------------------------------------------------------------------------------
--                                                                          --
--                         GNAT COMPILER COMPONENTS                         --
--                                                                          --
--                              S E M . C H 8                               --
--                                                                          --
--                                 B o d y                                  --
--                                                                          --
--                            $Revision: 1.193 $                            --
--                                                                          --
--           Copyright (c) 1992,1993,1994 NYU, All Rights Reserved          --
--                                                                          --
-- GNAT is free software;  you can  redistribute it  and/or modify it under --
-- terms of the  GNU General Public License as published  by the Free Soft- --
-- ware  Foundation;  either version 2,  or (at your option) any later ver- --
-- sion.  GNAT is distributed in the hope that it will be useful, but WITH- --
-- OUT ANY WARRANTY;  without even the  implied warranty of MERCHANTABILITY --
-- or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License --
-- for  more details.  You should have  received  a copy of the GNU General --
-- Public License  distributed with GNAT;  see file COPYING.  If not, write --
-- to the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA. --
--                                                                          --
------------------------------------------------------------------------------

with Atree;    use Atree;
with Debug;    use Debug;
with Einfo;    use Einfo;
with Elists;   use Elists;
with Errout;   use Errout;
with Features; use Features;
with Namet;    use Namet;
with Nlists;   use Nlists;
with Nmake;    use Nmake;
with Opt;      use Opt;
with Output;   use Output;
with Sem;      use Sem;
with Sem_Attr; use Sem_Attr;
with Sem_Ch3;  use Sem_Ch3;
with Sem_Ch4;  use Sem_Ch4;
with Sem_Ch5;  use Sem_Ch5;
with Sem_Ch6;  use Sem_Ch6;
with Sem_Ch8;  use Sem_Ch8;
with Sem_Res;  use Sem_Res;
with Sem_Util; use Sem_Util;
with Sem_Type; use Sem_Type;
with Stand;    use Stand;
with Sinfo;    use Sinfo;
with Sinfo.CN; use Sinfo.CN;
with Snames;   use Snames;
with Table;
with Tbuild;   use Tbuild;

package body Sem_Ch8 is

   ------------------------------------
   -- Visibility and Name Resolution --
   ------------------------------------

   --  This package handles name resolution and the collection of
   --  interpretations for overloaded names, prior to overload resolution.

   --  Name resolution is the process that establishes a mapping between source
   --  identifiers and the entities they denote at each point in the program.
   --  Each entity is represented by a defining occurrence. Each identifier
   --  that denotes an entity points to the corresponding defining occurrence.
   --  This is the entity of the applied occurrence. Each occurrence holds
   --  an index into the names table, where source identifiers are stored.

   --  Each entry in the names table for an identifier or designator uses the
   --  Info pointer to hold a link to the currently visible entity that has
   --  this name (see subprograms Get_Name_Entity_Id and Set_Name_Entity_Id
   --  in package Sem_Util). The visibility is initialized at the beginning of
   --  semantic processing to make entities in package Standard immediately
   --  visible. The visibility table is used in a more subtle way when
   --  compiling subunits (see below).

   --  Entities that have the same name (i.e. homonyms) are chained. In the
   --  case of overloaded entities, this chain holds all the possible meanings
   --  of a given identifier. The process of overload resolution uses type
   --  information to select from this chain the unique meaning of a given
   --  identifier.

   --  Entities are also chained in their scope, through the Next_Entity link.
   --  As a consequence, the name space is organized as a sparse matrix, where
   --  each row corresponds to a scope, and each column to a source identifier.
   --  Open scopes, that is to say scopes currently being compiled, have their
   --  corresponding rows of entities in order, innermost scope first.

   --  The scopes of packages that are mentioned in  context clauses appear in
   --  no particular order, interspersed among open scopes. This is because
   --  in the course of analyzing the context of a compilation, a package
   --  declaration is first an open scope, and subsequently an element of the
   --  context. If subunits or child units are present, a parent unit may
   --  appear under various guises at various times in the compilation.

   --  When the compilation of the innermost scope is complete, the entities
   --  defined therein are no longer visible. If the scope is not a package
   --  declaration, these entities are never visible subsequently, and can be
   --  removed from visibility chains. If the scope is a package declaration,
   --  its visible declarations may still be accessible. Therefore the entities
   --  defined in such a scope are left on the visibility chains, and only
   --  their visibility (immediately visibility or potential use-visibility)
   --  is affected.

   --  The ordering of homonyms on their chain does not necessarily follow
   --  the order of their corresponding scopes on the scope stack. For
   --  example, if package P and the enclosing scope both contain entities
   --  named E, then when compiling the package body the chain for E will
   --  hold the global entity first,  and the local one (corresponding to
   --  the current inner scope) next. As a result, name resolution routines
   --  do not assume any relative ordering of the homonym chains, either
   --  for scope nesting or to order of appearance of context clauses.

   --  When compiling a child unit, entities in the parent scope are always
   --  immediately visible. When compiling the body of a child unit, private
   --  entities in the parent must also be made immediately visible. There
   --  are separate routines to make the visible and private declarations
   --  visible at various times (see package Sem_Ch7).

   --              +--------+         +-----+
   --              | In use |-------->| EU1 |-------------------------->
   --              +--------+         +-----+
   --                                    |                      |
   --      +--------+                 +-----+                +-----+
   --      | Stand. |---------------->| ES1 |--------------->| ES2 |--->
   --      +--------+                 +-----+                +-----+
   --                                    |                      |
   --              +---------+           |                   +-----+
   --              | with'ed |------------------------------>| EW2 |--->
   --              +---------+           |                   +-----+
   --                                    |                      |
   --      +--------+                 +-----+                +-----+
   --      | Scope2 |---------------->| E12 |--------------->| E22 |--->
   --      +--------+                 +-----+                +-----+
   --                                    |                      |
   --      +--------+                 +-----+                +-----+
   --      | Scope1 |---------------->| E11 |--------------->| E12 |--->
   --      +--------+                 +-----+                +-----+
   --          ^                         |                      |
   --          |                         |                      |
   --          |   +---------+           |                      |
   --          |   | with'ed |----------------------------------------->
   --          |   +---------+           |                      |
   --          |                         |                      |
   --      Scope stack                   |                      |
   --      (innermost first)             |                      |
   --                                 +----------------------------+
   --      Names  table =>            | Id1 |     |    |     | Id2 |
   --                                 +----------------------------+

   --  Name resolution must deal with several syntactic forms: simple names,
   --  qualified names, indexed names, and various forms of calls.

   --  Each identifier points to an entry in the names table. The resolution
   --  of a simple name consists in traversing the homonym chain, starting
   --  from the names table. If an entry is immediately visible, it is the one
   --  designated by the identifier. If only potemtially use-visible entities
   --  are on the chain, we must verify that they do not hide each other. If
   --  the entity we find is overloadable, we collect all other overloadable
   --  entities on the chain as long as they are not hidden.
   --
   --  To resolve expanded names, we must find the entity at the intersection
   --  of the entity chain for the scope (the prefix) and the homonym chain
   --  for the selector. In general, homonym chains will be much shorter than
   --  entity chains, so it is preferable to start from the names table as
   --  well. If the entity found is overloadable, we must collect all other
   --  interpretations that are defined in the scope denoted by the prefix.

   --  For records, protected types, and tasks, their local entities are
   --  removed from visibility chains on exit from the corresponding scope.
   --  From the outside, these entities are always accessed by selected
   --  notation, and the entity chain for the record type, protected type,
   --  etc. is traversed sequentially in  order to find the designated entity.

   --  The discriminants of a type and the operations of a protected type or
   --  task are unchained on  exit from the first view of the type, (such as
   --  a private or incomplete type declaration, or a protected type speci-
   --  fication) and rechained when compiling the second view.

   --  In the case of operators,  we do not make operators on derived types
   --  explicit. As a result, the notation P."+" may denote either a user-
   --  defined function with name "+", or else an implicit declaration of the
   --  operator "+" in package P. The resolution of expanded names always
   --  tries to resolve an operator name as such an implicitly defined entity,
   --  in addition to looking for explicit declarations.

   --  All forms of names that denote entities (simple names, expanded names,
   --  character literals in some cases) have a Entity attribute, which
   --  identifies the entity denoted by the name.

   ---------------------
   -- The Scope Stack --
   ---------------------

   --  The Scope stack keeps track of the scopes currently been compiled.
   --  Every entity that contains declarations (including records) is placed
   --  on the scope stack while it is being processed, and removed at the end.
   --  Whenever a non-package scope is exited, the entities defined therein
   --  are removed from the visibility table, so that entities in outer scopes
   --  become visible (see previous description). On entry to Sem, the scope
   --  stack only contains the package Standard. As usual, subunits complicate
   --  this picture ever so slightly.

   --  The Rtsfind mechanism can force a call to Semantics while another
   --  compilation is in progress. The unit retrieved by Rtsfind must be
   --  compiled in  its own context, and has no access to the visibility of
   --  the unit currently being compiled. The procedures Save_Scope_Stack and
   --  Restore_Scope_Stack make entities in current open scopes invisible
   --  before compiling the retrieved unit, and restore the compilation
   --  environment afterwards.

   ------------------------
   -- Compiling subunits --
   ------------------------

   --  Subunits must be compiled in the environment of the corresponding
   --  stub, that is to say with the same visibility into the parent (and its
   --  context) that is available at the point of the stub declaration, but
   --  with the additional visibility provided by the context clause of the
   --  subunit itself. As a result, compilation of a subunit forces compilation
   --  of the parent (see description in lib-). At the point of the stub
   --  declaration, Analyze is called recursively to compile the proper body
   --  of the subunit, but without reinitializing the names table, nor the
   --  scope stack (i.e. standard is not pushed on the stack). In this fashion
   --  the context of the subunit is added to the context of the parent, and
   --  the subunit is compiled in the correct environment. Note that in the
   --  course of processing the context of a subunit, Standard will appear
   --  twice on the scope stack: once for the parent of the subunit, and
   --  once for the unit in the context clause being compiled. However, the
   --  two sets of entities are not linked by homonym chains, so that the
   --  compilation of any context unit happens in a fresh visibility
   --  environment.

   -------------------------------
   -- Processing of USE Clauses --
   -------------------------------

   --  Every defining occurrence has a flag indicating if it is potentially use
   --  visible. Resolution of simple names examines this flag. The processing
   --  of use clauses consists in setting this flag on all visible entities
   --  defined in the corresponding package. On exit from the scope of the use
   --  clause, the corresponding flag must be reset. However, a package may
   --  appear in several nested use clauses (pathological but legal, alas!)
   --  which forces us to use a slightly more involved scheme:

   --    a) The defining occurrence for a package holds a flag -In_Use- to
   --    indicate that it is currently in the scope of a use clause. If a
   --    redundant use clause is encountered, then the corresponding occurence
   --    of the package name is flagged -Redundant_Use-.

   --    b) On exit from a scope, the use clauses in its declarative part are
   --    scanned. The visibility flag is reset in all entities declared in
   --    package named in a use clause, as long as the package is not flagged
   --    as being in a redundant use clause (in which case the outer use
   --    clause is still in effect, and the direct visibility of its entities
   --    must be retained).

   --  Note that entities are not removed from their homonym chains on exit
   --  from the package specification. A subsequent use clause does not need
   --  to rechain the visible entities, but only to establish their direct
   --  visibility.

   -----------------------------------
   -- Handling private declarations --
   -----------------------------------

   --  The principle that each entity has a single defining occurrence clashes
   --  with the presence of two separate definitions for private types: the
   --  first is the private type declaration, and second is the full type
   --  declaration. It is important that all references to the type point to
   --  the same defining occurence, namely the first one. To enforce the two
   --  separate views of the entity, the corresponding information is swapped
   --  between the two declarations. Outside of the package, the defining
   --  occurence only contains the private declaration information, while in
   --  the private part and the body of the package the defining occurrence
   --  contains the full declaration. To simplify the swap, the defining
   --  occurrence that currently holds the private declaration points to the
   --  full declaration. During semantic processing the defining occurence
   --  also points to a list of private dependents, that is to say access
   --  types or composite types whose designated types or component types are
   --  subtypes or derived types of the private type in question. After the
   --  full declaration has been seen, the private dependents are updated to
   --  indicate that they have full definitions.

   ------------------------------------
   -- Handling of Undefined Messages --
   ------------------------------------

   --  In normal mode, only the first use of an undefined identifier generates
   --  a message. The table Urefs is used to record error messages that have
   --  been issued so that second and subsequent ones do not generate further
   --  messages. However, the second reference causes text to be added to the
   --  original undefined message noting "(more references follow)". The
   --  full error list option (-gnatf) forces messages to be generated for
   --  every reference and disconnects the use of this table.

   type Uref_Entry is record
      Node : Node_Id;
      --  Node for identifier for which original message was posted. The
      --  Chars field of this identifier is used to detect later references
      --  to the same identifier.

      Err : Error_Msg_Id;
      --  Records error message Id of original undefined message. Reset to
      --  No_Error_Msg after the second occurrence, where it is used to add
      --  text to the original message as described above.

      ND_Vis : Boolean;
      --  Set if the message is not directly visible rather than undefined

   end record;

   package Urefs is new Table (
     Table_Component_Type => Uref_Entry,
     Table_Index_Type     => Nat,
     Table_Low_Bound      => 1,
     Table_Initial        => 10,
     Table_Increment      => 100,
     Table_Name           => "Urefs");

   -----------------------
   -- Local Subprograms --
   -----------------------

   procedure Analyze_Generic_Renaming
     (N : Node_Id;
      K : Entity_Kind);
   --  Common processing for all three kinds of generic renaming declarations.
   --  Enter new name and indicate that it renames the generic unit.

   procedure Attribute_Renaming (N : Node_Id);
   --  Analyze renaming of attribute as function. The renaming declaration
   --  is rewritten as a function body that returns the attribute reference
   --  applied to the formals of the function.

   function Entity_Matches_Spec (Old_S,  New_S : Entity_Id) return Boolean;
   --  Used to resolve subprogram renaming. Old_S is a possible interpretation
   --  of the entity being renamed, New_S has an explicit signature. If Old_S
   --  is a subprogram, type conformance is required.

   function Get_Full_View (T_Name : Entity_Id) return Entity_Id;
   --  If T_Name is an incomplete type and the full declaration has been
   --  seen, or is the name of a private type and the current context is
   --  one in which the full declaration is visible, return the full
   --  declaration.

   procedure Use_One_Package (P : Entity_Id);
   --  Make visible entities declarated in  package P potentially use-visible
   --  in the current context.

   procedure Write_Info;
   --  Write debugging information on entities declared in current scope

   --------------------------------
   -- Analyze_Exception_Renaming --
   --------------------------------

   --  The language only allows a single identifier, but the tree holds
   --  an identifier list. The parser has already issued an error message
   --  if there is more than one element in the list.

   procedure Analyze_Exception_Renaming (N : Node_Id) is
      Id  : constant Node_Id   := Defining_Identifier (N);
      Nam : constant Node_Id   := Name (N);

   begin
      Enter_Name (Id);
      Set_Ekind (Id,  E_Exception);
      Set_Etype (Id, Standard_Exception_Type);
      Analyze (Nam);

      if Ekind (Entity (Nam)) /= E_Exception then
         Error_Msg_N ("invalid exception name in renaming", Nam);
      else
         if Present (Renamed_Object (Entity (Nam))) then
            Set_Renamed_Object (Id, Renamed_Object (Entity (Nam)));
         else
            Set_Renamed_Object (Id, Entity (Nam));
         end if;
      end if;
   end Analyze_Exception_Renaming;

   ----------------------------------------
   --  Analyze_Generic_Function_Renaming --
   ----------------------------------------

   procedure Analyze_Generic_Function_Renaming  (N : Node_Id) is
   begin
      Analyze_Generic_Renaming (N, E_Generic_Function);
   end Analyze_Generic_Function_Renaming;

   ---------------------------------------
   --  Analyze_Generic_Package_Renaming --
   ---------------------------------------

   procedure Analyze_Generic_Package_Renaming   (N : Node_Id) is
   begin
      Analyze_Generic_Renaming (N, E_Generic_Package);
   end Analyze_Generic_Package_Renaming;

   -----------------------------------------
   --  Analyze_Generic_Procedure_Renaming --
   -----------------------------------------

   procedure Analyze_Generic_Procedure_Renaming (N : Node_Id) is
   begin
      Analyze_Generic_Renaming (N, E_Generic_Procedure);
   end Analyze_Generic_Procedure_Renaming;

   -------------------------------
   --  Analyze_Generic_Renaming --
   -------------------------------

   procedure Analyze_Generic_Renaming
     (N : Node_Id;
      K : Entity_Kind)
   is
      New_P : Entity_Id := Defining_Unit_Simple_Name (N);
      Old_P : Entity_Id;
   begin
      Analyze (Name (N));
      Old_P := Entity (Name (N));
      Enter_Name (New_P);
      Set_Ekind (New_P, K);

      if Etype (Old_P) = Any_Type then
         null;

      elsif Ekind (Old_P) /= K then
         Error_Msg_N ("invalid generic unit name", Name (N));

      else
         if Present (Renamed_Object (Old_P)) then
            Set_Renamed_Object (New_P,  Renamed_Object (Old_P));
         else
            Set_Renamed_Object (New_P,  Old_P);
         end if;

         Set_Has_Completion (New_P);
      end if;
   end Analyze_Generic_Renaming;

   -----------------------------
   -- Analyze_Object_Renaming --
   -----------------------------

   procedure Analyze_Object_Renaming (N : Node_Id) is
      Id  : constant Entity_Id := Defining_Identifier (N);
      Nam : constant Node_Id   := Name (N);
      S   : constant Entity_Id := Subtype_Mark (N);
      T   : Entity_Id;
      T2  : Entity_Id;

   begin
      Enter_Name (Id);
      Analyze (Nam);
      T2 := Etype (Nam);
      Find_Type (S);
      T := Entity (S);
      Resolve (Nam, T);
      Set_Ekind (Id, E_Variable);

      if T = Any_Type or else Etype (Nam) = Any_Type then
         return;

      elsif not Is_Constrained (T) then

         --  The constraints are inherited from the renamed object, they
         --  are not affected by the given subtype mark.

         Set_Etype (Id, T2);

      else
         Set_Etype (Id, T);
      end if;

      if not Is_Variable (Nam) then
         Set_Ekind (Id, E_Constant);
      end if;

      Set_Renamed_Object (Id, Nam);
   end Analyze_Object_Renaming;

   ------------------------------
   -- Analyze_Package_Renaming --
   ------------------------------

   procedure Analyze_Package_Renaming (N : Node_Id) is
      New_P : constant Entity_Id := Defining_Unit_Simple_Name (N);
      Old_P : Entity_Id;

   begin
      Analyze (Name (N));
      Old_P := Entity (Name (N));

      if Etype (Old_P) = Any_Type then
         null;

      elsif Ekind (Old_P) /= E_Package then
         Error_Msg_N ("invalid package name", Name (N));

      else
         --  Entities in the old package are accessible through the
         --  renaming entity. The simplest implementation is to have
         --  both packages share the entity list.

         Enter_Name (New_P);
         Set_Ekind (New_P, E_Package);
         Set_Etype (New_P, Standard_Void_Type);

         if Present (Renamed_Object (Old_P)) then
            Set_Renamed_Object (New_P,  Renamed_Object (Old_P));
         else
            Set_Renamed_Object (New_P,  Old_P);
         end if;

         Set_Has_Completion (New_P);

         Set_First_Entity (New_P,  First_Entity (Old_P));
         Set_Last_Entity  (New_P,  Last_Entity  (Old_P));
         Set_First_Private_Entity (New_P, First_Private_Entity (Old_P));
      end if;
   end Analyze_Package_Renaming;

   ---------------------------------
   -- Analyze_Subprogram_Renaming --
   ---------------------------------

   procedure Analyze_Subprogram_Renaming (N : Node_Id) is
      Loc   : constant Source_Ptr := Sloc (N);
      Nam   : constant Node_Id := Name (N);
      Spec  : constant Node_Id := Specification (N);
      New_S : Entity_Id;
      Old_S : Entity_Id;
      Prev  : Entity_Id;
      I     : Interp_Index;
      It    : Interp;

   begin
      if Nkind (Nam) = N_Attribute_Reference then
         Attribute_Renaming (N);
         return;
      end if;

      --  The renaming defines a new overloaded entity, which is analyzed
      --  like a subprogram declaration.

      New_S := Analyze_Spec (Spec);
      Prev := Find_Corresponding_Spec (N);

      if Present (Prev) then

         --  Renaming_As_Body. Renaming declaration is the completion of
         --  the declaration of Prev.

         Note_Feature (Subprogram_Bodies_By_Renaming, Sloc (N));

         if not Type_Conformant (Prev, New_S) then
            Error_Msg_N
              ("renamed entity not type conformant with declaration", N);

         elsif Ada_83 then
            Error_Msg_N ("(Ada 83) renaming cannot serve as a body", N);
         end if;

         New_S := Prev;

      else
         New_Overloaded_Entity (New_S);
      end if;

      --  We must test for the attribute renaming case before the Analyze
      --  call because otherwise Sem_Attr will complain that the attribute
      --  is missing an argument when it is analyzed.

      --  All other cases

      Set_Has_Completion (New_S);
      Analyze (Nam);

      if Etype (Nam) = Any_Type then
         null;

      elsif not Is_Overloadable (Entity (Nam)) then
         Error_Msg_N ("expect valid subprogram name in renaming", N);

      else
         --  Now we must resolve the subprogram renaming

         Old_S := Any_Id;

         if not Is_Overloaded (Name (N)) then
            if Entity_Matches_Spec (Entity (Name (N)), New_S) then
               Old_S := Entity (Name (N));
            end if;

         else
            Get_First_Interp (Name (N),  I,  It);

            while Present (It.Nam) loop

               if Entity_Matches_Spec (It.Nam, New_S) then
                  if Old_S /= Any_Id then
                     Old_S := Disambiguate (It.Nam, Old_S, Etype (Old_S));

                     if Old_S = Any_Id then
                        Error_Msg_N ("ambiguous renaming", N);
                        exit;
                     end if;
                  else
                     Old_S := It.Nam;
                  end if;
               end if;

               Get_Next_Interp (I, It);
            end loop;
         end if;

         if Old_S /= Any_Id then

            --  The subprogram being renamed must be mode conformant
            --  with the given specification.

            if Ekind (Old_S) /= E_Operator then
               if not Mode_Conformant (New_S,  Old_S) then
                  Error_Msg_N
                   ("renamed entity not mode conformant with declaration",
                    N);
               end if;
            end if;

            if Present (Prev) then

               --  Rewrite renaming declaration as a subprogram body, whose
               --  single statement is a call to Old_S.

               declare
                  Actuals   : List_Id := New_List;
                  Call_Node : Node_Id;
                  Body_Node : Node_Id;
                  Formal    : Entity_Id;

               begin
                  --  Reset completion flag, so this body is seen as the
                  --  proper completion.

                  Set_Has_Completion (New_S, False);
                  Formal := First_Formal (New_S);

                  while Present (Formal) loop
                     Append (New_Reference_To (Formal, Loc), Actuals);
                     Formal := Next_Formal (Formal);
                  end loop;

                  if Ekind (Old_S) = E_Function then
                     Call_Node := Make_Return_Statement (Loc,
                        Expression =>
                          Make_Function_Call (Loc,
                            Name => New_Reference_To (Old_S,  Loc),
                            Parameter_Associations => Actuals));
                  else
                     Call_Node := Make_Procedure_Call_Statement (Loc,
                       Name => New_Reference_To (Old_S,  Loc),
                       Parameter_Associations => Actuals);
                  end if;

                  Body_Node := Make_Subprogram_Body (Loc,
                    Specification => Spec,
                    Declarations => New_List,
                    Handled_Statement_Sequence =>
                      Make_Handled_Sequence_Of_Statements (Loc,
                        Statements => New_List (Call_Node)));

                  Analyze (Body_Node);
                  Rewrite_Substitute_Tree (N,  Body_Node);
               end;

            else
               if Present (Alias (Old_S)) then
                  Set_Alias (New_S, Alias (Old_S));
               else
                  Set_Alias (New_S, Old_S);
               end if;

               if Old_S = New_S then
                  Error_Msg_N ("subprogram cannot rename itself", N);
               end if;
            end if;

         else
            Error_Msg_N
              ("No visible subprogram matches this specification", Spec);
         end if;
      end if;
   end Analyze_Subprogram_Renaming;

   -------------------------
   -- Analyze_Use_Package --
   -------------------------

   --  Resolve the package names in the use clause, and make all the visible
   --  entities defined in the package potentially use-visible. If the package
   --  is already in use from a previous use clause, its visible entities are
   --  already use-visible. In that case, mark the occurrence as a redundant
   --  use. If the package is an open scope, i.e. if the use clause occurs
   --  within the package itself, ignore it.

   procedure Analyze_Use_Package (N : Node_Id) is
      Pack_Name : Node_Id;
      Pack      : Entity_Id;

      function Find_Package_Unit return Entity_Id;
      --  Internal function used to scan the context clause to find the
      --  previous occurence of package name in with-clause, or report error.

      function Find_Package_Unit return Entity_Id is
         Item : Node_Id := First (Context_Items (Parent (N)));

      begin
         while Present (Item) and then Item /= N loop
            if Nkind (Item) = N_With_Clause
              and then Same_Name (Name (Item), Pack_Name)
            then
               return Entity (Name (Item));
            end if;

            Item := Next (Item);
         end loop;

         Error_Msg_N ("unknown package in USE clause", N);
         return Standard_Standard;
      end Find_Package_Unit;

   --  Start processing for Use_Package_Clause

   begin
      --  Loop through package names to identify referenced packages

      Pack_Name := First (Names (N));
      while Present (Pack_Name) loop

         --  If this use clause appears in the context clause of a compilation
         --  unit, then we have to find the corresponding with clause to
         --  identify the referenced package.

         if Nkind (Parent (N)) = N_Compilation_Unit then
            Pack := Find_Package_Unit;

            if Nkind (Pack_Name) = N_Selected_Component then
               Change_Selected_Component_To_Expanded_Name (Pack_Name);
            end if;

            Set_Entity (Pack_Name, Pack);

         --  If the use clause is appearing in a normal declaration context,
         --  then it references a package name that is visible by normal rules.

         else
            Analyze (Pack_Name);
         end if;

         Pack_Name := Next (Pack_Name);
      end loop;

      --  Loop through package names to mark all entities as potentially
      --  use visible

      Pack_Name := First (Names (N));
      while Present (Pack_Name) loop
         Pack := Entity (Pack_Name);

         if Ekind (Pack) /= E_Package
           and then Etype (Pack) /= Any_Type
         then
            Error_Msg_N ("& is not a usable package", Pack_Name);

         else
            if In_Open_Scopes (Pack) then
               null;

            elsif Present (Renamed_Object (Pack))
              and then In_Use (Renamed_Object (Pack))
            then
               Set_Redundant_Use (Pack_Name, True);

            elsif not In_Use (Pack)
              or else Nkind (Parent (N)) = N_Compilation_Unit
            then
               Use_One_Package (Pack);

            else
               Set_Redundant_Use (Pack_Name, True);
            end if;
         end if;

         Pack_Name := Next (Pack_Name);
      end loop;

   end Analyze_Use_Package;

   ----------------------
   -- Analyze_Use_Type --
   ----------------------

   procedure Analyze_Use_Type (N : Node_Id) is
      Id      : Entity_Id;
      Op_List : Elist_Id;
      Elmt    : Elmt_Id;

   begin
      Id := First (Subtype_Marks (N));
      while Present (Id) loop
         Find_Type (Id);

         if Entity (Id) /= Any_Type then

            --  Save current visibility status of type, before setting.

            Set_Redundant_Use (Id, Is_Potentially_Use_Visible (Entity (Id)));
            Set_Is_Potentially_Use_Visible (Entity (Id));

            --  If the base type is anonymous this indicates that the entity
            --  is really a first named subtype and we need to make the
            --  base type of the entity "use type" visible as well. Otherwise,
            --  operators which are defined on the type, and which the user
            --  wanted to make directly visible by means of this use clause,
            --  would in fact not become visible.

            if not Comes_From_Source (Base_Type (Entity (Id))) then
               Set_Is_Potentially_Use_Visible (Base_Type (Entity (Id)));
            end if;

            if not Redundant_Use (Id) then

               Op_List := Collect_Primitive_Operations (Entity (Id));
               Elmt := First_Elmt (Op_List);

               while Present (Elmt) loop

                  if Nkind (Node (Elmt)) = N_Defining_Operator_Symbol then
                     Set_Is_Potentially_Use_Visible (Node (Elmt));
                  end if;

                  Elmt := Next_Elmt (Elmt);
               end loop;
            end if;
         end if;

         Id := Next (Id);
      end loop;
   end Analyze_Use_Type;

   ------------------------
   -- Attribute_Renaming --
   ------------------------

   procedure Attribute_Renaming (N : Node_Id) is
      Loc         : constant Source_Ptr := Sloc (N);
      Nam         : constant Node_Id    := Name (N);
      Spec        : constant Node_Id    := Specification (N);
      Attr_Node   : Node_Id;
      Body_Node   : Node_Id;
      Expr_List   : List_Id;
      Param_Spec  : Node_Id;

   begin
      Param_Spec := First (Parameter_Specifications (Spec));

      if No (Param_Spec) then
         Error_Msg_N ("function renaming an attribute must have formals", N);
         return;
      else
         Expr_List := New_List (
           Make_Identifier (Loc,
             Chars => Chars (Defining_Identifier (Param_Spec))));
      end if;

      if Present (Next (Param_Spec)) then
         Append (
           Make_Identifier (Loc,
             Chars => Chars (Defining_Identifier (Next (Param_Spec)))),
           Expr_List);

         if Present (Next (Next (Param_Spec))) then
            Error_Msg_N ("too many formals for attribute", N);
         end if;

         --  Other mismatches in the number of parameters are detected
         --  in the subsequent analysis of the attribute reference.

      end if;

      Attr_Node :=
        Make_Attribute_Reference (Loc,
          Prefix => Prefix (Nam),
          Attribute_Name => Attribute_Name (Nam),
          Expressions => Expr_List);

      Body_Node :=
        Make_Subprogram_Body (Loc,
          Specification => Spec,
          Declarations => New_List,
          Handled_Statement_Sequence =>
            Make_Handled_Sequence_Of_Statements (Loc,
                Statements => New_List (
                  Make_Return_Statement (Loc,
                    Expression => Attr_Node))));

      Rewrite_Substitute_Tree (N, Body_Node);
      Analyze (N);

   end Attribute_Renaming;

   ---------------
   -- End_Scope --
   ---------------

   procedure End_Scope is
      Id    : Entity_Id;
      Prev  : Entity_Id;
      Outer : Entity_Id;

   begin
      Id := First_Entity (Current_Scope);

      while Present (Id) loop
         --  An entity in the current scope is not necessarily the first one
         --  on its homonym chain. Find its predecessor if any,
         --  If it is an internal entity, it will not be in the visibility
         --  chain altogether,  and there is nothing to unchain.

         if Id /= Current_Entity (Id) then
            Prev := Current_Entity (Id);
            while Present (Prev)
              and then Present (Homonym (Prev))
              and then Homonym (Prev) /= Id
            loop
               Prev := Homonym (Prev);
            end loop;

            --  Skip to end of loop if Id is not in the visibility chain

            if No (Prev) or else Homonym (Prev) /= Id then
               goto Next_Ent;
            end if;

         else
            Prev := Empty;
         end if;

         Outer := Homonym (Id);
         Set_Is_Immediately_Visible (Id, False);

         while Present (Outer) and then Scope (Outer) = Current_Scope loop
            Outer := Homonym (Outer);
         end loop;

         if No (Prev) then
            Set_Name_Entity_Id (Chars (Id), Outer);
         else
            Set_Homonym (Prev,  Outer);
         end if;

         <<Next_Ent>> Id  := Next_Entity (Id);
      end loop;

      Pop_Scope;
   end End_Scope;

   -------------
   -- End_Use --
   -------------

   procedure End_Use (L : List_Id) is
      Decl    : Node_Id;
      Id      : Entity_Id;

   begin
      if Present (L) then
         Decl := First (L);

         while Present (Decl) loop
            if Nkind (Decl) = N_Use_Package_Clause then
               End_Use_Package (Decl);

            elsif Nkind (Decl) = N_Use_Type_Clause then
               End_Use_Type (Decl);
            end if;

            Decl := Next (Decl);
         end loop;
      end if;

   end End_Use;

   ---------------------
   -- End_Use_Package --
   ---------------------

   procedure End_Use_Package (N : Node_Id) is
      Pack_Name : Node_Id;
      Pack      : Entity_Id;
      Id        : Entity_Id;

   begin
      Pack_Name := First (Names (N));

      while Present (Pack_Name) loop
         Pack := Entity (Pack_Name);

         if Ekind (Pack) = E_Package then
            if not Redundant_Use (Pack_Name) then
               Set_In_Use (Pack, False);
               Id := First_Entity (Pack);

               while Present (Id) loop
                  Set_Is_Potentially_Use_Visible (Id, False);
                  Id := Next_Entity (Id);
               end loop;

               if Present (Renamed_Object (Pack)) then
                  Set_In_Use (Renamed_Object (Pack), False);
               end if;

            end if;

         end if;

         Pack_Name := Next (Pack_Name);
      end loop;
   end End_Use_Package;

   ------------------
   -- End_Use_Type --
   ------------------

   procedure End_Use_Type (N : Node_Id) is
      Id      : Entity_Id;
      Op_List : Elist_Id;
      Elmt    : Elmt_Id;

   begin
      Id := First (Subtype_Marks (N));

      while Present (Id) loop

         if Entity (Id) /= Any_Type then

            --  Reset visibility status.

            Set_Is_Potentially_Use_Visible
              (Entity (Id), Redundant_Use (Id));
            Set_Is_Potentially_Use_Visible
              (Base_Type (Entity (Id)), Redundant_Use (Id));
         end if;

         if not Redundant_Use (Id) then
            Op_List := Collect_Primitive_Operations (Entity (Id));
            Elmt := First_Elmt (Op_List);

            while Present (Elmt) loop

               if Nkind (Node (Elmt)) = N_Defining_Operator_Symbol then
                  Set_Is_Potentially_Use_Visible (Node (Elmt), False);
               end if;

               Elmt := Next_Elmt (Elmt);
            end loop;
         end if;

         Id := Next (Id);
      end loop;
   end End_Use_Type;

   -------------------------
   -- Entity_Matches_Spec --
   -------------------------

   function Entity_Matches_Spec
    (Old_S,  New_S : Entity_Id)
     return          Boolean
   is
   begin
      --  Simple case: same entity kinds, type conformance is required.
      --  A parameterless function can also rename a literal.

      if Ekind (Old_S) = Ekind (New_S)
        or else (Ekind (New_S) = E_Function
                  and then Ekind (Old_S) = E_Enumeration_Literal)
      then
         return Type_Conformant (Old_S,  New_S);

      elsif Ekind (New_S) = E_Function and then Ekind (Old_S) = E_Operator then
         return Operator_Matches_Spec (Old_S, New_S);

      elsif Ekind (New_S) = E_Procedure and then Ekind (Old_S) = E_Entry then
         Unimplemented (Old_S, "entries renamed as procedures");
         return False;

      else
         return False;
      end if;
   end Entity_Matches_Spec;

   ----------------------
   -- Find_Direct_Name --
   ----------------------

   procedure Find_Direct_Name (N : Node_Id) is
      E   : Entity_Id;
      E2  : Entity_Id;
      Msg : Boolean;

      Homonyms : Entity_Id;
      --  Saves start of homonym chain

      Multiple_Overloadable_Entities : Boolean := False;
      --  This flag is set only if there are multiple overloadable entities
      --  that match (used at the end of processing to determine whether it
      --  is necessary to collect overloaded interpretations).
      --  ??? not used for this purpose yet, pending resolving some open
      --  issues with how Collect_Interps operates.

      procedure ND_Vis_Messages;
      --  Called if there are no directly visible entries for N, but there
      --  is at least one non-directly visible, or hidden declaration. This
      --  procedure outputs an appropriate set of error messages.

      procedure Undefined (ND_Vis : Boolean);
      --  This function is called if the current node has no corresponding
      --  visible entity or entities. The value set in Msg indicates whether
      --  an error message was generated (multiple error messages for the
      --  same variable are generally suppressed, see body for details).
      --  Msg is True if an error message was generated, False if not. This
      --  value is used by the caller to determine whether or not to output
      --  additional messages where appropriate. The parameter is set False
      --  to get the message "X is undefined", and True to get the message
      --  "X is not directly visible".

      procedure ND_Vis_Messages is
         Ent : Entity_Id;

      begin
         Undefined (ND_Vis => True);

         if Msg then

            --  First loop does hidden declarations

            Ent := Homonyms;
            while Present (Ent) loop
               if Is_Potentially_Use_Visible (Ent) then
                  Error_Msg_Sloc := Sloc (Ent);
                  Error_Msg_N ("hidden declaration#!", N);
               end if;

               Ent := Homonym (Ent);
            end loop;

            --  Second loop does non-directly visible declarations

            Ent := Homonyms;
            while Present (Ent) loop
               if not Is_Potentially_Use_Visible (Ent) then
                  Error_Msg_Sloc := Sloc (Ent);
                  Error_Msg_N ("non-visible declaration#!", N);
               end if;

               Ent := Homonym (Ent);
            end loop;

         end if;
      end ND_Vis_Messages;

      procedure Undefined (ND_Vis : Boolean) is
      begin
         Set_Entity (N, Any_Id);
         Set_Etype  (N, Any_Type);

         --  We use the table Urefs to keep track of entities for which we
         --  have issued errors for undefined references. Multiple errors
         --  for a single name are normally suppressed, however we modify
         --  the error message to alert the programmer to this effect.

         for J in Urefs.First .. Urefs.Last loop
            if Chars (N) = Chars (Urefs.Table (J).Node) then
               if Urefs.Table (J).Err /= No_Error_Msg then
                  Error_Msg_Node_1 := Urefs.Table (J).Node;

                  if Urefs.Table (J).ND_Vis then
                     Change_Error_Text (Urefs.Table (J).Err,
                       "& is not directly visible (more references follow)");
                  else
                     Change_Error_Text (Urefs.Table (J).Err,
                       "& is undefined (more references follow)");
                  end if;

                  Urefs.Table (J).Err := No_Error_Msg;
               end if;

               --  Although we will set Msg False, and thus suppress the
               --  message, we also set Error_Posted True, to avoid any
               --  cascaded messages resulting from the undefined reference.

               Msg := False;
               Set_Error_Posted (N, True);
               return;
            end if;
         end loop;

         --  If entry not found, this is first undefined occurrence

         if ND_Vis then
            Error_Msg_N ("& is not directly visible!", N);
         else
            Error_Msg_N ("& is undefined!", N);
         end if;

         --  Make entry in undefined references table unless the full
         --  errors switch is set, in which case by refraining from
         --  generating the table entry, we guarantee that we get an
         --  error message for every undefined reference.

         if not All_Errors_Mode then
            Urefs.Increment_Last;
            Urefs.Table (Urefs.Last).Node   := N;
            Urefs.Table (Urefs.Last).Err    := Get_Msg_Id;
            Urefs.Table (Urefs.Last).ND_Vis := ND_Vis;
         end if;

         Msg := True;
      end Undefined;

   --  Start of processing for Find_Direct_Name

   begin
      --  If the entity pointer is already set, this is an internal node, or
      --  a node that is analyzed more than once, after a tree modification.
      --  In such a case there is no resolution to perform, just set the type.

      if Present (Entity (N)) then
         if Is_Type (Entity (N)) then
            Set_Etype (N, Entity (N));
         else
            Set_Etype (N, Etype (Entity (N)));
         end if;

         return;
      end if;

      --  Here if Entity pointer was not set, we need full visibility analysis
      --  First we generate debugging output if the debug E flag is set.

      if Debug_Flag_E then
         Write_Str ("Looking for ");
         Write_Name (Chars (N));
         Write_Eol;
      end if;

      Homonyms := Current_Entity (N);

      --  If no entries on homonym chain, then we have a simple undefined
      --  reference, with no additional explanation required!

      if No (Homonyms) then
         Undefined (ND_Vis => False);
         return;

      --  Otherwise search homonym chain for matching entry

      else
         E := Homonyms;
         loop
            if Is_Immediately_Visible (E) then
               goto Immediately_Visible_Entity;

            elsif Is_Potentially_Use_Visible (E) then
               goto Potentially_Use_Visible_Entity;

            else
               E := Homonym (E);
               exit when No (E);
            end if;
         end loop;

         --  We fall through the loop if there are entries on the homonynm
         --  chain, but none of them is currently visible.

         ND_Vis_Messages;
         return;
      end if;

      --  Processing for a potentially use visible entry found. We must search
      --  the rest of the homonym chain for two reasons. First, if there is a
      --  directly visible entry, then none of the potentially use-visible
      --  entities are directly visible (RM 8.4(10)). Second, we need to check
      --  for the case of multiple potentially use-visible entries hiding one
      --  another and as a result being non-directly visible (RM 8.4(11)).

      <<Potentially_Use_Visible_Entity>> declare
         Only_One_Visible : Boolean := True;
         All_Overloadable : Boolean := Is_Overloadable (E);

      begin
         E2 := Homonym (E);

         while Present (E2) loop
            if Is_Immediately_Visible (E2) then
               E := E2;
               goto Immediately_Visible_Entity;

            elsif Is_Potentially_Use_Visible (E2) then
               Only_One_Visible := False;
               All_Overloadable := All_Overloadable and Is_Overloadable (E2);
            end if;

            E2 := Homonym (E2);
         end loop;

         --  On falling through this loop, we have checked that there are no
         --  immediately visible entities. Only_One_Visible is set if exactly
         --  one potentially use visible entity exists. All_Overloadable is
         --  set if all the potentially use visible entities are overloadable.
         --  The condition for legality is that either there is one potentially
         --  use visible entity, or if there is more than one, then all of them
         --  are overloadable.

         if Only_One_Visible or All_Overloadable then
            goto Found;

         --  If there is more than one potentially use-visible entity and at
         --  least one of them non-overloadable, we have an error (RM 8.4(11).
         --  Note that E points to the first such entity on the homonym list.

         else
            ND_Vis_Messages;
            return;
         end if;
      end;

      --  Come here with E set to the first immediately visible entity on
      --  the homonym chain. This is the one we want unless there is another
      --  immediately visible entity further on in the chain for a more
      --  inner scope (RM 8.3(8)).

      <<Immediately_Visible_Entity>> declare
         Level : Int;
         Scop  : Entity_Id;
      begin
         --  Find scope level of initial entity

         Level := Scope_Stack.Last;
         loop
            Scop := Scope_Stack.Table (Level).Entity;
            exit when Scop = Scope (E) or else Scop = Standard_Standard;
            Level := Level - 1;
         end loop;

         --  Now search remainder of homonym chain for more inner entry

         E2 := Homonym (E);
         while Present (E2) loop
            if Is_Immediately_Visible (E2) then
               for J in Level + 1 .. Scope_Stack.Last loop
                  if Scope_Stack.Table (J).Entity = Scope (E2) then
                     Level := J;
                     E := E2;
                     exit;
                  end if;
               end loop;
            end if;

            E2 := Homonym (E2);
         end loop;

         --  At the end of that loop, E is the innermost immediately
         --  visible entity, so we are all set.
      end;

      --  Come here with entity found, and stored in E

      <<Found>> begin
         Set_Entity (N, E);

         if Is_Type (E) then
            Set_Etype (N, E);
         else
            Set_Etype (N, Get_Full_View (Etype (E)));
         end if;

         if Debug_Flag_E then
            Write_Str (" found  ");
            Write_Entity_Info (E, "      ");
         end if;

         --  If the Ekind of the entity is Void, it means that all
         --  homonyms are hidden from all visibility (RM 8.3(5,14-20))

         if Ekind (E) = E_Void then
            Error_Msg_N ("premature usage of&!", N);

         --  If the entity is overloadable, collect all interpretations
         --  of the name for subsequent overload resolution. We optimize
         --  a bit here to do this only if we have an overloadable entity
         --  that is not on its own on the homonym chain.

         elsif Is_Overloadable (E)
           and then (Present (Homonym (E)) or else Current_Entity (N) /= E)
         then
            Collect_Interps (N);
         else
            Set_Entity_With_Style_Check (N, E);
         end if;
      end;
   end Find_Direct_Name;

   ------------------------
   -- Find_Expanded_Name --
   ------------------------

   --  This routine searches the homonym chain of the entity until it finds
   --  an entity declared in the scope denoted by the prefix. If the entity
   --  is private, it may nevertheless be immediately visible, if we are in
   --  the scope of its declaration.

   procedure Find_Expanded_Name (N : Node_Id) is
      Selector : constant Node_Id := Selector_Name (N);
      P_Name   : Entity_Id;
      Id       : Entity_Id;

   begin
      P_Name := Entity (Prefix (N));

      --  If the prefix is a renamed package, look for the entity
      --  in the original package.

      if Ekind (P_Name) = E_Package
        and then Present (Renamed_Object (P_Name))
      then
         P_Name := Renamed_Object (P_Name);
      end if;

      Id := Current_Entity (Selector);

      while Present (Id) loop
         exit when  Scope (Id) = P_Name
           and then (not Is_Private (Id) or else Is_Immediately_Visible (Id));
         Id := Homonym (Id);
      end loop;

      if No (Id) or else Chars (Id) /=  Chars (Selector) then
         if (Nkind (Selector) = N_Operator_Symbol
           and then Has_Implicit_Operator (N))
         then
            --  There is an implicit instance of the predefined operator in the
            --  given scope. Find the predefined operator in scope Standard.

            Id := Current_Entity (Selector);
            while Present (Id) and then Scope (Id) /= Standard_Standard loop
               Id := Homonym (Id);
            end loop;

         else
            Error_Msg_Node_2 := P_Name;
            Error_Msg_NE ("& not declared in&", N, Selector);
            Id := Any_Id;
         end if;
      end if;

      Change_Selected_Component_To_Expanded_Name (N);
      Set_Entity_With_Style_Check (N, Id);

      if Is_Type (Id) then
         Set_Etype (N, Id);
      else
         Set_Etype (N, Get_Full_View (Etype (Id)));
      end if;

      if Is_Overloadable (Id) and then Present (Homonym (Id))
        and then Scope (Id) = Scope (Homonym (Id))
      then
         Collect_Interps (N);
      end if;

      if (Nkind (Selector_Name (N)) = N_Operator_Symbol
        and then Has_Implicit_Operator (N)
        and then Scope (Id) /= Standard_Standard)
      then
         --  In addition to user-defined operators in the given scope,
         --  there is also an implicit instance of the predefined
         --  operator. Find the predefined operator in scope
         --  Standard, and add it as well to the interpretations.
         --  Procedure Add_One_Interp will determine which hides which.

         Id := Current_Entity (Selector);

         while Present (Id)
           and then Scope (Id) /= Standard_Standard
         loop
            Id := Homonym (Id);
         end loop;

         Add_One_Interp (N, Id,  Etype (Id));
      end if;
   end Find_Expanded_Name;

   -----------------------------
   -- Find_Selected_Component --
   -----------------------------

   procedure Find_Selected_Component (N : Node_Id) is
      P : Node_Id := Prefix (N);

      P_Name : Entity_Id;
      --  Entity denoted by prefix

      P_Type : Entity_Id;
      --  and its type

   begin
      Analyze (P);

      if Nkind (P) = N_Error then
         return;

      --  If the selector already has an entity, the node has been
      --  constructed in the course of expansion, and is known to be
      --  valid. Do not verify that it is defined for the type (it may
      --  be a private component used in the expansion of record equality).

      elsif Present (Entity (Selector_Name (N))) then
         Set_Etype (Selector_Name (N),  Etype (Entity (Selector_Name (N))));
         Set_Etype (N, Etype (Entity (Selector_Name (N))));
         return;

      elsif Is_Entity_Name (P) then
         P_Name := Entity (P);
         P_Type := Etype (P);

         if Debug_Flag_E then
            Write_Str ("Found prefix type to be ");
            Write_Entity_Info (P_Type, "      "); Write_Eol;
         end if;

         if Is_Appropriate_For_Record (P_Type) then

            --  Selected component of record. Type checking will validate
            --  name of selector.

            Analyze_Selected_Component (N);

         elsif Is_Appropriate_For_Entry_Prefix (P_Type)
           and then not In_Open_Scopes (P_Name)
         then
            --  Call to protected operation or entry. Type checking is
            --  needed on the prefix.

            Analyze_Selected_Component (N);

         elsif In_Open_Scopes (P_Name)
           and then (Ekind (P_Name) /= E_Void
           and then not Is_Overloadable (P_Name))
         then
            --  Prefix denotes an enclosing loop, block, or task, i.e. an
            --  enclosing construct that is not a subprogram or accept.

            Find_Expanded_Name (N);

         elsif Ekind (P_Name) = E_Package then
            Find_Expanded_Name (N);

         elsif Ekind (P_Name) = E_Generic_Package then

            --  If the prefix is a generic_package,  this can only be the name
            --  of a generic child unit. This is not a selected component but
            --  a compilation unit name. (10.1.6 (6)).

            Find_Expanded_Name (N);

            if Nkind (Get_Declaration_Node (Entity (N)))
                                       not in N_Generic_Declaration
              or else No (Parent_Spec (Get_Declaration_Node (Entity (N))))
            then
               Error_Msg_N ("expect generic child unit as selector",  N);
            end if;

         elsif Is_Overloadable (P_Name) then

            if Is_Overloaded (P) then

               --  The prefix must resolve to a unique enclosing construct.

               declare
                  Found : Boolean := False;
                  I     : Interp_Index;
                  It    : Interp;
               begin
                  Get_First_Interp (P, I, It);

                  while Present (It.Nam) loop

                     if In_Open_Scopes (It.Nam) then
                        if Found then
                           Error_Msg_N (
                              "prefix must be unique enclosing scope", N);
                           Set_Entity (N, Any_Id);
                           Set_Etype  (N, Any_Type);
                           return;

                        else
                           Found := True;
                           P_Name := It.Nam;
                        end if;
                     end if;

                     Get_Next_Interp (I, It);
                  end loop;
               end;
            end if;

            if In_Open_Scopes (P_Name) then
               Set_Entity (P, P_Name);
               Set_Is_Overloaded (P, False);
               Find_Expanded_Name (N);

            else
               --  If no interpretation as an expanded name is possible, it
               --  must be a selected component of a record returned by a
               --  function call. Reformat prefix as a function call, the
               --  rest is done by type resolution.

               Change_Node (P, N_Function_Call);
            end if;

         --  Remaining cases generate various error messages

         else
            --  Format node as expanded name, to avoid cascaded errors

            Change_Node (N, N_Expanded_Name);
            Set_Prefix  (N, P);
            Set_Entity  (N, Any_Id);
            Set_Etype   (N, Any_Type);

            --  Set_Selector_Name (N, Empty); ????

            --  Issue error message, but avoid this if error issued already

            if P_Name = Any_Id  then
               null;

            elsif Ekind (P_Name) = E_Void then
               Error_Msg_N ("premature usage of&", P);

            else
               Error_Msg_N (
                "invalid prefix in selected component&", P);
            end if;
         end if;

      else
         --  If prefix is not the name of an entity, it must be an expression,
         --  whose type is appropriate for a record. This is determined by
         --  type resolution.

         Analyze_Selected_Component (N);
      end if;
   end Find_Selected_Component;

   ---------------
   -- Find_Type --
   ---------------

   procedure Find_Type (N : Node_Id) is
      C      : Entity_Id;
      T      : Entity_Id;
      T_Name : Entity_Id;

   begin
      if Nkind (N) = N_Attribute_Reference then

         --  Class attribute. This is only valid in Ada 9X mode, but we don't
         --  do a check, since the tagged type referenced could only exist if
         --  we were in 9X mode when it was declared (or, if we were in Ada
         --  83 mode, then an error message would already have been issued).

         if Attribute_Name (N) = Name_Class then
            Find_Type (Prefix (N));
            T := Base_Type (Entity (Prefix (N)));

            if not Is_Tagged_Type (T) then
               if Ekind (T) = E_Incomplete_Type then

                  --  It is legal to denote the class type of an incomplete
                  --  type. The full type will have to be tagged, of course.

                  Set_Is_Tagged_Type (T);
                  Make_Class_Wide_Type (T);
                  Set_Entity (N, Class_Wide_Type (T));
                  Set_Etype  (N, Class_Wide_Type (T));

               else
                  Error_Msg_N
                    ("type& is not a tagged type", Prefix (N));
                  Set_Entity (N, Any_Type);
               end if;

            else
               C := Class_Wide_Type (T);
               Set_Entity_With_Style_Check (N, C);
               Set_Etype (N, C);
            end if;

         --  Base attribute, allowed in Ada 9X mode only

         elsif Attribute_Name (N) = Name_Base then
            Note_Feature (Base_Attribute_In_Subtype_Mark, Sloc (N));

            if Ada_83 then
               Error_Msg_N
                 ("(Ada 83) Base attribute not allowed in subtype mark", N);

            else
               Find_Type (Prefix (N));
               T := Base_Type (Entity (Prefix (N)));
               Set_Entity (N, T);
               Set_Etype (N, T);
            end if;

         --  Huge_Integer attribute

         elsif Attribute_Name (N) = Name_Huge_Integer then
            Analyze (N);

         --  All other attributes are invalid in a subtype mark

         else
            Error_Msg_N ("invalid attribute in subtype mark", N);
         end if;

      else
         Analyze (N);
         T_Name := Entity (N);

         if T_Name  = Any_Id or else Etype (N) = Any_Type then

            --  Undefined id. Make it into a valid type

            Set_Entity (N, Any_Type);

         elsif not Is_Type (T_Name)
           and then T_Name /= Standard_Void_Type
         then
            Error_Msg_N ("subtype mark required in this context", N);
            Set_Entity (N, Any_Type);

         else
            T_Name := Get_Full_View (T_Name);

            if In_Open_Scopes (T_Name) then
               if Ekind (Base_Type (T_Name)) = E_Task_Type then
                  Error_Msg_N ("task type cannot be used as type mark " &
                     "within its own body", N);
               else
                  Error_Msg_N ("Type declaration cannot refer to itself", N);
               end if;
            end if;

            Set_Entity (N, T_Name);
         end if;
      end if;
   end Find_Type;

   -------------------
   -- Get_Full_View --
   -------------------

   function Get_Full_View (T_Name : Entity_Id) return Entity_Id is
   begin
      if (Ekind (T_Name) = E_Incomplete_Type
          and then Present (Full_View (T_Name)))
      then
         return Full_View (T_Name);

      elsif Is_Class_Wide_Type (T_Name)
        and then Ekind (Etype (T_Name)) = E_Incomplete_Type
        and then Present (Full_View (Etype (T_Name)))
      then
         return Class_Wide_Type (Full_View (Etype (T_Name)));

      else
         return T_Name;
      end if;
   end Get_Full_View;

   ---------------------------
   -- Has_Implicit_Operator --
   ---------------------------

   function Has_Implicit_Operator (N : Node_Id) return Boolean is
      Op_Id : constant Name_Id    := Chars (Selector_Name (N));
      P     : constant Entity_Id  := Entity (Prefix (N));
      Id    : Entity_Id;

   begin
      Id := First_Entity (P);

      --  Boolean operators: implicit declaration exists if scope contains
      --  declaration for derived boolean type, or for array of boolean type.

      if Op_Id = Name_Op_And
        or else Op_Id = Name_Op_Not
        or else Op_Id = Name_Op_Or
        or else Op_Id = Name_Op_Xor
      then
         while Present (Id) loop
            if Valid_Boolean_Arg (Id) then
               return true;
            end if;

            Id := Next_Entity (Id);
         end loop;

      --  Equality: look for any non-limited type.

      elsif Op_Id = Name_Op_Eq
        or else Op_Id = Name_Op_Ne
      then
         while Present (Id) loop
            if Is_Type (Id) and not Is_Limited_Type (Id) then
               return true;
            end if;

            Id := Next_Entity (Id);
         end loop;

      --  Comparison operators: scalar type, or array of scalar.

      elsif Op_Id = Name_Op_Lt
        or else Op_Id = Name_Op_Le
        or else Op_Id = Name_Op_Gt
        or else Op_Id = Name_Op_Ge
      then
         while Present (Id) loop
            if Is_Scalar_Type (Id)
              or else (Is_Array_Type (Id)
                        and then Is_Scalar_Type (Component_Type (Id)))
            then
               return true;
            end if;

            Id := Next_Entity (Id);
         end loop;

      --  Arithmetic operators: any numeric type

      elsif Op_Id = Name_Op_Abs
        or else Op_Id = Name_Op_Add
        or else Op_Id = Name_Op_Mod
        or else Op_Id = Name_Op_Rem
        or else Op_Id = Name_Op_Subtract
        or else Op_Id = Name_Op_Multiply
        or else Op_Id = Name_Op_Divide
        or else Op_Id = Name_Op_Expon
      then
         while Present (Id) loop
            if Is_Numeric_Type (Id) then
               return True;
            end if;

            Id := Next_Entity (Id);
         end loop;

      --  Concatenation: any one-dimensional array type

      elsif Op_Id = Name_Op_Concat then
         while Present (Id) loop
            if Is_Array_Type (Id) and then Number_Dimensions (Id) = 1 then
               return true;
            end if;

            Id := Next_Entity (Id);
         end loop;
      else
         return False;
      end if;

      return False;
   end Has_Implicit_Operator;

   ----------------
   -- Initialize --
   ----------------

   procedure Initialize is
   begin
      Urefs.Init;
   end Initialize;

   --------------------
   -- In_Open_Scopes --
   --------------------

   function In_Open_Scopes (S : Entity_Id) return Boolean is
   begin
      --  Since there are several scope stacks maintained by Scope_Stack each
      --  delineated by Standard (see comments by definition of Scope_Stack)
      --  it is necessary to end the search when Standard is reached.

      for J in reverse 0 .. Scope_Stack.Last loop
         if Scope_Stack.Table (J).Entity = S then
            return True;
         end if;

         exit when Scope_Stack.Table (J).Entity = Standard_Standard;
      end loop;

      return False;
   end In_Open_Scopes;

   -------------------------------------
   -- Is_Appropriate_For_Entry_Prefix --
   -------------------------------------

   function Is_Appropriate_For_Entry_Prefix (T : Entity_Id) return Boolean is
      P_Type : Entity_Id := T;

   begin
      if Is_Access_Type (P_Type) then
         P_Type := Designated_Type (P_Type);
      end if;

      return Is_Task_Type (P_Type) or else Is_Protected_Type (P_Type);
   end Is_Appropriate_For_Entry_Prefix;

   -------------------------------
   -- Is_Appropriate_For_Record --
   -------------------------------

   function Is_Appropriate_For_Record
     (T    : Entity_Id)
      return Boolean
   is
      function Has_Components (T1 : Entity_Id) return Boolean;
      --  Determine if given type has components (i.e. is either a record
      --  type or a type that has discriminants).

      function Has_Components (T1 : Entity_Id) return Boolean is
      begin
         return Is_Record_Type (T1)
           or else (Is_Private_Type (T1) and then Has_Discriminants (T1))
           or else (Is_Task_Type (T1) and then Has_Discriminants (T1));
      end Has_Components;

   --  Start of processing for Is_Appropriate_For_Record

   begin
      return
        Present (T)
          and then (Has_Components (T)
            or else (Is_Access_Type (T)
              and then Has_Components (Designated_Type (T))));
   end Is_Appropriate_For_Record;

   ---------------
   -- New_Scope --
   ---------------

   procedure New_Scope (S : Entity_Id) is
   begin
      Scope_Stack.Increment_Last;

      Scope_Stack.Table (Scope_Stack.Last).Entity := S;

      Scope_Stack.Table (Scope_Stack.Last).Save_Scope_Suppress  :=
        Scope_Suppress;

      Scope_Stack.Table (Scope_Stack.Last).Save_Entity_Suppress :=
        Entity_Suppress.Last;

      Scope_Stack.Table (Scope_Stack.Last).Is_Transient       := False;
      Scope_Stack.Table (Scope_Stack.Last).Uses_Sec_Stack     := False;
      Scope_Stack.Table (Scope_Stack.Last).Node_To_Be_Wrapped := Empty;

      if Debug_Flag_W then
         Write_Str ("--> new scope: ");
         Write_Name (Chars (Current_Scope));
         Write_Int (Int (Current_Scope));
         Write_Eol;
      end if;

   end New_Scope;

   ---------------
   -- Pop_Scope --
   ---------------

   procedure Pop_Scope is
      E : Entity_Id;

   begin
      if Debug_Flag_E then
         Write_Info;
      end if;

      Scope_Suppress :=
        Scope_Stack.Table (Scope_Stack.Last).Save_Scope_Suppress;

      while Entity_Suppress.Last >
                 Scope_Stack.Table (Scope_Stack.Last).Save_Entity_Suppress
      loop
         E := Entity_Suppress.Table (Entity_Suppress.Last).Entity;

         case Entity_Suppress.Table (Entity_Suppress.Last).Check is

            when Access_Check        =>
               Set_Suppress_Access_Checks        (E, False);

            when Accessibility_Check =>
               Set_Suppress_Accessibility_Checks (E, False);

            when Discriminant_Check  =>
               Set_Suppress_Discriminant_Checks  (E, False);

            when Division_Check      =>
               Set_Suppress_Division_Checks      (E, False);

            when Elaboration_Check   =>
               Set_Suppress_Elaboration_Checks   (E, False);

            when Index_Check         =>
               Set_Suppress_Index_Checks         (E, False);

            when Length_Check        =>
               Set_Suppress_Length_Checks        (E, False);

            when Overflow_Check      =>
               Set_Suppress_Overflow_Checks      (E, False);

            when Range_Check         =>
               Set_Suppress_Range_Checks         (E, False);

            when Storage_Check       =>
               Set_Suppress_Storage_Checks       (E, False);

            when Tag_Check           =>
               Set_Suppress_Tag_Checks           (E, False);

            --  All_Checks should not appear here (since it is entered as a
            --  series of its separate checks). Bomb if it is encountered

            when All_Checks =>
               pragma Assert (False); null;
         end case;

         Entity_Suppress.Decrement_Last;
      end loop;

      if Debug_Flag_W then
         Write_Str ("--> exiting scope: ");
         Write_Name (Chars (Current_Scope));
         Write_Eol;
      end if;

      Scope_Stack.Decrement_Last;
   end Pop_Scope;

   -------------------------
   -- Restore_Scope_Stack --
   -------------------------

   procedure Restore_Scope_Stack is
      E : Entity_Id;
      S : Entity_Id;

   begin
      --  Restore visibility of previous scope stack, if any.

      for J in reverse 0 .. Scope_Stack.Last loop
         exit when  Scope_Stack.Table (J).Entity = Standard_Standard
            or else No (Scope_Stack.Table (J).Entity);

         S := Scope_Stack.Table (J).Entity;
         Set_Is_Immediately_Visible (S, True);
         E := First_Entity (S);

         while Present (E) loop
            Set_Is_Immediately_Visible (E, True);
            E := Next_Entity (E);
         end loop;
      end loop;
   end Restore_Scope_Stack;

   ----------------------
   -- Save_Scope_Stack --
   ----------------------

   procedure Save_Scope_Stack is
      E : Entity_Id;
      S : Entity_Id;

   begin

      if Scope_Stack.Table (Scope_Stack.Last).Entity /= Standard_Standard then

         --  If the call is from within a compilation unit, as when
         --  called from Rtsfind,  make current entries in scope stack
         --  invisible while we analyze the new unit.

         for J in reverse 0 .. Scope_Stack.Last loop
            exit when  Scope_Stack.Table (J).Entity = Standard_Standard
               or else No (Scope_Stack.Table (J).Entity);

            S := Scope_Stack.Table (J).Entity;
            Set_Is_Immediately_Visible (S, False);
            E := First_Entity (S);

            while Present (E) loop
               Set_Is_Immediately_Visible (E, False);
               E := Next_Entity (E);
            end loop;
         end loop;

      end if;
   end Save_Scope_Stack;

   -------------
   -- Set_Use --
   -------------

   procedure Set_Use (L : List_Id) is
      Decl      : Node_Id;
      Pack_Name : Node_Id;
      Pack      : Entity_Id;
      Id        : Entity_Id;

   begin
      if Present (L) then
         Decl := First (L);

         while Present (Decl) loop
            if Nkind (Decl) = N_Use_Package_Clause then
               Pack_Name := First (Names (Decl));

               while Present (Pack_Name) loop
                  Pack := Entity (Pack_Name);

                  if Ekind (Pack) = E_Package then
                     if not In_Use (Pack) then
                        Use_One_Package (Pack);

                     else
                        Set_Redundant_Use (Pack_Name, True);
                     end if;
                  end if;

                  Pack_Name := Next (Pack_Name);
               end loop;

            elsif Nkind (Decl) = N_Use_Type_Clause  then
               Id := First (Subtype_Marks (Decl));
               while Present (Id) loop
                  if Entity (Id) /= Any_Type then
                     Set_Is_Potentially_Use_Visible (Entity (Id),  True);

                     --  If Id is a first-named subtype, indicate that
                     --  anonymous parent is also potentially use visible,
                     --  so that operators become accessible.

                     if not Comes_From_Source (Base_Type (Entity (Id))) then
                        Set_Is_Potentially_Use_Visible
                          (Base_Type (Entity (Id)));
                     end if;

                  end if;

                  Id := Next (Id);
               end loop;
            end if;

            Decl := Next (Decl);
         end loop;
      end if;

   end Set_Use;

   ---------------------
   -- Use_One_Package --
   ---------------------

   procedure Use_One_Package (P : Entity_Id) is
      Id   : Entity_Id;
      Prev : Entity_Id;

   begin
      Set_In_Use (P);

      --  If unit is a package renaming, indicate that the renamed
      --  package is also in use (the flags on both entities must
      --  remain consistent, and a subsequent use of either of them
      --  should be recognized as redundant).

      if Present (Renamed_Object (P)) then
         Set_In_Use (Renamed_Object (P));
      end if;

      --  Loop through entities in one package making them potentially
      --  use-visible.

      Id := First_Entity (P);
      while Present (Id)
        and then Id /= First_Private_Entity (P)
      loop
         Prev := Current_Entity (Id);

         while Present (Prev) loop
            if Is_Immediately_Visible (Prev)
              and then (not Is_Overloadable (Prev)
              or else not Is_Overloadable (Id)
              or else (Type_Conformant (Id, Prev)))
            then
               --  Potentially use-visible entity remains hidden

               goto Next_Usable_Entity;
            end if;
            Prev := Homonym (Prev);
         end loop;

         --  On exit, we know entity is not hidden, unless it is private.

         if not Is_Private (Id) then
            Set_Is_Potentially_Use_Visible (Id);
         end if;

         <<Next_Usable_Entity>> Id := Next_Entity (Id);
      end loop;
   end Use_One_Package;

   --------
   -- Wh --
   --------

   procedure Wh (N : Node_Id) is
      E : Entity_Id := Current_Entity (N);

   begin
      while Present (E) loop
         Write_Entity_Info (E, " --- ");
         Write_Str (" *** ");
         E := Homonym (E);
      end loop;
   end Wh;

   ----------------
   -- Write_Info --
   ----------------

   procedure Write_Info is
      Id : Entity_Id := First_Entity (Current_Scope);

   begin
      --  No point in dumping standard entities

      if Current_Scope = Standard_Standard then
         return;
      end if;

      Write_Str ("========================================================");
      Write_Eol;
      Write_Str ("        Defined Entities in ");
      Write_Name (Chars (Current_Scope));
      Write_Eol;
      Write_Str ("========================================================");
      Write_Eol;

      if No (Id) then
         Write_Str ("-- none --");
         Write_Eol;

      else
         while Present (Id) loop
            Write_Entity_Info (Id, " ");
            Id := Next_Entity (Id);
         end loop;
      end if;

      if Scope (Current_Scope) = Standard_Standard then

         --  Print information on the current unit itself

         Write_Entity_Info (Current_Scope, " ");
      end if;

      Write_Eol;
   end Write_Info;

   --------
   -- Ws --
   --------

   procedure Ws is
      S : Entity_Id;

   begin
      Write_Str ("Scope Stack");
      Write_Eol;

      for J in reverse 0 .. Scope_Stack.Last loop
         S := Scope_Stack.Table (J).Entity;
         Write_Str ("scope ");
         Write_Int (J);
         Write_Name (Chars (S));
         Write_Eol;
      end loop;
   end Ws;

end Sem_Ch8;


----------------------
-- REVISION HISTORY --
----------------------

--  ----------------------------
--  revision 1.191
--  date: Fri Aug 26 14:42:15 1994;  author: schonber
--  (Analyze_Selected_Component): the rule that the prefix must be a unique
--   enclosing scope (if not a package) is a resolution rule.
--  ----------------------------
--  revision 1.192
--  date: Mon Aug 29 12:32:19 1994;  author: schonber
--  (Analyze_Use_Type, End_Use_Type): make all primitive operators of the
--   type potentially use-visible (not just the predefined operators).
--  ----------------------------
--  revision 1.193
--  date: Mon Aug 29 23:43:00 1994;  author: dewar
--  Minor reformatting
--  ----------------------------
--  New changes after this line.  Each line starts with: "--  "
