------------------------------------------------------------------------------
--                                                                          --
--                         GNAT COMPILER COMPONENTS                         --
--                                                                          --
--                              S E M _ C H 3                               --
--                                                                          --
--                                 B o d y                                  --
--                                                                          --
--                            $Revision: 1.442 $                            --
--                                                                          --
--           Copyright (c) 1992,1993,1994 NYU, All Rights Reserved          --
--                                                                          --
-- GNAT is free software;  you can  redistribute it  and/or modify it under --
-- terms of the  GNU General Public License as published  by the Free Soft- --
-- ware  Foundation;  either version 2,  or (at your option) any later ver- --
-- sion.  GNAT is distributed in the hope that it will be useful, but WITH- --
-- OUT ANY WARRANTY;  without even the  implied warranty of MERCHANTABILITY --
-- or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License --
-- for  more details.  You should have  received  a copy of the GNU General --
-- Public License  distributed with GNAT;  see file COPYING.  If not, write --
-- to the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA. --
--                                                                          --
------------------------------------------------------------------------------

with Atree;    use Atree;
with Elists;   use Elists;
with Einfo;    use Einfo;
with Errout;   use Errout;
with Exp_Ch3;  use Exp_Ch3;
with Exp_Util; use Exp_Util;
with Features; use Features;
with Itypes;   use Itypes;
with Lib;      use Lib;
with Namet;    use Namet;
with Nlists;   use Nlists;
with Nmake;    use Nmake;
with Opt;      use Opt;
with Output;   use Output;
with Sem;      use Sem;
with Sem_Ch5;  use Sem_Ch5;
with Sem_Ch6;  use Sem_Ch6;
with Sem_Ch7;  use Sem_Ch7;
with Sem_Ch8;  use Sem_Ch8;
with Sem_Ch12; use Sem_Ch12;
with Sem_Ch13; use Sem_Ch13;
with Sem_Eval; use Sem_Eval;
with Sem_Res;  use Sem_Res;
with Sem_Type; use Sem_Type;
with Sem_Util; use Sem_Util;
with Stand;    use Stand;
with Sinfo;    use Sinfo;
with Snames;   use Snames;
with Tbuild;   use Tbuild;
with Ttypes;   use Ttypes;
with Uintp;    use Uintp;
with Urealp;   use Urealp;

package body Sem_Ch3 is

   -----------------------
   -- Local Subprograms --
   -----------------------

   procedure Build_Derived_Type
     (N            : Node_Id;
      Parent_Type  : Entity_Id;
      Derived_Type : Entity_Id);
   --  The attributes of a derived type are a copy of the attributes of
   --  the parent type. In some cases, additional entities (copies of
   --  components of the parent type) must also be created.

   procedure Build_Derived_Enumeration_Type
     (N            : Node_Id;
      Parent_Type  : Entity_Id;
      Derived_Type : Entity_Id);
   --  Subsidiary procedure to precious one. For a derived enumeration type, we
   --  must create a new list of literals. Types derived from Character are
   --  special-cased.

   procedure Build_Derived_Numeric_Type
     (N            : Node_Id;
      Parent_Type  : Entity_Id;
      Derived_Type : Entity_Id);
   --  Susbsidiary procedude to Build_Derived_Type. For numeric types, create
   --  an anonymous base type, and propagate constraint to subtype if needed.

   procedure Build_Derived_Record_Type
     (N            : Node_Id;
      Parent_Type  : Entity_Id;
      Derived_Type : Entity_Id);
   --  Subsidiary procedure to Build_derived_Type. For non tagged record types,
   --  copy the declaration of the parent, so that the derived type has its own
   --  declaration tree, discriminants, and possibly its own representation.

   procedure Derived_Standard_Character
     (N             : Node_Id;
      Parent_Type   : Entity_Id;
      Derived_Type  : Entity_Id);
   --  Subsidiary procedure to Build_Derived_Enumeration_Type which handles
   --  derivations from types Standard.Character and Standard.Wide_Character.

   procedure Build_Derived_Tagged_Type
     (N            : Node_Id;
      Type_Def     : Node_Id;
      Parent_Type  : Entity_Id;
      Derived_Type : Entity_Id);
   --  Used for building Tagged Extensions, either private or not. N is the
   --  type declaration node, Type_Def is the type definition node. For private
   --  extensions this is the same node.

   function Build_Discriminant_Constraints
     (T           : Entity_Id;
      Def         : Node_Id;
      Related_Nod : Node_Id)
      return        Elist_Id;
   --  Validate discriminant constraints, and build list of expressions in
   --  order of discriminant declarations. Used for subtypes and for derived
   --  types of record types.

   procedure Check_Or_Process_Discriminants (N : Node_Id; T : Entity_Id);
   --  If T is the full declaration of an incomplete or private type, check
   --  the conformance of the discriminants, otherwise process them.

   procedure Check_Digits_Expression (E : Node_Id; C : Node_Id);
   --  Check that the expression represented by E is suitable for use as
   --  a digits expression, i.e. it is of integer type and is static. Give
   --  appropriate error messages, posted at node C, if not.

   procedure Check_Incomplete (T : Entity_Id);
   --  Called to verify that an incomplete type is not used prematurely

   procedure Check_Initialization (T : Entity_Id; Exp : Node_Id);
   --  Validate the initialization of an object declaration. T is the
   --  required type, and Exp is the initialization expression.

   procedure Check_Non_Static_Default_Expr (L : List_Id);
   --  Iterate through the component list of a record definition, check
   --  that no component is declared with a non-static default value.

   procedure Constrain_Access
     (Def_Id      : in out Entity_Id;
      S           : Node_Id;
      Related_Nod : Node_Id);
   --  Apply a list of constraints to an access type. If Def_If is emtpy,
   --  it is an anonymous type created for a subtype indication. In that
   --  case it is created in the procedure and attached to Related_Nod.

   procedure Constrain_Array
     (Def_Id      : in out Entity_Id;
      SI          : Node_Id;
      Related_Nod : Node_Id;
      Related_Id  : Entity_Id;
      Suffix      : Character);
   --  Apply a list of index constraints to an unconstrained array type. The
   --  first parameter is the entity for the resulting subtype. A value of
   --  Empty for Def_Id indicates that an implicit type must be created, but
   --  creation is delayed (and must be done by this procedure) because other
   --  subsidiary implicit types must be created first (which is why Def_Id
   --  is an in/out parameter). Related_Nod gives the place where this type has
   --  to be inserted in the tree. The last two arguments are used to create
   --  its external name if needed.

   procedure Constrain_Concurrent
     (Def_Id      : in out Entity_Id;
      SI          : Node_Id;
      Related_Nod : Node_Id;
      Related_Id  : Entity_Id;
      Suffix      : Character);
   --  Apply list of discriminant constraints to an unconstrained concurrent
   --  type. The first parameter is the entity for the resulting subtype. A
   --  value of Empty for Def_Id indicates that an implicit type must be
   --  created, but creation is delayed (and must be done by this procedure)
   --  because other subsidiary implicit types must be created first (which is
   --  why Def_Id is an in/out parameter).  Related_Nod gives the place where
   --  this type has to be inserted in the tree. The last two arguments are
   --  used to create its external name if needed.

   procedure Constrain_Enumeration
     (Def_Id      : Node_Id;
      S           : Node_Id;
      Related_Nod : Node_Id);
   --  Constrain an enumeration type with a range constraint. This is
   --  identical to Constrain_Integer, but for the Ekind of the
   --  resulting subtype.

   procedure Constrain_Float
     (Def_Id      : Node_Id;
      S           : Node_Id;
      Related_Nod : Node_Id);
   --  Constrain a floating point type with either a digits constraint
   --  and/or a range constraint, building a E_Floating_Point_Subtype.

   procedure Constrain_Index
     (Index        : Node_Id;
      S            : Node_Id;
      Related_Nod  : Node_Id;
      Related_Id   : Entity_Id;
      Suffix       : Character;
      Suffix_Index : Nat);
   --  Process an index constraint in a constrained array declaration.
   --  The constraint can be a subtype name, or a range with or without
   --  an explicit subtype mark. The index is the corresponding index of the
   --  unconstrained array. The three last parameters are used to build the
   --  name for the implicit type that is created.

   procedure Constrain_Integer
     (Def_Id      : Node_Id;
      S           : Node_Id;
      Related_Nod : Node_Id);
   --  Build subtype of a signed or modular integer type.

   procedure Constrain_Ordinary_Fixed
     (Def_Id      : Node_Id;
      S           : Node_Id;
      Related_Nod : Node_Id);
   --  Constrain an ordinary fixed point type with a range constraint, and
   --  build an E_Ordinary_Fixed_Point_Subtype entity.

   procedure Constrain_Decimal
     (Def_Id      : Node_Id;
      S           : Node_Id;
      Related_Nod : Node_Id);
   --  Constrain a decimal fixed point type with a digits constraint and a
   --  range constraint if present, and build a E_Decimal_Fixed_Point_Subtype
   --  entity.

   procedure Constrain_Discriminated_Type
   (Def_Id      : Entity_Id;
    S           : Node_Id;
    Related_Nod : Node_Id);
   --  Process discriminant constraints of composite type. Verify that values
   --  have been provided for all discriminants, that the original type is
   --  unconstrained, and that the types of the supplied expressions match
   --  the discriminant types.

   procedure Constant_Redeclaration (Id : Entity_Id; N : Node_Id);
   --  Processes full declaration of deferred constant. Id is the entity for
   --  the redeclaration, and N is the N_Object_Declaration node. The caller
   --  has not done an Enter_Name or Set_Ekind on this entity.

   procedure Prepare_Private_Subtype_Completion
     (Id          : Entity_Id;
      Related_Nod : Node_Id);
   --  Id is a subtype of some private type. Creates the full declaration
   --  associated with Id whenever possible, ie when the full declaration of
   --  the base type is already known. Otherwise, records this entity into its
   --  base type Private_Subtype_List.

   procedure Copy_And_Swap (Privat, Full : Entity_Id);
   --  Copy the Privat entity into the entity of its full declaration
   --  then swap the 2 entities in such a manner that the former private
   --  type is now seen as a full type.

   procedure Derived_Type_Declaration (T : Entity_Id; N : Node_Id);
   --  Process derived type declaration

   function  Determine_Enum_Size (T : Entity_Id) return Uint;
   --  Determine the size in bits necessary to store enumeration literals
   --  of type "T". The sizes are rounded to 8, 16 or 32 bit quantites.
   --  This determination is made in the absence of representation clauses
   --  for the enumeration type.

   procedure Derive_Subprograms (Parent_Type, Derived_Type : Entity_Id);
   --  To complete type derivation, collect or retrieve the primitive
   --  operations of the parent type, and replace the subsidiary subtypes
   --  with the derived type, to build the specs of the inherited ops.

   procedure Discriminant_Redeclaration (T : Entity_Id; D_List : List_Id);
   --  Verify conformance of discriminant part on redeclarations of types

   procedure Enumeration_Type_Declaration (T : Entity_Id; Def : Node_Id);
   --  Insert each literal in symbol table, as an overloadable identifier
   --  Each enumeration type is mapped into a sequence of integers, and
   --  each literal is defined as a constant with integer value. If any
   --  of the literals are character literals, the type is a character
   --  type, which means that strings are legal aggregates for arrays of
   --  components of the type.

   procedure Expand_Others_Choice
     (Case_Table     : Case_Table_Type;
      Others_Choice  : Node_Id;
      Choice_Type    : Entity_Id);
   --  In the case of a variant part of a record type that has an OTHERS
   --  choice, this procedure expands the OTHERS into the actual choices
   --  that it represents. This new list of choice nodes is attached to
   --  the OTHERS node via the Others_Discrete_Choices field. The Case_Table
   --  contains all choices that have been given explicitly in the variant.

   function Find_Type_Of_Object
     (Obj_Def     : Node_Id;
      Related_Nod : Node_Id)
      return        Entity_Id;
   --  Get type entity for object referenced by Obj_Def, attaching the
   --  implicit types generated to Related_Nod

   procedure Floating_Point_Type_Declaration (T : Entity_Id; Def : Node_Id);
   --  Create a new float, and apply the constraint to obtain subtype of it

   function Inherit_Components
     (N            : Node_Id;
      Parent_Type  : Entity_Id;
      Derived_Type : Entity_Id)
      return Elist_Id;
   --  Used by derived types and type extensions to copy components of Parent.
   --  The returned value is an association list:
   --  (old_component => new_component).

   procedure Signed_Integer_Type_Declaration (T : Entity_Id; Def : Node_Id);
   --  Create a new signed integer entity, and apply the constraint to obtain
   --  the required first named subtype of this type.

   procedure Modular_Type_Declaration (T : Entity_Id; Def : Node_Id);
   --  Create new modular type. Verify that modulus is in  bounds and is
   --  a power of two (implementation restriction).

   function In_Visible_Part (Scope_Id : Entity_Id) return Boolean;
   --  Determine whether a declaration occurs within the visible part of a
   --  package specification. The package must be on the scope stack, and the
   --  corresponding private part must not.

   function Is_Null_Range (Lo : Node_Id; Hi : Node_Id) return Boolean;
   --  Predicate that determines if the expressions Lo and Hi represent a
   --  "Ada null range". The nodes passed are assumed to be static.

   procedure New_Binary_Operator (Op_Name : Name_Id; Typ : Entity_Id);
   --  Create an abbreviated declaration for an operator,  in order to
   --  materialize minimally operators on derived types.

   procedure Ordinary_Fixed_Point_Type_Declaration
     (T   : Entity_Id;
      Def : Node_Id);
   --  Create a new ordinary fixed point type, and apply the constrain to
   --  obtain subtype of it.

   procedure Decimal_Fixed_Point_Type_Declaration
     (T   : Entity_Id;
      Def : Node_Id);
   --  Create a new decimal fixed point type, and apply the constraint to
   --  obtain a subtype of this new type.

   procedure Process_Range_Expr_In_Decl
     (R           : Node_Id;
      T           : Entity_Id;
      Related_Nod : Node_Id);
   --  Process a range expression that appears in a declaration context. The
   --  range is analyzed and resolved with the base type of the given type,
   --  and an appropriate check for expressions in non-static contexts made
   --  on the bounds.

   procedure Process_Full_View (N : Node_Id; Full_T, Priv_T : Entity_Id);
   --  Process some semantic actions when the full view of a private type is
   --  encountered and analyzed. The first actions is to create the full views
   --  of the dependant private subtypes. The second action is to recopy the
   --  primitive operations of the private view (in the tagged case).

   procedure Record_Type_Declaration (T : Entity_Id; N : Node_Id);
   --  Process non-tagged record type declaration

   procedure Tagged_Record_Type_Declaration (T : Entity_Id; N : Node_Id);
   --  Process tagged record type declaration. T is the typ being defined,
   --  N is the declaration node.

   procedure Record_Type_Definition (Def : Node_Id; T : Entity_Id);
   --  Def is a record type definition node. This procedure analyzes the
   --  components in this record type definition. T is the entity for
   --  the enclosing type. It is provided so that its Has_Tasks flag
   --  can be set if any of the component have Has_Tasks set.

   function Static_Discriminant_Expr (L : List_Id) return Boolean;
   --  Iterate through the discriminant list, check if all discriminants
   --  have static default expr., return True if so.

   --------------------------
   -- Analyze_Declarations --
   --------------------------

   procedure Analyze_Declarations (L : List_Id) is
      D         : Node_Id;
      Next_Node : Node_Id;
      Frozen    : Boolean := False;

   begin
      D := First (L);
      while Present (D) loop

         --  Complete analysis of declaration

         Analyze (D);
         Next_Node := Next (D);

         if No (Next_Node) then

            --  At the end of a declarative part, freeze remaining entities
            --  declared in it.  The end of the visible declarations of a
            --  package specification is not the end of a declarative part if
            --  private declarations are present.

            if not Frozen
              and then Nkind (Parent (L)) /= N_Component_List
              and then (Nkind (Parent (L)) /= N_Package_Specification
                or else L /= Visible_Declarations (Parent (L))
                or else No (Private_Declarations (Parent (L)))
                or else Is_Empty_List (Private_Declarations (Parent (L))))
            then
               Frozen := True;           -- else it will be fluid for ever...
               Append_List (Freeze_All (Current_Scope), L);
            end if;

         elsif Nkind (Next_Node) /= N_Subprogram_Body
           and then Nkind (Next_Node) /= N_Entry_Body
           and then Nkind (Next_Node) /= N_Package_Body
           and then Nkind (Next_Node) /= N_Protected_Body
           and then Nkind (Next_Node) /= N_Task_Body
           and then Nkind (Next_Node) not in N_Body_Stub
         then
            null;

         --  Case of internally generated declaration

         elsif Analyzed (Next_Node) then
            null;

         --  A body that is not an instance nor an internally generated,
         --  freezes all entities so far.

         else
            Insert_List_After (D, Freeze_All (Current_Scope));
         end if;

         D := Next (D);
      end loop;

   end Analyze_Declarations;

   -----------------------------
   --  Analyze_Implicit_Types --
   -----------------------------

   --  Nothing to do, since the only descendent is the head of the list of
   --  itypes, and all itype entities were analyzed when the implicit types
   --  were constructed (this is the whole point of implicit types!)

   procedure Analyze_Implicit_Types (N : Node_Id) is
   begin
      null;
   end Analyze_Implicit_Types;

   --------------------------------
   -- Analyze_Object_Declaration --
   --------------------------------

   procedure Analyze_Object_Declaration (N : Node_Id) is
      Id         : constant Entity_Id := Defining_Identifier (N);
      E          : constant Node_Id   := Expression (N);
      Odf        : constant Node_Id   := Object_Definition (N);
      T          : Entity_Id;
      Itype_Node : Node_Id;

   begin
      if Constant_Present (N)
         and then Present (Current_Entity_In_Scope (Id))
      then
         Constant_Redeclaration (Id, N);

      --  In the normal case, enter identifiers at the start to catch
      --  premature usage in the initialization expression.

      else
         Enter_Name (Id);
      end if;

      --  There are three kinds of implicit types generated by an
      --  object declaration:

      --   1. those for generated by the original Object Definition

      --   2. those generated by the Expression

      --   3. those used to constrained the Object Definition with the
      --       expression constraints when it is unconstrained

      --   The first category is attached to a N_Implicit_Types Node just
      --   before the node. The second is attached to the expression the
      --   third is attached to the Object Declaration node itself

      Itype_Node := New_Node (N_Implicit_Types, Sloc (N));
      T := Find_Type_Of_Object (Odf, Itype_Node);

      if Present (First_Itype (Itype_Node)) then
         Insert_Before (N, Itype_Node);
      end if;

      --  If deferred constant, make sure context is appropriate

      if Constant_Present (N) and then No (E) then
         if (Ekind (Current_Scope) /= E_Package
              and then Ekind (Current_Scope) /= E_Generic_Package)
           or else In_Private_Part (Current_Scope)
         then
            Error_Msg_N
              ("invalid context for deferred constant declaration", N);
            Set_Constant_Present (N, False);

         --  In Ada 83, deferred constant must be of private type

         elsif not Is_Private_Type (T) then
            Note_Feature (Deferred_Constants_Of_Any_Type, Sloc (N));

            if Ada_83 then
               Error_Msg_N
                 ("(Ada 83) deferred constant must be private type", N);
            end if;
         end if;

      --  If not a deferred constant, then object declaration freezes its type

      else
         Freeze_Before (N, Base_Type (T));
      end if;

      --  Process initialization expression if present

      if Present (E) then
         Analyze (E);
         Check_Initialization (T, E);
         Check_Non_Static_Context (E);
         Apply_Range_Check (E, Etype (E), T);
      end if;

      --  Abstract type is never permitted for a variable or constant

      if Is_Abstract (T) then
         Error_Msg_N ("type of object cannot be abstract", Odf);

      --  Case of unconstrained type

      elsif Is_Indefinite_Subtype (T) then

         --  Nothing to do in deferred constant case

         if Constant_Present (N) and then No (E) then
            null;

         --  Otherwise must have an initialization

         elsif No (E) then
            if not Constant_Present (N) then
               Note_Feature (Unconstrained_Variables, Sloc (Odf));

               if Ada_83 then
                  Error_Msg_N
                    ("(Ada 83) unconstrained variable not allowed", Odf);
               end if;
            end if;

            if Is_Class_Wide_Type (T) then
               Error_Msg_N
                 ("initialization required in class-wide declaration ", N);
            else
               Error_Msg_N
                 ("unconstrained subtype not allowed (need initialization)",
                  Odf);
            end if;

         --  All OK, constrain the type with the expression size

         else
            Expand_Subtype_From_Expr (N, T, Odf, E);
            T := Find_Type_Of_Object (Object_Definition (N), N);
         end if;
      end if;

      --  Now establish the proper kind and type of the object.

      if Constant_Present (N) then
         Set_Ekind (Id, E_Constant);

      else
         Set_Ekind (Id, E_Variable);
         Check_Fully_Declared (T, N);
      end if;

      Set_Etype      (Id, T);
      Set_Is_Aliased (Id, Aliased_Present (N));

      --  Check that if we are in preelaborated elaboration code, then we
      --  do not have an instance of a default initialized private, task or
      --  protected object declaration which would violate (RM 10.2.1(9)).
      --  Note that constants are never default initialized (and the test
      --  below also filters out deferred constants). A variable is default
      --  initialized if it does *not* have an initialization expression.

      --  Filter out cases that are not declaration of a variable from source.

      if Nkind (N) /= N_Object_Declaration
        or else Constant_Present (N)
        or else not Comes_From_Source (Id)
      then
         return;
      end if;

      if Inside_Preelaborated_Unit (N)
        and then not Inside_Subprogram_Unit (N)
      then
         if No (E) then
            declare
               Ent : Entity_Id;

            begin
               --  Note: there is no need to test for controlled objects,
               --  since any unit declaring such objects must with (directly
               --  or indirectly) Ada.Finalization, which is not preelaborable
               --  so this case will be caught by the normal dependency test.

               --  Private Extension (of semantic analysis) is not implemented.
               --  What does this comment mean ???

               --  Object decl. that is of record type and has no default expr.
               --  should check if there is any non-static default expression
               --  in component decl. of the record type decl.

               if Is_Record_Type (T) then
                  if Nkind (Parent (Etype (Id))) = N_Full_Type_Declaration then
                     Check_Non_Static_Default_Expr (Component_Items
                       (Component_List (Type_Definition (Parent (Etype (Id)
                       )))));
                  end if;
               end if;

               --  ??? suspicious code, to be checked

               if Is_Private (Id)
                 or else
                   (Is_Access_Type (T)
                     and then
                       Depends_On_Private (Directly_Designated_Type (T)))
                 or else Depends_On_Private (T)
               then
                  Error_Msg_N
                    ("?private object not allowed in preelaborated unit", N);
                  return;

               --  Access to Task or Protected type

               elsif Nkind (Odf) = N_Identifier
                 and then Present (Etype (Odf))
                 and then Is_Access_Type (Etype (Odf))
               then
                  Ent := Directly_Designated_Type (Etype (Odf));

               elsif Nkind (Odf) = N_Identifier then
                  Ent := Entity (Odf);

               elsif Nkind (Odf) = N_Subtype_Indication then
                  Ent := Etype (Subtype_Mark (Odf));

               elsif
                  Nkind (Odf) = N_Constrained_Array_Definition
               then
                  Ent := Etype (Subtype_Indication (Odf));

               else
                  return;
               end if;

               if Is_Task_Type (Ent)
                 or else (Is_Protected_Type (Ent)
                           and then Number_Simple_Entries (Ent) /= 0)
               then
                  Error_Msg_N
                    ("?concurrent object not allowed in preelaborated unit",
                    N);
                  return;

               end if;
            end;
         end if;

         --  Evaluation of discriminant default expr. is done when obj.
         --  is created. And it has to be static expr.

         if Is_Record_Type (Etype (Id))
           and then Has_Discriminants (Etype (Id))
           and then Present (Etype (Etype (Id)))
           and then Nkind (Parent (Etype (Etype (Id)))) =
                                N_Full_Type_Declaration
           and then not Static_Discriminant_Expr (Discriminant_Specifications (
                Parent (Etype (Etype (Id)))))
         then
            Error_Msg_N
              ("?non-static discriminant default expr. in preelaborated unit"
              , Id);
         end if;
      end if;

      --  RM 10.2.1(16).
      --  A Pure library_item ... does not contain the declaration of any
      --  variable ... except within  a subprogram, generic subprogram, task
      --  unit or protected unit.

      if Inside_Pure_Unit (N)
        and then not Inside_Subpgm_Task_Protected_Unit (N)
      then
         Error_Msg_N ("?declaration of variable not allowed in pure unit", N);

      --  RM E.2.3(9).
      --  ... visible part of RCI lib. uniti, it shall not contain the decl.
      --  of a variable.

      elsif Inside_Remote_Call_Interface_Unit (N) then
         Error_Msg_N ("?declaration of variable not allowed in rci unit", N);
      end if;

   end Analyze_Object_Declaration;

   ----------------------------
   -- Constant_Redeclaration --
   ----------------------------

   procedure Constant_Redeclaration (Id : Entity_Id; N : Node_Id) is
      E    : constant Node_Id   := Expression (N);
      Prev : constant Entity_Id := Current_Entity_In_Scope (Id);
      T    : Entity_Id;

   begin
      T := Find_Type_Of_Object (Object_Definition (N), N);
      Freeze_Before (N, T);

      --  Case of a constant with a previous declaration that was either not
      --  a constant, or was a full constant declaration. In either case, it
      --  seems best to let Enter_Name treat it as an illegal duplicate decl.

      if Ekind (Prev) /= E_Constant
        or else Present (Expression (Parent (Prev)))
      then
         Enter_Name (Id);

      --  Case of full declaration of constant has wrong type

      elsif Etype (Prev) /= T then
         Error_Msg_Sloc := Sloc (Prev);
         Error_Msg_N ("type does not match declaration#", N);
         Set_Full_View (Prev, Id);
         Set_Etype (Id, Any_Type);

      --  Otherwise process the full constant declaration

      else
         Set_Full_View (Prev, Id);
         Set_Is_Public (Id, Is_Public (Prev));
         Set_Is_Internal (Id);
         Append_Entity (Id, Current_Scope);

         --  Check ALIASED present if present before (RM 7.4(7))

         if Is_Aliased (Prev)
           and then not Aliased_Present (N)
         then
            Error_Msg_Sloc := Sloc (Prev);
            Error_Msg_N ("ALIASED required (see declaration#)", N);
         end if;

         if Present (E) and then No (Etype (E)) then
            --  How can E be not present here ???

            Analyze (E);
            Check_Initialization (T, E);

            if Is_Indefinite_Subtype (T) then
               Expand_Subtype_From_Expr (N, T, Object_Definition (N), E);
               Set_Etype (Id, Find_Type_Of_Object (Object_Definition (N), N));
            end if;
         end if;
      end if;
   end Constant_Redeclaration;

   --------------------------------
   -- Analyze_Number_Declaration --
   --------------------------------

   procedure Analyze_Number_Declaration (N : Node_Id) is
      Id    : constant Entity_Id := Defining_Identifier (N);
      E     : constant Node_Id   := Expression (N);
      T     : Entity_Id;
      K     : Entity_Kind;
      Index : Interp_Index;
      It    : Interp;
      This  : Entity_Id;

   begin
      Analyze (E);

      --  Verify that the expression is static and numeric. If
      --  the expression is overloaded, we apply the preference
      --  rule that favors root numeric types.

      if not Is_Overloaded (E) then
         T := Etype (E);

      else
         T := Any_Type;
         Get_First_Interp (E, Index, It);

         while Present (It.Typ) loop
            if (Is_Integer_Type (It.Typ)
                 or else Is_Real_Type (It.Typ))
              and then (Scope (Base_Type (It.Typ))) = Standard_Standard
            then
               T := It.Typ;
               exit;
               --  could the expression ever be ambiguous ???
            end if;

            Get_Next_Interp (Index, It);
         end loop;
      end if;

      Enter_Name (Id);

      if Is_Integer_Type (T)  then
         Resolve (E, T);
         Set_Etype (Id, Universal_Integer);
         Set_Ekind (Id, E_Named_Integer);

      elsif Is_Real_Type (T) then
         Resolve (E, T);
         Set_Etype (Id, Universal_Real);
         Set_Ekind (Id, E_Named_Real);

      else
         Error_Msg_N ("numeric type required for number declaration", N);
         Set_Etype (Id, Any_Type);
         Set_Ekind (Id, E_Constant);
      end if;

      if Nkind (E) = N_Integer_Literal
        or else Nkind (E) = N_Real_Literal
      then
         Set_Etype (E, Etype (Id));
      end if;

      Check_Static_Expression (E);

   end Analyze_Number_Declaration;

   -------------------------
   -- Find_Type_Of_Object --
   -------------------------

   function Find_Type_Of_Object
     (Obj_Def     : Node_Id;
      Related_Nod : Node_Id)
      return        Entity_Id
   is
      T  : Entity_Id;

   begin
      if Nkind (Obj_Def) = N_Constrained_Array_Definition then

         --  Case of an anonymous array subtype

         T := Empty;
         Array_Type_Declaration (T, Obj_Def);

      else
         T := Process_Subtype (
                Obj_Def, Related_Nod,
                Defining_Identifier (Parent (Obj_Def)), 'S');
      end if;

      return T;
   end Find_Type_Of_Object;

   --------------------------------
   -- Analyze_Subtype_Indication --
   --------------------------------

   procedure Analyze_Subtype_Indication (N : Node_Id) is
      T : constant Node_Id := Subtype_Mark (N);
      R : constant Node_Id := Range_Expression (Constraint (N));

   begin
      Analyze (T);
      Analyze (R);
      Set_Etype (N, Etype (R));
   end Analyze_Subtype_Indication;

   -----------------------------
   -- Check_Digits_Expression --
   -----------------------------

   procedure Check_Digits_Expression (E : Node_Id; C : Node_Id) is
   begin
      if not (Is_Integer_Type (Etype (E))) then
         Error_Msg_N ("digits expression must be of integer type", C);
      end if;

      Check_Static_Expression (E);
   end Check_Digits_Expression;

   --------------------------
   -- Check_Initialization --
   --------------------------

   procedure Check_Initialization (T : Entity_Id; Exp : Node_Id) is
   begin
      if Is_Limited_Type (T) then
         Error_Msg_N
           ("cannot initialize entities of limited type", Exp);
      end if;

      Resolve (Exp, T);
      Check_Non_Static_Context (Exp);

   end Check_Initialization;

   -----------------------------------
   -- Check_Non_Static_Default_Expr --
   -----------------------------------

   procedure Check_Non_Static_Default_Expr (L : List_Id) is
      Component_Decl : Node_Id := First (L);
   begin
      while Present (Component_Decl) loop
         if Nkind (Component_Decl) = N_Component_Declaration
           and then Present (Expression (Component_Decl))
           and then not Potentially_Static (Expression (Component_Decl))
         then
            Error_Msg_N
              ("?non-static default expr in comp. decl. in preelaborated unit",
              Component_Decl);
         end if;
         Component_Decl := Next (Component_Decl);
      end loop;
   end Check_Non_Static_Default_Expr;

   ------------------------------
   -- Analyze_Type_Declaration --
   ------------------------------

   procedure Analyze_Type_Declaration (N : Node_Id) is
      Def    : constant Node_Id   := Type_Definition (N);
      Def_Id : constant Entity_Id := Defining_Identifier (N);
      T      : Node_Id;

   begin
      T := Find_Type_Name (N);

      --  Elaborate the type definition according to kind, and generate
      --  susbsidiary (implicit) subtypes where needed.

      case Nkind (Def) is

         when N_Access_To_Subprogram_Definition =>
            Access_Subprogram_Declaration (T, Def);

            --  RM 10.2.1(16)
            --  A Pure library_item ... does not contain the declaration
            --  of .. named access type, except within a subprogram, generic
            --  subprogram, task unit, or protected unit.

            if Comes_From_Source (T)
              and then Inside_Pure_Unit (N)
              and then not Inside_Subpgm_Task_Protected_Unit (N)
            then
               Error_Msg_N
                 ("?decl. of named access type not allowed in pure unit", N);
            end if;

         when N_Access_To_Object_Definition =>
            Access_Type_Declaration (T, Def);

            --  Check comments above in N_Access_To_Subprogram_Definition.

            if Comes_From_Source (T)
              and then Inside_Pure_Unit (N)
              and then not Inside_Subpgm_Task_Protected_Unit (N)
            then
               Error_Msg_N
                 ("?decl. of named access type not allowed in pure unit", N);
            end if;

         when N_Array_Type_Definition =>
            Array_Type_Declaration (T, Def);

         when N_Derived_Type_Definition =>
            Derived_Type_Declaration (T, N);

         when N_Enumeration_Type_Definition =>
            Enumeration_Type_Declaration (T, Def);

         when N_Floating_Point_Definition =>
            Floating_Point_Type_Declaration (T, Def);

         when N_Decimal_Fixed_Point_Definition =>
            Decimal_Fixed_Point_Type_Declaration (T, Def);

         when N_Ordinary_Fixed_Point_Definition =>
            Ordinary_Fixed_Point_Type_Declaration (T, Def);

         when N_Signed_Integer_Type_Definition =>
            Signed_Integer_Type_Declaration (T, Def);

         when N_Modular_Type_Definition =>
            Modular_Type_Declaration (T, Def);

         when N_Record_Definition =>
            if Tagged_Present (Def) then
               Tagged_Record_Type_Declaration (T, N);
            else
               Record_Type_Declaration (T, N);
            end if;

         when others =>
            pragma Assert (False); null;

      end case;

      --  Some common processing for all types

      Set_Depends_On_Private (T, Has_Private_Component (T));
      Set_Is_Delayed      (T, Depends_On_Private (T));

      --  Case of T is the full declaration of some private type which has
      --  been swapped in Defining_Identifier (N).

      if T /= Def_Id and then Is_Private_Type (Def_Id) then
         Process_Full_View (N, T, Def_Id);
      end if;
   end Analyze_Type_Declaration;

   -----------------------
   -- Process_Full_View --
   -----------------------

   procedure Process_Full_View (N : Node_Id; Full_T, Priv_T : Entity_Id) is
   begin

      --  Create a full declaration for all its subtypes recorded in
      --  Private_Subtype_List and swap them similarly to the base type.

      declare
         Priv : Entity_Id := Private_Subtype_List (Priv_T);
         Full : Entity_Id;

      begin

         while Present (Priv) loop
            Full := Make_Defining_Identifier (Sloc (Priv), Chars (Priv));
            Attach_Itype_To (N, Full);
            Copy_And_Swap (Priv, Full);
            Set_Full_View (Full, Priv);
            Set_Ekind (Priv, Subtype_Kind (Ekind (Full_T)));
            Set_Etype (Priv, Full_T);
            Set_Depends_On_Private (Priv, Has_Private_Component (Priv));
            Priv := Private_Subtype_List (Full);
         end loop;
      end;

      --  if the private view was tagged, copy the new Primitive
      --  operations from the private view to the full view.

      if Is_Tagged_Type (Priv_T) then
         declare
            Parent_List : constant Elist_Id :=
                            Primitive_Operations (Etype (Priv_T));
            Priv_List   : constant Elist_Id :=
                            Primitive_Operations (Priv_T);
            Full_List   : constant Elist_Id :=
                            Primitive_Operations (Full_T);

            Prim1 : Elmt_Id := First_Elmt (Parent_List);
            Prim2 : Elmt_Id := First_Elmt (Priv_List);

         begin
            --  Skip the parent primitive operations if the private type is
            --  an extension. (Usually the test for root types is T = Etype (T)
            --  but in this case the full-view and the private view ave been
            --  skipped hence the next test)

            if Full_View (Priv_T) /= Etype (Priv_T) then
               while Present (Prim1) loop
                  Prim1 := Next_Elmt (Prim1);
                  Prim2 := Next_Elmt (Prim2);
               end loop;
            end if;

            --  Copy the new primitives

            while Present (Prim2) loop
               Append_Elmt (Node (Prim2), Full_List);
               Prim2 := Next_Elmt (Prim2);
            end loop;

            --  Now the 2 views can share the same Primitive Operation list

            Set_Primitive_Operations (Priv_T, Full_List);
         end;
      end if;
   end Process_Full_View;

   -------------------
   -- Copy_And_Swap --
   -------------------

   procedure Copy_And_Swap (Privat, Full : Entity_Id) is
      Loc : constant Source_Ptr := Sloc (Full);
   begin
      --  Initialize new full declaration entity by copying all existing
      --  fields of the corresponding private declaration entity, and
      --  then resetting the private and limited flags which do not
      --  apply to the full declaration view of the type.

      Copy_Node (Privat, Full);
      Set_Sloc (Full, Loc);
      Set_Depends_On_Private (Full, False);

      --  Swap the two entities. Now Privat is the full type entity and
      --  Full is the private one. They will be swapped back at the end
      --  of the private part. This swapping ensures that the entity that
      --  is visible in the private part is the full declaration.

      Exchange_Entities (Privat, Full);
      Set_Full_View (Full, Privat);
      Append_Entity (Full, Current_Scope);
   end Copy_And_Swap;

   --------------------
   -- Find_Type_Name --
   --------------------

   function Find_Type_Name (N : Node_Id) return Entity_Id is
      Id     : constant Entity_Id := Defining_Identifier (N);
      Prev   : Entity_Id;
      New_Id : Entity_Id;

   begin
      --  Find incomplete declaration, if some was given.

      Prev := Current_Entity_In_Scope (Id);

      if Present (Prev) then

         --  Previous declaration exists. Error if not incomplete/private case

         if not Is_Incomplete_Or_Private_Type (Prev) then
            Error_Msg_Sloc := Sloc (Prev);
            Error_Msg_N ("invalid redeclaration of& declared#", Id);
            New_Id := Id;

         elsif Nkind (N) /= N_Full_Type_Declaration
           and then Nkind (N) /= N_Task_Type_Declaration
           and then Nkind (N) /= N_Protected_Type_Declaration
         then
            --  Completion must be a full type declarations (RM 7.3(4))

            Error_Msg_Sloc := Sloc (Prev);
            Error_Msg_N ("invalid completion of& declared#", Id);
            New_Id := Id;

         --  Case of full declaration of incomplete type

         elsif Ekind (Prev) = E_Incomplete_Type then

            --  Indicate that the incomplete declaration has a matching
            --  full declaration. The defining occurrence of the incomplete
            --  declaration remains the visible one, and the procedure
            --  Get_Full_View dereferences it whenever the type is used.

            Set_Full_View (Prev,  Id);
            Append_Entity (Id, Current_Scope);
            Set_Is_Public (Id, Is_Public (Prev));
            Set_Is_Internal (Id);
            New_Id := Id;

            if Nkind (N) = N_Full_Type_Declaration
              and then Nkind (Type_Definition (N)) =
                           N_Unconstrained_Array_Definition
            then
               Unimplemented
                 (N, "incomplete types completed with unconstrained arrays");
            end if;


         --  Case of full declaration of private type

         else
            if Nkind (Parent (Prev)) = N_Private_Extension_Declaration
              and then not (Abstract_Present (Parent (Prev)))
              and then Nkind (N) = N_Full_Type_Declaration
              and then Abstract_Present (Type_Definition (N))
            then
               Error_Msg_N
                 ("full view of non-abstract extension cannot be abstract", N);
            end if;

            Copy_And_Swap (Prev, Id);
            New_Id := Prev;
         end if;

         --  Verify that full declaration conforms to incomplete one

         if Present (Discriminant_Specifications (N))
           and then Is_Incomplete_Or_Private_Type (Prev)
         then
            Discriminant_Redeclaration (Prev, Discriminant_Specifications (N));

         elsif Is_Incomplete_Or_Private_Type (Prev)
           and then Has_Discriminants (Prev)
         then
            Error_Msg_N ("missing discriminants in full type declaration", N);

         elsif Is_Tagged_Type (Prev) then
            Note_Feature (Tagged_Types, Sloc (N));

            --  The full declaration is either a tagged record or an
            --  extension otherwise this is an error

            if Nkind (Type_Definition (N)) = N_Record_Definition
              and then Tagged_Present (Type_Definition (N))
            then
               null;

            elsif Nkind (Type_Definition (N)) = N_Derived_Type_Definition
              and then Present (Record_Extension_Part (Type_Definition (N)))
            then
               null;

            else
               Error_Msg_N ("full declaration of type& must be tagged", Id);
            end if;
         end if;

         return New_Id;

      else
         --  New type declaration

         Enter_Name (Id);
         return Id;
      end if;
   end Find_Type_Name;

   ---------------------
   -- Process_Subtype --
   ---------------------

   function Process_Subtype
     (S           : Node_Id;
      Related_Nod : Node_Id;
      Related_Id  : Entity_Id := Empty;
      Suffix      : Character := ' ')
      return        Entity_Id
   is
      P               : Node_Id;
      Def_Id          : Entity_Id;
      Subtype_Mark_Id : Entity_Id;
      N_Dynamic_Ityp  : Node_Id := Empty;

   begin
      --  Case of constraint present, so that we have an N_Subtype_Indication
      --  node (this node is created only if constraints are present).

      if Nkind (S) = N_Subtype_Indication then
         Find_Type (Subtype_Mark (S));
         P := Parent (S);
         Subtype_Mark_Id := Entity (Subtype_Mark (S));

         --  Explicit subtype declaration case

         if Nkind (P) = N_Subtype_Declaration then
            Def_Id := Defining_Identifier (P);

         --  Explicit derived type definition case

         elsif Nkind (P) = N_Derived_Type_Definition then
            Def_Id := Defining_Identifier (Parent (P));

         --  Implicit case, the Def_Id must be created as an implicit type.
         --  The one exception arises in the case of concurrent types,
         --  array and access types, where other subsidiary implicit types
         --  may be created and must appear before the main implicit type.
         --  In these cases we leave Def_Id set to Empty as a signal that the
         --  call to New_Itype has not yet been made to create Def_Id.

         else
            if Is_Array_Type (Subtype_Mark_Id)
              or else Is_Concurrent_Type (Subtype_Mark_Id)
              or else Is_Access_Type (Subtype_Mark_Id)
            then
               Def_Id := Empty;
            else
               Def_Id := New_Itype (E_Void, Related_Nod, Related_Id, Suffix);
            end if;

            --  only set Has_Dynamic_Itypes if the type is Implicit

            N_Dynamic_Ityp := Related_Nod;
         end if;

         --  Remaining processing depends on type

         case Ekind (Subtype_Mark_Id) is

            --  If the type is a access type, the constraint applies to the
            --  type being accessed. Create the corresponding subtype of it,
            --  promote it to an implicit type, and return an access to it.

            when Access_Kind =>
               Constrain_Access (Def_Id, S, Related_Nod);

            when Array_Kind =>
               Constrain_Array (Def_Id, S, Related_Nod, Related_Id, Suffix);

            when Decimal_Fixed_Point_Kind =>
               Constrain_Decimal (Def_Id, S, N_Dynamic_Ityp);

            when Enumeration_Kind =>
               Constrain_Enumeration (Def_Id, S, N_Dynamic_Ityp);

            when Ordinary_Fixed_Point_Kind =>
               Constrain_Ordinary_Fixed (Def_Id, S, N_Dynamic_Ityp);

            when Float_Kind =>
               Constrain_Float (Def_Id, S, N_Dynamic_Ityp);

            when Integer_Kind =>
               Constrain_Integer (Def_Id, S, N_Dynamic_Ityp);

            when E_Record_Type     |
                 E_Record_Subtype  |
                 Class_Wide_Kind   |
                 E_Incomplete_Type =>
               Constrain_Discriminated_Type (Def_Id, S, Related_Nod);

            when Private_Kind =>
               Constrain_Discriminated_Type (Def_Id, S, Related_Nod);
               Prepare_Private_Subtype_Completion (Def_Id, Related_Nod);

            when Concurrent_Kind  =>
               Constrain_Concurrent (Def_Id, S,
                 Related_Nod, Related_Id, Suffix);

            when others =>
               Error_Msg_N ("invalid subtype mark in subtype indication", S);
         end case;

         return Def_Id;

      --  Case of no constraints present

      else
         Find_Type (S);
         Check_Incomplete (S);
         return Entity (S);
      end if;
   end Process_Subtype;

   ----------------------
   -- Check_Incomplete --
   ----------------------

   procedure Check_Incomplete (T : Entity_Id) is
   begin
      if Ekind (Entity (T)) = E_Incomplete_Type then
         Error_Msg_N ("invalid use of type before its full declaration", T);
      end if;
   end Check_Incomplete;

   -----------------------
   --  Check_Completion --
   -----------------------

   procedure Check_Completion (Body_Id : Node_Id := Empty) is
      E : Entity_Id;

      procedure Post_Error is
      begin
         if No (Body_Id) then

            if not Comes_From_Source (E) then

            --  if a generated entity has no completion, then either previous
            --   semantic errors have disabled the expansion phase, or else
            --  something is very wrong.

               if Errors_Detected > 0 then
                  return;
               else
                  pragma Assert (False); null;
               end if;
            end if;

            --  Check on a declarative part: post error on the declaration
            --  that has no completion.

            if Is_Type (E) then
               Error_Msg_NE ("Missing full declaration for type &",
                  Parent (E), E);
            else
               Error_Msg_NE ("Missing body for &", Parent (E), E);
            end if;

         else
            --  Package body has no completion for a declaration that appears
            --  in the corresponding spec. Post error on the body, with a
            --  reference to the non-completed declaration. However, do not
            --  post the message if the item is internal, and we have any
            --  errors so far (otherwise it could easily be an artifact of
            --  expansion, which is turned off if any errors occur, e.g. in
            --  the case of a missing task body procedure, where expansion of
            --  the task body was suppressed because of other errors.

            if not Is_Internal (E)
              or else Errors_Detected = 0
            then
               Error_Msg_Sloc := Sloc (E);

               if Is_Type (E) then
                  Error_Msg_NE (
                     "Missing full declaration for type & declared#!",
                     Body_Id, E);
               else
                  Error_Msg_NE ("Missing body for & declared#!",
                     Body_Id, E);
               end if;
            end if;
         end if;
      end Post_Error;

   --  Start processing for Check_Completion

   begin
      E := First_Entity (Current_Scope);
      while Present (E) loop
         if Is_Internal (E) then
            null;

         --  The following situation requires special handling: a child
         --  unit that appears in the context clause of the body of its
         --  parent:

         --    procedure Parent.Child (...);
         --
         --    with Parent.Child;
         --    package body Parent is

         --  Here Parent.Child appears as a local entity, but should not
         --  be flagged as requiring completion, because it is a
         --  compilation unit.

         elsif Ekind (E) = E_Function
           or else Ekind (E) = E_Procedure then
            if not Has_Completion (E)
              and then not Is_Abstract (E)
              and then Nkind (Parent (Get_Declaration_Node (E)))
                                               /= N_Compilation_Unit
              and then Chars (E) /= Name_uSize
              and then Chars (E) /= Name_uEquality
            then
               Post_Error;
            end if;

         elsif Ekind (E) = E_Package then
            if Unit_Requires_Body (E) then
               if not Has_Completion (E)
                 and then Nkind (Parent (Get_Declaration_Node (E)))
                                                  /= N_Compilation_Unit
               then
                  Post_Error;
               end if;
            else
               May_Need_Implicit_Body (E);
            end if;

         elsif Ekind (E) = E_Incomplete_Type
           and then No (Underlying_Type (E))
         then
            Post_Error;

         elsif (Ekind (E) = E_Task_Type or else Ekind (E) = E_Protected_Type)
           and then not Has_Completion (E)
         then
            Post_Error;

         elsif Ekind (E) = E_Constant
           and then Ekind (Etype (E)) = E_Task_Type
           and then not Has_Completion (Etype (E))
         then
            Post_Error;

         elsif Ekind (E) = E_Protected_Object
           and then not Has_Completion (Etype (E))
         then
            Post_Error;

         end if;

         E := Next_Entity (E);
      end loop;
   end Check_Completion;

   ----------------------------------------
   -- Prepare_Private_Subtype_Completion --
   ----------------------------------------

   procedure Prepare_Private_Subtype_Completion
     (Id          : Entity_Id;
      Related_Nod : Node_Id)
   is
      Id_B   : constant Entity_Id := Base_Type (Id);
      Full_B : constant Entity_Id := Full_View (Id_B);
      Full   : Entity_Id;

   begin
      if Present (Full_B) then

         --  The Base_Type is already completed, we can complete the
         --  subtype now. We have to create a new entity with the same name,
         --  Thus we can't use New_Itype.

         Full := Make_Defining_Identifier (Sloc (Id), Chars (Id));
         Copy_Node (Id, Full);

         Set_Ekind            (Full, Subtype_Kind (Ekind (Full_B)));
         Set_Esize            (Full, Esize (Full_B));
         Set_Alignment_Clause (Full, Alignment_Clause (Full_B));

         --  Attach the full declaration to the beginning of the list of
         --  implicit types in order to be sure that it will appear before
         --  its private view

         Set_Next_Itype  (Full, First_Itype (Related_Nod));
         Set_First_Itype (Related_Nod, Full);

         --  Set semantic attributes for new (implicit) subtype

         case Ekind (Full) is

            when Array_Kind =>
               Set_First_Index       (Full, First_Index (Full_B));
               Set_Component_Type    (Full, Component_Type (Full_B));

            when Discrete_Kind =>
               Set_Scalar_Range      (Full, Scalar_Range (Full_B));
               Set_Parent_Subtype    (Full, Full_B);

               if Is_Modular_Integer_Type (Full) then
                  Set_Modulus (Full, Modulus (Full_B));

               elsif Is_Enumeration_Type (Full) then
                  Set_Lit_Name_Table  (Full, Lit_Name_Table (Full_B));
               end if;

            when Private_Kind     |
                 E_Record_Type    |
                 E_Record_Subtype |
                 Class_Wide_Kind  =>
               Set_Has_Discriminants (Full, Has_Discriminants (Full_B));
               Set_First_Entity      (Full, First_Entity (Full_B));
               Set_Last_Entity       (Full, Last_Entity (Full_B));

            when Access_Kind =>
               Set_Directly_Designated_Type
                                     (Full, Designated_Type (Full_B));

            when Task_Kind =>
               null;  --  All attributes already copied.

            when others =>
               pragma Assert (False); null;
         end case;

         Set_Depends_On_Private (Full, Has_Private_Component (Full));
         Set_Etype (Full, Full_B);

      else
         --  The base type full declaration has not yet been reached
         --  The new subtype is attached to the list of private subtypes,
         --  it will be completed after the base type full declaration.

         Set_Private_Subtype_List (Id, Private_Subtype_List (Id_B));
         Set_Private_Subtype_List (Id_B, Id);
      end if;
   end Prepare_Private_Subtype_Completion;

   ---------------------------------
   -- Analyze_Subtype_Declaration --
   ---------------------------------

   procedure Analyze_Subtype_Declaration (N : Node_Id) is
      Id : constant Entity_Id := Defining_Identifier (N);
      T  : Entity_Id;

   begin
      --  The following guard condition on Enter_Name is to handle cases
      --  where the defining identifier has already been entered into the
      --  scope but the the declaration as a whole needs to be analyzed.

      --  This case in particular happens for derived enumeration types.
      --  The derived enumeration type is processed as an inserted enumeration
      --  type declaration followed by a rewritten subtype declaration. The
      --  defining identifier, however, is entered into the name scope very
      --  early in the processing of the original type declaration and
      --  therefore needs to be avoided here, when the created subtype
      --  declaration is analyzed. (See Build_Derived_Types)
      --  This also happens when the full view of a private type is a
      --  derived type with constraints. In this case the entity has been
      --  introduced in the private declaration.

      if Present (Etype (Id))
        and then (Is_Private_Type (Etype (Id))
          or else Is_Rewrite_Substitution (N))
      then
         null;

      else
         Enter_Name (Id);
      end if;

      T := Process_Subtype (Subtype_Indication (N), N, Id, 'P');

      --  Inherit common attributes.

      Set_Is_Generic_Type (Id, Is_Generic_Type (Base_Type (T)));

      --  In the case where there is no constraint given in the subtype
      --  indication, Process_Subtype just returns the Subtype_Mark,
      --  so its semantic attributes must be established here.

      if Nkind (Subtype_Indication (N)) /= N_Subtype_Indication then
         Set_Etype (Id, Base_Type (T));

         case Ekind (T) is
            when Array_Kind =>
               Set_Ekind                (Id, E_Array_Subtype);
               Set_First_Index          (Id, First_Index (T));
               Set_Component_Type       (Id, Component_Type (T));
               Set_Is_Constrained       (Id, Is_Constrained (T));
               Set_Esize                (Id, Uint_0);

            when Decimal_Fixed_Point_Kind =>
               Set_Ekind                (Id, E_Decimal_Fixed_Point_Subtype);
               Set_Esize                (Id, Esize (T));
               Set_Parent_Subtype       (Id, T);
               Set_Digits_Value         (Id, Digits_Value (T));
               Set_Delta_Value          (Id, Delta_Value (T));
               Set_Small_Value          (Id, Small_Value (T));
               Set_Scalar_Range         (Id, Scalar_Range (T));
               Set_Corresponding_Integer_Type
                 (Id, Corresponding_Integer_Type (T));

            when Enumeration_Kind =>
               Set_First_Literal        (Id, First_Literal (Base_Type (T)));
               Set_Ekind                (Id, E_Enumeration_Subtype);
               Set_Lit_Name_Table       (Id, Lit_Name_Table (T));
               Set_Scalar_Range         (Id, Scalar_Range (T));
               Set_Parent_Subtype       (Id, T);
               Set_Esize                (Id, Esize (T));
               Set_Is_Character_Type    (Id, Is_Character_Type (T));
               Set_Corresponding_Integer_Type
                 (Id, Corresponding_Integer_Type (T));

            when Ordinary_Fixed_Point_Kind =>
               Set_Ekind                (Id, E_Ordinary_Fixed_Point_Subtype);
               Set_Scalar_Range         (Id, Scalar_Range (T));
               Set_Parent_Subtype       (Id, T);
               Set_Esize                (Id, Esize (T));
               Set_Small_Value          (Id, Small_Value (T));
               Set_Delta_Value          (Id, Delta_Value (T));
               Set_Corresponding_Integer_Type
                 (Id, Corresponding_Integer_Type (T));

            when Float_Kind =>
               Set_Ekind                (Id, E_Floating_Point_Subtype);
               Set_Scalar_Range         (Id, Scalar_Range (T));
               Set_Parent_Subtype       (Id, T);
               Set_Esize                (Id, Esize (T));
               Set_Digits_Value         (Id, Digits_Value (T));

            when Signed_Integer_Kind =>
               Set_Ekind                (Id, E_Signed_Integer_Subtype);
               Set_Scalar_Range         (Id, Scalar_Range (T));
               Set_Parent_Subtype       (Id, T);
               Set_Esize                (Id, Esize (T));

            when Modular_Integer_Kind =>
               Set_Ekind                (Id, E_Modular_Integer_Subtype);
               Set_Scalar_Range         (Id, Scalar_Range (T));
               Set_Parent_Subtype       (Id, T);
               Set_Esize                (Id, Esize (T));
               Set_Modulus              (Id, Modulus (T));
               Set_Non_Binary_Modulus   (Id, Non_Binary_Modulus (T));

            when Class_Wide_Kind =>
               Note_Feature (Class_Wide_Types, Sloc (Id));
               Set_First_Entity         (Id, First_Entity (T));
               Set_Last_Entity          (Id, Last_Entity (T));
               Set_Esize                (Id, Uint_0);
               Set_Is_Tagged_Type       (Id, True);
               Set_Ekind                (Id, E_Class_Wide_Subtype);
               Set_Etype                (Id, Etype (T));

               if Ekind (T) = E_Class_Wide_Subtype then
                  Set_Equivalent_Type   (Id, Equivalent_Type (T));
               end if;


            when E_Record_Type | E_Record_Subtype =>
               Set_First_Entity         (Id, First_Entity (T));
               Set_Last_Entity          (Id, Last_Entity (T));
               Set_Esize                (Id, Uint_0);
               Set_Is_Tagged_Type       (Id, Is_Tagged_Type (T));
               Set_Ekind                (Id, E_Record_Subtype);
               Set_Has_Discriminants    (Id, Has_Discriminants (T));
               Set_Is_Constrained       (Id, Is_Constrained (T));

               if Has_Discriminants (T) then
                  Set_Discriminant_Constraint (Id,
                    Discriminant_Constraint (T));
               end if;

               if Is_Tagged_Type (T) then
                  Set_Primitive_Operations (Id, Primitive_Operations (T));
               end if;

            when Private_Kind =>
               Set_Ekind             (Id, Subtype_Kind (Ekind (T)));
               Set_Esize             (Id, Uint_0);
               Set_Has_Discriminants (Id, Has_Discriminants (T));
               Set_Is_Constrained    (Id, Is_Constrained (T));
               Set_First_Entity      (Id, First_Entity (T));
               Set_Last_Entity       (Id, Last_Entity (T));

               if Has_Discriminants (T) then
                  Set_Discriminant_Constraint
                    (Id, Discriminant_Constraint (T));
               end if;

               Prepare_Private_Subtype_Completion (Id, N);

               --  ??? special stuff for e_record_(sub)type_with_private

            when Access_Kind =>
               Set_Ekind             (Id, E_Access_Subtype);
               Set_Directly_Designated_Type
                                     (Id, Designated_Type (T));
               Set_Esize             (Id, UI_From_Int (System_Address_Size));

               --  RM 10.2.1(16)
               --  A Pure library_item ... does not contain the declaration
               --  of .. named access type, except within a subprogram, generic
               --  subprogram, task unit, or protected unit.

               if Comes_From_Source (Id)
                 and then Inside_Pure_Unit (N)
                 and then not Inside_Subpgm_Task_Protected_Unit (N)
               then
                  Error_Msg_N
                    ("?decl. of named access type not allowed in pure unit"
                    , N);
               end if;

            when Task_Kind =>
               Copy_Node (T, Id);
               Set_Ekind             (Id,  E_Task_Subtype);

            when others =>
               pragma Assert (False); null;
         end case;

         --  Some processing common to all types

         Set_Alignment_Clause (Id, Alignment_Clause (T));
      end if;

      T := Etype (Id);

      Set_Is_Immediately_Visible (Id, True);
      Set_Depends_On_Private     (Id, Has_Private_Component (T));

      if Is_Private_Type (T)
        and then Present (Full_View (T))
      then
         Set_Is_Delayed (Id, Is_Delayed (Full_View (T))
                               and then not Is_Frozen (Full_View (T)));
      else
         Set_Is_Delayed (Id, Is_Delayed (T) and then not Is_Frozen (T));
      end if;

      Set_Has_Tasks      (Id, Has_Tasks (T));
      Set_Has_Controlled (Id, Has_Controlled (T));
      Set_Is_Controlled  (Id, Is_Controlled (T));

      if Has_Controlled (Id) then
         Note_Feature (Controlled_Types, Sloc (Id));
      end if;
   end Analyze_Subtype_Declaration;

   ----------------------
   -- Constrain_Float --
   ----------------------

   procedure Constrain_Float
     (Def_Id      : Node_Id;
      S           : Node_Id;
      Related_Nod : Node_Id)
   is
      T : constant Node_Id := Entity (Subtype_Mark (S));
      C : Node_Id;
      R : Node_Id;
      D : Node_Id;

   begin
      Set_Ekind (Def_Id, E_Floating_Point_Subtype);
      C := Constraint (S);

      --  Digits constraint present

      if Nkind (C) = N_Digits_Constraint then
         D := Digits_Expression (C);
         Analyze (D);
         Resolve (D, Any_Integer);
         Check_Digits_Expression (D, C);
         Set_Digits_Value (Def_Id, Expr_Value (D));

         --  Check that digits value is in range. Note that strictly this is
         --  runtime semantics and should raise constraint error, but that
         --  seems silly since obviously this is always known statically.

         if UI_Gt (Digits_Value (Def_Id), Digits_Value (T)) then
            Error_Msg_Uint_1 := Digits_Value (T);
            Error_Msg_N ("digits value is too large, max here = ^", D);
         end if;

         C := Range_Constraint (C);

      --  No digits constraint present

      else
         Set_Digits_Value (Def_Id, Digits_Value (T));
      end if;

      --  Range constraint present

      if Nkind (C) = N_Range_Constraint then
         R := Range_Expression (C);
         Process_Range_Expr_In_Decl (R, T, Related_Nod);
         Set_Scalar_Range (Def_Id, R);

      --  No range constraint present

      elsif No (C) then
         Set_Scalar_Range (Def_Id, Scalar_Range (T));

      --  Some incorrect constraint present

      else
         Error_Msg_N ("expect digits or range constraint for float type", C);
         Set_Etype (Def_Id, Any_Type);
         return;
      end if;

      Set_Etype            (Def_Id, Base_Type (T));
      Set_Parent_Subtype   (Def_Id, T);
      Set_Esize            (Def_Id, Esize (T));
      Set_Alignment_Clause (Def_Id, Alignment_Clause (T));
   end Constrain_Float;

   -----------------------
   -- Constrain_Decimal --
   -----------------------

   procedure Constrain_Decimal
     (Def_Id      : Node_Id;
      S           : Node_Id;
      Related_Nod : Node_Id)
   is
      T           : constant Entity_Id  := Entity (Subtype_Mark (S));
      Int_Type    : constant Entity_Id  := Corresponding_Integer_Type (T);
      C           : constant Node_Id    := Constraint (S);
      Loc         : constant Source_Ptr := Sloc (C);
      R           : constant Node_Id    := Range_Constraint (C);
      Digits_Expr : constant Node_Id    := Digits_Expression (C);
      Range_Val   : Node_Id;
      Digits_Val  : Uint;
      Bound_Val   : Uint;

   begin
      Set_Ekind (Def_Id, E_Decimal_Fixed_Point_Subtype);

      Analyze (Digits_Expr);
      Resolve (Digits_Expr, Any_Integer);

      if Nkind (C) /= N_Digits_Constraint then
         Error_Msg_N ("expect digits constraint for decimal type", C);
         Set_Etype (Def_Id, Any_Type);
         return;
      end if;

      Check_Digits_Expression (Digits_Expr, C);
      Digits_Val := Expr_Value (Digits_Expr);

      if UI_Gt (Digits_Val, Digits_Value (T)) then
         Error_Msg_N ("digits expression is incompatible with subtype", C);

      elsif UI_Le (Digits_Val, Uint_0) then
         Error_Msg_N ("digits expression must be positive", C);
      end if;

      Bound_Val :=
        UI_Difference (UI_Exponentiate (Uint_10, Digits_Val), Uint_1);

      Set_Etype                      (Def_Id, Base_Type (T));
      Set_Parent_Subtype             (Def_Id, T);
      Set_Esize                      (Def_Id, Esize (T));
      Set_Alignment_Clause           (Def_Id, Alignment_Clause (T));
      Set_Corresponding_Integer_Type (Def_Id, Int_Type);
      Set_Delta_Value                (Def_Id, Delta_Value (T));
      Set_Small_Value                (Def_Id, Small_Value (T));
      Set_Digits_Value               (Def_Id, Digits_Val);

      if Present (R) then
         declare
            Low  : constant Node_Id := Low_Bound (Range_Expression (R));
            High : constant Node_Id := High_Bound (Range_Expression (R));

         begin
            Check_Non_Static_Context (Low);
            Check_Non_Static_Context (High);

            Range_Val :=
              Make_Range (Loc,
                Low_Bound =>
                  Make_Type_Conversion (Loc,
                     Subtype_Mark => New_Reference_To (Int_Type, Loc),
                     Expression =>
                       Make_Op_Divide (Loc,
                         Left_Opnd => Relocate_Node (Low),
                         Right_Opnd =>
                           Make_Real_Literal (Loc, Delta_Value (T)))),

                High_Bound =>
                  Make_Type_Conversion (Loc,
                     Subtype_Mark => New_Reference_To (Int_Type, Loc),
                     Expression =>
                       Make_Op_Divide (Loc,
                         Left_Opnd => Relocate_Node (High),
                         Right_Opnd =>
                           Make_Real_Literal (Loc, Delta_Value (T)))));
         end;

      else
         Range_Val :=
            Make_Range (Loc,
               Low_Bound =>
                 Make_Qualified_Expression (Loc,
                    Subtype_Mark => New_Reference_To (Int_Type, Loc),
                    Expression =>
                      Make_Integer_Literal (Loc, UI_Negate (Bound_Val))),

               High_Bound =>
                 Make_Qualified_Expression (Loc,
                    Subtype_Mark => New_Reference_To (Int_Type, Loc),
                    Expression => Make_Integer_Literal (Loc, Bound_Val)));

      end if;

      Analyze (Range_Val);
      Resolve (Range_Val, Int_Type);
      Set_Scalar_Range (Def_Id, Range_Val);

   end Constrain_Decimal;

   ------------------------------
   -- Constrain_Ordinary_Fixed --
   ------------------------------

   procedure Constrain_Ordinary_Fixed
     (Def_Id      : Node_Id;
      S           : Node_Id;
      Related_Nod : Node_Id)
   is
      Loc       : constant Source_Ptr := Sloc (Def_Id);
      T         : constant Entity_Id  := Entity (Subtype_Mark (S));
      Int_Type  : constant Entity_Id  := Corresponding_Integer_Type (T);
      C         : constant Node_Id    := Constraint (S);
      Low       : Node_Id;
      High      : Node_Id;
      Range_Val : Node_Id;

   begin
      Set_Ekind (Def_Id, E_Ordinary_Fixed_Point_Subtype);

      if Nkind (C) /= N_Range_Constraint then
         Error_Msg_N
           ("expect range constraint for ordinary fixed point type", C);
         Set_Etype (Def_Id, Any_Type);
         return;
      end if;

      Set_Etype                      (Def_Id, Base_Type (T));
      Set_Parent_Subtype             (Def_Id, T);
      Set_Esize                      (Def_Id, Esize (T));
      Set_Alignment_Clause           (Def_Id, Alignment_Clause (T));
      Set_Corresponding_Integer_Type (Def_Id, Corresponding_Integer_Type (T));
      Set_Small_Value                (Def_Id, Small_Value (T));
      Set_Delta_Value                (Def_Id, Delta_Value (T));

      Range_Val :=
        Make_Range (Loc,

          Low_Bound =>
            Make_Type_Conversion (Loc,
              Subtype_Mark =>
                New_Reference_To (Int_Type, Loc),
              Expression =>
                Make_Op_Divide (Loc,
                  Left_Opnd =>
                    Relocate_Node (Low_Bound (Range_Expression (C))),
                  Right_Opnd => Make_Real_Literal (Loc, Small_Value (T)))),

          High_Bound =>
            Make_Type_Conversion (Loc,
              Subtype_Mark => New_Reference_To (Int_Type, Loc),
              Expression =>
                Make_Op_Divide (Loc,
                  Left_Opnd =>
                    Relocate_Node (High_Bound (Range_Expression (C))),
                  Right_Opnd => Make_Real_Literal (Loc, Small_Value (T)))));

      Analyze (Range_Val);
      Resolve (Range_Val, Int_Type);
      Set_Scalar_Range (Def_Id, Range_Val);

   end Constrain_Ordinary_Fixed;

   ---------------------------
   -- Constrain_Enumeration --
   ---------------------------

   procedure Constrain_Enumeration
     (Def_Id      : Node_Id;
      S           : Node_Id;
      Related_Nod : Node_Id)
   is
      T : constant Entity_Id := Entity (Subtype_Mark (S));
      C : constant Node_Id   := Constraint (S);
      R : Node_Id;

   begin
      Set_Ekind (Def_Id, E_Enumeration_Subtype);

      if Nkind (C) /= N_Range_Constraint then
         Error_Msg_N ("expect range constraint for enumeration type", C);
         Set_Etype (Def_Id, Any_Type);
         return;
      end if;

      R := Range_Expression (C);
      Process_Range_Expr_In_Decl (R, T, Related_Nod);

      Set_Scalar_Range      (Def_Id, R);
      Set_First_Literal     (Def_Id, First_Literal (Base_Type (T)));
      Set_Etype             (Def_Id, Base_Type (T));
      Set_Lit_Name_Table    (Def_Id, Lit_Name_Table (T));
      Set_Esize             (Def_Id, Esize (T));
      Set_Alignment_Clause  (Def_Id, Alignment_Clause (T));
      Set_Is_Character_Type (Def_Id, Is_Character_Type (T));
   end Constrain_Enumeration;

   -----------------------
   -- Constrain_Integer --
   -----------------------

   procedure Constrain_Integer
     (Def_Id      : Node_Id;
      S           : Node_Id;
      Related_Nod : Node_Id)
   is
      T : constant Node_Id := Entity (Subtype_Mark (S));
      C : constant Node_Id := Constraint (S);
      R : Node_Id;

   begin
      if Nkind (C) /= N_Range_Constraint then
         Error_Msg_N ("expect range constraint for integer type", C);
         Set_Etype (Def_Id, Any_Type);
         return;
      end if;

      R := Range_Expression (C);
      Process_Range_Expr_In_Decl (R, T, Related_Nod);

      if Is_Modular_Integer_Type (T) then
         Set_Ekind         (Def_Id, E_Modular_Integer_Subtype);
      else
         Set_Ekind         (Def_Id, E_Signed_Integer_Subtype);
      end if;

      Set_Scalar_Range     (Def_Id, R);
      Set_Etype            (Def_Id, Base_Type (T));
      Set_Parent_Subtype   (Def_Id, T);
      Set_Esize            (Def_Id, Esize (T));
      Set_Alignment_Clause (Def_Id, Alignment_Clause (T));
   end Constrain_Integer;

   -------------------------------------
   -- Floating_Point_Type_Declaration --
   -------------------------------------

   procedure Floating_Point_Type_Declaration (T : Entity_Id; Def : Node_Id) is
      Digs          : constant Node_Id := Digits_Expression (Def);
      Digs_Val      : Uint;
      Base_Type     : Entity_Id;
      Implicit_Base : constant Entity_Id :=
                        New_Itype
                          (E_Floating_Point_Type, Parent (Def), T, 'B');

      --  Find if given digits value allows derivation from specified type

      function Can_Derive_From (E : Entity_Id) return Boolean is
      begin
         return UI_Le
           (Digs_Val,
            Expr_Value (Digits_Expression (Type_Definition (Parent (E)))));
      end Can_Derive_From;

   --  Start of processing for Floating_Point_Type_Declaration

   begin
      Analyze (Digs);
      Resolve (Digs, Any_Integer);
      Check_Digits_Expression (Digs, Def);
      Digs_Val := Expr_Value (Digs);

      if Can_Derive_From (Standard_Short_Float) then
         Base_Type := Standard_Short_Float;
      elsif Can_Derive_From (Standard_Float) then
         Base_Type := Standard_Float;
      elsif Can_Derive_From (Standard_Long_Float) then
         Base_Type := Standard_Long_Float;
      elsif Can_Derive_From (Standard_Long_Long_Float) then
         Base_Type := Standard_Long_Long_Float;
      else
         Base_Type := Standard_Long_Long_Float;
         Error_Msg_N ("digits expression out of range", Def);
      end if;

      Set_Scalar_Range     (Implicit_Base, Scalar_Range (Base_Type));
      Set_Etype            (Implicit_Base, Base_Type);
      Set_Esize            (Implicit_Base, Esize (Base_Type));
      Set_Alignment_Clause (Implicit_Base, Alignment_Clause (Base_Type));
      Set_Digits_Value     (Implicit_Base, Digs_Val);

      Set_Ekind            (T, E_Floating_Point_Subtype);
      Set_Etype            (T, Implicit_Base);
      Set_Esize            (T, Esize (Implicit_Base));
      Set_Alignment_Clause (T, Alignment_Clause (Implicit_Base));
      Set_Parent_Subtype   (T, Implicit_Base);
      Set_Digits_Value     (T, Digs_Val);

      --  If there are bounds given in the declaration use them as the bounds
      --  of the type, otherwise use the bounds of the predefined base type
      --  that was chosen based on the Digits value.

      if Present (Real_Range_Specification (Def)) then
         Analyze (Low_Bound (Real_Range_Specification (Def)));
         Analyze (High_Bound (Real_Range_Specification (Def)));
         Resolve (Low_Bound (Real_Range_Specification (Def)), Implicit_Base);
         Resolve (High_Bound (Real_Range_Specification (Def)), Implicit_Base);
         Set_Scalar_Range (T, Real_Range_Specification (Def));
      else
         Set_Scalar_Range (T, Scalar_Range (Base_Type));
      end if;
   end Floating_Point_Type_Declaration;

   -------------------------------------------
   -- Ordinary_Fixed_Point_Type_Declaration --
   -------------------------------------------

   procedure Ordinary_Fixed_Point_Type_Declaration
     (T   : Entity_Id;
      Def : Node_Id)
   is
      Loc           : constant Source_Ptr := Sloc (Def);
      Delta_Expr    : constant Node_Id    := Delta_Expression (Def);
      Implicit_Base : constant Entity_Id  :=
                        New_Itype
                          (E_Ordinary_Fixed_Point_Type, Parent (Def), T, 'B');
      Low           : constant Node_Id    :=
                        Low_Bound (Real_Range_Specification (Def));
      High          : constant Node_Id    :=
                        High_Bound (Real_Range_Specification (Def));
      Small_Min     : constant Ureal      :=
                        UR_Exponentiate
                          (Ureal_2,
                           UI_Difference
                             (Uint_1,
                              UI_From_Int (Standard_Long_Long_Integer_Size)));

      Delta_Val  : Ureal;
      Base_Type  : Entity_Id;
      Base_Range : Node_Id;
      Type_Range : Node_Id;
      Maxi       : Uint;
      Small_Val  : Ureal;
      Low_Val    : Uint;
      High_Val   : Uint;
      Min_Size   : Uint;

      function Get_Power_2 (Real : Ureal) return Ureal;
      --  Get the power of 2 just below the value of Real

      function Get_Best_Small (Size_Given : Int) return Ureal;
      --  Get the optimum Small value in order to have the best precision
      --  though keeping the smallest size of integer to represent the new
      --  ordinary fixed point type.

      function Can_Derive_From (E : Entity_Id) return Boolean;
      --  Find if given delta exprssion and range specification allows
      --  derivation from the specified type.

      function Get_Power_2 (Real : Ureal) return Ureal is
         Tmp    : Ureal;
         Scale  : Uint := Uint_0;

      begin
         Tmp := Ureal_1;

         if UR_Lt (Real, Tmp) then
            while UR_Gt (Tmp, Real) loop
               Tmp := UR_Quotient (Tmp, Ureal_2);
               Scale := UI_Sum (Scale, Uint_1);
            end loop;

         else
            while UR_Le (Tmp, Real) loop
               Tmp := UR_Product (Tmp, Ureal_2);
               Scale := UI_Difference (Scale, Uint_1);
            end loop;

            Tmp := UR_Quotient (Tmp, Ureal_2);
            Scale := UI_Sum (Scale, Uint_1);
         end if;

         return UR_From_Components (Uint_1, Scale, 2);
      end Get_Power_2;

      function Get_Best_Small (Size_Given : Int) return Ureal is
         Factor : constant Ureal :=
                    Get_Power_2  (
                      UR_Quotient (
                        UR_Exponentiate (Ureal_2, UI_From_Int (Size_Given)),
                        UR_From_Uint (Min_Size)));
      begin
         if UR_Ge (Factor, UR_Quotient (Small_Val, Small_Min)) then
            return Small_Min;
         else
            return UR_Quotient (Small_Val, Factor);
         end if;
      end Get_Best_Small;

      function Can_Derive_From (E : Entity_Id) return boolean is
      begin
         return UI_Le
                  (Min_Size,
                   UI_Exponentiate (Uint_2, Esize (E)));
      end Can_Derive_From;

   --  Start of processing for Ordinary_Fixed_Point_Type_Declaration

   begin
      Analyze (Delta_Expr);
      Resolve (Delta_Expr, Universal_Real);

      if not (Is_Real_Type (Etype (Delta_Expr))) then
         Error_Msg_N ("delta expression must be of real type", Def);
      end if;

      Check_Static_Expression (Delta_Expr);
      Delta_Val := Expr_Value (Delta_Expr);

      if not (UI_Is_Positive (Numerator (Delta_Val))) then
         Error_Msg_N ("delta expression must be positive", Def);
         return;
      end if;

      Small_Val := Get_Power_2 (Delta_Val);

      if UR_Lt (Small_Val, Small_Min) then
         Error_Msg_N ("delta value must be greater than Fine_Delta", Def);
      end if;

      Analyze (Low);
      Analyze (High);
      Resolve (Low, Universal_Real);
      Resolve (High, Universal_Real);

      Check_Static_Expression (Low);
      Check_Static_Expression (High);
      Low_Val := UR_To_Uint (UR_Quotient (Expr_Value (Low), Small_Val));
      High_Val := UR_To_Uint (UR_Quotient (Expr_Value (High), Small_Val));
      Maxi := UI_Max (UI_Abs (Low_Val), UI_Abs (High_Val));

      Min_Size := UI_Product (UI_Sum (Maxi, Uint_1), Uint_2);

      if Can_Derive_From (Standard_Short_Short_Integer) then
         Small_Val := Get_Best_Small (Standard_Short_Short_Integer_Size);
         Base_Type := Standard_Short_Short_Integer;

      elsif Can_Derive_From (Standard_Short_Integer) then
         Small_Val := Get_Best_Small (Standard_Short_Integer_Size);
         Base_Type := Standard_Short_Integer;

      elsif Can_Derive_From (Standard_Integer) then
         Small_Val := Get_Best_Small (Standard_Integer_Size);
         Base_Type := Standard_Integer;

      elsif Can_Derive_From (Standard_Long_Integer) then
         Small_Val := Get_Best_Small (Standard_Long_Integer_Size);
         Base_Type := Standard_Long_Integer;

      elsif Can_Derive_From (Standard_Long_Long_Integer) then
         Small_Val := Get_Best_Small (Standard_Long_Long_Integer_Size);
         Base_Type := Standard_Long_Long_Integer;

      else
         Base_Type := Standard_Long_Long_Integer;
         Error_Msg_N (
           "delta expression with range specification is out of range", Def);
      end if;

      Low_Val := UR_To_Uint (UR_Quotient (Expr_Value (Low), Small_Val));
      High_Val := UR_To_Uint (UR_Quotient (Expr_Value (High), Small_Val));
      Maxi := UI_Max (UI_Abs (Low_Val), UI_Abs (High_Val));

      Set_Etype            (Implicit_Base, Implicit_Base);
      Set_Esize            (Implicit_Base, Esize (Base_Type));
      Set_Alignment_Clause (Implicit_Base, Alignment_Clause (Base_Type));
      Set_Small_Value      (Implicit_Base, Small_Val);
      Set_Delta_Value      (Implicit_Base, Delta_Val);
      Set_Corresponding_Integer_Type (Implicit_Base, Base_Type);

      Base_Range :=
        Make_Range (Loc,
          Low_Bound =>
            Make_Qualified_Expression (Loc,
               Subtype_Mark => New_Reference_To (Base_Type, Loc),
               Expression => Make_Integer_Literal (Loc, UI_Negate (Maxi))),
          High_Bound =>
            Make_Qualified_Expression (Loc,
               Subtype_Mark => New_Reference_To (Base_Type, Loc),
               Expression => Make_Integer_Literal (Loc, Maxi)));

      Analyze (Base_Range);
      Resolve (Base_Range, Base_Type);
      Set_Scalar_Range (Implicit_Base, Base_Range);

      Set_Ekind                      (T, E_Ordinary_Fixed_Point_Subtype);
      Set_Etype                      (T, Implicit_Base);
      Set_Esize                      (T, Esize (Implicit_Base));
      Set_Alignment_Clause           (T, Alignment_Clause (Implicit_Base));
      Set_Corresponding_Integer_Type (T, Base_Type);
      Set_Small_Value                (T, Small_Val);
      Set_Delta_Value                (T, Delta_Val);

      Type_Range :=
        Make_Range (Loc,
          Low_Bound =>
            Make_Qualified_Expression (Loc,
               Subtype_Mark => New_Reference_To (Base_Type, Loc),
               Expression => Make_Integer_Literal (Loc, Low_Val)),
          High_Bound =>
            Make_Qualified_Expression (Loc,
               Subtype_Mark => New_Reference_To (Base_Type, Loc),
               Expression => Make_Integer_Literal (Loc, High_Val)));

      Analyze (Type_Range);
      Resolve (Type_Range, Base_Type);
      Set_Scalar_Range (T, Type_Range);

   end Ordinary_Fixed_Point_Type_Declaration;

   ------------------------------------------
   -- Decimal_Fixed_Point_Type_Declaration --
   ------------------------------------------

   procedure  Decimal_Fixed_Point_Type_Declaration
     (T : Entity_Id;
      Def : Node_Id)
   is
      Loc           : constant Source_Ptr := Sloc (Def);
      Digs          : constant Node_Id    := Digits_Expression (Def);
      Delta_Expr    : constant Node_Id    := Delta_Expression (Def);
      Implicit_Base : constant Entity_Id  :=
                        New_Itype
                          (E_Decimal_Fixed_Point_Type, Parent (Def), T, 'B');
      Delta_Min     : constant Ureal      :=
                        UR_Exponentiate (
                          Ureal_2,
                          UI_Difference (
                            Uint_1,
                            UI_From_Int (Standard_Long_Long_Integer_Size)));

      Base_Range : Node_Id;
      Digs_Val   : Uint;
      Delta_Val  : Ureal;
      Base_Type  : Entity_Id;
      Bound_Val  : Uint;

      function Is_Power_10 (Real : Ureal) return Boolean;
      --  Check if the given delta expression is a power of ten.

      function Can_Derive_From (E : Entity_Id) return Boolean;
      --  Find if given digits value allows derivation from specified type

      function Is_Power_10 (Real : Ureal) return Boolean is
         Val : Ureal := Real;

      begin
         while (UR_Lt (Val, Ureal_1)) loop
            Val := UR_Product (Val, Ureal_10);
         end loop;

         while (UR_Gt (Val, Ureal_1)) loop
            Val := UR_Quotient (Val, Ureal_10);
         end loop;

         if UR_Eq (Val, Ureal_1) then
            return True;
         else
            return False;
         end if;
      end Is_Power_10;

      function Can_Derive_From (E : Entity_Id) return Boolean is
      begin
         return UI_Le
                  (UI_Product (Uint_2, Bound_Val),
                   UI_Exponentiate (Uint_2, Esize (E)));
      end Can_Derive_From;

   --  Start of processing for Decimal_Fixed_Point_Type_Declaration

   begin
      Analyze (Delta_Expr);
      Resolve (Delta_Expr, Universal_Real);

      if not Is_Static_Expression (Delta_Expr) then
         Check_Static_Expression (Delta_Expr);
         return;

      elsif not Is_Real_Type (Etype (Delta_Expr)) then
         Error_Msg_N ("delta expression must be of real type", Def);
         return;

      else
         Delta_Val := Expr_Value (Delta_Expr);

         if not UR_Is_Positive (Delta_Val) then
            Error_Msg_N ("delta expression must be positive", Def);
            return;

         else
            Delta_Val := Expr_Value (Delta_Expr);

            if not Is_Power_10 (Delta_Val) then
               Error_Msg_N ("delta expression must be a power of 10", Def);
               return;

            elsif UR_Lt (Delta_Val, Delta_Min) then
               Error_Msg_N ("delta must be greater than Fine_Delta", Def);
               return;
            end if;
         end if;
      end if;

      Analyze (Digs);
      Resolve (Digs, Any_Integer);
      Check_Digits_Expression (Digs, Def);
      Digs_Val := Expr_Value (Digs);

      if UI_Le (Digs_Val, Uint_0) then
         Error_Msg_N ("digits value mus be greater then 0", Def);
      end if;

      Bound_Val := UI_Difference (UI_Exponentiate (Uint_10, Digs_Val), Uint_1);

      if Can_Derive_From (Standard_Short_Short_Integer) then
         Base_Type := Standard_Short_Short_Integer;

      elsif Can_Derive_From (Standard_Short_Integer) then
         Base_Type := Standard_Short_Integer;

      elsif Can_Derive_From (Standard_Integer) then
         Base_Type := Standard_Integer;

      elsif Can_Derive_From (Standard_Long_Integer) then
         Base_Type := Standard_Long_Integer;

      elsif Can_Derive_From (Standard_Long_Long_Integer) then
         Base_Type := Standard_Long_Long_Integer;

      else
         Base_Type := Standard_Long_Long_Integer;
         Error_Msg_N ("digits expression out of range", Def);
      end if;

      Set_Etype            (Implicit_Base, Implicit_Base);
      Set_Esize            (Implicit_Base, Esize (Base_Type));
      Set_Alignment_Clause (Implicit_Base, Alignment_Clause (Base_Type));
      Set_Digits_Value     (Implicit_Base, Digs_Val);
      Set_Delta_Value      (Implicit_Base, Delta_Val);
      Set_Small_Value      (Implicit_Base, Delta_Val);
      Set_Corresponding_Integer_Type (Implicit_Base, Base_Type);

      Base_Range :=
        Make_Range (Loc,
          Low_Bound =>
            Make_Qualified_Expression (Loc,
               Subtype_Mark => New_Reference_To (Base_Type, Loc),
               Expression =>
                 Make_Integer_Literal (Loc, UI_Negate (Bound_Val))),
          High_Bound =>
            Make_Qualified_Expression (Loc,
               Subtype_Mark => New_Reference_To (Base_Type, Loc),
               Expression => Make_Integer_Literal (Loc, Bound_Val)));

      Analyze (Base_Range);
      Resolve (Base_Range, Base_Type);
      Set_Scalar_Range (Implicit_Base, Base_Range);

      Set_Ekind                      (T, E_Decimal_Fixed_Point_Subtype);
      Set_Etype                      (T, Implicit_Base);
      Set_Esize                      (T, Esize (Implicit_Base));
      Set_Alignment_Clause           (T, Alignment_Clause (Implicit_Base));
      Set_Corresponding_Integer_Type (T, Base_Type);
      Set_Parent_Subtype             (T, Implicit_Base);
      Set_Digits_Value               (T, Digs_Val);
      Set_Delta_Value                (T, Delta_Val);
      Set_Small_Value                (T, Delta_Val);
      --  If there are bounds given in the declaration use them as the bounds
      --  of the type, otherwise use the bounds of the predefined base type
      --  that was chosen based on the Digits value.

      if Present (Real_Range_Specification (Def)) then
         declare
            Low        : constant Node_Id :=
                           Low_Bound (Real_Range_Specification (Def));
            High       : constant Node_Id :=
                           High_Bound (Real_Range_Specification (Def));
            Type_Range : Node_Id;
            Low_Val    : Uint;
            High_Val   : Uint;

         begin

            Analyze (Low);
            Analyze (High);
            Resolve (Low, Universal_Real);
            Resolve (High, Universal_Real);

            Check_Static_Expression (Low);
            Check_Static_Expression (High);
            Low_Val :=
              UI_Max (
                UR_To_Uint (UR_Quotient (Expr_Value (Low), Delta_Val)),
                UI_Negate (Bound_Val));
            High_Val :=
              UI_Min (
                UR_To_Uint (UR_Quotient (Expr_Value (High), Delta_Val)),
                Bound_Val);

            Type_Range :=
              Make_Range (Loc,
                Low_Bound =>
                  Make_Qualified_Expression (Loc,
                    Subtype_Mark => New_Reference_To (Base_Type, Loc),
                    Expression => Make_Integer_Literal (Loc, Low_Val)),
                High_Bound =>
                  Make_Qualified_Expression (Loc,
                    Subtype_Mark => New_Reference_To (Base_Type, Loc),
                    Expression => Make_Integer_Literal (Loc, High_Val)));

            Analyze (Type_Range);
            Resolve (Type_Range, Base_Type);
            Set_Scalar_Range (T, Type_Range);
         end;

      else
         Set_Scalar_Range (T, Base_Range);
      end if;

   end Decimal_Fixed_Point_Type_Declaration;

   ------------------------
   -- Is_Indefinite_Subtype --
   ------------------------

   function Is_Indefinite_Subtype (E : Entity_Id) return Boolean is
      K : constant Entity_Kind := Ekind (E);

   begin
      if Is_Constrained (E) then
         return False;

      elsif K in Array_Kind then
         return True;

      elsif K in Class_Wide_Kind then
         return True;

      elsif K in Record_Kind or else Is_Incomplete_Or_Private_Type (E) then

         --  known discriminants : indefinite if there are no defaults value

         if Has_Discriminants (E) then
            return No (Discriminant_Default_Value (First_Discriminant (E)));

         --  unknown discriminants : always indefinite

         else
            return True;
         end if;
      end if;

      return False;
   end Is_Indefinite_Subtype;

   -------------------------------------
   -- Signed_Integer_Type_Declaration --
   -------------------------------------

   procedure Signed_Integer_Type_Declaration (T : Entity_Id; Def : Node_Id) is
      Lo            : constant Node_Id   := Low_Bound (Def);
      Hi            : constant Node_Id   := High_Bound (Def);
      Implicit_Base : constant Entity_Id :=
                        New_Itype
                          (E_Signed_Integer_Type, Parent (Def), T, 'B');
      Base_Type     : Entity_Id;
      Lo_Val        : Uint;
      Hi_Val        : Uint;

      --  Determine whether given bounds allow derivation from specified type

      function Can_Derive_From (E : Entity_Id) return Boolean is
      begin
         return UI_Ge (Lo_Val, Expr_Value (Type_Low_Bound (E)))
           and then UI_Le (Hi_Val,  Expr_Value (Type_High_Bound (E)));
      end Can_Derive_From;

      --  If the bounds are literals, their type is the chosen base type,
      --  and the not the largest integer type that would otherwise be
      --  chosen for a literal in a non-specific context. If the bounds
      --  are constants of some other integer type, convert them explicitly
      --  to the new type.

      procedure Set_Type_Bound (Bound : Node_Id) is
         Loc  : constant Source_Ptr := Sloc (Bound);

      begin
         if Nkind (Bound) = N_Integer_Literal then
            Set_Etype (Bound, Implicit_Base);

         else
            Rewrite_Substitute_Tree (Bound,
              Make_Type_Conversion (Loc,
                Expression => New_Copy_Tree (Bound),
                Subtype_Mark => New_Reference_To (Implicit_Base, Loc)));
            Analyze (Bound);
         end if;
      end Set_Type_Bound;

   --  Start of processing for Signed_Integer_Type_Declaration

   begin
      Analyze (Lo);
      Analyze (Hi);
      Resolve (Lo, Any_Integer);
      Resolve (Hi, Any_Integer);

      --  If a range constraint is used as an integer type definition, each
      --  bound of the range must be defined by a static expression of some
      --  integer type, but the two bounds need not have the same integer type.
      --  (Negative bounds are allowed.) [LRM 3.5.4]

      if not (Is_Integer_Type (Etype (Lo))
        and then Is_Integer_Type (Etype (Hi)))
      then
         Error_Msg_N
           ("integer type definition bounds must be of integer type", Def);

      elsif not Is_Static_Expression (Lo)
        or else not Is_Static_Expression (Hi)
      then
         Error_Msg_N ("integer type definition bounds must be static", Def);
         return;
      end if;

      Lo_Val := Expr_Value (Lo);
      Hi_Val := Expr_Value (Hi);

      if Can_Derive_From (Standard_Short_Short_Integer) then
         Base_Type := Standard_Short_Short_Integer;
      elsif Can_Derive_From (Standard_Short_Integer) then
         Base_Type := Standard_Short_Integer;
      elsif Can_Derive_From (Standard_Integer) then
         Base_Type := Standard_Integer;
      elsif Can_Derive_From (Standard_Long_Integer) then
         Base_Type := Standard_Long_Integer;
      elsif Can_Derive_From (Standard_Long_Long_Integer) then
         Base_Type := Standard_Long_Long_Integer;
      else
         Base_Type := Standard_Long_Long_Integer;
         Error_Msg_N ("integer type definition bounds out of range", Def);
      end if;

      Set_Scalar_Range     (Implicit_Base, Scalar_Range (Base_Type));
      Set_Etype            (Implicit_Base, Base_Type);
      Set_Esize            (Implicit_Base, Esize (Base_Type));
      Set_Alignment_Clause (Implicit_Base, Alignment_Clause (Base_Type));

      Set_Ekind            (T, E_Signed_Integer_Subtype);
      Set_Etype            (T, Implicit_Base);
      Set_Esize            (T, Esize (Implicit_Base));
      Set_Alignment_Clause (T, Alignment_Clause (Implicit_Base));
      Set_Scalar_Range     (T, Def);
      Set_Parent_Subtype   (T, Implicit_Base);

      Set_Type_Bound (Lo);
      Set_Type_Bound (Hi);
   end Signed_Integer_Type_Declaration;

   ------------------------------
   -- Modular_Type_Declaration --
   ------------------------------

   procedure Modular_Type_Declaration (T : Entity_Id; Def : Node_Id) is
      Mod_Expr : constant Node_Id := Expression (Def);
      M_Val    : Uint;
      M        : Uint;
      Bits     : Nat;

   begin
      Set_Etype (T, T);
      Set_Ekind (T, E_Modular_Integer_Type);
      Analyze (Mod_Expr);
      Resolve (Mod_Expr, Any_Integer);

      if not Is_Static_Expression (Mod_Expr) then
         Check_Static_Expression (Mod_Expr);
         M_Val := UI_Exponentiate
           (Uint_2, UI_From_Int (System_Max_Binary_Modulus_Power));
      else
         M_Val := Expr_Value (Mod_Expr);
      end if;

      if UI_Le (M_Val, Uint_1) then
         Error_Msg_N ("modulus value must be greater than 1", Mod_Expr);
         M_Val := UI_Exponentiate
           (Uint_2, UI_From_Int (System_Max_Binary_Modulus_Power));
      end if;

      Set_Modulus (T, M_Val);

      --   Create bounds for the modular type based on the modulus given in
      --   the type declaration and then analyze and resolve those bounds.

      Set_Scalar_Range (T,
        Make_Range (Sloc (Mod_Expr),
          Low_Bound  =>
            Make_Integer_Literal (Sloc (Mod_Expr),
              Intval => Uint_0),
          High_Bound =>
            Make_Integer_Literal (Sloc (Mod_Expr),
              Intval => UI_Difference (M_Val, Uint_1))));

      Analyze (Low_Bound  (Scalar_Range (T)));
      Analyze (High_Bound (Scalar_Range (T)));
      Resolve (Low_Bound  (Scalar_Range (T)), T);
      Resolve (High_Bound (Scalar_Range (T)), T);

      --  Loop through powers of 2 to find number of bits required

      for Bits in Int range 1 .. System_Max_Binary_Modulus_Power loop

         --  Binary case

         if UI_Eq (M_Val, UI_Exponentiate (Uint_2, UI_From_Int (Bits))) then
            Set_Esize (T, UI_From_Int (Bits));
            return;

         --  Non-binary case

         elsif UI_Lt (M_Val, UI_Exponentiate (Uint_2, UI_From_Int (Bits))) then
            Set_Non_Binary_Modulus (T);

            if Bits > System_Max_Nonbinary_Modulus_Power then
               Error_Msg_Uint_1 :=
                 UI_From_Int (System_Max_Nonbinary_Modulus_Power);
               Error_Msg_N
                 ("nonbinary modulus exceeds limit (2'*'*^ - 1)", Mod_Expr);
               Set_Esize (T, UI_From_Int (System_Max_Binary_Modulus_Power));
               return;

            else
               --  In the non-binary case, we must have the actual size
               --  of the object be at least enough to hold the square
               --  of the modulus.

               Set_Esize (T, UI_From_Int (Bits * 2));
               return;
            end if;
         end if;

      end loop;

      --  If we fall through, then the size exceed System.Max_Binary_Modulus
      --  so we just signal an error and set the maximum size.

      Error_Msg_Uint_1 := UI_From_Int (System_Max_Binary_Modulus_Power);
      Error_Msg_N ("modulus exceeds limit (2'*'*^)", Mod_Expr);
      Set_Esize (T, UI_From_Int (System_Max_Binary_Modulus_Power));

   end Modular_Type_Declaration;

   ----------------------------------
   -- Enumeration_Type_Declaration --
   ----------------------------------

   procedure Enumeration_Type_Declaration (T : Entity_Id; Def : Node_Id) is
      Ev             : Uint;
      L              : Node_Id;
      Int_Lit        : Node_Id;
      R_Node, B_Node : Node_Id;
      Table_Obj      : Entity_Id;
      Table_Type     : Entity_Id;

   begin
      --  Create identifier node representing lower bound

      B_Node := New_Node (N_Identifier, Sloc (Def));
      L := First (Literals (Def));
      Set_Chars (B_Node, Chars (L));
      Set_Entity (B_Node,  L);
      Set_Etype (B_Node, T);
      Set_Potentially_Static (B_Node, True);

      R_Node := New_Node (N_Range, Sloc (Def));
      Set_Low_Bound  (R_Node, B_Node);

      Set_Ekind (T, E_Enumeration_Type);
      Set_First_Literal (T, L);
      Set_Etype (T, T);

      Ev := Uint_0;

      --  Loop through literals of enumeration type

      while Present (L) loop
         Set_Ekind (L, E_Enumeration_Literal);
         Set_Etype (L, T);
         Set_Enumeration_Pos (L, Ev);
         Set_Enumeration_Rep (L, Ev);
         New_Overloaded_Entity (L);

         if Nkind (L) = N_Defining_Character_Literal then
            Set_Is_Character_Type (T, True);
         end if;

         Ev := UI_Sum (Ev, Uint_1);
         L := Next (L);
      end loop;

      --  Now create a node representing upper bound

      B_Node := New_Node (N_Identifier, Sloc (Def));
      Set_Chars (B_Node, Chars (Last (Literals (Def))));
      Set_Entity (B_Node,  Last (Literals (Def)));
      Set_Etype (B_Node, T);
      Set_Potentially_Static (B_Node, True);

      Set_High_Bound (R_Node, B_Node);
      Set_Scalar_Range (T, R_Node);
      Set_Esize (T, Determine_Enum_Size (T));

      --  Create two defining occurrences corresponding to a enumeration
      --  table containing the literal names and its type. This table is
      --  used in conjunction with calls to 'Image on enumeration values.
      --  This table is filled in by the back-end.

      Table_Obj :=
        Make_Defining_Identifier (Sloc (Def),
          Chars => New_External_Name (Chars (T), 'T'));

      Set_Is_Internal (Table_Obj);
      Append_Entity (Table_Obj, Current_Scope);
      Set_Current_Entity (Table_Obj);

      Table_Type := New_Itype (E_Enum_Table_Type, Parent (Def), T, 'T');

      Set_Etype         (Table_Obj, Table_Type);
      Set_Ekind         (Table_Obj, E_Variable);
      Set_Public_Status (Table_Obj);

      Set_Etype (Table_Type, Table_Type);
      Set_Public_Status (Table_Type);
      Set_Component_Type (Table_Type, Standard_A_String);
      Set_First_Index (Table_Type,
        First (New_List (
          New_Occurrence_Of (Standard_Positive, Sloc (Def)))));
      Int_Lit := New_Node (N_Integer_Literal, Sloc (Def));
      Set_Intval
        (Int_Lit, Enumeration_Pos (Entity (Type_High_Bound (T))));
      Set_Etype (Int_Lit, Standard_Integer);
      Set_Potentially_Static (Int_Lit, True);
      Set_Table_High_Bound (Table_Type, Int_Lit);
      Set_Lit_Name_Table (T, Table_Obj);
   end Enumeration_Type_Declaration;

   -------------------------
   -- Determine_Enum_Size --
   -------------------------

   function Determine_Enum_Size  (T : Entity_Id) return Uint is
      Count : Nat;
      Lit   : Entity_Id;

   begin
      Lit := First_Literal (T);
      Count := 0;

      while Present (Lit) loop
         Count := Count + 1;
         Lit := Next_Literal (Lit);
      end loop;

      if Count <= 2 ** 8 then
         return Uint_8;
      elsif Count <= 2 ** 16 then
         return Uint_16;
      else
         return Uint_32;
      end if;
   end Determine_Enum_Size;

   ----------------------------
   -- Array_Type_Declaration --
   ----------------------------

   procedure Array_Type_Declaration (T : in out Entity_Id; Def : Node_Id) is
      Component_Def : constant Node_Id := Subtype_Indication (Def);
      Element_Type  : Entity_Id;
      Implicit_Base : Entity_Id;
      Index         : Node_Id;
      Related_Id    : Entity_Id := Empty;
      Nb_Index      : Nat;
      P             : constant Node_Id := Parent (Def);

   begin
      if Nkind (Def) = N_Constrained_Array_Definition then

         Index := First (Discrete_Subtype_Definitions (Def));

         --  Find proper names for the implicit types which may be public.
         --  in case of anonymous arrays we use the name of the first object
         --  of that type as prefix.

         if No (T) then
            Related_Id :=  Defining_Identifier (P);
         else
            Related_Id := T;
         end if;

      else
         Index := First (Subtype_Marks (Def));
      end if;

      Nb_Index := 1;

      while Present (Index) loop
         Analyze (Index);
         Make_Index (Index, P, Related_Id, Nb_Index);
         Index := Next_Index (Index);
         Nb_Index := Nb_Index + 1;
      end loop;

      Element_Type := Process_Subtype (Component_Def, P, Related_Id, 'C');

      --  Constrained array case

      if Nkind (Def) = N_Constrained_Array_Definition then

         --  Establish Implicit_Base as unconstrained base type

         Implicit_Base := New_Itype (E_Array_Type, P, Related_Id, 'B');

         Set_Esize     (Implicit_Base, Uint_0);
         Set_Etype     (Implicit_Base, Implicit_Base);
         Set_Scope     (Implicit_Base, Current_Scope);

         if No (T) then
            T := New_Itype (E_Void, P, Related_Id, 'T');
         end if;

         --  The constrained array type is a subtype of the unconstrained one

         Set_Ekind          (T, E_Array_Subtype);
         Set_Esize          (T, Uint_0);
         Set_Etype          (T, Implicit_Base);
         Set_Scope          (T, Current_Scope);
         Set_Is_Constrained (T, True);
         Set_First_Index    (T, First (Discrete_Subtype_Definitions (Def)));

         --  Complete setup of implicit base type

         Set_First_Index    (Implicit_Base, First_Index (T));
         Set_Component_Type (Implicit_Base, Element_Type);
         Set_Has_Tasks      (Implicit_Base, Has_Tasks (Element_Type));
         Set_Has_Controlled (Implicit_Base,
           Has_Controlled (Element_Type) or else Is_Controlled (Element_Type));

      --  Unconstrained array case

      else
         Set_Ekind       (T, E_Array_Type);
         Set_Esize       (T, Uint_0);
         Set_Etype       (T, T);
         Set_Scope       (T, Current_Scope);
         Set_Is_Constrained (T, False);
         Set_First_Index (T, First (Subtype_Marks (Def)));
      end if;

      Set_Component_Type (T, Element_Type);
      Set_Has_Tasks      (T, Has_Tasks (Element_Type));
      Set_Has_Controlled (T,
        Has_Controlled (Element_Type) or else Is_Controlled (Element_Type));

      if Aliased_Present (Def) then
         Set_Is_Aliased (T);
         Set_Is_Aliased (Etype (T));
      end if;

      if Number_Dimensions (T) = 1 then
         New_Binary_Operator (Name_Op_Concat, T);
      end if;

      --  In the case of an unconstrained array the parser has already
      --  verified that all the indices are unconstrained but we still
      --  need to make sure that the element type is constrained.

      if Is_Indefinite_Subtype (Element_Type) then
         Error_Msg_N
           ("unconstrained element type in array declaration ",
            Component_Def);

      elsif Is_Abstract (Element_Type) then
         Error_Msg_N ("The type of a component cannot be abstract ",
              Component_Def);
      end if;

   end Array_Type_Declaration;

   ----------------
   -- Make_Index --
   ----------------

   procedure Make_Index
     (I            : Node_Id;
      Related_Nod  : Node_Id;
      Related_Id   : Entity_Id := Empty;
      Suffix_Index : Nat := 1)
   is
      L : Entity_Id;
      --  Type of low bound of range

      H : Entity_Id;
      --  Type of high bound of range

      R      : Node_Id;
      T      : Entity_Id;
      Def_Id : Entity_Id;

   begin
      --  For a discrete range used in a constrained array definition and
      --  defined by a range, an implicit conversion to the predefined type
      --  INTEGER is assumed if each bound is either a numeric literal, a named
      --  number, or an attribute, and the type of both bounds (prior to the
      --  implicit conversion) is the type universal_integer. Otherwise, both
      --  bounds must be of the same discrete type, other than universal
      --  integer; this type must be determinable independently of the
      --  context, but using the fact that the type must be discrete and that
      --  both bounds must have the same type.

      --  Character literals also have a universal type in the absence of
      --  of additional context,  and are resolved to Standard_Character.

      if Nkind (I) = N_Range then

         --  The index is given by a range constraint. The bounds are known
         --  to be of a consistent type.

         L := Etype (Low_Bound (I));
         H := Etype (High_Bound (I));

         if L = Universal_Integer and H = Universal_Integer then
            T := Standard_Integer;
         elsif L = Universal_Integer then
            T := H;
         elsif L = Any_Character and then H = Any_Character then
            T := Standard_Character;
         else
            T := L;
         end if;

         R := I;
         Process_Range_Expr_In_Decl (R, T, Related_Nod);

      elsif Nkind (I) = N_Subtype_Indication then

         --  The index is given by a subtype with a range constraint.

         T :=  Base_Type (Entity (Subtype_Mark (I)));
         R := Range_Expression (Constraint (I));
         Resolve (R, T);
         Process_Range_Expr_In_Decl (R,
           Entity (Subtype_Mark (I)), Related_Nod);

      elsif Nkind (I) = N_Attribute_Reference then

         --  The parser guarantees that the attribute is a RANGE attribute

         Analyze (I);
         T := Etype (I);
         Resolve (I, T);
         R := I;

      --  If none of the above, must be a subtype (nothing to do)

      else
         if not Is_Entity_Name (I) or else not Is_Type (Entity (I)) then
            Error_Msg_N ("invalid subtype mark in discrete range ", I);
            Set_Etype (I, Any_Integer);
         end if;

         return;
      end if;

      if not Is_Discrete_Type (T) then
         Error_Msg_N ("discrete type required for range", I);
         Set_Etype (I, Any_Type);
         return;

      elsif T = Any_Type then
         Set_Etype (I, Any_Type);
         return;
      end if;

      Def_Id := New_Itype (E_Void, Related_Nod, Related_Id, 'X', Suffix_Index);
      Set_Etype (Def_Id, Base_Type (T));

      --  ??? what about modular types in the following situation

      if Is_Integer_Type (T) then
         Set_Ekind (Def_Id, E_Signed_Integer_Subtype);
      else
         Set_Ekind (Def_Id, E_Enumeration_Subtype);
         Set_Is_Character_Type (Def_Id, Is_Character_Type (T));
      end if;

      Set_Esize            (Def_Id, Esize (T));
      Set_Alignment_Clause (Def_Id, Alignment_Clause (T));
      Set_Scalar_Range     (Def_Id, R);
      Set_Parent_Subtype   (Def_Id, T);

      Set_Etype (I, Def_Id);
   end Make_Index;

   ---------------------
   -- Constrain_Array --
   ---------------------

   procedure Constrain_Array
     (Def_Id      : in out Entity_Id;
      SI          : Node_Id;
      Related_Nod : Node_Id;
      Related_Id  : Entity_Id;
      Suffix      : Character)
   is
      C                     : constant Node_Id := Constraint (SI);
      Number_Of_Constraints : Nat := 0;
      Index                 : Node_Id;
      S, T                  : Entity_Id;

   begin
      if Nkind (C) /= N_Index_Or_Discriminant_Constraint then
         Error_Msg_N ("incorrect constraint given for array type", C);
         return;
      end if;

      T := Entity (Subtype_Mark (SI));

      if Ekind (T) in Access_Kind then
         T := Designated_Type (T);
      end if;

      --  If an index constraint follows a subtype mark in a subtype indication
      --  then the type or subtype denoted by the subtype mark must not already
      --  impose an index constraint. The subtype mark must denote either an
      --  unconstrained array type or an access type whose designated type
      --  is such an array type... [LRM 3.6.1]

      if Is_Constrained (T) then
         Error_Msg_N
          ("array type is already constrained", Subtype_Mark (SI));
         return;
      end if;

      S := First (Constraints (C));

      while Present (S) loop
         Number_Of_Constraints := Number_Of_Constraints + 1;
         S := Next (S);
      end loop;

      --  In either case, the index constraint must provide a discrete range
      --  for each index of the array type and the type of each discrete range
      --  must be the same as that of the corresponding index. [LRM 3.6.1]

      if Number_Of_Constraints /= Number_Dimensions (T) then
         Error_Msg_NE
           ("incorrect no. of index constraints for type&", C, T);
         return;
      end if;

      S := First (Constraints (C));
      Index := First_Index (T);
      Analyze (Index);

      --  Apply constraints to each index type

      for J in 1 .. Number_Of_Constraints loop
         Constrain_Index (Index, S, Related_Nod, Related_Id, Suffix, J);
         Index := Next (Index);
         S := Next (S);
      end loop;

      if No (Def_Id) then
         Def_Id :=
           New_Itype (E_Array_Subtype, Related_Nod, Related_Id, Suffix);
      else
         Set_Ekind (Def_Id, E_Array_Subtype);
      end if;

      Set_Esize              (Def_Id, Uint_0);
      Set_Etype              (Def_Id, Base_Type (T));
      Set_First_Index        (Def_Id, First (Constraints (C)));
      Set_Component_Type     (Def_Id, Component_Type (T));
      Set_Has_Tasks          (Def_Id, Has_Tasks (T));
      Set_Has_Controlled     (Def_Id, Has_Controlled (T));
      Set_Is_Constrained     (Def_Id);
      Set_Depends_On_Private (Def_Id, Has_Private_Component (Def_Id));

   end Constrain_Array;

   --------------------------
   -- Constrain_Concurrent --
   --------------------------

   --  For concurrent types, the associated record value type carries the same
   --  discriminants, so when we constrain a concurrent type, we must constrain
   --  the value type as well.

   procedure Constrain_Concurrent
     (Def_Id      : in out Entity_Id;
      SI          : Node_Id;
      Related_Nod : Node_Id;
      Related_Id  : Entity_Id;
      Suffix      : Character)
   is
      T_Ent : constant Entity_Id := Entity (Subtype_Mark (SI));
      T_Val : constant Entity_Id := Corresponding_Record_Type (T_Ent);
      T_Sub : Entity_Id;

   begin
      if Present (T_Val) then
         T_Sub := New_Itype (E_Record_Subtype, Related_Nod, Related_Id, 'V');

         --  Create Def_Id (after T_Sub) if not yet created

         if No (Def_Id) then
            Def_Id := New_Itype (E_Void, Related_Nod, Related_Id, Suffix);
         end if;

         Constrain_Discriminated_Type  (Def_Id, SI, Related_Nod);
         Set_Depends_On_Private (Def_Id, Has_Private_Component (Def_Id));
         Set_Corresponding_Record_Type (Def_Id, T_Sub);

         Set_Etype                   (T_Sub, T_Val);
         Set_Esize                   (T_Sub, Uint_0);
         Set_Has_Discriminants       (T_Sub, True);
         Set_Is_Constrained          (T_Sub, True);
         Set_First_Entity            (T_Sub, First_Entity (T_Val));
         Set_Last_Entity             (T_Sub, Last_Entity (T_Val));
         Set_Discriminant_Constraint (T_Sub, Discriminant_Constraint (Def_Id));
         Set_Depends_On_Private      (T_Sub, Has_Private_Component (T_Sub));
      end if;
   end Constrain_Concurrent;

   ---------------------
   -- Constrain_Index --
   ---------------------

   procedure Constrain_Index
     (Index        : Node_Id;
      S            : Node_Id;
      Related_Nod  : Node_Id;
      Related_Id   : Entity_Id;
      Suffix       : Character;
      Suffix_Index : Nat)
   is
      Def_Id : Entity_Id;
      R      : Node_Id;
      T      : constant Entity_Id := Etype (Index);

   begin
      if Nkind (S) = N_Range
        or else Nkind (S) = N_Attribute_Reference
      then
         --  A Range attribute will transformed into N_Range by Resolve.

         Analyze (S);
         Set_Etype (S, T);
         R := S;
         Process_Range_Expr_In_Decl (R, T, Related_Nod);

         if Nkind (S) /= N_Range
           or else Base_Type (T) /= Base_Type (Etype (Low_Bound (S)))
           or else Base_Type (T) /= Base_Type (Etype (High_Bound (S)))
         then
            Error_Msg_NE ("invalid index constraint for&", S, Index);
         end if;

      elsif Nkind (S) = N_Subtype_Indication then
         Analyze (Subtype_Mark (S));
         R := Range_Expression (Constraint (S));
         Analyze (R);

         if Base_Type (Entity (Subtype_Mark (S))) /= Base_Type (T) then
            Error_Msg_NE ("invalid index constraint for&", R, Index);
            Set_Etype (S, Any_Type);
            return;
         else
            Resolve (R, T);
         end if;

      --  Subtype_Mark case, no anonymous subtypes to construct

      else
         Analyze (S);
         if Is_Entity_Name (S) then

            if not Is_Type (Entity (S))
              or else Base_Type (Entity (S)) /= Base_Type (T)
            then
               Error_Msg_NE ("invalid index constraint for&", S, Index);
            end if;

            return;

         else
            Error_Msg_N ("invalid index constraint", S);
            return;
         end if;
      end if;

      Def_Id :=
        New_Itype (E_Void, Related_Nod, Related_Id, Suffix, Suffix_Index);

      Set_Etype (Def_Id, Base_Type (T));

      --  What about modular types in the following test ???

      if Is_Integer_Type (T) then
         Set_Ekind (Def_Id, E_Signed_Integer_Subtype);
      else
         Set_Ekind (Def_Id, E_Enumeration_Subtype);
         Set_Is_Character_Type (Def_Id, Is_Character_Type (T));
      end if;

      Set_Esize            (Def_Id, Esize (T));
      Set_Alignment_Clause (Def_Id, Alignment_Clause (T));
      Set_Scalar_Range     (Def_Id, R);
      Set_Parent_Subtype   (Def_Id, T);

      Set_Etype (S, Def_Id);
   end Constrain_Index;


   procedure Check_Or_Process_Discriminants (N : Node_Id; T : Entity_Id) is

   --  If an incomplete or private type declaration was already given for
   --  the type, the discriminants may have already been processed if they
   --  were present on the incomplete declaration. In this case a full
   --  conformance check is performed otherwise just process them.

   begin
      if Has_Discriminants (T) then

         --  ??? conformance checks not implemented

         null;

         --  Make the discriminants visible to component declarations.

         declare
            D    : Entity_Id := First_Discriminant (T);
            Prev : Entity_Id;

         begin
            while Present (D) loop
               Prev := Current_Entity (D);
               Set_Current_Entity (D);
               Set_Is_Immediately_Visible (D);
               Set_Homonym (D, Prev);
               D := Next_Discriminant (D);
            end loop;
         end;

      else
         if Present (Discriminant_Specifications (N)) then
            Process_Discriminants (N);
         end if;
      end if;
   end Check_Or_Process_Discriminants;

   -----------------------------
   -- Record_Type_Declaration --
   -----------------------------

   procedure Record_Type_Declaration (T : Entity_Id; N : Node_Id) is
      Def : constant Node_Id := Type_Definition (N);

   begin
      --  Records constitute a scope for the component declarations within.
      --  The scope is created prior to the processing of these declarations.
      --  Discriminants are processed first, so that they are visible when
      --  processing the other components. The Ekind of the record type itself
      --  is set to E_Record_Type (subtypes appear as E_Record_Subtype).
      --  If an incomplete or private type declaration was already given for
      --  the type, then this scope already exists, and the discriminants have
      --  been declared within. We must verify that the full declaration
      --  matches the incomplete one.

      New_Scope (T); -- Enter record scope
      Set_Is_Limited_Record (T, Limited_Present (Def));

      Check_Or_Process_Discriminants (N, T);

      Set_Ekind          (T, E_Record_Type);
      Set_Etype          (T, T);
      Set_Esize          (T, Uint_0);
      Set_Is_Constrained (T, not Has_Discriminants (T));

      Record_Type_Definition (Def, T);

      End_Scope; -- Exit record scope

   end Record_Type_Declaration;

   ------------------------------------
   -- Tagged_Record_Type_Declaration --
   ------------------------------------

   procedure Tagged_Record_Type_Declaration (T : Entity_Id; N : Node_Id) is
      Def : constant Node_Id := Type_Definition (N);

   begin

      New_Scope (T); -- Enter record scope

      Set_Is_Tagged_Type (T);
      Set_Is_Limited_Record (T, Limited_Present (Def));

      --  Type is abstract if full declaration carries keyword, or if
      --  previous partial view did.

      Set_Is_Abstract (T, Is_Abstract (T) or else Abstract_Present (Def));
      Check_Or_Process_Discriminants (N, T);
      Set_Ekind          (T, E_Record_Type);
      Set_Etype          (T, T);
      Set_Esize          (T, Uint_0);
      Set_Is_Constrained (T, not Has_Discriminants (T));

      Record_Type_Definition (Def, T);

      Make_Class_Wide_Type (T);
      Set_Primitive_Operations (T, New_Elmt_List);

      End_Scope; -- Exit record scope
   end Tagged_Record_Type_Declaration;

   ---------------------------
   -- Process_Discriminants --
   ---------------------------

   procedure Process_Discriminants (N : Node_Id) is
      Id                  : Node_Id;
      Discr               : Node_Id;
      Discr_Type          : Entity_Id;
      Default_Present     : Boolean := False;
      Default_Not_Present : Boolean := False;
      D_Minal             : Entity_Id;
      Elist               : Elist_Id;

   begin
      --  A composite type other than an array type can have discriminants.
      --  Discriminants of non-limited types must have a discrete type.
      --  On entry, the current scope is the composite type.

      --  The discriminants are initially entered into the scope of the type
      --  via Enter_Name with the default Ekind of E_Void to prevent premature
      --  use, as explained at the end of this procedure.

      Elist := New_Elmt_List;

      Discr := First (Discriminant_Specifications (N));
      while Present (Discr) loop
         Enter_Name (Defining_Identifier (Discr));

         if Nkind (Discriminant_Type (Discr)) = N_Access_Definition then
            Discr_Type := Access_Definition (N, Discriminant_Type (Discr));

         else
            Analyze (Discriminant_Type (Discr));
            Discr_Type := Etype (Discriminant_Type (Discr));
         end if;

         if Is_Access_Type (Discr_Type) then
            Note_Feature (Access_Discriminants, Sloc (Discr));

            --  A discriminant_specification for an access discriminant
            --  shall appear only in the declaration for a task or protected
            --  type, or for a type with the reserved word 'limited' in
            --  its definition or in one of its ancestors. [LRM 3.7(10)]

            if not Is_Concurrent_Type (Current_Scope)
              and then not Is_Limited_Record (Current_Scope)
              and then Ekind (Current_Scope) /= E_Limited_Private_Type
            then
               Error_Msg_N
                ("access discriminants allowed only for limited types", Discr);
            end if;

            if Ada_83 then
               Error_Msg_N
                 ("(Ada 83) access discriminant not allowed", Discr);
            end if;

         elsif not Is_Discrete_Type (Discr_Type) then
            Error_Msg_N ("discriminants must have a discrete or access type",
              Discriminant_Type (Discr));
         end if;

         Set_Etype (Defining_Identifier (Discr), Discr_Type);

         --  If a discriminant specification includes the assignment compound
         --  delimiter followed by an expression, the expression is the default
         --  expression of the discriminant; the default expression must be of
         --  the type of the discriminant. [LRM 3.7.1]

         if Present (Expression (Discr)) then
            Analyze (Expression (Discr));
            Resolve (Expression (Discr), Discr_Type);
            Check_Non_Static_Context (Expression (Discr));
            Default_Present := True;
            Append_Elmt (Expression (Discr), Elist);

            --  Tag the defining identifiers for the discriminants with their
            --  corresponding default expressions from the tree.

            Set_Discriminant_Default_Value
              (Defining_Identifier (Discr), Expression (Discr));

         else
            Default_Not_Present := True;
         end if;

         Discr := Next (Discr);
      end loop;

      --  An element list consisting of the default expressions of the
      --  discriminants is constructed in the above loop and used to set
      --  the Discriminant_Constraint attribute for the type. If an object
      --  is declared of this (record or task) type without any explicit
      --  discriminant constraint given, this element list will form the
      --  actual parameters for the corresponding initialization procedure
      --  for the type.

      Set_Discriminant_Constraint (Current_Scope, Elist);

      --  Default expressions must be provided either for all or for none
      --  of the discriminants of a discriminant part. [LRM 3.7.1]

      if Default_Present and Default_Not_Present then
         Error_Msg_N
          ("incomplete specification of defaults for discriminants", N);
      end if;

      --  The use of the name of a discriminant is not allowed in default
      --  expressions of a discriminant part if the specification of the
      --  discriminant is itself given in the discriminant part. [LRM 3.7.1]

      --  To detect this, the discriminant names are entered initially with an
      --  Ekind of E_Void (which is the default Ekind given by Enter_Name). Any
      --  attempt to use a void entity (for example in an expression that is
      --  type-checked) produces the error message: premature usage.  Now after
      --  completing the semantic analysis of the discriminant part, we can set
      --  the Ekind of all the discriminants appropriately.

      Discr := First (Discriminant_Specifications (N));

      while Present (Discr) loop
         Id := Defining_Identifier (Discr);
         Set_Ekind (Id, E_Discriminant);

         --  Initialize the Original_Record_Component to the entity itself
         --  the New_Copy call in Build_Derived_Type will automatically
         --  propagate the right value to descendants

         Set_Original_Record_Component (Id, Id);

         --  Create discriminal, that is to say the associated entity
         --  to be used in initialization procedures for the type,
         --  in which a discriminal is a formal parameter whose actual
         --  is the value of the corresponding discriminant constraint.
         --  Discriminals are not used during semantic analysis, and are
         --  not fully defined entities until expansion. Thus they are not
         --  given a scope until intialization procedures are built.

         D_Minal :=
           Make_Defining_Identifier
             (Sloc (N), New_External_Name (Chars (Id), 'D'));
         Set_Ekind (D_Minal, E_In_Parameter);
         Set_Etype (D_Minal, Etype (Id));
         Set_Discriminal (Id, D_Minal);

         Discr := Next (Discr);
      end loop;

      Set_Has_Discriminants (Current_Scope);
   end Process_Discriminants;

   --------------------------------
   -- Discriminant_Redeclaration --
   --------------------------------

   procedure Discriminant_Redeclaration (T : Entity_Id; D_List : List_Id) is
   begin
      null; -- For now ???
   end Discriminant_Redeclaration;

   ----------------------------
   -- Record_Type_Definition --
   ----------------------------

   procedure Record_Type_Definition (Def : Node_Id; T : Entity_Id) is
      Anon_List : List_Id;
      Component : Entity_Id;

   begin
      if Tagged_Present (Def) then
         Expand_Tagged_Root (Def);
      end if;

      --  If the component list of a record type is defined by the reserved
      --  word null and there is no discriminant part, then the record type has
      --  no components and all records of the type are null records. [LRM 3.7]

      if No (Component_List (Def))
        or else Null_Present (Component_List (Def))
      then
         return;
      end if;

      Analyze_Declarations (Component_Items (Component_List (Def)));

      if Present (Variant_Part (Component_List (Def))) then
         Analyze (Variant_Part (Component_List (Def)));
      end if;

      Component := First_Entity (Current_Scope);

      --  After completing the semantic analysis of the record definition,
      --  then record components are accessible.

      while Present (Component) loop
         if Ekind (Component) = E_Void then
            Set_Ekind (Component, E_Component);
         end if;

         if Has_Tasks (Etype (Component)) then
            Set_Has_Tasks (T, True);
         end if;

         if Has_Controlled (Etype (Component))
           or else Is_Controlled (Etype (Component))
         then
            Note_Feature (Controlled_Types, Sloc (T));
            Set_Has_Controlled (T, True);
         end if;

         Component := Next_Entity (Component);
      end loop;
   end Record_Type_Definition;

   -----------------------------------
   -- Analyze_Component_Declaration --
   -----------------------------------

   procedure Analyze_Component_Declaration (N : Node_Id) is
      Id : constant Entity_Id := Defining_Identifier (N);
      T  : Entity_Id;

   begin
      Enter_Name (Defining_Identifier (N));
      T := Find_Type_Of_Object (Subtype_Indication (N), N);

      --  If a component declaration includes the assignment compound delimiter
      --  followed by an expression, the expression is the default expression
      --  of the record component; the default expression must be of the type
      --  of the component. Default expressions are not allowed for components
      --  that are of a limited type. [LRM 3.7 (5)]

      if Present (Expression (N)) then
         Analyze (Expression (N));
         Check_Initialization (T, Expression (N));
      end if;

      if Is_Indefinite_Subtype (T) then
         Error_Msg_N
           ("unconstrained subtype in component declaration",
            Subtype_Indication (N));

      --  Components cannot be abstract, except for the special case of
      --  the _Parent field (case of extending an abstract tagged type)

      elsif Is_Abstract (T) and then Chars (Id) /= Name_uParent then
         Error_Msg_N ("The type of a component cannot be abstract", N);
      end if;

      Set_Ekind (Id, E_Component);
      Set_Etype (Id, T);
      Set_Is_Aliased (Id, Aliased_Present (N));

      --  Initialyze the Original_Record_Component to the entity itself
      --  the New_Copy call in Build_Derived_Type will automatically
      --  propagate the right value to descendants

      Set_Original_Record_Component (Id, Id);
   end Analyze_Component_Declaration;

   ---------------------------
   -- Analyze_Others_Choice --
   ---------------------------

   --  Nothing to do for the others choice node itself, the semantic analysis
   --  of the others choice will occur as part of the processing of the parent

   procedure Analyze_Others_Choice (N : Node_Id) is
   begin
      null;
   end Analyze_Others_Choice;

   --------------------------
   -- Analyze_Variant_Part --
   --------------------------

   procedure Analyze_Variant_Part (N : Node_Id) is
      Case_Table     : Case_Table_Type (1 .. Number_Of_Case_Choices (N));
      Choice         : Node_Id;
      Choice_Count   : Nat := 0;
      Discr_Name     : Node_Id;
      Discr_Type     : Entity_Id;
      E              : Entity_Id;
      Hi             : Node_Id;
      Invalid_Case   : Boolean := False;
      Kind           : Node_Kind;
      Lo             : Node_Id;
      Others_Present : Boolean := False;
      Variant        : Node_Id;

      procedure Check_Choice (Lo, Hi : Node_Id; Position : Node_Id);
      --  Check_Choice checks whether the given bounds of a choice are
      --  static. If not a message is issued, otherwise the bounds are
      --  entered into the case table.

      procedure Check_Choice (Lo, Hi : Node_Id; Position : Node_Id) is
      begin
         --  The simple expressions and discrete ranges given as choices
         --  in a variant part must be static. [LRM 3.7.3]

         if not Is_Static_Expression (Lo)
           or else not Is_Static_Expression (Hi)
         then
            Error_Msg_N ("choice given in variant part not static", Position);
            Invalid_Case := True;
         else
            Choice_Count := Choice_Count + 1;
            Case_Table (Choice_Count).Choice_Lo := Lo;
            Case_Table (Choice_Count).Choice_Hi := Hi;
            Case_Table (Choice_Count).Choice_Node := Position;
         end if;
      end Check_Choice;

   --  Start of processing for Analyze_Variant_Part

   begin
      Discr_Name := Name (N);
      Analyze (Discr_Name);

      if Ekind (Entity (Discr_Name)) /= E_Discriminant then
         Error_Msg_N ("invalid discriminant name in variant part", Discr_Name);
      end if;

      Discr_Type := Etype (Entity (Discr_Name));

      --  The type of the discriminant of a variant part must not be a
      --  generic formal type. [LRM 3.7.3]

      if Is_Generic_Type (Discr_Type) then
         Error_Msg_N
           ("discriminant of variant part cannot be generic", Discr_Name);
         return;
      end if;

      --  Now check each of the case choices against Exp_Base_Type.

      Variant := First (Variants (N));

      while Present (Variant) loop
         Choice := First (Discrete_Choices (Variant));

         while Present (Choice) loop
            Analyze (Choice);
            Kind := Nkind (Choice);

            if Kind = N_Range then
               Resolve (Choice, Discr_Type);
               Check_Choice (Low_Bound (Choice), High_Bound (Choice), Choice);

            elsif (Kind = N_Identifier or else Kind = N_Selected_Component)
              and then Is_Type (Entity (Choice))
            then
               E  := Entity (Choice);
               Lo := Type_Low_Bound (E);
               Hi := Type_High_Bound (E);
               Check_Choice (Lo, Hi, Choice);

            elsif Kind = N_Subtype_Indication then
               pragma Assert (False); null;        -- for now ???

            --  The choice others is only allowed for the last variant and as
            --  its only choice; it stands for all values (possibly none) not
            --  given in the choices of previous variants. [LRM 3.7.3]

            elsif Kind = N_Others_Choice then
               if not (Choice = First (Discrete_Choices (Variant))
                 and then Choice = Last (Discrete_Choices (Variant))
                 and then Variant = Last (Variants (N)))
               then
                  Error_Msg_N
                    ("the choice OTHERS must appear alone and last", Choice);
                  return;
               end if;

               Others_Present := True;

            else
               --  Must be an expression

               Resolve (Choice, Discr_Type);
               Check_Choice (Choice, Choice, Choice);
            end if;

            Choice := Next (Choice);
         end loop;

         if not Null_Present (Component_List (Variant)) then
            Analyze_Declarations (Component_Items (Component_List (Variant)));

            if Present (Variant_Part (Component_List (Variant))) then
               Analyze (Variant_Part (Component_List (Variant)));
            end if;
         end if;

         Variant := Next (Variant);
      end loop;

      if not Invalid_Case
        and then Case_Table'Length > 0
      then
         Check_Case_Choices (Case_Table, N, Discr_Type, Others_Present);
      end if;

      if not Invalid_Case
        and then Others_Present
      then

         --  Fill in Others_Discrete_Choices field of the OTHERS choice

         Choice := Last (Discrete_Choices (Last (Variants (N))));
         Expand_Others_Choice (Case_Table, Choice, Discr_Type);
      end if;

   end Analyze_Variant_Part;

   --------------------------
   -- Expand_Others_Choice --
   --------------------------

   procedure Expand_Others_Choice
     (Case_Table    : Case_Table_Type;
      Others_Choice : Node_Id;
      Choice_Type   : Entity_Id)
   is
      Choice      : Node_Id;
      Choice_List : List_Id := New_List;
      Exp_Lo      : Node_Id;
      Exp_Hi      : Node_Id;
      Hi          : Uint;
      Lo          : Uint;
      Loc         : Source_Ptr := Sloc (Others_Choice);
      Previous_Hi : Uint;

      function Lit_Of (Value : Uint) return Node_Id;
      --  Returns the Node_Id for the enumeration literal corresponding to the
      --  position given by Value within the enumeration type Choice_Type.

      function Build_Choice (Value1, Value2 : Uint) return Node_Id;
      --  Builds a node representing the missing choices given by the
      --  Value1 and Value2. A N_Range node is built if there is more than
      --  one literal value missing. Otherwise a single N_Integer_Literal,
      --  N_Identifier or N_Character_Literal is built depending on what
      --  Choice_Type is.

      ------------
      -- Lit_Of --
      ------------

      function Lit_Of (Value : Uint) return Node_Id is
         Lit : Entity_Id;

      begin
         --  In the case where the literal is of type Character, there needs
         --  to be some special handling since there is no explicit chain
         --  of literals to search. Instead, a N_Character_Literal node
         --  is created with the appropriate Char_Code and Chars fields.

         if Root_Type (Choice_Type) = Standard_Character then
            Set_Character_Literal_Name (Char_Code (Value));
            Lit := New_Node (N_Character_Literal, Loc);
            Set_Chars (Lit, Name_Find);
            Set_Char_Literal_Value (Lit, Char_Code (UI_To_Int (Value)));
            Set_Etype (Lit, Choice_Type);
            Set_Potentially_Static (Lit, True);
            return Lit;

         --  Otherwise, iterate through the literals list of Choice_Type
         --  "Value" number of times until the desired literal is reached
         --  and then return an occurrence of it.

         else
            Lit := First_Literal (Choice_Type);
            for J in 1 .. UI_To_Int (Value) loop
               Lit := Next_Literal (Lit);
            end loop;

            return New_Occurrence_Of (Lit, Loc);
         end if;
      end Lit_Of;

      ------------------
      -- Build_Choice --
      ------------------

      function Build_Choice (Value1, Value2 : Uint) return Node_Id is
         Lit_Node : Node_Id;
         Lo, Hi   : Node_Id;
         Lo_Val   : Uint;
         Hi_Val   : Uint;

      begin
         --  If there is only one choice value missing between Value1 and
         --  Value2, build an integer or enumeration literal to represent it.

         if UI_Eq (UI_Difference (Value2, Value1), Uint_1) then

            if Is_Integer_Type (Choice_Type) then
               Lit_Node := Make_Integer_Literal (Loc, Value1);
               Set_Etype (Lit_Node, Choice_Type);
            else
               Lit_Node := Lit_Of (Value1);
            end if;

         --  Otherwise is more that one choice value that is missing between
         --  Value1 and Value2, therefore build a N_Range node of either
         --  integer or enumeration literals.

         else
            Hi_Val := UI_Difference (Value2, Uint_1);

            if Is_Integer_Type (Choice_Type) then
               Lo := Make_Integer_Literal (Loc, Value1);
               Set_Etype (Lo, Choice_Type);
               Hi := Make_Integer_Literal (Loc, Hi_Val);
               Set_Etype (Hi, Choice_Type);
               Lit_Node :=
                 Make_Range (Loc,
                   Low_Bound  => Lo,
                   High_Bound => Hi);

            else
               Lit_Node :=
                 Make_Range (Loc,
                   Low_Bound  => Lit_Of (Value1),
                   High_Bound => Lit_Of (Hi_Val));
            end if;
         end if;

         return Lit_Node;
      end Build_Choice;

   --  Start of processing for Expand_Others_Choice

   begin

      if Case_Table'Length = 0 then

         --  Pathological case: only an others case is present.
         --  The others case covers the full range of the type.

         if Is_Static_Subtype (Choice_Type) then
            Choice := New_Occurrence_Of (Choice_Type, Loc);
         else
            Choice := New_Occurrence_Of (Base_Type (Choice_Type), Loc);
         end if;

         Set_Others_Discrete_Choices (Others_Choice, New_List (Choice));
         return;
      end if;

      --  Establish the bound values for the variant depending upon whether
      --  the type of the discriminant name is static or not.

      if Is_Static_Subtype (Choice_Type) then
         Exp_Lo := Type_Low_Bound (Choice_Type);
         Exp_Hi := Type_High_Bound (Choice_Type);
      else
         Exp_Lo := Type_Low_Bound (Base_Type (Choice_Type));
         Exp_Hi := Type_High_Bound (Base_Type (Choice_Type));
      end if;

      Lo := Expr_Value (Case_Table (Case_Table'First).Choice_Lo);
      Hi := Expr_Value (Case_Table (Case_Table'First).Choice_Hi);
      Previous_Hi := Expr_Value (Case_Table (Case_Table'First).Choice_Hi);

      --  Build the node for any missing choices that are smaller than any
      --  explicit choices given in the variant.

      if UI_Lt (Expr_Value (Exp_Lo), Lo) then
         Append (Build_Choice (Expr_Value (Exp_Lo), Lo), Choice_List);
      end if;

      --  Build the nodes representing any missing choices that lie between
      --  the explicit ones given in the variant.

      for J in Case_Table'First + 1 .. Case_Table'Last loop
         Lo := Expr_Value (Case_Table (J).Choice_Lo);
         Hi := Expr_Value (Case_Table (J).Choice_Hi);
         Choice := Case_Table (J).Choice_Node;

         if UI_Ne (Lo, UI_Sum (Previous_Hi, Uint_1)) then
            Append_To (Choice_List,
              Build_Choice (UI_Sum (Previous_Hi, Uint_1), Lo));
         end if;

         Previous_Hi := Hi;
      end loop;

      --  Build the node for any missing choices that are greater than any
      --  explicit choices given in the variant.

      if UI_Gt (Expr_Value (Exp_Hi), Hi) then
         Append (Build_Choice (UI_Sum (Hi, Uint_1), Expr_Value (Exp_Hi)),
                 Choice_List);
      end if;

      Set_Others_Discrete_Choices (Others_Choice, Choice_List);
   end Expand_Others_Choice;

   ------------------------------------
   -- Build_Discriminant_Constraints --
   ------------------------------------

   function Build_Discriminant_Constraints
     (T           : Entity_Id;
      Def         : Node_Id;
      Related_Nod : Node_Id)
      return        Elist_Id
   is
      C          : Node_Id := Constraint (Def);
      Discr_Expr : array (1 .. Number_Discriminants (T)) of Node_Id;
      Discr      : Entity_Id;
      E          : Entity_Id;
      Elist      : Elist_Id := New_Elmt_List;
      Position   : Nat := 1;
      Id         : Entity_Id;
      Id2        : Entity_Id;
      N          : Node_Id;
      Not_Found  : Boolean;

      function Pos_Of_Discr (T : Entity_Id; Discr : Entity_Id) return Nat;
      --  Return the Position number (starting at 1) of a discriminant
      --  (Discr) within the discriminant list of the record type (T).

      function Pos_Of_Discr (T : Entity_Id; Discr : Entity_Id) return Nat is
         J : Nat := 1;
         D : Entity_Id;

      begin
         D := First_Discriminant (T);

         while Present (D) loop
            if D = Discr then
               return J;
            end if;

            D := Next_Discriminant (D);
            J := J + 1;
         end loop;

         --  Note: Since this function is called on discriminants that are
         --  known to belong to the record type, falling through the loop
         --  with no match signals an internal compiler error.

         pragma Assert (False);
      end Pos_Of_Discr;

   begin
      for J in Discr_Expr'range loop
         Discr_Expr (J) := Empty;
      end loop;

      Discr := First_Discriminant (T);
      N := First (Constraints (C));

      --  The following loop will process the positional associations only
      --  and will exit when a named association is seen. The named
      --  associations will then be processed by the subsequent loop.

      while Present (N) loop
         exit when Nkind (N) = N_Discriminant_Association; -- Named Assoc

         --  For a positional association, the (single) discriminant is
         --  implicitly specified by position, in textual order. [LRM 3.7.2]

         if No (Discr) then
            Error_Msg_N ("too many constraints given for record", C);
            return Elist;

         elsif Nkind (N) = N_Range then
            Error_Msg_N
              ("a range is not a valid discriminant constraint", N);
            Discr_Expr (Position) := Error;
            Position := Position + 1;
            Discr := Next_Discriminant (Discr);

         else
            Analyze (N);
            Discr_Expr (Position) := N;
            Resolve (N, Base_Type (Etype (Discr)));
            Position := Position + 1;
            Discr := Next_Discriminant (Discr);

            if Present (Related_Nod)
              and then not Is_Static_Expression (N)
            then
               Set_Has_Dynamic_Itype (Related_Nod);
            end if;
         end if;

         N := Next (N);
      end loop;

      --  There should only be named associations left on the discriminant
      --  constraint. Any positional assoication are in error.

      while Present (N) loop

         if Nkind (N) = N_Discriminant_Association then
            E := Empty;
            Analyze (Expression (N));

            --  Search the entity list of the record looking at only the
            --  discriminants (which always appear first) to see if the
            --  simple name given in the constraint matches any of them.

            Id := First (Selector_Names (N));

            while Present (Id) loop
               Not_Found := True;
               Id2 := First_Entity (T);

               while Present (Id2)
                 and then Ekind (Id2) = E_Discriminant
               loop
                  if Chars (Id2) = Chars (Id) then
                     Not_Found := False;
                     exit;
                  end if;

                  Id2 := Next_Entity (Id2);
               end loop;

               if Not_Found then
                  Error_Msg_N ("& does not match any discriminant", N);
                  return Elist;
               end if;

               Position := Pos_Of_Discr (T, Id2);

               if No (Discr_Expr (Position)) then
                  Discr_Expr (Position) := Expression (N);
                  Resolve (Expression (N), Base_Type (Etype (Id2)));
               else
                  Error_Msg_N
                    ("duplicate constraint for discriminant&", Id);
               end if;

               --  A discriminant association with more than one
               --  discriminant name is only allowed if the named
               --  discriminants are all of the same type ... [LRM 3.7.2]

               if E = Empty then
                  E := Etype (Id2);

               elsif Etype (Id2) /= E then
                  Error_Msg_N ("all discriminants in an association " &
                               "must have the same type", N);
               end if;

               Id := Next (Id);
            end loop;

         else
            --  Positional Association

            --  Named associations can be given in any order, but if both
            --  positional and named associations are used in the same
            --  discriminant constraint, then positional associations must
            --  occur first, at their normal position. Hence once a named
            --  association is used, the rest of the discriminant constraint
            --  must use only named associations.

            Error_Msg_N ("positional association follows named one", N);
            return Elist;
         end if;

         N := Next (N);
      end loop;

      --  Furthermore, for each discriminant association (whether named or
      --  positional), the expression and the associated discriminants must
      --  have the same type. A discriminant constraint must provide exactly
      --  one value for each discriminant of the type. [LRM 3.7.2]

      --  missing code here???

      for I in Discr_Expr'Range loop
         if No (Discr_Expr (I)) then
            Error_Msg_N ("too few constraints given for record", C);
            return Elist;
         end if;
      end loop;

      --  Build an element list consisting of the expressions given in the
      --  discriminant constraint. The list is constructed after resolving
      --  any named discriminant associations and therefore the expressions
      --  appear in the textual order of the discriminants.

      Discr := First_Discriminant (T);

      for I in Discr_Expr'Range loop
         Append_Elmt (Discr_Expr (I), Elist);

         --  If any of the discriminant constraints is given by a discriminant
         --  the context may be a derived type derivation that renames them.
         --  Establish link between new and old discriminant.

         if Is_Entity_Name (Discr_Expr (I))
            and then Ekind (Entity (Discr_Expr (I))) = E_Discriminant
         then
            Set_Corresponding_Discriminant (Entity (Discr_Expr (I)), Discr);
         end if;

         Discr := Next_Discriminant (Discr);
      end loop;

      return Elist;
   end Build_Discriminant_Constraints;

   ----------------------------------
   -- Constrain_Discriminated_Type --
   ----------------------------------

   procedure Constrain_Discriminated_Type
   (Def_Id      : Entity_Id;
    S           : Node_Id;
    Related_Nod : Node_Id)
   is
      T     : Entity_Id;
      C     : Node_Id;
      N     : Node_Id;
      Elist : Elist_Id;
      Discr : Entity_Id;
      Num_D : Nat := 0;

   begin
      C := Constraint (S);

      if Nkind (C) /= N_Index_Or_Discriminant_Constraint then
         Error_Msg_N ("incorrect constraint given for record type", C);
         return;
      end if;

      --  A discriminant constraint is only allowed in a subtype indication,
      --  after a subtype mark. This subtype mark must denote either a type
      --  with discriminants, or an access type whose designated type is a
      --  type with discriminants. A discriminant constraint specifies the
      --  values of these discriminants. [LRM 3.7.2 (5)]

      T := Base_Type (Entity (Subtype_Mark (S)));

      if Ekind (T) in Access_Kind then
         T := Designated_Type (T);
      end if;

      if Is_Constrained (T) then
         Error_Msg_N
           ("record type is already constrained", Subtype_Mark (S));
         return;

      elsif not Has_Discriminants (T) then
         Error_Msg_N
           ("invalid constraint: record type has no discriminant", C);
         Set_Etype (Def_Id, Any_Type);
         return;
      end if;

      if Is_Itype (Def_Id) then
         Elist := Build_Discriminant_Constraints (T, S, Related_Nod);
      else
         Elist := Build_Discriminant_Constraints (T, S, Empty);
      end if;

      if Ekind (T) = E_Record_Type then
         Set_Ekind (Def_Id, E_Record_Subtype);

      elsif Ekind (T) = E_Task_Type then
         Set_Ekind (Def_Id, E_Task_Subtype);

      elsif Ekind (T) = E_Protected_Type then
         Set_Ekind (Def_Id, E_Protected_Subtype);

      elsif Is_Private_Type (T) then
         Set_Ekind (Def_Id, Subtype_Kind (Ekind (T)));

      else
         --  Incomplete type.

         Set_Ekind (Def_Id, Ekind (T));
      end if;

      Set_Etype                   (Def_Id, T);
      Set_Is_Tagged_Type          (Def_Id, Is_Tagged_Type (T));
      Set_Esize                   (Def_Id, Uint_0);
      Set_Has_Discriminants       (Def_Id);
      Set_Is_Constrained          (Def_Id);
      Set_First_Entity            (Def_Id, First_Entity (T));
      Set_Last_Entity             (Def_Id, Last_Entity (T));
      Set_Discriminant_Constraint (Def_Id, Elist);
      Set_Has_Tasks               (Def_Id, Has_Tasks (T));
      Set_Has_Controlled          (Def_Id, Has_Controlled (T));

      if Is_Tagged_Type (T) then
         Set_Primitive_Operations (Def_Id, Primitive_Operations (T));
         Set_Is_Controlled (Def_Id, Is_Controlled (T));
      end if;

      --  Subtypes introduced by component declarations do not need to be
      --  marked as delayed, and do not get freeze nodes, because the semantics
      --  verifies that the parents of the subtypes are frozen before the
      --  enclosing record is frozen.

      if not Is_Type (Scope (Def_Id)) then
         Set_Depends_On_Private (Def_Id, Depends_On_Private (T));

         if Is_Private_Type (T)
           and then Present (Full_View (T))
         then
            Set_Is_Delayed (Def_Id,
              Is_Delayed (Full_View (T))
                and then not Is_Frozen (Full_View (T)));
         else
            Set_Is_Delayed (Def_Id,
              Is_Delayed (T) and then not Is_Frozen (T));
         end if;
      end if;

   end Constrain_Discriminated_Type;

   ------------------------
   -- Check_Discriminant --
   ------------------------

   --  Within a record type definition the only allowed uses of the name
   --  of a discriminant of the record type are: in the default expressions
   --  for record components; in a variant part as the discriminant name;
   --  and in a component subtype definition, either as a bound in an index
   --  constraint, or to specify a discriminant value in a discriminant
   --  constraint. A discriminant name used in these component subtype
   --  definitions must appear by itself, not as part of a larger expression.
   --  [LRM 3.7.1]

   --  Verify that when a discriminant appears in an index constraint or a
   --  discriminant constraint of a record component, it appears by itself
   --  and not as part of a larger expression.
   --  Dummy procedure for now ???

   function Check_Discriminant (Exp_Node : Node_Id) return Boolean is
   begin
      return True;
   end Check_Discriminant;

   ------------------------------
   -- Derived_Type_Declaration --
   ------------------------------

   procedure Derived_Type_Declaration (T : Entity_Id; N : Node_Id) is
      Def             : constant Node_Id := Type_Definition (N);
      Derived_Type    : Entity_Id;
      Derived_Subtype : Entity_Id;
      Indic           : constant Node_Id := Subtype_Indication (Def);
      Extension       : constant Node_Id := Record_Extension_Part (Def);
      Loc             : Source_Ptr := Sloc (N);
      Parent_Type     : Entity_Id;
      Parent_Subtype  : Entity_Id;
      Sub_Decl        : Node_Id;

   begin
      if Nkind (Indic) = N_Subtype_Indication then
         Find_Type (Subtype_Mark (Indic));
         Parent_Type := Entity (Subtype_Mark (Indic));

         --  Otherwise we have a subtype mark without a constraint

      else
         Find_Type (Indic);
         Parent_Type := Entity (Indic);
      end if;

      if Parent_Type = Any_Type then
         Set_Etype (T, Any_Type);
         return;
      end if;

      --  In Ada 83, a derived type defined in a package specification cannot
      --  be used for further derivation until the end of its visible part.
      --  Note that derivation in the private part of the package is allowed.

      if (Ada_83 or Features_On)
        and then Is_Derived_Type (Parent_Type)
        and then In_Visible_Part (Scope (Parent_Type))
      then
         Note_Feature (Inheritance_At_Local_Derivation, Sloc (Indic));

         if Ada_83 then
            Error_Msg_N
              ("(Ada 83): premature use of type for derivation", Indic);
         end if;
      end if;

      --  Check for early use of incomplete or private type

      if (Is_Incomplete_Or_Private_Type (Parent_Type)
        and then not Is_Generic_Type (Parent_Type)
        and then No (Underlying_Type (Parent_Type)))
      then
         Error_Msg_N
           ("premature derivation of derived or private type", Indic);
      end if;

      if Present (Extension) and then not Is_Tagged_Type (Parent_Type) then
         Error_Msg_N
           ("a type derived from a non tagged type cannot have an extension",
            Indic);

      elsif No (Extension) and then Is_Tagged_Type (Parent_Type) then
         Error_Msg_N (
           "a type derived from a tagged type must have an extension",
           Indic);
      end if;

      --  If subtype indication, constraint to be applied later to derived type
      --  unless declaration has a discriminant part, in  which case the
      --  constraint on the parent type does not make the derived type into a
      --  constrained type.

      if Nkind (Indic) = N_Subtype_Indication
        and then No (Discriminant_Specifications (N))
        and then not (Is_Tagged_Type (Parent_Type))
        and then Ekind (Parent_Type) not in Scalar_Kind
      then

         if Is_Record_Type (Parent_Type) then

            --  the implicit type is not attached because it will be replaced
            --  by a brand new non-implicit type later in the analysis

            Derived_Type := New_Itype_Not_Attached (E_Void, Sloc (N), T, 'B');

         else
            Derived_Type := New_Itype (E_Void, N,  T, 'B');
         end if;

         --  If a constraint is present, add subtype declaration for the
         --  derived subtype of anonymous parent we just created.
         --  Numeric types are treated specially in the routine
         --  Build_Derived_Numeric_Type. If the parent type is tagged, the
         --  _parent component of the derived type is constrained, not the
         --  derived type itself.

         Sub_Decl :=
           Make_Subtype_Declaration (Loc,
              Defining_Identifier => New_Copy (T),
              Subtype_Indication =>
                Make_Subtype_Indication (Loc,
                  Subtype_Mark => (New_Occurrence_Of (Derived_Type, Loc)),
                  Constraint => Constraint (Indic)));
         Insert_After (N, Sub_Decl);

      else
         Derived_Type := T;
      end if;

      Build_Derived_Type (N, Parent_Type, Derived_Type);
      Derive_Subprograms (Parent_Type, Derived_Type);

   end Derived_Type_Declaration;

   ------------------------
   -- Build_Derived_Type --
   ------------------------

   procedure Build_Derived_Type
     (N            : Node_Id;
      Parent_Type  : Entity_Id;
      Derived_Type : Entity_Id)
   is
      Indic : constant Node_Id := Subtype_Indication (Type_Definition (N));
      Implicit_Base  : Entity_Id;
      T              : Entity_Id;

   begin
      --  Copy common attributes

      Set_Ekind            (Derived_Type, Ekind (Base_Type (Parent_Type)));
      Set_Esize            (Derived_Type, Esize (Parent_Type));
      Set_Alignment_Clause (Derived_Type, Alignment_Clause (Parent_Type));
      Set_Etype            (Derived_Type, Base_Type (Parent_Type));
      Set_Scope            (Derived_Type, Current_Scope);

      case Ekind (Parent_Type) is
         when Numeric_Kind =>
            Build_Derived_Numeric_Type (N, Parent_Type, Derived_Type);

         when Array_Kind =>
            Set_First_Index    (Derived_Type, First_Index    (Parent_Type));
            Set_Component_Type (Derived_Type, Component_Type (Parent_Type));
            Set_Is_Constrained (Derived_Type, Is_Constrained (Parent_Type));

         when E_Record_Type | E_Record_Subtype =>
            if Is_Tagged_Type (Parent_Type) then
               Build_Derived_Tagged_Type (N,
                 Type_Definition (N), Parent_Type, Derived_Type);
            else
               Build_Derived_Record_Type (N, Parent_Type, Derived_Type);
            end if;

         when Class_Wide_Kind =>
            Build_Derived_Record_Type (N, Parent_Type, Derived_Type);

         when Enumeration_Kind =>
            Build_Derived_Enumeration_Type (N, Parent_Type, Derived_Type);

         when Access_Kind =>
            Set_Directly_Designated_Type
              (Derived_Type, Designated_Type (Parent_Type));
            Set_Is_Access_Constant (Derived_Type,
                Is_Access_Constant (Parent_Type));

         when Incomplete_Or_Private_Kind =>
            if Is_Tagged_Type (Parent_Type) then
               Build_Derived_Tagged_Type (N,
                 Type_Definition (N), Parent_Type, Derived_Type);
            else
               Set_Is_Constrained  (Derived_Type,
                 Is_Constrained (Parent_Type));
            end if;
         when others =>
            pragma Assert (False); null;
      end case;
   end Build_Derived_Type;

   --------------------------------
   -- Build_Derived_Numeric_Type --
   --------------------------------

   procedure Build_Derived_Numeric_Type
     (N            : Node_Id;
      Parent_Type  : Entity_Id;
      Derived_Type : Entity_Id)
   is
      Indic : constant Node_Id := Subtype_Indication (Type_Definition (N));
      Implicit_Base  : Entity_Id;
      T              : Entity_Id;
   begin

      --  Process the subtype indication including a validation check on
      --  the constraint if any.

      T := Process_Subtype (Indic, N);

      --  Introduce an implicit base type for the derived type even if
      --  there is no constraint attached to it, since this seems closer
      --  to the Ada semantics.

      Implicit_Base :=
        New_Itype (Ekind (Base_Type (Parent_Type)), N, Derived_Type, 'B');

      Set_Etype (Implicit_Base, Parent_Type);
      Set_Esize (Implicit_Base, Esize (Parent_Type));
      Set_Alignment_Clause
                (Implicit_Base, Alignment_Clause (Parent_Type));

      Set_Scalar_Range    (Implicit_Base, Scalar_Range (Parent_Type));

      --  Make "Derived_Type", which is the defining identifier of the
      --  derived type declaration, be a subtype of the introduced
      --  implicit base type. In the case where there is no constraint
      --  given the Ekind will have to be set here since it is not set
      --  by Process_Subtype.

      Set_Etype (Derived_Type, Implicit_Base);
      Set_Parent_Subtype (Derived_Type, Implicit_Base);

      if Nkind (Indic) /= N_Subtype_Indication then
         Set_Ekind (Derived_Type, Subtype_Kind (Ekind (Parent_Type)));
         Set_Scalar_Range (Derived_Type, Scalar_Range (Parent_Type));
      end if;

      if Is_Modular_Integer_Type (Parent_Type) then
         Set_Modulus (Implicit_Base, Modulus (Parent_Type));
         Set_Modulus (Derived_Type, Modulus (Parent_Type));

      elsif Is_Floating_Point_Type (Parent_Type) then
         Set_Digits_Value (Derived_Type, Digits_Value (Parent_Type));

      elsif Is_Fixed_Point_Type (Parent_Type) then
         Set_Delta_Value (Derived_Type, Delta_Value (Parent_Type));
         Set_Corresponding_Integer_Type (
           Derived_Type, Corresponding_Integer_Type (Parent_Type));
      end if;
   end Build_Derived_Numeric_Type;

   ------------------------------------
   -- Build_Derived_Enumeration_Type --
   ------------------------------------

   procedure Build_Derived_Enumeration_Type
     (N            : Node_Id;
      Parent_Type  : Entity_Id;
      Derived_Type : Entity_Id)
   is
      Implicit_Base : Entity_Id;
      Def           : constant Node_Id    := Type_Definition (N);
      Indic         : constant Node_Id    := Subtype_Indication (Def);
      Loc           : constant Source_Ptr := Sloc (N);
      Literal       : Entity_Id;
      Literals_List : List_Id;
      Type_Decl     : Node_Id;
      I_Node        : Node_Id;
   begin
      --  Since types Standard.Character and Standard.Wide_Character do
      --  not have explicit literals lists we need to process types derived
      --  from them specially. This is handled by Derived_Standard_Character.

      if Root_Type (Parent_Type) = Standard_Character
        or else Root_Type (Parent_Type) = Standard_Wide_Character
      then
         Derived_Standard_Character (N, Parent_Type, Derived_Type);
         return;
      end if;

      --  Introduce an implicit base type for the derived type even
      --  if there is no constraint attached to it, since this seems
      --  closer to the Ada semantics. Build a full type declaration
      --  tree for the derived type using the implicit base type as
      --  the defining identifier. The build a subtype declaration
      --  tree which applies the constraint (if any) have it replace
      --  the derived type declaration.

      Literal := First_Literal (Parent_Type);
      Literals_List := New_List;

      while Present (Literal)
        and then Ekind (Literal) = E_Enumeration_Literal
      loop
         Append (New_Copy (Literal), Literals_List);
         Literal := Next_Literal (Literal);
      end loop;

      Implicit_Base :=
        Make_Defining_Identifier (Loc,
          New_External_Name (Chars (Derived_Type), 'B'));

      Type_Decl :=
        Make_Full_Type_Declaration (Loc,
          Defining_Identifier => Implicit_Base,
          Discriminant_Specifications => No_List,
          Type_Definition =>
            Make_Enumeration_Type_Definition (Loc, Literals_List));

      Mark_Rewrite_Insertion (Type_Decl);
      Insert_Before (N, Type_Decl);
      Analyze (Type_Decl);

      --  After the implicit base is analyzed its Etype needs to be
      --  changed to reflect the fact that it is derived from the
      --  parent type which was ignored during analysis.

      Set_Etype (Implicit_Base, Parent_Type);

      --  Process the subtype indication including a validation check
      --  on the constraint if any.

      if Nkind (Indic) = N_Subtype_Indication then
         I_Node :=
           Make_Subtype_Indication (Loc,
             Subtype_Mark => New_Occurrence_Of (Implicit_Base, Loc),
             Constraint => Constraint (Indic));
      else
         I_Node := New_Occurrence_Of (Implicit_Base, Loc);
      end if;

      Rewrite_Substitute_Tree (N,
        Make_Subtype_Declaration (Loc,
          Defining_Identifier => Derived_Type,
          Subtype_Indication => I_Node));
      Analyze (N);
   end Build_Derived_Enumeration_Type;

   -------------------------------
   -- Build_Derived_Record_Type --
   -------------------------------

   procedure Build_Derived_Record_Type
     (N            : Node_Id;
      Parent_Type  : Entity_Id;
      Derived_Type : Entity_Id)
   is
      Type_Def   : constant Node_Id := Type_Definition (N);
      Indic      : constant Node_Id := Subtype_Indication (Type_Def);
      Discs      : Elist_Id;



   begin
      --  A derived record type has the same fields and types as the parent.
      --  The declaration may have a discriminant part, in which case the
      --  new discriminants can be used to constrain discriminants of the
      --  parent. For non-tagged types this is the only legal use of new
      --  discriminants.

      if Present (Discriminant_Specifications (N)) then
         New_Scope (Derived_Type);
         Process_Discriminants (N);

         if  Nkind (Indic) = N_Subtype_Indication then
            Discs :=
              Build_Discriminant_Constraints (Parent_Type, Indic, Empty);
         end if;

         End_Scope;
      end if;

      Set_Is_Constrained (Derived_Type, Is_Constrained (Parent_Type));
      Set_Is_Limited_Record (Derived_Type, Is_Limited_Record (Parent_Type));
      Set_Has_Discriminants (Derived_Type, Has_Discriminants (Parent_Type));

      Rewrite_Substitute_Tree (N,
        New_Copy_With_Replacement (Parent (Parent_Type),
          Inherit_Components (N, Parent_Type, Derived_Type)));
   end Build_Derived_Record_Type;

   --------------------------------
   -- Derived_Standard_Character --
   --------------------------------

   procedure Derived_Standard_Character
     (N             : Node_Id;
      Parent_Type   : Entity_Id;
      Derived_Type  : Entity_Id)
   is
      Def           : constant Node_Id   := Type_Definition (N);
      Indic         : constant Node_Id   := Subtype_Indication (Def);
      Implicit_Base : constant Entity_Id :=
                        New_Itype
                          (E_Enumeration_Type, N, Parent_Type, 'B');

      Lo, Hi        : Node_Id;
      R_Node        : Node_Id;

   begin
      Set_Etype (Implicit_Base, Base_Type (Parent_Type));
      Set_Is_Character_Type (Implicit_Base, True);

      R_Node := New_Node (N_Range, Sloc (N));
      Set_Low_Bound (R_Node, New_Copy (Type_Low_Bound (Parent_Type)));
      Set_High_Bound (R_Node, New_Copy (Type_High_Bound (Parent_Type)));
      Set_Scalar_Range (Implicit_Base, R_Node);

      R_Node := New_Node (N_Range, Sloc (N));

      Set_Ekind (Derived_Type, E_Enumeration_Subtype);
      Set_Etype (Derived_Type, Implicit_Base);
      Set_Is_Character_Type (Derived_Type, True);

      if Nkind (Indic) = N_Subtype_Indication then
         Lo := New_Copy (Low_Bound (Range_Expression (Constraint (Indic))));
         Hi := New_Copy (High_Bound (Range_Expression (Constraint (Indic))));
      else
         Lo := New_Copy (Type_Low_Bound (Parent_Type));
         Hi := New_Copy (Type_High_Bound (Parent_Type));
      end if;

      Set_Low_Bound (R_Node, Lo);
      Set_High_Bound (R_Node, Hi);
      Set_Scalar_Range (Derived_Type, R_Node);

      Analyze (Lo);
      Analyze (Hi);
      Resolve (Lo, Derived_Type);
      Resolve (Hi, Derived_Type);

   end Derived_Standard_Character;

   -------------------------------
   -- Build_Derived_Tagged_Type --
   -------------------------------

   procedure Build_Derived_Tagged_Type
     (N            : Node_Id;
      Type_Def     : Node_Id;
      Parent_Type  : Entity_Id;
      Derived_Type : Entity_Id)
   is
      New_Decl   : Node_Id;
      Assoc_List : Elist_Id;
      Discs      : Elist_Id;

      Subtype_Indic_Case : constant Boolean :=
        Nkind (Subtype_Indication (Type_Def)) = N_Subtype_Indication;
   begin

      Set_Is_Tagged_Type (Derived_Type);
      Set_Primitive_Operations (Derived_Type, New_Elmt_List);
      New_Scope (Derived_Type);

      if Present (Discriminant_Specifications (N)) then
         if Is_Constrained (Parent_Type) or else Subtype_Indic_Case then
            Process_Discriminants (N);

         else
            --  RM 3.7 (13): if a known_discriminant_part is provided then
            --  the parent subtype shall be constrained

            Error_Msg_N ("unconstrained type not allowed in this context",
              Subtype_Indication (Type_Def));
         end if;

      else
         --  The derived type can only have inherited discriminants if the
         --  parent type is unconstrained

         if Is_Constrained (Parent_Type) or else Subtype_Indic_Case then
            Set_Has_Discriminants (Derived_Type, False);
         else
            Set_Has_Discriminants (Derived_Type, True);
         end if;
      end if;

      Set_Is_Constrained (Derived_Type, not Has_Discriminants (Derived_Type));
      Set_Is_Limited_Record (Derived_Type, (Is_Limited_Record (Parent_Type)));

      --  Analyze the extension

      if Nkind (N) = N_Private_Extension_Declaration then
         Set_Ekind (Derived_Type, E_Record_Type_With_Private);
      else
         Set_Ekind (Derived_Type, E_Record_Type);
         Assoc_List := Inherit_Components (N, Parent_Type, Derived_Type);
         Expand_Derived_Record (Derived_Type, Type_Def);
         Record_Type_Definition (
           Record_Extension_Part (Type_Def), Derived_Type);
      end if;

      End_Scope;

      --  The parent type is frozen for tagged types (RM 13.13(7)).

      Freeze_Before (N, Parent_Type);

      --  All tagged types defined in Ada.Finalization are controlled

      if Chars (Scope (Derived_Type)) = Name_Finalization
        and then Chars (Scope (Scope (Derived_Type))) = Name_Ada
        and then Scope (Scope (Scope (Derived_Type))) = Standard_Standard
      then
         Note_Feature (Controlled_Types, Sloc (Derived_Type));
         Set_Is_Controlled (Derived_Type);
      else
         Set_Is_Controlled (Derived_Type, Is_Controlled (Parent_Type));
      end if;

      Make_Class_Wide_Type (Derived_Type);
      Set_Is_Abstract (Derived_Type, Abstract_Present (Type_Def));

   end Build_Derived_Tagged_Type;

   ------------------------
   -- Inherit_Components --
   ------------------------

   function Inherit_Components
     (N            : Node_Id;
      Parent_Type  : Entity_Id;
      Derived_Type : Entity_Id)
      return Elist_Id
   is
      Assoc_List : Elist_Id := New_Elmt_List;
      Comp       : Entity_Id;
      New_Comp   : Entity_Id;
      D_Minal    : Entity_Id;
      New_Decl   : Node_Id;
      Old_Disc   : Entity_Id;
      Discr_Elmt : Elmt_Id;

      function Assoc (C : Entity_Id) return Entity_Id;
      --  This function searches the association list, and returns the entity
      --  that is associated with C. A matching entry is assumed to be present.

      procedure Inherit_Discriminant (Old_Disc : Entity_Id);
      --  Procedure to do discriminant inheritance processing for one discr

      function Assoc (C : Entity_Id) return Entity_Id is
         Elmt : Elmt_Id;

      begin
         Elmt := First_Elmt (Assoc_List);

         while Present (Elmt) loop

            if Node (Elmt) = C then
               return Node (Next_Elmt (Elmt));
            end if;

            Elmt := Next_Elmt (Elmt);
         end loop;

         return Empty;
      end Assoc;

      procedure Inherit_Discriminant (Old_Disc : Entity_Id) is
         D_Minal : Node_Id;

      begin
         New_Comp := New_Copy (Old_Disc);
         Append_Elmt   (Old_Disc, Assoc_List);
         Append_Elmt   (New_Comp, Assoc_List);
         Append_Entity (New_Comp, Derived_Type);

         D_Minal :=
           Make_Defining_Identifier
                (Sloc (N), New_External_Name (Chars (Old_Disc), 'D'));
         Set_Ekind (D_Minal, E_In_Parameter);
         Set_Etype (D_Minal, Etype (Old_Disc));

         Set_Discriminal (New_Comp, D_Minal);
      end Inherit_Discriminant;

   --  Start of processing for Inherit_Components

   begin
      Append_Elmt (Parent_Type,  Assoc_List);
      Append_Elmt (Derived_Type, Assoc_List);

      --  If the declaration has a discriminant part, the discriminants
      --  are already analyzed. If the parent type has discriminants,
      --  then some or all of them may correspond to the new discriminants.
      --  In  the case of untagged types, all of them must correspond.
      --  The correspondence determines the list of components that is built
      --  for the derived type. The discriminant part itself is not used
      --  further. It there are inherited discriminants, the discriminant
      --  part is incomplete,  but this does not affect subsequent expansion
      --  or translation in Gigi.

      if not Is_Tagged_Type (Parent_Type) then

         if Present (Discriminant_Specifications (N)) then
            New_Comp := First_Discriminant (Derived_Type);

            while Present (New_Comp) loop
               Old_Disc := Corresponding_Discriminant (New_Comp);

               if Present (Old_Disc) then
                  Append_Elmt (Old_Disc, Assoc_List);
                  Append_Elmt (New_Comp, Assoc_List);

               else
                  Error_Msg_N ("new discriminants must constrain old ones", N);
               end if;

               New_Comp := Next_Discriminant (New_Comp);
            end loop;

         elsif Has_Discriminants (Parent_Type) then

            --  Inherit all discriminants of parent.

            Old_Disc := First_Discriminant (Parent_Type);

            while Present (Old_Disc) loop
               Inherit_Discriminant (Old_Disc);
               Old_Disc := Next_Discriminant (Old_Disc);
            end loop;
         end if;

      else
         --  Parent type is tagged. Some of the discriminants may be
         --  renamed, some constrained, and some inherited.
         --  First we mark the renamed discriminants.  These renamed
         --  discriminants are not visible components of the derived
         --  type (3.4 (11)).

         if Present (Discriminant_Specifications (N)) then
            New_Comp := First_Discriminant (Derived_Type);

            while Present (New_Comp) loop
               Old_Disc := Corresponding_Discriminant (New_Comp);

               if Present (Old_Disc) then
                  Append_Elmt (Old_Disc, Assoc_List);
                  Append_Elmt (New_Comp, Assoc_List);
               end if;

               New_Comp := Next_Discriminant (New_Comp);
            end loop;
         end if;

         --  Next we inherit the discriminants of the parent which have
         --  not been renamed. If there is a discriminant constraint on
         --  the parent, the inherited components are not discriminants
         --  any longer, and cannot participate in  subsequent constraints
         --  on the derived type.

         if Has_Discriminants (Parent_Type) then
            Old_Disc := First_Discriminant (Parent_Type);

            while Present (Old_Disc) loop

               if No (Assoc (Old_Disc)) then
                  Inherit_Discriminant (Old_Disc);

                  if Is_Constrained (Parent_Type)
                    or else Nkind (Subtype_Indication (Type_Definition (N)))
                              = N_Subtype_Indication
                  then
                     Set_Ekind (New_Comp, E_Component);
                  end if;
               end if;

               Old_Disc := Next_Discriminant (Old_Disc);
            end loop;
         end if;

      end if;

      --  Finally, inherit non-discriminant components.

      Comp := First_Entity (Parent_Type);

      while Present (Comp) loop

         if Ekind (Comp) = E_Component
           and then Chars (Comp) /= Name_uParent
         then
            New_Comp := New_Copy (Comp);

            Append_Elmt   (Comp,     Assoc_List);
            Append_Elmt   (New_Comp, Assoc_List);
            Append_Entity (New_Comp, Derived_Type);
         end if;

         Comp := Next_Entity (Comp);
      end loop;

      return Assoc_List;

   end Inherit_Components;

   ---------------------
   -- Is_Derived_Type --
   ---------------------

   function Is_Derived_Type (Type_Id : Entity_Id) return Boolean is
   begin
      return Base_Type (Type_Id) /= Root_Type (Type_Id)
        and not Is_Generic_Type (Type_Id);
   end Is_Derived_Type;

   ---------------------
   --  Is_Null_Range --
   ---------------------

   function Is_Null_Range (Lo : Node_Id; Hi : Node_Id) return Boolean is
      Typ : Entity_Id := Etype (Lo);

   begin
      --  For discrete types, do the check against the bounds

      if Is_Discrete_Type (Typ) then
         return UI_Gt (Expr_Value (Lo), Expr_Value (Hi));

      --  For now, all other types are considered to be in range, TBSL ???

      else
         return False;
      end if;
   end Is_Null_Range;

   ---------------------
   -- In_Visible_Part --
   ---------------------

   function In_Visible_Part (Scope_Id : Entity_Id) return Boolean is
   begin
      return
        (Ekind (Scope_Id) = E_Package
            or else Ekind (Scope_Id) = E_Generic_Package)
          and then In_Open_Scopes (Scope_Id)
          and then not In_Package_Body (Scope_Id)
          and then not In_Private_Part (Scope_Id);
   end In_Visible_Part;

   ----------------------------------
   -- Collect_Primitive_Operations --
   ----------------------------------

   function Collect_Primitive_Operations (T : Entity_Id) return Elist_Id is
      B_Type  : constant Entity_Id := Base_Type (T);
      B_Scope : constant Entity_Id := Scope (B_Type);
      Op_List : Elist_Id  := New_Elmt_List;
      Formal  : Entity_Id;
      Is_Prim : Boolean;
      Id      : Entity_Id := Next_Entity (B_Type);

   begin
      --  For tagged types, the primitive operations are collected as they
      --  are declared, and held in an explicit list which is simply returned.

      if Is_Tagged_Type (B_Type) then
         Op_List := Primitive_Operations (B_Type);

      elsif B_Scope = Standard_Standard then
         null;

      elsif Ekind (B_Scope) /= E_Package
        and then not Is_Derived_Type (B_Type)
      then
         null;

      else

         while Present (Id) loop
            if Is_Overloadable (Id) then
               Is_Prim := False;

               if Base_Type (Etype (Id)) = B_Type then
                  Is_Prim := True;
               else
                  Formal := First_Formal (Id);
                  while Present (Formal) loop
                     if Base_Type (Etype (Formal)) = B_Type then
                        Is_Prim := True;
                        exit;
                     end if;

                     Formal := Next_Formal (Formal);
                  end loop;
               end if;

               if Is_Prim then
                  Append_Elmt (Id, Op_List);
               end if;
            end if;

            Id := Next_Entity (Id);

         end loop;

      end if;

      return Op_List;
   end Collect_Primitive_Operations;

   ------------------------
   -- Derive_Subprograms --
   ------------------------

   procedure Derive_Subprograms (Parent_Type, Derived_Type  : Entity_Id) is
      Op_List    : Elist_Id  := Collect_Primitive_Operations (Parent_Type);
      Elmt       : Elmt_Id;
      Subp       : Entity_Id;
      New_Subp   : Entity_Id;
      Formal     : Entity_Id;
      New_Formal : Entity_Id;

      procedure Replace_Type (Id, New_Id : Entity_Id);
      --  When the type is an anonymous access type, create a new access type
      --  designating the derived type. The implicit type mechanism doesn't
      --  need to be used because inherited subprograms are never used in Gigi.

      procedure Replace_Type (Id, New_Id : Entity_Id) is
         Acc_Type : Entity_Id;

      begin
         --  When the type is an anonymous access type, create a new access
         --  type designating the derived type. The implicit type mechanism
         --  doesn't need to be used because inherited subprograms are never
         --  used in Gigi.

         if Ekind (Etype (Id)) = E_Anonymous_Access_Type
           and then Base_Type (Designated_Type (Etype (Id)))
                  = Base_Type (Parent_Type)
         then
            Acc_Type := New_Copy (Etype (Id));
            Set_Etype (Acc_Type, Acc_Type);
            Set_Directly_Designated_Type (Acc_Type, Derived_Type);

            Set_Etype (New_Id, Acc_Type);

         elsif Base_Type (Etype (Id)) = Base_Type (Parent_Type) then
            Set_Etype (New_Id, Derived_Type);
         else
            Set_Etype (New_Id, Etype (Id));
         end if;
      end Replace_Type;

   --  Start of processing for Derive_Subprograms

   begin
      Elmt := First_Elmt (Op_List);

      while Present (Elmt) loop
         Subp := Node (Elmt);
         New_Subp := New_Entity (N_Defining_Identifier, Sloc (Derived_Type));
         Set_Ekind (New_Subp, Ekind (Subp));
         Set_Chars (New_Subp, Chars (Subp));
         Replace_Type (Subp, New_Subp);

         Formal := First_Formal (Subp);
         while Present (Formal) loop
            New_Formal := New_Copy (Formal);
            Append_Entity (New_Formal, New_Subp);
            Replace_Type (Formal, New_Formal);
            Formal := Next_Formal (Formal);
         end loop;

         Set_Alias (New_Subp, Subp);
         New_Overloaded_Entity (New_Subp);

         --  Indicate that a derived subprogram does not require a body.

         Set_Has_Completion (New_Subp);

         --  A derived function with a controlling result is abstract.

         if Is_Abstract (Subp)
           or else Etype (New_Subp) = Derived_Type
         then
            Set_Is_Abstract (New_Subp);
         end if;

         Elmt := Next_Elmt (Elmt);
      end loop;
   end Derive_Subprograms;

   -------------------------------------------
   -- Analyze_Private_Extension_Declaration --
   -------------------------------------------

   procedure Analyze_Private_Extension_Declaration (N : Node_Id) is
      Assoc       : Elist_Id;
      T           : constant Entity_Id := Defining_Identifier (N);
      Indic       : constant Node_Id   := Subtype_Indication (N);
      Parent_Type : Entity_Id;

   begin
      Enter_Name (T);

      Find_Type (Indic);
      Parent_Type := Entity (Indic);

      if not Is_Tagged_Type (Parent_Type) then
         Error_Msg_N
           ("parent of type extension must be a tagged type ", Indic);
         return;
      end if;

      if Ekind (Current_Scope) /= E_Package
        and then Ekind (Current_Scope) /= E_Generic_Package
      then
         Error_Msg_N ("invalid context for private extension", N);
      end if;
      Set_Is_Tagged_Type     (T);
      Set_Ekind              (T, E_Record_Type_With_Private);
      Set_Esize              (T, Uint_0);
      Set_Alignment_Clause   (T, Alignment_Clause (Parent_Type));
      Set_Etype              (T, Base_Type (Parent_Type));
      Set_Scope              (T, Current_Scope);
      Set_Depends_On_Private (T);
      Set_Is_Delayed         (T);

      Build_Derived_Tagged_Type (N, N, Parent_Type, T);
      Derive_Subprograms (Parent_Type, T);
   end Analyze_Private_Extension_Declaration;

   --------------------------
   -- Make_Class_Wide_Type --
   --------------------------

   procedure Make_Class_Wide_Type  (T : Entity_Id) is
      CW_Type : constant Entity_Id :=
                  New_External_Entity
                    (E_Void, Scope (T), Sloc (T), T, 'C', 0, 'T');
      CW_Name : Name_Id;
      Next_E  : Entity_Id;

   begin
      --  Inherit root type characteristics

      CW_Name := Chars (CW_Type);
      Next_E  := Next_Entity (CW_Type);
      Copy_Node (T, CW_Type);
      Set_Chars (CW_Type, CW_Name);
      Set_Next_Entity (CW_Type, Next_E);

      --  Customize the class-wide type: It has no prim. op., it cannot be
      --  abstract and its Etype points back to the root type

      Set_Ekind (CW_Type, E_Class_Wide_Type);
      Set_Primitive_Operations (CW_Type,  New_Elmt_List);
      Set_Is_Abstract (CW_Type, False);
      Set_Etype (CW_Type, T);
      Set_Is_Constrained (CW_Type, False);

      Set_Class_Wide_Type (T, CW_Type);

      --  The class-wide type of a class-wide type is itself (RM 3.9(14))

      Set_Class_Wide_Type (CW_Type, CW_Type);

   end Make_Class_Wide_Type;

   ----------------------------------
   -- Analyze_Incomplete_Type_Decl --
   ----------------------------------

   procedure Analyze_Incomplete_Type_Decl (N : Node_Id) is
      T : Node_Id;

   begin
      --  Process an incomplete declaration. The identifier must not have been
      --  declared already in the scope. However, an incomplete declaration may
      --  appear in the private part of a package, for a private type that has
      --  already been declared.

      --  In this case, the discriminants (if any) must match.

      T := Find_Type_Name (N);
      Set_Ekind (T, E_Incomplete_Type);
      Set_Etype (T, T);
      New_Scope (T);

      if Present (Discriminant_Specifications (N)) then
         Process_Discriminants (N);
      end if;

      End_Scope;
   end Analyze_Incomplete_Type_Decl;

   ----------------------------
   -- Access_Type_Declaration --
   ----------------------------

   procedure Access_Type_Declaration (T : Entity_Id; Def : Node_Id) is
      S : constant Node_Id := Subtype_Indication (Def);
      P : constant Node_Id := Parent (Def);

   begin
      --  Check for permissible use of incomplete type

      if Nkind (S) /= N_Subtype_Indication then
         Analyze (S);

         if Ekind (Entity (S)) = E_Incomplete_Type then
            Set_Directly_Designated_Type (T, Entity (S));
         else
            Set_Directly_Designated_Type (T,
              Process_Subtype (S, P, T, 'P'));
         end if;

      else
         Set_Directly_Designated_Type (T,
           Process_Subtype (S, P, T, 'P'));
      end if;

      if All_Present (Def) or Constant_Present (Def) then
         Set_Ekind (T, E_General_Access_Type);
      else
         Set_Ekind (T, E_Access_Type);
      end if;

      Set_Etype     (T, T);
      Set_Esize     (T, UI_From_Int (System_Address_Size));
      Set_Is_Access_Constant (T, Constant_Present (Def));

      --  Note that Has_Tasks is always false, since the access type itself
      --  is not a task type. See Einfo for more description on this point.
      --  Exactly the same consideration applies to Has_Controlled.

      Set_Has_Tasks      (T, False);
      Set_Has_Controlled (T, False);
   end Access_Type_Declaration;

   -----------------------------------
   -- Access_Subprogram_Declaration --
   -----------------------------------

   procedure Access_Subprogram_Declaration
     (T_Name : Entity_Id;
      T_Def  : Node_Id)
   is
      Formals    : constant List_Id   := Parameter_Specifications (T_Def);

      --  The attachment of the itype is delayed otherwise it would be at
      --  the beginning of the itype list which is incorrect in presence
      --  of access parameters.

      Desig_Type : constant Entity_Id :=
        New_Itype_Not_Attached (E_Subprogram_Type, Sloc (Parent (T_Def)));

   begin
      if Nkind (T_Def) = N_Access_Function_Definition then
         Analyze (Subtype_Mark (T_Def));
         Set_Etype (Desig_Type, Entity (Subtype_Mark (T_Def)));
      else
         Set_Etype (Desig_Type, Standard_Void_Type);
      end if;

      if Present (Formals) then
         New_Scope (Desig_Type);
         Process_Formals (Desig_Type, Formals, Parent (T_Def));
         End_Scope;
      end if;

      Attach_Itype_To (Parent (T_Def), Desig_Type);
      Check_Delayed_Subprogram (Desig_Type);

      Set_Ekind     (T_Name, E_Access_Subprogram_Type);
      Set_Etype     (T_Name, T_Name);
      Set_Esize     (T_Name, UI_From_Int (System_Address_Size));
      Set_Directly_Designated_Type (T_Name, Desig_Type);
   end Access_Subprogram_Declaration;

   ----------------------
   -- Constrain_Access --
   ----------------------

   procedure Constrain_Access
     (Def_Id      : in out Entity_Id;
      S           : Node_Id;
      Related_Nod : Node_Id)
   is
      T             : constant Entity_Id := Entity (Subtype_Mark (S));
      Desig_Type    : constant Entity_Id := Designated_Type (T);
      Desig_Subtype : Entity_Id := Empty;

   begin

      if Ekind (Desig_Type) = E_Array_Type
        or else Ekind (Desig_Type) = E_String_Type
      then
         Constrain_Array (Desig_Subtype, S, Related_Nod, Def_Id, 'P');

      elsif Ekind (Desig_Type) = E_Record_Type
        or else Ekind (Desig_Type) = E_Task_Type
        or else Ekind (Desig_Type) = E_Protected_Type
        or else Is_Private_Type (Desig_Type)
      then
         Desig_Subtype := New_Itype (E_Void, Related_Nod);
         Constrain_Discriminated_Type (Desig_Subtype, S, Related_Nod);
         if Is_Private_Type (Desig_Type) then
            Prepare_Private_Subtype_Completion (Desig_Subtype, Related_Nod);
         end if;

      else
         Error_Msg_N ("invalid constraint on access type", S);
         Set_Etype (Def_Id, Any_Type);
         return;
      end if;

      if No (Def_Id) then
         Def_Id := New_Itype (E_Access_Subtype, Related_Nod);
      else
         Set_Ekind (Def_Id, E_Access_Subtype);
      end if;

      Set_Etype                    (Def_Id, T);
      Set_Esize                    (Def_Id, Esize (T));
      Set_Directly_Designated_Type (Def_Id, Desig_Subtype);
      Set_Depends_On_Private       (Def_Id, Has_Private_Component (Def_Id));
      Set_Is_Access_Constant       (Def_Id, Is_Access_Constant (T));
   end Constrain_Access;

   -----------------------
   -- Access_Definition --
   -----------------------

   function Access_Definition
     (Related_Nod : Node_Id;
      N           : Node_Id)
      return        Entity_Id
   is
      Anon_Type : constant Entity_Id :=
        New_Itype (E_Anonymous_Access_Type, Related_Nod,
          Scope_Id => Scope (Current_Scope));

   begin
      if (Ekind (Current_Scope) = E_Entry
        or else Ekind (Current_Scope) = E_Entry_Family)
        and then Is_Task_Type (Etype (Scope (Current_Scope)))
      then
         Error_Msg_N ("Task entries cannot have access parameters", N);
      end if;

      Find_Type (Subtype_Mark (N));

      Set_Etype                    (Anon_Type, Anon_Type);
      Set_Directly_Designated_Type (Anon_Type, Entity (Subtype_Mark (N)));
      Set_Depends_On_Private       (Anon_Type,
                                     Has_Private_Component (Anon_Type));
      return Anon_Type;
   end Access_Definition;

   -------------------------
   -- New_Binary_Operator --
   -------------------------

   procedure New_Binary_Operator (Op_Name : Name_Id; Typ : Entity_Id) is
      Loc : constant Source_Ptr := Sloc (Typ);
      Op  : Entity_Id;

      function Make_Op_Formal (Typ, Op : Entity_Id) return Entity_Id;
      --  Create abbreviated declaration for the formal of a predefined
      --  Operator 'Op' of type 'Typ'

      function Make_Op_Formal (Typ, Op : Entity_Id) return Entity_Id is
         Formal : Entity_Id;

      begin
         Formal := New_Internal_Entity (E_In_Parameter, Op, Loc, 'P');
         Set_Etype (Formal, Typ);
         return Formal;
      end Make_Op_Formal;

   --  Start of processing for Make_Op_Formal

   begin
      Op :=  New_Internal_Entity (E_Operator, Current_Scope, Loc, 'F');

      Set_Etype                  (Op, Typ);
      Set_Chars                  (Op, Op_Name);
      Set_Homonym                (Op, Get_Name_Entity_Id (Op_Name));
      Set_Is_Immediately_Visible (Op, True);
      Set_Is_Internal            (Op, True);
      Set_Has_Completion         (Op, True);
      Append_Entity              (Op, Current_Scope);

      Set_Name_Entity_Id (Op_Name, Op);

      Append_Entity (Make_Op_Formal (Typ, Op), Op);
      Append_Entity (Make_Op_Formal (Typ, Op), Op);

   end New_Binary_Operator;

   --------------------------------
   -- Process_Range_Expr_In_Decl --
   --------------------------------

   procedure Process_Range_Expr_In_Decl
     (R           : Node_Id;
      T           : Entity_Id;
      Related_Nod : Node_Id)
   is
      Lo : Node_Id;
      Hi : Node_Id;

   begin
      Analyze (R);
      Resolve (R, Base_Type (T));

      if Nkind (R) = N_Range then
         Lo := Low_Bound (R);
         Hi := High_Bound (R);
         Check_Non_Static_Context (Lo);
         Check_Non_Static_Context (Hi);

         --  Check_Non_Static_Context will check that the bounds are in the
         --  range of the base type. Here we check whether the bounds are in
         --  the range of the subtype itself. This is complicated by the
         --  fact that the bounds may represent the null range in which case
         --  the Constraint_Error exception should not be raised.



         if Is_Static_Expression (Lo) and then Is_Static_Expression (Hi) then

            if Is_Static_Subtype (T) then
               if not Is_Null_Range (Lo, Hi) then
                  if not Is_In_Range (Lo, T) then
                     Constraint_Error_Warning (Lo,
                       "static value out of range?!");
                  end if;
                  if not Is_In_Range (Hi, T) then
                     Constraint_Error_Warning (Hi,
                       "static value out of range?!");
                  end if;
               end if;
            end if;

         else
            --  One of the two expressions is not static
            if Present (Related_Nod) then
               Set_Has_Dynamic_Itype (Related_Nod);
            end if;
         end if;
      end if;
   end Process_Range_Expr_In_Decl;

   ------------------------------
   -- Static_Discriminant_Expr --
   ------------------------------

   function Static_Discriminant_Expr (L : List_Id) return Boolean is
      Discriminant_Spec : Node_Id := First (L);

   begin
      while Present (Discriminant_Spec) loop
         if Present (Expression (Discriminant_Spec))
           and then not Potentially_Static (Expression (Discriminant_Spec))
         then
            return False;
         end if;
         Discriminant_Spec := Next (Discriminant_Spec);
      end loop;
      return True;
   end Static_Discriminant_Expr;

end Sem_Ch3;


----------------------
-- REVISION HISTORY --
----------------------

--  ----------------------------
--  revision 1.440
--  date: Mon Aug 29 12:31:34 1994;  author: schonber
--  Make Collect_Primitive_Operations public.
--  (Analyze_Array_Type_Declaration): reset constrained flag explicitly in
--   the case of an unconstrained type (was incorrect if this was full view
--   of private type).
--  ----------------------------
--  revision 1.441
--  date: Mon Aug 29 23:42:29 1994;  author: dewar
--  Minor reformatting
--  ----------------------------
--  revision 1.442
--  date: Tue Aug 30 11:58:39 1994;  author: dewar
--  (Constant_Redeclaration): Check ALIASED present if present before
--  (Constant_Redeclaration): Change to error messages and comments
--  ----------------------------
--  New changes after this line.  Each line starts with: "--  "
