------------------------------------------------------------------------------
--                                                                          --
--                         GNAT COMPILER COMPONENTS                         --
--                                                                          --
--                              I T Y P E S                                 --
--                                                                          --
--                                 B o d y                                  --
--                                                                          --
--                            $Revision: 1.10 $                             --
--                                                                          --
--           Copyright (c) 1992,1993,1994 NYU, All Rights Reserved          --
--                                                                          --
-- GNAT is free software;  you can  redistribute it  and/or modify it under --
-- terms of the  GNU General Public License as published  by the Free Soft- --
-- ware  Foundation;  either version 2,  or (at your option) any later ver- --
-- sion.  GNAT is distributed in the hope that it will be useful, but WITH- --
-- OUT ANY WARRANTY;  without even the  implied warranty of MERCHANTABILITY --
-- or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License --
-- for  more details.  You should have  received  a copy of the GNU General --
-- Public License  distributed with GNAT;  see file COPYING.  If not, write --
-- to the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA. --
--                                                                          --
------------------------------------------------------------------------------

with Alloc;    use Alloc;
with Atree;    use Atree;
with Einfo;    use Einfo;
with Nlists;   use Nlists;
with Sem_Util; use Sem_Util;
with Sinfo;    use Sinfo;
with Stand;    use Stand;
with Table;

package body Itypes is

   -----------------------
   -- Local subprograms --
   -----------------------

   procedure Append_Itype (N : Node_Id; E : Entity_Id);
   --  Add the Itype E at the end of implicit types list attached to N. If
   --  E is the head of an implicit type list, the full list is appened.

   ------------------
   -- Append_Itype --
   ------------------

   procedure Append_Itype (N : Node_Id; E : Entity_Id) is
      Ityp : Entity_Id;

   begin
      pragma Assert (Nkind (N) not in N_Entity and Nkind (E) in N_Entity);

      if No (First_Itype (N)) then
         Set_First_Itype (N, E);
      else
         Ityp := First_Itype (N);
         while Present (Next_Itype (Ityp)) loop
            Ityp := Next_Itype (Ityp);
         end loop;

         Set_Next_Itype (Ityp, E);
      end if;
   end Append_Itype;

   ---------------------
   -- Attach_Itype_To --
   ---------------------

   procedure Attach_Itype_To (N : Node_Id; E : Entity_Id) is
   begin
      pragma Assert (Nkind (N) not in N_Entity and Nkind (E) in N_Entity);
      pragma Assert (Next_Itype (E) = Empty);
      Append_Itype (N, E);
   end Attach_Itype_To;

   --------------
   -- Is_Itype --
   --------------

   function Is_Itype (E : Entity_Id) return Boolean is
   begin
      --  The only entities which have no parents are the implicit types

      return Ekind (E) in Type_Kind and No (Parent (E));
   end Is_Itype;

   ---------------------------
   -- New_Itype_Not_Attched --
   ---------------------------

   function New_Itype_Not_Attached
     (Ekind        : Entity_Kind;
      Loc          : Source_Ptr;
      Related_Id   : Entity_Id := Empty;
      Suffix       : Character := ' ';
      Suffix_Index : Nat       := 0;
      Scope_Id     : Entity_Id := Current_Scope)
      return         Entity_Id
   is
      Typ : Entity_Id;

   begin
      if Related_Id = Empty then
         Typ := New_Internal_Entity (Ekind, Scope_Id, Loc, 'T');

         Set_Public_Status (Typ);
         --  This must surely be a bug, how can an internal name be public???

      else
         Typ := New_External_Entity
           (Ekind, Scope_Id, Loc, Related_Id, Suffix, Suffix_Index, 'T');
      end if;

      Set_Etype (Typ, Any_Type);
      return Typ;
   end New_Itype_Not_Attached;

   ---------------
   -- New_Itype --
   ---------------

   function New_Itype
     (Ekind        : Entity_Kind;
      In_Node      : Node_Id;
      Related_Id   : Entity_Id   := Empty;
      Suffix       : Character   := ' ';
      Suffix_Index : Nat         := 0;
      Scope_Id     : Entity_Id   := Current_Scope)
     return         Entity_Id
   is
      Typ : constant Entity_Id :=
        New_Itype_Not_Attached (Ekind, Sloc (In_Node),
          Related_Id, Suffix, Suffix_Index, Scope_Id);

   begin
      Attach_Itype_To (In_Node, Typ);
      return Typ;
   end New_Itype;

   ---------------------
   -- Transfer_Itypes --
   ---------------------

   procedure Transfer_Itypes (From : Node_Id; To : Node_Id) is
   begin
      pragma Assert (Nkind (From) not in N_Entity
                       and Nkind (To) not in N_Entity);

      if From /= To
        and then Nkind (From) in N_Has_Itypes
        and then Present (First_Itype (From))
      then
         Append_Itype (To, First_Itype (From));
         Set_First_Itype (From, Empty);
         if Has_Dynamic_Itype (From) then
            Set_Has_Dynamic_Itype (To,   True);
            Set_Has_Dynamic_Itype (From, False);
         end if;
      end if;
   end Transfer_Itypes;

end Itypes;


----------------------
-- REVISION HISTORY --
----------------------

--  ----------------------------
--  revision 1.8
--  date: Mon Jun 27 01:16:18 1994;  author: dewar
--  (New_Itype): Remove call for Set_Public_Status for external name
--   case (now done in New_External_Entity), keep the Set_Public_Status
--   call for the internal name and flag as suspicious with ???
--  ----------------------------
--  revision 1.9
--  date: Sun Aug 28 08:48:49 1994;  author: comar
--  (New_Itype): add the Ekind as firstr parameter
--  (New_Itype_Not_Attached): new subprogram.
--  (Transfer_Itypes): also transfer Has_Dynamic_Itype flag
--  ----------------------------
--  revision 1.10
--  date: Mon Aug 29 23:42:06 1994;  author: dewar
--  Minor reformatting
--  ----------------------------
--  New changes after this line.  Each line starts with: "--  "
