------------------------------------------------------------------------------
--                                                                          --
--                         GNAT COMPILER COMPONENTS                         --
--                                                                          --
--                             E X P _ U T I L                              --
--                                                                          --
--                                 B o d y                                  --
--                                                                          --
--                            $Revision: 1.76 $                             --
--                                                                          --
--           Copyright (c) 1992,1993,1994 NYU, All Rights Reserved          --
--                                                                          --
-- GNAT is free software;  you can  redistribute it  and/or modify it under --
-- terms of the  GNU General Public License as published  by the Free Soft- --
-- ware  Foundation;  either version 2,  or (at your option) any later ver- --
-- sion.  GNAT is distributed in the hope that it will be useful, but WITH- --
-- OUT ANY WARRANTY;  without even the  implied warranty of MERCHANTABILITY --
-- or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License --
-- for  more details.  You should have  received  a copy of the GNU General --
-- Public License  distributed with GNAT;  see file COPYING.  If not, write --
-- to the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA. --
--                                                                          --
------------------------------------------------------------------------------

with Atree;    use Atree;
with Debug;    use Debug;
with Einfo;    use Einfo;
with Elists;   use Elists;
with Errout;   use Errout;
with Itypes;   use Itypes;
with Lib;      use Lib;
with Nlists;   use Nlists;
with Nmake;    use Nmake;
with Sem;      use Sem;
with Sem_Ch13; use Sem_Ch13;
with Sem_Res;  use Sem_Res;
with Sem_Util; use Sem_Util;
with Sinfo;    use Sinfo;
with Sinput;   use Sinput;
with Snames;   use Snames;
with Stand;    use Stand;
with Table;
with Tbuild;   use Tbuild;
with Ttypes;   use Ttypes;
with Uintp;    use Uintp;

package body Exp_Util is

   -----------------------
   -- Local Subprograms --
   -----------------------

   function Make_Literal_Range
     (Loc         : Source_Ptr;
      Literal_Typ : Entity_Id;
      Index_Typ   : Entity_Id)
      return        Node_Id;
   --  Produce a Range node whose bounds are:
   --    Index_Typ'first .. Index_Typ'First + Length (Literal_Typ)
   --  this is used for expanding declarations like X : String := "sdfgdfg";

   function Make_Subtype_From_Expr
     (N                  : Node_Id;
      Expression_Entity  : Entity_Id;
      Unconstrained_Type : Entity_Id)
      return               Node_Id;
   --  Creates an appropriate Subtype Indication for unconstrained object
   --  declarations. Unconstrained_Type can be an unc. array, an unc. record
   --  or a classwide type. The expression has to be transformed in order to
   --  be accessible through an Entity (Expression_Entity), See
   --  Expand_Subtype_From_Expr for detail on this transformation.

   ----------------
   -- Local Data --
   ----------------

   --  The following table is used to save values of the Expander_Active
   --  flag when they are saved by Expander_Mode_Save_And_Set. We use an
   --  extendible table (which is a bit of overkill) because it is easier
   --  than figuring out a maximum value or bothering with range checks!

   package Expander_Flags is new Table (
     Table_Component_Type => Boolean,
     Table_Index_Type     => Int,
     Table_Low_Bound      => 0,
     Table_Initial        => 32,
     Table_Increment      => 200,
     Table_Name           => "Expander_Flags");

   ------------------------
   -- Build_Runtime_Call --
   ------------------------

   function Build_Runtime_Call (Loc : Source_Ptr; RE : RE_Id) return Node_Id is
   begin
      return
        Make_Procedure_Call_Statement (Loc,
          Name => New_Reference_To (RTE (RE), Loc));
   end Build_Runtime_Call;

   ---------------------------
   -- Expander_Mode_Restore --
   ---------------------------

   procedure Expander_Mode_Restore is
   begin
      Expander_Active := Expander_Flags.Table (Expander_Flags.Last);
      Expander_Flags.Decrement_Last;

      if Errors_Detected /= 0 then
         Expander_Active := False;
      end if;
   end Expander_Mode_Restore;

   --------------------------------
   -- Expander_Mode_Save_And_Set --
   --------------------------------

   procedure Expander_Mode_Save_And_Set (Status : Boolean) is
   begin
      Expander_Flags.Increment_Last;
      Expander_Flags.Table (Expander_Flags.Last) := Expander_Active;
      Expander_Active := Status;
   end Expander_Mode_Save_And_Set;

   -------------------------------
   -- Expand_Class_Wide_Subtype --
   -------------------------------

   --  Create a record type used as an equivalent of any member of the class
   --  which takes its size from exp.

   --  Generate the following code:

   --   type Equiv_T is record
   --     _parent :  T (List of discriminant constaints taken from Exp);
   --     Ext__50 : Storage_Array (1 .. (Exp'size - Typ'size) / Storage_Unit);
   --   end Equiv_T;

   function Expand_Class_Wide_Subtype
     (N                 : Node_Id;
      Class_Type        : Entity_Id;
      Expression_Entity : Entity_Id)
      return              List_Id
   is
      Loc         : constant Source_Ptr := Sloc (N);
      Root_Type   : constant Entity_Id  := Etype (Class_Type);
      Equiv_Type  : Entity_Id;
      Range_Type  : Entity_Id;
      Str_Type    : Entity_Id;
      List_Def    : List_Id := Empty_List;
      Constr_Root : Entity_Id;
      Sizexpr     : Node_Id;
      Expr        : Node_Id;

   begin
      if not Has_Discriminants (Root_Type) then
         Constr_Root := Root_Type;
      else
         Constr_Root :=
           Make_Defining_Identifier (Loc, New_Internal_Name ('R'));

         --  subtype cstr__n is T (List of discr constraints taken from Exp)

         Append_To (List_Def,
           Make_Subtype_Declaration (Loc,
             Defining_Identifier => Constr_Root,
               Subtype_Indication =>
                 Make_Subtype_From_Expr (N, Expression_Entity, Root_Type)));
      end if;

      --  subtype rg__xx is Storage_Offset range
      --                           (Expr'size - typ'size) / Storage_Unit

      Range_Type := Make_Defining_Identifier (Loc, New_Internal_Name ('G'));

      Expr := New_Reference_To (Expression_Entity, Loc);

      if Is_Access_Type (Etype (Expression_Entity)) then
         Expr := Make_Explicit_Dereference (Loc, Prefix => Expr);
      end if;

      Sizexpr :=
        Make_Op_Subtract (Loc,
          Left_Opnd =>
            Make_Attribute_Reference (Loc,
              Prefix => Expr,
              Attribute_Name => Name_Size),
          Right_Opnd =>
            Make_Attribute_Reference (Loc,
              Prefix => New_Reference_To (Constr_Root, Loc),
              Attribute_Name => Name_Size));

      Set_Paren_Count (Sizexpr, 1);

      Append_To (List_Def,
        Make_Subtype_Declaration (Loc,
          Defining_Identifier => Range_Type,
          Subtype_Indication =>
            Make_Subtype_Indication (Loc,
              Subtype_Mark => New_Reference_To (RTE (RE_Storage_Offset), Loc),
              Constraint => Make_Range_Constraint (Loc,
                Range_Expression =>
                  Make_Range (Loc,
                    Low_Bound => Make_Integer_Literal (Loc, Uint_1),
                    High_Bound =>
                      Make_Op_Divide (Loc,
                        Left_Opnd => Sizexpr,
                        Right_Opnd => Make_Integer_Literal (Loc,
                          Intval =>
                            UI_From_Int (System_Storage_Unit))))))));

      --  subtype str__nn is Storage_Array (rg__x);

      Str_Type := Make_Defining_Identifier (Loc, New_Internal_Name ('S'));
      Append_To (List_Def,
        Make_Subtype_Declaration (Loc,
          Defining_Identifier => Str_Type,
          Subtype_Indication =>
            Make_Subtype_Indication (Loc,
              Subtype_Mark => New_Reference_To (RTE (RE_Storage_Array), Loc),
              Constraint =>
                Make_Index_Or_Discriminant_Constraint (Loc,
                  Constraints =>
                    New_List (New_Reference_To (Range_Type, Loc))))));

      --  type Equiv_T is record
      --    _parent : Tnn;
      --    E : Str_Type;
      --  end Equiv_T;

      Equiv_Type := Make_Defining_Identifier (Loc, New_Internal_Name ('T'));

      --  Avoid the generation of an init procedure

      Set_Is_Frozen (Equiv_Type);

      Append_To (List_Def,
        Make_Full_Type_Declaration (Loc,
          Defining_Identifier => Equiv_Type,
          Type_Definition =>
            Make_Record_Definition (Loc,
              Component_List => Make_Component_List (Loc,
                Component_Items => New_List (
                  Make_Component_Declaration (Loc,
                    Defining_Identifier =>
                      Make_Defining_Identifier (Loc, Name_uParent),
                    Subtype_Indication => New_Reference_To (Constr_Root, Loc)),
                  Make_Component_Declaration (Loc,
                    Defining_Identifier =>
                      Make_Defining_Identifier (Loc,
                        Chars => New_Internal_Name ('X')),
                    Subtype_Indication => New_Reference_To (Str_Type, Loc))),
                Variant_Part => Empty))));

      Set_Equivalent_Type (Class_Type, Equiv_Type);
      return List_Def;
   end Expand_Class_Wide_Subtype;

   ------------------------------
   -- Expand_Subtype_From_Expr --
   ------------------------------

   --  This function is applicable for both static and dynamic allocation of
   --  objects which are constrained by an initial expression. Basically it
   --  transforms an unconstrained subtype indication into a constrained one.
   --  The expression may also be transformed in certain cases in order to
   --  avoid multiple evaulation. In the static allocation case, the general
   --  scheme is :
   --     Val : T := Expr;
   --        is transformed into
   --     Val : Constrained_Subtype_of_T := Maybe_Modified_Expr;
   --
   --  Here are the main cases :
   --
   --  <if Expr is a Slice>
   --    Val : T ([Slice_Range (Expr)]) := Expr;
   --
   --  <elsif Expr is a String Literal>
   --    Val : T (T'First .. T'First + Length (string literal) - 1) := Expr;
   --
   --  <elsif Expr is Constrained>
   --    Val : Type_Of_Expr := Expr;
   --
   --  <elsif Expr is an entity_name>
   --    Val : T (contraints taken from Expr) := Expr;
   --
   --  <else>
   --    type Axxx is access all T;
   --    Rval : Axxx := Expr'ref;
   --    Val  : T (contraints taken from Rval) := Rval.all;
   --    ??? note: when the Expression is allocated in the secondary stack
   --              we could use it directly instead of copying it by declaring
   --              Val : T (...) renames Rval.all

   procedure Expand_Subtype_From_Expr
     (N             : Node_Id;
      Unc_Type      : Entity_Id;
      Subtype_Indic : Node_Id;
      Exp           : Node_Id)
   is
      Loc           : constant Source_Ptr := Sloc (N);
      Exp_Typ       : constant Entity_Id  := Etype (Exp);

      Ref_Node      : Node_Id;
      Ref_Type_Node : Node_Id;
      New_Exp       : Node_Id;
      Ref_Type      : Entity_Id;
      Ref_Id        : Entity_Id;
      Expanded_Code : List_Id := New_List;

   begin
      if not Expander_Active then
         return;
      end if;

      if Ekind (Exp_Typ) = E_Slice_Subtype then

         Rewrite_Substitute_Tree (Subtype_Indic,
           Make_Subtype_Indication (Loc,
             Subtype_Mark => New_Reference_To (Unc_Type, Loc),
             Constraint =>
               Make_Index_Or_Discriminant_Constraint (Loc,
                 Constraints => New_List (Slice_Range (Exp_Typ)))));

      elsif Ekind (Exp_Typ) = E_String_Literal_Subtype then

         Rewrite_Substitute_Tree (Subtype_Indic,
           Make_Subtype_Indication (Loc,
             Subtype_Mark => New_Reference_To (Unc_Type, Loc),
             Constraint =>
               Make_Index_Or_Discriminant_Constraint (Loc,
                 Constraints => New_List (
                   Make_Literal_Range (Loc,
                     Literal_Typ => Exp_Typ,
                     Index_Typ   => Etype (First_Index (Unc_Type)))))));

      elsif Is_Constrained (Exp_Typ)
        and then not Is_Class_Wide_Type (Unc_Type)
      then

         Rewrite_Substitute_Tree (Subtype_Indic,
           New_Reference_To (Exp_Typ, Loc));

      elsif Is_Entity_Name (Exp) then

         Rewrite_Substitute_Tree (Subtype_Indic,
           Make_Subtype_From_Expr (N,
             Expression_Entity  => Entity (Exp),
             Unconstrained_Type => Unc_Type));

      else
         --  Expand: type Axxx is access all Unc_Type;

         Ref_Type := Make_Defining_Identifier (Loc, New_Internal_Name ('A'));
         Ref_Type_Node :=
           Make_Full_Type_Declaration (Loc,
             Defining_Identifier => Ref_Type,
             Type_Definition =>
              Make_Access_To_Object_Definition (Loc,
                 All_Present => True,
                 Subtype_Indication => New_Reference_To (Unc_Type, Loc)));

         Insert_Before (N, Ref_Type_Node);
         Mark_Rewrite_Insertion (Ref_Type_Node);
         Analyze (Ref_Type_Node);

         --  Expand: Rval : Axxx := Expr'ref;

         Ref_Id :=
           Make_Defining_Identifier (Loc,
             Chars =>
               New_External_Name (Chars (Defining_Identifier (N)), 'R'));
         Set_Etype (Ref_Id, Ref_Type);

         Ref_Node := Make_Object_Declaration (Loc,
           Defining_Identifier => Ref_Id,
           Object_Definition   => New_Reference_To (Ref_Type, Loc),
           Expression          => Make_Reference (Loc, Relocate_Node (Exp)));

         Insert_Before (N, Ref_Node);
         Mark_Rewrite_Insertion (Ref_Node);
         Analyze (Ref_Node);

         Rewrite_Substitute_Tree (Subtype_Indic,
           Make_Subtype_From_Expr (N,
             Expression_Entity  => Ref_Id,
             Unconstrained_Type => Unc_Type));

         Rewrite_Substitute_Tree (Exp,
           Make_Explicit_Dereference (Loc,
             Prefix => New_Occurrence_Of (Ref_Id, Loc)));
         Set_Etype (Exp, Exp_Typ);
      end if;
   end Expand_Subtype_From_Expr;

   ------------------
   -- Find_Prim_Op --
   ------------------

   function Find_Prim_Op (T : Entity_Id; Name : Name_Id) return Entity_Id is
      Prim : Elmt_Id;
      Typ  : Entity_Id := T;

   begin
      if Is_Class_Wide_Type (Typ) then
         Typ := Etype (Typ);
      end if;

      Prim := First_Elmt (Primitive_Operations (Underlying_Type (Typ)));
      while Chars (Node (Prim)) /= Name loop
         Prim := Next_Elmt (Prim);
      end loop;

      return Node (Prim);
   end Find_Prim_Op;

   ------------------------------------
   -- Insert_List_Before_And_Analyze --
   ------------------------------------

   procedure Insert_List_Before_And_Analyze (N : Node_Id; L : List_Id) is
      Node : Node_Id;
      Lend : Node_Id;

   begin
      --  Capture first and last nodes in list

      Node := First (L);
      Lend := Last (L);

      --  Now do the insertion

      Insert_List_Before (N, L);

      --  The insertion does not change the Id's of any of the nodes in the
      --  list, and the are still linked, so we can simply loop from the
      --  first to the last to get them analyzed.

      loop
         Analyze (Node);
         exit when Node = Lend;
         Node := Next (Node);
      end loop;

   end Insert_List_Before_And_Analyze;

   --------------------------------
   -- Make_Constraints_From_Expr --
   --------------------------------

   --  1. if Expr is an uncontrained array expression, creates
   --    Unc_Type(Expr'first(1)..Expr'Last(1),..., Expr'first(n)..Expr'last(n))

   --  2. if Expr is a unconstrained discriminated type expression, creates
   --    Unc_Type(Expr.Discr1, ... , Expr.Discr_n)

   --  3. if Expr is class-wide, creates an implicit class wide subtype

   function Make_Subtype_From_Expr
     (N                  : Node_Id;
      Expression_Entity  : Entity_Id;
      Unconstrained_Type : Entity_Id)
      return Node_Id
   is
      Loc         : constant Source_Ptr := Sloc (N);
      List_Constr : List_Id := New_List;
      D           : Entity_Id;
      Elmt        : Elmt_Id;

   begin

      if Is_Array_Type (Unconstrained_Type) then
         for I in 1 .. Number_Dimensions (Unconstrained_Type) loop
            Append_To (List_Constr,
              Make_Range (Loc,
                Low_Bound =>
                  Make_Attribute_Reference (Loc,
                    Prefix => New_Occurrence_Of (Expression_Entity, Loc),
                    Attribute_Name => Name_First,
                    Expressions => New_List (
                      Make_Integer_Literal (Loc, Intval => UI_From_Int (I)))),
                High_Bound =>
                  Make_Attribute_Reference (Loc,
                    Prefix => New_Occurrence_Of (Expression_Entity, Loc),
                    Attribute_Name => Name_Last,
                    Expressions => New_List (
                      Make_Integer_Literal (Loc,
                        Intval => UI_From_Int (I))))));
         end loop;

      elsif Is_Class_Wide_Type (Unconstrained_Type) then
         declare
            Class_Wide_Subtype      : Entity_Id;
            Class_Wide_Subtype_Name : Name_Id;

         begin
            Class_Wide_Subtype :=
              New_Itype (E_Void, N, Unconstrained_Type, 'S');
            Set_Public_Status (Class_Wide_Subtype);
            Class_Wide_Subtype_Name := Chars (Class_Wide_Subtype);
            Copy_Node (Unconstrained_Type, Class_Wide_Subtype);
            Set_Chars (Class_Wide_Subtype, Class_Wide_Subtype_Name);
            Set_Ekind (Class_Wide_Subtype, E_Class_Wide_Subtype);
            Set_Next_Entity (Class_Wide_Subtype, Empty);

            if Expander_Active then
               Insert_List_Before_And_Analyze (N,
                 Expand_Class_Wide_Subtype (N,
                   Class_Wide_Subtype,
                   Expression_Entity));
            end if;

            return New_Occurrence_Of (Class_Wide_Subtype, Loc);
         end;

      else
         D := First_Discriminant (Unconstrained_Type);
         while (Present (D)) loop

            Append_To (List_Constr,
              Make_Selected_Component (Loc,
                Prefix => New_Occurrence_Of (Expression_Entity, Loc),
                Selector_Name => New_Reference_To (D, Loc)));

            D := Next_Discriminant (D);
         end loop;
      end if;

      return
        Make_Subtype_Indication (Loc,
          Subtype_Mark => New_Reference_To (Unconstrained_Type, Loc),
          Constraint   =>
            Make_Index_Or_Discriminant_Constraint (Loc,
              Constraints => List_Constr));
   end Make_Subtype_From_Expr;

   ------------------------
   -- Make_Literal_Range --
   ------------------------

   function Make_Literal_Range
     (Loc         : Source_Ptr;
      Literal_Typ : Entity_Id;
      Index_Typ   : Entity_Id)
      return        Node_Id
   is
   begin
         return
           Make_Range (Loc,
             Low_Bound =>
               Make_Attribute_Reference (Loc,
                 Prefix => New_Occurrence_Of (Index_Typ, Loc),
                 Attribute_Name => Name_First),
             High_Bound =>
               Make_Op_Subtract (Loc,
                  Left_Opnd =>
                    Make_Op_Add (Loc,
                      Left_Opnd =>
                        Make_Attribute_Reference (Loc,
                          Prefix => New_Occurrence_Of (Index_Typ, Loc),
                          Attribute_Name => Name_First),
                      Right_Opnd => Make_Integer_Literal (Loc,
                        String_Literal_Length (Literal_Typ))),
                  Right_Opnd => Make_Integer_Literal (Loc, Uint_1)));
   end Make_Literal_Range;

   ----------------------
   -- Make_Tagged_Copy --
   ----------------------
   --  Generate :
   --    declare
   --       subtype _A is Storage_Offset range (1 .. Address_Size)/Store_Unit;
   --                                            3 * Address_Size  <CTRL>
   --       subtype _B is Storage_Array (_A);
   --       type _C is record
   --          D : Tags.Tag;
   --          E : Finalizable_Ptr;   <CTRL>
   --          F : Finalizable_Ptr;   <CTRL>
   --          G : _B;
   --       end record;
   --   begin
   --      _C!(Lhr).G := _C!(Rhs).G;
   --   end;

   function Make_Tagged_Copy
     (Loc      : Source_Ptr;
      L_Entity : Entity_Id;
      R_Entity : Entity_Id;
      Typ      : Entity_Id)
      return     Node_Id
   is
      Decls      : constant List_Id    := New_List;
      Stmts      : constant List_Id    := New_List;
      Comp_List  : constant List_Id    := New_List;
      Ptrs_Size  : Node_Id;
      Sizexpr    : Node_Id;

      A : constant Node_Id
            := Make_Defining_Identifier (Loc, New_Internal_Name ('A'));
      B : constant Node_Id
            := Make_Defining_Identifier (Loc, New_Internal_Name ('B'));
      C : constant Node_Id
            := Make_Defining_Identifier (Loc, New_Internal_Name ('C'));

   begin
      --  Sizexpr : rhs'size - Standard_Address_Size
      --     or     rhs'size - 3 * Standard_Address_Size   <CTRL>
      --  (we assume all 3 pointers, the tag and the finalization pointers,
      --   are thin pointers)

      Ptrs_Size :=
        Make_Integer_Literal (Loc,
          Intval => UI_From_Int (System_Address_Size));

      if Is_Controlled (Typ) then
         Ptrs_Size :=
           Make_Op_Multiply (Loc,
             Left_Opnd =>
               Make_Integer_Literal (Loc, Intval => UI_From_Int (3)),
             Right_Opnd => Ptrs_Size);
      end if;

      Sizexpr :=
        Make_Op_Subtract (Loc,
          Left_Opnd =>
            Make_Attribute_Reference (Loc,
              Prefix => New_Ref_To_Expr (R_Entity, Loc),
              Attribute_Name => Name_Size),
          Right_Opnd => Ptrs_Size);

      Set_Paren_Count (Sizexpr, 1);

      --  Subtype _A is Storage_Offset range 1 .. size_exp/Storage_Unit;

      Append_To (Decls,
        Make_Subtype_Declaration (Loc,
          Defining_Identifier => A,
          Subtype_Indication =>
            Make_Subtype_Indication (Loc,
              Subtype_Mark =>
                New_Reference_To (RTE (RE_Storage_Offset), Loc),
              Constraint => Make_Range_Constraint (Loc,
                Range_Expression =>
                  Make_Range (Loc,
                    Low_Bound => Make_Integer_Literal (Loc, Uint_1),
                    High_Bound =>
                      Make_Op_Divide (Loc,
                        Left_Opnd => Sizexpr,
                        Right_Opnd => Make_Integer_Literal (Loc,
                          Intval =>
                            UI_From_Int (System_Storage_Unit))))))));

      --  subtype _B is Storage_Array (_A);

      Append_To (Decls,
        Make_Subtype_Declaration (Loc,
          Defining_Identifier => B,
          Subtype_Indication =>
            Make_Subtype_Indication (Loc,
              Subtype_Mark =>
                New_Reference_To (RTE (RE_Storage_Array), Loc),
              Constraint =>
                Make_Index_Or_Discriminant_Constraint (Loc,
                  Constraints =>
                    New_List (New_Reference_To (A, Loc))))));

      --  type _C is record
      --     D : Tags.Tag;
      --     E : Finalizable_Ptr;                            <CTRL>
      --     F : Finalizable_Ptr;                            <CTRL>
      --     G : _B;
      --  end record;

      Append_To (Comp_List,
        Make_Component_Declaration (Loc,
          Defining_Identifier => Make_Defining_Identifier (Loc, Name_D),
          Subtype_Indication  => New_Reference_To (RTE (RE_Tag), Loc)));

      if Is_Controlled (Typ) then

         Append_To (Comp_List,
            Make_Component_Declaration (Loc,
              Defining_Identifier =>
                Make_Defining_Identifier (Loc, Name_E),
              Subtype_Indication  =>
                New_Reference_To (RTE (RE_Finalizable_Ptr), Loc)));

         Append_To (Comp_List,
            Make_Component_Declaration (Loc,
              Defining_Identifier =>
                Make_Defining_Identifier (Loc, Name_F),
              Subtype_Indication  =>
                New_Reference_To (RTE (RE_Finalizable_Ptr), Loc)));
      end if;

      Append_To (Comp_List,
        Make_Component_Declaration (Loc,
          Defining_Identifier => Make_Defining_Identifier (Loc, Name_G),
          Subtype_Indication  => New_Reference_To (B, Loc)));

      Append_To (Decls,
        Make_Full_Type_Declaration (Loc,
          Defining_Identifier => C,
          Type_Definition =>
            Make_Record_Definition (Loc,
              Component_List =>
                Make_Component_List (Loc,
                  Component_Items => Comp_List,
                  Variant_Part => Empty))));

      --  _C!(Lhr).G := _C!(Rhs).G

      Append_To (Stmts,
        Make_Assignment_Statement (Loc,
          Name =>
            Make_Selected_Component (Loc,
              Prefix =>
                Make_Unchecked_Type_Conversion (Loc,
                  Subtype_Mark => New_Reference_To (C, Loc),
                  Expression => New_Ref_To_Expr (L_Entity, Loc)),
              Selector_Name => Make_Identifier (Loc, Name_G)),

         Expression =>
            Make_Selected_Component (Loc,
              Prefix =>
                Make_Unchecked_Type_Conversion (Loc,
                  Subtype_Mark => New_Reference_To (C, Loc),
                  Expression => New_Ref_To_Expr (R_Entity, Loc)),
              Selector_Name => Make_Identifier (Loc, Name_G))));

      --  This assignment must work even for constant target (it is
      --  used for initializing tagged object)

      Set_Assignment_OK (Name (First (Stmts)));

      return
        Make_Block_Statement (Loc,
          Identifier => Empty,
          Declarations => Decls,
          Handled_Statement_Sequence =>
            Make_Handled_Sequence_Of_Statements (Loc, Stmts));
   end Make_Tagged_Copy;

   ----------------------
   -- Is_Unconstrained --
   ----------------------

   function Is_Unconstrained (N : Node_Id) return Boolean is
      Typ : Entity_Id;

   begin
      Typ := Underlying_Type (Etype (N));

      return Present (Typ)
        and then (Has_Discriminants (Typ)
                  or else (Is_Array_Type (Typ)
                           and then not Is_Constrained (Typ)));
   end Is_Unconstrained;

   -------------------------------
   -- Prepare_Multi_Use_Of_Expr --
   -------------------------------

   procedure Prepare_Multi_Use_Of_Expr
     (Exp_In        :     Node_Id;
      Entity_Exp    : out Entity_Id;
      Expanded_Code : out List_Id)
   is
      Loc      : constant Source_Ptr := Sloc (Exp_In);
      Ref_Type : Entity_Id;

   begin
      if Is_Entity_Name (Exp_In) then
         Entity_Exp    := Entity (Exp_In);
         Expanded_Code := No_List;

      else
         Ref_Type := Make_Defining_Identifier (Loc, New_Internal_Name ('A'));
         Expanded_Code := New_List;

         Append_To (Expanded_Code,
           Make_Full_Type_Declaration (Loc,
             Defining_Identifier => Ref_Type,
             Type_Definition =>
              Make_Access_To_Object_Definition (Loc,
                 All_Present => True,
                 Subtype_Indication =>
                   New_Reference_To (Etype (Exp_In), Loc))));

         Entity_Exp := Make_Defining_Identifier (Loc, New_Internal_Name ('R'));
         Set_Etype (Entity_Exp, Ref_Type);

         Append_To (Expanded_Code,
           Make_Object_Declaration (Loc,
             Defining_Identifier => Entity_Exp,
             Object_Definition => New_Reference_To (Ref_Type, Loc),
             Expression => Make_Reference (Loc, Relocate_Node (Exp_In))));
      end if;
   end Prepare_Multi_Use_Of_Expr;

   ---------------------
   -- New_Ref_To_Expr --
   ---------------------

   function New_Ref_To_Expr
     (Entity_Exp : Entity_Id;
      Loc        : Source_Ptr)
      return       Node_Id
   is
   begin
      if Comes_From_Source (Entity_Exp) then
         return New_Reference_To (Entity_Exp, Loc);
      else
         return
           Make_Explicit_Dereference (Loc,
             Prefix => New_Reference_To (Entity_Exp, Loc));
      end if;
   end New_Ref_To_Expr;


   ------------------------
   -- Protect_Statements --
   ------------------------

   procedure Protect_Statements (N : Node_Id; E : Entity_Id) is
      Loc : constant Source_Ptr := Sloc (N);
      Stm : constant Node_Id    := Handled_Statement_Sequence (N);
      Dec : List_Id := No_List;

   begin
      --  If the existing statement sequence has no exception handlers, then
      --  all we need to do is to add the specified cleanup call. If there
      --  are exception handlers present, then we have to wrap an extra
      --  block around to hold the cleanup call because of the current rule
      --  that a block cannot have both a cleanup and exception handlers.


      Dec := Declarations (N);
      Set_Declarations (N, Empty_List);

      Set_Handled_Statement_Sequence (N,
        Make_Handled_Sequence_Of_Statements (Loc,
          Statements => New_List (
            Make_Block_Statement (Loc,
              Identifier => Empty,
              Declarations => Dec,
              Handled_Statement_Sequence => Stm)),
          Identifier => New_Occurrence_Of (E, Loc)));

   end Protect_Statements;

   ----------------------------
   -- Wrap_Cleanup_Procedure --
   ----------------------------

   procedure Wrap_Cleanup_Procedure (N : Node_Id) is
      Loc   : constant Source_Ptr := Sloc (N);
      Stseq : constant Node_Id    := Handled_Statement_Sequence (N);
      Stmts : constant List_Id    := Statements (Stseq);

   begin
      Prepend_To (Stmts, Build_Runtime_Call (Loc, RE_Abort_Defer));
      Append_To  (Stmts, Build_Runtime_Call (Loc, RE_Abort_Undefer));
   end Wrap_Cleanup_Procedure;

end Exp_Util;


----------------------
-- REVISION HISTORY --
----------------------

--  ----------------------------
--  revision 1.74
--  date: Thu Aug 18 16:26:20 1994;  author: comar
--  Bodies for Prepare_Multi_Use_Of_Expr and New_Ref_To_Expr
--  Change Make_Tagged_Copy in order to use the 2 former subprograms.
--  ----------------------------
--  revision 1.75
--  date: Thu Aug 18 20:09:18 1994;  author: dewar
--  Minor reformatting
--  ----------------------------
--  revision 1.76
--  date: Sun Aug 28 08:48:42 1994;  author: comar
--  (Make_Subtype_From_Expr): Use new protocol for New_Itype
--  ----------------------------
--  New changes after this line.  Each line starts with: "--  "
