/* Mouse managing
   Copyright (C) 1994 Miguel de Icaza.
   
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
   
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.  */

/* Events received by clients of this library have their coordinates 0 */
/* based */

static char *rcsid = "$Id: mouse.c,v 1.3 1994/08/02 04:40:06 miguel Exp $";

#ifdef use_mouse
#include <sys/types.h>
#include <sys/fcntl.h>
#include <termios.h>
#include "mouse.h"
#include "util.h"		/* xmalloc */

static int mouse_d;		/* Handle to the mouse server */
static int console_fd;		/* Handle to the console */

#ifdef DEBUGMOUSE
FILE *log;
#endif

/* A linked list of events */
static MouseEvent *mouse_events = 0;
static MouseEvent *last_mouse_event = 0;
static Frame *frames = 0;

/* Only used for debugging */
static int top_event = 0;

/* Used by mouse_handler to keep track of last position (needed to redraw) */
static int last_x, last_y;

/* When the mouse generates an event in the box (x1,y1)-(x2,y2),
   call the callback procedure with the data void *argument.
   If callback return 1, then it means autorepeat if no other event comes,
   it callback returns 0, then wait for the next event to be generated */
void push_event (int x1, int y1, int x2, int y2, mouse_h callback,
		 void *data)
{
    MouseEvent *new;
    int extra_x, extra_y;
    
    new = xmalloc (sizeof (MouseEvent), "add_event");
    new->next = (void *) mouse_events;
    new->mouse_callback = callback;
    if (frames){
	extra_x = frames->x;
	extra_y = frames->y;
    } else
	extra_x = extra_y = 0;
    
    new->x1 = x1 + extra_x;
    new->x2 = x2 + extra_x;
    new->y1 = y1 + extra_y;
    new->y2 = y2 + extra_y;
    new->data = data;
    mouse_events = new;
    DEBUGM ((log, "Event: %d\n", ++top_event));
}

void pop_event ()
{
    MouseEvent *head = mouse_events;

    if (!mouse_events)
	return;
    mouse_events = (MouseEvent *) mouse_events->next;
    free (head);
    DEBUGM ((log, "Event: %d\n", --top_event));
}

/* Used to establish a new origin. */
void push_frame (int x, int y, int relative_to_previous)
{
    Frame *new;

    new = xmalloc (sizeof (Frame), "push_frame");
    new->next = (void *) frames;
    new->x = x;
    new->y = y;
    new->mouse_events = mouse_events;
    if (relative_to_previous && frames){
	new->x += frames->x;
	new->y += frames->y;
    }
    frames = new;
}

void pop_frame ()
{
    Frame *head = frames;

    if (!frames)
	return;
    while (mouse_events != frames->mouse_events)
	pop_event ();
    frames = (Frame *) frames->next;
    free (head);
}

void show_mouse_pointer (int x, int y)
{
    unsigned char buf[sizeof(char) + 5 * sizeof(short)];
    unsigned short *arg = (unsigned short *)(buf + 1);
    
    /* Show the pointer */
    buf[0] = 2;     arg[4] = 3;
    arg[0] = x;     arg[1] = y;
    arg[2] = x;     arg[3] = y;
    ioctl (console_fd, TIOCLINUX, &buf);
}

int mouse_handler (Gpm_Event *gpm_event)
{
    MouseEvent *event = mouse_events;
    int x = last_x = gpm_event->x;
    int y = last_y = gpm_event->y;
    int redo = 0;
    
/*    DEBUGM ((log, "Mouse [%d, %d]\n", x, y)); */

    /* Call any registered event handlers */
    for (; event; event = (MouseEvent *) event->next){
	if ((event->x1 <= x) && (x <= event->x2)
	    && (event->y1 <= y) && (y <= event->y2)){
	    gpm_event->x -= event->x1;
	    gpm_event->y -= event->y1;
	    last_mouse_event = event;
	    redo = (*(event->mouse_callback))(gpm_event, event->data);
	    break;
	}
    }
    Gpm_DrawPointer (last_x, last_y, console_fd);
    return redo;
}

int redo_mouse (Gpm_Event *event)
{
    if (last_mouse_event){
	return (*(last_mouse_event->mouse_callback))
	    (event,last_mouse_event->data);
    }
    return MOU_NORMAL;
}

/* Simple shell to functions that only respond to a click */
/* The extra data (fn) is used as the pointer to the function to be called */
int click (Gpm_Event *event, void *fn)
{
    if (event->type & GPM_DOWN)
	(*((mouse_h) fn))(event, 0);
    return MOU_NORMAL;
}

int click_auto_repeat (Gpm_Event *event, void *fn)
{
    if (event->type & (GPM_DOWN | GPM_DRAG))
	(*((mouse_h) fn))(event, 0);
    return MOU_REPEAT;
}

int click_may_break_loop (Gpm_Event *event, void *fn)
{
    int v;
    
    if (event->type & GPM_DOWN){
	v = (*((mouse_h) fn))(event, 0);
	return v ? MOU_ENDLOOP : MOU_NORMAL;
    }
    return MOU_NORMAL;
}

int null_event (Gpm_Event *event, void *x)
{
    return MOU_NORMAL;
}

int quit_event (Gpm_Event *event, void *x)
{
    return MOU_ENDLOOP;
}

void init_mouse ()
{
    Gpm_Connect conn;

    conn.eventMask   = ~GPM_MOVE;
    conn.defaultMask = GPM_MOVE;
    conn.minMod      = 0;
    conn.maxMod      = 0;

    if ((console_fd = open("/dev/console", O_WRONLY)) == -1)
	return;

    if ((mouse_d = Gpm_Open (&conn, 0)) == -1)
	return;

#ifdef DEBUGMOUSE
    log = fopen ("/dev/tty7", "w");
#endif
    push_frame (0, 0, 0);
}

void shut_mouse ()
{
    close (console_fd);
    Gpm_Close ();
}

#else
void init_mouse ()
{
    /* Nothing */
}

void shut_mouse ()
{
    /* Nothing */
}
#endif
