/*
 * man.c
 *
 * Copyright (c) 1990, 1991, John W. Eaton.
 *
 * You may distribute under the terms of the GNU General Public
 * License as specified in the file COPYING that comes with the man
 * distribution.
 *
 * John W. Eaton
 * jwe@che.utexas.edu
 * Department of Chemical Engineering
 * The University of Texas at Austin
 * Austin, Texas  78712
 *
 *
 * Wed Dec 23 13:26:03 1992: Rik Faith (faith@cs.unc.edu) applied bug fixes
 * supplied by Willem Kasdorp (wkasdo@nikhefk.nikef.nl)
 *
 * Sat Apr 30 11:54:08 BST 1994 Wilf. (G.Wilford@ee.surrey.ac.uk)
 * Extensively rewritten, ANSI-fied, and upgraded to include self maintaining 
 * global database of available man pages, and to (optionally) adhere to the 
 * new filesystem standard (FSSTND). Also use newer glob routines, culled from
 * the bash-1.13.5 distribution.
 * New code mandb.c added to initialise the global database. 
 */

#define MAN_MAIN

#ifdef FSSTND
#define CATPATH(path)	catpath(path)
char *catpath (char *path);
#else
#define CATPATH(path)	path
#endif

#include <stdio.h>
#include <ctype.h>
#include <string.h>
#include <sys/file.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <signal.h>
#include <stdlib.h>
#include <unistd.h>
#include <time.h>

#include "config.h"
#include "mydbm.h"
#include "gripes.h"
#include "util.h"
#include "statdir.h"
#include "manp.h"
#include "newman.h"
#include "dbver.h"

/* version.c */
void ver(void);
/* db_management.c */
int try_db_manpage(char *fullpath, datum key, char *manp, char *section);
void remove_db_entry(char *fullpath, datum key);
int add_db_entry(char *manpage);

/* man.c */
void usage(void);
char **add_dir_to_mpath_list(char **mp, char *p);
void man_getopt(int argc, char *argv[]);
char *is_section(char *name);
void do_extern(char *prog, char *name);
char **glob_for_file(char *path, char *section, char *name, int cat);
char **make_name(char *path, char *section, char *name, int cat);
char *get_expander(char *file);
int display_cat_file(char *file);
void add_directive(int *first, char *d, char *file, char *buf);
int parse_roff_directive(char *cp, char *file, char *buf);
char *make_roff_command(char *file);
char **get_section_list (void);
int man (char *name);

extern char **glob_vector ();
extern char **glob_filename ();

/* externals */
char *prognam;
char *database;
int debug;
short quiet = 1;
MYDBM_FILE dbf; 
short db;
char *manp;
short findall;
short dups = DUPS;

/* locals */
static char *pager;
static char *manpathlist[MAXDIRS];
static char *section;
static char *colon_sep_section_list;
static char **section_list;
static char *roff_directive;
static short apropos;
static short whatis;
static short print_where;

#ifdef ALT_SYSTEMS
static int alt_system;
static char *alt_system_name;
#endif

static int troff = 0;

#ifdef HAS_TROFF
#ifdef ALT_SYSTEMS
static char args[] = "M:P:S:adfhkVm:p:tw?";
#else
static char args[] = "M:P:S:adfhkVp:tw?";
#endif
#else
#ifdef ALT_SYSTEMS
static char args[] = "M:P:S:adfhkVm:p:w?";
#else
static char args[] = "M:P:S:adfhkVp:w?";
#endif
#endif

int main (int argc, char *argv[])
{
	int status = 0;
	char *nextarg;
	char *tmp;
	extern int optind;

	prognam = mkprogname (argv[0]);

	man_getopt (argc, argv);

	if (optind == argc)
		gripe_no_name ((char *) NULL);

	section_list = get_section_list ();

	if (optind == argc - 1){
		tmp = is_section (argv[optind]);

		if (tmp != NULL)
			gripe_no_name (tmp);
	}

	while (optind < argc){
		nextarg = argv[optind++];

		/*
     		 * See if this argument is a valid section name.  If not,
      		 * is_section returns NULL.
      		 */

		tmp = is_section (nextarg);

		if (tmp != NULL){
			section = tmp;

			if (debug)
				fprintf (stderr, "\nsection: %s\n", section);

			continue;
		}

		if (apropos)
			do_extern (APROPOS, nextarg);
		else if (whatis)
			do_extern (WHATIS, nextarg);

		/* this is where we actually start looking for the man page */

		else {
			status = (int) do_manual(nextarg);			

			if (status == 0)
				gripe_not_found (nextarg, section);
			else
				if (debug)
					fprintf(stderr,
					  "\nFound %d man pages\n", status);
		}
	}
	return 0;
}

void usage ()
{
	static char usage_string[1024]; 

#ifdef HAS_TROFF
#ifdef ALT_SYSTEMS
	static char s1[] =
	"usage: %s [-adfhktw] [section] [-M path] [-P pager] [-S list]\n\
           [-m system] [-p string] name ...\n\n";
#else
	static char s1[] =
	"usage: %s [-adfhktw] [section] [-M path] [-P pager] [-S list]\n\
           [-p string] name ...\n\n";
#endif
#else
#ifdef ALT_SYSTEMS
	static char s1[] =
	"usage: %s [-adfhkw] [section] [-M path] [-P pager] [-S list]\n\
           [-m system] [-p string] name ...\n\n";
#else
	static char s1[] =
	"usage: %s [-adfhkw] [section] [-M path] [-P pager] [-S list]\n\
           [-p string] name ...\n\n";
#endif
#endif

	static char s2[] = "  a : find all matching entries\n\
  d : print gobs of debugging information\n\
  f : same as whatis(1)\n\
  h : print this help message\n\
  k : same as apropos(1)\n";

#ifdef HAS_TROFF
	static char s3[] = "  t : use troff to format pages for printing\n";
#endif

	static char s4[] = "  w : print location of man page(s) that would be displayed\n\n\
  M path   : set search path for manual pages to `path'\n\
  P pager  : use program `pager' to display pages\n\
  S list   : colon separated section list\n";

#ifdef ALT_SYSTEMS
	static char s5[] = "  m system : search for alternate system's man pages\n";
#endif

	static char s6[] = "  p string : string tells which preprocessors to run\n\
               e - [n]eqn(1)   p - pic(1)    t - tbl(1)\n\
               g - grap(1)     r - refer(1)  v - vgrind(1)\n";

	strcat (usage_string, s1);
	strcat (usage_string, s2);

#ifdef HAS_TROFF
	strcat (usage_string, s3);
#endif

	strcat (usage_string, s4);

#ifdef ALT_SYSTEMS
	strcat (usage_string, s5);
#endif

	strcat (usage_string, s6);

	printf (usage_string, prognam);
}

char **add_dir_to_mpath_list (char **mp, char *p)
{
	int status;

	status = is_directory (p);

	if (status < 0){
		fprintf (stderr, "Warning: couldn't stat file %s!\n", p);
	}
	else if (status == 0){
		fprintf (stderr, "Warning: %s isn't a directory!\n", p);
	}
	else if (status == 1){
		if (debug)
			fprintf (stderr, "adding %s to manpathlist\n", p);

		*mp++ = strdup (p);
	}
	return mp;
}

/*
 * Get options from the command line and user environment.
 */
void man_getopt (int argc, char *argv[])
{
	int c;
	char *p;
	char *end;
	char **mp;
	extern char *optarg;

	while ((c = getopt (argc, argv, args)) != -1){

		switch (c){

			case 'M':
				manp = strdup (optarg);
				db++;
				break;
		    	case 'P':
				pager = strdup (optarg);
				break;
		    	case 'S':
				colon_sep_section_list = strdup (optarg);
				db++;
				break;
			case 'V':
				ver();
				exit(0);
		    	case 'a':
				findall++;
				break;
		    	case 'd':
				debug++;
				break;
		    	case 'f':
				if (troff)
					gripe_incompatible ("-f and -t");
				if (apropos)
					gripe_incompatible ("-f and -k");
				if (print_where)
					gripe_incompatible ("-f and -w");
				whatis++;
				break;
		    	case 'k':
				if (troff)
					gripe_incompatible ("-k and -t");
				if (whatis)
					gripe_incompatible ("-k and -f");
				if (print_where)
					gripe_incompatible ("-k and -w");
				apropos++;
				break;
#ifdef ALT_SYSTEMS
		    	case 'm':
				alt_system++;
				alt_system_name = strdup (optarg);
				break;
#endif
		    	case 'p':
				roff_directive = strdup (optarg);
				break;
#ifdef HAS_TROFF
		    	case 't':
				if (apropos)
					gripe_incompatible ("-t and -k");
				if (whatis)
					gripe_incompatible ("-t and -f");
				if (print_where)
					gripe_incompatible ("-t and -w");
				troff++;
				break;
#endif
		    	case 'w':
				if (apropos)
					gripe_incompatible ("-w and -k");
				if (whatis)
					gripe_incompatible ("-w and -f");
				if (troff)
					gripe_incompatible ("-w and -t");
				print_where++;
				break;
		    	case 'h':
		    	case '?':
		    		usage();
		    		exit(0);
		    	default:
				usage ();
				exit(1);
			}
		}

	if (pager == NULL || *pager == '\0')
		if ((pager = getenv ("PAGER")) == NULL)
			pager = strdup (PAGER);

	if (debug)
		fprintf (stderr, "\nusing %s as pager\n", pager);

	if (manp == NULL){
		if ((manp = manpath (0)) == NULL)
			gripe_manpath ();

		if (debug)
			fprintf (stderr,
				"\nsearch path for pages determined by manpath is\n%s\n\n",
				   manp);
	}

#ifdef ALT_SYSTEMS
	if (alt_system_name == NULL || *alt_system_name == '\0')
		if ((alt_system_name = getenv ("SYSTEM")) != NULL)
			alt_system_name = strdup (alt_system_name);

	if (alt_system_name != NULL && *alt_system_name != '\0')
		downcase (alt_system_name);
#endif

	/*
  	 * Expand the manpath into a list for easier handling.
  	 */

	mp = manpathlist;
	for (p = manp;; p = end + 1){
		if ((end = strchr (p, ':')) != NULL)
			*end = '\0';

#ifdef ALT_SYSTEMS
		if (alt_system){
			char buf[BUFSIZ];

			 if (debug)
				fprintf (stderr, "Alternate system `%s' specified\n",
					alt_system_name);

			strcpy (buf, p);
			strcat (buf, "/");
			strcat (buf, alt_system_name);

			mp = add_dir_to_mpath_list (mp, buf);
		}
		else {
			mp = add_dir_to_mpath_list (mp, p);
		}
#else
		mp = add_dir_to_mpath_list (mp, p);
#endif
		if (end == NULL)
			break;

		*end = ':';
	}
	*mp = NULL;
}

/*
 * Check to see if the argument is a valid section number.  If the
 * first character of name is a numeral, or the name matches one of
 * the sections listed in section_list, we'll assume that it's a section.
 * The list of sections in config.h simply allows us to specify oddly
 * named directories like .../man3f.  Yuk.
 */
char *is_section (char *name)
{
	char **vs;

	for (vs = section_list; *vs != NULL; vs++)
		if ((strcmp (*vs, name) == 0) || (isdigit (name[0])
						  && !isdigit (name[1])))
			return strdup (name);

	return NULL;
}

/*
 * Try to find the man page corresponding to the given name.  The
 * reason we do this with globbing is because some systems have man
 * page directories named man3 which contain files with names like
 * XtPopup.3Xt.  Rather than requiring that this program know about
 * all those possible names, we simply try to match things like
 * .../man[sect]/name[sect]*.  This is *much* easier.
 *
 * Note that globbing is only done when the section is unspecified.
 */
char **glob_for_file (char *path, char *section, char *name, int cat)
{
	char pathname[BUFSIZ];
	char **gf;

	if (cat)
		sprintf (pathname, "%s/cat%s/%s.%s*", CATPATH (path), section, name, section);
	else
		sprintf (pathname, "%s/man%s/%s.%s*", path, section, name, section);

	if (debug)
		fprintf (stderr, "globbing %s\n", pathname);

	gf = glob_filename (pathname);

	if ((gf == (char **) -1 || *gf == NULL) && isdigit (*section)){
		if (cat)
			sprintf (pathname, "%s/cat%s/%s.%c*", CATPATH (path), section, name, *section);
		else
			sprintf (pathname, "%s/man%s/%s.%c*", path, section, name, *section);

		gf = glob_filename (pathname);
	}

	/*
  	 * If we're given a section that looks like `3f', we may want to try
  	 * file names like .../man3/foo.3f as well.  This seems a bit
  	 * kludgey to me, but what the hey...
  	 */

	if (section[1] != '\0'){
		if (cat)
			sprintf (pathname, "%s/cat%c/%s.%s", CATPATH (path), section[0], name, section);
		else
			sprintf (pathname, "%s/man%c/%s.%s", path, section[0], name, section);

		gf = glob_filename (pathname);
	}
	return gf;
}

/*
 * Return an un-globbed name in the same form as if we were doing
 * globbing.
 */
char **make_name (char *path, char *section, char *name, int cat)
{
	int i = 0;
	static char *names[3];
	char buf[BUFSIZ];

	if (cat)
		sprintf (buf, "%s/cat%s/%s.%s", CATPATH (path), section, name, section);
	else
		sprintf (buf, "%s/man%s/%s.%s", path, section, name, section);

	if (access (buf, R_OK) == 0)
		names[i++] = strdup (buf);

	/*
  	 * If we're given a section that looks like `3f', we may want to try
  	 * file names like .../man3/foo.3f as well.  This seems a bit
  	 * kludgey to me, but what the hey...
  	 */

	if (section[1] != '\0'){
		if (cat)
			sprintf (buf, "%s/cat%c/%s.%s", CATPATH (path), section[0], name, section);
		else
			sprintf (buf, "%s/man%c/%s.%s", path, section[0], name, section);

		if (access (buf, R_OK) == 0)
			names[i++] = strdup (buf);
	}
	names[i] = NULL;
	return &names[0];
}

#ifdef DO_UNCOMPRESS
char *get_expander (char *file)
{
	char *expander = NULL;
	char *suff;

	if ( (suff = strrchr(file, '.')) != NULL){

		if (*(++suff) == 'F'){
			if (strcmp (FCAT, "") != 0)
				expander = strdup (FCAT);
		} else if (*suff == 'Y'){
			if (strcmp (YCAT, "") != 0)
				expander = strdup (YCAT);
		} else if (*suff == 'Z' ||
		     strcmp(suff, COMPRESS_EXT + 1) == 0){
			if (strcmp (ZCAT, "") != 0)
				expander = strdup (ZCAT);
		}
	}
	return expander;
}
#endif /* DO_COMPRESS */
	

/*
	int len = strlen (file);

	if (file[len - 2] == '.'){
	
		  switch (file[len - 1]){
#ifdef FCAT
		    case 'F':
			    if (strcmp (FCAT, "") != 0)
				    expander = strdup (FCAT);
			    break;
#endif
#ifdef YCAT
		    case 'Y':
			    if (strcmp (YCAT, "") != 0)
				    expander = strdup (YCAT);
			    break;
#endif
#ifdef ZCAT
		    case 'Z':
			    if (strcmp (ZCAT, "") != 0)
				    expander = strdup (ZCAT);
			    break;
#endif
		    default:
			    break;
		    }
	  }
	return expander;
}
*/

/*
 * Simply display the preformatted page.
 */
int display_cat_file (char *file)
{
	char command[BUFSIZ];

	if (access (file, R_OK) == 0){

#ifdef DO_UNCOMPRESS
		char *expander = get_expander (file);

		if (expander != NULL)
			sprintf (command, "%s %s | %s", expander, file, pager);
		else
			sprintf (command, "%s %s", pager, file);
#else
		sprintf (command, "%s %s", pager, file);
#endif

		return do_system_command (command);
	}
	return 0;
}

/* 
 * what does this do?
 */
void add_directive (int *first, char *d, char *file, char *buf)
{
	if (strcmp (d, "") != 0){
		if (*first){
			*first = 0;
			strcpy (buf, d);
			strcat (buf, " ");
			strcat (buf, file);
		}
		else {
			strcat (buf, " | ");
			strcat (buf, d);
		}
	}
}

int parse_roff_directive (char *cp, char *file, char *buf)
{
	char c;
	int first = 1;
	int tbl_found = 0;

	while ( !isspace(c = *cp++) && c != '\0'){
		switch (c){
		  
		case 'e':

			if (debug)
				fprintf (stderr, "found eqn(1) directive\n");

			if (troff)
				add_directive (&first, EQN, file, buf);
			else
				add_directive (&first, NEQN, file, buf);

			break;

		case 'g':

			if (debug)
				fprintf (stderr, "found grap(1) directive\n");

			add_directive (&first, GRAP, file, buf);

			break;

		case 'p':

			if (debug)
				fprintf (stderr, "found pic(1) directive\n");

			add_directive (&first, PIC, file, buf);

			break;

		case 't':

			if (debug)
				fprintf (stderr, "found tbl(1) directive\n");

			tbl_found++;
			add_directive (&first, TBL, file, buf);
			break;

		case 'v':

			if (debug)
				fprintf (stderr, "found vgrind(1) directive\n");

			add_directive (&first, VGRIND, file, buf);
			break;

		case 'r':

			if (debug)
				fprintf (stderr, "found refer(1) directive\n");

			add_directive (&first, REFER, file, buf);
			break;

		default:

			return -1;
		}
	}

	if (first)
		return 1;

#ifdef HAS_TROFF
	if (troff){
		strcat (buf, " | ");
		strcat (buf, TROFF);
	}
	else
#endif
	{
		strcat (buf, " | ");
		strcat (buf, NROFF);
	}

	if (tbl_found && !troff && strcmp (COL, "") != 0){
		strcat (buf, " | ");
		strcat (buf, COL);
	}
	return 0;
}

char *make_roff_command (char *file)
{
	FILE *fp;
	char line[BUFSIZ];
	static char buf[BUFSIZ];
	int status;
	char *cp;

	if (roff_directive != NULL){
		if (debug)
			fprintf (stderr, "parsing directive from command line\n");

		status = parse_roff_directive (roff_directive, file, buf);

		if (status == 0)
			return buf;

		if (status == -1)
			gripe_roff_command_from_command_line (file);
	}

	if ((fp = fopen (file, "r")) != NULL){
		cp = &line[0];
		fgets (line, 100, fp);
		if (*cp++ == '\'' && *cp++ == '\\' && *cp++ == '"' && *cp++ == ' '){

			if (debug)
				fprintf (stderr, "parsing directive from file\n");

			status = parse_roff_directive (cp, file, buf);

			fclose (fp);

			if (status == 0)
				return buf;

			if (status == -1)
				gripe_roff_command_from_file (file);
		}
	}
	else {
		/*
      		 * Is there really any point in continuing to look for
      		 * preprocessor options if we can't even read the man page source?
      		 */
		  gripe_reading_man_file (file);
		  return NULL;
	}

	if ((cp = getenv ("MANROFFSEQ")) != NULL){
	
		if (debug)
			fprintf (stderr, "parsing directive from environment\n");

		status = parse_roff_directive (cp, file, buf);

		if (status == 0)
			return buf;

		if (status == -1)
			gripe_roff_command_from_env ();
	}
	if (debug)
		fprintf (stderr, "using default preprocessor sequence\n");

#ifdef HAS_TROFF
	if (troff){
		if (strcmp (TBL, "") != 0){
			strcpy (buf, TBL);
			strcat (buf, " ");
			strcat (buf, file);
			strcat (buf, " | ");
			strcat (buf, TROFF);
		}
		else {
			strcpy (buf, TROFF);
			strcat (buf, " ");
			strcat (buf, file);
		}
	}
	else
#endif
	{
		if (strcmp (TBL, "") != 0){
			strcpy (buf, TBL);
			strcat (buf, " ");
			strcat (buf, file);
			strcat (buf, " | ");
			strcat (buf, NROFF);
		}
		else {
			strcpy (buf, NROFF);
			strcat (buf, " ");
			strcat (buf, file);
		}

		if (strcmp (COL, "") != 0){
			strcat (buf, " | ");
			strcat (buf, COL);
		}
	}
	return buf;
}

/*
 * Try to format the man page and create a new formatted file.  Return
 * 1 for success and 0 for failure.
 */
int make_cat_file (char *path, char *man_file, char *cat_file)
{
	int status;
	int mode;
	FILE *fp;
	char *roff_command;
	char command[BUFSIZ];

	if ((fp = fopen (cat_file, "w")) != NULL){
		fclose (fp);
		unlink (cat_file);

		roff_command = make_roff_command (man_file);
		if (roff_command == NULL)
			return 0;
		else
#ifdef DO_COMPRESS
			sprintf (command, "(cd %s ; %s | %s > %s)", path,
				roff_command, COMPRESSOR, cat_file);
#else
			sprintf (command, "(cd %s ; %s > %s)", path,
				roff_command, cat_file);
#endif
		/*
      		 * Don't let the user interrupt the system () call and screw up
      		 * the formmatted man page if we're not done yet.
      		 */

		signal (SIGINT, SIG_IGN);

		fputs ("Formatting page, please wait... ", stderr);

		status = do_system_command (command);

		if (status == 1){
			mode = CATMODE;
			chmod (cat_file, mode);

			if (debug)
				fprintf (stderr, "mode of %s is now %o\n", cat_file, mode);
		}

		signal (SIGINT, SIG_DFL);
		fputs("done\n", stderr);

		return 1;
	}
	else {
		fputc('\n', stderr);
		if (debug)
			fprintf (stderr, "Couldn't open %s for writing.\n", cat_file);

		return 0;
	}
}

/*
 * Try to format the man page source and save it, then display it.  If
 * that's not possible, try to format the man page source and display
 * it directly.
 *
 * Note that we've already been handed the name of the ultimate source
 * file at this point.
 */
int format_and_display (char *path, char *man_file, char *cat_file)
{
	int status;
	register int found;
	char *roff_command;
	char command[BUFSIZ];

	found = 0;

	if (access (man_file, R_OK) != 0)
		return 0;

	if (troff){
		roff_command = make_roff_command (man_file);
		if (roff_command == NULL)
			return 0;
		else
			sprintf (command, "(cd %s ; %s)", path, roff_command);

		found = do_system_command (command);
	}
	else {
		status = is_newer (man_file, cat_file);
		if (debug)
			fprintf (stderr, "status from is_newer() = %d\n", status);

		if (status == 1 || status == -2){
			/*
	  		 * Cat file is out of date.  Try to format and save it.
	  		 */

			if (print_where){
				printf ("%s\n", man_file);
				found++;
			}
			else {
				found = make_cat_file (path, man_file, cat_file);
#ifdef SECURE_MAN_UID
				if (!found){
					/*
		  			 * Try again as real user.  Note that for private
		  			 * man pages, we won't even get this far unless the
		  			 * effective user can read the real user's man page
		  			 * source.  Also, if we are trying to find all the
		  			 * man pages, this will probably make it impossible
		  			 * to make cat files in the system directories if
		  			 * the real user's man directories are searched
		  			 * first, because there's no way to undo this (is
		  			 * there?).  Yikes, am I missing something obvious?
		  			 */
		  			 
					setuid (getuid ());

					found = make_cat_file (path, man_file, cat_file);
				}
#endif
				if (found){
					/*
		  			 * Creating the cat file worked.  Now just display it.
		  			 */
					(void) display_cat_file (cat_file);
				}
				else {
					/*
		  			 * Couldn't create cat file.  Just format it and
		  			 * display it through the pager.
		  			 */
					roff_command = make_roff_command (man_file);
					if (roff_command == NULL)
						return 0;
					else
						sprintf (command, "(cd %s ; %s | %s)", path,
						roff_command, pager);

					found = do_system_command (command);
				}
			}
		}
		else if (access (cat_file, R_OK) == 0) {
			/*
	  		 * Formatting not necessary.  Cat file is newer than source
	  		 * file, or source file is not present but cat file is.
	  		 */
			if (print_where){
				printf ("%s (source: %s)\n", cat_file, 
				  status & 2 ? "NONE - straycat." : man_file);
				found++;
			}
			else {
				found = display_cat_file (cat_file);
			}
		}
	}
	return found;
}

/*
 * See if the preformatted man page or the source exists in the given
 * section.
 */
int try_section (char *path, char *section, char *name, int glob)
{
	int found = 0;
	int cat;
	char **names;
	char **np;

	if (debug){
		if (glob)
			fprintf (stderr, "trying section %s with globbing\n", section);
		else
			fprintf (stderr, "trying section %s without globbing\n", section);
	}

#ifndef NROFF_MISSING /* #ifdef NROFF */
	/*
  	 * Look for man page source files.
  	 */

	cat = 0;
	if (glob)
		names = glob_for_file (path, section, name, cat);
	else
		names = make_name (path, section, name, cat);

	if (names == (char **) -1 || *names == NULL)
		/*
    		 * No files match.  
    		 * See if there's a preformatted page around that
    		 * we can display.
    		 */
#endif /* NROFF_MISSING */
	{
		if (!troff){
			cat = 1;
			if (glob)
				names = glob_for_file (path, section, name, cat);
			else
				names = make_name (path, section, name, cat);

			if (names != (char **) -1 && *names != NULL){
				for (np = names; *np != NULL; np++){
					if (print_where){
						printf ("%s\n", *np);
						found++;
					}
					else {
						found += display_cat_file (*np);
					}
				}
			}
		}
	}
#ifndef NROFF_MISSING
	else {
		for (np = names; *np != NULL; np++){
			char *cat_file = NULL;
			char *man_file;

			if ( (man_file = ult_src (*np, path)) == NULL)
				return 0;
		/*
			if (!db && add_db_entry(*np)){
		 */			
		                /*
                                 * If we get to here, we've found a 
                                 * src file that is
                                 * supposedly not in the database - 
                                 * we should add it. (as above)
			         *
			         * If it can't be added 
			         * DUE TO ALREADY EXISTING IN THE db,
			         * man was probably called with -a.
			         * Ignore this page entirely. 
			         * Don't even display it.
			         */
		/*
			        free(man_file);
		         	return 0;
 		        }
		 */

			if (debug)
				fprintf (stderr, "found ultimate source file %s\n", man_file);

			if (!troff){
				cat_file = convert_name (man_file);

				if (debug)
					fprintf (stderr, "will try to write %s if needed\n", cat_file);
			}

			found += format_and_display (path, man_file, cat_file);
			free(man_file);
		}
	}
#endif /* NROFF_MISSING */
	return found;
}

/*
 * Search for manual pages.
 *
 * If preformatted manual pages are supported, look for the formatted
 * file first, then the man page source file.  If they both exist and
 * the man page source file is newer, or only the source file exists,
 * try to reformat it and write the results in the cat directory.  If
 * it is not possible to write the cat file, simply format and display
 * the man file.
 *
 * If preformatted pages are not supported, or the troff option is
 * being used, only look for the man page source file.
 *
 */
int man (char *name)
{
	int found = 0;
	int glob = 0;
	char **mp;
	char **sp;

	fflush (stdout);

	/* first of all check the global database */

	db++;  /* set the static db identifier */

	if (db == 1){
		if ((dbf = MYDBM_REOPEN (database)) != NULL){

  /*
   *  Good, means we get a chance to do things properly.
   *
   *  First we try to find the page within the db. If it's found, great, we
   *  look it up. If there are multiple entries, we can show them all as usual
   *  BUT, if there are multiple entries and a new page which is not yet
   *  in the db, it won't be found :-(. Otherwise we will see all of the db
   *  entries + all entries via globbing. Maybe this is better, at least we
   *  actually get to see the new page!
   *
   * If (for some reason), the page found in the db does not exist in reality
   * we remove it.
   *
   *  (We could verify that globbed entries are not in the db before showing
   *  them ?)
   *
   *  If we find no trace of the man page we give control to the globbing
   *  routines. It's not all over yet, though, if we find a src page via 
   *  globbing, we'll add it to the db if possible.
   */
		
			datum key, content;
	
			if (debug)
				  fprintf (stderr, "\nopened database file %s\n", database);

			if ( (dbver(dbf)) != 0 ) {
				MYDBM_CLOSE(dbf);
				return 0;
			}

			key.dptr = name;
			key.dsize = strlen (key.dptr) + 1;
	
			content = MYDBM_FETCH (dbf, key);
	
			MYDBM_CLOSE(dbf); 	/* for sanity's sake */
	
			/* if closing the db frees up content.dptr, we've had it here */
	
			if (content.dptr != NULL){
			
				/* 
				 * The db has an entry for the manpage, 
				 * which may be several
				 * pages concattenated with ':'s
				 */
	
				char *c1;
	
				if (debug)
					fprintf (stderr, "found man page(s) in db at %s\n",
					  content.dptr);
	
				while ((c1 = strrchr(content.dptr, ':')) != NULL){
					*c1 = '\0';
	
					found += try_db_manpage(++c1, key, manp, section);
	
					if (found && !findall)  
						return found;
			 	}
	
				found += try_db_manpage(content.dptr, key, manp, section);
	
				if (found && !findall)
					return found;
				
				MYDBM_FREE (content.dptr);
			}
			else {	
	
				/*
       	   		 	 * name not found in db, 
       	   		 	 * if we find it by globbing,
       	   		 	 * we should add it
       	   		 	 */
	
       	   		if (debug)
       	   			fprintf(stderr, "man page not found in %s\n", database);
			}
			return found;
		}
		else 
			/* db not found */
	
			if (debug){
				fputs("couldn't open db, even in READ_ONLY mode.\n", stderr);
		}
	/*	return found; gone up 8 lines !*/
	}
	else
		if (debug)
			fputs("Not using any databases.\n\n", stderr);

	db--; /* unset the static db identifier */

	if (strcmp(database, DB) != 0)
		return found;

	if (section != NULL){
		for (mp = manpathlist; *mp != NULL; mp++){
			if (debug)
				fprintf (stderr, "\nsearching in %s\n", *mp);

			found += try_section (*mp, section, name, glob);

			if (found && !findall)	/* i.e. only do this section... */
				return found;
		}
	}
	else {
		for (sp = section_list; *sp != NULL; sp++){
			for (mp = manpathlist; *mp != NULL; mp++){
				if (debug)
					fprintf (stderr, "\nsearching in %s\n", *mp);

				glob = 1;

				found += try_section (*mp, *sp, name, glob);

				if (found && !findall)	/* i.e. only do this section... */
					return found;
			}
		}
	}
	return found;
}

char **get_section_list (void)
{
	int i;
	char *p;
	char *end;
	static char *tmp_section_list[100];

	if (colon_sep_section_list == NULL){

		  if ((p = getenv ("MANSECT")) == NULL)
			return std_sections;
		  else
			colon_sep_section_list = strdup (p);
	}

	i = 0;
	for (p = colon_sep_section_list;; p = end + 1){
		  if ((end = strchr (p, ':')) != NULL)
			*end = '\0';

		  tmp_section_list[i++] = strdup (p);

		  if (end == NULL)
			break;
	}

	tmp_section_list[i] = NULL;
	return tmp_section_list;
}
