/* 
    umount.c - by Peter Orbaek <poe@daimi.aau.dk> 

    Copyright (C) 1992,93 Peter Orbaek.

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#define _BSD_SOURCE 1
#define _POSIX_SOURCE 1
#include <stdio.h>
#include <mntent.h>
#include <unistd.h>
#include <fcntl.h>
#include <string.h>
#include <stdlib.h>
#include <errno.h>
#include <getopt.h>
#include <sys/mount.h>
#include <sys/stat.h>

#define MTAB_FILE MOUNTED
#define LOCK_FILE "/etc/mtab~"
#define UFSTAB    "/etc/ufstab"

int opt_all          = 0;
int opt_verbose      = 0;
int opt_usermount    = 0;

char mnttype[200] = "";
char *mntdev;

void usage(), err(char *);
void lock_mtab();
int do_umount(struct mntent *);

void usage()
{
    fprintf(stderr,
	    "Usage: umount -a [-v] [-t type]\n"
	    "       umount [-uv] dev|dir\n");
    exit(1);
}

int main(int argc, char *argv[])
{
    char c;
    FILE *mtab, *newmtab;
    struct mntent *mp;
    struct mntent mnt;
    int perms;

    int flag;

    while((c = getopt(argc, argv, "avut:")) != EOF) {
	switch(c) {
	  case 'a':
	    opt_all = 1;
	    break;
	  case 'v':
	    opt_verbose = 1;
	    break;
	  case 't':
	    strncpy(mnttype, optarg, sizeof(mnttype));
	    break;
	  case 'u':
	    opt_usermount = 1;
	    break;
	  default:
	    usage();
	    break;
	}
    }
    if(optind < argc) mntdev = argv[optind++];
    if(optind < argc) usage();

    if((!opt_all && !mntdev) || (opt_all && mntdev)) usage();

    perms = 0666;
    if(getuid())
    {
	struct stat statbuf;

	umask (0);
	opt_usermount = 1;
	if ( stat ( MTAB_FILE, &statbuf ) < 0 ) {
	    perror ( "umount: can't stat mtab!!" );
	    exit(1);
	}
	perms = statbuf.st_mode & 0777;
    }

    if(geteuid()) {
	fprintf(stderr, "umount: You must be root to umount filesystems.\n");
	exit(1);
    }

    if(opt_usermount) {
	if(opt_all) {
	    fprintf(stderr, "umount: Ordinary users can't umount all filesystems.\n");
	    exit(1);
	}
	/* scan ufstab for filesystems that ordinary users can umount */
	flag = 0;
	if(!(mtab = setmntent(UFSTAB, "r"))) err(UFSTAB);
	while((mp = getmntent(mtab))) {
	    if(!strcmp(mntdev, mp->mnt_fsname)
	       || !strcmp(mntdev, mp->mnt_dir)) {
		flag = 1;
		break;
	    }
	}
	endmntent(mtab);

	if(!flag) {
	    fprintf(stderr, "umount: %s isn't user-unmountable\n", mntdev);
	    exit(1);
	}
    }

    if(!(newmtab = fdopen(open(LOCK_FILE, O_WRONLY|O_CREAT|O_EXCL, perms), "w"))) {
	fprintf(stderr, "umount: A lockfile exists, unmount denied\n");
	exit(1);
    }
    if(!(mtab = setmntent(MTAB_FILE, "r"))) err(MTAB_FILE);

    if(opt_all) {
	/*
	 * To work properly, filesystems have to unmounted in the reverse order
	 * of how they were mounted.  Otherwise, you may try to umount a
	 * filesystem that has another filesystem mounted to it (which will
	 * fail).
	 */
	struct mntentnode {
	    struct mntent m;
	    int mounted;
	    struct mntentnode *next;
	} *mnp = 0, *tmnp, *next;
	/*
	 * Build the list of things to umount, bottom-up...
	 */
	while((mp = getmntent(mtab))) {
	    if(!mp->mnt_opts || !mp->mnt_opts[0]) mp->mnt_opts = "defaults";

	    if ( ! ( tmnp = malloc ( sizeof ( struct mntentnode ) ) ) ) {
		/*
		 * If malloc failed, then the system is probably in dire
		 * straits, so we exit in order to avoid damage...
		 */
		err("Out of memory!!!  Aborting...");
	    }

	    tmnp->next = mnp;
	    mnp = tmnp;
	    mnp->mounted = 1;	/* It's currently mounted */
	    mnp->m = *mp;
	    /*
	     * We're safer in copying the various strings out of the
	     * mntent struct rather than just copying the pointers.
	     * For the case of Linux, this proves to be a prudent
	     * assumption, inasmuch as the string space is reused by
	     * getmntent()..
	     */
	    mnp->m.mnt_fsname = strdup ( mp->mnt_fsname );
	    mnp->m.mnt_dir = strdup ( mp->mnt_dir );
	    mnp->m.mnt_type = strdup ( mp->mnt_type );
	    mnp->m.mnt_opts = strdup ( mp->mnt_opts );

	    if(!(mnp->m.mnt_fsname && mnp->m.mnt_dir && mnp->m.mnt_type && mnp->m.mnt_opts))
	      err("Out of memory!!!  Aborting...");

	}
	for ( tmnp = mnp ; tmnp ; tmnp = next ) {
	    next = tmnp->next;
	    if(mnttype[0] == '\0' || !strcmp(mnttype, mp->mnt_type)) {
		if(do_umount(&(tmnp->m)) < 0) {
		    /*
		     * Put this at the front of the list.  The end result
		     * will be a list of failed and skipped filesystems
		     * that will be in forward order (reverse of reverse).
		     * We use this to preserve the proper order of the
		     * mtab file.
		     */
		    tmnp->next = mnp;
		    mnp = tmnp;
		    fprintf(stderr, "umount: Umount of %s failed\n",
			    tmnp->m.mnt_fsname);
		} else
		    tmnp->mounted = 0;
	    } else {
		tmnp->next = mnp;
		mnp = tmnp;
	    }
	}
	for ( tmnp = mnp ; tmnp && tmnp->mounted ; tmnp = next ) {
	    addmntent(newmtab, &(tmnp->m));
	    /*
	     * We don't *have* to clean up behind ourselves, since we'll
	     * probably exit soon anyway, but it's always better to be
	     * safe...
	     */
	    next = tmnp->next;
	    free ( tmnp->m.mnt_fsname );
	    free ( tmnp->m.mnt_dir );
	    free ( tmnp->m.mnt_type );
	    free ( tmnp->m.mnt_opts );
	    free ( tmnp );
	}
    } else {
	flag = 0;
	while((mp = getmntent(mtab))) {
	    if(!mp->mnt_opts || !mp->mnt_opts[0]) mp->mnt_opts = "defaults";
	    if(!strcmp(mntdev, mp->mnt_fsname)
	       || !strcmp(mntdev, mp->mnt_dir)) {
		flag = 1;
		if(do_umount(mp) < 0) {
		    addmntent(newmtab, mp);
		    fprintf(stderr, "umount: Umount of %s failed\n",
			    mp->mnt_fsname);
		}
	    } else addmntent(newmtab, mp);
	}

	/* if it wasn't in mtab, try it anyway... */
	if(!flag) {
	    mnt.mnt_type = mnttype;
	    mnt.mnt_fsname = mntdev;
	    if(do_umount(&mnt) < 0) {
		fprintf(stderr, "umount: Umount of %s failed\n", mntdev);
	    }
	}
    }
    endmntent(newmtab);
    endmntent(mtab);

    unlink(MTAB_FILE);
    link(LOCK_FILE, MTAB_FILE);
    unlink(LOCK_FILE);

    exit(0);
}

void err(char * str)
{
    fprintf(stderr, "umount: %s: %s\n", str, strerror(errno));
    if(access(LOCK_FILE, 0) == 0) unlink(LOCK_FILE);
    exit(1);
}

int do_umount(struct mntent *mp)
{
    if(mp->mnt_type && strcmp(mp->mnt_type, MNTTYPE_SWAP) == 0) {
	if(swapoff(mp->mnt_fsname) < 0) return -1;
    } else {
	if(umount(mp->mnt_fsname) < 0) return -1;
    }

    if(opt_verbose)
      fprintf(stderr, "Unmounted %s\n", mp->mnt_fsname);

    return 0;
}

