/*
 *
 * Simple VFS definitions for fileio.
 *
 * Authors: Marco van Wieringen <v892273@si.hhs.nl> <mvw@mcs.ow.org>
 *          Edvard Tuinder <v892231@si.hhs.nl> <ed@delirium.ow.org>
 *
 * Version: $Id: fileio.c,v 1.10 1994/10/17 15:59:26 mvw Exp mvw $
 *
 */

#include <linux/config.h>
#include <linux/kernel.h>
#include <linux/sched.h>
#include <linux/stat.h>
#include <linux/string.h>
#include <linux/fcntl.h>

#ifdef CONFIG_QUOTA
extern int lookup(struct inode *, const char *, int, struct inode **);

int vfs_write(struct inode *inode, struct file *filp, char *addr, size_t bytes)
{
   size_t written;
   u_long cur_blocks, wanted_blocks = 0, avail_blocks = 0;

   if (S_ISREG(inode->i_mode)) {
      cur_blocks = isize_to_blocks(inode->i_size, inode->i_blksize);
      if ((filp->f_pos + bytes) > inode->i_size) {
         wanted_blocks = isize_to_blocks(filp->f_pos + bytes,
                         inode->i_blksize) - cur_blocks;
         if (wanted_blocks && quota_alloc(inode, 0, wanted_blocks,
                                          &avail_blocks) == NO_QUOTA)
            return -EDQUOT;
         if (wanted_blocks && (avail_blocks < wanted_blocks))
            bytes = blocks_to_isize((cur_blocks + avail_blocks),
                    inode->i_blksize) - filp->f_pos;
      }
      if ((written = filp->f_op->write(inode, filp, addr, bytes)) != bytes) {
         quota_remove(inode, 0, avail_blocks -
                     (isize_to_blocks(inode->i_size, inode->i_blksize) -
                      isize_to_blocks((inode->i_size - written), inode->i_blksize)));
      }
      current->io_usage += written;
      if (wanted_blocks && (avail_blocks < wanted_blocks))
         return -EDQUOT;

      return written;
   } else {
      current->io_usage += bytes;
      return filp->f_op->write(inode, filp, (char *)addr, bytes);
   }
}

int vfs_create(struct inode *dir, const char *basename,
               int namelen, int mode, struct inode **res_ino)
{
   int error;
   struct inode new_inode;

   memset(&new_inode, 0, sizeof(struct inode));
   new_inode.i_dev = dir->i_dev;
   new_inode.i_uid = current->fsuid;
   new_inode.i_gid = current->fsgid;
   getinoquota(&new_inode, -1);

   if (quota_alloc(&new_inode, 1, 0, (u_long *)0) == NO_QUOTA) {
      putinoquota(&new_inode);
      return -EDQUOT;
   }
   error = dir->i_op->create(dir, basename, namelen, mode, res_ino);
   if (error)
      quota_remove(&new_inode, 1, 0);
   putinoquota(&new_inode);

   return error;
}

int vfs_truncate(struct inode *ino, size_t lenght)
{
   int error;
   size_t old_isize;

   old_isize = ino->i_size;
   ino->i_size = lenght;
   if (ino->i_op && ino->i_op->truncate)
      ino->i_op->truncate(ino);
   ino->i_ctime = ino->i_mtime = CURRENT_TIME;
   ino->i_dirt = 1;
   if ((error = notify_change(NOTIFY_SIZE, ino)))
      return error;
   getinoquota(ino, -1);
   quota_remove(ino, 0, isize_to_blocks(old_isize, ino->i_blksize));
   putinoquota(ino);

   return error;
}

int vfs_mknod(struct inode *dir, const char *basename,
              int namelen, int mode, dev_t dev)
{
   int error;
   struct inode new_inode;

   memset(&new_inode, 0, sizeof(struct inode));
   new_inode.i_dev = dir->i_dev;
   new_inode.i_uid = current->fsuid;
   new_inode.i_gid = current->fsgid;
   getinoquota(&new_inode, -1);

   if (quota_alloc(&new_inode, 1, 0, (u_long *)0) == NO_QUOTA) {
      putinoquota(&new_inode);
      iput(dir);
      return -EDQUOT;
   }
   dir->i_count++;
   error = dir->i_op->mknod(dir, basename, namelen, mode, dev);
   if (error)
      quota_remove(&new_inode, 1, 0);
   putinoquota(&new_inode);
   iput(dir);

   return error;
}

int vfs_mkdir(struct inode *dir, const char *basename, int namelen, int mode)
{
   int error;
   struct inode new_inode;

   memset(&new_inode, 0, sizeof(struct inode));
   new_inode.i_dev = dir->i_dev;
   new_inode.i_uid = current->fsuid;
   new_inode.i_gid = current->fsgid;
   getinoquota(&new_inode, -1);

   if (quota_alloc(&new_inode, 1, 1, (u_long *)0) == NO_QUOTA) {
      putinoquota(&new_inode);
      iput(dir);
      return -EDQUOT;
   }
   dir->i_count++;
   error = dir->i_op->mkdir(dir, basename, namelen, mode);
   if (error)
      quota_remove(&new_inode, 1, 1);
   putinoquota(&new_inode);
   iput(dir);

   return error;
}

int vfs_rmdir(struct inode *dir, const char *basename, int namelen)
{
   int error;
   struct inode *old_inode;

   /*
    * Need inode entry of directory for quota operations
    */
   dir->i_count++;
   if ((error = lookup(dir, basename, namelen, &old_inode))) {
      iput(dir);
      return error;
   }
   getinoquota(old_inode, -1);
   if (!(error = dir->i_op->rmdir(dir, basename, namelen)))
      quota_remove(old_inode, 1, 1);
   putinoquota(old_inode);
   iput(old_inode);

   return error;
}

int vfs_unlink(struct inode *dir, const char *basename, int namelen)
{
   int error;
   struct inode *old_inode;

   /*
    * Need inode info of to remove file for quota operations.
    */
   dir->i_count++;
   if ((error = lookup(dir, basename, namelen, &old_inode))) {
      iput(dir);
      return error;
   }
   getinoquota(old_inode, -1);
   error = dir->i_op->unlink(dir, basename, namelen);
   /*
    * Remove blocks and inode. Only if link-count is 0 !
    */
   if (!error && old_inode->i_nlink == 0)
      quota_remove(old_inode, 1,
                   isize_to_blocks(old_inode->i_size, old_inode->i_blksize));
   putinoquota(old_inode);
   iput(old_inode);

   return error;
}

int vfs_symlink(struct inode *dir, const char *basename,
                int namelen, const char *oldname)
{
   int error;
   struct inode new_inode;

   memset(&new_inode, 0, sizeof(struct inode));
   new_inode.i_dev = dir->i_dev;
   new_inode.i_uid = current->fsuid;
   new_inode.i_gid = current->fsgid;
   getinoquota(&new_inode, -1);

   if (quota_alloc(&new_inode, 1, 1, (u_long *)0) == NO_QUOTA) {
      putinoquota(&new_inode);
      iput(dir);
      return -EDQUOT;
   }
   dir->i_count++;
   if (!(error = dir->i_op->symlink(dir, basename, namelen, oldname)))
      quota_remove(&new_inode, 1, 1);
   putinoquota(&new_inode);
   iput(dir);

   return error;
}

int vfs_chown(struct inode *ino, uid_t uid, gid_t gid)
{
   int error;
   uid_t olduid, oldgid;
   int notify_flag = 0;

   olduid = ino->i_uid;
   oldgid = ino->i_gid;
   getinoquota(ino, -1);
   if (quota_transfer(ino, uid, gid, 1,
                      isize_to_blocks(ino->i_size,
                      ino->i_blksize)) == NO_QUOTA) {
      putinoquota(ino);
      return -EDQUOT;
   }
   /*
    * If the owner has been changed, remove the setuid bit
    */
   if (ino->i_uid != uid && ino->i_mode & S_ISUID) {
      ino->i_mode &= ~S_ISUID;
      notify_flag = NOTIFY_MODE;
   }
   /*
    * If the group has been changed, remove the setgid bit
    */
   if (ino->i_gid != gid && ino->i_mode & S_ISGID) {
      ino->i_mode &= ~S_ISGID;
      notify_flag = NOTIFY_MODE;
   }
   ino->i_uid = uid;
   ino->i_gid = gid;
   notify_flag |= NOTIFY_UIDGID;
   ino->i_ctime = CURRENT_TIME;
   ino->i_dirt = 1;
   if ((error = notify_change(notify_flag, ino)))
      quota_transfer(ino, olduid, oldgid, 1,
                     isize_to_blocks(ino->i_size, ino->i_blksize));
   putinoquota(ino);
   return error;
}

int vfs_rename(struct inode *old_dir, const char *old_base, int old_len,
               struct inode *new_dir, const char *new_base, int new_len)
{
   int error;
   struct inode *old_inode, *new_inode;

   /*
    * Check if target file already exists, drop quota of file if
    * it already exists and is overwritten. Extra check needed for
    * renames of file to the same file.
    */
   old_dir->i_count++;
   if ((error = lookup(old_dir, old_base, old_len, &old_inode))) {
      iput(old_dir);
      iput(new_dir);
      return error;
   }
   new_dir->i_count++;
   if (!lookup(new_dir, new_base, new_len, &new_inode)) {
      if (old_dir != new_dir && old_inode != new_inode) {
         iput(old_inode);
         error = old_dir->i_op->rename(old_dir, old_base, old_len, 
                                       new_dir, new_base, new_len);
         if (!error) {
            getinoquota(new_inode, -1);
            quota_remove(new_inode, 1,
                         isize_to_blocks(new_inode->i_size, new_inode->i_blksize));
            putinoquota(new_inode);
            iput(new_inode);
         }
         return error;
      }
      iput(new_inode);
   }
   iput(old_inode);
   return old_dir->i_op->rename(old_dir, old_base, old_len, 
                                new_dir, new_base, new_len);
}

void vfs_open(struct file *filp)
{
   if (filp->f_inode && S_ISREG(filp->f_inode->i_mode) && (filp->f_mode & 2)) {
      filp->f_inode->i_writecount++;
      getinoquota(filp->f_inode, -1); 
   }
}

void vfs_close(struct file *filp)
{
   if (filp->f_inode && S_ISREG(filp->f_inode->i_mode) && (filp->f_mode & 2)) {
      filp->f_inode->i_writecount--;
      putinoquota(filp->f_inode); 
   }
}

#else /* CONFIG_QUOTA */

int vfs_truncate(struct inode *ino, size_t lenght)
{
   int error;

   ino->i_size = lenght;
   if (ino->i_op && ino->i_op->truncate)
      ino->i_op->truncate(ino);
   if ((error = notify_change(NOTIFY_SIZE, ino))) {
      return error;
   }
   ino->i_dirt = 1;

   return error;
}

int vfs_chown(struct inode *ino, uid_t uid, gid_t gid)
{
   int notify_flag = 0;

   /*
    * If the owner has been changed, remove the setuid bit
    */
   if (ino->i_uid != uid && ino->i_mode & S_ISUID) {
      ino->i_mode &= ~S_ISUID;
      notify_flag = NOTIFY_MODE;
   }
   /*
    * If the group has been changed, remove the setgid bit
    */
   if (ino->i_gid != gid && ino->i_mode & S_ISGID) {
      ino->i_mode &= ~S_ISGID;
      notify_flag = NOTIFY_MODE;
   }
   ino->i_uid = uid;
   ino->i_gid = gid;
   notify_flag |= NOTIFY_UIDGID;
   ino->i_ctime = CURRENT_TIME;
   ino->i_dirt = 1;
   return(notify_change(notify_flag, ino));
}

void vfs_open(struct file *filp)
{
   if (filp->f_inode && S_ISREG(filp->f_inode->i_mode) && (filp->f_mode & 2))
      filp->f_inode->i_writecount++;
}

void vfs_close(struct file *filp)
{
   if (filp->f_inode && S_ISREG(filp->f_inode->i_mode) && (filp->f_mode & 2))
      filp->f_inode->i_writecount--;
}

#endif /* CONFIG_QUOTA */

