#include <OI/oi.H>

/*
 *	KeyHelp
 *	
 *	This program demonstrates the use of the OI help mechanism.
 *	The help text file, keyhelp.hp, must exist in this directory
 *	for this program to correctly function.  Please refer to the
 *	README file for information on a second program which demonstrates
 *	how to use resources to specify the help mechanism.
 *
 *	The reader should refer to xmodmap man pages for information
 *	on how to specify which key on the keyboard should initiate
 *	help.  In this example, I have the following line:
 *		xmodmap -e "keysym KP_F4 = Help"
 *	in my ~/.xsession (~/.Xdefaults) file, to specify that key PF4
 *	should initiate help.
 *
 *	The reader should refer to the OI documentation for
 *	information on the following member functions.
 *		- OIIntro
 *			OI_init()
 *			OI_begin_interaction()
 *			OI_fini()
 *		- OI_d_tech
 *			layout_associated_object()
 *			set_associated_object()
 *			set_help_str()
 *			set_layout()
 *		- OI_app_window
 *			oi_create_app_window()
 *			push_help()
 *			set_help()
 *			set_key_help()
 *		- OI_static_text
 *			oi_create_static_text()
 *
 *	The general structure of a help file is discussed within the
 *	OI_app_window man pages under the set_help member function.
 *	However, in this example, the help text file would take the following form:
 *		@Topic@Hello
 *		@Sub Topics@World
 *		@Sub Topics@Universe
 *		@Related Topics@Query
 *		This is the help text for general "Hello".
 *		@Topic@World
 *		@Related Topics@Hello
 *		@Related Topics@Universe
 *		@Related Topics@Query
 *		This is the help text for "Hello World".
 *		@Topic@Universe
 *		@Related Topics@Hello
 *		@Related Topics@World
 *		@Related Topics@Query
 *		This is the help text for "Hello Universe".
 *		@Topic@Query
 *		@Related Topics@Hello
 *		This is the help text for "Is anyone home?"
 *	This indicates that 
 *		Topic (Hello) has subtopics (World, Universe) and a related
 *			topic (Query).  The help for topic (Hello) consists of:
 *				(This is the help text for general "Hello".)
 *		Topic (World) has no subtopics, but has related topics
 *			(Hello, Universe, Query).  The help for this topic 
 *			consists of:
 *				(This is the help text for "Hello World".)
 *		etc.
 */
void
main(int argc, char **argv)
{
	int		key_callback(OI_d_tech*,void*) ;
	OI_connection	*conp ;
	OI_app_window	*wp ;
	OI_static_text	*tp ;

	/*
	 *	Open a connection.
	 */
	if (conp=OI_init(&argc,argv,"test")) {

		/*
		 *	Create application main window.  Make it row layout.
		 */
		wp = oi_create_app_window("main",300,200,"Main") ;
		wp->set_layout(OI_layout_row) ;

		/*
		 *	Create three static text objects, and place them
		 *	within the application window.
		 */
		tp = oi_create_static_text("World","Hello, World!") ;
		tp->layout_associated_object(wp,0,1,OI_ACTIVE) ;

		tp = oi_create_static_text("Universe","Hello, Universe!") ;
		tp->layout_associated_object(wp,0,2,OI_ACTIVE) ;

		tp = oi_create_static_text("Query","Is anyone home?") ;
		tp->layout_associated_object(wp,0,3,OI_ACTIVE) ;

		/*
		 *	Display the main window.
		 */
		wp->set_associated_object(wp->root(),OI_DEF_LOC,OI_DEF_LOC,OI_ACTIVE) ;

		/*
		 *	Set the current help context to topic "Hello" in file "keyhelp.hp".
		 */
		wp->push_help("Hello","keyhelp.hp") ;

		/*
		 *	Specify the function to call whenever help is initiated from the 
		 *	keyboard over this app window.
		 */
		wp->set_key_help(key_callback) ;

		/*
		 *	Begin Interaction.
		 */
		OI_begin_interaction() ;

		/*
		 * Cleanup.  Make sure that we cleanup the library.
		 */
		OI_fini();
	}
}
 

/*
 *	key callback
 *
 *	Within main, the function set_key_help() is used to enable this function
 *	to be called whenever help is initiated from the keyboard.
 */
int
key_callback(OI_d_tech *objp, void*)
{
	char		str[101] ;
	int		ret = 0 ;


	/*
	 *	verify that an object was clicked on.
	 */
	if (objp) {
		/*
		 *	verify that this is a named object (since we are keying off the
		 *	object name for setting the help context.
		 */
		if (objp->name()) {
			/*
			 *	set the current help context to the specified topic and file.
			 *	In this example, our help contexts are specified by the name
			 *	of the object just clicked on.
			 */
			if (objp->app_window()->set_help(objp->name()) == OI_OK) {
				sprintf(str,"Object under pointer is \"%s\"",objp->name()) ;
				ret = 1 ;
			}
			else
				sprintf(str,"Unable to set help context for \"%s\"",objp->name()) ;
			/*
			 *	Set the help string in the nearest application window to either
			 *	the object name, or error message indicating no help for this object.
			 */
			objp->set_help_str(str) ;
		}
		else
			objp->set_help_str("Object under pointer has no name") ;
	}
	return(ret) ;
}
