/*
 * Copyright (c) 1992/3 Theo de Raadt <deraadt@fsa.ca>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. The name of the author may not be used to endorse or promote
 *    products derived from this software without specific prior written
 *    permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS
 * OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#include <sys/param.h>
#include <sys/types.h>
#include <sys/ioctl.h>
#include <sys/signal.h>
#include <sys/socket.h>
#include <sys/file.h>
#include <sys/fcntl.h>
#include <sys/syslog.h>
#include <sys/stat.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <ctype.h>
#include <dirent.h>
#include <netdb.h>
#include <string.h>
#include <rpc/rpc.h>
#include <rpc/xdr.h>
#include <net/if.h>
#include <arpa/inet.h>
#include <rpc/pmap_clnt.h>
#include <rpc/pmap_prot.h>
#include <rpc/pmap_rmt.h>
#include <unistd.h>
#include <rpcsvc/yp_prot.h>
#include <rpcsvc/ypclnt.h>

#include "xdr_yp.h"

#ifndef BINDINGDIR
#define BINDINGDIR "/var/yp/binding"
#endif

static char rcsid[] = "ypbind.c,v 2.1 1993/09/15 12:55:39 swen Exp";

struct _dom_binding
{
  struct _dom_binding *dom_pnext;
  char dom_domain[YPMAXDOMAIN + 1];
  struct sockaddr_in dom_server_addr;
  unsigned short int dom_server_port;
  int dom_socket;
  CLIENT *dom_client;
  long int dom_vers;
  time_t dom_check_t;
  int dom_lockfd;
  int dom_alive;
};

void *ypbindproc_null_2 __P((SVCXPRT *, void *, CLIENT *));
struct ypbind_resp *ypbindproc_domain_2 __P((SVCXPRT *, char *, CLIENT *));
bool_t *ypbindproc_setdom_2 __P((SVCXPRT *, struct ypbind_setdom *, CLIENT *));
static void ypbindprog_2 __P((struct svc_req *, SVCXPRT *));
static void checkwork __P((void));
static bool_t eachresult __P((caddr_t, struct sockaddr_in *));
static void broadcast __P((char *));
static void rpc_received __P((char *, struct sockaddr_in *, int));

char *domainname;

struct _dom_binding *ypbindlist;
int check;

#define YPSET_NO	0
#define YPSET_LOCAL	1
#define YPSET_ALL	2
int ypsetmode = YPSET_NO;

void *
ypbindproc_null_2(transp, argp, clnt)
     SVCXPRT *transp;
     void *argp;
     CLIENT *clnt;
{
  static char res;
  
  (void)memset(&res, 0, sizeof(res));
  return (void *)&res;
}

struct ypbind_resp *
ypbindproc_domain_2(transp, argp, clnt)
     SVCXPRT *transp;
     char *argp;
     CLIENT *clnt;
{
  static struct ypbind_resp res;
  struct _dom_binding *ypdb;
  char path[MAXPATHLEN];
  
  (void)memset(&res, 0, sizeof(res));
  res.ypbind_status = YPBIND_FAIL_VAL;
  
  for(ypdb=ypbindlist; ypdb; ypdb=ypdb->dom_pnext)
    if( strcmp(ypdb->dom_domain, argp) == 0)
      break;
  
  if(ypdb==NULL)
    {
      ypdb = (struct _dom_binding *)malloc(sizeof *ypdb);
      (void)memset(ypdb, 0, sizeof(*ypdb));
      strncpy(ypdb->dom_domain, argp, sizeof ypdb->dom_domain);
      ypdb->dom_vers = YPVERS;
      ypdb->dom_alive = 0;
      ypdb->dom_lockfd = -1;
      sprintf(path, "%s/%s.%d", BINDINGDIR, ypdb->dom_domain, ypdb->dom_vers);
      unlink(path);
      ypdb->dom_pnext = ypbindlist;
      ypbindlist = ypdb;
      check++;
      return NULL;
    }
  
  if(ypdb->dom_alive==0)
    return NULL;
  
#if 0
  delta = ypdb->dom_check_t - ypdb->dom_ask_t;
  if( !(ypdb->dom_ask_t==0 || delta > 5))
    {
      ypdb->dom_ask_t = time(NULL);
      /*
       * Hmm. More than 2 requests in 5 seconds have indicated that my
       * binding is possibly incorrect. Ok, make myself unalive, and
       * find out what the actual state is.
       */
      if(ypdb->dom_lockfd!=-1)
	close(ypdb->dom_lockfd);
      ypdb->dom_lockfd = -1;
      ypdb->dom_alive = 0;
      ypdb->dom_lockfd = -1;
      sprintf(path, "%s/%s.%d", BINDINGDIR, ypdb->dom_domain, ypdb->dom_vers);
      unlink(path);
      check++;
      return NULL;
    }
#endif
  
  res.ypbind_status = YPBIND_SUCC_VAL;
  res.ypbind_respbody.ypbind_bindinfo.ypbind_binding_addr.s_addr =
    ypdb->dom_server_addr.sin_addr.s_addr;
  res.ypbind_respbody.ypbind_bindinfo.ypbind_binding_port =
    ypdb->dom_server_port;
  /*printf("domain %s at %s/%d\n", ypdb->dom_domain,
    inet_ntoa(ypdb->dom_server_addr.sin_addr),
    ntohs(ypdb->dom_server_addr.sin_port));*/
  return &res;
}

bool_t *
ypbindproc_setdom_2(transp, argp, clnt)
     SVCXPRT *transp;
     struct ypbind_setdom *argp;
     CLIENT *clnt;
{
  struct sockaddr_in *fromsin, bindsin;
  char res;
  
  (void)memset(&res, 0, sizeof(res));
  fromsin = svc_getcaller(transp);
  
  switch(ypsetmode)
    {
    case YPSET_LOCAL:
      if( fromsin->sin_addr.s_addr != htonl(INADDR_LOOPBACK))
	return (void *)NULL;
      break;
    case YPSET_ALL:
      break;
    case YPSET_NO:
    default:
      return (void *)NULL;
    }
  
  if(argp->ypsetdom_vers != YPVERS)
    return (void *)&res;
  
  (void)memset(&bindsin, 0, sizeof(bindsin));
  bindsin.sin_family = AF_INET;
  bindsin.sin_addr.s_addr = argp->ypsetdom_addr.s_addr;
  bindsin.sin_port = argp->ypsetdom_port;
  rpc_received(argp->ypsetdom_domain, &bindsin, 1);
  
  res = 1;
  return (void *)&res;
}

void
ypbindprog_2(rqstp, transp)
     struct svc_req *rqstp;
     SVCXPRT *transp;
{
  union
    {
      char ypbindproc_domain_2_arg[MAXHOSTNAMELEN];
      struct ypbind_setdom ypbindproc_setdom_2_arg;
    } argument;
  struct authunix_parms *creds;
  char *result;
  bool_t (*xdr_argument)(), (*xdr_result)();
  char *(*local)();
  
  switch (rqstp->rq_proc)
    {
    case YPBINDPROC_NULL:
      xdr_argument = xdr_void;
      xdr_result = xdr_void;
      local = (char *(*)()) ypbindproc_null_2;
      break;
      
    case YPBINDPROC_DOMAIN:
      xdr_argument = xdr_domainname;
      xdr_result = xdr_ypbind_resp;
      local = (char *(*)()) ypbindproc_domain_2;
      break;
      
    case YPBINDPROC_SETDOM:
      switch(rqstp->rq_cred.oa_flavor)
	{
	case AUTH_UNIX:
	  creds = (struct authunix_parms *)rqstp->rq_clntcred;
	  if( creds->aup_uid != 0)
	    {
	      svcerr_auth(transp, AUTH_BADCRED);
	      return;
	    }
	  break;
	default:
	  svcerr_auth(transp, AUTH_TOOWEAK);
	  return;
	}
      
      xdr_argument = xdr_ypbind_setdom;
      xdr_result = xdr_void;
      local = (char *(*)()) ypbindproc_setdom_2;
      break;
      
    default:
      svcerr_noproc(transp);
      return;
    }
  (void)memset(&argument, 0, sizeof(argument));
  if (!svc_getargs(transp, xdr_argument, &argument))
    {
      svcerr_decode(transp);
      return;
    }
  result = (*local)(transp, &argument, rqstp);
  if (result != NULL && !svc_sendreply(transp, xdr_result, result))
    {
      svcerr_systemerr(transp);
    }
  return;
}

/*
 * change to do something like this:
 *
 * STATE	TIME		ACTION		NEWTIME	NEWSTATE
 * no binding	t==*		broadcast 	t=2	no binding
 * binding	t==60		check server	t=10	binding
 * binding	t=10		broadcast	t=2	no binding
 */
void
checkwork()
{
  struct _dom_binding *ypdb;
  time_t t;
  
  check = 0;
  
  time(&t);
  for(ypdb=ypbindlist; ypdb; ypdb=ypdb->dom_pnext)
    {
      if(ypdb->dom_alive==0 || ypdb->dom_check_t < t)
	{
	  broadcast(ypdb->dom_domain);
	  time(&t);
	  ypdb->dom_check_t = t + 60;
	}
    }
}

bool_t
eachresult(out, addr)
     caddr_t out;
     struct sockaddr_in *addr;
{
  if (*(bool_t *)out)
    {
#ifndef DAEMON
      struct hostent *hostentp;
      hostentp = gethostbyaddr((char *)&addr->sin_addr.s_addr, 
			       sizeof(addr->sin_addr.s_addr), AF_INET);
      fprintf(stderr,"Answer from server %s\n", hostentp->h_name);
#endif
      rpc_received(ypbindlist->dom_domain, addr, 0);
      return(1);
    }
  else
    return(0);
}

void
broadcast(dom)
     char *dom;
{
  char out[256];
  
  enum clnt_stat stat;
  
  stat = clnt_broadcast(YPPROG, YPVERS, YPPROC_DOMAIN_NONACK,
			xdr_domainname, dom, xdr_bool, (caddr_t)&out, eachresult);
  if (stat != RPC_SUCCESS)
    {
      fprintf(stderr, "broadcast: %s\n", clnt_sperrno(stat));
    }
}


/*
 * LOOPBACK IS MORE IMPORTANT: PUT IN HACK
 */
void
rpc_received(dom, raddrp, force)
     char *dom;
     struct sockaddr_in *raddrp;
     int force;
{
  struct _dom_binding *ypdb;
  char path[MAXPATHLEN];
  int fd;
  
  /*printf("returned from %s about %s\n", inet_ntoa(raddrp->sin_addr), dom);*/
  
  if(dom==NULL)
    return;
  
  for(ypdb=ypbindlist; ypdb; ypdb=ypdb->dom_pnext)
    if( strcmp(ypdb->dom_domain, dom) == 0)
      break;
  
  if(ypdb==NULL)
    {
      if(force==0)
	return;
      ypdb = (struct _dom_binding *)malloc(sizeof *ypdb);
      (void)memset(ypdb, 0, sizeof(*ypdb));
      strncpy(ypdb->dom_domain, dom, sizeof ypdb->dom_domain);
      ypdb->dom_lockfd = -1;
      ypdb->dom_pnext = ypbindlist;
      ypbindlist = ypdb;
    }
  
  /* soft update, alive, less than 30 seconds old */
  if(ypdb->dom_alive==1 && force==0 && ypdb->dom_check_t<time(NULL)+30)
    return;
  
  (void)memcpy(&ypdb->dom_server_addr, raddrp, sizeof(ypdb->dom_server_addr));
  ypdb->dom_check_t = time(NULL) + 60;	/* recheck binding in 60 seconds */
  ypdb->dom_vers = YPVERS;
  ypdb->dom_alive = 1;
  
  if(ypdb->dom_lockfd != -1)
    close(ypdb->dom_lockfd);
  
  sprintf(path, "%s/%s.%d", BINDINGDIR,
	  ypdb->dom_domain, ypdb->dom_vers);
#ifdef O_SHLOCK
  if( (fd=open(path, O_CREAT|O_SHLOCK|O_RDWR|O_TRUNC, 0644)) == -1) {
    (void)mkdir(BINDINGDIR, 0755);
    if( (fd=open(path, O_CREAT|O_SHLOCK|O_RDWR|O_TRUNC, 0644)) == -1)
      return;
  }
#else
  if( (fd=open(path, O_CREAT|O_RDWR|O_TRUNC, 0644)) == -1)
    {
      (void)mkdir(BINDINGDIR, 0755);
      if( (fd=open(path, O_CREAT|O_RDWR|O_TRUNC, 0644)) == -1)
	return;
    }
  flock(fd, LOCK_SH);
#endif

  /*
   * ok, if BINDINGDIR exists, and we can create the binding file,
   * then write to it..
   */
  ypdb->dom_lockfd = fd;
  if( write(ypdb->dom_lockfd, raddrp, sizeof *raddrp) != sizeof *raddrp)
    {
      perror("write");
      close(ypdb->dom_lockfd);
      ypdb->dom_lockfd = -1;
      return;
    }
}

void
svc_run()
{
#ifdef FD_SETSIZE
  fd_set readfds;
#else
  int readfds;
#endif /* def FD_SETSIZE */
  int dtblsize = _rpc_dtablesize();
  struct timeval timeout;
  extern int errno;
  
  for(;;)
    {
      timeout.tv_sec = 3;
      timeout.tv_usec = 0;
#ifdef FD_SETSIZE	 
      readfds = svc_fdset;
#else
      readfds = svc_fds;
#endif /* def FD_SETSIZE */
      switch(select(dtblsize, &readfds, (fd_set *)0, (fd_set *)0, &timeout))
	{
	case 0:
	  checkwork();
	  continue;
	case -1:
	  if (errno != EBADF)
	    {
	      continue;
	    }
	  perror("svc_run: select failed\n");
	  return;
	default:
	  svc_getreqset(&readfds);
	  if(check)
	    checkwork();
	  break;
	}
    }
}

int
main(argc, argv)
     int argc;
     char **argv;
{
  char path[MAXPATHLEN];
  SVCXPRT *transp;
  int i;
  
  yp_get_default_domain(&domainname);
  if( domainname[0] == '\0')
    {
      fprintf(stderr, "domainname not set. Aborting.\n");
      exit(1);
    }
  
  for(i=1; i<argc; i++)
    {
      if( strcmp("-ypset", argv[i]) == 0)
	ypsetmode = YPSET_ALL;
      else if (strcmp("-ypsetme", argv[i]) == 0)
	ypsetmode = YPSET_LOCAL;
    }
  
  /* blow away everything in BINDINGDIR */
  
  
  
#ifdef DAEMON
  switch(fork())
    {
    case 0:
      break;
    case -1:
      perror("fork");
      exit(1);
    default:
      exit(0);
    }
  setsid();
#endif
  
  transp = svcudp_create(RPC_ANYSOCK);
  if (transp == NULL)
    {
      fprintf(stderr, "cannot create udp service.");
      exit(1);
    }

  (void)pmap_unset(YPBINDPROG, YPBINDVERS);
  if (svc_register(transp, YPBINDPROG, YPBINDVERS, 
		   ypbindprog_2, IPPROTO_UDP) == FALSE)
    {
      fprintf(stderr, "unable to register (YPBINDPROG, YPBINDVERS, udp).");
      svc_destroy(transp);
      return(1);
    }
  
  transp = svctcp_create(RPC_ANYSOCK, 0, 0);
  if (transp == NULL)
    {
      fprintf(stderr, "cannot create tcp service.");
      return(1);
    }
  
  if (svc_register(transp, YPBINDPROG, YPBINDVERS, 
		   ypbindprog_2, IPPROTO_TCP) == FALSE)
    {
      fprintf(stderr, "unable to register (YPBINDPROG, YPBINDVERS, tcp).");
      svc_destroy(transp);
      return(1);
    }
  
  /* build initial domain binding, make it "unsuccessful" */
  ypbindlist = (struct _dom_binding *)malloc(sizeof *ypbindlist);
  (void)memset(ypbindlist, 0, sizeof(*ypbindlist));
  strncpy(ypbindlist->dom_domain, domainname, sizeof ypbindlist->dom_domain);
  ypbindlist->dom_vers = YPVERS;
  ypbindlist->dom_alive = 0;
  ypbindlist->dom_lockfd = -1;
  sprintf(path, "%s/%s.%d", BINDINGDIR, ypbindlist->dom_domain,
	  ypbindlist->dom_vers);
  (void)unlink(path);
  svc_run();
  fprintf(stderr,"Error: svc_run() shouldn't return\n");
  svc_unregister(YPBINDPROG, YPBINDVERS);
  svc_destroy(transp);
  return(1);
}


