/*
 *  isis_events.c,v 1.11 1993/01/07 22:39:15 jch Exp
 */

/* Gated Release 3.0 */
/* Copyright (c) 1990, 1991, 1992 by Cornell University. All rights reserved. */
/* Refer to Particulars and other Copyright notices at the end of this file. */


#include "include.h"
#include "isis_includes.h"

#ifdef	PROTO_SNMP
#include "isis_mib.h"				/* Isis-Mib */
#endif

void
adjInitFailureEvent(circuit, adj, reason, id, numIds)
CircuitEntry	*circuit;	/* circuit adjacency is on */
AdjacencyEntry	*adj;
ReasonCode	reason;		/* reason for failure */
SystemID6	*id;		/* list of IDs of failing system */
int			numIds;		/* number in list */
{
	trace(TR_ISIS, LOG_ERR, "%s: adjInitFailureEvent: reason %s", circuit->name,
		reasonCodeToStr(reason));
        circuit->initFailures++;
#ifdef	PROTO_SNMP
	{
	    IsisMibEntry	*pMibEntry;
	    IsisCircEntry	*pCircEntry;

	    pMibEntry = (IsisMibEntry *)locateEntry(&isisMibTable, 1);
	    pCircEntry = (IsisCircEntry *)locateEntry(&pMibEntry->isisCircTable,
								circuit->localId);
	    if (pCircEntry) {
	        ++pCircEntry->isisCircInitFails;		/* Isis-Mib */
	    }
	    else {
		trace(TR_ISIS, LOG_ERR, "adjInitFailureEvent(): cannot locate circ mib entry for localId==%d", circuit->localId);
	    }
	}
#endif	/* PROTO_SNMP */

}

void
adjUpEvent(circuit, adj)
CircuitEntry	*circuit;	/* circuit adjacency is on */
AdjacencyEntry	*adj;
{
	logAdjEvent(circuit, adj, "AdjUpEvent");
	lspAdjUp(circuit, adj);
#ifdef	PROTO_SNMP
	{
    	    IsisMibEntry	*pMibEntry;
    	    IsisCircEntry	*pCircEntry;
	    IsisISAdjEntry	*pISAdjEntry;

    	    pMibEntry = (IsisMibEntry *)locateEntry(&isisMibTable, 1);
    	    pCircEntry = (IsisCircEntry *)locateEntry(&pMibEntry->isisCircTable,
							       circuit->localId);
    	    if (pCircEntry) {
		pISAdjEntry = (IsisISAdjEntry *)locateEntry(&pCircEntry->isisISAdjTable, adj->name);
		if (pISAdjEntry) {
		    freshISAdjEntry(pISAdjEntry, adj);		/* Isis-Mib */
		    freshISAdjAreaAddrTable(pISAdjEntry, adj);
		    pISAdjEntry->isisISAdjState = 2;		/* state <= up */
		} else {
	            trace(TR_ISIS, LOG_ERR, "adjUpEvent(): cannot locate entry in IS adj table for localId==%d, adj.name==%d", 
			  circuit->localId, adj->name);
		}
            } else {
	        trace(TR_ISIS, LOG_ERR, "adjUpEvent(): cannot locate circ mib entry for localId==%d", circuit->localId);
            }
	}
#endif	/* PROTO_SNMP */
}


void
adjDownEvent(circuit, adj, reason)
CircuitEntry	*circuit;	/* circuit adjacency is on */
AdjacencyEntry	*adj;
ReasonCode	reason;		/* reason for failure */
{
	if ((adj->state == AdjUp) && (reason == OneWayConnection)) {
		/* we went from up to initializing - neighbor must have restarted */
		logAdjEvent(circuit, adj, "AdjDownEvent (up -> initializing)");
		lspAdjDown(circuit, adj);
		adj->state = AdjInitializing;
	} else if (adj->state == AdjUp) {
		logAdjEvent(circuit, adj, "AdjDownEvent");
		adj->state = AdjFailed;
		lspAdjDown(circuit, adj);
	} else if (adj->state == AdjInitializing) {
		/* we went from initializing to down */
		logAdjEvent(circuit, adj, "AdjDownEvent (initializing -> down)");
	}
#ifdef	PROTO_SNMP
	{
    	    IsisMibEntry	*pMibEntry;
    	    IsisCircEntry	*pCircEntry;
	    IsisISAdjEntry	*pISAdjEntry;

    	    pMibEntry = (IsisMibEntry *)locateEntry(&isisMibTable, 1);
    	    pCircEntry = (IsisCircEntry *)locateEntry(&pMibEntry->isisCircTable, circuit->localId);
    	    if (pCircEntry) {
		pISAdjEntry = (IsisISAdjEntry *)locateEntry(&pCircEntry->isisISAdjTable, adj->name);
		if (pISAdjEntry) {
		    if (adj->state == AdjFailed) {
		        if (queueDelete(&pCircEntry->isisISAdjTable, pISAdjEntry)) {
	                    trace(TR_ISIS, LOG_ERR, 
			          "adjDownEvent(): cannot delete entry in IS adj table for circIndex==%d, adjIndex==%d", 
			          pISAdjEntry->isisISAdjIndex, pISAdjEntry->isisISAdjIndex); /* Isis-Mib */
		        }
		    } else {
		    	pISAdjEntry->isisISAdjState = 1;		/* state <= initializing */
		    }
		} else {
	            trace(TR_ISIS, LOG_ERR, "adjDownEvent(): cannot locate entry in IS adj table for localId==%d, adj.name==%d", 
			  circuit->localId, adj->name);
		}
            } else {
	        trace(TR_ISIS, LOG_ERR, "adjDownEvent(): cannot locate circ mib entry for localId==%d", circuit->localId);
            }
	}
#endif	/* PROTO_SNMP */

	if (adj->state == AdjFailed) {
		removeAdjFromHash(adj);
		/* 
	 	 *	if we are called due to a timer expiration, the adj has already been
	 	 *	removed from the linked list. Otherwise, remove it now.
	 	 */
		if (reason != HoldingTimerExpired)
			DLLRemque(adj);

		/* schedule this adjacency to be deleted after next spf is run */
		addOldAdjacency(adj);
	}
}


void
desigSysChangeEvent(circuit, level, elected)
CircuitEntry	*circuit;
int				level;
Boolean			elected; 	/* true => elected, false => resigned */
{
	trace(TR_ISIS, LOG_INFO, "Designated System Change: %s: L%d %s", circuit->name, level, 
		elected ? "Elected" : "Resigned");
        if (level == 1) {
                circuit->l1DesignateChanges++;
#ifdef	PROTO_SNMP
		{
		    IsisMibEntry	*pMibEntry;
	    	    IsisCircEntry	*pCircEntry;

		    pMibEntry = (IsisMibEntry *)locateEntry(&isisMibTable, 1);
		    pCircEntry = (IsisCircEntry *)locateEntry(&pMibEntry->isisCircTable,
								circuit->localId);
		    if (pCircEntry) {
		        ++pCircEntry->isisCircLANL1DesISChanges;	/* Isis-Mib */
		    }
		    else {
			trace(TR_ISIS, LOG_ERR, "desigSysChangeEvent(): cannot locate circ mib entry for localId==%d", circuit->localId);
		    }
		}
#endif	/* PROTO_SNMP */

	}
        else {
                circuit->l2DesignateChanges++;
#ifdef	PROTO_SNMP
		{
	    	    IsisMibEntry	*pMibEntry;
	    	    IsisCircEntry	*pCircEntry;

		    pMibEntry = (IsisMibEntry *)locateEntry(&isisMibTable, 1);
		    pCircEntry = (IsisCircEntry *)locateEntry(&pMibEntry->isisCircTable,
								circuit->localId);
		    if (pCircEntry) {
		        ++pCircEntry->isisCircLANL2DesISChanges;	/* Isis-Mib */
		    }
		    else {
			trace(TR_ISIS, LOG_ERR, "desigSysChangeEvent: cannot locate circ mib entry for localId==%d", circuit->localId);
		    }
		}
#endif	/* PROTO_SNMP */

	}
}


void
logAdjEvent(circuit, adj, msg)
CircuitEntry	*circuit;
AdjacencyEntry	*adj;
char			*msg;
{
	int i;

	tracef("%s: %s: %s %s", msg, circuit->name,
		systemTypeToStr(adj->adjacencyType),
		adj->level2Only ? "(L2 only)" : "");

        circuit->adjacencyEvents++;
#ifdef	PROTO_SNMP
	{
	    IsisMibEntry	*pMibEntry;
	    IsisCircEntry	*pCircEntry;

	    pMibEntry = (IsisMibEntry *)locateEntry(&isisMibTable, 1);
	    pCircEntry = (IsisCircEntry *)locateEntry(&pMibEntry->isisCircTable,
								circuit->localId);
	    if (pCircEntry) {
	        ++pCircEntry->isisCircAdjChanges;	/* Isis-Mib */
	    }
	    else {
		trace(TR_ISIS, LOG_ERR, "logAdjEvent(): cannot locate circ mib entry for localId==%d", circuit->localId);
	    }

	}
#endif	/* PROTO_SNMP */

	switch (adj->adjacencyType) {
		case L1IS:
		case L2IS:
#define	HEXPRINT
#ifdef	HEXPRINT
			trace(TR_ISIS, LOG_INFO, "system %s", IDToStr(adj->neighborSysID, 6));
#else
			trace(TR_ISIS, LOG_INFO, "system %s", adj->neighborSysID);
#endif	/* HEXPRINT */
			break;
		case ES:
			if (circuit->circuitType == Broadcast)
				trace(TR_ISIS, LOG_INFO, "SNPA %s", IDToStr(adj->neighborMAC, 6));
			else {
				tracef("Neighbor%s", (adj->numIds>1) ? "s":" ");
                        	for (i=0; i<adj->numIds; i++) {
                                	tracef("%s%c", IDToStr(&adj->neighborInfo[i*6], 6),
                                        	i<(adj->numIds-1)?',':' ');
                        	}
                        	trace(TR_ISIS, LOG_INFO, NULL);
			}

			break;
	}
}

/*
 *	Called when a circuit is turned on
 *
 *	1) transmit an IIH PDU
 *	2) solicit the end system configuration
 *	3) set a timer to run the election process
 */
void
circuitUpEvent(c)
CircuitEntry *c;
{
	c->circuitChanges++;

	if (c->circuitType == Broadcast) {
		switch (systemType) {
			case L1IS:
				circuitL1IIHTimeout(c);
				break;

			case L2IS:
				if (!c->manualL2Only) {
					circuitL1IIHTimeout(c);
				}
				circuitL2IIHTimeout(c);
				break;
		}

		setTimer(sysIIHInterval * 2, lanFirstTimeElection, c);
		solicitESConfig(c);
	} else if (c->circuitType == Pt2Pt) {
		circuitP2PIIHTimeout(c);
	}

	lspCircuitUp(c);
}


/*
 * ------------------------------------------------------------------------
 * 
 * 	GateD, Release 3
 * 
 * 	Copyright (c) 1990,1991,1992,1993 by Cornell University
 * 	    All rights reserved.
 * 
 * 	THIS SOFTWARE IS PROVIDED "AS IS" AND WITHOUT ANY
 * 	EXPRESS OR IMPLIED WARRANTIES, INCLUDING, WITHOUT
 * 	LIMITATION, THE IMPLIED WARRANTIES OF MERCHANTABILITY
 * 	AND FITNESS FOR A PARTICULAR PURPOSE.
 * 
 * 	Royalty-free licenses to redistribute GateD Release
 * 	3 in whole or in part may be obtained by writing to:
 * 
 * 	    GateDaemon Project
 * 	    Information Technologies/Network Resources
 * 	    200 CCC
 * 	    Cornell University
 * 	    Ithaca, NY  14853-2601  USA
 * 
 * 	GateD is based on Kirton's EGP, UC Berkeley's routing
 * 	daemon	 (routed), and DCN's HELLO routing Protocol.
 * 	Development of GateD has been supported in part by the
 * 	National Science Foundation.
 * 
 * 	Please forward bug fixes, enhancements and questions to the
 * 	gated mailing list: gated-people@gated.cornell.edu.
 * 
 * 	Authors:
 * 
 * 		Jeffrey C Honig <jch@gated.cornell.edu>
 * 		Scott W Brim <swb@gated.cornell.edu>
 * 
 * ------------------------------------------------------------------------
 * 
 *       Portions of this software may fall under the following
 *       copyrights:
 * 
 * 	Copyright (c) 1988 Regents of the University of California.
 * 	All rights reserved.
 * 
 * 	Redistribution and use in source and binary forms are
 * 	permitted provided that the above copyright notice and
 * 	this paragraph are duplicated in all such forms and that
 * 	any documentation, advertising materials, and other
 * 	materials related to such distribution and use
 * 	acknowledge that the software was developed by the
 * 	University of California, Berkeley.  The name of the
 * 	University may not be used to endorse or promote
 * 	products derived from this software without specific
 * 	prior written permission.  THIS SOFTWARE IS PROVIDED
 * 	``AS IS'' AND WITHOUT ANY EXPRESS OR IMPLIED WARRANTIES,
 * 	INCLUDING, WITHOUT LIMITATION, THE IMPLIED WARRANTIES OF
 * 	MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 * ------------------------------------------------------------------------
 * 
 * 	Copyright 1991 D.L.S. Associates
 * 
 * 	Permission to use, copy, modify, distribute, and sell this software
 * 	and its documentation for any purpose is hereby granted without
 * 	fee, provided that the above copyright notice appear in all copies
 * 	and that both that copyright notice and this permission notice
 * 	appear in supporting documentation, and that the name of D.L.S. not
 * 	be used in advertising or publicity pertaining to distribution of
 * 	the software without specific, written prior permission.  D.L.S.
 * 	makes no representations about the suitability of this software for
 * 	any purpose.  It is provided "as is" without express or implied
 * 	warranty.
 * 
 * 	D.L.S. DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
 * 	INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN
 * 	NO EVENT SHALL D.L.S.  BE LIABLE FOR ANY SPECIAL, INDIRECT OR
 * 	CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS
 * 	OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT,
 * 	NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN
 * 	CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 * 
 * 	Authors:  Robert Hagens and Dan Schuh
 * 
 * 
 */
