#include <stdio.h>
#include <sys/types.h>
#include <sys/ptrace.h>
#define __KERNEL__
#include <linux/ipc.h>
#include <linux/sem.h>
#include <linux/msg.h>
#include <linux/shm.h>
#include "defs.h"


int sys_semop(), sys_semget(), sys_semctl();
int sys_msgsnd(), sys_msgrcv(), sys_msgget(), sys_msgctl();
int sys_shmat(), sys_shmdt(), sys_shmget(), sys_shmctl();

extern int errno;

static struct sysent {
      int nargs;
      int ifunc;
      int (*sys_func) ();
      char *sys_name;
} sysent[] = {
    {4, SEMOP, sys_semop, "semop"},
    {4, SEMGET, sys_semget, "semget"},
    {4, SEMCTL, sys_semctl, "semctl"},
    {4, MSGSND, sys_msgsnd, "msgsnd"},
    {4, MSGRCV, sys_msgrcv, "msgrcv"},
    {4, MSGGET, sys_msgget, "msgget"},
    {4, MSGCTL, sys_msgctl, "msgctl"},
    {4, SHMAT, sys_shmat, "shmat"},
    {4, SHMDT, sys_shmdt, "shmdt"},
    {4, SHMGET, sys_shmget, "shmget"},
    {4, SHMCTL, sys_shmctl, "shmctl"},
    {0, 0, NULL, NULL}
};


static Xlat msgctl[] = {
    IPC_RMID, 	"IPC_RMID",
    IPC_SET, 	"IPC_SET",
    IPC_STAT, 	"IPC_STAT",
    IPC_INFO, 	"IPC_INFO",
    MSG_STAT, 	"MSG_STAT",
    MSG_INFO, 	"MSG_INFO",
    0, 		NULL,
};



static Xlat semctl[] = {
    IPC_RMID, 	"IPC_RMID",
    IPC_SET, 	"IPC_SET",
    IPC_STAT, 	"IPC_STAT",
    IPC_INFO, 	"IPC_INFO",
    SEM_STAT, 	"SEM_STAT",
    SEM_INFO, 	"SEM_INFO",
    GETPID, 	"GETPID",
    GETVAL, 	"GETVAL",
    GETALL, 	"GETALL",
    GETNCNT, 	"GETNCNT",
    GETZCNT, 	"GETZCNT",
    SETVAL, 	"SETVAL",
    SETALL, 	"SETALL",
    0, 		NULL,
};


static Xlat shmctl[] = {
    IPC_RMID, 	"IPC_RMID",
    IPC_SET, 	"IPC_SET",
    IPC_STAT, 	"IPC_STAT",
    IPC_INFO, 	"IPC_INFO",
    SHM_STAT, 	"SHM_STAT",
    SHM_INFO, 	"SHM_INFO",
    SHM_LOCK, 	"SHM_LOCK",
    SHM_UNLOCK, "SHM_UNLOCK",
    0, 		NULL,
};


static Xlat resource_flags[] = {
    IPC_CREAT, 	"IPC_CREAT",
    IPC_EXCL, 	"IPC_EXCL",
    IPC_NOWAIT, "IPC_NOWAIT",
    0, 		NULL,
};


static Xlat shm_flags[] = {
    SHM_REMAP, 	"SHM_REMAP",
    SHM_RDONLY, "SHM_RDONLY",
    SHM_RND, 	"SHM_RND",
    0, 		NULL,
};


static Xlat msg_flags[] = {
    MSG_NOERROR, "MSG_NOERROR",
    MSG_EXCEPT, "MSG_EXCEPT",
    IPC_NOWAIT, "IPC_NOWAIT",
    0, 		NULL,
};



int sys_ipc(tcp)
struct tcb *tcp;
{
    int ipccall;
    struct sysent *se;
    char *addr;
    int i;

    if (entering(tcp))
	ipccall = tcp->u_args[7] = tcp->u_args[0];
    else
	ipccall = tcp->u_args[7];


    for (se = sysent; se->sys_name; se++) {
    	if (se->ifunc == ipccall)
		break;
    }

    if (!se->sys_name) {
	if (entering(tcp))
	    fprintf(outf, "ipc(%d, %#x", ipccall, tcp->u_args[1]);
	return 0;
    }

    if (entering(tcp)) {
	fprintf(outf, "%s(", se->sys_name);
	addr = (char *) tcp->u_args[1];
	for (i = 0; i < se->nargs; i++, addr += 4) {
	    tcp->u_args[i] = tcp->u_args[i + 1];
	}
    }
    return (*(se->sys_func)) (tcp);
}



int sys_msgget(tcp)
struct tcb *tcp;
{

    if (entering(tcp)) {
	if (tcp->u_args[0])
	    fprintf(outf, "%u", tcp->u_args[0]);
	else
	    fprintf(outf, "IPC_PRIVATE");
	fprintf(outf, ", ");
	if (printflags(resource_flags, tcp->u_args[1]) != 0)
	    fprintf(outf, "|");
	fprintf(outf, "%#.4o", tcp->u_args[1] & 0666);
    }
    return 0;
}



int sys_msgctl(tcp)
struct tcb *tcp;
{
    char *cmd = xlookup(msgctl, tcp->u_args[1]);

    if (entering(tcp)) {
	fprintf(outf, "%u", tcp->u_args[0]);
	fprintf(outf, ", %s", cmd == NULL ? "MSG_???" : cmd);
	fprintf(outf, ", %#x", tcp->u_args[3]);
    }
    return 0;
}


int sys_msgsnd(tcp)
struct tcb *tcp;
{
    long mtype;

    if (entering(tcp)) {
	fprintf(outf, "%u", tcp->u_args[0]);
	umove(tcp->pid, tcp->u_args[3], sizeof mtype, (char *) &mtype);
	fprintf(outf, ", {%u, ", mtype);
	printstr(tcp->pid, tcp->u_args[3] + sizeof(long), tcp->u_args[1]);
	fprintf(outf, "}, %u", tcp->u_args[1]);
	fprintf(outf, ", ");
	if (printflags(msg_flags, tcp->u_args[2]) == 0)
	    fprintf(outf, "0");
    }
    return 0;
}


int sys_msgrcv(tcp)
struct tcb *tcp;
{
    long mtype;
    struct ipc_kludge tmp;

    if (exiting(tcp)) {
	fprintf(outf, "%u", tcp->u_args[0]);
	umove(tcp->pid, tcp->u_args[3], sizeof tmp, (char *) &tmp);
	umove(tcp->pid, tmp.msgp, sizeof mtype, (char *) &mtype);
	fprintf(outf, ", {%u, ", mtype);
	printstr(tcp->pid, (int) (tmp.msgp) + sizeof(long), tcp->u_args[1]);
	fprintf(outf, "}, %u", tcp->u_args[1]);
	fprintf(outf, ", %d", tmp.msgtyp);
	fprintf(outf, ", ");
	if (printflags(msg_flags, tcp->u_args[2]) == 0)
	    fprintf(outf, "0");
    }
    return 0;
}



int sys_semop(tcp)
struct tcb *tcp;
{

    if (entering(tcp)) {
	fprintf(outf, "%u", tcp->u_args[0]);
	fprintf(outf, ", %#x", tcp->u_args[3]);
	fprintf(outf, ", %u", tcp->u_args[1]);
    }
    return 0;
}



int sys_semget(tcp)
struct tcb *tcp;
{

    if (entering(tcp)) {
	if (tcp->u_args[0])
	    fprintf(outf, "%u", tcp->u_args[0]);
	else
	    fprintf(outf, "IPC_PRIVATE");
	fprintf(outf, ", %u", tcp->u_args[1]);
	fprintf(outf, ", ");
	if (printflags(resource_flags, tcp->u_args[2]) != 0)
	    fprintf(outf, "|");
	fprintf(outf, "%#.4o", tcp->u_args[2] & 0666);
    }
    return 0;
}



int sys_semctl(tcp)
struct tcb *tcp;
{
    char *cmd = xlookup(semctl, tcp->u_args[2]);

    if (entering(tcp)) {
	fprintf(outf, "%u", tcp->u_args[0]);
	fprintf(outf, ", %u", tcp->u_args[1]);
	fprintf(outf, ", %s", cmd == NULL ? "SEM_???" : cmd);
	fprintf(outf, ", %#x", tcp->u_args[3]);
    }
    return 0;
}



int sys_shmget(tcp)
struct tcb *tcp;
{

    if (entering(tcp)) {
	if (tcp->u_args[0])
	    fprintf(outf, "%u", tcp->u_args[0]);
	else
	    fprintf(outf, "IPC_PRIVATE");
	fprintf(outf, ", %u", tcp->u_args[1]);
	fprintf(outf, ", ");
	if (printflags(resource_flags, tcp->u_args[2]) != 0)
	    fprintf(outf, "|");
	fprintf(outf, "%#.4o", tcp->u_args[2] & 0666);
    }
    return 0;
}


int sys_shmctl(tcp)
struct tcb *tcp;
{
    char *cmd = xlookup(shmctl, tcp->u_args[1]);

    if (entering(tcp)) {
	fprintf(outf, "%u", tcp->u_args[0]);
	fprintf(outf, ", %s", cmd == NULL ? "SHM_???" : cmd);
	fprintf(outf, ", %#x", tcp->u_args[3]);
    }
    return 0;
}


int sys_shmat(tcp)
struct tcb *tcp;
{
    unsigned long raddr;

    if (exiting(tcp)) {
	fprintf(outf, "%u", tcp->u_args[0]);
	fprintf(outf, ", %#x", tcp->u_args[3]);
	fprintf(outf, ", ");
	if (printflags(shm_flags, tcp->u_args[1]) == 0)
	    fprintf(outf, "0");
	umove(tcp->pid, tcp->u_args[2], sizeof raddr, (char *) &raddr);
	tcp->u_rval = raddr;
    }
    return RVAL_HEX;
}


int sys_shmdt(tcp)
struct tcb *tcp;
{
    if (entering(tcp)) {
	fprintf(outf, "%#x", tcp->u_args[3]);
    }
    return 0;
}

