#include <mbase.h>
#include <parse.h>

#ifdef MSDOS
#include <direct.h>   /* For getcwd() */
#endif


/*
 * PROTOTYPES -----------------------------------------------------------------
 *
 */

      relation *getrel    XARGS( (char *) );
      void      addTag    XARGS( (char *, char *, char *) );
      void      sortTags  XARGS( (void) );
      void      chap5     XARGS( (FParse *) );
      void      chap7     XARGS( (FParse *) );
      void      chap8to10 XARGS( (FParse *, char *) );
      void      chap12    XARGS( (FParse *) );
      void      getname   XARGS( (char *, char *) );
      bool      fContains XARGS( (char *, char *) );


/*
 * DOCTAGS searches through six documentation chapters in this directory
 * and creates a VI-compatible tagfile for looking up functions, error
 * codes, and compile-time options.
 *
 * The following chapters are searched:
 *
 *
 *    Compile-time options..................................Chapter 5
 *
 *    Callback services.....................................Chapter 7
 *
 *    Library: database routines............................Chapter 8
 *
 *    Library: time, date, phone number routines............Chapter 9
 *
 *    Library: general routines.............................Chapter 10
 *
 *    Error codes...........................................Chapter 12
 *
 *
 * If you don't use VI (or another editor which recognizes tagfiles), then
 * it's probably not worth your effort to build this.
 *
 * There are no arguments to DOCTAGS' command-line, nor any compile-time
 * options; DOCTAGS should be run from the same directory where the above
 * six chapters are found, and will create a TAGS file in that directory.
 * Note that this code is a huge hack, and will probably not work properly if
 * you change the text in the chapters.
 *
 * The tagfile will be given full pathnames to the documentation, so you can
 * add the tagfile to your TAGS= environment variable and use ctrl-] within
 * VI to look up relevant documentation from anywhere.
 *
 */

   relation *rel;
   file      fh;

void
main ()
{
   FParse *fp;
   char    tempname[64];

/*
 * Make sure we have a tagfile available, and a handle open to it:
 *
 */

   if (access ("tags", 0) != -1)  /* Following the CTAGS tradition, if there */
      {                           /* is an existing tag file, we'll just     */
      unlink ("tags");            /* delete it.  Rather unkind, eh?          */
      }

   printf ("this program will create a VI-compatible TAGS file in the\n");
   printf ("current directory.  Only chapters 5,7,8,9,10 and 12 will be\n");
   printf ("tagged.\n\n");

   if ((fh = creatx ("tags")) <= 0)
      {
      fprintf (stderr, "cannot create tagfile in current directory\n");
      mb_exit (1);
      }

   close (fh);
   if ((fh = openx ("tags", OPENMODE)) <= 0)
      {
      fprintf (stderr, "cannot write to tagfile in current directory\n");
      unlink ("tags");
      mb_exit (1);
      }

/*
 * Now create a temporary relation, into which we'll add our tags so they'll
 * be sorted automatically...
 *
 */

    if ((rel = getrel (tempname)) == RNULL)
       {
       fprintf (stderr, "cannot create temporary relation\n");
       unlink ("tags");
       mb_exit (1);
       }

/*
 * First, chapter 5...
 *
 */

   if ((fp = fpInit ("chapter.5", FALSE)) == NULL)
      fprintf (stderr, "cannot find chapter 5--no tags included for it\n");
   else
      {
      fprintf (stderr, "working on chapter 5...\n");
      chap5 (fp);
      fpClose (fp);
      }

/*
 * Then the callback services in chapter 7...
 *
 */

   if ((fp = fpInit ("chapter.7", FALSE)) == NULL)
      fprintf (stderr, "cannot find chapter 7--no tags included for it\n");
   else
      {
      fprintf (stderr, "working on chapter 7...\n");
      chap7 (fp);
      fpClose (fp);
      }

/*
 * Then the functions in 8, 9 and 10
 *
 */

   if ((fp = fpInit ("chapter.8", FALSE)) == NULL)
      fprintf (stderr, "cannot find chapter 8--no tags included for it\n");
   else
      {
      fprintf (stderr, "working on chapter 8...\n");
      chap8to10 (fp, "chapter.8");
      fpClose (fp);
      }

   if ((fp = fpInit ("chapter.9", FALSE)) == NULL)
      fprintf (stderr, "cannot find chapter 9--no tags included for it\n");
   else
      {
      fprintf (stderr, "working on chapter 9...\n");
      chap8to10 (fp, "chapter.9");
      fpClose (fp);
      }

   if ((fp = fpInit ("chapter.10", FALSE)) == NULL)
      fprintf (stderr, "cannot find chapter 10--no tags included for it\n");
   else
      {
      fprintf (stderr, "working on chapter 10...\n");
      chap8to10 (fp, "chapter.10");
      fpClose (fp);
      }

/*
 * Finally the errors in chapter 12...
 *
 */

   if ((fp = fpInit ("chapter.12", FALSE)) == NULL)
      fprintf (stderr, "cannot find chapter 12--no tags included for it\n");
   else
      {
      fprintf (stderr, "working on chapter 12...\n");
      chap12 (fp);
      fpClose (fp);
      }

/*
 * Now create the tags file
 *
 */

   fprintf (stderr, "creating tagfile...\n");

   sortTags ();

   fprintf (stderr, "tagfile created--%ld tags\n", mb_num(rel));

   close    (fh);
   unlink   (tempname);

   mb_exit (0);
}


/*
 * Tagfile format:
 *    |
 *    | keyword  -- function name, error code or compile-time option
 *    | tab
 *    | filename -- full path, so we can look up dox from anywhere
 *    | tab
 *    | ?        -- indicates we should begin a search
 *    | ^        -- indicates beginning-of-line
 *    | search   -- line to search for
 *    | ?        -- indicates end-of-search
 *    | CR       -- end-of-line; append LF for MS-DOS systems
 *    |
 *
 * Of course, the lines have to be sorted.  That's what sucks...
 *
 */

typedef struct
   {
   char  kw[20];
   char  name[64];
   char  search[50];
   } filestr;

relation *
getrel (name)
char   *name;
{
   relation *r;

   if (! *( GetTmpName(name)))
      {
      return RNULL;
      }
   strcat (name, ".rel");

   if ((r = mb_new ()) == RNULL)
      {
      fprintf (stderr, "%s\n", mb_error);
      return RNULL;
      }

   if (mb_addfield (r, "kw", T_CHAR, 20) != MB_OKAY)
      {
      fprintf (stderr, "%s\n", mb_error);
      return RNULL;
      }
   if (mb_addfield (r, "file", T_CHAR, 64) != MB_OKAY)
      {
      fprintf (stderr, "%s\n", mb_error);
      return RNULL;
      }
   if (mb_addfield (r, "search", T_CHAR, 50) != MB_OKAY)
      {
      fprintf (stderr, "%s\n", mb_error);
      return RNULL;
      }
   if (mb_addindex (r, "ix_kw", TRUE, "0") != MB_OKAY)
      {
      fprintf (stderr, "%s\n", mb_error);
      return RNULL;
      }
   if (mb_create (r, name, 0) != MB_OKAY)
      {
      fprintf (stderr, "%s\n", mb_error);
      return RNULL;
      }

   return mb_inc (name, "");
}

filestr rec;

void
addTag (kw, name, search)
char   *kw,*name,*search;
{
   strzcpy (rec.kw,     kw,     19);
   strzcpy (rec.name,   name,   63);
   strzcpy (rec.search, search, 49);

   mb_add (rel, &rec);
}

void
sortTags ()
{
   mb_action act;
   char      line[128];

   for (act = FIRST; ; act = NEXT)
      {
      if (mb_sel (rel, 0, &rec, act, NULL) != MB_OKAY)
         break;

      sprintf (line, "%s\t%s\t?^%s?%s", rec.kw, rec.name, rec.search, szEOL);
      writx (fh, line, strlen(line));
      }
}


/*
 *
 *  5 - search for 'by default',
 *         keyword = first word after spaces and before '.'
 *
 *  7 - search for '...'; keyword = first word after spaces and before '.'
 *
 *  8 -
 *  9 -
 * 10 - search for "Function.." or "Macro.."-- for each line,
 *         keyword = first word after '.' runs, before space or '('
 *      if '[' on line,
 *         keyword = first word after '[' and spaces, and before '(' or spaces
 *                   (skip 'or')
 *
 *      search for "Macros.."-- for each line,
 *         start after periods...
 *            keyword = word until '(' or ' '...
 *            try again after passing spaces, if not EOL
 *
 * 12 - search for '---', on lines not starting with '-'
 *         keyword = first word, from start of line and before ' '
 *
 */

void
chap5  (fp)
FParse *fp;
{
   char  kw[30], name[128];
   char *search;
   char *trg, *src;

   getname (name, "chapter.5");

   for ( ; !fpEOF(fp) && (search = fpLine (fp)) != NULL; )
      {
      if (! fContains (search, "by default"))
         continue;

      for (src = search; *src == ' '; src++)
         ;
      for (trg = kw; *src && *src != '.' && *src != ' '; src++, trg++)
         *trg = *src;
      *trg = 0;

      addTag (kw, name, search);
      }
}


void
chap7  (fp)
FParse *fp;
{
   char  kw[30], name[128];
   char *search;
   char *trg, *src;

   getname (name, "chapter.7");

   for ( ; !fpEOF(fp) && (search = fpLine (fp)) != NULL; )
      {
      if (! fContains (search, "..."))
         continue;

      for (src = search; *src == ' '; src++)
         ;
      for (trg = kw; *src && *src != '.' && *src != ' '; src++, trg++)
         *trg = *src;
      *trg = 0;

      addTag (kw, name, search);
      }
}


void
chap8to10 (fp, filename)
FParse    *fp;
char          *filename;
{
   char  kw[30], name[128];
   char *search;
   char *trg, *src;

   getname (name, filename);

   for ( ; !fpEOF(fp) && (search = fpLine (fp)) != NULL; )
      {
      if (fContains (search, "Function..") || fContains (search, "Macro....."))
         {
         for (src = search; *src && *src != '.'; src++)
            ;
         for ( ; *src == '.'; src++)
            ;
         for (trg = kw; *src && *src != ' ' && *src != '('; trg++,src++)
            *trg = *src;
         *trg = 0;

         addTag (kw, name, search);

         if ((src = strchr (search, '[')) != NULL)
            {
            src += 5;  /* +1=' ', +2='o', +3='r', +4=' ', +5=function name */
            for (trg = kw; *src && *src != '(' && *src != ' '; src++, trg++)
               *trg = *src;
            *trg = 0;

            addTag (kw, name, search);
            }
         }
      else
      if (fContains (search, "Macros...."))
         {
         for (src = search; *src && *src != '.'; src++)
            ;
         for ( ; *src == '.'; src++)
            ;

         for ( ; *src; )
            {
            for (trg = kw; *src && *src != ' ' && *src != '('; trg++,src++)
               *trg = *src;
            *trg = 0;

            addTag (kw, name, search);

            for ( ; *src && *src != ' '; src++)
               ;
            for ( ; *src == ' '; src++)
               ;
            }
         }
      }
}


void
chap12 (fp)
FParse *fp;
{
   char  kw[30], name[128];
   char *search;
   char *trg, *src;

   getname (name, "chapter.12");

   for ( ; !fpEOF(fp) && (search = fpLine (fp)) != NULL; )
      {
      if (! fContains (search, "---"))
         continue;
      if (*search == '-' || *search == ' ')
         continue;

      for (src = search; *src == ' '; src++)
         ;
      for (trg = kw; *src && *src != ' '; src++, trg++)
         *trg = *src;
      *trg = 0;

      addTag (kw, name, search);
      }
}

void
getname (trg, src)
char    *trg,*src;
{
   char  temp[64];

   getcwd (temp, 63);
   sprintf (trg, "%s%c%s", temp, DIRSEP, src);
}

/*
 * Normally I'd use strstr() for this, but some platforms don't have that
 * (Xenix, for example).
 *
 */

bool
fContains (src, trg)
char      *src,*trg;
{
   int  n, m;

   for (n = strlen(src), m = strlen(trg); *src; src++, n--)
      {
      if (! strncmp (src, trg, min(n,m)))
         return TRUE;
      }

   return FALSE;
}

