------------------------------------------------------------------------------
--                                                                          --
--                         GNAT COMPILER COMPONENTS                         --
--                                                                          --
--                             W I D E C H A R                              --
--                                                                          --
--                                 S p e c                                  --
--                                                                          --
--                            $Revision: 1.3 $                              --
--                                                                          --
--           Copyright (c) 1992,1993,1994 NYU, All Rights Reserved          --
--                                                                          --
-- GNAT is free software;  you can  redistribute it  and/or modify it under --
-- terms of the  GNU General Public License as published  by the Free Soft- --
-- ware  Foundation;  either version 2,  or (at your option) any later ver- --
-- sion.  GNAT is distributed in the hope that it will be useful, but WITH- --
-- OUT ANY WARRANTY;  without even the  implied warranty of MERCHANTABILITY --
-- or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License --
-- for  more details.  You should have  received  a copy of the GNU General --
-- Public License  distributed with GNAT;  see file COPYING.  If not, write --
-- to the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA. --
--                                                                          --
------------------------------------------------------------------------------

--  This package encapsulates the method used to represent wide characters
--  using ESC sequences. The assumption is that each wide character is always
--  represented with a separate ESC sequences, starting with an ESC character.
--  It is also assumed that the ESC sequence cannot contain an embedded
--  format effector character (CR, LF, FF, VT, HT).

--  Other methods of representing wide characters (Unicode, shift in/out etc)
--  will require more extensive changes to the scanner.

--  The representation used in this version is as follows:

--   If the character code is in the range 16#0000# .. 16#00FF# then it is
--   never represented using an ESC sequence, instead the second byte
--   appears as a Latin-1 character in the usual manner.

--   If both the high order and low order bytes of the 16-bit value are in
--   the range 16#20# .. 16#FF#, then the ESC sequence is simply:

--      ESC   high-order-byte   low-order-byte

--   If either the high order or low order bytes of the 16-bit value are in
--   the range 16#00# .. 16#1F#, then the ESC sequence is:

--      ESC  16#00#  flag-byte  high-order-byte  low-order-byte

--   where flag-byte has one of the following settings:

--      1 = subtract 16#20# from high order byte value
--      2 = subtract 16#20# from low order byte value
--      3 = subtract 16#20# from both low and high order byte value

--   This allows the encoding of these cases without control characters
--   appearing in the ESC sequence as required by the rules.

with Types; use Types;

package Widechar is

   procedure Scan_Wide
     (S : Source_Buffer_Ptr;
      P : in out Source_Ptr;
      C : out Char_Code;
      E : out Boolean);
   --  On entry S (P) points to an ESC character that starts a wide character
   --  escape sequence. If no error is found, the value stored in C is the
   --  character code for this wide character, P is updated past the escape
   --  sequence and E is set to False. If an error is found, then P points
   --  to the improper character, C is undefined, and E is set to True.

   procedure Set_Escape
     (C : Char_Code;
      S : in out String;
      P : in out Natural);
   --  The escape sequence (including the leading ESC character) for the
   --  given character code is stored starting at S (P + 1), and on return
   --  P points to the last stored character (i.e. P is the count of stored
   --  characters on entry and exit, and the escape sequence is appended to
   --  the end of the stored string). The character code C must represent a
   --  wide character value, i.e. be in the range 16#0100# .. 16#FFFF#.

   procedure Skip_Escape (S : String; P : in out Natural);
   --  On entry, S (P) points to an ESC character for a wide character
   --  escape sequence. On exit, P is bumped past the wide character escape
   --  sequence. No error checking is done, since this is only used on escape
   --  sequences generated by Set_Escape, which are known to be correct.

end Widechar;
