------------------------------------------------------------------------------
--                                                                          --
--                         GNAT COMPILER COMPONENTS                         --
--                                                                          --
--                               S I N P U T                                --
--                                                                          --
--                                 S p e c                                  --
--                                                                          --
--                            $Revision: 1.24 $                             --
--                                                                          --
--           Copyright (c) 1992,1993,1994 NYU, All Rights Reserved          --
--                                                                          --
-- GNAT is free software;  you can  redistribute it  and/or modify it under --
-- terms of the  GNU General Public License as published  by the Free Soft- --
-- ware  Foundation;  either version 2,  or (at your option) any later ver- --
-- sion.  GNAT is distributed in the hope that it will be useful, but WITH- --
-- OUT ANY WARRANTY;  without even the  implied warranty of MERCHANTABILITY --
-- or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License --
-- for  more details.  You should have  received  a copy of the GNU General --
-- Public License  distributed with GNAT;  see file COPYING.  If not, write --
-- to the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA. --
--                                                                          --
------------------------------------------------------------------------------

--  This package contains the input routines used for reading the
--  input source file. The actual I/O routines are in OS_Interface,
--  with this module containing only the system independent processing.

--  General Note: throughout the compiler, we use the term line or source
--  line to refer to a physical line in the source, terminated by the end of
--  physical line sequence. See Skip_Line_Terminators procedure for a full
--  description of the difference between logical and physical lines.

with Casing; use Casing;
with Table;
with Types;  use Types;

package Sinput is

   -----------------------
   -- Source File Table --
   -----------------------

   --  The source file table has an entry for each source file read in for
   --  this run of the compiler. This table is (default) initialized when
   --  the compiler is loaded, and simply accumulates entries as compilation
   --  proceeds and the Load_Source_File procedure is called to load required
   --  source files.

   --  In the case where there are multiple main units (e.g. in the case of
   --  the cross-reference tool), this table is not reset between these units,
   --  so that a given source file is only read once if it is used by two
   --  separate main units.

   --  The entries in the table are accessed using a Source_File_Index that
   --  ranges from 1 to Last_Source_File. Each entry has the following fields

   --  File_Name : File_Name_Type
   --    Name of the source file (simple name with no directory information).
   --    Set by Load_Source_File and cannot be subequently changed.

   --  Full_File_Name : File_Name_Type
   --    Full file name (full name with directory info), used for generation
   --    of error messages, etc. Set by Load_Source_File and cannot be
   --    subsequently changed.

   --  Source_Text : Source_Buffer_Ptr
   --    Text of source file. Note that every source file has a distinct set
   --    of non-overlapping bounds, so it is possible to determine which file
   --    is referenced from a given subscript (Source_Ptr) value. Set by
   --    Load_Source_File and cannot be subsequently changed.

   --  Time_Stamp : Time_Stamp_Type;
   --    Time stamp of the source file. Set by Load_Source_File, and cannot
   --    be subsequently changed.

   --  Lines_Table : Lines_Table_Ptr;
   --    Pointer to lines table for this source. Updated as additional
   --    lines are accessed using the Skip_Line_Terminators procedure.

   --  Last_Line : Line_Number_Type;
   --    Subscript of last entry in Lines_Table (may be different from 'Last
   --    value because of the use of expandable tables). On completion of
   --    compilation of a unit (status = loaded), this is the number of
   --    source lines in the file. Updated by the Skip_Line_Terminators
   --    procedure as new lines are read.

   --  Keyword_Casing : Casing_Type;
   --    Casing style used in file for keyword casing. This is initialized
   --    to Unknown, and then set from the first occurrence of a keyword. This
   --    value is used only for formatting of error messages.

   --  Identifier_Casing : Casing_Type;
   --    Casing style used in file for identifier casing. This is initialized
   --    to Unknown, and then set from an identifier in the program as soon as
   --    one is found whose casing is sufficiently clear to make a decision.
   --    This value is used for formatting of error messages, and also is used
   --    in the detection of keywords misused as identifiers.

   --  The source file table is accessed using the following subprogram
   --  interface:

   function File_Name         (S : Source_File_Index) return File_Name_Type;
   function Full_File_Name    (S : Source_File_Index) return File_Name_Type;
   function Source_Text       (S : Source_File_Index) return Source_Buffer_Ptr;
   function Time_Stamp        (S : Source_File_Index) return Time_Stamp_Type;
   function Lines_Table       (S : Source_File_Index) return Lines_Table_Ptr;
   function Last_Line         (S : Source_File_Index) return Line_Number_Type;
   function Keyword_Casing    (S : Source_File_Index) return Casing_Type;
   function Identifier_Casing (S : Source_File_Index) return Casing_Type;

   procedure Set_Keyword_Casing    (S : Source_File_Index; C : Casing_Type);
   procedure Set_Identifier_Casing (S : Source_File_Index; C : Casing_Type);

   function Last_Source_File return Source_File_Index;
   --  Index of last source file table entry

   function Num_Source_Files return Nat;
   --  Number of source file table entries

   pragma Inline (File_Name);
   pragma Inline (Full_File_Name);
   pragma Inline (Source_Text);
   pragma Inline (Time_Stamp);
   pragma Inline (Lines_Table);
   pragma Inline (Last_Line);
   pragma Inline (Keyword_Casing);
   pragma Inline (Identifier_Casing);
   pragma Inline (Set_Keyword_Casing);
   pragma Inline (Set_Identifier_Casing);
   pragma Inline (Last_Source_File);
   pragma Inline (Num_Source_Files);

   -----------------
   -- Global Data --
   -----------------

   Current_Source_File : Source_File_Index;
   --  Source_File table index of source file currently being scanned

   Current_Source_Unit : Unit_Number_Type;
   --  Unit number of source file currently being scanned

   Source : Source_Buffer_Ptr;
   --  Current source (copy of Source_File.Table (Current_Source_Unit).Source)

   Internal_Source : aliased Source_Buffer (1 .. 81);
   --  This buffer is used internally in the compiler when the lexical analyzer
   --  is used to scan a string from within the compiler. The procedure is to
   --  establish Internal_Source_Ptr as the value of Source, set the string to
   --  be scanned, appropriately terminated, in this buffer, and set Scan_Ptr
   --  to point to the start of the buffer. It is a fatal error if the scanner
   --  signals an error while scanning a token in this internal buffer.

   Internal_Source_Ptr : constant Source_Buffer_Ptr := Internal_Source'access;
   --  Pointer to internal source buffer

   -----------------
   -- Subprograms --
   -----------------

   procedure Backup_Line (P : in out Source_Ptr);
   --  Back up the argument pointer to the start of the previous line. On
   --  entry, P points to the start of a physical line in the source buffer.
   --  On return, P is updated to point to the start of the previous line.
   --  The caller has checked that a Line_Terminator character precedes P so
   --  that there definitely is a previous line in the source buffer.

   function Get_Col_Number (P : Source_Ptr) return Column_Number_Type;
   --  The ones-origin column number of the specified Source_Ptr value is
   --  determined and returned. Tab characters if present are assumed to
   --  represent the standard 1,9,17.. spacing pattern.

   function Get_Line_Number (P : Source_Ptr) return Line_Number_Type;
   --  The line number of the specified source position is obtained by
   --  doing a binary search on the source positions in the lines table
   --  for the unit containing the given source position.

   function Get_Source_File_Index (S : Source_Ptr) return Source_File_Index;
   --  Return file table index of file identified by given source pointer
   --  value. This call must always succeed, since any valid source pointer
   --  value belongs to some previously loaded source file.

   function Load_Source_File (N : File_Name_Type) return Source_File_Index;
   --  Given a source file name, returns the index of the corresponding entry
   --  in the the source file table. If the file is not currently loaded, then
   --  this is the call that causes the source file to be read and an entry
   --  made in the table. A new entry in the table has the file name and time
   --  stamp entries set and the Casing entries set to Unknown. Version is set
   --  to all blanks, and the lines table is initialized but only the first
   --  entry is set (and Last_Line is set to 1). If the given source file
   --  cannot be opened, then the value returned is No_Source_File.

   function Line_Start (P : Source_Ptr) return Source_Ptr;
   --  Finds the source position of the start of the line containing the
   --  given source location.

   procedure Skip_Line_Terminators (
     P        : in out Source_Ptr;
     Physical : out Boolean);
   --  On entry, Source (P) points to the line terminator character that
   --  terminates a line. The result set in P is the location of the first
   --  character of the following line (after skipping the sequence of line
   --  terminator characters terminating the current line). In addition, if
   --  the terminator sequence ends a physical line (the definition of what
   --  constitutes a physical line is embodied in the implementation of this
   --  function), and it is the first time this sequence is encountered, then
   --  an entry is made in the lines table to record the location for further
   --  use by functions such as Get_Line_Number. Physical is set to True if
   --  the line terminator was the end of a physical line.

   procedure Write_Location (P : Source_Ptr);
   --  Writes out a string of the form "fff", line nn(cc), where fff, nn, cc
   --  are the file name, line number and column corresponding to the given
   --  source location. No_Location and Standard_Location appear as the
   --  strings <no location> and <standard location>.

   procedure Write_Time_Stamp (S : Source_File_Index);
   --  Writes time stamp of specified file in YY-MM-DD HH:MM.SS format

end Sinput;
