/****************************************************************************/
/*                                                                          */
/*                         GNAT COMPILER COMPONENTS                         */
/*                                                                          */
/*                               G N A T B L                                */
/*                                                                          */
/*                          C Implementation File                           */
/*                                                                          */
/*                             $Revision: 1.16 $                            */
/*                                                                          */
/*           Copyright (c) 1992,1993,1994 NYU, All Rights Reserved          */
/*                                                                          */
/* GNAT is free software;  you can  redistribute it  and/or modify it under */
/* terms of the  GNU General Public License as published  by the Free Soft- */
/* ware  Foundation;  either version 2,  or (at your option) any later ver- */
/* sion.  GNAT is distributed in the hope that it will be useful, but WITH- */
/* OUT ANY WARRANTY;  without even the  implied warranty of MERCHANTABILITY */
/* or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License */
/* for  more details.  You should have  received  a copy of the GNU General */
/* Public License  distributed with GNAT;  see file COPYING.  If not, write */
/* to the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA. */
/*                                                                          */
/****************************************************************************/

#include <stdio.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <stdlib.h>
#include <string.h>
#ifdef OS2
#include <process.h>
#endif
#include "config.h"

/* These can be set by command line arguments */
char *binder_path = 0;
char *linker_path = 0;
int  verbose      = 0;

int  link_arg_max = -1;
char **link_args = (char **) 0;
int  link_arg_index = -1;
int  numerics_present = 0;  /* indicates presence of ada numerics packages */

/* By default, colon separates directories in a path.  */
#ifndef PATH_SEPARATOR
#define PATH_SEPARATOR ':'
#endif

#ifndef DIR_SEPARATOR
#define DIR_SEPARATOR '/'
#endif

static void proc_bind_file (char *aliname);
static int is_regular_file (char *name);
static void port_spawn (char *args[]);

static void
addarg (char *str)
{
  int i;

  if (++link_arg_index >= link_arg_max) {
    char **new_link_args
      = (char **) calloc (link_arg_max + 1000, sizeof (char *));
    for (i = 0; i <= link_arg_max; i++)
      new_link_args [i] = link_args [i];
    if (link_args) free (link_args);
    link_arg_max += 1000;
    link_args = new_link_args;
  }

  link_args [link_arg_index] = str;
}

static char *
locate_exec (char *exec_name, char *path_val)
{
  char buf [1000];
  int exec_len = strlen (exec_name);
  char *end_path = path_val + strlen (path_val);
  char *ptr;

  /* Handle absolute pathnames */
  if (exec_name [0] == DIR_SEPARATOR
#ifdef OS2
      || isalpha (exec_name [0]) && exec_name [1] == ':'
#endif
     ) {
    strncpy (buf, exec_name, sizeof buf);
    buf[sizeof buf - 1] = '\0';
#ifdef OS2
    ptr = strstr (buf, ".exe");
    if (ptr == NULL || strlen (ptr) != 4)
       strcat (buf, ".exe");
#endif
    if (is_regular_file (buf))
      return strdup (buf);
    else
      return (char *)0;
  }

  if (! path_val) return (char *) 0;

  for (;;) {
    for (; *path_val == PATH_SEPARATOR ; path_val++)
      ;
    if (! *path_val) return (char *) 0;

    for (ptr = buf; *path_val && *path_val != PATH_SEPARATOR; )
      *ptr++ = *path_val++;

    if (*--ptr != DIR_SEPARATOR)
      *++ptr = DIR_SEPARATOR;

    strcpy (++ptr, exec_name);

#ifdef OS2
    ptr = strstr (buf, ".exe");
    if (ptr == NULL || strlen (ptr) != 4)
       strcat (buf, ".exe");
#endif
    if (is_regular_file (buf))
      return strdup (buf);
  }

  return (char *) 0;
}

static int
is_regular_file (char *name)
{
  int ret;
  struct stat statbuf;

  ret = stat(name, &statbuf);
  return (!ret && S_ISREG(statbuf.st_mode));
}

static void
process_args (int *p_argc, char *argv [])
{
  int i, j;

  for (i = 1; i < *p_argc; i++) {

    /* -v turns on verbose option here and is passed on to gcc */
    if (! strcmp (argv [i], "-v"))
      verbose = 1;

    else if (! strcmp (argv [i], "-gnatbind")) {

      /* Explicit naming of binder.  Grab the value then remove the
	 two arguments from the argument list. */

      if ( i + 1 >= *p_argc ) {
	fprintf (stderr, "Missing argument for -gnatbind\n");
	exit (1);
      }

      binder_path = locate_exec (argv [i + 1], ".");
      if (!binder_path) {
	fprintf (stderr, "Could not locate binder: %s\n", argv [i + 1]);
	exit (1);
      }
      for (j = i + 2; j < *p_argc; j++)
	argv [j - 2] = argv [j];
      (*p_argc) -= 2;
      i--;
    }

    else if (! strcmp (argv [i], "-gnatlink")) {

      /* Explicit naming of binder.  Grab the value then remove the
	 two arguments from the argument list. */

      if ( i + 1 >= *p_argc ) {
	fprintf (stderr, "Missing argument for -gnatlink\n");
	exit (1);
      }
      linker_path = locate_exec (argv [i + 1], ".");
      if (!linker_path) {
	fprintf (stderr, "Could not locate linker: %s\n", argv [i + 1]);
	exit (1);
      }
      for (j = i + 2; j < *p_argc; j++)
	argv [j - 2] = argv [j];
      (*p_argc) -= 2;
      i--;
    }
  }
}

main (int argc, char *argv [])
{
  int i;
  int done_an_ali = 0;
  int file_name_index;
  char *pathval = getenv ("PATH");
  char *spawn_args [5];
  char *bind_src_file_name;
  char *bind_obj_file_name;
  int   bind_file_name_len;

#ifdef OS2
  char *tmppathval = malloc (strlen (pathval) + 3);
  strcpy (tmppathval, ".;");
  pathval = strcat (tmppathval, pathval);
#endif

  process_args (&argc , argv);

  if (!binder_path)
    binder_path = locate_exec ("gnatbind", pathval);
  if (!binder_path) {
    fprintf (stderr, "Couldn't locate gnatbind\n");
    exit (1);
  }
  if (!linker_path)
    linker_path = locate_exec ("gcc", pathval);
  if (!linker_path) {
    fprintf (stderr, "Couldn't locate gcc\n");
    exit (1);
  }
  addarg (linker_path);

  for (i = 1; i < argc; i++) {
    int arg_len = strlen (argv [i]);
    if (arg_len > 4 && ! strcmp (&argv [i][arg_len - 4], ".ali")) {

      if (done_an_ali) {
	fprintf (stderr, "Sorry - cannot handle more than one ALI file\n");
	exit (1);
      }
      done_an_ali = 1;

      if (is_regular_file (argv [i])) {
	int pid;

	/* Run gnatbind */
	spawn_args [0] = binder_path;
	spawn_args [1] = argv[i];
	spawn_args [2] = 0;
	port_spawn (spawn_args);

	for (file_name_index = arg_len - 4; ; file_name_index--)
	  if (file_name_index == 0
	      || argv [i][file_name_index - 1] == DIR_SEPARATOR)
	    break;
	bind_file_name_len = strlen (&(argv [i][file_name_index]));
	bind_src_file_name = malloc (bind_file_name_len + 1);
	strcpy  (bind_src_file_name, "b_");
	strncat (bind_src_file_name,
		 &(argv [i][file_name_index]), bind_file_name_len - 3);
	strcat  (bind_src_file_name, "c");

	/* Compile binder output */
	/* ??? should linker and compiler be separate */
	spawn_args [0] = linker_path;
	spawn_args [1] = "-c";
	spawn_args [2] = "-g";
	spawn_args [3] = bind_src_file_name;
	spawn_args [4] = 0;
	port_spawn (spawn_args);

#ifdef OS2
	bind_obj_file_name = malloc (strlen (bind_src_file_name) + 3);
	strcpy (bind_obj_file_name, bind_src_file_name);
	bind_obj_file_name [strlen (bind_obj_file_name) - 1] = '\0';
	strcat  (bind_obj_file_name, "obj");
#else
	bind_obj_file_name = strdup (bind_src_file_name);
	bind_obj_file_name [strlen (bind_obj_file_name) - 1] = 'o';
#endif
	addarg (bind_obj_file_name);

	proc_bind_file (bind_src_file_name);
      }
      else {
	addarg (argv [i]);
      }
    }
    else {
      addarg (argv [i]);
    }
  }

#ifdef OS2
  addarg ("-Wl,gnat.lib");
  addarg (NULL);
#else
  addarg ("-L" ADA_RTL_OBJ_DIR);
  addarg ("-lgnat");
  addarg ("-lpthreads");
  if (numerics_present) addarg ("-lm");
  addarg ((char *) 0);
#endif

  if (verbose) {
    int i;
    for (i = 0; i < link_arg_index; i++)
      printf ("%s ", link_args [i]);
    putchar ('\n');
  }

#ifdef OS2
  if (spawnvp (P_WAIT, linker_path, link_args) != 0) {
    fprintf (stderr, "Error executing %s\n", link_args[0]);
    exit (1);
  }
  exit (0);
#else
  execv (linker_path, link_args);
#endif

}

static void
proc_bind_file (char *bind_file_name)
{
  char *ptr;
  char buf [1000];
  FILE *bind_file;
  int aliname_len = strlen (bind_file_name);
  char *start_string = "/* BEGIN Object";
  int  start_string_len = strlen (start_string);
  char *end_string   = "   END Object";
  int  end_string_len = strlen (end_string);
/* If not set to strrchr by config.h */
#ifndef rindex
  char *rindex (char *, int);
#endif


  if ((bind_file = fopen (bind_file_name, "r")) == NULL) {
    fprintf (stderr, "Failed to open binder output\n");
    exit (1);
  }

  for (;;) {
    if (fgets (buf, 1000, bind_file) == NULL) {
      fprintf (stderr, "Error reading binder output\n");
      exit (1);
    }
    if (!strncmp (buf, start_string, start_string_len))
      break;
  }

  for (;;) {
    if (fgets (buf, 1000, bind_file) == NULL) {
      fprintf (stderr, "Error reading binder output\n");
      exit (1);
    }
    if (!strncmp (buf, end_string, end_string_len))
      break;

    for (ptr = buf; *ptr && *ptr != '\n'; ptr++)
      ;
    *ptr = '\0';

    addarg (strdup (buf));

    /* If the object file "a-numaux" is present set a flag and later
     * when the final arguments of the link are appended, the flag will be
     * queried and "-lm" will be added to the link since this is needed by
     * the numerics packages.
     */
    ptr = rindex (buf, DIR_SEPARATOR);
    if (ptr != NULL) {
      ptr++;  /* skip over the separator */
    }
    else {
      ptr = buf;
    }
    if (! strcmp (ptr, "a-numaux.o"))
      numerics_present++;

  }
  fclose (bind_file);
}

static void
port_spawn (char *args[]) {
  int status;
  int finished;
  int pid;

  if (verbose) {
    int i;
    for (i = 0; args [i] != (char *) 0; i++)
      printf ("%s ", args [i]);
    putchar ('\n');
  }

#ifdef OS2
  if (spawnvp (P_WAIT, args [0], args) != 0) {
    fprintf (stderr, "Error executing %s\n", args[0]);
    exit (1);
  }
#else
  pid = fork ();
  if (pid == -1) {
    fprintf (stderr, "Error starting %s\n", args [0]);
    exit (1);
  }
  if (pid == 0) {
    /* The child */
    execv (args [0], args);
    fprintf (stderr, "Failed to exec %s\n", args [0]);
    exit (1);
  }

  /* The parent */
  finished = wait (&status);
  if (finished != pid || status & 0xffff) {
    /* ??? figure out later what this test should be */
    fprintf (stderr, "%s failed\n", args [0]);
    exit (1);
  }
#endif
}
