/*
    chips.c - Part of libsensors, a Linux library for reading sensor data.
    Copyright (c) 1998, 1999  Frodo Looijaard <frodol@dds.nl> 

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include "chips.h"
#include "data.h"
#include "sensors.h"
#include "kernel/include/sensors.h"

/* For each chip, a list of features;
   For each feature, the number, its name, the logical mapping (for example,
   fan1_div belongs to fan1), the compute mapping, an access field, the 
   SYSCTL id of the /proc field, the value location of the /proc field,
   and the magnitude.
   The access field  indicates whether resources are readable and writable.
   Sometimes, a resource is neither; in this case, it is just a catch-all
   category, for mappings. 
   The value location is the byte-offset where this value can be found in
   the sysctl buffer.
   The magnitude is the 10-power through which the read value must be
   divided. */


#define VALUE(x) (x-1) * sizeof(long)
#define NOMAP SENSORS_NO_MAPPING
#define R SENSORS_MODE_R
#define RW SENSORS_MODE_RW
#define NOSYSCTL 0

static sensors_chip_feature lm78_features[] =
  { 
    { { SENSORS_LM78_IN0, "in0", NOMAP, NOMAP, R }, 
                          LM78_SYSCTL_IN0, VALUE(3), 2 },
    { { SENSORS_LM78_IN1, "in1", NOMAP, NOMAP, R }, 
                          LM78_SYSCTL_IN1, VALUE(3), 2 },
    { { SENSORS_LM78_IN2, "in2", NOMAP, NOMAP, R }, 
                          LM78_SYSCTL_IN2, VALUE(3), 2 },
    { { SENSORS_LM78_IN3, "in3", NOMAP, NOMAP, R }, 
                          LM78_SYSCTL_IN3, VALUE(3), 2 },
    { { SENSORS_LM78_IN4, "in4", NOMAP, NOMAP, R }, 
                          LM78_SYSCTL_IN4, VALUE(3), 2 },
    { { SENSORS_LM78_IN5, "in5", NOMAP, NOMAP, R }, 
                          LM78_SYSCTL_IN5, VALUE(3), 2 },
    { { SENSORS_LM78_IN6, "in6", NOMAP, NOMAP, R }, 
                          LM78_SYSCTL_IN6, VALUE(3), 2 },
    { { SENSORS_LM78_IN0_MIN, "in0_min", SENSORS_LM78_IN0, SENSORS_LM78_IN0, RW }, 
                              LM78_SYSCTL_IN0, VALUE(1), 2 },
    { { SENSORS_LM78_IN1_MIN, "in1_min", SENSORS_LM78_IN1, SENSORS_LM78_IN1, RW }, 
                              LM78_SYSCTL_IN1, VALUE(1), 2 },
    { { SENSORS_LM78_IN2_MIN, "in2_min", SENSORS_LM78_IN2, SENSORS_LM78_IN2, RW }, 
                              LM78_SYSCTL_IN2, VALUE(1), 2 },
    { { SENSORS_LM78_IN3_MIN, "in3_min", SENSORS_LM78_IN3, SENSORS_LM78_IN3, RW }, 
                              LM78_SYSCTL_IN3, VALUE(1), 2 },
    { { SENSORS_LM78_IN4_MIN, "in4_min", SENSORS_LM78_IN4, SENSORS_LM78_IN4, RW }, 
                              LM78_SYSCTL_IN4, VALUE(1), 2 },
/* swap 5 and 6 min/max because compute lines will be inverting */
    { { SENSORS_LM78_IN5_MIN, "in5_min", SENSORS_LM78_IN5, SENSORS_LM78_IN5, RW }, 
                              LM78_SYSCTL_IN5, VALUE(2), 2, "in5_max", 3 },
    { { SENSORS_LM78_IN6_MIN, "in6_min", SENSORS_LM78_IN6, SENSORS_LM78_IN6, RW }, 
                              LM78_SYSCTL_IN6, VALUE(2), 2, "in6_max", 3 },
    { { SENSORS_LM78_IN0_MAX, "in0_max", SENSORS_LM78_IN0, SENSORS_LM78_IN0, RW }, 
                              LM78_SYSCTL_IN0, VALUE(2), 2 },
    { { SENSORS_LM78_IN1_MAX, "in1_max", SENSORS_LM78_IN1, SENSORS_LM78_IN1, RW }, 
                              LM78_SYSCTL_IN1, VALUE(2), 2 },
    { { SENSORS_LM78_IN2_MAX, "in2_max", SENSORS_LM78_IN2, SENSORS_LM78_IN2, RW }, 
                              LM78_SYSCTL_IN2, VALUE(2), 2 },
    { { SENSORS_LM78_IN3_MAX, "in3_max", SENSORS_LM78_IN3, SENSORS_LM78_IN3, RW }, 
                              LM78_SYSCTL_IN3, VALUE(2), 2 },
    { { SENSORS_LM78_IN4_MAX, "in4_max", SENSORS_LM78_IN4, SENSORS_LM78_IN4, RW }, 
                              LM78_SYSCTL_IN4, VALUE(2), 2 },
    { { SENSORS_LM78_IN5_MAX, "in5_max", SENSORS_LM78_IN5, SENSORS_LM78_IN5, RW }, 
                              LM78_SYSCTL_IN5, VALUE(1), 2, "in5_min", 3 },
    { { SENSORS_LM78_IN6_MAX, "in6_max", SENSORS_LM78_IN6, SENSORS_LM78_IN6, RW }, 
                              LM78_SYSCTL_IN6, VALUE(1), 2, "in6_min", 3 },
    { { SENSORS_LM78_FAN1, "fan1", NOMAP, NOMAP, R }, 
                           LM78_SYSCTL_FAN1, VALUE(2), 0 },
    { { SENSORS_LM78_FAN2, "fan2", NOMAP, NOMAP, R }, 
                           LM78_SYSCTL_FAN2, VALUE(2), 0 },
    { { SENSORS_LM78_FAN3, "fan3", NOMAP, NOMAP, R }, 
                           LM78_SYSCTL_FAN3, VALUE(2), 0 },
    { { SENSORS_LM78_FAN1_MIN, "fan1_min", SENSORS_LM78_FAN1, SENSORS_LM78_FAN1, 
                               RW }, 
                               LM78_SYSCTL_FAN1, VALUE(1), 0 },
    { { SENSORS_LM78_FAN2_MIN, "fan2_min", SENSORS_LM78_FAN2, SENSORS_LM78_FAN2, 
                               RW }, 
                               LM78_SYSCTL_FAN2, VALUE(1), 0 },
    { { SENSORS_LM78_FAN3_MIN, "fan3_min", SENSORS_LM78_FAN3, SENSORS_LM78_FAN3, 
                               RW }, 
                               LM78_SYSCTL_FAN3, VALUE(1), 0 },
    { { SENSORS_LM78_TEMP, "temp", NOMAP, NOMAP, R }, 
                           LM78_SYSCTL_TEMP, VALUE(3), 1 },
    { { SENSORS_LM78_TEMP_HYST, "temp_hyst", SENSORS_LM78_TEMP, 
                                SENSORS_LM78_TEMP, RW }, 
                                LM78_SYSCTL_TEMP, VALUE(2), 1 },
    { { SENSORS_LM78_TEMP_OVER, "temp_over", SENSORS_LM78_TEMP, 
                                SENSORS_LM78_TEMP, RW }, 
                                LM78_SYSCTL_TEMP, VALUE(1), 1 },
    { { SENSORS_LM78_VID, "vid", NOMAP, NOMAP, R }, 
                          LM78_SYSCTL_VID, VALUE(1), 2 },
    { { SENSORS_LM78_FAN1_DIV, "fan1_div", SENSORS_LM78_FAN1, NOMAP, RW }, 
                               LM78_SYSCTL_FAN_DIV, VALUE(1), 0 },
    { { SENSORS_LM78_FAN2_DIV, "fan2_div", SENSORS_LM78_FAN2, NOMAP, RW }, 
                               LM78_SYSCTL_FAN_DIV, VALUE(2), 0 },
    { { SENSORS_LM78_FAN3_DIV, "fan3_div", SENSORS_LM78_FAN3, NOMAP, R }, 
                               LM78_SYSCTL_FAN_DIV, VALUE(3), 0 },
    { { SENSORS_LM78_ALARMS, "alarms", NOMAP, NOMAP, R }, 
                             LM78_SYSCTL_ALARMS, VALUE(1), 0 },
    { { 0 }, 0 }
  };

static sensors_chip_feature lm78j_features[] =
  { 
    { { SENSORS_LM78J_IN0, "in0", NOMAP, NOMAP, R }, 
                           LM78_SYSCTL_IN0, VALUE(3), 2 },
    { { SENSORS_LM78J_IN1, "in1", NOMAP, NOMAP, R }, 
                           LM78_SYSCTL_IN1, VALUE(3), 2 },
    { { SENSORS_LM78J_IN2, "in2", NOMAP, NOMAP, R }, 
                           LM78_SYSCTL_IN2, VALUE(3), 2 },
    { { SENSORS_LM78J_IN3, "in3", NOMAP, NOMAP, R }, 
                           LM78_SYSCTL_IN3, VALUE(3), 2 },
    { { SENSORS_LM78J_IN4, "in4", NOMAP, NOMAP, R }, 
                           LM78_SYSCTL_IN4, VALUE(3), 2 },
    { { SENSORS_LM78J_IN5, "in5", NOMAP, NOMAP, R }, 
                           LM78_SYSCTL_IN5, VALUE(3), 2 },
    { { SENSORS_LM78J_IN6, "in6", NOMAP, NOMAP, R }, 
                           LM78_SYSCTL_IN6, VALUE(3), 2 },
    { { SENSORS_LM78J_IN0_MIN, "in0_min", SENSORS_LM78J_IN0, SENSORS_LM78J_IN0, 
                               RW }, 
                               LM78_SYSCTL_IN0, VALUE(1), 2 },
    { { SENSORS_LM78J_IN1_MIN, "in1_min", SENSORS_LM78J_IN1, SENSORS_LM78J_IN1, 
                               RW }, 
                               LM78_SYSCTL_IN1, VALUE(1), 2 },
    { { SENSORS_LM78J_IN2_MIN, "in2_min", SENSORS_LM78J_IN2, SENSORS_LM78J_IN2, 
                               RW }, 
                               LM78_SYSCTL_IN2, VALUE(1), 2 },
    { { SENSORS_LM78J_IN3_MIN, "in3_min", SENSORS_LM78J_IN3, SENSORS_LM78J_IN3, 
                               RW }, 
                               LM78_SYSCTL_IN3, VALUE(1), 2 },
    { { SENSORS_LM78J_IN4_MIN, "in4_min", SENSORS_LM78J_IN4, SENSORS_LM78J_IN4, 
                               RW }, 
                               LM78_SYSCTL_IN4, VALUE(1), 2 },
/* swap 5 and 6 min/max because compute lines will be inverting */
    { { SENSORS_LM78J_IN5_MIN, "in5_min", SENSORS_LM78J_IN5, SENSORS_LM78J_IN5, 
                               RW }, 
                               LM78_SYSCTL_IN5, VALUE(2), 2, "in5_max", 3 },
    { { SENSORS_LM78J_IN6_MIN, "in6_min", SENSORS_LM78J_IN6, SENSORS_LM78J_IN6, 
                               RW }, 
                               LM78_SYSCTL_IN6, VALUE(2), 2, "in6_max", 3 },
    { { SENSORS_LM78J_IN0_MAX, "in0_max", SENSORS_LM78J_IN0, SENSORS_LM78J_IN0, 
                               RW }, 
                               LM78_SYSCTL_IN0, VALUE(2), 2 },
    { { SENSORS_LM78J_IN1_MAX, "in1_max", SENSORS_LM78J_IN1, SENSORS_LM78J_IN1, 
                               RW }, 
                               LM78_SYSCTL_IN1, VALUE(2), 2 },
    { { SENSORS_LM78J_IN2_MAX, "in2_max", SENSORS_LM78J_IN2, SENSORS_LM78J_IN2, 
                               RW }, 
                               LM78_SYSCTL_IN2, VALUE(2), 2 },
    { { SENSORS_LM78J_IN3_MAX, "in3_max", SENSORS_LM78J_IN3, SENSORS_LM78J_IN3, 
                               RW }, 
                               LM78_SYSCTL_IN3, VALUE(2), 2 },
    { { SENSORS_LM78J_IN4_MAX, "in4_max", SENSORS_LM78J_IN4, SENSORS_LM78J_IN4, 
                               RW }, 
                               LM78_SYSCTL_IN4, VALUE(2), 2 },
    { { SENSORS_LM78J_IN5_MAX, "in5_max", SENSORS_LM78J_IN5, SENSORS_LM78J_IN5, 
                               RW }, 
                               LM78_SYSCTL_IN5, VALUE(1), 2, "in5_min", 3 },
    { { SENSORS_LM78J_IN6_MAX, "in6_max", SENSORS_LM78J_IN6, SENSORS_LM78J_IN6, 
                               RW }, 
                               LM78_SYSCTL_IN6, VALUE(1), 2, "in6_min", 3 },
    { { SENSORS_LM78J_FAN1, "fan1", NOMAP, NOMAP, R }, 
                            LM78_SYSCTL_FAN1, VALUE(2), 0 },
    { { SENSORS_LM78J_FAN2, "fan2", NOMAP, NOMAP, R }, 
                            LM78_SYSCTL_FAN2, VALUE(2), 0 },
    { { SENSORS_LM78J_FAN3, "fan3", NOMAP, NOMAP, R }, 
                            LM78_SYSCTL_FAN3, VALUE(2), 0 },
    { { SENSORS_LM78J_FAN1_MIN, "fan1_min", SENSORS_LM78J_FAN1, 
                                SENSORS_LM78J_FAN1, RW }, 
                                LM78_SYSCTL_FAN1, VALUE(1), 0 },
    { { SENSORS_LM78J_FAN2_MIN, "fan2_min", SENSORS_LM78J_FAN2, 
                                SENSORS_LM78J_FAN2, RW }, 
                                LM78_SYSCTL_FAN2, VALUE(1), 0 },
    { { SENSORS_LM78J_FAN3_MIN, "fan3_min", SENSORS_LM78J_FAN3, 
                                SENSORS_LM78J_FAN3, RW }, 
                                LM78_SYSCTL_FAN3, VALUE(1), 0 },
    { { SENSORS_LM78J_TEMP, "temp", NOMAP, NOMAP, R }, 
                            LM78_SYSCTL_TEMP, VALUE(3), 1 },
    { { SENSORS_LM78J_TEMP_HYST, "temp_hyst", SENSORS_LM78J_TEMP, 
                                 SENSORS_LM78J_TEMP, RW }, 
                                 LM78_SYSCTL_TEMP, VALUE(2), 1 },
    { { SENSORS_LM78J_TEMP_OVER, "temp_over", SENSORS_LM78J_TEMP, 
                                 SENSORS_LM78J_TEMP, RW }, 
                                 LM78_SYSCTL_TEMP, VALUE(1), 1 },
    { { SENSORS_LM78J_VID, "vid", NOMAP, NOMAP, R }, 
                           LM78_SYSCTL_VID, VALUE(1), 2 },
    { { SENSORS_LM78J_FAN1_DIV, "fan1_div", SENSORS_LM78J_FAN1, NOMAP, RW }, 
                                LM78_SYSCTL_FAN_DIV, VALUE(1), 0 },
    { { SENSORS_LM78J_FAN2_DIV, "fan2_div", SENSORS_LM78J_FAN2, NOMAP, RW }, 
                                LM78_SYSCTL_FAN_DIV, VALUE(2), 0 },
    { { SENSORS_LM78J_FAN3_DIV, "fan3_div", SENSORS_LM78J_FAN3, NOMAP, R }, 
                                LM78_SYSCTL_FAN_DIV, VALUE(3), 0 },
    { { SENSORS_LM78J_ALARMS, "alarms", NOMAP, NOMAP, R }, 
                              LM78_SYSCTL_ALARMS, VALUE(1), 0 },
    { { 0 }, 0 }
  };

static sensors_chip_feature lm79_features[] =
  { 
    { { SENSORS_LM79_IN0, "in0", NOMAP, NOMAP, R }, 
                          LM78_SYSCTL_IN0, VALUE(3), 2 },
    { { SENSORS_LM79_IN1, "in1", NOMAP, NOMAP, R }, 
                          LM78_SYSCTL_IN1, VALUE(3), 2 },
    { { SENSORS_LM79_IN2, "in2", NOMAP, NOMAP, R }, 
                          LM78_SYSCTL_IN2, VALUE(3), 2 },
    { { SENSORS_LM79_IN3, "in3", NOMAP, NOMAP, R }, 
                          LM78_SYSCTL_IN3, VALUE(3), 2 },
    { { SENSORS_LM79_IN4, "in4", NOMAP, NOMAP, R }, 
                          LM78_SYSCTL_IN4, VALUE(3), 2 },
    { { SENSORS_LM79_IN5, "in5", NOMAP, NOMAP, R }, 
                          LM78_SYSCTL_IN5, VALUE(3), 2 },
    { { SENSORS_LM79_IN6, "in6", NOMAP, NOMAP, R }, 
                          LM78_SYSCTL_IN6, VALUE(3), 2 },
    { { SENSORS_LM79_IN0_MIN, "in0_min", SENSORS_LM79_IN0, SENSORS_LM79_IN0, RW }, 
                              LM78_SYSCTL_IN0, VALUE(1), 2 },
    { { SENSORS_LM79_IN1_MIN, "in1_min", SENSORS_LM79_IN1, SENSORS_LM79_IN1, RW }, 
                              LM78_SYSCTL_IN1, VALUE(1), 2 },
    { { SENSORS_LM79_IN2_MIN, "in2_min", SENSORS_LM79_IN2, SENSORS_LM79_IN2, RW }, 
                              LM78_SYSCTL_IN2, VALUE(1), 2 },
    { { SENSORS_LM79_IN3_MIN, "in3_min", SENSORS_LM79_IN3, SENSORS_LM79_IN3, RW }, 
                              LM78_SYSCTL_IN3, VALUE(1), 2 },
    { { SENSORS_LM79_IN4_MIN, "in4_min", SENSORS_LM79_IN4, SENSORS_LM79_IN4, RW }, 
                              LM78_SYSCTL_IN4, VALUE(1), 2 },
/* swap 5 and 6 min/max because compute lines will be inverting */
    { { SENSORS_LM79_IN5_MIN, "in5_min", SENSORS_LM79_IN5, SENSORS_LM79_IN5, RW }, 
                              LM78_SYSCTL_IN5, VALUE(2), 2, "in5_max", 3 },
    { { SENSORS_LM79_IN6_MIN, "in6_min", SENSORS_LM79_IN6, SENSORS_LM79_IN6, RW }, 
                              LM78_SYSCTL_IN6, VALUE(2), 2, "in6_max", 3 },
    { { SENSORS_LM79_IN0_MAX, "in0_max", SENSORS_LM79_IN0, SENSORS_LM79_IN0, RW }, 
                              LM78_SYSCTL_IN0, VALUE(2), 2 },
    { { SENSORS_LM79_IN1_MAX, "in1_max", SENSORS_LM79_IN1, SENSORS_LM79_IN1, RW }, 
                              LM78_SYSCTL_IN1, VALUE(2), 2 },
    { { SENSORS_LM79_IN2_MAX, "in2_max", SENSORS_LM79_IN2, SENSORS_LM79_IN2, RW }, 
                              LM78_SYSCTL_IN2, VALUE(2), 2 },
    { { SENSORS_LM79_IN3_MAX, "in3_max", SENSORS_LM79_IN3, SENSORS_LM79_IN3, RW }, 
                              LM78_SYSCTL_IN3, VALUE(2), 2 },
    { { SENSORS_LM79_IN4_MAX, "in4_max", SENSORS_LM79_IN4, SENSORS_LM79_IN4, RW }, 
                              LM78_SYSCTL_IN4, VALUE(2), 2 },
    { { SENSORS_LM79_IN5_MAX, "in5_max", SENSORS_LM79_IN5, SENSORS_LM79_IN5, RW }, 
                              LM78_SYSCTL_IN5, VALUE(1), 2, "in5_min", 3 },
    { { SENSORS_LM79_IN6_MAX, "in6_max", SENSORS_LM79_IN6, SENSORS_LM79_IN6, RW }, 
                              LM78_SYSCTL_IN6, VALUE(1), 2, "in6_min", 3 },
    { { SENSORS_LM79_FAN1, "fan1", NOMAP, NOMAP, R }, 
                           LM78_SYSCTL_FAN1, VALUE(2), 0 },
    { { SENSORS_LM79_FAN2, "fan2", NOMAP, NOMAP, R }, 
                           LM78_SYSCTL_FAN2, VALUE(2), 0 },
    { { SENSORS_LM79_FAN3, "fan3", NOMAP, NOMAP, R }, 
                           LM78_SYSCTL_FAN3, VALUE(2), 0 },
    { { SENSORS_LM79_FAN1_MIN, "fan1_min", SENSORS_LM79_FAN1, SENSORS_LM79_FAN1, 
                               RW }, 
                               LM78_SYSCTL_FAN1, VALUE(1), 0 },
    { { SENSORS_LM79_FAN2_MIN, "fan2_min", SENSORS_LM79_FAN2, SENSORS_LM79_FAN2, 
                               RW }, 
                               LM78_SYSCTL_FAN2, VALUE(1), 0 },
    { { SENSORS_LM79_FAN3_MIN, "fan3_min", SENSORS_LM79_FAN3, SENSORS_LM79_FAN3, 
                               RW }, 
                               LM78_SYSCTL_FAN3, VALUE(1), 0 },
    { { SENSORS_LM79_TEMP, "temp", NOMAP, NOMAP, R }, 
                           LM78_SYSCTL_TEMP, VALUE(3), 1 },
    { { SENSORS_LM79_TEMP_HYST, "temp_hyst", SENSORS_LM79_TEMP, 
                                SENSORS_LM79_TEMP, RW }, 
                                LM78_SYSCTL_TEMP, VALUE(2), 1 },
    { { SENSORS_LM79_TEMP_OVER, "temp_over", SENSORS_LM79_TEMP, 
                                SENSORS_LM79_TEMP, RW }, 
                                LM78_SYSCTL_TEMP, VALUE(1), 1 },
    { { SENSORS_LM79_VID, "vid", NOMAP, NOMAP, R }, 
                          LM78_SYSCTL_VID, VALUE(1), 2 },
    { { SENSORS_LM79_FAN1_DIV, "fan1_div", SENSORS_LM79_FAN1, NOMAP, RW }, 
                               LM78_SYSCTL_FAN_DIV, VALUE(1), 0 },
    { { SENSORS_LM79_FAN2_DIV, "fan2_div", SENSORS_LM79_FAN2, NOMAP, RW }, 
                               LM78_SYSCTL_FAN_DIV, VALUE(2), 0 },
    { { SENSORS_LM79_FAN3_DIV, "fan3_div", SENSORS_LM79_FAN3, NOMAP, R }, 
                               LM78_SYSCTL_FAN_DIV, VALUE(3), 0 },
    { { SENSORS_LM79_ALARMS, "alarms", NOMAP, NOMAP, R }, 
                             LM78_SYSCTL_ALARMS, VALUE(1), 0 },
    { { 0 }, 0 }
  };


static sensors_chip_feature lm75_features[] =
  {
    { { SENSORS_LM75_TEMP, "temp", NOMAP, NOMAP, R }, 
                           LM75_SYSCTL_TEMP, VALUE(3), 1 },
    { { SENSORS_LM75_TEMP_HYST, "temp_hyst", SENSORS_LM75_TEMP, 
                                SENSORS_LM75_TEMP, RW }, 
                                LM75_SYSCTL_TEMP, VALUE(2), 1 },
    { { SENSORS_LM75_TEMP_OVER, "temp_over", SENSORS_LM75_TEMP, 
                                SENSORS_LM75_TEMP, RW }, 
                                LM75_SYSCTL_TEMP, VALUE(1), 1 },
    { { 0 }, 0 }
  };

static sensors_chip_feature adm1021_features[] =
  {
    { { SENSORS_ADM1021_TEMP, "temp", NOMAP, NOMAP, R }, 
                              ADM1021_SYSCTL_TEMP, VALUE(3), 0 },
    { { SENSORS_ADM1021_TEMP_HYST, "temp_low", SENSORS_ADM1021_TEMP, 
                                   SENSORS_ADM1021_TEMP, RW }, 
                                   ADM1021_SYSCTL_TEMP, VALUE(2), 0 },
    { { SENSORS_ADM1021_TEMP_OVER, "temp_over", SENSORS_ADM1021_TEMP, 
                                   SENSORS_ADM1021_TEMP, RW }, 
                                   ADM1021_SYSCTL_TEMP, VALUE(1), 0 },
    { { SENSORS_ADM1021_REMOTE_TEMP, "remote_temp", NOMAP, NOMAP, R }, 
                                     ADM1021_SYSCTL_REMOTE_TEMP, VALUE(3), 0 },
    { { SENSORS_ADM1021_REMOTE_TEMP_HYST, "remote_temp_low", 
                                          SENSORS_ADM1021_REMOTE_TEMP, 
                                          SENSORS_ADM1021_REMOTE_TEMP, RW }, 
                                          ADM1021_SYSCTL_REMOTE_TEMP, VALUE(2), 
                                          0 },
    { { SENSORS_ADM1021_REMOTE_TEMP_OVER, "remote_temp_over", 
                                          SENSORS_ADM1021_REMOTE_TEMP, 
                                          SENSORS_ADM1021_REMOTE_TEMP, RW }, 
                                          ADM1021_SYSCTL_REMOTE_TEMP, VALUE(1), 
                                          0 },
    { { SENSORS_ADM1021_ALARMS, "alarms", NOMAP, NOMAP, R }, 
                                ADM1021_SYSCTL_ALARMS, VALUE(1), 0 },
    { { SENSORS_ADM1021_DIE_CODE, "die_code", NOMAP, NOMAP, R }, 
                                  ADM1021_SYSCTL_DIE_CODE, VALUE(1), 0 },
    { { 0 }, 0 }
  };

static sensors_chip_feature adm1023_features[] =
  {
    { { SENSORS_ADM1021_TEMP, "temp", NOMAP, NOMAP, R }, 
                              ADM1021_SYSCTL_TEMP, VALUE(3), 0 },
    { { SENSORS_ADM1021_TEMP_HYST, "temp_low", SENSORS_ADM1021_TEMP, 
                                   SENSORS_ADM1021_TEMP, RW }, 
                                   ADM1021_SYSCTL_TEMP, VALUE(2), 0 },
    { { SENSORS_ADM1021_TEMP_OVER, "temp_over", SENSORS_ADM1021_TEMP, 
                                   SENSORS_ADM1021_TEMP, RW }, 
                                   ADM1021_SYSCTL_TEMP, VALUE(1), 0 },
    { { SENSORS_ADM1021_REMOTE_TEMP, "remote_temp", NOMAP, NOMAP, R }, 
                                     ADM1021_SYSCTL_REMOTE_TEMP, VALUE(4), 3 },
    { { SENSORS_ADM1021_REMOTE_TEMP_HYST, "remote_temp_low", 
                                          SENSORS_ADM1021_REMOTE_TEMP, 
                                          SENSORS_ADM1021_REMOTE_TEMP, RW }, 
                                          ADM1021_SYSCTL_REMOTE_TEMP, VALUE(2), 
                                          3 },
    { { SENSORS_ADM1021_REMOTE_TEMP_OVER, "remote_temp_over", 
                                          SENSORS_ADM1021_REMOTE_TEMP, 
                                          SENSORS_ADM1021_REMOTE_TEMP, RW }, 
                                          ADM1021_SYSCTL_REMOTE_TEMP, VALUE(1), 
                                          3 },
    { { SENSORS_ADM1021_ALARMS, "alarms", NOMAP, NOMAP, R }, 
                                ADM1021_SYSCTL_ALARMS, VALUE(1), 0 },
    { { 0 }, 0 }
  };


static sensors_chip_feature adm1029_features[] =
  {
       { { SENSORS_ADM1029_TEMP1, "temp1", NOMAP, NOMAP, R },
		NOSYSCTL, VALUE(3), 3 },
       { { SENSORS_ADM1029_TEMP1_MAX, "temp1_max",
                SENSORS_ADM1029_TEMP1, SENSORS_ADM1029_TEMP1, RW },
		NOSYSCTL, VALUE(1), 3 },
       { { SENSORS_ADM1029_TEMP1_MIN, "temp1_min",
                SENSORS_ADM1029_TEMP1, SENSORS_ADM1029_TEMP1, RW },
		NOSYSCTL, VALUE(2), 3 },

       { { SENSORS_ADM1029_TEMP2, "temp2", NOMAP, NOMAP, R },
		NOSYSCTL, VALUE(3), 3 },
       { { SENSORS_ADM1029_TEMP2_MAX, "temp2_max",
                SENSORS_ADM1029_TEMP2, SENSORS_ADM1029_TEMP2, RW },
		NOSYSCTL, VALUE(1), 3 },
       { { SENSORS_ADM1029_TEMP2_MIN, "temp2_min",
                SENSORS_ADM1029_TEMP2, SENSORS_ADM1029_TEMP2, RW },
		NOSYSCTL, VALUE(2), 3 },

       { { SENSORS_ADM1029_TEMP3, "temp3", NOMAP, NOMAP, R },
		NOSYSCTL, VALUE(3), 3 },
       { { SENSORS_ADM1029_TEMP3_MAX, "temp3_max",
                SENSORS_ADM1029_TEMP3, SENSORS_ADM1029_TEMP3, RW },
		NOSYSCTL, VALUE(1), 3 },
       { { SENSORS_ADM1029_TEMP3_MIN, "temp3_min",
                SENSORS_ADM1029_TEMP3, SENSORS_ADM1029_TEMP3, RW },
		NOSYSCTL, VALUE(2), 3 },

       { { SENSORS_ADM1029_FAN1, "fan1", NOMAP, NOMAP, R },
		NOSYSCTL, VALUE(2), 0 },
       { { SENSORS_ADM1029_FAN1_MIN, "fan1_min",
                SENSORS_ADM1029_FAN1, SENSORS_ADM1029_FAN1, RW },
		NOSYSCTL, VALUE(1), 0 },
       { { SENSORS_ADM1029_FAN1_DIV, "fan1_div",
                SENSORS_ADM1029_FAN1, NOMAP, RW },
		NOSYSCTL, VALUE(1), 0 },

       { { SENSORS_ADM1029_FAN2, "fan2", NOMAP, NOMAP, R },
		NOSYSCTL, VALUE(2), 0 },
       { { SENSORS_ADM1029_FAN2_MIN, "fan2_min",
                SENSORS_ADM1029_FAN2, SENSORS_ADM1029_FAN2, RW },
		NOSYSCTL, VALUE(1), 0 },
       { { SENSORS_ADM1029_FAN2_DIV, "fan2_div",
                SENSORS_ADM1029_FAN2, NOMAP, RW },
		NOSYSCTL, VALUE(2), 0 },
       { { 0 }, 0 }
  };


static sensors_chip_feature adm1030_features[] =
  {
    { { SENSORS_ADM1031_TEMP1, "temp1", NOMAP, NOMAP, R }, 
                               ADM1031_SYSCTL_TEMP1, VALUE(4), 3 },
    { { SENSORS_ADM1031_TEMP1_MAX, "temp1_max", SENSORS_ADM1031_TEMP1, 
                                   SENSORS_ADM1031_TEMP1, RW }, 
                                   ADM1031_SYSCTL_TEMP1, VALUE(1), 3 },
    { { SENSORS_ADM1031_TEMP1_MIN, "temp1_min", SENSORS_ADM1031_TEMP1, 
                                   SENSORS_ADM1031_TEMP1, RW }, 
                                   ADM1031_SYSCTL_TEMP1, VALUE(2), 3 },
    { { SENSORS_ADM1031_TEMP1_CRIT, "temp1_crit", SENSORS_ADM1031_TEMP1, 
                                    SENSORS_ADM1031_TEMP1, RW }, 
                                    ADM1031_SYSCTL_TEMP1, VALUE(3), 3 },

    { { SENSORS_ADM1031_TEMP2, "temp2", NOMAP, NOMAP, R }, 
                               ADM1031_SYSCTL_TEMP2, VALUE(4), 3 },
    { { SENSORS_ADM1031_TEMP2_MAX, "temp2_max", SENSORS_ADM1031_TEMP2, 
                                   SENSORS_ADM1031_TEMP2, RW }, 
                                   ADM1031_SYSCTL_TEMP2, VALUE(1), 3 },
    { { SENSORS_ADM1031_TEMP2_MIN, "temp2_min", SENSORS_ADM1031_TEMP2, 
                                   SENSORS_ADM1031_TEMP2, RW }, 
                                   ADM1031_SYSCTL_TEMP2, VALUE(2), 3 },
    { { SENSORS_ADM1031_TEMP2_CRIT, "temp2_crit", SENSORS_ADM1031_TEMP2, 
                                    SENSORS_ADM1031_TEMP2, RW }, 
                                    ADM1031_SYSCTL_TEMP2, VALUE(3), 3 },

    { { SENSORS_ADM1031_FAN1, "fan1", NOMAP, NOMAP, R }, 
                              ADM1031_SYSCTL_FAN1, VALUE(2), 0 },
    { { SENSORS_ADM1031_FAN1_MIN, "fan1_min", SENSORS_ADM1031_FAN1, 
                                  SENSORS_ADM1031_FAN1, RW }, 
                                  ADM1031_SYSCTL_FAN1, VALUE(1), 0 },
    { { SENSORS_ADM1031_FAN1_DIV, "fan1_div", SENSORS_ADM1031_FAN1, NOMAP, RW }, 
                                  ADM1031_SYSCTL_FAN_DIV, VALUE(1), 0 },

    { { SENSORS_ADM1031_ALARMS, "alarms", NOMAP, NOMAP, R }, 
                                ADM1031_SYSCTL_ALARMS, VALUE(1), 0 },
    { { 0 }, 0 }
  };


static sensors_chip_feature adm1031_features[] =
  {
    { { SENSORS_ADM1031_TEMP1, "temp1", NOMAP, NOMAP, R }, 
                               ADM1031_SYSCTL_TEMP1, VALUE(4), 3 },
    { { SENSORS_ADM1031_TEMP1_MAX, "temp1_max", SENSORS_ADM1031_TEMP1, 
                                   SENSORS_ADM1031_TEMP1, RW }, 
                                   ADM1031_SYSCTL_TEMP1, VALUE(1), 3 },
    { { SENSORS_ADM1031_TEMP1_MIN, "temp1_min", SENSORS_ADM1031_TEMP1, 
                                   SENSORS_ADM1031_TEMP1, RW }, 
                                   ADM1031_SYSCTL_TEMP1, VALUE(2), 3 },
    { { SENSORS_ADM1031_TEMP1_CRIT, "temp1_crit", SENSORS_ADM1031_TEMP1, 
                                    SENSORS_ADM1031_TEMP1, RW }, 
                                    ADM1031_SYSCTL_TEMP1, VALUE(3), 3 },

    { { SENSORS_ADM1031_TEMP2, "temp2", NOMAP, NOMAP, R }, 
                               ADM1031_SYSCTL_TEMP2, VALUE(4), 3 },
    { { SENSORS_ADM1031_TEMP2_MAX, "temp2_max", SENSORS_ADM1031_TEMP2, 
                                   SENSORS_ADM1031_TEMP2, RW }, 
                                   ADM1031_SYSCTL_TEMP2, VALUE(1), 3 },
    { { SENSORS_ADM1031_TEMP2_MIN, "temp2_min", SENSORS_ADM1031_TEMP2, 
                                   SENSORS_ADM1031_TEMP2, RW }, 
                                   ADM1031_SYSCTL_TEMP2, VALUE(2), 3 },
    { { SENSORS_ADM1031_TEMP2_CRIT, "temp2_crit", SENSORS_ADM1031_TEMP2, 
                                    SENSORS_ADM1031_TEMP2, RW }, 
                                    ADM1031_SYSCTL_TEMP2, VALUE(3), 3 },

    { { SENSORS_ADM1031_TEMP3, "temp3", NOMAP, NOMAP, R }, 
                               ADM1031_SYSCTL_TEMP3, VALUE(4), 3 },
    { { SENSORS_ADM1031_TEMP3_MAX, "temp3_max", SENSORS_ADM1031_TEMP3, 
                                   SENSORS_ADM1031_TEMP3, RW }, 
                                   ADM1031_SYSCTL_TEMP3, VALUE(1), 3 },
    { { SENSORS_ADM1031_TEMP3_MIN, "temp3_min", SENSORS_ADM1031_TEMP3, 
                                   SENSORS_ADM1031_TEMP3, RW }, 
                                   ADM1031_SYSCTL_TEMP3, VALUE(2), 3 },
    { { SENSORS_ADM1031_TEMP3_CRIT, "temp3_crit", SENSORS_ADM1031_TEMP3, 
                                    SENSORS_ADM1031_TEMP3, RW }, 
                                    ADM1031_SYSCTL_TEMP3, VALUE(3), 3 },

    { { SENSORS_ADM1031_FAN1, "fan1", NOMAP, NOMAP, R }, 
                              ADM1031_SYSCTL_FAN1, VALUE(2), 0 },
    { { SENSORS_ADM1031_FAN1_MIN, "fan1_min", SENSORS_ADM1031_FAN1, 
                                  SENSORS_ADM1031_FAN1, RW }, 
                                  ADM1031_SYSCTL_FAN1, VALUE(1), 0 },
    { { SENSORS_ADM1031_FAN1_DIV, "fan1_div", SENSORS_ADM1031_FAN1, NOMAP, RW }, 
                                  ADM1031_SYSCTL_FAN_DIV, VALUE(1), 0 },

    { { SENSORS_ADM1031_FAN2, "fan2", NOMAP, NOMAP, R }, 
                              ADM1031_SYSCTL_FAN2, VALUE(2), 0 },
    { { SENSORS_ADM1031_FAN2_MIN, "fan2_min", SENSORS_ADM1031_FAN2, 
                                  SENSORS_ADM1031_FAN2, RW }, 
                                  ADM1031_SYSCTL_FAN2, VALUE(1), 0 },
    { { SENSORS_ADM1031_FAN2_DIV, "fan2_div", SENSORS_ADM1031_FAN2, NOMAP, RW }, 
                                  ADM1031_SYSCTL_FAN_DIV, VALUE(2), 0 },

    { { SENSORS_ADM1031_ALARMS, "alarms", NOMAP, NOMAP, R }, 
                                ADM1031_SYSCTL_ALARMS, VALUE(1), 0 },
    { { 0 }, 0 }
  };

static sensors_chip_feature max1617_features[] =
  {
    { { SENSORS_MAX1617_TEMP, "temp", NOMAP, NOMAP, R }, 
                              ADM1021_SYSCTL_TEMP, VALUE(3), 0 },
    { { SENSORS_MAX1617_TEMP_HYST, "temp_low", SENSORS_MAX1617_TEMP, 
                                   SENSORS_MAX1617_TEMP, RW }, 
                                   ADM1021_SYSCTL_TEMP, VALUE(2), 0 },
    { { SENSORS_MAX1617_TEMP_OVER, "temp_over", SENSORS_MAX1617_TEMP, 
                                   SENSORS_MAX1617_TEMP, RW }, 
                                   ADM1021_SYSCTL_TEMP, VALUE(1), 0 },
    { { SENSORS_MAX1617_REMOTE_TEMP, "remote_temp", NOMAP, NOMAP, R }, 
                                     ADM1021_SYSCTL_REMOTE_TEMP, VALUE(3), 0 },
    { { SENSORS_MAX1617_REMOTE_TEMP_HYST, "remote_temp_low", 
                                          SENSORS_MAX1617_REMOTE_TEMP, 
                                          SENSORS_MAX1617_REMOTE_TEMP, RW }, 
                                          ADM1021_SYSCTL_REMOTE_TEMP, VALUE(2), 
                                          0 },
    { { SENSORS_MAX1617_REMOTE_TEMP_OVER, "remote_temp_over", 
                                          SENSORS_MAX1617_REMOTE_TEMP, 
                                          SENSORS_MAX1617_REMOTE_TEMP, RW }, 
                                          ADM1021_SYSCTL_REMOTE_TEMP, VALUE(1), 
                                          0 },
    { { SENSORS_MAX1617_ALARMS, "alarms", NOMAP, NOMAP, R }, 
                                ADM1021_SYSCTL_ALARMS, VALUE(1), 0 },
    { { 0 }, 0 }
  };

static sensors_chip_feature max1617a_features[] =
  {
    { { SENSORS_MAX1617A_TEMP, "temp", NOMAP, NOMAP, R }, 
                               ADM1021_SYSCTL_TEMP, VALUE(3), 0 },
    { { SENSORS_MAX1617A_TEMP_HYST, "temp_low", SENSORS_MAX1617A_TEMP, 
                                    SENSORS_MAX1617A_TEMP, RW }, 
                                    ADM1021_SYSCTL_TEMP, VALUE(2), 0 },
    { { SENSORS_MAX1617A_TEMP_OVER, "temp_over", SENSORS_MAX1617A_TEMP, 
                                    SENSORS_MAX1617A_TEMP, RW }, 
                                    ADM1021_SYSCTL_TEMP, VALUE(1), 0 },
    { { SENSORS_MAX1617A_REMOTE_TEMP, "remote_temp", NOMAP, NOMAP, R }, 
                                      ADM1021_SYSCTL_REMOTE_TEMP, VALUE(3), 0 },
    { { SENSORS_MAX1617A_REMOTE_TEMP_HYST, "remote_temp_low", 
                                           SENSORS_MAX1617A_REMOTE_TEMP, 
                                           SENSORS_MAX1617A_REMOTE_TEMP, RW }, 
                                           ADM1021_SYSCTL_REMOTE_TEMP, VALUE(2), 
                                           0 },
    { { SENSORS_MAX1617A_REMOTE_TEMP_OVER, "remote_temp_over", 
                                           SENSORS_MAX1617A_REMOTE_TEMP, 
                                           SENSORS_MAX1617A_REMOTE_TEMP, RW }, 
                                           ADM1021_SYSCTL_REMOTE_TEMP, VALUE(1), 
                                           0 },
    { { SENSORS_MAX1617A_ALARMS, "alarms", NOMAP, NOMAP, R }, 
                                 ADM1021_SYSCTL_ALARMS, VALUE(1), 0 },
    { { 0 }, 0 }
  };

static sensors_chip_feature gl518_features[] =
  {
    { { SENSORS_GL518_VDD, "vdd", NOMAP, NOMAP, R }, 
                           GL518_SYSCTL_VDD, VALUE(3), 2, "in0_input", 3 },
    { { SENSORS_GL518_VIN1, "vin1", NOMAP, NOMAP, R }, 
                            GL518_SYSCTL_VIN1, VALUE(3), 2 },
    { { SENSORS_GL518_VIN2, "vin2", NOMAP, NOMAP, R }, 
                            GL518_SYSCTL_VIN2, VALUE(3), 2 },
    { { SENSORS_GL518_VIN3, "vin3", NOMAP, NOMAP, R }, 
                            GL518_SYSCTL_VIN3, VALUE(3), 2 },
    { { SENSORS_GL518_VDD_MIN, "vdd_min", SENSORS_GL518_VDD, SENSORS_GL518_VDD, 
                               RW }, 
                               GL518_SYSCTL_VDD, VALUE(1), 2 , "in0_min", 3 },
    { { SENSORS_GL518_VIN1_MIN, "vin1_min", SENSORS_GL518_VIN1, 
                                SENSORS_GL518_VIN1, RW }, 
                                GL518_SYSCTL_VIN1, VALUE(1), 2 },
    { { SENSORS_GL518_VIN2_MIN, "vin2_min", SENSORS_GL518_VIN2, 
                                SENSORS_GL518_VIN2, RW }, 
                                GL518_SYSCTL_VIN2, VALUE(1), 2 },
    { { SENSORS_GL518_VIN3_MIN, "vin3_min", SENSORS_GL518_VIN3, 
                                SENSORS_GL518_VIN3, RW }, 
                                GL518_SYSCTL_VIN3, VALUE(1), 2 },
    { { SENSORS_GL518_VDD_MAX, "vdd_max", SENSORS_GL518_VDD, SENSORS_GL518_VDD, 
                               RW }, 
                               GL518_SYSCTL_VDD, VALUE(2), 2 , "in0_max", 3 },
    { { SENSORS_GL518_VIN1_MAX, "vin1_max", SENSORS_GL518_VIN1, 
                                SENSORS_GL518_VIN1, RW }, 
                                GL518_SYSCTL_VIN1, VALUE(2), 2 },
    { { SENSORS_GL518_VIN2_MAX, "vin2_max", SENSORS_GL518_VIN2, 
                                SENSORS_GL518_VIN2, RW }, 
                                GL518_SYSCTL_VIN2, VALUE(2), 2 },
    { { SENSORS_GL518_VIN3_MAX, "vin3_max", SENSORS_GL518_VIN3, 
                                SENSORS_GL518_VIN3, RW }, 
                                GL518_SYSCTL_VIN3, VALUE(2), 2 },
    { { SENSORS_GL518_FAN1, "fan1", NOMAP, NOMAP, R }, 
                            GL518_SYSCTL_FAN1, VALUE(2), 0 },
    { { SENSORS_GL518_FAN2, "fan2", NOMAP, NOMAP, R }, 
                            GL518_SYSCTL_FAN2, VALUE(2), 0 },
    { { SENSORS_GL518_FAN1_MIN, "fan1_min", SENSORS_GL518_FAN1, 
                                SENSORS_GL518_FAN1, RW }, 
                                GL518_SYSCTL_FAN1, VALUE(1), 0 },
    { { SENSORS_GL518_FAN2_MIN, "fan2_min", SENSORS_GL518_FAN2, 
                                SENSORS_GL518_FAN2, RW }, 
                                GL518_SYSCTL_FAN2, VALUE(1), 0 },
    { { SENSORS_GL518_TEMP, "temp", NOMAP, NOMAP, R }, 
                            GL518_SYSCTL_TEMP, VALUE(3), 1 },
    { { SENSORS_GL518_TEMP_HYST, "temp_hyst", SENSORS_GL518_TEMP, 
                                 SENSORS_GL518_TEMP, RW }, 
                                 GL518_SYSCTL_TEMP, VALUE(2), 1 },
    { { SENSORS_GL518_TEMP_OVER, "temp_over", SENSORS_GL518_TEMP, 
                                 SENSORS_GL518_TEMP, RW }, 
                                 GL518_SYSCTL_TEMP, VALUE(1), 1 },
    { { SENSORS_GL518_FAN1_DIV, "fan1_div", SENSORS_GL518_FAN1, NOMAP, RW }, 
                                GL518_SYSCTL_FAN_DIV, VALUE(1), 0 },
    { { SENSORS_GL518_FAN2_DIV, "fan2_div", SENSORS_GL518_FAN2, NOMAP, RW }, 
                                GL518_SYSCTL_FAN_DIV, VALUE(2), 0 },
    { { SENSORS_GL518_ALARMS, "alarms", NOMAP, NOMAP, R }, 
                              GL518_SYSCTL_ALARMS, VALUE(1), 0 },
    { { SENSORS_GL518_BEEP_ENABLE, "beep_enable", SENSORS_GL518_ALARMS, NOMAP, 
                                   RW }, 
                                   GL518_SYSCTL_BEEP, VALUE(1), 0 },
    { { SENSORS_GL518_BEEPS, "beeps", SENSORS_GL518_ALARMS, SENSORS_GL518_ALARMS, 
                             RW }, 
                             GL518_SYSCTL_BEEP, VALUE(2), 0 },
    { { SENSORS_GL518_FAN1OFF, "fan1_off", SENSORS_GL518_FAN1, NOMAP, RW }, 
                               GL518_SYSCTL_FAN1OFF, VALUE(1), 0, "fan1_auto", 
                               0 },
    { { SENSORS_GL518_FAN1PIN, "fan1_off_pin", SENSORS_GL518_FAN1OFF, NOMAP, RW }, 
                               GL518_SYSCTL_FAN1OFF, VALUE(2), 0 },
    { { SENSORS_GL518_ITERATE, "iterate", NOMAP, NOMAP, RW }, 
                               GL518_SYSCTL_ITERATE, VALUE(1), 0 },
    { { 0 }, 0 }
  };

static sensors_chip_feature gl520_features[] =
  {
    { { SENSORS_GL520_VDD, "vdd", NOMAP, NOMAP, R }, 
                           GL520_SYSCTL_VDD, VALUE(3), 2, "in0_input", 3 },
    { { SENSORS_GL520_VIN1, "vin1", NOMAP, NOMAP, R }, 
                            GL520_SYSCTL_VIN1, VALUE(3), 2 },
    { { SENSORS_GL520_VIN2, "vin2", NOMAP, NOMAP, R }, 
                            GL520_SYSCTL_VIN2, VALUE(3), 2 },
    { { SENSORS_GL520_VIN3, "vin3", NOMAP, NOMAP, R }, 
                            GL520_SYSCTL_VIN3, VALUE(3), 2 },
    { { SENSORS_GL520_VIN4, "vin4", NOMAP, NOMAP, R }, 
                            GL520_SYSCTL_VIN4, VALUE(3), 2 },
    { { SENSORS_GL520_VDD_MIN, "vdd_min", SENSORS_GL520_VDD, SENSORS_GL520_VDD, 
                               RW }, 
                               GL520_SYSCTL_VDD, VALUE(1), 2, "in0_min", 3 },
    { { SENSORS_GL520_VIN1_MIN, "vin1_min", SENSORS_GL520_VIN1, 
                                SENSORS_GL520_VIN1, RW }, 
                                GL520_SYSCTL_VIN1, VALUE(1), 2 },
    { { SENSORS_GL520_VIN2_MIN, "vin2_min", SENSORS_GL520_VIN2, 
                                SENSORS_GL520_VIN2, RW }, 
                                GL520_SYSCTL_VIN2, VALUE(1), 2 },
    { { SENSORS_GL520_VIN3_MIN, "vin3_min", SENSORS_GL520_VIN3, 
                                SENSORS_GL520_VIN3, RW }, 
                                GL520_SYSCTL_VIN3, VALUE(1), 2 },
    { { SENSORS_GL520_VIN4_MIN, "vin4_min", SENSORS_GL520_VIN4, 
                                SENSORS_GL520_VIN4, RW }, 
                                GL520_SYSCTL_VIN4, VALUE(1), 2 },
    { { SENSORS_GL520_VDD_MAX, "vdd_max", SENSORS_GL520_VDD, SENSORS_GL520_VDD, 
                               RW }, 
                               GL520_SYSCTL_VDD, VALUE(2), 2, "in0_max", 3 },
    { { SENSORS_GL520_VIN1_MAX, "vin1_max", SENSORS_GL520_VIN1, 
                                SENSORS_GL520_VIN1, RW }, 
                                GL520_SYSCTL_VIN1, VALUE(2), 2 },
    { { SENSORS_GL520_VIN2_MAX, "vin2_max", SENSORS_GL520_VIN2, 
                                SENSORS_GL520_VIN2, RW }, 
                                GL520_SYSCTL_VIN2, VALUE(2), 2 },
    { { SENSORS_GL520_VIN3_MAX, "vin3_max", SENSORS_GL520_VIN3, 
                                SENSORS_GL520_VIN3, RW }, 
                                GL520_SYSCTL_VIN3, VALUE(2), 2 },
    { { SENSORS_GL520_VIN4_MAX, "vin4_max", SENSORS_GL520_VIN4, 
                                SENSORS_GL520_VIN4, RW }, 
                                GL520_SYSCTL_VIN4, VALUE(2), 2 },
    { { SENSORS_GL520_FAN1, "fan1", NOMAP, NOMAP, R }, 
                            GL520_SYSCTL_FAN1, VALUE(2), 0 },
    { { SENSORS_GL520_FAN2, "fan2", NOMAP, NOMAP, R }, 
                            GL520_SYSCTL_FAN2, VALUE(2), 0 },
    { { SENSORS_GL520_FAN1_MIN, "fan1_min", SENSORS_GL520_FAN1, 
                                SENSORS_GL520_FAN1, RW }, 
                                GL520_SYSCTL_FAN1, VALUE(1), 0 },
    { { SENSORS_GL520_FAN2_MIN, "fan2_min", SENSORS_GL520_FAN2, 
                                SENSORS_GL520_FAN2, RW }, 
                                GL520_SYSCTL_FAN2, VALUE(1), 0 },
    { { SENSORS_GL520_TEMP1, "temp1", NOMAP, NOMAP, R }, 
                             GL520_SYSCTL_TEMP1, VALUE(3), 1 },
    { { SENSORS_GL520_TEMP1_HYST, "temp1_hyst", SENSORS_GL520_TEMP1, 
                                  SENSORS_GL520_TEMP1, RW }, 
                                  GL520_SYSCTL_TEMP1, VALUE(2), 1 },
    { { SENSORS_GL520_TEMP1_OVER, "temp1_over", SENSORS_GL520_TEMP1, 
                                  SENSORS_GL520_TEMP1, RW }, 
                                  GL520_SYSCTL_TEMP1, VALUE(1), 1 },
    { { SENSORS_GL520_TEMP2, "temp2", NOMAP, NOMAP, R }, 
                             GL520_SYSCTL_TEMP2, VALUE(3), 1 },
    { { SENSORS_GL520_TEMP2_HYST, "temp2_hyst", SENSORS_GL520_TEMP2, 
                                  SENSORS_GL520_TEMP2, RW }, 
                                  GL520_SYSCTL_TEMP2, VALUE(2), 1 },
    { { SENSORS_GL520_TEMP2_OVER, "temp2_over", SENSORS_GL520_TEMP2, 
                                  SENSORS_GL520_TEMP2, RW }, 
                                  GL520_SYSCTL_TEMP2, VALUE(1), 1 },
    { { SENSORS_GL520_VID, "vid", NOMAP, NOMAP, R }, 
                           GL520_SYSCTL_VID, VALUE(1), 2 },
    { { SENSORS_GL520_FAN1_DIV, "fan1_div", SENSORS_GL520_FAN1, NOMAP, RW }, 
                                GL520_SYSCTL_FAN_DIV, VALUE(1), 0 },
    { { SENSORS_GL520_FAN2_DIV, "fan2_div", SENSORS_GL520_FAN2, NOMAP, RW }, 
                                GL520_SYSCTL_FAN_DIV, VALUE(2), 0 },
    { { SENSORS_GL520_ALARMS, "alarms", NOMAP, NOMAP, R }, 
                              GL520_SYSCTL_ALARMS, VALUE(1), 0 },
    { { SENSORS_GL520_BEEP_ENABLE, "beep_enable", SENSORS_GL520_ALARMS, NOMAP, 
                                   RW }, 
                                   GL520_SYSCTL_BEEP, VALUE(1), 0 },
    { { SENSORS_GL520_BEEPS, "beeps", SENSORS_GL520_ALARMS, SENSORS_GL520_ALARMS, 
                             RW }, 
                             GL520_SYSCTL_BEEP, VALUE(2), 0 },
    { { SENSORS_GL520_FAN1OFF, "fan1_off", SENSORS_GL520_FAN1, NOMAP, RW }, 
                               GL520_SYSCTL_FAN1OFF, VALUE(1), 0 },
    { { SENSORS_GL520_TWOTEMPS, "two_temps", NOMAP, NOMAP, RW }, 
                                GL520_SYSCTL_CONFIG, VALUE(1), 0 },
    { { 0 }, 0 }
  };

static sensors_chip_feature lm80_features[] =
  { 
    { { SENSORS_LM80_IN0, "in0", NOMAP, NOMAP, R }, 
                          LM80_SYSCTL_IN0, VALUE(3), 2 },
    { { SENSORS_LM80_IN1, "in1", NOMAP, NOMAP, R }, 
                          LM80_SYSCTL_IN1, VALUE(3), 2 },
    { { SENSORS_LM80_IN2, "in2", NOMAP, NOMAP, R }, 
                          LM80_SYSCTL_IN2, VALUE(3), 2 },
    { { SENSORS_LM80_IN3, "in3", NOMAP, NOMAP, R }, 
                          LM80_SYSCTL_IN3, VALUE(3), 2 },
    { { SENSORS_LM80_IN4, "in4", NOMAP, NOMAP, R }, 
                          LM80_SYSCTL_IN4, VALUE(3), 2 },
    { { SENSORS_LM80_IN5, "in5", NOMAP, NOMAP, R }, 
                          LM80_SYSCTL_IN5, VALUE(3), 2 },
    { { SENSORS_LM80_IN6, "in6", NOMAP, NOMAP, R }, 
                          LM80_SYSCTL_IN6, VALUE(3), 2 },
    { { SENSORS_LM80_IN0_MIN, "in0_min", SENSORS_LM80_IN0, SENSORS_LM80_IN0, RW }, 
                              LM80_SYSCTL_IN0, VALUE(1), 2 },
    { { SENSORS_LM80_IN1_MIN, "in1_min", SENSORS_LM80_IN1, SENSORS_LM80_IN1, RW }, 
                              LM80_SYSCTL_IN1, VALUE(1), 2 },
    { { SENSORS_LM80_IN2_MIN, "in2_min", SENSORS_LM80_IN2, SENSORS_LM80_IN2, RW }, 
                              LM80_SYSCTL_IN2, VALUE(1), 2 },
    { { SENSORS_LM80_IN3_MIN, "in3_min", SENSORS_LM80_IN3, SENSORS_LM80_IN3, RW }, 
                              LM80_SYSCTL_IN3, VALUE(1), 2 },
    { { SENSORS_LM80_IN4_MIN, "in4_min", SENSORS_LM80_IN4, SENSORS_LM80_IN4, RW }, 
                              LM80_SYSCTL_IN4, VALUE(1), 2 },
    { { SENSORS_LM80_IN5_MIN, "in5_min", SENSORS_LM80_IN5, SENSORS_LM80_IN5, RW }, 
                              LM80_SYSCTL_IN5, VALUE(1), 2 },
    { { SENSORS_LM80_IN6_MIN, "in6_min", SENSORS_LM80_IN6, SENSORS_LM80_IN6, RW }, 
                              LM80_SYSCTL_IN6, VALUE(1), 2 },
    { { SENSORS_LM80_IN0_MAX, "in0_max", SENSORS_LM80_IN0, SENSORS_LM80_IN0, RW }, 
                              LM80_SYSCTL_IN0, VALUE(2), 2 },
    { { SENSORS_LM80_IN1_MAX, "in1_max", SENSORS_LM80_IN1, SENSORS_LM80_IN1, RW }, 
                              LM80_SYSCTL_IN1, VALUE(2), 2 },
    { { SENSORS_LM80_IN2_MAX, "in2_max", SENSORS_LM80_IN2, SENSORS_LM80_IN2, RW }, 
                              LM80_SYSCTL_IN2, VALUE(2), 2 },
    { { SENSORS_LM80_IN3_MAX, "in3_max", SENSORS_LM80_IN3, SENSORS_LM80_IN3, RW }, 
                              LM80_SYSCTL_IN3, VALUE(2), 2 },
    { { SENSORS_LM80_IN4_MAX, "in4_max", SENSORS_LM80_IN4, SENSORS_LM80_IN4, RW }, 
                              LM80_SYSCTL_IN4, VALUE(2), 2 },
    { { SENSORS_LM80_IN5_MAX, "in5_max", SENSORS_LM80_IN5, SENSORS_LM80_IN5, RW }, 
                              LM80_SYSCTL_IN5, VALUE(2), 2 },
    { { SENSORS_LM80_IN6_MAX, "in6_max", SENSORS_LM80_IN6, SENSORS_LM80_IN6, RW }, 
                              LM80_SYSCTL_IN6, VALUE(2), 2 },
    { { SENSORS_LM80_FAN1, "fan1", NOMAP, NOMAP, R }, 
                           LM80_SYSCTL_FAN1, VALUE(2), 0 },
    { { SENSORS_LM80_FAN2, "fan2", NOMAP, NOMAP, R }, 
                           LM80_SYSCTL_FAN2, VALUE(2), 0 },
    { { SENSORS_LM80_FAN1_MIN, "fan1_min", SENSORS_LM80_FAN1, NOMAP, RW }, 
                               LM80_SYSCTL_FAN1, VALUE(1), 0 },
    { { SENSORS_LM80_FAN2_MIN, "fan2_min", SENSORS_LM80_FAN2, NOMAP, RW }, 
                               LM80_SYSCTL_FAN2, VALUE(1), 0 },
    { { SENSORS_LM80_TEMP, "temp", NOMAP, NOMAP, R }, 
                           LM80_SYSCTL_TEMP, VALUE(5), 2 },
    { { SENSORS_LM80_TEMP_HOT_HYST, "temp_hot_hyst", SENSORS_LM80_TEMP, 
                                    SENSORS_LM80_TEMP, RW }, 
                                    LM80_SYSCTL_TEMP, VALUE(2), 2, 
                                    "temp1_max_hyst", 3 },
    { { SENSORS_LM80_TEMP_HOT_MAX, "temp_hot_max", SENSORS_LM80_TEMP, 
                                   SENSORS_LM80_TEMP, RW }, 
                                   LM80_SYSCTL_TEMP, VALUE(1), 2, "temp1_max", 
                                   3 },
    { { SENSORS_LM80_TEMP_OS_HYST, "temp_os_hyst", SENSORS_LM80_TEMP, 
                                   SENSORS_LM80_TEMP, RW }, 
                                   LM80_SYSCTL_TEMP, VALUE(4), 2, 
                                   "temp1_crit_hyst", 3 },
    { { SENSORS_LM80_TEMP_OS_MAX, "temp_os_max", SENSORS_LM80_TEMP, 
                                  SENSORS_LM80_TEMP, RW }, 
                                  LM80_SYSCTL_TEMP, VALUE(3), 2, "temp1_crit", 
                                  3 },
    { { SENSORS_LM80_FAN1_DIV, "fan1_div", SENSORS_LM80_FAN1, NOMAP, RW }, 
                               LM80_SYSCTL_FAN_DIV, VALUE(1), 0 },
    { { SENSORS_LM80_FAN2_DIV, "fan2_div", SENSORS_LM80_FAN2, NOMAP, RW }, 
                               LM80_SYSCTL_FAN_DIV, VALUE(2), 0 },
    { { SENSORS_LM80_ALARMS, "alarms", NOMAP, NOMAP, R }, 
                             LM80_SYSCTL_ALARMS, VALUE(1), 0 },
    { { 0 }, 0 }
  };


static sensors_chip_feature lm83_features[] =
  {
    { { SENSORS_LM83_LOCAL_TEMP, "temp1", NOMAP, NOMAP, R }, 
                                 LM83_SYSCTL_LOCAL_TEMP, VALUE(2), 0 },
    { { SENSORS_LM83_LOCAL_HIGH, "temp1_high", SENSORS_LM83_LOCAL_TEMP, 
                                 SENSORS_LM83_LOCAL_TEMP, RW }, 
                                 LM83_SYSCTL_LOCAL_TEMP, VALUE(1), 0 },
    { { SENSORS_LM83_REMOTE1_TEMP, "temp2", NOMAP, NOMAP, R }, 
                                   LM83_SYSCTL_REMOTE1_TEMP, VALUE(2), 0 },
    { { SENSORS_LM83_REMOTE1_HIGH, "temp2_high", SENSORS_LM83_REMOTE1_TEMP, 
                                   SENSORS_LM83_REMOTE1_TEMP, RW }, 
                                   LM83_SYSCTL_REMOTE1_TEMP, VALUE(1), 0 },
    { { SENSORS_LM83_REMOTE2_TEMP, "temp3", NOMAP, NOMAP, R }, 
                                   LM83_SYSCTL_REMOTE2_TEMP, VALUE(2), 0 },
    { { SENSORS_LM83_REMOTE2_HIGH, "temp3_high", SENSORS_LM83_REMOTE2_TEMP, 
                                   SENSORS_LM83_REMOTE2_TEMP, RW }, 
                                   LM83_SYSCTL_REMOTE2_TEMP, VALUE(1), 0 },
    { { SENSORS_LM83_REMOTE3_TEMP, "temp4", NOMAP, NOMAP, R }, 
                                   LM83_SYSCTL_REMOTE3_TEMP, VALUE(2), 0 },
    { { SENSORS_LM83_REMOTE3_HIGH, "temp4_high", SENSORS_LM83_REMOTE3_TEMP, 
                                   SENSORS_LM83_REMOTE3_TEMP, RW }, 
                                   LM83_SYSCTL_REMOTE3_TEMP, VALUE(1), 0 },
    { { SENSORS_LM83_TCRIT, "tcrit", NOMAP, NOMAP, RW }, 
                            LM83_SYSCTL_TCRIT, VALUE(1), 0, "temp3_crit", 3, 
                            "temp_crit" },
    { { SENSORS_LM83_ALARMS, "alarms", NOMAP, NOMAP, R }, 
                             LM83_SYSCTL_ALARMS, VALUE(1), 0 },
    { { 0 }, 0 }
  };


static sensors_chip_feature lm90_features[] =
  {
    { { SENSORS_LM90_LOCAL_TEMP, "temp1", NOMAP, NOMAP, R }, 
                                 LM90_SYSCTL_LOCAL_TEMP, VALUE(3), 0 },
    { { SENSORS_LM90_LOCAL_HIGH, "temp1_high", SENSORS_LM90_LOCAL_TEMP, 
                                 SENSORS_LM90_LOCAL_TEMP, RW }, 
                                 LM90_SYSCTL_LOCAL_TEMP, VALUE(1), 0 },
    { { SENSORS_LM90_LOCAL_LOW, "temp1_low", SENSORS_LM90_LOCAL_TEMP, 
                                SENSORS_LM90_LOCAL_TEMP, RW }, 
                                LM90_SYSCTL_LOCAL_TEMP, VALUE(2), 0 },
    { { SENSORS_LM90_REMOTE_TEMP, "temp2", NOMAP, NOMAP, R }, 
                                  LM90_SYSCTL_REMOTE_TEMP, VALUE(3), 1 },
    { { SENSORS_LM90_REMOTE_HIGH, "temp2_high", SENSORS_LM90_REMOTE_TEMP, 
                                  SENSORS_LM90_REMOTE_TEMP, RW }, 
                                  LM90_SYSCTL_REMOTE_TEMP, VALUE(1), 1 },
    { { SENSORS_LM90_REMOTE_LOW, "temp2_low", SENSORS_LM90_REMOTE_TEMP, 
                                 SENSORS_LM90_REMOTE_TEMP, RW }, 
                                 LM90_SYSCTL_REMOTE_TEMP, VALUE(2), 1 },
    { { SENSORS_LM90_LOCAL_TCRIT, "tcrit1", SENSORS_LM90_LOCAL_TEMP, 
                                  SENSORS_LM90_LOCAL_TEMP, RW }, 
                                  LM90_SYSCTL_LOCAL_TCRIT, VALUE(1), 0, 
                                  "temp1_crit", 3 },
    { { SENSORS_LM90_REMOTE_TCRIT, "tcrit2", SENSORS_LM90_REMOTE_TEMP, 
                                   SENSORS_LM90_REMOTE_TEMP, RW }, 
                                   LM90_SYSCTL_REMOTE_TCRIT, VALUE(1), 0, 
                                   "temp2_crit", 3 },
    { { SENSORS_LM90_LOCAL_TCRIT_HYST, "hyst1", SENSORS_LM90_LOCAL_TEMP, 
                                       SENSORS_LM90_LOCAL_TEMP, RW }, 
                                       LM90_SYSCTL_LOCAL_HYST, VALUE(1), 0, 
                                       "temp1_crit_hyst", 3 },
    { { SENSORS_LM90_REMOTE_TCRIT_HYST, "hyst2", SENSORS_LM90_REMOTE_TEMP, 
                                        SENSORS_LM90_REMOTE_TEMP, R }, 
                                        LM90_SYSCTL_REMOTE_HYST, VALUE(1), 0, 
                                        "temp2_crit_hyst", 3 },
    { { SENSORS_LM90_ALARMS, "alarms", NOMAP, NOMAP, R }, 
                             LM90_SYSCTL_ALARMS, VALUE(1), 0 },
    { { 0 }, 0 }
  };


static sensors_chip_feature lm63_features[] =
  {
    { { SENSORS_LM63_LOCAL_TEMP, "temp1", NOMAP, NOMAP, R }, 
                                 LM63_SYSCTL_TEMP1, VALUE(2), 0 },
    { { SENSORS_LM63_LOCAL_HIGH, "temp1_high", SENSORS_LM63_LOCAL_TEMP, 
                                 SENSORS_LM63_LOCAL_TEMP, RW }, 
                                 LM63_SYSCTL_TEMP1, VALUE(1), 0 },
    { { SENSORS_LM63_REMOTE_TEMP, "temp2", NOMAP, NOMAP, R }, 
                                  LM63_SYSCTL_TEMP2, VALUE(3), 3 },
    { { SENSORS_LM63_REMOTE_HIGH, "temp2_high", SENSORS_LM63_REMOTE_TEMP, 
                                  SENSORS_LM63_REMOTE_TEMP, RW }, 
                                  LM63_SYSCTL_TEMP2, VALUE(1), 3 },
    { { SENSORS_LM63_REMOTE_LOW, "temp2_low", SENSORS_LM63_REMOTE_TEMP, 
                                 SENSORS_LM63_REMOTE_TEMP, RW }, 
                                 LM63_SYSCTL_TEMP2, VALUE(2), 3 },
    { { SENSORS_LM63_REMOTE_TCRIT, "temp2_crit", SENSORS_LM63_REMOTE_TEMP, 
                                   SENSORS_LM63_REMOTE_TEMP, R }, 
                                   LM63_SYSCTL_TEMP2_TCRIT, VALUE(1), 0 },
    { { SENSORS_LM63_REMOTE_TCRIT_HYST, "temp2_crit_hyst", 
                                        SENSORS_LM63_REMOTE_TEMP, 
                                        SENSORS_LM63_REMOTE_TEMP, RW }, 
                                        LM63_SYSCTL_TEMP2_TCRIT_HYST, VALUE(1), 
                                        0, "temp2_crit_hyst", 3 },
    { { SENSORS_LM63_ALARMS, "alarms", NOMAP, NOMAP, R }, 
                             LM63_SYSCTL_ALARMS, VALUE(1), 0 },
    { { SENSORS_LM63_FAN_LOW, "fan1_min", SENSORS_LM63_FAN, SENSORS_LM63_FAN, RW }, 
                              LM63_SYSCTL_FAN1, VALUE(1), 0 },
    { { SENSORS_LM63_FAN, "fan1", NOMAP, NOMAP, R }, 
                          LM63_SYSCTL_FAN1, VALUE(2), 0 },
    { { SENSORS_LM63_PWM, "pwm1", NOMAP, NOMAP, RW }, 
                          LM63_SYSCTL_PWM1, VALUE(1), 0 },
    { { SENSORS_LM63_PWM_ENABLE, "pwm1_enable", SENSORS_LM63_PWM, NOMAP, RW }, 
                                 LM63_SYSCTL_PWM1, VALUE(2), 0 },
    { { 0 }, 0 }
  };


static sensors_chip_feature w83781d_features[] =
  { 
    { { SENSORS_W83781D_IN0, "in0", NOMAP, NOMAP, R }, 
                             W83781D_SYSCTL_IN0, VALUE(3), 2 },
    { { SENSORS_W83781D_IN1, "in1", NOMAP, NOMAP, R }, 
                             W83781D_SYSCTL_IN1, VALUE(3), 2 },
    { { SENSORS_W83781D_IN2, "in2", NOMAP, NOMAP, R }, 
                             W83781D_SYSCTL_IN2, VALUE(3), 2 },
    { { SENSORS_W83781D_IN3, "in3", NOMAP, NOMAP, R }, 
                             W83781D_SYSCTL_IN3, VALUE(3), 2 },
    { { SENSORS_W83781D_IN4, "in4", NOMAP, NOMAP, R }, 
                             W83781D_SYSCTL_IN4, VALUE(3), 2 },
    { { SENSORS_W83781D_IN5, "in5", NOMAP, NOMAP, R }, 
                             W83781D_SYSCTL_IN5, VALUE(3), 2 },
    { { SENSORS_W83781D_IN6, "in6", NOMAP, NOMAP, R }, 
                             W83781D_SYSCTL_IN6, VALUE(3), 2 },
    { { SENSORS_W83781D_IN0_MIN, "in0_min", SENSORS_W83781D_IN0, 
                                 SENSORS_W83781D_IN0, RW }, 
                                 W83781D_SYSCTL_IN0, VALUE(1), 2 },
    { { SENSORS_W83781D_IN1_MIN, "in1_min", SENSORS_W83781D_IN1, 
                                 SENSORS_W83781D_IN1, RW }, 
                                 W83781D_SYSCTL_IN1, VALUE(1), 2 },
    { { SENSORS_W83781D_IN2_MIN, "in2_min", SENSORS_W83781D_IN2, 
                                 SENSORS_W83781D_IN2, RW }, 
                                 W83781D_SYSCTL_IN2, VALUE(1), 2 },
    { { SENSORS_W83781D_IN3_MIN, "in3_min", SENSORS_W83781D_IN3, 
                                 SENSORS_W83781D_IN3, RW }, 
                                 W83781D_SYSCTL_IN3, VALUE(1), 2 },
    { { SENSORS_W83781D_IN4_MIN, "in4_min", SENSORS_W83781D_IN4, 
                                 SENSORS_W83781D_IN4, RW }, 
                                 W83781D_SYSCTL_IN4, VALUE(1), 2 },
/* swap 5 and 6 min/max because compute lines will be inverting */
    { { SENSORS_W83781D_IN5_MIN, "in5_min", SENSORS_W83781D_IN5, 
                                 SENSORS_W83781D_IN5, RW }, 
                                 W83781D_SYSCTL_IN5, VALUE(2), 2, "in5_max", 3 },
    { { SENSORS_W83781D_IN6_MIN, "in6_min", SENSORS_W83781D_IN6, 
                                 SENSORS_W83781D_IN6, RW }, 
                                 W83781D_SYSCTL_IN6, VALUE(2), 2, "in6_max", 3 },
    { { SENSORS_W83781D_IN0_MAX, "in0_max", SENSORS_W83781D_IN0, 
                                 SENSORS_W83781D_IN0, RW }, 
                                 W83781D_SYSCTL_IN0, VALUE(2), 2 },
    { { SENSORS_W83781D_IN1_MAX, "in1_max", SENSORS_W83781D_IN1, 
                                 SENSORS_W83781D_IN1, RW }, 
                                 W83781D_SYSCTL_IN1, VALUE(2), 2 },
    { { SENSORS_W83781D_IN2_MAX, "in2_max", SENSORS_W83781D_IN2, 
                                 SENSORS_W83781D_IN2, RW }, 
                                 W83781D_SYSCTL_IN2, VALUE(2), 2 },
    { { SENSORS_W83781D_IN3_MAX, "in3_max", SENSORS_W83781D_IN3, 
                                 SENSORS_W83781D_IN3, RW }, 
                                 W83781D_SYSCTL_IN3, VALUE(2), 2 },
    { { SENSORS_W83781D_IN4_MAX, "in4_max", SENSORS_W83781D_IN4, 
                                 SENSORS_W83781D_IN4, RW }, 
                                 W83781D_SYSCTL_IN4, VALUE(2), 2 },
    { { SENSORS_W83781D_IN5_MAX, "in5_max", SENSORS_W83781D_IN5, 
                                 SENSORS_W83781D_IN5, RW }, 
                                 W83781D_SYSCTL_IN5, VALUE(1), 2, "in5_min", 3 },
    { { SENSORS_W83781D_IN6_MAX, "in6_max", SENSORS_W83781D_IN6, 
                                 SENSORS_W83781D_IN6, RW }, 
                                 W83781D_SYSCTL_IN6, VALUE(1), 2, "in6_min", 3 },
    { { SENSORS_W83781D_FAN1, "fan1", NOMAP, NOMAP, R }, 
                              W83781D_SYSCTL_FAN1, VALUE(2), 0 },
    { { SENSORS_W83781D_FAN2, "fan2", NOMAP, NOMAP, R }, 
                              W83781D_SYSCTL_FAN2, VALUE(2), 0 },
    { { SENSORS_W83781D_FAN3, "fan3", NOMAP, NOMAP, R }, 
                              W83781D_SYSCTL_FAN3, VALUE(2), 0 },
    { { SENSORS_W83781D_FAN1_MIN, "fan1_min", SENSORS_W83781D_FAN1, 
                                  SENSORS_W83781D_FAN1, RW }, 
                                  W83781D_SYSCTL_FAN1, VALUE(1), 0 },
    { { SENSORS_W83781D_FAN2_MIN, "fan2_min", SENSORS_W83781D_FAN2, 
                                  SENSORS_W83781D_FAN2, RW }, 
                                  W83781D_SYSCTL_FAN2, VALUE(1), 0 },
    { { SENSORS_W83781D_FAN3_MIN, "fan3_min", SENSORS_W83781D_FAN3, 
                                  SENSORS_W83781D_FAN3, RW }, 
                                  W83781D_SYSCTL_FAN3, VALUE(1), 0 },
    { { SENSORS_W83781D_TEMP1, "temp1", NOMAP, NOMAP, R }, 
                               W83781D_SYSCTL_TEMP1, VALUE(3), 1 },
    { { SENSORS_W83781D_TEMP1_HYST, "temp1_hyst", SENSORS_W83781D_TEMP1, 
                                    SENSORS_W83781D_TEMP1, RW }, 
                                    W83781D_SYSCTL_TEMP1, VALUE(2), 1 },
    { { SENSORS_W83781D_TEMP1_OVER, "temp1_over", SENSORS_W83781D_TEMP1, 
                                    SENSORS_W83781D_TEMP1, RW }, 
                                    W83781D_SYSCTL_TEMP1, VALUE(1), 1 },
    { { SENSORS_W83781D_TEMP2, "temp2", NOMAP, NOMAP, R }, 
                               W83781D_SYSCTL_TEMP2, VALUE(3), 1 },
    { { SENSORS_W83781D_TEMP2_HYST, "temp2_hyst", SENSORS_W83781D_TEMP2, 
                                    SENSORS_W83781D_TEMP2, RW }, 
                                    W83781D_SYSCTL_TEMP2, VALUE(2), 1 },
    { { SENSORS_W83781D_TEMP2_OVER, "temp2_over", SENSORS_W83781D_TEMP2, 
                                    SENSORS_W83781D_TEMP2, RW }, 
                                    W83781D_SYSCTL_TEMP2, VALUE(1), 1 },
    { { SENSORS_W83781D_TEMP3, "temp3", NOMAP, NOMAP, R }, 
                               W83781D_SYSCTL_TEMP3, VALUE(3), 1 },
    { { SENSORS_W83781D_TEMP3_HYST, "temp3_hyst", SENSORS_W83781D_TEMP3, 
                                    SENSORS_W83781D_TEMP3, RW }, 
                                    W83781D_SYSCTL_TEMP3, VALUE(2), 1 },
    { { SENSORS_W83781D_TEMP3_OVER, "temp3_over", SENSORS_W83781D_TEMP3, 
                                    SENSORS_W83781D_TEMP3, RW }, 
                                    W83781D_SYSCTL_TEMP3, VALUE(1), 1 },
    { { SENSORS_W83781D_VID, "vid", NOMAP, NOMAP, R }, 
                             W83781D_SYSCTL_VID, VALUE(1), 3 },
    { { SENSORS_W83781D_VRM, "vrm", NOMAP, NOMAP, RW }, 
                             W83781D_SYSCTL_VRM, VALUE(1), 1 },
    { { SENSORS_W83781D_FAN1_DIV, "fan1_div", SENSORS_W83781D_FAN1, NOMAP, RW }, 
                                  W83781D_SYSCTL_FAN_DIV, VALUE(1), 0 },
    { { SENSORS_W83781D_FAN2_DIV, "fan2_div", SENSORS_W83781D_FAN2, NOMAP, RW }, 
                                  W83781D_SYSCTL_FAN_DIV, VALUE(2), 0 },
    { { SENSORS_W83781D_FAN3_DIV, "fan3_div", SENSORS_W83781D_FAN3, NOMAP, RW }, 
                                  W83781D_SYSCTL_FAN_DIV, VALUE(3), 0 },
    { { SENSORS_W83781D_ALARMS, "alarms", NOMAP, NOMAP, R }, 
                                W83781D_SYSCTL_ALARMS, VALUE(1), 0 },
    { { SENSORS_W83781D_BEEP_ENABLE, "beep_enable", SENSORS_W83781D_ALARMS, 
                                     NOMAP, RW }, 
                                     W83781D_SYSCTL_BEEP, VALUE(1), 0 },
    { { SENSORS_W83781D_BEEPS, "beeps", SENSORS_W83781D_ALARMS, 
                               SENSORS_W83781D_ALARMS, RW }, 
                               W83781D_SYSCTL_BEEP, VALUE(2), 0 },
    { { 0 }, 0 }
  };
  
static sensors_chip_feature as99127f_features[] =
  { 
    { { SENSORS_W83782D_IN0, "in0", NOMAP, NOMAP, R }, 
                             W83781D_SYSCTL_IN0, VALUE(3), 2 },
    { { SENSORS_W83782D_IN1, "in1", NOMAP, NOMAP, R }, 
                             W83781D_SYSCTL_IN1, VALUE(3), 2 },
    { { SENSORS_W83782D_IN2, "in2", NOMAP, NOMAP, R }, 
                             W83781D_SYSCTL_IN2, VALUE(3), 2 },
    { { SENSORS_W83782D_IN3, "in3", NOMAP, NOMAP, R }, 
                             W83781D_SYSCTL_IN3, VALUE(3), 2 },
    { { SENSORS_W83782D_IN4, "in4", NOMAP, NOMAP, R }, 
                             W83781D_SYSCTL_IN4, VALUE(3), 2 },
    { { SENSORS_W83782D_IN5, "in5", NOMAP, NOMAP, R }, 
                             W83781D_SYSCTL_IN5, VALUE(3), 2 },
    { { SENSORS_W83782D_IN6, "in6", NOMAP, NOMAP, R }, 
                             W83781D_SYSCTL_IN6, VALUE(3), 2 },
    { { SENSORS_W83782D_IN0_MIN, "in0_min", SENSORS_W83782D_IN0, 
                                 SENSORS_W83782D_IN0, RW }, 
                                 W83781D_SYSCTL_IN0, VALUE(1), 2 },
    { { SENSORS_W83782D_IN1_MIN, "in1_min", SENSORS_W83782D_IN1, 
                                 SENSORS_W83782D_IN1, RW }, 
                                 W83781D_SYSCTL_IN1, VALUE(1), 2 },
    { { SENSORS_W83782D_IN2_MIN, "in2_min", SENSORS_W83782D_IN2, 
                                 SENSORS_W83782D_IN2, RW }, 
                                 W83781D_SYSCTL_IN2, VALUE(1), 2 },
    { { SENSORS_W83782D_IN3_MIN, "in3_min", SENSORS_W83782D_IN3, 
                                 SENSORS_W83782D_IN3, RW }, 
                                 W83781D_SYSCTL_IN3, VALUE(1), 2 },
    { { SENSORS_W83782D_IN4_MIN, "in4_min", SENSORS_W83782D_IN4, 
                                 SENSORS_W83782D_IN4, RW }, 
                                 W83781D_SYSCTL_IN4, VALUE(1), 2 },
/* swap 5 and 6 min/max because compute lines will be inverting */
    { { SENSORS_W83782D_IN5_MIN, "in5_min", SENSORS_W83782D_IN5, 
                                 SENSORS_W83782D_IN5, RW }, 
                                 W83781D_SYSCTL_IN5, VALUE(2), 2, "in5_max", 3 },
    { { SENSORS_W83782D_IN6_MIN, "in6_min", SENSORS_W83782D_IN6, 
                                 SENSORS_W83782D_IN6, RW }, 
                                 W83781D_SYSCTL_IN6, VALUE(2), 2, "in6_max", 3 },
    { { SENSORS_W83782D_IN0_MAX, "in0_max", SENSORS_W83782D_IN0, 
                                 SENSORS_W83782D_IN0, RW }, 
                                 W83781D_SYSCTL_IN0, VALUE(2), 2 },
    { { SENSORS_W83782D_IN1_MAX, "in1_max", SENSORS_W83782D_IN1, 
                                 SENSORS_W83782D_IN1, RW }, 
                                 W83781D_SYSCTL_IN1, VALUE(2), 2 },
    { { SENSORS_W83782D_IN2_MAX, "in2_max", SENSORS_W83782D_IN2, 
                                 SENSORS_W83782D_IN2, RW }, 
                                 W83781D_SYSCTL_IN2, VALUE(2), 2 },
    { { SENSORS_W83782D_IN3_MAX, "in3_max", SENSORS_W83782D_IN3, 
                                 SENSORS_W83782D_IN3, RW }, 
                                 W83781D_SYSCTL_IN3, VALUE(2), 2 },
    { { SENSORS_W83782D_IN4_MAX, "in4_max", SENSORS_W83782D_IN4, 
                                 SENSORS_W83782D_IN4, RW }, 
                                 W83781D_SYSCTL_IN4, VALUE(2), 2 },
    { { SENSORS_W83782D_IN5_MAX, "in5_max", SENSORS_W83782D_IN5, 
                                 SENSORS_W83782D_IN5, RW }, 
                                 W83781D_SYSCTL_IN5, VALUE(1), 2, "in5_min", 3 },
    { { SENSORS_W83782D_IN6_MAX, "in6_max", SENSORS_W83782D_IN6, 
                                 SENSORS_W83782D_IN6, RW }, 
                                 W83781D_SYSCTL_IN6, VALUE(1), 2, "in6_min", 3 },
    { { SENSORS_W83782D_FAN1, "fan1", NOMAP, NOMAP, R }, 
                              W83781D_SYSCTL_FAN1, VALUE(2), 0 },
    { { SENSORS_W83782D_FAN2, "fan2", NOMAP, NOMAP, R }, 
                              W83781D_SYSCTL_FAN2, VALUE(2), 0 },
    { { SENSORS_W83782D_FAN3, "fan3", NOMAP, NOMAP, R }, 
                              W83781D_SYSCTL_FAN3, VALUE(2), 0 },
    { { SENSORS_W83782D_FAN1_MIN, "fan1_min", SENSORS_W83782D_FAN1, 
                                  SENSORS_W83782D_FAN1, RW }, 
                                  W83781D_SYSCTL_FAN1, VALUE(1), 0 },
    { { SENSORS_W83782D_FAN2_MIN, "fan2_min", SENSORS_W83782D_FAN2, 
                                  SENSORS_W83782D_FAN2, RW }, 
                                  W83781D_SYSCTL_FAN2, VALUE(1), 0 },
    { { SENSORS_W83782D_FAN3_MIN, "fan3_min", SENSORS_W83782D_FAN3, 
                                  SENSORS_W83782D_FAN3, RW }, 
                                  W83781D_SYSCTL_FAN3, VALUE(1), 0 },
    { { SENSORS_W83782D_TEMP1, "temp1", NOMAP, NOMAP, R }, 
                               W83781D_SYSCTL_TEMP1, VALUE(3), 1 },
    { { SENSORS_W83782D_TEMP1_HYST, "temp1_hyst", SENSORS_W83782D_TEMP1, 
                                    SENSORS_W83782D_TEMP1, RW }, 
                                    W83781D_SYSCTL_TEMP1, VALUE(2), 1 },
    { { SENSORS_W83782D_TEMP1_OVER, "temp1_over", SENSORS_W83782D_TEMP1, 
                                    SENSORS_W83782D_TEMP1, RW }, 
                                    W83781D_SYSCTL_TEMP1, VALUE(1), 1 },
    { { SENSORS_W83782D_TEMP2, "temp2", NOMAP, NOMAP, R }, 
                               W83781D_SYSCTL_TEMP2, VALUE(3), 1 },
    { { SENSORS_W83782D_TEMP2_HYST, "temp2_hyst", SENSORS_W83782D_TEMP2, 
                                    SENSORS_W83782D_TEMP2, RW }, 
                                    W83781D_SYSCTL_TEMP2, VALUE(2), 1 },
    { { SENSORS_W83782D_TEMP2_OVER, "temp2_over", SENSORS_W83782D_TEMP2, 
                                    SENSORS_W83782D_TEMP2, RW }, 
                                    W83781D_SYSCTL_TEMP2, VALUE(1), 1 },
    { { SENSORS_W83782D_TEMP3, "temp3", NOMAP, NOMAP, R }, 
                               W83781D_SYSCTL_TEMP3, VALUE(3), 1 },
    { { SENSORS_W83782D_TEMP3_HYST, "temp3_hyst", SENSORS_W83782D_TEMP3, 
                                    SENSORS_W83782D_TEMP3, RW }, 
                                    W83781D_SYSCTL_TEMP3, VALUE(2), 1 },
    { { SENSORS_W83782D_TEMP3_OVER, "temp3_over", SENSORS_W83782D_TEMP3, 
                                    SENSORS_W83782D_TEMP3, RW }, 
                                    W83781D_SYSCTL_TEMP3, VALUE(1), 1 },
    { { SENSORS_W83782D_VID, "vid", NOMAP, NOMAP, R }, 
                             W83781D_SYSCTL_VID, VALUE(1), 3 },
    { { SENSORS_W83782D_VRM, "vrm", NOMAP, NOMAP, RW }, 
                             W83781D_SYSCTL_VRM, VALUE(1), 1 },
    { { SENSORS_W83782D_FAN1_DIV, "fan1_div", SENSORS_W83782D_FAN1, NOMAP, RW }, 
                                  W83781D_SYSCTL_FAN_DIV, VALUE(1), 0 },
    { { SENSORS_W83782D_FAN2_DIV, "fan2_div", SENSORS_W83782D_FAN2, NOMAP, RW }, 
                                  W83781D_SYSCTL_FAN_DIV, VALUE(2), 0 },
    { { SENSORS_W83782D_FAN3_DIV, "fan3_div", SENSORS_W83782D_FAN3, NOMAP, RW }, 
                                  W83781D_SYSCTL_FAN_DIV, VALUE(3), 0 },
    { { SENSORS_W83782D_ALARMS, "alarms", NOMAP, NOMAP, R }, 
                                W83781D_SYSCTL_ALARMS, VALUE(1), 0 },
    { { SENSORS_W83782D_BEEP_ENABLE, "beep_enable", SENSORS_W83782D_ALARMS, 
                                     NOMAP, RW }, 
                                     W83781D_SYSCTL_BEEP, VALUE(1), 0 },
    { { SENSORS_W83782D_BEEPS, "beeps", SENSORS_W83782D_ALARMS, 
                               SENSORS_W83782D_ALARMS, RW }, 
                               W83781D_SYSCTL_BEEP, VALUE(2), 0 },
    { { 0 }, 0 }
  };

/* macro for LM93 voltage in/min/max entries */
#define SENSORS_LM93_IN_MIN_MAX(nr) \
	{ { SENSORS_LM93_IN##nr, "in" #nr, NOMAP, \
		NOMAP, R }, \
		LM93_SYSCTL_IN##nr, VALUE(3), 2}, \
	{ { SENSORS_LM93_IN##nr##_MIN, "in" #nr "_min", SENSORS_LM93_IN##nr, \
		SENSORS_LM93_IN##nr, RW }, \
		LM93_SYSCTL_IN##nr, VALUE(1), 2 }, \
	{ { SENSORS_LM93_IN##nr##_MAX, "in" #nr "_max", SENSORS_LM93_IN##nr, \
		SENSORS_LM93_IN##nr, RW }, \
		LM93_SYSCTL_IN##nr, VALUE(2), 2 }

/* macro for LM93 fan/min entries */
#define SENSORS_LM93_FAN_MIN(nr) \
	{ { SENSORS_LM93_FAN##nr, "fan" #nr, NOMAP, NOMAP, R }, \
		LM93_SYSCTL_FAN##nr, VALUE(2), 0 }, \
	{ { SENSORS_LM93_FAN##nr##_MIN, "fan" #nr "_min", \
		SENSORS_LM93_FAN##nr, SENSORS_LM93_FAN##nr, RW }, \
	        LM93_SYSCTL_FAN##nr, VALUE(1), 0 }

/* macro for LM93 temp/min/max entries */
#define SENSORS_LM93_TEMP_MIN_MAX(nr) \
	{ { SENSORS_LM93_TEMP##nr, "temp" #nr, NOMAP, NOMAP, R }, \
		LM93_SYSCTL_TEMP##nr, VALUE(3), 1 }, \
	{ { SENSORS_LM93_TEMP##nr##_MAX, "temp" #nr "_max", \
		SENSORS_LM93_TEMP##nr, SENSORS_LM93_TEMP##nr, RW }, \
		LM93_SYSCTL_TEMP##nr, VALUE(1), 1 }, \
	{ { SENSORS_LM93_TEMP##nr##_MIN, "temp" #nr "_min", \
		SENSORS_LM93_TEMP##nr, SENSORS_LM93_TEMP##nr, RW }, \
		LM93_SYSCTL_TEMP##nr, VALUE(2), 1 }

static sensors_chip_feature lm93_features[] = {
	SENSORS_LM93_IN_MIN_MAX(1),
	SENSORS_LM93_IN_MIN_MAX(2),
	SENSORS_LM93_IN_MIN_MAX(3),
	SENSORS_LM93_IN_MIN_MAX(4),
	SENSORS_LM93_IN_MIN_MAX(5),
	SENSORS_LM93_IN_MIN_MAX(6),
	SENSORS_LM93_IN_MIN_MAX(7),
	SENSORS_LM93_IN_MIN_MAX(8),
	SENSORS_LM93_IN_MIN_MAX(9),
	SENSORS_LM93_IN_MIN_MAX(10),
	SENSORS_LM93_IN_MIN_MAX(11),
	SENSORS_LM93_IN_MIN_MAX(12),
	SENSORS_LM93_IN_MIN_MAX(13),
	SENSORS_LM93_IN_MIN_MAX(14),
	SENSORS_LM93_IN_MIN_MAX(15),
	SENSORS_LM93_IN_MIN_MAX(16),
	SENSORS_LM93_FAN_MIN(1),
	SENSORS_LM93_FAN_MIN(2),
	SENSORS_LM93_FAN_MIN(3),
	SENSORS_LM93_FAN_MIN(4),
	SENSORS_LM93_TEMP_MIN_MAX(1),
	SENSORS_LM93_TEMP_MIN_MAX(2),
	SENSORS_LM93_TEMP_MIN_MAX(3),
    { { SENSORS_LM93_VID1, "vid1", NOMAP, NOMAP, R },
                          LM93_SYSCTL_VID1, VALUE(1), 3, "cpu0_vid", 3, "vid1" },
    { { SENSORS_LM93_VID2, "vid2", NOMAP, NOMAP, R },
                          LM93_SYSCTL_VID2, VALUE(1), 3, "cpu1_vid", 3, "vid2" },
    { { SENSORS_LM93_ALARMS, "alarms", NOMAP, NOMAP, R }, 
                          LM93_SYSCTL_ALARMS, VALUE(1), 0 },
    { { 0 }, 0 }
};

/* macro for ASB100 Bach voltage in/min/max entries */
#define SENSORS_ASB100_IN_MIN_MAX(nr) \
	{ { SENSORS_ASB100_IN##nr, "in" #nr, NOMAP, NOMAP, R }, \
		ASB100_SYSCTL_IN##nr, \
		VALUE(3), 2 }, \
	{ { SENSORS_ASB100_IN##nr##_MIN, "in" #nr "_min", SENSORS_ASB100_IN##nr, \
		SENSORS_ASB100_IN##nr, RW }, \
		ASB100_SYSCTL_IN##nr, VALUE(1), 2 }, \
	{ { SENSORS_ASB100_IN##nr##_MAX, "in" #nr "_max", SENSORS_ASB100_IN##nr, \
		SENSORS_ASB100_IN##nr, RW }, \
		ASB100_SYSCTL_IN##nr, VALUE(2), 2 }

/* macro for ASB100 Bach fan/min/div entries */
#define SENSORS_ASB100_FAN_MIN_DIV(nr, divmode) \
	{ { SENSORS_ASB100_FAN##nr, "fan" #nr, NOMAP, \
		NOMAP, R}, \
		ASB100_SYSCTL_FAN##nr, VALUE(2), 0 }, \
	{ { SENSORS_ASB100_FAN##nr##_MIN, "fan" #nr "_min", \
		SENSORS_ASB100_FAN##nr, SENSORS_ASB100_FAN##nr, RW }, \
		ASB100_SYSCTL_FAN##nr, VALUE(1), 0 }, \
	{ { SENSORS_ASB100_FAN##nr##_DIV, "fan" #nr "_div", \
		SENSORS_ASB100_FAN##nr,  NOMAP, divmode }, \
		ASB100_SYSCTL_FAN_DIV, VALUE(nr), 0 }

/* macro for ASB100 Bach temperature/overtemp/hysteresis entries */
#define SENSORS_ASB100_TEMP_OVER_HYST(nr) \
	{ { SENSORS_ASB100_TEMP##nr, "temp" #nr, NOMAP, \
		NOMAP, R }, \
		ASB100_SYSCTL_TEMP##nr, VALUE(3), 1 }, \
	{ { SENSORS_ASB100_TEMP##nr##_OVER, "temp" #nr "_over", \
		SENSORS_ASB100_TEMP##nr, SENSORS_ASB100_TEMP##nr, RW }, \
		ASB100_SYSCTL_TEMP##nr, VALUE(1), 1 }, \
	{ { SENSORS_ASB100_TEMP##nr##_HYST, "temp" #nr "_hyst", \
		SENSORS_ASB100_TEMP##nr, SENSORS_ASB100_TEMP##nr, RW }, \
		ASB100_SYSCTL_TEMP##nr, VALUE(2), 1 }

static sensors_chip_feature asb100_features[] = { 
	SENSORS_ASB100_IN_MIN_MAX(0),
	SENSORS_ASB100_IN_MIN_MAX(1),
	SENSORS_ASB100_IN_MIN_MAX(2),
	SENSORS_ASB100_IN_MIN_MAX(3),
	SENSORS_ASB100_IN_MIN_MAX(4),
	SENSORS_ASB100_IN_MIN_MAX(5),
	SENSORS_ASB100_IN_MIN_MAX(6),
	SENSORS_ASB100_FAN_MIN_DIV(1, RW),
	SENSORS_ASB100_FAN_MIN_DIV(2, RW),
	SENSORS_ASB100_FAN_MIN_DIV(3, RW),
	SENSORS_ASB100_TEMP_OVER_HYST(1),
	SENSORS_ASB100_TEMP_OVER_HYST(2),
	SENSORS_ASB100_TEMP_OVER_HYST(3),
	SENSORS_ASB100_TEMP_OVER_HYST(4),
    { { SENSORS_ASB100_VID, "vid", NOMAP, NOMAP, R }, 
                         ASB100_SYSCTL_VID, VALUE(1), 3 },
    { { SENSORS_ASB100_VRM, "vrm", NOMAP, NOMAP, RW }, 
                         ASB100_SYSCTL_VRM, VALUE(1), 1 },
    { { SENSORS_ASB100_ALARMS, "alarms", NOMAP, NOMAP, R }, 
                            ASB100_SYSCTL_ALARMS, VALUE(1), 0 },
    { { 0 }, 0 }
};
  
static sensors_chip_feature w83782d_features[] =
  { 
    { { SENSORS_W83782D_IN0, "in0", NOMAP, NOMAP, R }, 
                             W83781D_SYSCTL_IN0, VALUE(3), 2 },
    { { SENSORS_W83782D_IN1, "in1", NOMAP, NOMAP, R }, 
                             W83781D_SYSCTL_IN1, VALUE(3), 2 },
    { { SENSORS_W83782D_IN2, "in2", NOMAP, NOMAP, R }, 
                             W83781D_SYSCTL_IN2, VALUE(3), 2 },
    { { SENSORS_W83782D_IN3, "in3", NOMAP, NOMAP, R }, 
                             W83781D_SYSCTL_IN3, VALUE(3), 2 },
    { { SENSORS_W83782D_IN4, "in4", NOMAP, NOMAP, R }, 
                             W83781D_SYSCTL_IN4, VALUE(3), 2 },
    { { SENSORS_W83782D_IN5, "in5", NOMAP, NOMAP, R }, 
                             W83781D_SYSCTL_IN5, VALUE(3), 2 },
    { { SENSORS_W83782D_IN6, "in6", NOMAP, NOMAP, R }, 
                             W83781D_SYSCTL_IN6, VALUE(3), 2 },
    { { SENSORS_W83782D_IN7, "in7", NOMAP, NOMAP, R }, 
                             W83781D_SYSCTL_IN7, VALUE(3), 2 },
    { { SENSORS_W83782D_IN8, "in8", NOMAP, NOMAP, R }, 
                             W83781D_SYSCTL_IN8, VALUE(3), 2 },
    { { SENSORS_W83782D_IN0_MIN, "in0_min", SENSORS_W83782D_IN0, 
                                 SENSORS_W83782D_IN0, RW }, 
                                 W83781D_SYSCTL_IN0, VALUE(1), 2 },
    { { SENSORS_W83782D_IN1_MIN, "in1_min", SENSORS_W83782D_IN1, 
                                 SENSORS_W83782D_IN1, RW }, 
                                 W83781D_SYSCTL_IN1, VALUE(1), 2 },
    { { SENSORS_W83782D_IN2_MIN, "in2_min", SENSORS_W83782D_IN2, 
                                 SENSORS_W83782D_IN2, RW }, 
                                 W83781D_SYSCTL_IN2, VALUE(1), 2 },
    { { SENSORS_W83782D_IN3_MIN, "in3_min", SENSORS_W83782D_IN3, 
                                 SENSORS_W83782D_IN3, RW }, 
                                 W83781D_SYSCTL_IN3, VALUE(1), 2 },
    { { SENSORS_W83782D_IN4_MIN, "in4_min", SENSORS_W83782D_IN4, 
                                 SENSORS_W83782D_IN4, RW }, 
                                 W83781D_SYSCTL_IN4, VALUE(1), 2 },
    { { SENSORS_W83782D_IN5_MIN, "in5_min", SENSORS_W83782D_IN5, 
                                 SENSORS_W83782D_IN5, RW }, 
                                 W83781D_SYSCTL_IN5, VALUE(1), 2 },
    { { SENSORS_W83782D_IN6_MIN, "in6_min", SENSORS_W83782D_IN6, 
                                 SENSORS_W83782D_IN6, RW }, 
                                 W83781D_SYSCTL_IN6, VALUE(1), 2 },
    { { SENSORS_W83782D_IN7_MIN, "in7_min", SENSORS_W83782D_IN7, 
                                 SENSORS_W83782D_IN7, RW }, 
                                 W83781D_SYSCTL_IN7, VALUE(1), 2 },
    { { SENSORS_W83782D_IN8_MIN, "in8_min", SENSORS_W83782D_IN8, 
                                 SENSORS_W83782D_IN8, RW }, 
                                 W83781D_SYSCTL_IN8, VALUE(1), 2 },
    { { SENSORS_W83782D_IN0_MAX, "in0_max", SENSORS_W83782D_IN0, 
                                 SENSORS_W83782D_IN0, RW }, 
                                 W83781D_SYSCTL_IN0, VALUE(2), 2 },
    { { SENSORS_W83782D_IN1_MAX, "in1_max", SENSORS_W83782D_IN1, 
                                 SENSORS_W83782D_IN1, RW }, 
                                 W83781D_SYSCTL_IN1, VALUE(2), 2 },
    { { SENSORS_W83782D_IN2_MAX, "in2_max", SENSORS_W83782D_IN2, 
                                 SENSORS_W83782D_IN2, RW }, 
                                 W83781D_SYSCTL_IN2, VALUE(2), 2 },
    { { SENSORS_W83782D_IN3_MAX, "in3_max", SENSORS_W83782D_IN3, 
                                 SENSORS_W83782D_IN3, RW }, 
                                 W83781D_SYSCTL_IN3, VALUE(2), 2 },
    { { SENSORS_W83782D_IN4_MAX, "in4_max", SENSORS_W83782D_IN4, 
                                 SENSORS_W83782D_IN4, RW }, 
                                 W83781D_SYSCTL_IN4, VALUE(2), 2 },
    { { SENSORS_W83782D_IN5_MAX, "in5_max", SENSORS_W83782D_IN5, 
                                 SENSORS_W83782D_IN5, RW }, 
                                 W83781D_SYSCTL_IN5, VALUE(2), 2 },
    { { SENSORS_W83782D_IN6_MAX, "in6_max", SENSORS_W83782D_IN6, 
                                 SENSORS_W83782D_IN6, RW }, 
                                 W83781D_SYSCTL_IN6, VALUE(2), 2 },
    { { SENSORS_W83782D_IN7_MAX, "in7_max", SENSORS_W83782D_IN7, 
                                 SENSORS_W83782D_IN7, RW }, 
                                 W83781D_SYSCTL_IN7, VALUE(2), 2 },
    { { SENSORS_W83782D_IN8_MAX, "in8_max", SENSORS_W83782D_IN8, 
                                 SENSORS_W83782D_IN8, RW }, 
                                 W83781D_SYSCTL_IN8, VALUE(2), 2 },
    { { SENSORS_W83782D_FAN1, "fan1", NOMAP, NOMAP, R }, 
                              W83781D_SYSCTL_FAN1, VALUE(2), 0 },
    { { SENSORS_W83782D_FAN2, "fan2", NOMAP, NOMAP, R }, 
                              W83781D_SYSCTL_FAN2, VALUE(2), 0 },
    { { SENSORS_W83782D_FAN3, "fan3", NOMAP, NOMAP, R }, 
                              W83781D_SYSCTL_FAN3, VALUE(2), 0 },
    { { SENSORS_W83782D_FAN1_MIN, "fan1_min", SENSORS_W83782D_FAN1, 
                                  SENSORS_W83782D_FAN1, RW }, 
                                  W83781D_SYSCTL_FAN1, VALUE(1), 0 },
    { { SENSORS_W83782D_FAN2_MIN, "fan2_min", SENSORS_W83782D_FAN2, 
                                  SENSORS_W83782D_FAN2, RW }, 
                                  W83781D_SYSCTL_FAN2, VALUE(1), 0 },
    { { SENSORS_W83782D_FAN3_MIN, "fan3_min", SENSORS_W83782D_FAN3, 
                                  SENSORS_W83782D_FAN3, RW }, 
                                  W83781D_SYSCTL_FAN3, VALUE(1), 0 },
    { { SENSORS_W83782D_TEMP1, "temp1", NOMAP, NOMAP, R }, 
                               W83781D_SYSCTL_TEMP1, VALUE(3), 1 },
    { { SENSORS_W83782D_TEMP1_HYST, "temp1_hyst", SENSORS_W83782D_TEMP1, 
                                    SENSORS_W83782D_TEMP1, RW }, 
                                    W83781D_SYSCTL_TEMP1, VALUE(2), 1 },
    { { SENSORS_W83782D_TEMP1_OVER, "temp1_over", SENSORS_W83782D_TEMP1, 
                                    SENSORS_W83782D_TEMP1, RW }, 
                                    W83781D_SYSCTL_TEMP1, VALUE(1), 1 },
    { { SENSORS_W83782D_TEMP2, "temp2", NOMAP, NOMAP, R }, 
                               W83781D_SYSCTL_TEMP2, VALUE(3), 1 },
    { { SENSORS_W83782D_TEMP2_HYST, "temp2_hyst", SENSORS_W83782D_TEMP2, 
                                    SENSORS_W83782D_TEMP2, RW }, 
                                    W83781D_SYSCTL_TEMP2, VALUE(2), 1 },
    { { SENSORS_W83782D_TEMP2_OVER, "temp2_over", SENSORS_W83782D_TEMP2, 
                                    SENSORS_W83782D_TEMP2, RW }, 
                                    W83781D_SYSCTL_TEMP2, VALUE(1), 1 },
    { { SENSORS_W83782D_TEMP3, "temp3", NOMAP, NOMAP, R }, 
                               W83781D_SYSCTL_TEMP3, VALUE(3), 1 },
    { { SENSORS_W83782D_TEMP3_HYST, "temp3_hyst", SENSORS_W83782D_TEMP3, 
                                    SENSORS_W83782D_TEMP3, RW }, 
                                    W83781D_SYSCTL_TEMP3, VALUE(2), 1 },
    { { SENSORS_W83782D_TEMP3_OVER, "temp3_over", SENSORS_W83782D_TEMP3, 
                                    SENSORS_W83782D_TEMP3, RW }, 
                                    W83781D_SYSCTL_TEMP3, VALUE(1), 1 },
    { { SENSORS_W83782D_VID, "vid", NOMAP, NOMAP, R }, 
                             W83781D_SYSCTL_VID, VALUE(1), 3 },
    { { SENSORS_W83782D_VRM, "vrm", NOMAP, NOMAP, RW }, 
                             W83781D_SYSCTL_VRM, VALUE(1), 1 },
    { { SENSORS_W83782D_FAN1_DIV, "fan1_div", SENSORS_W83782D_FAN1, NOMAP, RW }, 
                                  W83781D_SYSCTL_FAN_DIV, VALUE(1), 0 },
    { { SENSORS_W83782D_FAN2_DIV, "fan2_div", SENSORS_W83782D_FAN2, NOMAP, RW }, 
                                  W83781D_SYSCTL_FAN_DIV, VALUE(2), 0 },
    { { SENSORS_W83782D_FAN3_DIV, "fan3_div", SENSORS_W83782D_FAN3, NOMAP, RW }, 
                                  W83781D_SYSCTL_FAN_DIV, VALUE(3), 0 },
    { { SENSORS_W83782D_ALARMS, "alarms", NOMAP, NOMAP, R }, 
                                W83781D_SYSCTL_ALARMS, VALUE(1), 0 },
    { { SENSORS_W83782D_BEEP_ENABLE, "beep_enable", SENSORS_W83782D_ALARMS, 
                                     NOMAP, RW }, 
                                     W83781D_SYSCTL_BEEP, VALUE(1), 0 },
    { { SENSORS_W83782D_BEEPS, "beeps", SENSORS_W83782D_ALARMS, 
                               SENSORS_W83782D_ALARMS, RW }, 
                               W83781D_SYSCTL_BEEP, VALUE(2), 0 },
    { { SENSORS_W83782D_SENS1, "sensor1", SENSORS_W83782D_TEMP1, NOMAP, RW }, 
                               W83781D_SYSCTL_SENS1, VALUE(1), 0 },
    { { SENSORS_W83782D_SENS2, "sensor2", SENSORS_W83782D_TEMP2, NOMAP, RW }, 
                               W83781D_SYSCTL_SENS2, VALUE(1), 0 },
    { { SENSORS_W83782D_SENS3, "sensor3", SENSORS_W83782D_TEMP3, NOMAP, RW }, 
                               W83781D_SYSCTL_SENS3, VALUE(1), 0 },
    { { 0 }, 0 }
  };
  
static sensors_chip_feature w83783s_features[] =
  { 
    { { SENSORS_W83783S_IN0, "in0", NOMAP, NOMAP, R }, 
                             W83781D_SYSCTL_IN0, VALUE(3), 2 },
    { { SENSORS_W83783S_IN2, "in2", NOMAP, NOMAP, R }, 
                             W83781D_SYSCTL_IN2, VALUE(3), 2 },
    { { SENSORS_W83783S_IN3, "in3", NOMAP, NOMAP, R }, 
                             W83781D_SYSCTL_IN3, VALUE(3), 2 },
    { { SENSORS_W83783S_IN4, "in4", NOMAP, NOMAP, R }, 
                             W83781D_SYSCTL_IN4, VALUE(3), 2 },
    { { SENSORS_W83783S_IN5, "in5", NOMAP, NOMAP, R }, 
                             W83781D_SYSCTL_IN5, VALUE(3), 2 },
    { { SENSORS_W83783S_IN6, "in6", NOMAP, NOMAP, R }, 
                             W83781D_SYSCTL_IN6, VALUE(3), 2 },
    { { SENSORS_W83783S_IN0_MIN, "in0_min", SENSORS_W83783S_IN0, 
                                 SENSORS_W83783S_IN0, RW }, 
                                 W83781D_SYSCTL_IN0, VALUE(1), 2 },
    { { SENSORS_W83783S_IN2_MIN, "in2_min", SENSORS_W83783S_IN2, 
                                 SENSORS_W83783S_IN2, RW }, 
                                 W83781D_SYSCTL_IN2, VALUE(1), 2 },
    { { SENSORS_W83783S_IN3_MIN, "in3_min", SENSORS_W83783S_IN3, 
                                 SENSORS_W83783S_IN3, RW }, 
                                 W83781D_SYSCTL_IN3, VALUE(1), 2 },
    { { SENSORS_W83783S_IN4_MIN, "in4_min", SENSORS_W83783S_IN4, 
                                 SENSORS_W83783S_IN4, RW }, 
                                 W83781D_SYSCTL_IN4, VALUE(1), 2 },
    { { SENSORS_W83783S_IN5_MIN, "in5_min", SENSORS_W83783S_IN5, 
                                 SENSORS_W83783S_IN5, RW }, 
                                 W83781D_SYSCTL_IN5, VALUE(1), 2 },
    { { SENSORS_W83783S_IN6_MIN, "in6_min", SENSORS_W83783S_IN6, 
                                 SENSORS_W83783S_IN6, RW }, 
                                 W83781D_SYSCTL_IN6, VALUE(1), 2 },
    { { SENSORS_W83783S_IN0_MAX, "in0_max", SENSORS_W83783S_IN0, 
                                 SENSORS_W83783S_IN0, RW }, 
                                 W83781D_SYSCTL_IN0, VALUE(2), 2 },
    { { SENSORS_W83783S_IN2_MAX, "in2_max", SENSORS_W83783S_IN2, 
                                 SENSORS_W83783S_IN2, RW }, 
                                 W83781D_SYSCTL_IN2, VALUE(2), 2 },
    { { SENSORS_W83783S_IN3_MAX, "in3_max", SENSORS_W83783S_IN3, 
                                 SENSORS_W83783S_IN3, RW }, 
                                 W83781D_SYSCTL_IN3, VALUE(2), 2 },
    { { SENSORS_W83783S_IN4_MAX, "in4_max", SENSORS_W83783S_IN4, 
                                 SENSORS_W83783S_IN4, RW }, 
                                 W83781D_SYSCTL_IN4, VALUE(2), 2 },
    { { SENSORS_W83783S_IN5_MAX, "in5_max", SENSORS_W83783S_IN5, 
                                 SENSORS_W83783S_IN5, RW }, 
                                 W83781D_SYSCTL_IN5, VALUE(2), 2 },
    { { SENSORS_W83783S_IN6_MAX, "in6_max", SENSORS_W83783S_IN6, 
                                 SENSORS_W83783S_IN6, RW }, 
                                 W83781D_SYSCTL_IN6, VALUE(2), 2 },
    { { SENSORS_W83783S_FAN1, "fan1", NOMAP, NOMAP, R }, 
                              W83781D_SYSCTL_FAN1, VALUE(2), 0 },
    { { SENSORS_W83783S_FAN2, "fan2", NOMAP, NOMAP, R }, 
                              W83781D_SYSCTL_FAN2, VALUE(2), 0 },
    { { SENSORS_W83783S_FAN3, "fan3", NOMAP, NOMAP, R }, 
                              W83781D_SYSCTL_FAN3, VALUE(2), 0 },
    { { SENSORS_W83783S_FAN1_MIN, "fan1_min", SENSORS_W83783S_FAN1, 
                                  SENSORS_W83783S_FAN1, RW }, 
                                  W83781D_SYSCTL_FAN1, VALUE(1), 0 },
    { { SENSORS_W83783S_FAN2_MIN, "fan2_min", SENSORS_W83783S_FAN2, 
                                  SENSORS_W83783S_FAN2, RW }, 
                                  W83781D_SYSCTL_FAN2, VALUE(1), 0 },
    { { SENSORS_W83783S_FAN3_MIN, "fan3_min", SENSORS_W83783S_FAN3, 
                                  SENSORS_W83783S_FAN3, RW }, 
                                  W83781D_SYSCTL_FAN3, VALUE(1), 0 },
    { { SENSORS_W83783S_TEMP1, "temp1", NOMAP, NOMAP, R }, 
                               W83781D_SYSCTL_TEMP1, VALUE(3), 1 },
    { { SENSORS_W83783S_TEMP1_HYST, "temp1_hyst", SENSORS_W83783S_TEMP1, 
                                    SENSORS_W83783S_TEMP1, RW }, 
                                    W83781D_SYSCTL_TEMP1, VALUE(2), 1 },
    { { SENSORS_W83783S_TEMP1_OVER, "temp1_over", SENSORS_W83783S_TEMP1, 
                                    SENSORS_W83783S_TEMP1, RW }, 
                                    W83781D_SYSCTL_TEMP1, VALUE(1), 1 },
    { { SENSORS_W83783S_TEMP2, "temp2", NOMAP, NOMAP, R }, 
                               W83781D_SYSCTL_TEMP2, VALUE(3), 1 },
    { { SENSORS_W83783S_TEMP2_HYST, "temp2_hyst", SENSORS_W83783S_TEMP2, 
                                    SENSORS_W83783S_TEMP2, RW }, 
                                    W83781D_SYSCTL_TEMP2, VALUE(2), 1 },
    { { SENSORS_W83783S_TEMP2_OVER, "temp2_over", SENSORS_W83783S_TEMP2, 
                                    SENSORS_W83783S_TEMP2, RW }, 
                                    W83781D_SYSCTL_TEMP2, VALUE(1), 1 },
    { { SENSORS_W83783S_VID, "vid", NOMAP, NOMAP, R }, 
                             W83781D_SYSCTL_VID, VALUE(1), 3 },
    { { SENSORS_W83783S_VRM, "vrm", NOMAP, NOMAP, RW }, 
                             W83781D_SYSCTL_VRM, VALUE(1), 1 },
    { { SENSORS_W83783S_FAN1_DIV, "fan1_div", SENSORS_W83783S_FAN1, NOMAP, RW }, 
                                  W83781D_SYSCTL_FAN_DIV, VALUE(1), 0 },
    { { SENSORS_W83783S_FAN2_DIV, "fan2_div", SENSORS_W83783S_FAN2, NOMAP, RW }, 
                                  W83781D_SYSCTL_FAN_DIV, VALUE(2), 0 },
    { { SENSORS_W83783S_FAN3_DIV, "fan3_div", SENSORS_W83783S_FAN3, NOMAP, RW }, 
                                  W83781D_SYSCTL_FAN_DIV, VALUE(3), 0 },
    { { SENSORS_W83783S_ALARMS, "alarms", NOMAP, NOMAP, R }, 
                                W83781D_SYSCTL_ALARMS, VALUE(1), 0 },
    { { SENSORS_W83783S_BEEP_ENABLE, "beep_enable", SENSORS_W83783S_ALARMS, 
                                     NOMAP, RW }, 
                                     W83781D_SYSCTL_BEEP, VALUE(1), 0 },
    { { SENSORS_W83783S_BEEPS, "beeps", SENSORS_W83783S_ALARMS, 
                               SENSORS_W83783S_ALARMS, RW }, 
                               W83781D_SYSCTL_BEEP, VALUE(2), 0 },
    { { SENSORS_W83783S_SENS1, "sensor1", SENSORS_W83783S_TEMP1, NOMAP, RW }, 
                               W83781D_SYSCTL_SENS1, VALUE(1), 0 },
    { { SENSORS_W83783S_SENS2, "sensor2", SENSORS_W83783S_TEMP2, NOMAP, RW }, 
                               W83781D_SYSCTL_SENS2, VALUE(1), 0 },
    { { 0 }, 0 }
  };
  
static sensors_chip_feature w83791d_features[] =
  { 
    { { SENSORS_W83791D_IN0, "in0", NOMAP, NOMAP, R }, 
                             W83781D_SYSCTL_IN0, VALUE(3), 2 },
    { { SENSORS_W83791D_IN1, "in1", NOMAP, NOMAP, R }, 
                             W83781D_SYSCTL_IN1, VALUE(3), 2 },
    { { SENSORS_W83791D_IN2, "in2", NOMAP, NOMAP, R }, 
                             W83781D_SYSCTL_IN2, VALUE(3), 2 },
    { { SENSORS_W83791D_IN3, "in3", NOMAP, NOMAP, R }, 
                             W83781D_SYSCTL_IN3, VALUE(3), 2 },
    { { SENSORS_W83791D_IN4, "in4", NOMAP, NOMAP, R }, 
                             W83781D_SYSCTL_IN4, VALUE(3), 2 },
    { { SENSORS_W83791D_IN5, "in5", NOMAP, NOMAP, R }, 
                             W83781D_SYSCTL_IN5, VALUE(3), 2 },
    { { SENSORS_W83791D_IN6, "in6", NOMAP, NOMAP, R }, 
                             W83781D_SYSCTL_IN6, VALUE(3), 2 },
    { { SENSORS_W83791D_IN7, "in7", NOMAP, NOMAP, R }, 
                             W83781D_SYSCTL_IN7, VALUE(3), 2 },
    { { SENSORS_W83791D_IN8, "in8", NOMAP, NOMAP, R }, 
                             W83781D_SYSCTL_IN8, VALUE(3), 2 },
    { { SENSORS_W83791D_IN9, "in9", NOMAP, NOMAP, R }, 
                             W83781D_SYSCTL_IN9, VALUE(3), 2 },
    { { SENSORS_W83791D_IN0_MIN, "in0_min", SENSORS_W83791D_IN0, 
                                 SENSORS_W83791D_IN0, RW }, 
                                 W83781D_SYSCTL_IN0, VALUE(1), 2 },
    { { SENSORS_W83791D_IN1_MIN, "in1_min", SENSORS_W83791D_IN1, 
                                 SENSORS_W83791D_IN1, RW }, 
                                 W83781D_SYSCTL_IN1, VALUE(1), 2 },
    { { SENSORS_W83791D_IN2_MIN, "in2_min", SENSORS_W83791D_IN2, 
                                 SENSORS_W83791D_IN2, RW }, 
                                 W83781D_SYSCTL_IN2, VALUE(1), 2 },
    { { SENSORS_W83791D_IN3_MIN, "in3_min", SENSORS_W83791D_IN3, 
                                 SENSORS_W83791D_IN3, RW }, 
                                 W83781D_SYSCTL_IN3, VALUE(1), 2 },
    { { SENSORS_W83791D_IN4_MIN, "in4_min", SENSORS_W83791D_IN4, 
                                 SENSORS_W83791D_IN4, RW }, 
                                 W83781D_SYSCTL_IN4, VALUE(1), 2 },
    { { SENSORS_W83791D_IN5_MIN, "in5_min", SENSORS_W83791D_IN5, 
                                 SENSORS_W83791D_IN5, RW }, 
                                 W83781D_SYSCTL_IN5, VALUE(1), 2 },
    { { SENSORS_W83791D_IN6_MIN, "in6_min", SENSORS_W83791D_IN6, 
                                 SENSORS_W83791D_IN6, RW }, 
                                 W83781D_SYSCTL_IN6, VALUE(1), 2 },
    { { SENSORS_W83791D_IN7_MIN, "in7_min", SENSORS_W83791D_IN7, 
                                 SENSORS_W83791D_IN7, RW }, 
                                 W83781D_SYSCTL_IN7, VALUE(1), 2 },
    { { SENSORS_W83791D_IN8_MIN, "in8_min", SENSORS_W83791D_IN8, 
                                 SENSORS_W83791D_IN8, RW }, 
                                 W83781D_SYSCTL_IN8, VALUE(1), 2 },
    { { SENSORS_W83791D_IN9_MIN, "in9_min", SENSORS_W83791D_IN9, 
                                 SENSORS_W83791D_IN9, RW }, 
                                 W83781D_SYSCTL_IN9, VALUE(1), 2 },
    { { SENSORS_W83791D_IN0_MAX, "in0_max", SENSORS_W83791D_IN0, 
                                 SENSORS_W83791D_IN0, RW }, 
                                 W83781D_SYSCTL_IN0, VALUE(2), 2 },
    { { SENSORS_W83791D_IN1_MAX, "in1_max", SENSORS_W83791D_IN1, 
                                 SENSORS_W83791D_IN1, RW }, 
                                 W83781D_SYSCTL_IN1, VALUE(2), 2 },
    { { SENSORS_W83791D_IN2_MAX, "in2_max", SENSORS_W83791D_IN2, 
                                 SENSORS_W83791D_IN2, RW }, 
                                 W83781D_SYSCTL_IN2, VALUE(2), 2 },
    { { SENSORS_W83791D_IN3_MAX, "in3_max", SENSORS_W83791D_IN3, 
                                 SENSORS_W83791D_IN3, RW }, 
                                 W83781D_SYSCTL_IN3, VALUE(2), 2 },
    { { SENSORS_W83791D_IN4_MAX, "in4_max", SENSORS_W83791D_IN4, 
                                 SENSORS_W83791D_IN4, RW }, 
                                 W83781D_SYSCTL_IN4, VALUE(2), 2 },
    { { SENSORS_W83791D_IN5_MAX, "in5_max", SENSORS_W83791D_IN5, 
                                 SENSORS_W83791D_IN5, RW }, 
                                 W83781D_SYSCTL_IN5, VALUE(2), 2 },
    { { SENSORS_W83791D_IN6_MAX, "in6_max", SENSORS_W83791D_IN6, 
                                 SENSORS_W83791D_IN6, RW }, 
                                 W83781D_SYSCTL_IN6, VALUE(2), 2 },
    { { SENSORS_W83791D_IN7_MAX, "in7_max", SENSORS_W83791D_IN7, 
                                 SENSORS_W83791D_IN7, RW }, 
                                 W83781D_SYSCTL_IN7, VALUE(2), 2 },
    { { SENSORS_W83791D_IN8_MAX, "in8_max", SENSORS_W83791D_IN8, 
                                 SENSORS_W83791D_IN8, RW }, 
                                 W83781D_SYSCTL_IN8, VALUE(2), 2 },
    { { SENSORS_W83791D_IN9_MAX, "in9_max", SENSORS_W83791D_IN9, 
                                 SENSORS_W83791D_IN9, RW }, 
                                 W83781D_SYSCTL_IN9, VALUE(2), 2 },
    { { SENSORS_W83791D_FAN1, "fan1", NOMAP, NOMAP, R }, 
                              W83781D_SYSCTL_FAN1, VALUE(2), 0 },
    { { SENSORS_W83791D_FAN2, "fan2", NOMAP, NOMAP, R }, 
                              W83781D_SYSCTL_FAN2, VALUE(2), 0 },
    { { SENSORS_W83791D_FAN3, "fan3", NOMAP, NOMAP, R }, 
                              W83781D_SYSCTL_FAN3, VALUE(2), 0 },
    { { SENSORS_W83791D_FAN4, "fan4", NOMAP, NOMAP, R }, 
                              W83781D_SYSCTL_FAN4, VALUE(2), 0 },
    { { SENSORS_W83791D_FAN5, "fan5", NOMAP, NOMAP, R }, 
                              W83781D_SYSCTL_FAN5, VALUE(2), 0 },
    { { SENSORS_W83791D_FAN1_MIN, "fan1_min", SENSORS_W83791D_FAN1, 
                                  SENSORS_W83791D_FAN1, RW }, 
                                  W83781D_SYSCTL_FAN1, VALUE(1), 0 },
    { { SENSORS_W83791D_FAN2_MIN, "fan2_min", SENSORS_W83791D_FAN2, 
                                  SENSORS_W83791D_FAN2, RW }, 
                                  W83781D_SYSCTL_FAN2, VALUE(1), 0 },
    { { SENSORS_W83791D_FAN3_MIN, "fan3_min", SENSORS_W83791D_FAN3, 
                                  SENSORS_W83791D_FAN3, RW }, 
                                  W83781D_SYSCTL_FAN3, VALUE(1), 0 },
    { { SENSORS_W83791D_FAN4_MIN, "fan4_min", SENSORS_W83791D_FAN4, 
                                  SENSORS_W83791D_FAN4, RW }, 
                                  W83781D_SYSCTL_FAN4, VALUE(1), 0 },
    { { SENSORS_W83791D_FAN5_MIN, "fan5_min", SENSORS_W83791D_FAN5, 
                                  SENSORS_W83791D_FAN5, RW }, 
                                  W83781D_SYSCTL_FAN5, VALUE(1), 0 },
    { { SENSORS_W83791D_TEMP1, "temp1", NOMAP, NOMAP, R }, 
                               W83781D_SYSCTL_TEMP1, VALUE(3), 1 },
    { { SENSORS_W83791D_TEMP1_HYST, "temp1_hyst", SENSORS_W83791D_TEMP1, 
                                    SENSORS_W83791D_TEMP1, RW }, 
                                    W83781D_SYSCTL_TEMP1, VALUE(2), 1 },
    { { SENSORS_W83791D_TEMP1_OVER, "temp1_over", SENSORS_W83791D_TEMP1, 
                                    SENSORS_W83791D_TEMP1, RW }, 
                                    W83781D_SYSCTL_TEMP1, VALUE(1), 1 },
    { { SENSORS_W83791D_TEMP2, "temp2", NOMAP, NOMAP, R }, 
                               W83781D_SYSCTL_TEMP2, VALUE(3), 1 },
    { { SENSORS_W83791D_TEMP2_HYST, "temp2_hyst", SENSORS_W83791D_TEMP2, 
                                    SENSORS_W83791D_TEMP2, RW }, 
                                    W83781D_SYSCTL_TEMP2, VALUE(2), 1 },
    { { SENSORS_W83791D_TEMP2_OVER, "temp2_over", SENSORS_W83791D_TEMP2, 
                                    SENSORS_W83791D_TEMP2, RW }, 
                                    W83781D_SYSCTL_TEMP2, VALUE(1), 1 },
    { { SENSORS_W83791D_TEMP3, "temp3", NOMAP, NOMAP, R }, 
                               W83781D_SYSCTL_TEMP3, VALUE(3), 1 },
    { { SENSORS_W83791D_TEMP3_HYST, "temp3_hyst", SENSORS_W83791D_TEMP3, 
                                    SENSORS_W83791D_TEMP3, RW }, 
                                    W83781D_SYSCTL_TEMP3, VALUE(2), 1 },
    { { SENSORS_W83791D_TEMP3_OVER, "temp3_over", SENSORS_W83791D_TEMP3, 
                                    SENSORS_W83791D_TEMP3, RW }, 
                                    W83781D_SYSCTL_TEMP3, VALUE(1), 1 },
    { { SENSORS_W83791D_VID, "vid", NOMAP, NOMAP, R }, 
                             W83781D_SYSCTL_VID, VALUE(1), 3 },
    { { SENSORS_W83791D_VRM, "vrm", NOMAP, NOMAP, RW }, 
                             W83781D_SYSCTL_VRM, VALUE(1), 1 },
    { { SENSORS_W83791D_FAN1_DIV, "fan1_div", SENSORS_W83791D_FAN1, NOMAP, RW }, 
                                  W83781D_SYSCTL_FAN_DIV, VALUE(1), 0 },
    { { SENSORS_W83791D_FAN2_DIV, "fan2_div", SENSORS_W83791D_FAN2, NOMAP, RW }, 
                                  W83781D_SYSCTL_FAN_DIV, VALUE(2), 0 },
    { { SENSORS_W83791D_FAN3_DIV, "fan3_div", SENSORS_W83791D_FAN3, NOMAP, RW }, 
                                  W83781D_SYSCTL_FAN_DIV, VALUE(3), 0 },
    { { SENSORS_W83791D_FAN4_DIV, "fan4_div", SENSORS_W83791D_FAN4, NOMAP, RW }, 
                                  W83781D_SYSCTL_FAN_DIV, VALUE(4), 0 },
    { { SENSORS_W83791D_FAN5_DIV, "fan5_div", SENSORS_W83791D_FAN5, NOMAP, RW }, 
                                  W83781D_SYSCTL_FAN_DIV, VALUE(5), 0 },
    { { SENSORS_W83791D_ALARMS, "alarms", NOMAP, NOMAP, R }, 
                                W83781D_SYSCTL_ALARMS, VALUE(1), 0 },
    { { SENSORS_W83791D_BEEP_ENABLE, "beep_enable", SENSORS_W83791D_ALARMS, 
                                     NOMAP, RW }, 
                                     W83781D_SYSCTL_BEEP, VALUE(1), 0 },
    { { SENSORS_W83791D_BEEPS, "beeps", SENSORS_W83791D_ALARMS, 
                               SENSORS_W83791D_ALARMS, RW }, 
                               W83781D_SYSCTL_BEEP, VALUE(2), 0 },
    { { 0 }, 0 }
  };


static sensors_chip_feature w83792d_features[] =
  { 
    { { SENSORS_W83792D_IN0, "in0", NOMAP, NOMAP, R }, 
                             W83792D_SYSCTL_IN0, VALUE(3), 3 },
    { { SENSORS_W83792D_IN1, "in1", NOMAP, NOMAP, R }, 
                             W83792D_SYSCTL_IN1, VALUE(3), 3 },
    { { SENSORS_W83792D_IN2, "in2", NOMAP, NOMAP, R }, 
                             W83792D_SYSCTL_IN2, VALUE(3), 3 },
    { { SENSORS_W83792D_IN3, "in3", NOMAP, NOMAP, R }, 
                             W83792D_SYSCTL_IN3, VALUE(3), 3 },
    { { SENSORS_W83792D_IN4, "in4", NOMAP, NOMAP, R }, 
                             W83792D_SYSCTL_IN4, VALUE(3), 3 },
    { { SENSORS_W83792D_IN5, "in5", NOMAP, NOMAP, R }, 
                             W83792D_SYSCTL_IN5, VALUE(3), 3 },
    { { SENSORS_W83792D_IN6, "in6", NOMAP, NOMAP, R }, 
                             W83792D_SYSCTL_IN6, VALUE(3), 3 },
    { { SENSORS_W83792D_IN7, "in7", NOMAP, NOMAP, R }, 
                             W83792D_SYSCTL_IN7, VALUE(3), 3 },
    { { SENSORS_W83792D_IN8, "in8", NOMAP, NOMAP, R }, 
                             W83792D_SYSCTL_IN8, VALUE(3), 3 },
    { { SENSORS_W83792D_IN0_MIN, "in0_min", SENSORS_W83792D_IN0, 
                                 SENSORS_W83792D_IN0, RW }, 
                                 W83792D_SYSCTL_IN0, VALUE(1), 3 },
    { { SENSORS_W83792D_IN1_MIN, "in1_min", SENSORS_W83792D_IN1, 
                                 SENSORS_W83792D_IN1, RW }, 
                                 W83792D_SYSCTL_IN1, VALUE(1), 3 },
    { { SENSORS_W83792D_IN2_MIN, "in2_min", SENSORS_W83792D_IN2, 
                                 SENSORS_W83792D_IN2, RW }, 
                                 W83792D_SYSCTL_IN2, VALUE(1), 3 },
    { { SENSORS_W83792D_IN3_MIN, "in3_min", SENSORS_W83792D_IN3, 
                                 SENSORS_W83792D_IN3, RW }, 
                                 W83792D_SYSCTL_IN3, VALUE(1), 3 },
    { { SENSORS_W83792D_IN4_MIN, "in4_min", SENSORS_W83792D_IN4, 
                                 SENSORS_W83792D_IN4, RW }, 
                                 W83792D_SYSCTL_IN4, VALUE(1), 3 },
    { { SENSORS_W83792D_IN5_MIN, "in5_min", SENSORS_W83792D_IN5, 
                                 SENSORS_W83792D_IN5, RW }, 
                                 W83792D_SYSCTL_IN5, VALUE(1), 3 },
    { { SENSORS_W83792D_IN6_MIN, "in6_min", SENSORS_W83792D_IN6, 
                                 SENSORS_W83792D_IN6, RW }, 
                                 W83792D_SYSCTL_IN6, VALUE(1), 3 },
    { { SENSORS_W83792D_IN7_MIN, "in7_min", SENSORS_W83792D_IN7, 
                                 SENSORS_W83792D_IN7, RW }, 
                                 W83792D_SYSCTL_IN7, VALUE(1), 3 },
    { { SENSORS_W83792D_IN8_MIN, "in8_min", SENSORS_W83792D_IN8, 
                                 SENSORS_W83792D_IN8, RW }, 
                                 W83792D_SYSCTL_IN8, VALUE(1), 3 },
    { { SENSORS_W83792D_IN0_MAX, "in0_max", SENSORS_W83792D_IN0, 
                                 SENSORS_W83792D_IN0, RW }, 
                                 W83792D_SYSCTL_IN0, VALUE(2), 3 },
    { { SENSORS_W83792D_IN1_MAX, "in1_max", SENSORS_W83792D_IN1, 
                                 SENSORS_W83792D_IN1, RW }, 
                                 W83792D_SYSCTL_IN1, VALUE(2), 3 },
    { { SENSORS_W83792D_IN2_MAX, "in2_max", SENSORS_W83792D_IN2, 
                                 SENSORS_W83792D_IN2, RW }, 
                                 W83792D_SYSCTL_IN2, VALUE(2), 3 },
    { { SENSORS_W83792D_IN3_MAX, "in3_max", SENSORS_W83792D_IN3, 
                                 SENSORS_W83792D_IN3, RW }, 
                                 W83792D_SYSCTL_IN3, VALUE(2), 3 },
    { { SENSORS_W83792D_IN4_MAX, "in4_max", SENSORS_W83792D_IN4, 
                                 SENSORS_W83792D_IN4, RW }, 
                                 W83792D_SYSCTL_IN4, VALUE(2), 3 },
    { { SENSORS_W83792D_IN5_MAX, "in5_max", SENSORS_W83792D_IN5, 
                                 SENSORS_W83792D_IN5, RW }, 
                                 W83792D_SYSCTL_IN5, VALUE(2), 3 },
    { { SENSORS_W83792D_IN6_MAX, "in6_max", SENSORS_W83792D_IN6, 
                                 SENSORS_W83792D_IN6, RW }, 
                                 W83792D_SYSCTL_IN6, VALUE(2), 3 },
    { { SENSORS_W83792D_IN7_MAX, "in7_max", SENSORS_W83792D_IN7, 
                                 SENSORS_W83792D_IN7, RW }, 
                                 W83792D_SYSCTL_IN7, VALUE(2), 3 },
    { { SENSORS_W83792D_IN8_MAX, "in8_max", SENSORS_W83792D_IN8, 
                                 SENSORS_W83792D_IN8, RW }, 
                                 W83792D_SYSCTL_IN8, VALUE(2), 3 },
    { { SENSORS_W83792D_FAN1, "fan1", NOMAP, NOMAP, R }, 
                              W83792D_SYSCTL_FAN1, VALUE(2), 0 },
    { { SENSORS_W83792D_FAN2, "fan2", NOMAP, NOMAP, R }, 
                              W83792D_SYSCTL_FAN2, VALUE(2), 0 },
    { { SENSORS_W83792D_FAN3, "fan3", NOMAP, NOMAP, R }, 
                              W83792D_SYSCTL_FAN3, VALUE(2), 0 },
    { { SENSORS_W83792D_FAN4, "fan4", NOMAP, NOMAP, R }, 
                              W83792D_SYSCTL_FAN4, VALUE(2), 0 },
    { { SENSORS_W83792D_FAN5, "fan5", NOMAP, NOMAP, R }, 
                              W83792D_SYSCTL_FAN5, VALUE(2), 0 },
    { { SENSORS_W83792D_FAN6, "fan6", NOMAP, NOMAP, R }, 
                              W83792D_SYSCTL_FAN6, VALUE(2), 0 },
    { { SENSORS_W83792D_FAN7, "fan7", NOMAP, NOMAP, R }, 
                              W83792D_SYSCTL_FAN7, VALUE(2), 0 },
    { { SENSORS_W83792D_FAN1_MIN, "fan1_min", SENSORS_W83792D_FAN1, 
                                  SENSORS_W83792D_FAN1, RW }, 
                                  W83792D_SYSCTL_FAN1, VALUE(1), 0 },
    { { SENSORS_W83792D_FAN2_MIN, "fan2_min", SENSORS_W83792D_FAN2, 
                                  SENSORS_W83792D_FAN2, RW }, 
                                  W83792D_SYSCTL_FAN2, VALUE(1), 0 },
    { { SENSORS_W83792D_FAN3_MIN, "fan3_min", SENSORS_W83792D_FAN3, 
                                  SENSORS_W83792D_FAN3, RW }, 
                                  W83792D_SYSCTL_FAN3, VALUE(1), 0 },
    { { SENSORS_W83792D_FAN4_MIN, "fan4_min", SENSORS_W83792D_FAN4, 
                                  SENSORS_W83792D_FAN4, RW }, 
                                  W83792D_SYSCTL_FAN4, VALUE(1), 0 },
    { { SENSORS_W83792D_FAN5_MIN, "fan5_min", SENSORS_W83792D_FAN5, 
                                  SENSORS_W83792D_FAN5, RW }, 
                                  W83792D_SYSCTL_FAN5, VALUE(1), 0 },
    { { SENSORS_W83792D_FAN6_MIN, "fan6_min", SENSORS_W83792D_FAN6, 
                                  SENSORS_W83792D_FAN6, RW }, 
                                  W83792D_SYSCTL_FAN6, VALUE(1), 0 },
    { { SENSORS_W83792D_FAN7_MIN, "fan7_min", SENSORS_W83792D_FAN7, 
                                  SENSORS_W83792D_FAN7, RW }, 
                                  W83792D_SYSCTL_FAN7, VALUE(1), 0 },
    { { SENSORS_W83792D_TEMP1, "temp1", NOMAP, NOMAP, R }, 
                               W83792D_SYSCTL_TEMP1, VALUE(3), 1 },
    { { SENSORS_W83792D_TEMP1_HYST, "temp1_hyst", SENSORS_W83792D_TEMP1, 
                                    SENSORS_W83792D_TEMP1, RW }, 
                                    W83792D_SYSCTL_TEMP1, VALUE(2), 1 },
    { { SENSORS_W83792D_TEMP1_OVER, "temp1_over", SENSORS_W83792D_TEMP1, 
                                    SENSORS_W83792D_TEMP1, RW }, 
                                    W83792D_SYSCTL_TEMP1, VALUE(1), 1 },
    { { SENSORS_W83792D_TEMP2, "temp2", NOMAP, NOMAP, R }, 
                               W83792D_SYSCTL_TEMP2, VALUE(3), 1 },
    { { SENSORS_W83792D_TEMP2_HYST, "temp2_hyst", SENSORS_W83792D_TEMP2, 
                                    SENSORS_W83792D_TEMP2, RW }, 
                                    W83792D_SYSCTL_TEMP2, VALUE(2), 1 },
    { { SENSORS_W83792D_TEMP2_OVER, "temp2_over", SENSORS_W83792D_TEMP2, 
                                    SENSORS_W83792D_TEMP2, RW }, 
                                    W83792D_SYSCTL_TEMP2, VALUE(1), 1 },
    { { SENSORS_W83792D_TEMP3, "temp3", NOMAP, NOMAP, R }, 
                               W83792D_SYSCTL_TEMP3, VALUE(3), 1 },
    { { SENSORS_W83792D_TEMP3_HYST, "temp3_hyst", SENSORS_W83792D_TEMP3, 
                                    SENSORS_W83792D_TEMP3, RW }, 
                                    W83792D_SYSCTL_TEMP3, VALUE(2), 1 },
    { { SENSORS_W83792D_TEMP3_OVER, "temp3_over", SENSORS_W83792D_TEMP3, 
                                    SENSORS_W83792D_TEMP3, RW }, 
                                    W83792D_SYSCTL_TEMP3, VALUE(1), 1 },
    /* { { SENSORS_W83792D_VID, "vid", NOMAP, NOMAP, R }, 
                                W83792D_SYSCTL_VID, VALUE(1), 3 },
    { { SENSORS_W83792D_VRM, "vrm", NOMAP, NOMAP, RW }, 
                             W83792D_SYSCTL_VRM, VALUE(1), 1 }, */
    { { SENSORS_W83792D_CHASSIS, "chassis", SENSORS_W83792D_CHASSIS, NOMAP, RW }, 
                                 W83792D_SYSCTL_CHASSIS, VALUE(1), 0 },
    { { SENSORS_W83792D_FAN1_DIV, "fan1_div", SENSORS_W83792D_FAN1, NOMAP, RW }, 
                                  W83792D_SYSCTL_FAN_DIV, VALUE(1), 0 },
    { { SENSORS_W83792D_FAN2_DIV, "fan2_div", SENSORS_W83792D_FAN2, NOMAP, RW }, 
                                  W83792D_SYSCTL_FAN_DIV, VALUE(2), 0 },
    { { SENSORS_W83792D_FAN3_DIV, "fan3_div", SENSORS_W83792D_FAN3, NOMAP, RW }, 
                                  W83792D_SYSCTL_FAN_DIV, VALUE(3), 0 },
    { { SENSORS_W83792D_FAN4_DIV, "fan4_div", SENSORS_W83792D_FAN4, NOMAP, RW }, 
                                  W83792D_SYSCTL_FAN_DIV, VALUE(4), 0 },
    { { SENSORS_W83792D_FAN5_DIV, "fan5_div", SENSORS_W83792D_FAN5, NOMAP, RW }, 
                                  W83792D_SYSCTL_FAN_DIV, VALUE(5), 0 },
    { { SENSORS_W83792D_FAN6_DIV, "fan6_div", SENSORS_W83792D_FAN6, NOMAP, RW }, 
                                  W83792D_SYSCTL_FAN_DIV, VALUE(6), 0 },
    { { SENSORS_W83792D_FAN7_DIV, "fan7_div", SENSORS_W83792D_FAN7, NOMAP, RW }, 
                                  W83792D_SYSCTL_FAN_DIV, VALUE(7), 0 },
    { { SENSORS_W83792D_ALARMS, "alarms", NOMAP, NOMAP, R }, 
                                W83792D_SYSCTL_ALARMS, VALUE(1), 0 },
    { { 0 }, 0 }
  };


/* w83793 */
#define SENSORS_W83793_FEATURES_IN(nr)					\
	{ { SENSORS_W83793_IN(nr), "in" #nr, NOMAP, NOMAP, R },		\
		NOSYSCTL, VALUE(3), 3 },				\
	{ { SENSORS_W83793_IN_MIN(nr), "in" #nr "_min",			\
		SENSORS_W83793_IN(nr), SENSORS_W83793_IN(nr), RW },	\
                NOSYSCTL, VALUE(1), 3 },				\
	{ { SENSORS_W83793_IN_MAX(nr), "in" #nr "_max",			\
		SENSORS_W83793_IN(nr), SENSORS_W83793_IN(nr), RW },	\
                NOSYSCTL, VALUE(2), 3 },				\
	{ { SENSORS_W83793_IN_ALARM(nr), "in" #nr "_alarm",		\
		SENSORS_W83793_IN(nr), NOMAP, R},			\
		NOSYSCTL, VALUE(1), 0 }

#define SENSORS_W83793_FEATURES_FAN(nr)					\
	{ { SENSORS_W83793_FAN(nr), "fan" #nr, NOMAP, NOMAP, R }, 	\
		NOSYSCTL, VALUE(2), 0 },				\
	{ { SENSORS_W83793_FAN_MIN(nr), "fan" #nr "_min",		\
		SENSORS_W83793_FAN(nr), SENSORS_W83793_FAN(nr), RW },	\
		NOSYSCTL, VALUE(1), 0 },				\
	{ { SENSORS_W83793_FAN_ALARM(nr), "fan" #nr "_alarm",		\
		SENSORS_W83793_FAN(nr), NOMAP, R }, 			\
		NOSYSCTL, VALUE(1), 0 }

#define SENSORS_W83793_FEATURES_TEMP(nr)				\
	{ { SENSORS_W83793_TEMP(nr), "temp" #nr, NOMAP, NOMAP, R },	\
		NOSYSCTL, VALUE(3), 3 },				\
	{ { SENSORS_W83793_TEMP_CRIT(nr), "temp" #nr "_max",		\
		SENSORS_W83793_TEMP(nr), SENSORS_W83793_TEMP(nr), RW }, \
		NOSYSCTL, VALUE(1), 3 },				\
	{ { SENSORS_W83793_TEMP_CRIT_HYST(nr), "temp" #nr "_max_hyst",	\
		SENSORS_W83793_TEMP(nr), SENSORS_W83793_TEMP(nr),	\
		RW }, 							\
		NOSYSCTL, VALUE(2), 3 },				\
	{ { SENSORS_W83793_TEMP_ALARM(nr), "temp" #nr "_alarm",		\
		SENSORS_W83793_TEMP(nr), NOMAP, R },			\
		NOSYSCTL, VALUE(1), 0 }

/* No support for Linux 2.4 yet (sysctl) */
static sensors_chip_feature w83793_features[] =
{
	SENSORS_W83793_FEATURES_IN(0),
	SENSORS_W83793_FEATURES_IN(1),
	SENSORS_W83793_FEATURES_IN(2),
	SENSORS_W83793_FEATURES_IN(3),
	SENSORS_W83793_FEATURES_IN(4),
	SENSORS_W83793_FEATURES_IN(5),
	SENSORS_W83793_FEATURES_IN(6),
	SENSORS_W83793_FEATURES_IN(7),
	SENSORS_W83793_FEATURES_IN(8),
	SENSORS_W83793_FEATURES_IN(9),
	SENSORS_W83793_FEATURES_FAN(1),
	SENSORS_W83793_FEATURES_FAN(2),
	SENSORS_W83793_FEATURES_FAN(3),
	SENSORS_W83793_FEATURES_FAN(4),
	SENSORS_W83793_FEATURES_FAN(5),
	SENSORS_W83793_FEATURES_FAN(6),
	SENSORS_W83793_FEATURES_FAN(7),
	SENSORS_W83793_FEATURES_FAN(8),
	SENSORS_W83793_FEATURES_FAN(9),
	SENSORS_W83793_FEATURES_FAN(10),
	SENSORS_W83793_FEATURES_FAN(11),
	SENSORS_W83793_FEATURES_FAN(12),
	SENSORS_W83793_FEATURES_TEMP(1),
	SENSORS_W83793_FEATURES_TEMP(2),
	SENSORS_W83793_FEATURES_TEMP(3),
	SENSORS_W83793_FEATURES_TEMP(4),
	SENSORS_W83793_FEATURES_TEMP(5),
	SENSORS_W83793_FEATURES_TEMP(6),
    { { SENSORS_W83793_VID0, "cpu0_vid", NOMAP, NOMAP, R }, 
                          NOSYSCTL, VALUE(1), 3 },
    { { SENSORS_W83793_VID1, "cpu1_vid", NOMAP, NOMAP, R }, 
                          NOSYSCTL, VALUE(1), 3 },
    { { SENSORS_W83793_VRM, "vrm", NOMAP, NOMAP, RW }, 
                         NOSYSCTL, VALUE(1), 1 },
    { { SENSORS_W83793_CHASSIS, "chassis", NOMAP, NOMAP, RW }, 
                             NOSYSCTL, VALUE(1), 0 },
    { { 0 }, 0 }
};


static sensors_chip_feature w83l785ts_features[] =
  {
    { { SENSORS_W83L785TS_TEMP, "temp", NOMAP, NOMAP, R }, 
                                W83L785TS_SYSCTL_TEMP, VALUE(2), 0 },
    { { SENSORS_W83L785TS_TEMP_OVER, "temp_over", SENSORS_W83L785TS_TEMP, 
                                     SENSORS_W83L785TS_TEMP, R }, 
                                     W83L785TS_SYSCTL_TEMP, VALUE(1), 0 },
    { { 0 }, 0 }
  };

/* W83627DHG uses same structure excluding in9* */
static sensors_chip_feature w83627ehf_features[] =
  {
    { { SENSORS_W83627EHF_IN0, "in0", NOMAP, NOMAP, R }, 
                               W83627EHF_SYSCTL_IN0, VALUE(3), 3 },
    { { SENSORS_W83627EHF_IN1, "in1", NOMAP, NOMAP, R }, 
                               W83627EHF_SYSCTL_IN1, VALUE(3), 3 },
    { { SENSORS_W83627EHF_IN2, "in2", NOMAP, NOMAP, R }, 
                               W83627EHF_SYSCTL_IN2, VALUE(3), 3 },
    { { SENSORS_W83627EHF_IN3, "in3", NOMAP, NOMAP, R }, 
                               W83627EHF_SYSCTL_IN3, VALUE(3), 3 },
    { { SENSORS_W83627EHF_IN4, "in4", NOMAP, NOMAP, R }, 
                               W83627EHF_SYSCTL_IN4, VALUE(3), 3 },
    { { SENSORS_W83627EHF_IN5, "in5", NOMAP, NOMAP, R }, 
                               W83627EHF_SYSCTL_IN5, VALUE(3), 3 },
    { { SENSORS_W83627EHF_IN6, "in6", NOMAP, NOMAP, R }, 
                               W83627EHF_SYSCTL_IN6, VALUE(3), 3 },
    { { SENSORS_W83627EHF_IN7, "in7", NOMAP, NOMAP, R }, 
                               W83627EHF_SYSCTL_IN7, VALUE(3), 3 },
    { { SENSORS_W83627EHF_IN8, "in8", NOMAP, NOMAP, R }, 
                               W83627EHF_SYSCTL_IN8, VALUE(3), 3 },
    { { SENSORS_W83627EHF_IN9, "in9", NOMAP, NOMAP, R }, 
                               W83627EHF_SYSCTL_IN9, VALUE(3), 3 },
    { { SENSORS_W83627EHF_IN0_ALARM, "in0_alarm", SENSORS_W83627EHF_IN0, NOMAP, 
                                     R }, 
                                     NOSYSCTL, VALUE(1), 0 },
    { { SENSORS_W83627EHF_IN1_ALARM, "in1_alarm", SENSORS_W83627EHF_IN1, NOMAP, 
                                     R }, 
                                     NOSYSCTL, VALUE(1), 0 },
    { { SENSORS_W83627EHF_IN2_ALARM, "in2_alarm", SENSORS_W83627EHF_IN2, NOMAP, 
                                     R }, 
                                     NOSYSCTL, VALUE(1), 0 },
    { { SENSORS_W83627EHF_IN3_ALARM, "in3_alarm", SENSORS_W83627EHF_IN3, NOMAP, 
                                     R }, 
                                     NOSYSCTL, VALUE(1), 0 },
    { { SENSORS_W83627EHF_IN4_ALARM, "in4_alarm", SENSORS_W83627EHF_IN4, NOMAP, 
                                     R }, 
                                     NOSYSCTL, VALUE(1), 0 },
    { { SENSORS_W83627EHF_IN5_ALARM, "in5_alarm", SENSORS_W83627EHF_IN5, NOMAP, 
                                     R }, 
                                     NOSYSCTL, VALUE(1), 0 },
    { { SENSORS_W83627EHF_IN6_ALARM, "in6_alarm", SENSORS_W83627EHF_IN6, NOMAP, 
                                     R }, 
                                     NOSYSCTL, VALUE(1), 0 },
    { { SENSORS_W83627EHF_IN7_ALARM, "in7_alarm", SENSORS_W83627EHF_IN7, NOMAP, 
                                     R }, 
                                     NOSYSCTL, VALUE(1), 0 },
    { { SENSORS_W83627EHF_IN8_ALARM, "in8_alarm", SENSORS_W83627EHF_IN8, NOMAP, 
                                     R }, 
                                     NOSYSCTL, VALUE(1), 0 },
    { { SENSORS_W83627EHF_IN9_ALARM, "in9_alarm", SENSORS_W83627EHF_IN9, NOMAP, 
                                     R }, 
                                     NOSYSCTL, VALUE(1), 0 },
    { { SENSORS_W83627EHF_IN0_MIN, "in0_min", SENSORS_W83627EHF_IN0, 
                                   SENSORS_W83627EHF_IN0, RW }, 
                                   W83627EHF_SYSCTL_IN0, VALUE(1), 3 },
    { { SENSORS_W83627EHF_IN1_MIN, "in1_min", SENSORS_W83627EHF_IN1, 
                                   SENSORS_W83627EHF_IN1, RW }, 
                                   W83627EHF_SYSCTL_IN1, VALUE(1), 3 },
    { { SENSORS_W83627EHF_IN2_MIN, "in2_min", SENSORS_W83627EHF_IN2, 
                                   SENSORS_W83627EHF_IN2, RW }, 
                                   W83627EHF_SYSCTL_IN2, VALUE(1), 3 },
    { { SENSORS_W83627EHF_IN3_MIN, "in3_min", SENSORS_W83627EHF_IN3, 
                                   SENSORS_W83627EHF_IN3, RW }, 
                                   W83627EHF_SYSCTL_IN3, VALUE(1), 3 },
    { { SENSORS_W83627EHF_IN4_MIN, "in4_min", SENSORS_W83627EHF_IN4, 
                                   SENSORS_W83627EHF_IN4, RW }, 
                                   W83627EHF_SYSCTL_IN4, VALUE(1), 3 },
    { { SENSORS_W83627EHF_IN5_MIN, "in5_min", SENSORS_W83627EHF_IN5, 
                                   SENSORS_W83627EHF_IN5, RW }, 
                                   W83627EHF_SYSCTL_IN5, VALUE(1), 3 },
    { { SENSORS_W83627EHF_IN6_MIN, "in6_min", SENSORS_W83627EHF_IN6, 
                                   SENSORS_W83627EHF_IN6, RW }, 
                                   W83627EHF_SYSCTL_IN6, VALUE(1), 3 },
    { { SENSORS_W83627EHF_IN7_MIN, "in7_min", SENSORS_W83627EHF_IN7, 
                                   SENSORS_W83627EHF_IN7, RW }, 
                                   W83627EHF_SYSCTL_IN7, VALUE(1), 3 },
    { { SENSORS_W83627EHF_IN8_MIN, "in8_min", SENSORS_W83627EHF_IN8, 
                                   SENSORS_W83627EHF_IN8, RW }, 
                                   W83627EHF_SYSCTL_IN8, VALUE(1), 3 },
    { { SENSORS_W83627EHF_IN9_MIN, "in9_min", SENSORS_W83627EHF_IN9, 
                                   SENSORS_W83627EHF_IN9, RW }, 
                                   W83627EHF_SYSCTL_IN9, VALUE(1), 3 },
    { { SENSORS_W83627EHF_IN0_MAX, "in0_max", SENSORS_W83627EHF_IN0, 
                                   SENSORS_W83627EHF_IN0, RW }, 
                                   W83627EHF_SYSCTL_IN0, VALUE(2), 3 },
    { { SENSORS_W83627EHF_IN1_MAX, "in1_max", SENSORS_W83627EHF_IN1, 
                                   SENSORS_W83627EHF_IN1, RW }, 
                                   W83627EHF_SYSCTL_IN1, VALUE(2), 3 },
    { { SENSORS_W83627EHF_IN2_MAX, "in2_max", SENSORS_W83627EHF_IN2, 
                                   SENSORS_W83627EHF_IN2, RW }, 
                                   W83627EHF_SYSCTL_IN2, VALUE(2), 3 },
    { { SENSORS_W83627EHF_IN3_MAX, "in3_max", SENSORS_W83627EHF_IN3, 
                                   SENSORS_W83627EHF_IN3, RW }, 
                                   W83627EHF_SYSCTL_IN3, VALUE(2), 3 },
    { { SENSORS_W83627EHF_IN4_MAX, "in4_max", SENSORS_W83627EHF_IN4, 
                                   SENSORS_W83627EHF_IN4, RW }, 
                                   W83627EHF_SYSCTL_IN4, VALUE(2), 3 },
    { { SENSORS_W83627EHF_IN5_MAX, "in5_max", SENSORS_W83627EHF_IN5, 
                                   SENSORS_W83627EHF_IN5, RW }, 
                                   W83627EHF_SYSCTL_IN5, VALUE(2), 3 },
    { { SENSORS_W83627EHF_IN6_MAX, "in6_max", SENSORS_W83627EHF_IN6, 
                                   SENSORS_W83627EHF_IN6, RW }, 
                                   W83627EHF_SYSCTL_IN6, VALUE(2), 3 },
    { { SENSORS_W83627EHF_IN7_MAX, "in7_max", SENSORS_W83627EHF_IN7, 
                                   SENSORS_W83627EHF_IN7, RW }, 
                                   W83627EHF_SYSCTL_IN7, VALUE(2), 3 },
    { { SENSORS_W83627EHF_IN8_MAX, "in8_max", SENSORS_W83627EHF_IN8, 
                                   SENSORS_W83627EHF_IN8, RW }, 
                                   W83627EHF_SYSCTL_IN8, VALUE(2), 3 },
    { { SENSORS_W83627EHF_IN9_MAX, "in9_max", SENSORS_W83627EHF_IN9, 
                                   SENSORS_W83627EHF_IN9, RW }, 
                                   W83627EHF_SYSCTL_IN9, VALUE(2), 3 },
    { { SENSORS_W83627EHF_FAN1, "fan1", NOMAP, NOMAP, R }, 
                                W83627EHF_SYSCTL_FAN1, VALUE(2), 0 },
    { { SENSORS_W83627EHF_FAN2, "fan2", NOMAP, NOMAP, R }, 
                                W83627EHF_SYSCTL_FAN2, VALUE(2), 0 },
    { { SENSORS_W83627EHF_FAN3, "fan3", NOMAP, NOMAP, R }, 
                                W83627EHF_SYSCTL_FAN3, VALUE(2), 0 },
    { { SENSORS_W83627EHF_FAN4, "fan4", NOMAP, NOMAP, R }, 
                                W83627EHF_SYSCTL_FAN4, VALUE(2), 0 },
    { { SENSORS_W83627EHF_FAN5, "fan5", NOMAP, NOMAP, R }, 
                                W83627EHF_SYSCTL_FAN5, VALUE(2), 0 },
    { { SENSORS_W83627EHF_FAN1_ALARM, "fan1_alarm", SENSORS_W83627EHF_FAN1, 
                                      NOMAP, R }, 
                                      NOSYSCTL, VALUE(1), 0 },
    { { SENSORS_W83627EHF_FAN2_ALARM, "fan2_alarm", SENSORS_W83627EHF_FAN2, 
                                      NOMAP, R }, 
                                      NOSYSCTL, VALUE(1), 0 },
    { { SENSORS_W83627EHF_FAN3_ALARM, "fan3_alarm", SENSORS_W83627EHF_FAN3, 
                                      NOMAP, R }, 
                                      NOSYSCTL, VALUE(1), 0 },
    { { SENSORS_W83627EHF_FAN4_ALARM, "fan4_alarm", SENSORS_W83627EHF_FAN4, 
                                      NOMAP, R }, 
                                      NOSYSCTL, VALUE(1), 0 },
    { { SENSORS_W83627EHF_FAN5_ALARM, "fan5_alarm", SENSORS_W83627EHF_FAN5, 
                                      NOMAP, R }, 
                                      NOSYSCTL, VALUE(1), 0 },
    { { SENSORS_W83627EHF_FAN1_MIN, "fan1_min", SENSORS_W83627EHF_FAN1, 
                                    SENSORS_W83627EHF_FAN1, RW }, 
                                    W83627EHF_SYSCTL_FAN1, VALUE(1), 0 },
    { { SENSORS_W83627EHF_FAN2_MIN, "fan2_min", SENSORS_W83627EHF_FAN2, 
                                    SENSORS_W83627EHF_FAN2, RW }, 
                                    W83627EHF_SYSCTL_FAN2, VALUE(1), 0 },
    { { SENSORS_W83627EHF_FAN3_MIN, "fan3_min", SENSORS_W83627EHF_FAN3, 
                                    SENSORS_W83627EHF_FAN3, RW }, 
                                    W83627EHF_SYSCTL_FAN3, VALUE(1), 0 },
    { { SENSORS_W83627EHF_FAN4_MIN, "fan4_min", SENSORS_W83627EHF_FAN4, 
                                    SENSORS_W83627EHF_FAN4, RW }, 
                                    W83627EHF_SYSCTL_FAN4, VALUE(1), 0 },
    { { SENSORS_W83627EHF_FAN5_MIN, "fan5_min", SENSORS_W83627EHF_FAN5, 
                                    SENSORS_W83627EHF_FAN5, RW }, 
                                    W83627EHF_SYSCTL_FAN5, VALUE(1), 0 },
    { { SENSORS_W83627EHF_FAN1_DIV, "fan1_div", SENSORS_W83627EHF_FAN1, NOMAP, R }, 
                                    W83627EHF_SYSCTL_FAN_DIV, VALUE(1), 0 },
    { { SENSORS_W83627EHF_FAN2_DIV, "fan2_div", SENSORS_W83627EHF_FAN2, NOMAP, R }, 
                                    W83627EHF_SYSCTL_FAN_DIV, VALUE(2), 0 },
    { { SENSORS_W83627EHF_FAN3_DIV, "fan3_div", SENSORS_W83627EHF_FAN3, NOMAP, R }, 
                                    W83627EHF_SYSCTL_FAN_DIV, VALUE(3), 0 },
    { { SENSORS_W83627EHF_FAN4_DIV, "fan4_div", SENSORS_W83627EHF_FAN4, NOMAP, R }, 
                                    W83627EHF_SYSCTL_FAN_DIV, VALUE(4), 0 },
    { { SENSORS_W83627EHF_FAN5_DIV, "fan5_div", SENSORS_W83627EHF_FAN5, NOMAP, R }, 
                                    W83627EHF_SYSCTL_FAN_DIV, VALUE(5), 0 },
    { { SENSORS_W83627EHF_TEMP1, "temp1", NOMAP, NOMAP, R }, 
                                 W83627EHF_SYSCTL_TEMP1, VALUE(3), 1 },
    { { SENSORS_W83627EHF_TEMP2, "temp2", NOMAP, NOMAP, R }, 
                                 W83627EHF_SYSCTL_TEMP2, VALUE(3), 1 },
    { { SENSORS_W83627EHF_TEMP3, "temp3", NOMAP, NOMAP, R }, 
                                 W83627EHF_SYSCTL_TEMP3, VALUE(3), 1 },
    { { SENSORS_W83627EHF_TEMP1_ALARM, "temp1_alarm", SENSORS_W83627EHF_TEMP1, 
                                       NOMAP, R }, 
                                       NOSYSCTL, VALUE(1), 0 },
    { { SENSORS_W83627EHF_TEMP2_ALARM, "temp2_alarm", SENSORS_W83627EHF_TEMP2, 
                                       NOMAP, R }, 
                                       NOSYSCTL, VALUE(1), 0 },
    { { SENSORS_W83627EHF_TEMP3_ALARM, "temp3_alarm", SENSORS_W83627EHF_TEMP3, 
                                       NOMAP, R }, 
                                       NOSYSCTL, VALUE(1), 0 },
    { { SENSORS_W83627EHF_TEMP1_OVER, "temp1_over", SENSORS_W83627EHF_TEMP1, 
                                      SENSORS_W83627EHF_TEMP1, RW }, 
                                      W83627EHF_SYSCTL_TEMP1, VALUE(1), 1 },
    { { SENSORS_W83627EHF_TEMP2_OVER, "temp2_over", SENSORS_W83627EHF_TEMP2, 
                                      SENSORS_W83627EHF_TEMP2, RW }, 
                                      W83627EHF_SYSCTL_TEMP2, VALUE(1), 1 },
    { { SENSORS_W83627EHF_TEMP3_OVER, "temp3_over", SENSORS_W83627EHF_TEMP3, 
                                      SENSORS_W83627EHF_TEMP3, RW }, 
                                      W83627EHF_SYSCTL_TEMP3, VALUE(1), 1 },
    { { SENSORS_W83627EHF_TEMP1_HYST, "temp1_hyst", SENSORS_W83627EHF_TEMP1, 
                                      SENSORS_W83627EHF_TEMP1, RW }, 
                                      W83627EHF_SYSCTL_TEMP1, VALUE(2), 1 },
    { { SENSORS_W83627EHF_TEMP2_HYST, "temp2_hyst", SENSORS_W83627EHF_TEMP2, 
                                      SENSORS_W83627EHF_TEMP2, RW }, 
                                      W83627EHF_SYSCTL_TEMP2, VALUE(2), 1 },
    { { SENSORS_W83627EHF_TEMP3_HYST, "temp3_hyst", SENSORS_W83627EHF_TEMP3, 
                                      SENSORS_W83627EHF_TEMP3, RW }, 
                                      W83627EHF_SYSCTL_TEMP3, VALUE(2), 1 },
    { { SENSORS_W83627EHF_TEMP1_TYPE, "sensor1", SENSORS_W83627EHF_TEMP1, 
                                      NOMAP, R }, 
                                      W83627EHF_SYSCTL_SENSOR1, VALUE(1), 0 },
    { { SENSORS_W83627EHF_TEMP2_TYPE, "sensor2", SENSORS_W83627EHF_TEMP2, 
                                      NOMAP, R }, 
                                      W83627EHF_SYSCTL_SENSOR2, VALUE(1), 0 },
    { { SENSORS_W83627EHF_TEMP3_TYPE, "sensor3", SENSORS_W83627EHF_TEMP3, 
                                      NOMAP, R }, 
                                      W83627EHF_SYSCTL_SENSOR3, VALUE(1), 0 },
    { { SENSORS_W83627EHF_VID, "vid", NOMAP, NOMAP, R },
                                      W83627EHF_SYSCTL_VID, VALUE(1), 3 },
    { { SENSORS_W83627EHF_VRM, "vrm", NOMAP, NOMAP, RW },
                                      W83627EHF_SYSCTL_VRM, VALUE(1), 1 },
    { { SENSORS_W83627EHF_ALARMS, "alarms", NOMAP, NOMAP, R }, 
                                      W83627EHF_SYSCTL_ALARMS, VALUE(1), 0 },
    { { 0 }, 0 }
  };

static sensors_chip_feature adm9240_features[] =
  { 
    { { SENSORS_ADM9240_IN0, "2.5V", NOMAP, NOMAP, R }, 
                             ADM9240_SYSCTL_IN0, VALUE(3), 2, "in0_input", 3 },
    { { SENSORS_ADM9240_IN1, "Vccp1", NOMAP, NOMAP, R }, 
                             ADM9240_SYSCTL_IN1, VALUE(3), 2, "in1_input", 3 },
    { { SENSORS_ADM9240_IN2, "3.3V", NOMAP, NOMAP, R }, 
                             ADM9240_SYSCTL_IN2, VALUE(3), 2, "in2_input", 3 },
    { { SENSORS_ADM9240_IN3, "5V", NOMAP, NOMAP, R }, 
                             ADM9240_SYSCTL_IN3, VALUE(3), 2, "in3_input", 3 },
    { { SENSORS_ADM9240_IN4, "12V", NOMAP, NOMAP, R }, 
                             ADM9240_SYSCTL_IN4, VALUE(3), 2, "in4_input", 3 },
    { { SENSORS_ADM9240_IN5, "Vccp2", NOMAP, NOMAP, R }, 
                             ADM9240_SYSCTL_IN5, VALUE(3), 2, "in5_input", 3 },
    { { SENSORS_ADM9240_IN0_MIN, "2.5V_min", SENSORS_ADM9240_IN0, 
                                 SENSORS_ADM9240_IN0, RW }, 
                                 ADM9240_SYSCTL_IN0, VALUE(1), 2, "in0_min", 3 },
    { { SENSORS_ADM9240_IN1_MIN, "Vccp1_min", SENSORS_ADM9240_IN1, 
                                 SENSORS_ADM9240_IN1, RW }, 
                                 ADM9240_SYSCTL_IN1, VALUE(1), 2, "in1_min", 3 },
    { { SENSORS_ADM9240_IN2_MIN, "3.3V_min", SENSORS_ADM9240_IN2, 
                                 SENSORS_ADM9240_IN2, RW }, 
                                 ADM9240_SYSCTL_IN2, VALUE(1), 2, "in2_min", 3 },
    { { SENSORS_ADM9240_IN3_MIN, "5V_min", SENSORS_ADM9240_IN3, 
                                 SENSORS_ADM9240_IN3, RW }, 
                                 ADM9240_SYSCTL_IN3, VALUE(1), 2, "in3_min", 3 },
    { { SENSORS_ADM9240_IN4_MIN, "12V_min", SENSORS_ADM9240_IN4, 
                                 SENSORS_ADM9240_IN4, RW }, 
                                 ADM9240_SYSCTL_IN4, VALUE(1), 2, "in4_min", 3 },
    { { SENSORS_ADM9240_IN5_MIN, "Vccp2_min", SENSORS_ADM9240_IN5, 
                                 SENSORS_ADM9240_IN5, RW }, 
                                 ADM9240_SYSCTL_IN5, VALUE(1), 2, "in5_min", 3 },
    { { SENSORS_ADM9240_IN0_MAX, "2.5V_max", SENSORS_ADM9240_IN0, 
                                 SENSORS_ADM9240_IN0, RW }, 
                                 ADM9240_SYSCTL_IN0, VALUE(2), 2, "in0_max", 3 },
    { { SENSORS_ADM9240_IN1_MAX, "Vccp1_max", SENSORS_ADM9240_IN1, 
                                 SENSORS_ADM9240_IN1, RW }, 
                                 ADM9240_SYSCTL_IN1, VALUE(2), 2, "in1_max", 3 },
    { { SENSORS_ADM9240_IN2_MAX, "3.3V_max", SENSORS_ADM9240_IN2, 
                                 SENSORS_ADM9240_IN2, RW }, 
                                 ADM9240_SYSCTL_IN2, VALUE(2), 2, "in2_max", 3 },
    { { SENSORS_ADM9240_IN3_MAX, "5V_max", SENSORS_ADM9240_IN3, 
                                 SENSORS_ADM9240_IN3, RW }, 
                                 ADM9240_SYSCTL_IN3, VALUE(2), 2, "in3_max", 3 },
    { { SENSORS_ADM9240_IN4_MAX, "12V_max", SENSORS_ADM9240_IN4, 
                                 SENSORS_ADM9240_IN4, RW }, 
                                 ADM9240_SYSCTL_IN4, VALUE(2), 2, "in4_max", 3 },
    { { SENSORS_ADM9240_IN5_MAX, "Vccp2_max", SENSORS_ADM9240_IN5, 
                                 SENSORS_ADM9240_IN5, RW }, 
                                 ADM9240_SYSCTL_IN5, VALUE(2), 2, "in5_max", 3 },
    { { SENSORS_ADM9240_FAN1, "fan1", NOMAP, NOMAP, R }, 
                              ADM9240_SYSCTL_FAN1, VALUE(2), 0 },
    { { SENSORS_ADM9240_FAN2, "fan2", NOMAP, NOMAP, R }, 
                              ADM9240_SYSCTL_FAN2, VALUE(2), 0 },
    { { SENSORS_ADM9240_FAN1_MIN, "fan1_min", SENSORS_ADM9240_FAN1, 
                                  SENSORS_ADM9240_FAN1, RW }, 
                                  ADM9240_SYSCTL_FAN1, VALUE(1), 0 },
    { { SENSORS_ADM9240_FAN2_MIN, "fan2_min", SENSORS_ADM9240_FAN2, 
                                  SENSORS_ADM9240_FAN2, RW }, 
                                  ADM9240_SYSCTL_FAN2, VALUE(1), 0 },
    { { SENSORS_ADM9240_TEMP, "temp", NOMAP, NOMAP, R }, 
                              ADM9240_SYSCTL_TEMP, VALUE(3), 1 },
    { { SENSORS_ADM9240_TEMP_HYST, "temp_hyst", SENSORS_ADM9240_TEMP, 
                                   SENSORS_ADM9240_TEMP, RW }, 
                                   ADM9240_SYSCTL_TEMP, VALUE(2), 1 },
    { { SENSORS_ADM9240_TEMP_OVER, "temp_over", SENSORS_ADM9240_TEMP, 
                                   SENSORS_ADM9240_TEMP, RW }, 
                                   ADM9240_SYSCTL_TEMP, VALUE(1), 1 },
    { { SENSORS_ADM9240_VID, "vid", NOMAP, NOMAP, R }, 
                             ADM9240_SYSCTL_VID, VALUE(1), 2 },
    { { SENSORS_ADM9240_FAN1_DIV, "fan1_div", SENSORS_ADM9240_FAN1, NOMAP, RW }, 
                                  ADM9240_SYSCTL_FAN_DIV, VALUE(1), 0 },
    { { SENSORS_ADM9240_FAN2_DIV, "fan2_div", SENSORS_ADM9240_FAN2, NOMAP, RW }, 
                                  ADM9240_SYSCTL_FAN_DIV, VALUE(2), 0 },
    { { SENSORS_ADM9240_ALARMS, "alarms", NOMAP, NOMAP, R }, 
                                ADM9240_SYSCTL_ALARMS, VALUE(1), 0 },
    { { SENSORS_ADM9240_ANALOG_OUT, "analog_out", NOMAP, NOMAP, RW }, 
                                    ADM9240_SYSCTL_ANALOG_OUT, VALUE(1), 0, 
                                    "aout_output", 3 },
    { { 0 }, 0 }
  };

static sensors_chip_feature adm1024_features[] =
  { 
    { { SENSORS_ADM1024_IN0, "2.5V", NOMAP, NOMAP, R }, 
                             ADM1024_SYSCTL_IN0, VALUE(3), 2, "in0_input", 3 },
    { { SENSORS_ADM1024_IN1, "Vccp1", NOMAP, NOMAP, R }, 
                             ADM1024_SYSCTL_IN1, VALUE(3), 2, "in1_input", 3 },
    { { SENSORS_ADM1024_IN2, "3.3V", NOMAP, NOMAP, R }, 
                             ADM1024_SYSCTL_IN2, VALUE(3), 2, "in2_input", 3 },
    { { SENSORS_ADM1024_IN3, "5V", NOMAP, NOMAP, R }, 
                             ADM1024_SYSCTL_IN3, VALUE(3), 2, "in3_input", 3 },
    { { SENSORS_ADM1024_IN4, "12V", NOMAP, NOMAP, R }, 
                             ADM1024_SYSCTL_IN4, VALUE(3), 2, "in4_input", 3 },
    { { SENSORS_ADM1024_IN5, "Vccp2", NOMAP, NOMAP, R }, 
                             ADM1024_SYSCTL_IN5, VALUE(3), 2, "in5_input", 3 },
    { { SENSORS_ADM1024_IN0_MIN, "2.5V_min", SENSORS_ADM1024_IN0, 
                                 SENSORS_ADM1024_IN0, RW }, 
                                 ADM1024_SYSCTL_IN0, VALUE(1), 2, "in0_min", 3 },
    { { SENSORS_ADM1024_IN1_MIN, "Vccp1_min", SENSORS_ADM1024_IN1, 
                                 SENSORS_ADM1024_IN1, RW }, 
                                 ADM1024_SYSCTL_IN1, VALUE(1), 2, "in1_min", 3 },
    { { SENSORS_ADM1024_IN2_MIN, "3.3V_min", SENSORS_ADM1024_IN2, 
                                 SENSORS_ADM1024_IN2, RW }, 
                                 ADM1024_SYSCTL_IN2, VALUE(1), 2, "in2_min", 3 },
    { { SENSORS_ADM1024_IN3_MIN, "5V_min", SENSORS_ADM1024_IN3, 
                                 SENSORS_ADM1024_IN3, RW }, 
                                 ADM1024_SYSCTL_IN3, VALUE(1), 2, "in3_min", 3 },
    { { SENSORS_ADM1024_IN4_MIN, "12V_min", SENSORS_ADM1024_IN4, 
                                 SENSORS_ADM1024_IN4, RW }, 
                                 ADM1024_SYSCTL_IN4, VALUE(1), 2, "in4_min", 3 },
    { { SENSORS_ADM1024_IN5_MIN, "Vccp2_min", SENSORS_ADM1024_IN5, 
                                 SENSORS_ADM1024_IN5, RW }, 
                                 ADM1024_SYSCTL_IN5, VALUE(1), 2, "in5_min", 3 },
    { { SENSORS_ADM1024_IN0_MAX, "2.5V_max", SENSORS_ADM1024_IN0, 
                                 SENSORS_ADM1024_IN0, RW }, 
                                 ADM1024_SYSCTL_IN0, VALUE(2), 2, "in0_max", 3 },
    { { SENSORS_ADM1024_IN1_MAX, "Vccp1_max", SENSORS_ADM1024_IN1, 
                                 SENSORS_ADM1024_IN1, RW }, 
                                 ADM1024_SYSCTL_IN1, VALUE(2), 2, "in1_max", 3 },
    { { SENSORS_ADM1024_IN2_MAX, "3.3V_max", SENSORS_ADM1024_IN2, 
                                 SENSORS_ADM1024_IN2, RW }, 
                                 ADM1024_SYSCTL_IN2, VALUE(2), 2, "in2_max", 3 },
    { { SENSORS_ADM1024_IN3_MAX, "5V_max", SENSORS_ADM1024_IN3, 
                                 SENSORS_ADM1024_IN3, RW }, 
                                 ADM1024_SYSCTL_IN3, VALUE(2), 2, "in3_max", 3 },
    { { SENSORS_ADM1024_IN4_MAX, "12V_max", SENSORS_ADM1024_IN4, 
                                 SENSORS_ADM1024_IN4, RW }, 
                                 ADM1024_SYSCTL_IN4, VALUE(2), 2, "in4_max", 3 },
    { { SENSORS_ADM1024_IN5_MAX, "Vccp2_max", SENSORS_ADM1024_IN5, 
                                 SENSORS_ADM1024_IN5, RW }, 
                                 ADM1024_SYSCTL_IN5, VALUE(2), 2, "in5_max", 3 },
    { { SENSORS_ADM1024_FAN1, "fan1", NOMAP, NOMAP, R }, 
                              ADM1024_SYSCTL_FAN1, VALUE(2), 0 },
    { { SENSORS_ADM1024_FAN2, "fan2", NOMAP, NOMAP, R }, 
                              ADM1024_SYSCTL_FAN2, VALUE(2), 0 },
    { { SENSORS_ADM1024_FAN1_MIN, "fan1_min", SENSORS_ADM1024_FAN1, 
                                  SENSORS_ADM1024_FAN1, RW }, 
                                  ADM1024_SYSCTL_FAN1, VALUE(1), 0 },
    { { SENSORS_ADM1024_FAN2_MIN, "fan2_min", SENSORS_ADM1024_FAN2, 
                                  SENSORS_ADM1024_FAN2, RW }, 
                                  ADM1024_SYSCTL_FAN2, VALUE(1), 0 },
    { { SENSORS_ADM1024_TEMP, "temp", NOMAP, NOMAP, R }, 
                              ADM1024_SYSCTL_TEMP, VALUE(3), 1 },
    { { SENSORS_ADM1024_TEMP1, "temp1", NOMAP, NOMAP, R }, 
                               ADM1024_SYSCTL_TEMP1, VALUE(3), 1, "temp2_input", 3 },
    { { SENSORS_ADM1024_TEMP2, "temp2", NOMAP, NOMAP, R }, 
                               ADM1024_SYSCTL_TEMP2, VALUE(3), 1, "temp3_input", 3 },
    { { SENSORS_ADM1024_TEMP_HYST, "temp_hyst", SENSORS_ADM1024_TEMP, 
                                   SENSORS_ADM1024_TEMP, RW }, 
                                   ADM1024_SYSCTL_TEMP, VALUE(2), 1, "temp1_min", 3 },
    { { SENSORS_ADM1024_TEMP_OVER, "temp_over", SENSORS_ADM1024_TEMP, 
                                   SENSORS_ADM1024_TEMP, RW }, 
                                   ADM1024_SYSCTL_TEMP, VALUE(1), 1 },
    { { SENSORS_ADM1024_TEMP1_HYST, "temp1_hyst", SENSORS_ADM1024_TEMP1, 
                                    SENSORS_ADM1024_TEMP1, RW }, 
                                    ADM1024_SYSCTL_TEMP1, VALUE(2), 1, "temp2_min", 3 },
    { { SENSORS_ADM1024_TEMP1_OVER, "temp1_over", SENSORS_ADM1024_TEMP1, 
                                    SENSORS_ADM1024_TEMP1, RW }, 
                                    ADM1024_SYSCTL_TEMP1, VALUE(1), 1, "temp2_max", 3 },
    { { SENSORS_ADM1024_TEMP2_HYST, "temp2_hyst", SENSORS_ADM1024_TEMP2, 
                                    SENSORS_ADM1024_TEMP2, RW }, 
                                    ADM1024_SYSCTL_TEMP2, VALUE(2), 1, "temp3_min", 3 },
    { { SENSORS_ADM1024_TEMP2_OVER, "temp2_over", SENSORS_ADM1024_TEMP2, 
                                    SENSORS_ADM1024_TEMP2, RW }, 
                                    ADM1024_SYSCTL_TEMP2, VALUE(1), 1, "temp3_max", 3 },
    { { SENSORS_ADM1024_VID, "vid", NOMAP, NOMAP, R }, 
                             ADM1024_SYSCTL_VID, VALUE(1), 2 },
    { { SENSORS_ADM1024_FAN1_DIV, "fan1_div", SENSORS_ADM1024_FAN1, NOMAP, RW }, 
                                  ADM1024_SYSCTL_FAN_DIV, VALUE(1), 0 },
    { { SENSORS_ADM1024_FAN2_DIV, "fan2_div", SENSORS_ADM1024_FAN2, NOMAP, RW }, 
                                  ADM1024_SYSCTL_FAN_DIV, VALUE(2), 0 },
    { { SENSORS_ADM1024_ALARMS, "alarms", NOMAP, NOMAP, R }, 
                                ADM1024_SYSCTL_ALARMS, VALUE(1), 0 },
    { { SENSORS_ADM1024_ANALOG_OUT, "analog_out", NOMAP, NOMAP, RW }, 
                                    ADM1024_SYSCTL_ANALOG_OUT, VALUE(1), 0 },
    { { 0 }, 0 }
  };

static sensors_chip_feature sis5595_features[] =
  { 
    { { SENSORS_SIS5595_IN0, "in0", NOMAP, NOMAP, R }, 
                             SIS5595_SYSCTL_IN0, VALUE(3), 2 },
    { { SENSORS_SIS5595_IN1, "in1", NOMAP, NOMAP, R }, 
                             SIS5595_SYSCTL_IN1, VALUE(3), 2 },
    { { SENSORS_SIS5595_IN2, "in2", NOMAP, NOMAP, R }, 
                             SIS5595_SYSCTL_IN2, VALUE(3), 2 },
    { { SENSORS_SIS5595_IN3, "in3", NOMAP, NOMAP, R }, 
                             SIS5595_SYSCTL_IN3, VALUE(3), 2 },
    { { SENSORS_SIS5595_IN4, "in4", NOMAP, NOMAP, R }, 
                             SIS5595_SYSCTL_IN4, VALUE(3), 2 },
    { { SENSORS_SIS5595_IN0_MIN, "in0_min", SENSORS_SIS5595_IN0, 
                                 SENSORS_SIS5595_IN0, RW }, 
                                 SIS5595_SYSCTL_IN0, VALUE(1), 2 },
    { { SENSORS_SIS5595_IN1_MIN, "in1_min", SENSORS_SIS5595_IN1, 
                                 SENSORS_SIS5595_IN1, RW }, 
                                 SIS5595_SYSCTL_IN1, VALUE(1), 2 },
    { { SENSORS_SIS5595_IN2_MIN, "in2_min", SENSORS_SIS5595_IN2, 
                                 SENSORS_SIS5595_IN2, RW }, 
                                 SIS5595_SYSCTL_IN2, VALUE(1), 2 },
    { { SENSORS_SIS5595_IN3_MIN, "in3_min", SENSORS_SIS5595_IN3, 
                                 SENSORS_SIS5595_IN3, RW }, 
                                 SIS5595_SYSCTL_IN3, VALUE(1), 2 },
    { { SENSORS_SIS5595_IN4_MIN, "in4_min", SENSORS_SIS5595_IN4, 
                                 SENSORS_SIS5595_IN4, RW }, 
                                 SIS5595_SYSCTL_IN4, VALUE(1), 2 },
    { { SENSORS_SIS5595_IN0_MAX, "in0_max", SENSORS_SIS5595_IN0, 
                                 SENSORS_SIS5595_IN0, RW }, 
                                 SIS5595_SYSCTL_IN0, VALUE(2), 2 },
    { { SENSORS_SIS5595_IN1_MAX, "in1_max", SENSORS_SIS5595_IN1, 
                                 SENSORS_SIS5595_IN1, RW }, 
                                 SIS5595_SYSCTL_IN1, VALUE(2), 2 },
    { { SENSORS_SIS5595_IN2_MAX, "in2_max", SENSORS_SIS5595_IN2, 
                                 SENSORS_SIS5595_IN2, RW }, 
                                 SIS5595_SYSCTL_IN2, VALUE(2), 2 },
    { { SENSORS_SIS5595_IN3_MAX, "in3_max", SENSORS_SIS5595_IN3, 
                                 SENSORS_SIS5595_IN3, RW }, 
                                 SIS5595_SYSCTL_IN3, VALUE(2), 2 },
    { { SENSORS_SIS5595_IN4_MAX, "in4_max", SENSORS_SIS5595_IN4, 
                                 SENSORS_SIS5595_IN4, RW }, 
                                 SIS5595_SYSCTL_IN4, VALUE(2), 2 },
    { { SENSORS_SIS5595_FAN1, "fan1", NOMAP, NOMAP, R }, 
                              SIS5595_SYSCTL_FAN1, VALUE(2), 0 },
    { { SENSORS_SIS5595_FAN2, "fan2", NOMAP, NOMAP, R }, 
                              SIS5595_SYSCTL_FAN2, VALUE(2), 0 },
    { { SENSORS_SIS5595_FAN1_MIN, "fan1_min", SENSORS_SIS5595_FAN1, 
                                  SENSORS_SIS5595_FAN1, RW }, 
                                  SIS5595_SYSCTL_FAN1, VALUE(1), 0 },
    { { SENSORS_SIS5595_FAN2_MIN, "fan2_min", SENSORS_SIS5595_FAN2, 
                                  SENSORS_SIS5595_FAN2, RW }, 
                                  SIS5595_SYSCTL_FAN2, VALUE(1), 0 },
    { { SENSORS_SIS5595_TEMP, "temp", NOMAP, NOMAP, R }, 
                              SIS5595_SYSCTL_TEMP, VALUE(3), 1 },
    { { SENSORS_SIS5595_TEMP_HYST, "temp_hyst", SENSORS_SIS5595_TEMP, 
                                   SENSORS_SIS5595_TEMP, RW }, 
                                   SIS5595_SYSCTL_TEMP, VALUE(2), 1 },
    { { SENSORS_SIS5595_TEMP_OVER, "temp_over", SENSORS_SIS5595_TEMP, 
                                   SENSORS_SIS5595_TEMP, RW }, 
                                   SIS5595_SYSCTL_TEMP, VALUE(1), 1 },
    { { SENSORS_SIS5595_FAN1_DIV, "fan1_div", SENSORS_SIS5595_FAN1, NOMAP, RW }, 
                                  SIS5595_SYSCTL_FAN_DIV, VALUE(1), 0 },
    { { SENSORS_SIS5595_FAN2_DIV, "fan2_div", SENSORS_SIS5595_FAN2, NOMAP, RW }, 
                                  SIS5595_SYSCTL_FAN_DIV, VALUE(2), 0 },
    { { SENSORS_SIS5595_ALARMS, "alarms", NOMAP, NOMAP, R }, 
                                SIS5595_SYSCTL_ALARMS, VALUE(1), 0 },
    { { 0 }, 0 }
  };

static sensors_chip_feature maxi_cg_features[] =
  { 
    { { SENSORS_MAXI_CG_FAN1, "fan1", NOMAP, NOMAP, R }, 
                              MAXI_SYSCTL_FAN1, VALUE(3), 0 },
    { { SENSORS_MAXI_CG_FAN2, "fan2", NOMAP, NOMAP, R }, 
                              MAXI_SYSCTL_FAN2, VALUE(3), 0 },
    { { SENSORS_MAXI_CG_FAN3, "fan3", NOMAP, NOMAP, R }, 
                              MAXI_SYSCTL_FAN3, VALUE(3), 0 },
    { { SENSORS_MAXI_CG_FAN1_MIN, "fan1_min", SENSORS_MAXI_CG_FAN1, 
                                  SENSORS_MAXI_CG_FAN1, RW }, 
                                  MAXI_SYSCTL_FAN1, VALUE(1), 0 },
    { { SENSORS_MAXI_CG_FAN2_MIN, "fan2_min", SENSORS_MAXI_CG_FAN2, 
                                  SENSORS_MAXI_CG_FAN2, RW }, 
                                  MAXI_SYSCTL_FAN2, VALUE(1), 0 },
    { { SENSORS_MAXI_CG_FAN3_MIN, "fan3_min", SENSORS_MAXI_CG_FAN3, 
                                  SENSORS_MAXI_CG_FAN3, RW }, 
                                  MAXI_SYSCTL_FAN3, VALUE(1), 0 },
    { { SENSORS_MAXI_CG_FAN1_DIV, "fan1_div", SENSORS_MAXI_CG_FAN1, NOMAP, R }, 
                                  MAXI_SYSCTL_FAN1, VALUE(2), 0 },
    { { SENSORS_MAXI_CG_FAN2_DIV, "fan2_div", SENSORS_MAXI_CG_FAN2, NOMAP, R }, 
                                  MAXI_SYSCTL_FAN2, VALUE(2), 0 },
    { { SENSORS_MAXI_CG_FAN3_DIV, "fan3_div", SENSORS_MAXI_CG_FAN3, NOMAP, R }, 
                                  MAXI_SYSCTL_FAN3, VALUE(2), 0 },
    { { SENSORS_MAXI_CG_TEMP1, "temp1", NOMAP, NOMAP, R }, 
                               MAXI_SYSCTL_TEMP1, VALUE(3), 1 },
    { { SENSORS_MAXI_CG_TEMP2, "temp2", NOMAP, NOMAP, R }, 
                               MAXI_SYSCTL_TEMP2, VALUE(3), 1 },
    { { SENSORS_MAXI_CG_TEMP3, "temp3", NOMAP, NOMAP, R }, 
                               MAXI_SYSCTL_TEMP3, VALUE(3), 1 },
    { { SENSORS_MAXI_CG_TEMP4, "temp4", NOMAP, NOMAP, R }, 
                               MAXI_SYSCTL_TEMP4, VALUE(3), 1 },
    { { SENSORS_MAXI_CG_TEMP5, "temp5", NOMAP, NOMAP, R }, 
                               MAXI_SYSCTL_TEMP5, VALUE(3), 1 },
    { { SENSORS_MAXI_CG_TEMP1_MAX, "temp1_max", SENSORS_MAXI_CG_TEMP1, 
                                   SENSORS_MAXI_CG_TEMP1, R }, 
                                   MAXI_SYSCTL_TEMP1, VALUE(1), 1 },
    { { SENSORS_MAXI_CG_TEMP2_MAX, "temp2_max", SENSORS_MAXI_CG_TEMP2, 
                                   SENSORS_MAXI_CG_TEMP2, R }, 
                                   MAXI_SYSCTL_TEMP2, VALUE(1), 1 },
    { { SENSORS_MAXI_CG_TEMP3_MAX, "temp3_max", SENSORS_MAXI_CG_TEMP3, 
                                   SENSORS_MAXI_CG_TEMP3, R }, 
                                   MAXI_SYSCTL_TEMP3, VALUE(1), 1 },
    { { SENSORS_MAXI_CG_TEMP4_MAX, "temp4_max", SENSORS_MAXI_CG_TEMP4, 
                                   SENSORS_MAXI_CG_TEMP4, R }, 
                                   MAXI_SYSCTL_TEMP4, VALUE(1), 1 },
    { { SENSORS_MAXI_CG_TEMP5_MAX, "temp5_max", SENSORS_MAXI_CG_TEMP5, 
                                   SENSORS_MAXI_CG_TEMP5, R }, 
                                   MAXI_SYSCTL_TEMP5, VALUE(1), 1 },
    { { SENSORS_MAXI_CG_TEMP1_HYST, "temp1_hyst", SENSORS_MAXI_CG_TEMP1, 
                                    SENSORS_MAXI_CG_TEMP1, R }, 
                                    MAXI_SYSCTL_TEMP1, VALUE(2), 1 },
    { { SENSORS_MAXI_CG_TEMP2_HYST, "temp2_hyst", SENSORS_MAXI_CG_TEMP2, 
                                    SENSORS_MAXI_CG_TEMP2, R }, 
                                    MAXI_SYSCTL_TEMP2, VALUE(2), 1 },
    { { SENSORS_MAXI_CG_TEMP3_HYST, "temp3_hyst", SENSORS_MAXI_CG_TEMP3, 
                                    SENSORS_MAXI_CG_TEMP3, R }, 
                                    MAXI_SYSCTL_TEMP3, VALUE(2), 1 },
    { { SENSORS_MAXI_CG_TEMP4_HYST, "temp4_hyst", SENSORS_MAXI_CG_TEMP4, 
                                    SENSORS_MAXI_CG_TEMP4, R }, 
                                    MAXI_SYSCTL_TEMP4, VALUE(2), 1 },
    { { SENSORS_MAXI_CG_TEMP5_HYST, "temp5_hyst", SENSORS_MAXI_CG_TEMP5, 
                                    SENSORS_MAXI_CG_TEMP5, R }, 
                                    MAXI_SYSCTL_TEMP5, VALUE(2), 1 },
    { { SENSORS_MAXI_CG_PLL, "pll", NOMAP, NOMAP, R }, 
                             MAXI_SYSCTL_PLL, VALUE(3), 2 },
    { { SENSORS_MAXI_CG_PLL_MIN, "pll_min", SENSORS_MAXI_CG_PLL, 
                                 SENSORS_MAXI_CG_PLL, RW }, 
                                 MAXI_SYSCTL_PLL, VALUE(1), 2 },
    { { SENSORS_MAXI_CG_PLL_MAX, "pll_max", SENSORS_MAXI_CG_PLL, 
                                 SENSORS_MAXI_CG_PLL, RW }, 
                                 MAXI_SYSCTL_PLL, VALUE(2), 2 },
    { { SENSORS_MAXI_CG_VID1, "vid1", NOMAP, NOMAP, R }, 
                              MAXI_SYSCTL_VID1, VALUE(3), 4, "in0_input", 3 },
    { { SENSORS_MAXI_CG_VID2, "vid2", NOMAP, NOMAP, R }, 
                              MAXI_SYSCTL_VID2, VALUE(3), 4, "in1_input", 3 },
    { { SENSORS_MAXI_CG_VID3, "vid3", NOMAP, NOMAP, R }, 
                              MAXI_SYSCTL_VID3, VALUE(3), 4, "in2_input", 3 },
    { { SENSORS_MAXI_CG_VID4, "vid4", NOMAP, NOMAP, R }, 
                              MAXI_SYSCTL_VID4, VALUE(3), 4, "in3_input", 3 },
    { { SENSORS_MAXI_CG_VID1_MIN, "vid1_min", SENSORS_MAXI_CG_VID1, 
                                  SENSORS_MAXI_CG_VID1, RW }, 
                                  MAXI_SYSCTL_VID1, VALUE(1), 4, "in0_min", 3 },
    { { SENSORS_MAXI_CG_VID2_MIN, "vid2_min", SENSORS_MAXI_CG_VID2, 
                                  SENSORS_MAXI_CG_VID2, RW }, 
                                  MAXI_SYSCTL_VID2, VALUE(1), 4, "in1_min", 3 },
    { { SENSORS_MAXI_CG_VID3_MIN, "vid3_min", SENSORS_MAXI_CG_VID3, 
                                  SENSORS_MAXI_CG_VID3, RW }, 
                                  MAXI_SYSCTL_VID3, VALUE(1), 4, "in2_min", 3 },
    { { SENSORS_MAXI_CG_VID4_MIN, "vid4_min", SENSORS_MAXI_CG_VID4, 
                                  SENSORS_MAXI_CG_VID4, RW }, 
                                  MAXI_SYSCTL_VID4, VALUE(1), 4, "in3_min", 3 },
    { { SENSORS_MAXI_CG_VID1_MAX, "vid1_max", SENSORS_MAXI_CG_VID1, 
                                  SENSORS_MAXI_CG_VID1, RW }, 
                                  MAXI_SYSCTL_VID1, VALUE(2), 4, "in0_max", 3 },
    { { SENSORS_MAXI_CG_VID2_MAX, "vid2_max", SENSORS_MAXI_CG_VID2, 
                                  SENSORS_MAXI_CG_VID2, RW }, 
                                  MAXI_SYSCTL_VID2, VALUE(2), 4, "in1_max", 3 },
    { { SENSORS_MAXI_CG_VID3_MAX, "vid3_max", SENSORS_MAXI_CG_VID3, 
                                  SENSORS_MAXI_CG_VID3, RW }, 
                                  MAXI_SYSCTL_VID3, VALUE(2), 4, "in2_max", 3 },
    { { SENSORS_MAXI_CG_VID4_MAX, "vid4_max", SENSORS_MAXI_CG_VID4, 
                                  SENSORS_MAXI_CG_VID4, RW }, 
                                  MAXI_SYSCTL_VID4, VALUE(2), 4, "in3_max", 3 },
    { { SENSORS_MAXI_CG_ALARMS, "alarms", NOMAP, NOMAP, R }, 
                                MAXI_SYSCTL_ALARMS, VALUE(1), 0 },
    { { 0 }, 0 }
  };

static sensors_chip_feature maxi_co_features[] =
  { 
    { { SENSORS_MAXI_CO_FAN1, "fan1", NOMAP, NOMAP, R }, 
                              MAXI_SYSCTL_FAN1, VALUE(3), 0 },
    { { SENSORS_MAXI_CO_FAN2, "fan2", NOMAP, NOMAP, R }, 
                              MAXI_SYSCTL_FAN2, VALUE(3), 0 },
    { { SENSORS_MAXI_CO_FAN3, "fan3", NOMAP, NOMAP, R }, 
                              MAXI_SYSCTL_FAN3, VALUE(3), 0 },
    { { SENSORS_MAXI_CO_FAN1_MIN, "fan1_min", SENSORS_MAXI_CO_FAN1, 
                                  SENSORS_MAXI_CO_FAN1, RW }, 
                                  MAXI_SYSCTL_FAN1, VALUE(1), 0 },
    { { SENSORS_MAXI_CO_FAN2_MIN, "fan2_min", SENSORS_MAXI_CO_FAN2, 
                                  SENSORS_MAXI_CO_FAN2, RW }, 
                                  MAXI_SYSCTL_FAN2, VALUE(1), 0 },
    { { SENSORS_MAXI_CO_FAN3_MIN, "fan3_min", SENSORS_MAXI_CO_FAN3, 
                                  SENSORS_MAXI_CO_FAN3, RW }, 
                                  MAXI_SYSCTL_FAN3, VALUE(1), 0 },
    { { SENSORS_MAXI_CO_FAN1_DIV, "fan1_div", SENSORS_MAXI_CO_FAN1, NOMAP, R }, 
                                  MAXI_SYSCTL_FAN1, VALUE(2), 0 },
    { { SENSORS_MAXI_CO_FAN2_DIV, "fan2_div", SENSORS_MAXI_CO_FAN2, NOMAP, R }, 
                                  MAXI_SYSCTL_FAN2, VALUE(2), 0 },
    { { SENSORS_MAXI_CO_FAN3_DIV, "fan3_div", SENSORS_MAXI_CO_FAN3, NOMAP, R }, 
                                  MAXI_SYSCTL_FAN3, VALUE(2), 0 },
    { { SENSORS_MAXI_CO_TEMP1, "temp1", NOMAP, NOMAP, R }, 
                               MAXI_SYSCTL_TEMP1, VALUE(3), 1 },
    { { SENSORS_MAXI_CO_TEMP2, "temp2", NOMAP, NOMAP, R }, 
                               MAXI_SYSCTL_TEMP2, VALUE(3), 1 },
    { { SENSORS_MAXI_CO_TEMP3, "temp3", NOMAP, NOMAP, R }, 
                               MAXI_SYSCTL_TEMP3, VALUE(3), 1 },
    { { SENSORS_MAXI_CO_TEMP4, "temp4", NOMAP, NOMAP, R }, 
                               MAXI_SYSCTL_TEMP4, VALUE(3), 1 },
    { { SENSORS_MAXI_CO_TEMP5, "temp5", NOMAP, NOMAP, R }, 
                               MAXI_SYSCTL_TEMP5, VALUE(3), 1 },
    { { SENSORS_MAXI_CO_TEMP1_MAX, "temp1_max", SENSORS_MAXI_CO_TEMP1, 
                                   SENSORS_MAXI_CO_TEMP1, R }, 
                                   MAXI_SYSCTL_TEMP1, VALUE(1), 1 },
    { { SENSORS_MAXI_CO_TEMP2_MAX, "temp2_max", SENSORS_MAXI_CO_TEMP2, 
                                   SENSORS_MAXI_CO_TEMP2, R }, 
                                   MAXI_SYSCTL_TEMP2, VALUE(1), 1 },
    { { SENSORS_MAXI_CO_TEMP3_MAX, "temp3_max", SENSORS_MAXI_CO_TEMP3, 
                                   SENSORS_MAXI_CO_TEMP3, R }, 
                                   MAXI_SYSCTL_TEMP3, VALUE(1), 1 },
    { { SENSORS_MAXI_CO_TEMP4_MAX, "temp4_max", SENSORS_MAXI_CO_TEMP4, 
                                   SENSORS_MAXI_CO_TEMP4, R }, 
                                   MAXI_SYSCTL_TEMP4, VALUE(1), 1 },
    { { SENSORS_MAXI_CO_TEMP5_MAX, "temp5_max", SENSORS_MAXI_CO_TEMP5, 
                                   SENSORS_MAXI_CO_TEMP5, R }, 
                                   MAXI_SYSCTL_TEMP5, VALUE(1), 1 },
    { { SENSORS_MAXI_CO_TEMP1_HYST, "temp1_hyst", SENSORS_MAXI_CO_TEMP1, 
                                    SENSORS_MAXI_CO_TEMP1, R }, 
                                    MAXI_SYSCTL_TEMP1, VALUE(2), 1 },
    { { SENSORS_MAXI_CO_TEMP2_HYST, "temp2_hyst", SENSORS_MAXI_CO_TEMP2, 
                                    SENSORS_MAXI_CO_TEMP2, R }, 
                                    MAXI_SYSCTL_TEMP2, VALUE(2), 1 },
    { { SENSORS_MAXI_CO_TEMP3_HYST, "temp3_hyst", SENSORS_MAXI_CO_TEMP3, 
                                    SENSORS_MAXI_CO_TEMP3, R }, 
                                    MAXI_SYSCTL_TEMP3, VALUE(2), 1 },
    { { SENSORS_MAXI_CO_TEMP4_HYST, "temp4_hyst", SENSORS_MAXI_CO_TEMP4, 
                                    SENSORS_MAXI_CO_TEMP4, R }, 
                                    MAXI_SYSCTL_TEMP4, VALUE(2), 1 },
    { { SENSORS_MAXI_CO_TEMP5_HYST, "temp5_hyst", SENSORS_MAXI_CO_TEMP5, 
                                    SENSORS_MAXI_CO_TEMP5, R }, 
                                    MAXI_SYSCTL_TEMP5, VALUE(2), 1 },
    { { SENSORS_MAXI_CO_PLL, "pll", NOMAP, NOMAP, R }, 
                             MAXI_SYSCTL_PLL, VALUE(3), 2 },
    { { SENSORS_MAXI_CO_PLL_MIN, "pll_min", SENSORS_MAXI_CO_PLL, 
                                 SENSORS_MAXI_CO_PLL, RW }, 
                                 MAXI_SYSCTL_PLL, VALUE(1), 2 },
    { { SENSORS_MAXI_CO_PLL_MAX, "pll_max", SENSORS_MAXI_CO_PLL, 
                                 SENSORS_MAXI_CO_PLL, RW }, 
                                 MAXI_SYSCTL_PLL, VALUE(2), 2 },
    { { SENSORS_MAXI_CO_VID1, "vid1", NOMAP, NOMAP, R }, 
                              MAXI_SYSCTL_VID1, VALUE(3), 4, "in0_input", 3 },
    { { SENSORS_MAXI_CO_VID2, "vid2", NOMAP, NOMAP, R }, 
                              MAXI_SYSCTL_VID2, VALUE(3), 4, "in1_input", 3 },
    { { SENSORS_MAXI_CO_VID3, "vid3", NOMAP, NOMAP, R }, 
                              MAXI_SYSCTL_VID3, VALUE(3), 4, "in2_input", 3 },
    { { SENSORS_MAXI_CO_VID4, "vid4", NOMAP, NOMAP, R }, 
                              MAXI_SYSCTL_VID4, VALUE(3), 4, "in3_input", 3 },
    { { SENSORS_MAXI_CO_VID1_MIN, "vid1_min", SENSORS_MAXI_CO_VID1, 
                                  SENSORS_MAXI_CO_VID1, RW }, 
                                  MAXI_SYSCTL_VID1, VALUE(1), 4, "in0_min", 3 },
    { { SENSORS_MAXI_CO_VID2_MIN, "vid2_min", SENSORS_MAXI_CO_VID2, 
                                  SENSORS_MAXI_CO_VID2, RW }, 
                                  MAXI_SYSCTL_VID2, VALUE(1), 4, "in1_min", 3 },
    { { SENSORS_MAXI_CO_VID3_MIN, "vid3_min", SENSORS_MAXI_CO_VID3, 
                                  SENSORS_MAXI_CO_VID3, RW }, 
                                  MAXI_SYSCTL_VID3, VALUE(1), 4, "in2_min", 3 },
    { { SENSORS_MAXI_CO_VID4_MIN, "vid4_min", SENSORS_MAXI_CO_VID4, 
                                  SENSORS_MAXI_CO_VID4, RW }, 
                                  MAXI_SYSCTL_VID4, VALUE(1), 4, "in3_min", 3 },
    { { SENSORS_MAXI_CO_VID1_MAX, "vid1_max", SENSORS_MAXI_CO_VID1, 
                                  SENSORS_MAXI_CO_VID1, RW }, 
                                  MAXI_SYSCTL_VID1, VALUE(2), 4, "in0_max", 3 },
    { { SENSORS_MAXI_CO_VID2_MAX, "vid2_max", SENSORS_MAXI_CO_VID2, 
                                  SENSORS_MAXI_CO_VID2, RW }, 
                                  MAXI_SYSCTL_VID2, VALUE(2), 4, "in1_max", 3 },
    { { SENSORS_MAXI_CG_VID3_MAX, "vid3_max", SENSORS_MAXI_CO_VID3, 
                                  SENSORS_MAXI_CO_VID3, RW }, 
                                  MAXI_SYSCTL_VID3, VALUE(2), 4, "in2_max", 3 },
    { { SENSORS_MAXI_CO_VID4_MAX, "vid4_max", SENSORS_MAXI_CO_VID4, 
                                  SENSORS_MAXI_CG_VID4, RW }, 
                                  MAXI_SYSCTL_VID4, VALUE(2), 4, "in3_max", 3 },
    { { SENSORS_MAXI_CO_ALARMS, "alarms", NOMAP, NOMAP, R }, 
                                MAXI_SYSCTL_ALARMS, VALUE(1), 0 },
    { { 0 }, 0 }
  };

static sensors_chip_feature maxi_as_features[] =
  { 
    { { SENSORS_MAXI_AS_FAN1, "fan1", NOMAP, NOMAP, R }, 
                              MAXI_SYSCTL_FAN1, VALUE(3), 0 },
    { { SENSORS_MAXI_AS_FAN2, "fan2", NOMAP, NOMAP, R }, 
                              MAXI_SYSCTL_FAN2, VALUE(3), 0 },
    { { SENSORS_MAXI_AS_FAN3, "fan3", NOMAP, NOMAP, R }, 
                              MAXI_SYSCTL_FAN3, VALUE(3), 0 },
    { { SENSORS_MAXI_AS_FAN1_MIN, "fan1_min", SENSORS_MAXI_AS_FAN1, 
                                  SENSORS_MAXI_AS_FAN1, RW }, 
                                  MAXI_SYSCTL_FAN1, VALUE(1), 0 },
    { { SENSORS_MAXI_AS_FAN2_MIN, "fan2_min", SENSORS_MAXI_AS_FAN2, 
                                  SENSORS_MAXI_AS_FAN2, RW }, 
                                  MAXI_SYSCTL_FAN2, VALUE(1), 0 },
    { { SENSORS_MAXI_AS_FAN3_MIN, "fan3_min", SENSORS_MAXI_AS_FAN3, 
                                  SENSORS_MAXI_AS_FAN3, RW }, 
                                  MAXI_SYSCTL_FAN3, VALUE(1), 0 },
    { { SENSORS_MAXI_AS_FAN1_DIV, "fan1_div", SENSORS_MAXI_AS_FAN1, NOMAP, R }, 
                                  MAXI_SYSCTL_FAN1, VALUE(2), 0 },
    { { SENSORS_MAXI_AS_FAN2_DIV, "fan2_div", SENSORS_MAXI_AS_FAN2, NOMAP, R }, 
                                  MAXI_SYSCTL_FAN2, VALUE(2), 0 },
    { { SENSORS_MAXI_AS_FAN3_DIV, "fan3_div", SENSORS_MAXI_AS_FAN3, NOMAP, R }, 
                                  MAXI_SYSCTL_FAN3, VALUE(2), 0 },
    { { SENSORS_MAXI_AS_TEMP1, "temp1", NOMAP, NOMAP, R }, 
                               MAXI_SYSCTL_TEMP1, VALUE(3), 1 },
    { { SENSORS_MAXI_AS_TEMP2, "temp2", NOMAP, NOMAP, R }, 
                               MAXI_SYSCTL_TEMP2, VALUE(3), 1 },
    { { SENSORS_MAXI_AS_TEMP3, "temp3", NOMAP, NOMAP, R }, 
                               MAXI_SYSCTL_TEMP3, VALUE(3), 1 },
    { { SENSORS_MAXI_AS_TEMP4, "temp4", NOMAP, NOMAP, R }, 
                               MAXI_SYSCTL_TEMP4, VALUE(3), 1 },
    { { SENSORS_MAXI_AS_TEMP5, "temp5", NOMAP, NOMAP, R }, 
                               MAXI_SYSCTL_TEMP5, VALUE(3), 1 },
    { { SENSORS_MAXI_AS_TEMP1_MAX, "temp1_max", SENSORS_MAXI_AS_TEMP1, 
                                   SENSORS_MAXI_AS_TEMP1, R }, 
                                   MAXI_SYSCTL_TEMP1, VALUE(1), 1 },
    { { SENSORS_MAXI_AS_TEMP2_MAX, "temp2_max", SENSORS_MAXI_AS_TEMP2, 
                                   SENSORS_MAXI_AS_TEMP2, R }, 
                                   MAXI_SYSCTL_TEMP2, VALUE(1), 1 },
    { { SENSORS_MAXI_AS_TEMP3_MAX, "temp3_max", SENSORS_MAXI_AS_TEMP3, 
                                   SENSORS_MAXI_AS_TEMP3, R }, 
                                   MAXI_SYSCTL_TEMP3, VALUE(1), 1 },
    { { SENSORS_MAXI_AS_TEMP4_MAX, "temp4_max", SENSORS_MAXI_AS_TEMP4, 
                                   SENSORS_MAXI_AS_TEMP4, R }, 
                                   MAXI_SYSCTL_TEMP4, VALUE(1), 1 },
    { { SENSORS_MAXI_AS_TEMP5_MAX, "temp5_max", SENSORS_MAXI_AS_TEMP5, 
                                   SENSORS_MAXI_AS_TEMP5, R }, 
                                   MAXI_SYSCTL_TEMP5, VALUE(1), 1 },
    { { SENSORS_MAXI_AS_TEMP1_HYST, "temp1_hyst", SENSORS_MAXI_AS_TEMP1, 
                                    SENSORS_MAXI_AS_TEMP1, R }, 
                                    MAXI_SYSCTL_TEMP1, VALUE(2), 1 },
    { { SENSORS_MAXI_AS_TEMP2_HYST, "temp2_hyst", SENSORS_MAXI_AS_TEMP2, 
                                    SENSORS_MAXI_AS_TEMP2, R }, 
                                    MAXI_SYSCTL_TEMP2, VALUE(2), 1 },
    { { SENSORS_MAXI_AS_TEMP3_HYST, "temp3_hyst", SENSORS_MAXI_AS_TEMP3, 
                                    SENSORS_MAXI_AS_TEMP3, R }, 
                                    MAXI_SYSCTL_TEMP3, VALUE(2), 1 },
    { { SENSORS_MAXI_AS_TEMP4_HYST, "temp4_hyst", SENSORS_MAXI_AS_TEMP4, 
                                    SENSORS_MAXI_AS_TEMP4, R }, 
                                    MAXI_SYSCTL_TEMP4, VALUE(2), 1 },
    { { SENSORS_MAXI_AS_TEMP5_HYST, "temp5_hyst", SENSORS_MAXI_AS_TEMP5, 
                                    SENSORS_MAXI_AS_TEMP5, R }, 
                                    MAXI_SYSCTL_TEMP5, VALUE(2), 1 },
    { { SENSORS_MAXI_AS_PLL, "pll", NOMAP, NOMAP, R }, 
                             MAXI_SYSCTL_PLL, VALUE(3), 2 },
    { { SENSORS_MAXI_AS_PLL_MIN, "pll_min", SENSORS_MAXI_AS_PLL, 
                                 SENSORS_MAXI_AS_PLL, RW }, 
                                 MAXI_SYSCTL_PLL, VALUE(1), 2 },
    { { SENSORS_MAXI_AS_PLL_MAX, "pll_max", SENSORS_MAXI_AS_PLL, 
                                 SENSORS_MAXI_AS_PLL, RW }, 
                                 MAXI_SYSCTL_PLL, VALUE(2), 2 },
    { { SENSORS_MAXI_AS_VID1, "vid1", NOMAP, NOMAP, R }, 
                              MAXI_SYSCTL_VID1, VALUE(3), 4, "in0_input", 3 },
    { { SENSORS_MAXI_AS_VID2, "vid2", NOMAP, NOMAP, R }, 
                              MAXI_SYSCTL_VID2, VALUE(3), 4, "in1_input", 3 },
    { { SENSORS_MAXI_AS_VID3, "vid3", NOMAP, NOMAP, R }, 
                              MAXI_SYSCTL_VID3, VALUE(3), 4, "in2_input", 3 },
    { { SENSORS_MAXI_AS_VID4, "vid4", NOMAP, NOMAP, R }, 
                              MAXI_SYSCTL_VID4, VALUE(3), 4, "in3_input", 3 },
    { { SENSORS_MAXI_AS_VID1_MIN, "vid1_min", SENSORS_MAXI_AS_VID1, 
                                  SENSORS_MAXI_AS_VID1, RW }, 
                                  MAXI_SYSCTL_VID1, VALUE(1), 4, "in0_min", 3 },
    { { SENSORS_MAXI_AS_VID2_MIN, "vid2_min", SENSORS_MAXI_AS_VID2, 
                                  SENSORS_MAXI_AS_VID2, RW }, 
                                  MAXI_SYSCTL_VID2, VALUE(1), 4, "in1_min", 3 },
    { { SENSORS_MAXI_AS_VID3_MIN, "vid3_min", SENSORS_MAXI_AS_VID3, 
                                  SENSORS_MAXI_AS_VID3, RW }, 
                                  MAXI_SYSCTL_VID3, VALUE(1), 4, "in2_min", 3 },
    { { SENSORS_MAXI_AS_VID4_MIN, "vid4_min", SENSORS_MAXI_AS_VID4, 
                                  SENSORS_MAXI_AS_VID4, RW }, 
                                  MAXI_SYSCTL_VID4, VALUE(1), 4, "in3_min", 3 },
    { { SENSORS_MAXI_AS_VID1_MAX, "vid1_max", SENSORS_MAXI_AS_VID1, 
                                  SENSORS_MAXI_AS_VID1, RW }, 
                                  MAXI_SYSCTL_VID1, VALUE(2), 4, "in0_max", 3 },
    { { SENSORS_MAXI_AS_VID2_MAX, "vid2_max", SENSORS_MAXI_AS_VID2, 
                                  SENSORS_MAXI_AS_VID2, RW }, 
                                  MAXI_SYSCTL_VID2, VALUE(2), 4, "in1_max", 3 },
    { { SENSORS_MAXI_AS_VID3_MAX, "vid3_max", SENSORS_MAXI_AS_VID3, 
                                  SENSORS_MAXI_AS_VID3, RW }, 
                                  MAXI_SYSCTL_VID3, VALUE(2), 4, "in2_max", 3 },
    { { SENSORS_MAXI_AS_VID4_MAX, "vid4_max", SENSORS_MAXI_AS_VID4, 
                                  SENSORS_MAXI_CG_VID4, RW }, 
                                  MAXI_SYSCTL_VID4, VALUE(2), 4, "in3_max", 3 },
    { { SENSORS_MAXI_AS_ALARMS, "alarms", NOMAP, NOMAP, R }, 
                                MAXI_SYSCTL_ALARMS, VALUE(1), 0 },
    { { 0 }, 0 }
  };

static sensors_chip_feature thmc50_features[] =
  {
    { { SENSORS_THMC50_TEMP, "temp", NOMAP, NOMAP, R }, 
                             THMC50_SYSCTL_TEMP, VALUE(3), 0 },
    { { SENSORS_THMC50_TEMP_HYST, "temp_hyst", SENSORS_THMC50_TEMP, 
                                  SENSORS_THMC50_TEMP, RW }, 
                                  THMC50_SYSCTL_TEMP, VALUE(2), 0,
                                  "temp1_min", 3 },
    { { SENSORS_THMC50_TEMP_OVER, "temp_over", SENSORS_THMC50_TEMP, 
                                  SENSORS_THMC50_TEMP, RW }, 
                                  THMC50_SYSCTL_TEMP, VALUE(1), 0 },
    { { SENSORS_THMC50_REMOTE_TEMP, "remote_temp", NOMAP, NOMAP, R }, 
                                    THMC50_SYSCTL_REMOTE_TEMP, VALUE(3), 0 },
    { { SENSORS_THMC50_REMOTE_TEMP_HYST, "remote_temp_hyst", 
                                         SENSORS_THMC50_REMOTE_TEMP, 
                                         SENSORS_THMC50_REMOTE_TEMP, RW }, 
                                         THMC50_SYSCTL_REMOTE_TEMP, VALUE(2), 
                                         0, "temp2_min", 3 },
    { { SENSORS_THMC50_REMOTE_TEMP_OVER, "remote_temp_over", 
                                         SENSORS_THMC50_REMOTE_TEMP, 
                                         SENSORS_THMC50_REMOTE_TEMP, RW }, 
                                         THMC50_SYSCTL_REMOTE_TEMP, VALUE(1), 
                                         0 },
    { { SENSORS_ADM1022_REMOTE_TEMP2, "remote_temp2", NOMAP, NOMAP, R }, 
                                      NOSYSCTL, VALUE(3),
                                      0, "temp3_input", 3 },
    { { SENSORS_ADM1022_REMOTE_TEMP2_HYST, "remote_temp2_hyst", 
                                           SENSORS_ADM1022_REMOTE_TEMP2, 
                                           SENSORS_ADM1022_REMOTE_TEMP2, RW }, 
                                           NOSYSCTL, VALUE(2), 
                                           0, "temp3_min", 3 },
    { { SENSORS_ADM1022_REMOTE_TEMP2_OVER, "remote_temp2_over", 
                                           SENSORS_ADM1022_REMOTE_TEMP2, 
                                           SENSORS_ADM1022_REMOTE_TEMP2, RW }, 
                                           NOSYSCTL, VALUE(1), 
                                           0, "temp3_max", 3 },
    { { SENSORS_THMC50_ANALOG_OUT, "analog output", NOMAP, NOMAP, RW }, 
                                   THMC50_SYSCTL_ANALOG_OUT, VALUE(1), 0,
                                   "pwm1", 0 },
    { { SENSORS_THMC50_INTER, "interrupts", NOMAP, NOMAP, R }, 
                              THMC50_SYSCTL_INTER, VALUE(1), 0 },
    { { SENSORS_THMC50_INTER_MASK, "interrupt mask", NOMAP, NOMAP, RW }, 
                                   THMC50_SYSCTL_INTER_MASK, VALUE(1), 0 },
    { { SENSORS_THMC50_DIE_CODE, "die_code", NOMAP, NOMAP, R }, 
                                 THMC50_SYSCTL_DIE_CODE, VALUE(1), 0 },
    { { 0 }, 0 }
  };

static sensors_chip_feature adm1025_features[] =
  { 
    { { SENSORS_ADM1025_IN0, "in0", NOMAP, NOMAP, R }, 
                             ADM1025_SYSCTL_IN0, VALUE(3), 2 },
    { { SENSORS_ADM1025_IN1, "in1", NOMAP, NOMAP, R }, 
                             ADM1025_SYSCTL_IN1, VALUE(3), 2 },
    { { SENSORS_ADM1025_IN2, "in2", NOMAP, NOMAP, R }, 
                             ADM1025_SYSCTL_IN2, VALUE(3), 2 },
    { { SENSORS_ADM1025_IN3, "in3", NOMAP, NOMAP, R }, 
                             ADM1025_SYSCTL_IN3, VALUE(3), 2 },
    { { SENSORS_ADM1025_IN4, "in4", NOMAP, NOMAP, R }, 
                             ADM1025_SYSCTL_IN4, VALUE(3), 2 },
    { { SENSORS_ADM1025_IN5, "in5", NOMAP, NOMAP, R }, 
                             ADM1025_SYSCTL_IN5, VALUE(3), 2 },
    { { SENSORS_ADM1025_IN0_MIN, "in0_min", SENSORS_ADM1025_IN0, 
                                 SENSORS_ADM1025_IN0, RW }, 
                                 ADM1025_SYSCTL_IN0, VALUE(1), 2 },
    { { SENSORS_ADM1025_IN1_MIN, "in1_min", SENSORS_ADM1025_IN1, 
                                 SENSORS_ADM1025_IN1, RW }, 
                                 ADM1025_SYSCTL_IN1, VALUE(1), 2 },
    { { SENSORS_ADM1025_IN2_MIN, "in2_min", SENSORS_ADM1025_IN2, 
                                 SENSORS_ADM1025_IN2, RW }, 
                                 ADM1025_SYSCTL_IN2, VALUE(1), 2 },
    { { SENSORS_ADM1025_IN3_MIN, "in3_min", SENSORS_ADM1025_IN3, 
                                 SENSORS_ADM1025_IN3, RW }, 
                                 ADM1025_SYSCTL_IN3, VALUE(1), 2 },
    { { SENSORS_ADM1025_IN4_MIN, "in4_min", SENSORS_ADM1025_IN4, 
                                 SENSORS_ADM1025_IN4, RW }, 
                                 ADM1025_SYSCTL_IN4, VALUE(1), 2 },
    { { SENSORS_ADM1025_IN5_MIN, "in5_min", SENSORS_ADM1025_IN5, 
                                 SENSORS_ADM1025_IN5, RW }, 
                                 ADM1025_SYSCTL_IN5, VALUE(1), 2 },
    { { SENSORS_ADM1025_IN0_MAX, "in0_max", SENSORS_ADM1025_IN0, 
                                 SENSORS_ADM1025_IN0, RW }, 
                                 ADM1025_SYSCTL_IN0, VALUE(2), 2 },
    { { SENSORS_ADM1025_IN1_MAX, "in1_max", SENSORS_ADM1025_IN1, 
                                 SENSORS_ADM1025_IN1, RW }, 
                                 ADM1025_SYSCTL_IN1, VALUE(2), 2 },
    { { SENSORS_ADM1025_IN2_MAX, "in2_max", SENSORS_ADM1025_IN2, 
                                 SENSORS_ADM1025_IN2, RW }, 
                                 ADM1025_SYSCTL_IN2, VALUE(2), 2 },
    { { SENSORS_ADM1025_IN3_MAX, "in3_max", SENSORS_ADM1025_IN3, 
                                 SENSORS_ADM1025_IN3, RW }, 
                                 ADM1025_SYSCTL_IN3, VALUE(2), 2 },
    { { SENSORS_ADM1025_IN4_MAX, "in4_max", SENSORS_ADM1025_IN4, 
                                 SENSORS_ADM1025_IN4, RW }, 
                                 ADM1025_SYSCTL_IN4, VALUE(2), 2 },
    { { SENSORS_ADM1025_IN5_MAX, "in5_max", SENSORS_ADM1025_IN5, 
                                 SENSORS_ADM1025_IN5, RW }, 
                                 ADM1025_SYSCTL_IN5, VALUE(2), 2 },
    { { SENSORS_ADM1025_TEMP1, "temp1", NOMAP, NOMAP, R }, 
                               ADM1025_SYSCTL_RTEMP, VALUE(3), 1 },
    { { SENSORS_ADM1025_TEMP1_LOW, "temp1_low", SENSORS_ADM1025_TEMP1, 
                                   SENSORS_ADM1025_TEMP1, RW }, 
                                   ADM1025_SYSCTL_RTEMP, VALUE(2), 1 },
    { { SENSORS_ADM1025_TEMP1_HIGH, "temp1_high", SENSORS_ADM1025_TEMP1, 
                                    SENSORS_ADM1025_TEMP1, RW }, 
                                    ADM1025_SYSCTL_RTEMP, VALUE(1), 1 },
    { { SENSORS_ADM1025_TEMP2, "temp2", NOMAP, NOMAP, R }, 
                               ADM1025_SYSCTL_TEMP, VALUE(3), 1 },
    { { SENSORS_ADM1025_TEMP2_LOW, "temp2_low", SENSORS_ADM1025_TEMP2, 
                                   SENSORS_ADM1025_TEMP2, RW }, 
                                   ADM1025_SYSCTL_TEMP, VALUE(2), 1 },
    { { SENSORS_ADM1025_TEMP2_HIGH, "temp2_high", SENSORS_ADM1025_TEMP2, 
                                    SENSORS_ADM1025_TEMP2, RW }, 
                                    ADM1025_SYSCTL_TEMP, VALUE(1), 1 },
    { { SENSORS_ADM1025_VID, "vid", NOMAP, NOMAP, R }, 
                             ADM1025_SYSCTL_VID, VALUE(1), 3, "cpu0_vid", 3, 
                             "in1_ref" },
    { { SENSORS_ADM1025_VRM, "vrm", NOMAP, NOMAP, RW }, 
                             ADM1025_SYSCTL_VRM, VALUE(1), 1 },
    { { SENSORS_ADM1025_ALARMS, "alarms", NOMAP, NOMAP, R }, 
                                ADM1025_SYSCTL_ALARMS, VALUE(1), 0 },
    { { 0 }, 0 }
  };


static sensors_chip_feature adm1026_features[] = {
    { { SENSORS_ADM1026_ALARMS, "alarms", NOMAP, NOMAP, R }, 
                                ADM1026_SYSCTL_ALARMS, VALUE(1), 0 },
    { { SENSORS_ADM1026_ALARM_MASK, "alarm_mask", SENSORS_ADM1026_ALARMS, NOMAP, 
                                    R }, 
                                    ADM1026_SYSCTL_ALARM_MASK, VALUE(1), 0 },
    { { SENSORS_ADM1026_GPIO, "gpio", NOMAP, NOMAP, R }, 
                              ADM1026_SYSCTL_GPIO, VALUE(1), 0 },
    { { SENSORS_ADM1026_GPIO_MASK, "gpio_mask", SENSORS_ADM1026_GPIO, NOMAP, R }, 
                                   ADM1026_SYSCTL_GPIO_MASK, VALUE(1), 0 },
    { { SENSORS_ADM1026_VID, "vid", NOMAP, NOMAP, R }, 
                             ADM1026_SYSCTL_VID, VALUE(1), 3 },
    { { SENSORS_ADM1026_VRM, "vrm", NOMAP, NOMAP, RW }, 
                             ADM1026_SYSCTL_VRM, VALUE(1), 1 },
    { { SENSORS_ADM1026_PWM, "pwm", NOMAP, NOMAP, RW }, 
                             ADM1026_SYSCTL_PWM, VALUE(1), 0 },
    { { SENSORS_ADM1026_AFC_PWM, "afc_pwm", NOMAP, NOMAP, RW }, 
                                 ADM1026_SYSCTL_PWM, VALUE(1), 0 },
    { { SENSORS_ADM1026_DAC, "analog_out", NOMAP, NOMAP, RW }, 
                             ADM1026_SYSCTL_ANALOG_OUT, VALUE(1), 0 },
    { { SENSORS_ADM1026_AFC_DAC, "afc_analog_out", SENSORS_ADM1026_DAC, NOMAP, 
                                 RW }, 
                                 ADM1026_SYSCTL_AFC, VALUE(2), 0 },

    { { SENSORS_ADM1026_IN0, "in0", NOMAP, NOMAP, R }, 
                             ADM1026_SYSCTL_IN0, VALUE(3), 3 },
    { { SENSORS_ADM1026_IN0_MIN, "in0_min", SENSORS_ADM1026_IN0, 
                                 SENSORS_ADM1026_IN0, RW }, 
                                 ADM1026_SYSCTL_IN0, VALUE(1), 3 },
    { { SENSORS_ADM1026_IN0_MAX, "in0_max", SENSORS_ADM1026_IN0, 
                                 SENSORS_ADM1026_IN0, RW }, 
                                 ADM1026_SYSCTL_IN0, VALUE(2), 3 },
    { { SENSORS_ADM1026_IN1, "in1", NOMAP, NOMAP, R }, 
                             ADM1026_SYSCTL_IN1, VALUE(3), 3 },
    { { SENSORS_ADM1026_IN1_MIN, "in1_min", SENSORS_ADM1026_IN1, 
                                 SENSORS_ADM1026_IN1, RW }, 
                                 ADM1026_SYSCTL_IN1, VALUE(1), 3 },
    { { SENSORS_ADM1026_IN1_MAX, "in1_max", SENSORS_ADM1026_IN1, 
                                 SENSORS_ADM1026_IN1, RW }, 
                                 ADM1026_SYSCTL_IN1, VALUE(2), 3 },
    { { SENSORS_ADM1026_IN2, "in2", NOMAP, NOMAP, R }, 
                             ADM1026_SYSCTL_IN2, VALUE(3), 3 },
    { { SENSORS_ADM1026_IN2_MIN, "in2_min", SENSORS_ADM1026_IN2, 
                                 SENSORS_ADM1026_IN2, RW }, 
                                 ADM1026_SYSCTL_IN2, VALUE(1), 3 },
    { { SENSORS_ADM1026_IN2_MAX, "in2_max", SENSORS_ADM1026_IN2, 
                                 SENSORS_ADM1026_IN2, RW }, 
                                 ADM1026_SYSCTL_IN2, VALUE(2), 3 },
    { { SENSORS_ADM1026_IN3, "in3", NOMAP, NOMAP, R }, 
                             ADM1026_SYSCTL_IN3, VALUE(3), 3 },
    { { SENSORS_ADM1026_IN3_MIN, "in3_min", SENSORS_ADM1026_IN3, 
                                 SENSORS_ADM1026_IN3, RW }, 
                                 ADM1026_SYSCTL_IN3, VALUE(1), 3 },
    { { SENSORS_ADM1026_IN3_MAX, "in3_max", SENSORS_ADM1026_IN3, 
                                 SENSORS_ADM1026_IN3, RW }, 
                                 ADM1026_SYSCTL_IN3, VALUE(2), 3 },
    { { SENSORS_ADM1026_IN4, "in4", NOMAP, NOMAP, R }, 
                             ADM1026_SYSCTL_IN4, VALUE(3), 3 },
    { { SENSORS_ADM1026_IN4_MIN, "in4_min", SENSORS_ADM1026_IN4, 
                                 SENSORS_ADM1026_IN4, RW }, 
                                 ADM1026_SYSCTL_IN4, VALUE(1), 3 },
    { { SENSORS_ADM1026_IN4_MAX, "in4_max", SENSORS_ADM1026_IN4, 
                                 SENSORS_ADM1026_IN4, RW }, 
                                 ADM1026_SYSCTL_IN4, VALUE(2), 3 },
    { { SENSORS_ADM1026_IN5, "in5", NOMAP, NOMAP, R }, 
                             ADM1026_SYSCTL_IN5, VALUE(3), 3 },
    { { SENSORS_ADM1026_IN5_MIN, "in5_min", SENSORS_ADM1026_IN5, 
                                 SENSORS_ADM1026_IN5, RW }, 
                                 ADM1026_SYSCTL_IN5, VALUE(1), 3 },
    { { SENSORS_ADM1026_IN5_MAX, "in5_max", SENSORS_ADM1026_IN5, 
                                 SENSORS_ADM1026_IN5, RW }, 
                                 ADM1026_SYSCTL_IN5, VALUE(2), 3 },
    { { SENSORS_ADM1026_IN6, "in6", NOMAP, NOMAP, R }, 
                             ADM1026_SYSCTL_IN6, VALUE(3), 3 },
    { { SENSORS_ADM1026_IN6_MIN, "in6_min", SENSORS_ADM1026_IN6, 
                                 SENSORS_ADM1026_IN6, RW }, 
                                 ADM1026_SYSCTL_IN6, VALUE(1), 3 },
    { { SENSORS_ADM1026_IN6_MAX, "in6_max", SENSORS_ADM1026_IN6, 
                                 SENSORS_ADM1026_IN6, RW }, 
                                 ADM1026_SYSCTL_IN6, VALUE(2), 3 },
    { { SENSORS_ADM1026_IN7, "in7", NOMAP, NOMAP, R }, 
                             ADM1026_SYSCTL_IN7, VALUE(3), 3 },
    { { SENSORS_ADM1026_IN7_MIN, "in7_min", SENSORS_ADM1026_IN7, 
                                 SENSORS_ADM1026_IN7, RW }, 
                                 ADM1026_SYSCTL_IN7, VALUE(1), 3 },
    { { SENSORS_ADM1026_IN7_MAX, "in7_max", SENSORS_ADM1026_IN7, 
                                 SENSORS_ADM1026_IN7, RW }, 
                                 ADM1026_SYSCTL_IN7, VALUE(2), 3 },
    { { SENSORS_ADM1026_IN8, "in8", NOMAP, NOMAP, R }, 
                             ADM1026_SYSCTL_IN8, VALUE(3), 3 },
    { { SENSORS_ADM1026_IN8_MIN, "in8_min", SENSORS_ADM1026_IN8, 
                                 SENSORS_ADM1026_IN8, RW }, 
                                 ADM1026_SYSCTL_IN8, VALUE(1), 3 },
    { { SENSORS_ADM1026_IN8_MAX, "in8_max", SENSORS_ADM1026_IN8, 
                                 SENSORS_ADM1026_IN8, RW }, 
                                 ADM1026_SYSCTL_IN8, VALUE(2), 3 },
    { { SENSORS_ADM1026_IN9, "in9", NOMAP, NOMAP, R }, 
                             ADM1026_SYSCTL_IN9, VALUE(3), 3 },
    { { SENSORS_ADM1026_IN9_MIN, "in9_min", SENSORS_ADM1026_IN9, 
                                 SENSORS_ADM1026_IN9, RW }, 
                                 ADM1026_SYSCTL_IN9, VALUE(1), 3 },
    { { SENSORS_ADM1026_IN9_MAX, "in9_max", SENSORS_ADM1026_IN9, 
                                 SENSORS_ADM1026_IN9, RW }, 
                                 ADM1026_SYSCTL_IN9, VALUE(2), 3 },
    { { SENSORS_ADM1026_IN10, "in10", NOMAP, NOMAP, R }, 
                              ADM1026_SYSCTL_IN10, VALUE(3), 3 },
    { { SENSORS_ADM1026_IN10_MIN, "in10_min", SENSORS_ADM1026_IN10, 
                                  SENSORS_ADM1026_IN10, RW }, 
                                  ADM1026_SYSCTL_IN10, VALUE(1), 3 },
    { { SENSORS_ADM1026_IN10_MAX, "in10_max", SENSORS_ADM1026_IN10, 
                                  SENSORS_ADM1026_IN10, RW }, 
                                  ADM1026_SYSCTL_IN10, VALUE(2), 3 },
    { { SENSORS_ADM1026_IN11, "in11", NOMAP, NOMAP, R }, 
                              ADM1026_SYSCTL_IN11, VALUE(3), 3 },
    { { SENSORS_ADM1026_IN11_MIN, "in11_min", SENSORS_ADM1026_IN11, 
                                  SENSORS_ADM1026_IN11, RW }, 
                                  ADM1026_SYSCTL_IN11, VALUE(1), 3 },
    { { SENSORS_ADM1026_IN11_MAX, "in11_max", SENSORS_ADM1026_IN11, 
                                  SENSORS_ADM1026_IN11, RW }, 
                                  ADM1026_SYSCTL_IN11, VALUE(2), 3 },
    { { SENSORS_ADM1026_IN12, "in12", NOMAP, NOMAP, R }, 
                              ADM1026_SYSCTL_IN12, VALUE(3), 3 },
    { { SENSORS_ADM1026_IN12_MIN, "in12_min", SENSORS_ADM1026_IN12, 
                                  SENSORS_ADM1026_IN12, RW }, 
                                  ADM1026_SYSCTL_IN12, VALUE(1), 3 },
    { { SENSORS_ADM1026_IN12_MAX, "in12_max", SENSORS_ADM1026_IN12, 
                                  SENSORS_ADM1026_IN12, RW }, 
                                  ADM1026_SYSCTL_IN12, VALUE(2), 3 },
    { { SENSORS_ADM1026_IN13, "in13", NOMAP, NOMAP, R }, 
                              ADM1026_SYSCTL_IN13, VALUE(3), 3 },
    { { SENSORS_ADM1026_IN13_MIN, "in13_min", SENSORS_ADM1026_IN13, 
                                  SENSORS_ADM1026_IN13, RW }, 
                                  ADM1026_SYSCTL_IN13, VALUE(1), 3 },
    { { SENSORS_ADM1026_IN13_MAX, "in13_max", SENSORS_ADM1026_IN13, 
                                  SENSORS_ADM1026_IN13, RW }, 
                                  ADM1026_SYSCTL_IN13, VALUE(2), 3 },
    { { SENSORS_ADM1026_IN14, "in14", NOMAP, NOMAP, R }, 
                              ADM1026_SYSCTL_IN14, VALUE(3), 3 },
    { { SENSORS_ADM1026_IN14_MIN, "in14_min", SENSORS_ADM1026_IN14, 
                                  SENSORS_ADM1026_IN14, RW }, 
                                  ADM1026_SYSCTL_IN14, VALUE(1), 3 },
    { { SENSORS_ADM1026_IN14_MAX, "in14_max", SENSORS_ADM1026_IN14, 
                                  SENSORS_ADM1026_IN14, RW }, 
                                  ADM1026_SYSCTL_IN14, VALUE(2), 3 },
    { { SENSORS_ADM1026_IN15, "in15", NOMAP, NOMAP, R }, 
                              ADM1026_SYSCTL_IN15, VALUE(3), 3 },
    { { SENSORS_ADM1026_IN15_MIN, "in15_min", SENSORS_ADM1026_IN15, 
                                  SENSORS_ADM1026_IN15, RW }, 
                                  ADM1026_SYSCTL_IN15, VALUE(1), 3 },
    { { SENSORS_ADM1026_IN15_MAX, "in15_max", SENSORS_ADM1026_IN15, 
                                  SENSORS_ADM1026_IN15, RW }, 
                                  ADM1026_SYSCTL_IN15, VALUE(2), 3 },
    { { SENSORS_ADM1026_IN16, "in16", NOMAP, NOMAP, R }, 
                              ADM1026_SYSCTL_IN16, VALUE(3), 3 },
    { { SENSORS_ADM1026_IN16_MIN, "in16_min", SENSORS_ADM1026_IN16, 
                                  SENSORS_ADM1026_IN16, RW }, 
                                  ADM1026_SYSCTL_IN16, VALUE(1), 3 },
    { { SENSORS_ADM1026_IN16_MAX, "in16_max", SENSORS_ADM1026_IN16, 
                                  SENSORS_ADM1026_IN16, RW }, 
                                  ADM1026_SYSCTL_IN16, VALUE(2), 3 },
    { { SENSORS_ADM1026_FAN0, "fan0", NOMAP, NOMAP, R }, 
                              ADM1026_SYSCTL_FAN0, VALUE(2), 0, "fan8_input", 
                              0 },
    { { SENSORS_ADM1026_FAN0_MIN, "fan0_min", SENSORS_ADM1026_FAN0, 
                                  SENSORS_ADM1026_FAN0, RW }, 
                                  ADM1026_SYSCTL_FAN0, VALUE(1), 0, "fan8_min", 
                                  0 },
    { { SENSORS_ADM1026_FAN0_DIV, "fan0_div", SENSORS_ADM1026_FAN0, NOMAP, RW }, 
                                  ADM1026_SYSCTL_FAN_DIV, VALUE(1), 0, 
                                  "fan8_div", 0 },
    { { SENSORS_ADM1026_FAN1, "fan1", NOMAP, NOMAP, R }, 
                              ADM1026_SYSCTL_FAN1, VALUE(2), 0 },
    { { SENSORS_ADM1026_FAN1_MIN, "fan1_min", SENSORS_ADM1026_FAN1, 
                                  SENSORS_ADM1026_FAN1, RW }, 
                                  ADM1026_SYSCTL_FAN1, VALUE(1), 0 },
    { { SENSORS_ADM1026_FAN1_DIV, "fan1_div", SENSORS_ADM1026_FAN1, NOMAP, RW }, 
                                  ADM1026_SYSCTL_FAN_DIV, VALUE(2), 0 },
    { { SENSORS_ADM1026_FAN2, "fan2", NOMAP, NOMAP, R }, 
                              ADM1026_SYSCTL_FAN2, VALUE(2), 0 },
    { { SENSORS_ADM1026_FAN2_MIN, "fan2_min", SENSORS_ADM1026_FAN2, 
                                  SENSORS_ADM1026_FAN2, RW }, 
                                  ADM1026_SYSCTL_FAN2, VALUE(1), 0 },
    { { SENSORS_ADM1026_FAN2_DIV, "fan2_div", SENSORS_ADM1026_FAN2, NOMAP, RW }, 
                                  ADM1026_SYSCTL_FAN_DIV, VALUE(3), 0 },
    { { SENSORS_ADM1026_FAN3, "fan3", NOMAP, NOMAP, R }, 
                              ADM1026_SYSCTL_FAN3, VALUE(2), 0 },
    { { SENSORS_ADM1026_FAN3_MIN, "fan3_min", SENSORS_ADM1026_FAN3, 
                                  SENSORS_ADM1026_FAN3, RW }, 
                                  ADM1026_SYSCTL_FAN3, VALUE(1), 0 },
    { { SENSORS_ADM1026_FAN3_DIV, "fan3_div", SENSORS_ADM1026_FAN3, NOMAP, RW }, 
                                  ADM1026_SYSCTL_FAN_DIV, VALUE(4), 0 },
    { { SENSORS_ADM1026_FAN4, "fan4", NOMAP, NOMAP, R }, 
                              ADM1026_SYSCTL_FAN4, VALUE(2), 0 },
    { { SENSORS_ADM1026_FAN4_MIN, "fan4_min", SENSORS_ADM1026_FAN4, 
                                  SENSORS_ADM1026_FAN4, RW }, 
                                  ADM1026_SYSCTL_FAN4, VALUE(1), 0 },
    { { SENSORS_ADM1026_FAN4_DIV, "fan4_div", SENSORS_ADM1026_FAN4, NOMAP, RW }, 
                                  ADM1026_SYSCTL_FAN_DIV, VALUE(5), 0 },
    { { SENSORS_ADM1026_FAN5, "fan5", NOMAP, NOMAP, R }, 
                              ADM1026_SYSCTL_FAN5, VALUE(2), 0 },
    { { SENSORS_ADM1026_FAN5_MIN, "fan5_min", SENSORS_ADM1026_FAN5, 
                                  SENSORS_ADM1026_FAN5, RW }, 
                                  ADM1026_SYSCTL_FAN5, VALUE(1), 0 },
    { { SENSORS_ADM1026_FAN5_DIV, "fan5_div", SENSORS_ADM1026_FAN5, NOMAP, RW }, 
                                  ADM1026_SYSCTL_FAN_DIV, VALUE(6), 0 },
    { { SENSORS_ADM1026_FAN6, "fan6", NOMAP, NOMAP, R }, 
                              ADM1026_SYSCTL_FAN6, VALUE(2), 0 },
    { { SENSORS_ADM1026_FAN6_MIN, "fan6_min", SENSORS_ADM1026_FAN6, 
                                  SENSORS_ADM1026_FAN6, RW }, 
                                  ADM1026_SYSCTL_FAN6, VALUE(1), 0 },
    { { SENSORS_ADM1026_FAN6_DIV, "fan6_div", SENSORS_ADM1026_FAN6, NOMAP, RW }, 
                                  ADM1026_SYSCTL_FAN_DIV, VALUE(7), 0 },
    { { SENSORS_ADM1026_FAN7, "fan7", NOMAP, NOMAP, R }, 
                              ADM1026_SYSCTL_FAN7, VALUE(2), 0 },
    { { SENSORS_ADM1026_FAN7_MIN, "fan7_min", SENSORS_ADM1026_FAN7, 
                                  SENSORS_ADM1026_FAN7, RW }, 
                                  ADM1026_SYSCTL_FAN7, VALUE(1), 0 },
    { { SENSORS_ADM1026_FAN7_DIV, "fan7_div", SENSORS_ADM1026_FAN7, NOMAP, RW }, 
                                  ADM1026_SYSCTL_FAN_DIV, VALUE(8), 0 },
    { { SENSORS_ADM1026_TEMP1, "temp1", NOMAP, NOMAP, R }, 
                               ADM1026_SYSCTL_TEMP1, VALUE(3), 0 },
    { { SENSORS_ADM1026_TEMP1_MIN, "temp1_min", SENSORS_ADM1026_TEMP1, 
                                   SENSORS_ADM1026_TEMP1, RW }, 
                                   ADM1026_SYSCTL_TEMP1, VALUE(1), 0 },
    { { SENSORS_ADM1026_TEMP1_MAX, "temp1_max", SENSORS_ADM1026_TEMP1, 
                                   SENSORS_ADM1026_TEMP1, RW }, 
                                   ADM1026_SYSCTL_TEMP1, VALUE(2), 0 },
    { { SENSORS_ADM1026_TEMP1_OFFSET, "temp1_offset", SENSORS_ADM1026_TEMP1, 
                                      NOMAP, RW }, 
                                      ADM1026_SYSCTL_TEMP_OFFSET1, VALUE(1), 0 },
    { { SENSORS_ADM1026_TEMP1_TMIN, "temp1_tmin", SENSORS_ADM1026_TEMP1, 
                                    SENSORS_ADM1026_TEMP1, RW }, 
                                    ADM1026_SYSCTL_TEMP_TMIN1, VALUE(1), 0 },
    { { SENSORS_ADM1026_TEMP1_THERM, "temp1_therm", SENSORS_ADM1026_TEMP1, 
                                     SENSORS_ADM1026_TEMP1, RW }, 
                                     ADM1026_SYSCTL_TEMP_THERM1, VALUE(1), 0, "temp1_crit", 3 },
    { { SENSORS_ADM1026_TEMP2, "temp2", NOMAP, NOMAP, R }, 
                               ADM1026_SYSCTL_TEMP2, VALUE(3), 0 },
    { { SENSORS_ADM1026_TEMP2_MIN, "temp2_min", SENSORS_ADM1026_TEMP2, 
                                   SENSORS_ADM1026_TEMP2, RW }, 
                                   ADM1026_SYSCTL_TEMP2, VALUE(1), 0 },
    { { SENSORS_ADM1026_TEMP2_MAX, "temp2_max", SENSORS_ADM1026_TEMP2, 
                                   SENSORS_ADM1026_TEMP2, RW }, 
                                   ADM1026_SYSCTL_TEMP2, VALUE(2), 0 },
    { { SENSORS_ADM1026_TEMP2_OFFSET, "temp2_offset", SENSORS_ADM1026_TEMP2, 
                                      NOMAP, RW }, 
                                      ADM1026_SYSCTL_TEMP_OFFSET2, VALUE(1), 0 },
    { { SENSORS_ADM1026_TEMP2_TMIN, "temp2_tmin", SENSORS_ADM1026_TEMP2, 
                                    SENSORS_ADM1026_TEMP2, RW }, 
                                    ADM1026_SYSCTL_TEMP_TMIN2, VALUE(1), 0 },
    { { SENSORS_ADM1026_TEMP2_THERM, "temp2_therm", SENSORS_ADM1026_TEMP2, 
                                     SENSORS_ADM1026_TEMP2, RW }, 
                                     ADM1026_SYSCTL_TEMP_THERM2, VALUE(1), 0, "temp2_crit", 3 },
    { { SENSORS_ADM1026_TEMP3, "temp3", NOMAP, NOMAP, R }, 
                               ADM1026_SYSCTL_TEMP3, VALUE(3), 0 },
    { { SENSORS_ADM1026_TEMP3_MIN, "temp3_min", SENSORS_ADM1026_TEMP3, 
                                   SENSORS_ADM1026_TEMP3, RW }, 
                                   ADM1026_SYSCTL_TEMP3, VALUE(1), 0 },
    { { SENSORS_ADM1026_TEMP3_MAX, "temp3_max", SENSORS_ADM1026_TEMP3, 
                                   SENSORS_ADM1026_TEMP3, RW }, 
                                   ADM1026_SYSCTL_TEMP3, VALUE(2), 0 },
    { { SENSORS_ADM1026_TEMP3_OFFSET, "temp3_offset", SENSORS_ADM1026_TEMP3, 
                                      NOMAP, RW }, 
                                      ADM1026_SYSCTL_TEMP_OFFSET3, VALUE(1), 0 },
    { { SENSORS_ADM1026_TEMP3_TMIN, "temp3_tmin", SENSORS_ADM1026_TEMP3, 
                                    SENSORS_ADM1026_TEMP3, RW }, 
                                    ADM1026_SYSCTL_TEMP_TMIN3, VALUE(1), 0 },
    { { SENSORS_ADM1026_TEMP3_THERM, "temp3_therm", SENSORS_ADM1026_TEMP3, 
                                     SENSORS_ADM1026_TEMP3, RW }, 
                                     ADM1026_SYSCTL_TEMP_THERM3, VALUE(1), 0, "temp3_crit", 3 },
    { { 0 }, 0 }
  };


static sensors_chip_feature via686a_features[] =
  { 
    { { SENSORS_VIA686A_IN0, "2.0V", NOMAP, NOMAP, R }, 
                             VIA686A_SYSCTL_IN0, VALUE(3), 2, "in0_input", 3 },
    { { SENSORS_VIA686A_IN1, "2.5V", NOMAP, NOMAP, R }, 
                             VIA686A_SYSCTL_IN1, VALUE(3), 2, "in1_input", 3 },
    { { SENSORS_VIA686A_IN2, "3.3V", NOMAP, NOMAP, R }, 
                             VIA686A_SYSCTL_IN2, VALUE(3), 2, "in2_input", 3 },
    { { SENSORS_VIA686A_IN3, "5.0V", NOMAP, NOMAP, R }, 
                             VIA686A_SYSCTL_IN3, VALUE(3), 2, "in3_input", 3 },
    { { SENSORS_VIA686A_IN4, "12V", NOMAP, NOMAP, R }, 
                             VIA686A_SYSCTL_IN4, VALUE(3), 2, "in4_input", 3 },
    { { SENSORS_VIA686A_IN0_MIN, "in0_min", SENSORS_VIA686A_IN0, 
                                 SENSORS_VIA686A_IN0, RW }, 
                                 VIA686A_SYSCTL_IN0, VALUE(1), 2 },
    { { SENSORS_VIA686A_IN1_MIN, "in1_min", SENSORS_VIA686A_IN1, 
                                 SENSORS_VIA686A_IN1, RW }, 
                                 VIA686A_SYSCTL_IN1, VALUE(1), 2 },
    { { SENSORS_VIA686A_IN2_MIN, "in2_min", SENSORS_VIA686A_IN2, 
                                 SENSORS_VIA686A_IN2, RW }, 
                                 VIA686A_SYSCTL_IN2, VALUE(1), 2 },
    { { SENSORS_VIA686A_IN3_MIN, "in3_min", SENSORS_VIA686A_IN3, 
                                 SENSORS_VIA686A_IN3, RW }, 
                                 VIA686A_SYSCTL_IN3, VALUE(1), 2 },
    { { SENSORS_VIA686A_IN4_MIN, "in4_min", SENSORS_VIA686A_IN4, 
                                 SENSORS_VIA686A_IN4, RW }, 
                                 VIA686A_SYSCTL_IN4, VALUE(1), 2 },
    { { SENSORS_VIA686A_IN0_MAX, "in0_max", SENSORS_VIA686A_IN0, 
                                 SENSORS_VIA686A_IN0, RW }, 
                                 VIA686A_SYSCTL_IN0, VALUE(2), 2 },
    { { SENSORS_VIA686A_IN1_MAX, "in1_max", SENSORS_VIA686A_IN1, 
                                 SENSORS_VIA686A_IN1, RW }, 
                                 VIA686A_SYSCTL_IN1, VALUE(2), 2 },
    { { SENSORS_VIA686A_IN2_MAX, "in2_max", SENSORS_VIA686A_IN2, 
                                 SENSORS_VIA686A_IN2, RW }, 
                                 VIA686A_SYSCTL_IN2, VALUE(2), 2 },
    { { SENSORS_VIA686A_IN3_MAX, "in3_max", SENSORS_VIA686A_IN3, 
                                 SENSORS_VIA686A_IN3, RW }, 
                                 VIA686A_SYSCTL_IN3, VALUE(2), 2 },
    { { SENSORS_VIA686A_IN4_MAX, "in4_max", SENSORS_VIA686A_IN4, 
                                 SENSORS_VIA686A_IN4, RW }, 
                                 VIA686A_SYSCTL_IN4, VALUE(2), 2 },
    { { SENSORS_VIA686A_FAN1, "fan1", NOMAP, NOMAP, R }, 
                              VIA686A_SYSCTL_FAN1, VALUE(2), 0 },
    { { SENSORS_VIA686A_FAN2, "fan2", NOMAP, NOMAP, R }, 
                              VIA686A_SYSCTL_FAN2, VALUE(2), 0 },
    { { SENSORS_VIA686A_FAN1_MIN, "fan1_min", SENSORS_VIA686A_FAN1, 
                                  SENSORS_VIA686A_FAN1, RW }, 
                                  VIA686A_SYSCTL_FAN1, VALUE(1), 0 },
    { { SENSORS_VIA686A_FAN2_MIN, "fan2_min", SENSORS_VIA686A_FAN2, 
                                  SENSORS_VIA686A_FAN2, RW }, 
                                  VIA686A_SYSCTL_FAN2, VALUE(1), 0 },
    { { SENSORS_VIA686A_TEMP, "temp1", NOMAP, NOMAP, R }, 
                              VIA686A_SYSCTL_TEMP, VALUE(3), 1 },
    { { SENSORS_VIA686A_TEMP_HYST, "temp1_hyst", SENSORS_VIA686A_TEMP, 
                                   SENSORS_VIA686A_TEMP, RW }, 
                                   VIA686A_SYSCTL_TEMP, VALUE(2), 1 },
    { { SENSORS_VIA686A_TEMP_OVER, "temp1_over", SENSORS_VIA686A_TEMP, 
                                   SENSORS_VIA686A_TEMP, RW }, 
                                   VIA686A_SYSCTL_TEMP, VALUE(1), 1 },
    { { SENSORS_VIA686A_TEMP2, "temp2", NOMAP, NOMAP, R }, 
                               VIA686A_SYSCTL_TEMP2, VALUE(3), 1 },
    { { SENSORS_VIA686A_TEMP2_HYST, "temp2_hyst", SENSORS_VIA686A_TEMP2, 
                                    SENSORS_VIA686A_TEMP2, RW }, 
                                    VIA686A_SYSCTL_TEMP2, VALUE(2), 1 },
    { { SENSORS_VIA686A_TEMP2_OVER, "temp2_over", SENSORS_VIA686A_TEMP2, 
                                    SENSORS_VIA686A_TEMP2, RW }, 
                                    VIA686A_SYSCTL_TEMP2, VALUE(1), 1 },
    { { SENSORS_VIA686A_TEMP3, "temp3", NOMAP, NOMAP, R }, 
                               VIA686A_SYSCTL_TEMP3, VALUE(3), 1 },
    { { SENSORS_VIA686A_TEMP3_HYST, "temp3_hyst", SENSORS_VIA686A_TEMP3, 
                                    SENSORS_VIA686A_TEMP3, RW }, 
                                    VIA686A_SYSCTL_TEMP3, VALUE(2), 1 },
    { { SENSORS_VIA686A_TEMP3_OVER, "temp3_over", SENSORS_VIA686A_TEMP3, 
                                    SENSORS_VIA686A_TEMP3, RW }, 
                                    VIA686A_SYSCTL_TEMP3, VALUE(1), 1 },
    { { SENSORS_VIA686A_FAN1_DIV, "fan1_div", SENSORS_VIA686A_FAN1, NOMAP, RW }, 
                                  VIA686A_SYSCTL_FAN_DIV, VALUE(1), 0 },
    { { SENSORS_VIA686A_FAN2_DIV, "fan2_div", SENSORS_VIA686A_FAN2, NOMAP, RW }, 
                                  VIA686A_SYSCTL_FAN_DIV, VALUE(2), 0 },
    { { SENSORS_VIA686A_ALARMS, "alarms", NOMAP, NOMAP, R }, 
                                VIA686A_SYSCTL_ALARMS, VALUE(1) },
    { { 0 }, 0 }
  };

static sensors_chip_feature ddcmon_features[] =
  {
    { { SENSORS_DDCMON_MAN_ID, "Manufacturer ID", NOMAP, NOMAP, R }, 
                               DDCMON_SYSCTL_ID, VALUE(1), 0 },
    { { SENSORS_DDCMON_PROD_ID, "Model Number", NOMAP, NOMAP, R }, 
                                DDCMON_SYSCTL_ID, VALUE(2), 0 },
    { { SENSORS_DDCMON_VERSIZE, "Monitor Size (cm)", NOMAP, NOMAP, R }, 
                                DDCMON_SYSCTL_SIZE, VALUE(1), 0 },
    { { SENSORS_DDCMON_HORSIZE, "hsize", SENSORS_DDCMON_VERSIZE, NOMAP, R }, 
                                DDCMON_SYSCTL_SIZE, VALUE(2), 0 },
    { { SENSORS_DDCMON_VERSYNCMIN, "Vertical Sync (Hz)", NOMAP, NOMAP, R }, 
                                   DDCMON_SYSCTL_SYNC, VALUE(1), 0 },
    { { SENSORS_DDCMON_VERSYNCMAX, "v_sync_max", SENSORS_DDCMON_VERSYNCMIN, 
                                   NOMAP, R }, 
                                   DDCMON_SYSCTL_SYNC, VALUE(2), 0 },
    { { SENSORS_DDCMON_HORSYNCMIN, "Horizontal Sync (KHz)", NOMAP, NOMAP, R }, 
                                   DDCMON_SYSCTL_SYNC, VALUE(3), 0 },
    { { SENSORS_DDCMON_HORSYNCMAX, "h_sync_max", SENSORS_DDCMON_HORSYNCMIN, 
                                   NOMAP, R }, 
                                   DDCMON_SYSCTL_SYNC, VALUE(4), 0 },
    { { SENSORS_DDCMON_TIMINGS, "Established Timings", NOMAP, NOMAP, R }, 
                                DDCMON_SYSCTL_TIMINGS, VALUE(1), 0 },
    { { SENSORS_DDCMON_SERIAL, "Serial Number", NOMAP, NOMAP, R }, 
                               DDCMON_SYSCTL_SERIAL, VALUE(1), 0 },
    { { SENSORS_DDCMON_YEAR, "Manufacture Time", NOMAP, NOMAP, R }, 
                             DDCMON_SYSCTL_TIME, VALUE(1), 0 },
    { { SENSORS_DDCMON_WEEK, "week", SENSORS_DDCMON_YEAR, NOMAP, R }, 
                             DDCMON_SYSCTL_TIME, VALUE(2), 0 },
    { { SENSORS_DDCMON_EDID_VER, "EDID Version", NOMAP, NOMAP, R }, 
                                 DDCMON_SYSCTL_EDID, VALUE(1), 0 },
    { { SENSORS_DDCMON_EDID_REV, "revision", SENSORS_DDCMON_EDID_VER, NOMAP, R }, 
                                 DDCMON_SYSCTL_EDID, VALUE(2), 0 },
    { { SENSORS_DDCMON_GAMMA, "Gamma Factor", NOMAP, NOMAP, R }, 
                              DDCMON_SYSCTL_GAMMA, VALUE(1), 2 },
    { { SENSORS_DDCMON_DPMS, "DPMS Modes", NOMAP, NOMAP, R }, 
                             DDCMON_SYSCTL_DPMS, VALUE(1), 0 },
    { { SENSORS_DDCMON_TIMING1_HOR, "Standard Timing 1", NOMAP, NOMAP, R }, 
                                    DDCMON_SYSCTL_TIMING1, VALUE(1), 0 },
    { { SENSORS_DDCMON_TIMING1_VER, "vertical", SENSORS_DDCMON_TIMING1_HOR, 
                                    NOMAP, R }, 
                                    DDCMON_SYSCTL_TIMING1, VALUE(2), 0 },
    { { SENSORS_DDCMON_TIMING1_REF, "refresh", SENSORS_DDCMON_TIMING1_HOR, NOMAP, 
                                    R }, 
                                    DDCMON_SYSCTL_TIMING1, VALUE(3), 0 },
    { { SENSORS_DDCMON_TIMING2_HOR, "Standard Timing 2", NOMAP, NOMAP, R }, 
                                    DDCMON_SYSCTL_TIMING2, VALUE(1), 0 },
    { { SENSORS_DDCMON_TIMING2_VER, "vertical", SENSORS_DDCMON_TIMING2_HOR, 
                                    NOMAP, R }, 
                                    DDCMON_SYSCTL_TIMING2, VALUE(2), 0 },
    { { SENSORS_DDCMON_TIMING2_REF, "refresh", SENSORS_DDCMON_TIMING2_HOR, NOMAP, 
                                    R }, 
                                    DDCMON_SYSCTL_TIMING2, VALUE(3), 0 },
    { { SENSORS_DDCMON_TIMING3_HOR, "Standard Timing 3", NOMAP, NOMAP, R }, 
                                    DDCMON_SYSCTL_TIMING3, VALUE(1), 0 },
    { { SENSORS_DDCMON_TIMING3_VER, "vertical", SENSORS_DDCMON_TIMING3_HOR, 
                                    NOMAP, R }, 
                                    DDCMON_SYSCTL_TIMING3, VALUE(2), 0 },
    { { SENSORS_DDCMON_TIMING3_REF, "refresh", SENSORS_DDCMON_TIMING3_HOR, NOMAP, 
                                    R }, 
                                    DDCMON_SYSCTL_TIMING3, VALUE(3), 0 },
    { { SENSORS_DDCMON_TIMING4_HOR, "Standard Timing 4", NOMAP, NOMAP, R }, 
                                    DDCMON_SYSCTL_TIMING4, VALUE(1), 0 },
    { { SENSORS_DDCMON_TIMING4_VER, "vertical", SENSORS_DDCMON_TIMING4_HOR, 
                                    NOMAP, R }, 
                                    DDCMON_SYSCTL_TIMING4, VALUE(2), 0 },
    { { SENSORS_DDCMON_TIMING4_REF, "refresh", SENSORS_DDCMON_TIMING4_HOR, NOMAP, 
                                    R }, 
                                    DDCMON_SYSCTL_TIMING4, VALUE(3), 0 },
    { { SENSORS_DDCMON_TIMING5_HOR, "Standard Timing 5", NOMAP, NOMAP, R }, 
                                    DDCMON_SYSCTL_TIMING5, VALUE(1), 0 },
    { { SENSORS_DDCMON_TIMING5_VER, "vertical", SENSORS_DDCMON_TIMING5_HOR, 
                                    NOMAP, R }, 
                                    DDCMON_SYSCTL_TIMING5, VALUE(2), 0 },
    { { SENSORS_DDCMON_TIMING5_REF, "refresh", SENSORS_DDCMON_TIMING5_HOR, NOMAP, 
                                    R }, 
                                    DDCMON_SYSCTL_TIMING5, VALUE(3), 0 },
    { { SENSORS_DDCMON_TIMING6_HOR, "Standard Timing 6", NOMAP, NOMAP, R }, 
                                    DDCMON_SYSCTL_TIMING6, VALUE(1), 0 },
    { { SENSORS_DDCMON_TIMING6_VER, "vertical", SENSORS_DDCMON_TIMING6_HOR, 
                                    NOMAP, R }, 
                                    DDCMON_SYSCTL_TIMING6, VALUE(2), 0 },
    { { SENSORS_DDCMON_TIMING6_REF, "refresh", SENSORS_DDCMON_TIMING6_HOR, NOMAP, 
                                    R }, 
                                    DDCMON_SYSCTL_TIMING6, VALUE(3), 0 },
    { { SENSORS_DDCMON_TIMING7_HOR, "Standard Timing 7", NOMAP, NOMAP, R }, 
                                    DDCMON_SYSCTL_TIMING7, VALUE(1), 0 },
    { { SENSORS_DDCMON_TIMING7_VER, "vertical", SENSORS_DDCMON_TIMING7_HOR, 
                                    NOMAP, R }, 
                                    DDCMON_SYSCTL_TIMING7, VALUE(2), 0 },
    { { SENSORS_DDCMON_TIMING7_REF, "refresh", SENSORS_DDCMON_TIMING7_HOR, NOMAP, 
                                    R }, 
                                    DDCMON_SYSCTL_TIMING7, VALUE(3), 0 },
    { { SENSORS_DDCMON_TIMING8_HOR, "Standard Timing 8", NOMAP, NOMAP, R }, 
                                    DDCMON_SYSCTL_TIMING8, VALUE(1), 0 },
    { { SENSORS_DDCMON_TIMING8_VER, "vertical", SENSORS_DDCMON_TIMING8_HOR, 
                                    NOMAP, R }, 
                                    DDCMON_SYSCTL_TIMING8, VALUE(2), 0 },
    { { SENSORS_DDCMON_TIMING8_REF, "refresh", SENSORS_DDCMON_TIMING8_HOR, NOMAP, 
                                    R }, 
                                    DDCMON_SYSCTL_TIMING8, VALUE(3), 0 },
    { { SENSORS_DDCMON_MAXCLOCK, "Max Pixel Clock (MHz)", NOMAP, NOMAP, R }, 
                                 DDCMON_SYSCTL_MAXCLOCK, VALUE(1), 0 },
    { { 0 }, 0 }
  };

static sensors_chip_feature eeprom_features[] =
  {
    { { SENSORS_EEPROM_TYPE, "Memory type", NOMAP, NOMAP, R }, 
                             EEPROM_SYSCTL1, VALUE(3), 0 },
/* cheat and use the row-address-bits label for the total size in sensors */
    { { SENSORS_EEPROM_ROWADDR, "Memory size (MB)", NOMAP, NOMAP, R }, 
                                EEPROM_SYSCTL1, VALUE(4), 0 },
    { { SENSORS_EEPROM_COLADDR, "col", NOMAP, NOMAP, R }, 
                                EEPROM_SYSCTL1, VALUE(5), 0 },
    { { SENSORS_EEPROM_NUMROWS, "num rows", NOMAP, NOMAP, R }, 
                                EEPROM_SYSCTL1, VALUE(6), 0 },
    { { SENSORS_EEPROM_BANKS, "banks", NOMAP, NOMAP, R }, 
                              EEPROM_SYSCTL2, VALUE(2), 0 },
/* for Sony Vaio's eeproms */
    { { SENSORS_EEPROM_VAIO_NAME, "Machine name", NOMAP, NOMAP, R }, 
                                  EEPROM_SYSCTL9, VALUE(1), 0 },
    { { SENSORS_EEPROM_VAIO_NAME+1, "name2", NOMAP, NOMAP, R }, 
                                    EEPROM_SYSCTL9, VALUE(2), 0 },
    { { SENSORS_EEPROM_VAIO_NAME+2, "name3", NOMAP, NOMAP, R }, 
                                    EEPROM_SYSCTL9, VALUE(3), 0 },
    { { SENSORS_EEPROM_VAIO_NAME+3, "name4", NOMAP, NOMAP, R }, 
                                    EEPROM_SYSCTL9, VALUE(4), 0 },
    { { SENSORS_EEPROM_VAIO_NAME+4, "name5", NOMAP, NOMAP, R }, 
                                    EEPROM_SYSCTL9, VALUE(5), 0 },
    { { SENSORS_EEPROM_VAIO_NAME+5, "name6", NOMAP, NOMAP, R }, 
                                    EEPROM_SYSCTL9, VALUE(6), 0 },
    { { SENSORS_EEPROM_VAIO_NAME+6, "name7", NOMAP, NOMAP, R }, 
                                    EEPROM_SYSCTL9, VALUE(7), 0 },
    { { SENSORS_EEPROM_VAIO_NAME+7, "name8", NOMAP, NOMAP, R }, 
                                    EEPROM_SYSCTL9, VALUE(8), 0 },
    { { SENSORS_EEPROM_VAIO_NAME+8, "name9", NOMAP, NOMAP, R }, 
                                    EEPROM_SYSCTL9, VALUE(9), 0 },
    { { SENSORS_EEPROM_VAIO_NAME+9, "name10", NOMAP, NOMAP, R }, 
                                    EEPROM_SYSCTL9, VALUE(10), 0 },
    { { SENSORS_EEPROM_VAIO_NAME+10, "name11", NOMAP, NOMAP, R }, 
                                     EEPROM_SYSCTL9, VALUE(11), 0 },
    { { SENSORS_EEPROM_VAIO_NAME+11, "name12", NOMAP, NOMAP, R }, 
                                     EEPROM_SYSCTL9, VALUE(12), 0 },
    { { SENSORS_EEPROM_VAIO_NAME+12, "name13", NOMAP, NOMAP, R }, 
                                     EEPROM_SYSCTL9, VALUE(13), 0 },
    { { SENSORS_EEPROM_VAIO_NAME+13, "name14", NOMAP, NOMAP, R }, 
                                     EEPROM_SYSCTL9, VALUE(14), 0 },
    { { SENSORS_EEPROM_VAIO_NAME+14, "name15", NOMAP, NOMAP, R }, 
                                     EEPROM_SYSCTL9, VALUE(15), 0 },
    { { SENSORS_EEPROM_VAIO_NAME+15, "name16", NOMAP, NOMAP, R }, 
                                     EEPROM_SYSCTL9, VALUE(16), 0 },
    { { SENSORS_EEPROM_VAIO_NAME+16, "name17", NOMAP, NOMAP, R }, 
                                     EEPROM_SYSCTL10, VALUE(1), 0 },
    { { SENSORS_EEPROM_VAIO_NAME+17, "name18", NOMAP, NOMAP, R }, 
                                     EEPROM_SYSCTL10, VALUE(2), 0 },
    { { SENSORS_EEPROM_VAIO_NAME+18, "name19", NOMAP, NOMAP, R }, 
                                     EEPROM_SYSCTL10, VALUE(3), 0 },
    { { SENSORS_EEPROM_VAIO_NAME+19, "name20", NOMAP, NOMAP, R }, 
                                     EEPROM_SYSCTL10, VALUE(4), 0 },
    { { SENSORS_EEPROM_VAIO_NAME+20, "name21", NOMAP, NOMAP, R }, 
                                     EEPROM_SYSCTL10, VALUE(5), 0 },
    { { SENSORS_EEPROM_VAIO_NAME+21, "name22", NOMAP, NOMAP, R }, 
                                     EEPROM_SYSCTL10, VALUE(6), 0 },
    { { SENSORS_EEPROM_VAIO_NAME+22, "name23", NOMAP, NOMAP, R }, 
                                     EEPROM_SYSCTL10, VALUE(7), 0 },
    { { SENSORS_EEPROM_VAIO_NAME+23, "name24", NOMAP, NOMAP, R }, 
                                     EEPROM_SYSCTL10, VALUE(8), 0 },
    { { SENSORS_EEPROM_VAIO_NAME+24, "name25", NOMAP, NOMAP, R }, 
                                     EEPROM_SYSCTL10, VALUE(9), 0 },
    { { SENSORS_EEPROM_VAIO_NAME+25, "name26", NOMAP, NOMAP, R }, 
                                     EEPROM_SYSCTL10, VALUE(10), 0 },
    { { SENSORS_EEPROM_VAIO_NAME+26, "name27", NOMAP, NOMAP, R }, 
                                     EEPROM_SYSCTL10, VALUE(11), 0 },
    { { SENSORS_EEPROM_VAIO_NAME+27, "name28", NOMAP, NOMAP, R }, 
                                     EEPROM_SYSCTL10, VALUE(12), 0 },
    { { SENSORS_EEPROM_VAIO_NAME+28, "name29", NOMAP, NOMAP, R }, 
                                     EEPROM_SYSCTL10, VALUE(13), 0 },
    { { SENSORS_EEPROM_VAIO_NAME+29, "name30", NOMAP, NOMAP, R }, 
                                     EEPROM_SYSCTL10, VALUE(14), 0 },
    { { SENSORS_EEPROM_VAIO_NAME+30, "name31", NOMAP, NOMAP, R }, 
                                     EEPROM_SYSCTL10, VALUE(15), 0 },
    { { SENSORS_EEPROM_VAIO_NAME+31, "name32", NOMAP, NOMAP, R }, 
                                     EEPROM_SYSCTL10, VALUE(16), 0 },
    { { SENSORS_EEPROM_VAIO_SERIAL, "Serial number", NOMAP, NOMAP, R }, 
                                    EEPROM_SYSCTL13, VALUE(1), 0 },
    { { SENSORS_EEPROM_VAIO_SERIAL+1, "serial2", NOMAP, NOMAP, R }, 
                                      EEPROM_SYSCTL13, VALUE(2), 0 },
    { { SENSORS_EEPROM_VAIO_SERIAL+2, "serial3", NOMAP, NOMAP, R }, 
                                      EEPROM_SYSCTL13, VALUE(3), 0 },
    { { SENSORS_EEPROM_VAIO_SERIAL+3, "serial4", NOMAP, NOMAP, R }, 
                                      EEPROM_SYSCTL13, VALUE(4), 0 },
    { { SENSORS_EEPROM_VAIO_SERIAL+4, "serial5", NOMAP, NOMAP, R }, 
                                      EEPROM_SYSCTL13, VALUE(5), 0 },
    { { SENSORS_EEPROM_VAIO_SERIAL+5, "serial6", NOMAP, NOMAP, R }, 
                                      EEPROM_SYSCTL13, VALUE(6), 0 },
    { { SENSORS_EEPROM_VAIO_SERIAL+6, "serial7", NOMAP, NOMAP, R }, 
                                      EEPROM_SYSCTL13, VALUE(7), 0 },
    { { SENSORS_EEPROM_VAIO_SERIAL+7, "serial8", NOMAP, NOMAP, R }, 
                                      EEPROM_SYSCTL13, VALUE(8), 0 },
    { { SENSORS_EEPROM_VAIO_SERIAL+8, "serial9", NOMAP, NOMAP, R }, 
                                      EEPROM_SYSCTL13, VALUE(9), 0 },
    { { SENSORS_EEPROM_VAIO_SERIAL+9, "serial10", NOMAP, NOMAP, R }, 
                                      EEPROM_SYSCTL13, VALUE(10), 0 },
    { { SENSORS_EEPROM_VAIO_SERIAL+10, "serial11", NOMAP, NOMAP, R }, 
                                       EEPROM_SYSCTL13, VALUE(11), 0 },
    { { SENSORS_EEPROM_VAIO_SERIAL+11, "serial12", NOMAP, NOMAP, R }, 
                                       EEPROM_SYSCTL13, VALUE(12), 0 },
    { { SENSORS_EEPROM_VAIO_SERIAL+12, "serial13", NOMAP, NOMAP, R }, 
                                       EEPROM_SYSCTL13, VALUE(13), 0 },
    { { SENSORS_EEPROM_VAIO_SERIAL+13, "serial14", NOMAP, NOMAP, R }, 
                                       EEPROM_SYSCTL13, VALUE(14), 0 },
    { { SENSORS_EEPROM_VAIO_SERIAL+14, "serial15", NOMAP, NOMAP, R }, 
                                       EEPROM_SYSCTL13, VALUE(15), 0 },
    { { SENSORS_EEPROM_VAIO_SERIAL+15, "serial16", NOMAP, NOMAP, R }, 
                                       EEPROM_SYSCTL13, VALUE(16), 0 },
    { { SENSORS_EEPROM_VAIO_SERIAL+16, "serial17", NOMAP, NOMAP, R }, 
                                       EEPROM_SYSCTL14, VALUE(1), 0 },
    { { SENSORS_EEPROM_VAIO_SERIAL+17, "serial18", NOMAP, NOMAP, R }, 
                                       EEPROM_SYSCTL14, VALUE(2), 0 },
    { { SENSORS_EEPROM_VAIO_SERIAL+18, "serial19", NOMAP, NOMAP, R }, 
                                       EEPROM_SYSCTL14, VALUE(3), 0 },
    { { SENSORS_EEPROM_VAIO_SERIAL+19, "serial20", NOMAP, NOMAP, R }, 
                                       EEPROM_SYSCTL14, VALUE(4), 0 },
    { { SENSORS_EEPROM_VAIO_SERIAL+20, "serial21", NOMAP, NOMAP, R }, 
                                       EEPROM_SYSCTL14, VALUE(5), 0 },
    { { SENSORS_EEPROM_VAIO_SERIAL+21, "serial22", NOMAP, NOMAP, R }, 
                                       EEPROM_SYSCTL14, VALUE(6), 0 },
    { { SENSORS_EEPROM_VAIO_SERIAL+22, "serial23", NOMAP, NOMAP, R }, 
                                       EEPROM_SYSCTL14, VALUE(7), 0 },
    { { SENSORS_EEPROM_VAIO_SERIAL+23, "serial24", NOMAP, NOMAP, R }, 
                                       EEPROM_SYSCTL14, VALUE(8), 0 },
    { { SENSORS_EEPROM_VAIO_SERIAL+24, "serial25", NOMAP, NOMAP, R }, 
                                       EEPROM_SYSCTL14, VALUE(9), 0 },
    { { SENSORS_EEPROM_VAIO_SERIAL+25, "serial26", NOMAP, NOMAP, R }, 
                                       EEPROM_SYSCTL14, VALUE(10), 0 },
    { { SENSORS_EEPROM_VAIO_SERIAL+26, "serial27", NOMAP, NOMAP, R }, 
                                       EEPROM_SYSCTL14, VALUE(11), 0 },
    { { SENSORS_EEPROM_VAIO_SERIAL+27, "serial28", NOMAP, NOMAP, R }, 
                                       EEPROM_SYSCTL14, VALUE(12), 0 },
    { { SENSORS_EEPROM_VAIO_SERIAL+28, "serial29", NOMAP, NOMAP, R }, 
                                       EEPROM_SYSCTL14, VALUE(13), 0 },
    { { SENSORS_EEPROM_VAIO_SERIAL+29, "serial30", NOMAP, NOMAP, R }, 
                                       EEPROM_SYSCTL14, VALUE(14), 0 },
    { { SENSORS_EEPROM_VAIO_SERIAL+30, "serial31", NOMAP, NOMAP, R }, 
                                       EEPROM_SYSCTL14, VALUE(15), 0 },
    { { SENSORS_EEPROM_VAIO_SERIAL+31, "serial32", NOMAP, NOMAP, R }, 
                                       EEPROM_SYSCTL14, VALUE(16), 0 },
/* for monitor's edid */
    { { SENSORS_EEPROM_EDID_HEADER, "EDID header", NOMAP, NOMAP, R }, 
                                    EEPROM_SYSCTL1, VALUE(1), 0 },
/* for Shuttle EEPROMS */
    { { SENSORS_EEPROM_SHUTTLE, "Ethernet Address", NOMAP, NOMAP, R }, 
                                EEPROM_SYSCTL1, VALUE(1), 0 },
    { { SENSORS_EEPROM_SHUTTLE, "shuttle2", NOMAP, NOMAP, R }, 
                                EEPROM_SYSCTL1, VALUE(2), 0 },
    { { SENSORS_EEPROM_SHUTTLE, "shuttle3", NOMAP, NOMAP, R }, 
                                EEPROM_SYSCTL1, VALUE(3), 0 },
    { { SENSORS_EEPROM_SHUTTLE, "shuttle4", NOMAP, NOMAP, R }, 
                                EEPROM_SYSCTL1, VALUE(4), 0 },
    { { SENSORS_EEPROM_SHUTTLE, "shuttle5", NOMAP, NOMAP, R }, 
                                EEPROM_SYSCTL1, VALUE(5), 0 },
    { { SENSORS_EEPROM_SHUTTLE, "shuttle6", NOMAP, NOMAP, R }, 
                                EEPROM_SYSCTL1, VALUE(6), 0 },
    { { 0 }, 0 }
  };

/* NOTE: Many shared values with adm1027_features !!! */
static sensors_chip_feature lm85_features[] =
  { 
    { { SENSORS_LM85_IN0, "in0", NOMAP, NOMAP, R }, 
                          LM85_SYSCTL_IN0, VALUE(3), 3 },
    { { SENSORS_LM85_IN1, "in1", NOMAP, NOMAP, R }, 
                          LM85_SYSCTL_IN1, VALUE(3), 3 },
    { { SENSORS_LM85_IN2, "in2", NOMAP, NOMAP, R }, 
                          LM85_SYSCTL_IN2, VALUE(3), 3 },
    { { SENSORS_LM85_IN3, "in3", NOMAP, NOMAP, R }, 
                          LM85_SYSCTL_IN3, VALUE(3), 3 },
    { { SENSORS_LM85_IN4, "in4", NOMAP, NOMAP, R }, 
                          LM85_SYSCTL_IN4, VALUE(3), 3 },
    { { SENSORS_LM85_IN5, "in5", NOMAP, NOMAP, R }, 
                          EMC6D100_SYSCTL_IN5, VALUE(3), 3 },
    { { SENSORS_LM85_IN6, "in6", NOMAP, NOMAP, R }, 
                          EMC6D100_SYSCTL_IN6, VALUE(3), 3 },
    { { SENSORS_LM85_IN7, "in7", NOMAP, NOMAP, R }, 
                          EMC6D100_SYSCTL_IN7, VALUE(3), 3 },
    { { SENSORS_LM85_IN0_MIN, "in0_min", SENSORS_LM85_IN0, SENSORS_LM85_IN0, RW }, 
                              LM85_SYSCTL_IN0, VALUE(1), 3 },
    { { SENSORS_LM85_IN1_MIN, "in1_min", SENSORS_LM85_IN1, SENSORS_LM85_IN1, RW }, 
                              LM85_SYSCTL_IN1, VALUE(1), 3 },
    { { SENSORS_LM85_IN2_MIN, "in2_min", SENSORS_LM85_IN2, SENSORS_LM85_IN2, RW }, 
                              LM85_SYSCTL_IN2, VALUE(1), 3 },
    { { SENSORS_LM85_IN3_MIN, "in3_min", SENSORS_LM85_IN3, SENSORS_LM85_IN3, RW }, 
                              LM85_SYSCTL_IN3, VALUE(1), 3 },
    { { SENSORS_LM85_IN4_MIN, "in4_min", SENSORS_LM85_IN4, SENSORS_LM85_IN4, RW }, 
                              LM85_SYSCTL_IN4, VALUE(1), 3 },
    { { SENSORS_LM85_IN5_MIN, "in5_min", SENSORS_LM85_IN5, SENSORS_LM85_IN5, RW }, 
                              EMC6D100_SYSCTL_IN5, VALUE(1), 3 },
    { { SENSORS_LM85_IN6_MIN, "in6_min", SENSORS_LM85_IN6, SENSORS_LM85_IN6, RW }, 
                              EMC6D100_SYSCTL_IN6, VALUE(1), 3 },
    { { SENSORS_LM85_IN7_MIN, "in7_min", SENSORS_LM85_IN7, SENSORS_LM85_IN7, RW }, 
                              EMC6D100_SYSCTL_IN7, VALUE(1), 3 },
    { { SENSORS_LM85_IN0_MAX, "in0_max", SENSORS_LM85_IN0, SENSORS_LM85_IN0, RW }, 
                              LM85_SYSCTL_IN0, VALUE(2), 3 },
    { { SENSORS_LM85_IN1_MAX, "in1_max", SENSORS_LM85_IN1, SENSORS_LM85_IN1, RW }, 
                              LM85_SYSCTL_IN1, VALUE(2), 3 },
    { { SENSORS_LM85_IN2_MAX, "in2_max", SENSORS_LM85_IN2, SENSORS_LM85_IN2, RW }, 
                              LM85_SYSCTL_IN2, VALUE(2), 3 },
    { { SENSORS_LM85_IN3_MAX, "in3_max", SENSORS_LM85_IN3, SENSORS_LM85_IN3, RW }, 
                              LM85_SYSCTL_IN3, VALUE(2), 3 },
    { { SENSORS_LM85_IN4_MAX, "in4_max", SENSORS_LM85_IN4, SENSORS_LM85_IN4, RW }, 
                              LM85_SYSCTL_IN4, VALUE(2), 3 },
    { { SENSORS_LM85_IN5_MAX, "in5_max", SENSORS_LM85_IN5, SENSORS_LM85_IN5, RW }, 
                              EMC6D100_SYSCTL_IN5, VALUE(2), 3 },
    { { SENSORS_LM85_IN6_MAX, "in6_max", SENSORS_LM85_IN6, SENSORS_LM85_IN6, RW }, 
                              EMC6D100_SYSCTL_IN6, VALUE(2), 3 },
    { { SENSORS_LM85_IN7_MAX, "in7_max", SENSORS_LM85_IN7, SENSORS_LM85_IN7, RW }, 
                              EMC6D100_SYSCTL_IN7, VALUE(2), 3 },
    { { SENSORS_LM85_FAN1, "fan1", NOMAP, NOMAP, R }, 
                           LM85_SYSCTL_FAN1, VALUE(2), 0 },
    { { SENSORS_LM85_FAN2, "fan2", NOMAP, NOMAP, R }, 
                           LM85_SYSCTL_FAN2, VALUE(2), 0 },
    { { SENSORS_LM85_FAN3, "fan3", NOMAP, NOMAP, R }, 
                           LM85_SYSCTL_FAN3, VALUE(2), 0 },
    { { SENSORS_LM85_FAN4, "fan4", NOMAP, NOMAP, R }, 
                           LM85_SYSCTL_FAN4, VALUE(2), 0 },
    { { SENSORS_LM85_FAN1_MIN, "fan1_min", SENSORS_LM85_FAN1, SENSORS_LM85_FAN1, 
                               RW }, 
                               LM85_SYSCTL_FAN1, VALUE(1), 0 },
    { { SENSORS_LM85_FAN2_MIN, "fan2_min", SENSORS_LM85_FAN2, SENSORS_LM85_FAN2, 
                               RW }, 
                               LM85_SYSCTL_FAN2, VALUE(1), 0 },
    { { SENSORS_LM85_FAN3_MIN, "fan3_min", SENSORS_LM85_FAN3, SENSORS_LM85_FAN3, 
                               RW }, 
                               LM85_SYSCTL_FAN3, VALUE(1), 0 },
    { { SENSORS_LM85_FAN4_MIN, "fan4_min", SENSORS_LM85_FAN4, SENSORS_LM85_FAN4, 
                               RW }, 
                               LM85_SYSCTL_FAN4, VALUE(1), 0 },
    { { SENSORS_LM85_TEMP1, "temp1", NOMAP, NOMAP, R }, 
                            LM85_SYSCTL_TEMP1, VALUE(3), 2 },
    { { SENSORS_LM85_TEMP2, "temp2", NOMAP, NOMAP, R }, 
                            LM85_SYSCTL_TEMP2, VALUE(3), 2 },
    { { SENSORS_LM85_TEMP3, "temp3", NOMAP, NOMAP, R }, 
                            LM85_SYSCTL_TEMP3, VALUE(3), 2 },
    { { SENSORS_LM85_TEMP1_MIN, "temp1_min", SENSORS_LM85_TEMP1, 
                                SENSORS_LM85_TEMP1, RW }, 
                                LM85_SYSCTL_TEMP1, VALUE(1), 2 },
    { { SENSORS_LM85_TEMP2_MIN, "temp2_min", SENSORS_LM85_TEMP2, 
                                SENSORS_LM85_TEMP2, RW }, 
                                LM85_SYSCTL_TEMP2, VALUE(1), 2 },
    { { SENSORS_LM85_TEMP3_MIN, "temp3_min", SENSORS_LM85_TEMP3, 
                                SENSORS_LM85_TEMP3, RW }, 
                                LM85_SYSCTL_TEMP3, VALUE(1), 2 },
    { { SENSORS_LM85_TEMP1_MAX, "temp1_max", SENSORS_LM85_TEMP1, 
                                SENSORS_LM85_TEMP1, RW }, 
                                LM85_SYSCTL_TEMP1, VALUE(2), 2 },
    { { SENSORS_LM85_TEMP2_MAX, "temp2_max", SENSORS_LM85_TEMP2, 
                                SENSORS_LM85_TEMP2, RW }, 
                                LM85_SYSCTL_TEMP2, VALUE(2), 2 },
    { { SENSORS_LM85_TEMP3_MAX, "temp3_max", SENSORS_LM85_TEMP3, 
                                SENSORS_LM85_TEMP3, RW }, 
                                LM85_SYSCTL_TEMP3, VALUE(2), 2 },
    { { SENSORS_LM85_VID, "vid", NOMAP, NOMAP, R }, 
                          LM85_SYSCTL_VID, VALUE(1), 3 },
    { { SENSORS_LM85_VRM, "vrm", NOMAP, NOMAP, RW }, 
                          LM85_SYSCTL_VRM, VALUE(1), 1 },
    { { SENSORS_LM85_ALARMS, "alarms", NOMAP, NOMAP, R }, 
                             LM85_SYSCTL_ALARMS, VALUE(1), 0 },
    { { SENSORS_LM85_PWM1, "pwm1", NOMAP, NOMAP, RW }, 
                           LM85_SYSCTL_PWM1, VALUE(1), 0 },
    { { SENSORS_LM85_PWM2, "pwm2", NOMAP, NOMAP, RW }, 
                           LM85_SYSCTL_PWM2, VALUE(1), 0 },
    { { SENSORS_LM85_PWM3, "pwm3", NOMAP, NOMAP, RW }, 
                           LM85_SYSCTL_PWM3, VALUE(1), 0 },
    { { SENSORS_LM85_PWM1_SPINUP, "pwm1_spinup", SENSORS_LM85_PWM1, NOMAP, RW }, 
                                  LM85_SYSCTL_PWM_CFG1, VALUE(1), 1 },
    { { SENSORS_LM85_PWM2_SPINUP, "pwm2_spinup", SENSORS_LM85_PWM2, NOMAP, RW }, 
                                  LM85_SYSCTL_PWM_CFG2, VALUE(1), 1 },
    { { SENSORS_LM85_PWM3_SPINUP, "pwm3_spinup", SENSORS_LM85_PWM3, NOMAP, RW }, 
                                  LM85_SYSCTL_PWM_CFG3, VALUE(1), 1 },
    { { SENSORS_LM85_PWM1_MIN, "pwm1_min", SENSORS_LM85_PWM1, NOMAP, RW }, 
                               LM85_SYSCTL_PWM_CFG1, VALUE(2), 1 },
    { { SENSORS_LM85_PWM2_MIN, "pwm2_min", SENSORS_LM85_PWM2, NOMAP, RW }, 
                               LM85_SYSCTL_PWM_CFG2, VALUE(2), 1 },
    { { SENSORS_LM85_PWM3_MIN, "pwm3_min", SENSORS_LM85_PWM3, NOMAP, RW }, 
                               LM85_SYSCTL_PWM_CFG3, VALUE(2), 1 },
    { { SENSORS_LM85_PWM1_FREQ, "pwm1_freq", SENSORS_LM85_PWM1, NOMAP, RW }, 
                                LM85_SYSCTL_PWM_CFG1, VALUE(3), 1 },
    { { SENSORS_LM85_PWM2_FREQ, "pwm2_freq", SENSORS_LM85_PWM2, NOMAP, RW }, 
                                LM85_SYSCTL_PWM_CFG2, VALUE(3), 1 },
    { { SENSORS_LM85_PWM3_FREQ, "pwm3_freq", SENSORS_LM85_PWM3, NOMAP, RW }, 
                                LM85_SYSCTL_PWM_CFG3, VALUE(3), 1 },
    { { SENSORS_LM85_PWM1_MIN_CTL, "pwm1_min_ctl", SENSORS_LM85_PWM1, NOMAP, RW }, 
                                   LM85_SYSCTL_PWM_CFG1, VALUE(4), 1 },
    { { SENSORS_LM85_PWM2_MIN_CTL, "pwm2_min_ctl", SENSORS_LM85_PWM2, NOMAP, RW }, 
                                   LM85_SYSCTL_PWM_CFG2, VALUE(4), 1 },
    { { SENSORS_LM85_PWM3_MIN_CTL, "pwm3_min_ctl", SENSORS_LM85_PWM3, NOMAP, RW }, 
                                   LM85_SYSCTL_PWM_CFG3, VALUE(4), 1 },
    { { SENSORS_LM85_PWM1_INVERT, "pwm1_invert", SENSORS_LM85_PWM1, NOMAP, RW }, 
                                  LM85_SYSCTL_PWM_CFG1, VALUE(5), 0 },
    { { SENSORS_LM85_PWM2_INVERT, "pwm2_invert", SENSORS_LM85_PWM2, NOMAP, RW }, 
                                  LM85_SYSCTL_PWM_CFG2, VALUE(5), 0 },
    { { SENSORS_LM85_PWM3_INVERT, "pwm3_invert", SENSORS_LM85_PWM3, NOMAP, RW }, 
                                  LM85_SYSCTL_PWM_CFG3, VALUE(5), 0 },
    { { SENSORS_LM85_PWM1_ZONE, "pwm1_zone", SENSORS_LM85_PWM1, NOMAP, RW }, 
                                LM85_SYSCTL_PWM_ZONE1, VALUE(1), 0 },
    { { SENSORS_LM85_PWM2_ZONE, "pwm2_zone", SENSORS_LM85_PWM2, NOMAP, RW }, 
                                LM85_SYSCTL_PWM_ZONE2, VALUE(1), 0 },
    { { SENSORS_LM85_PWM3_ZONE, "pwm3_zone", SENSORS_LM85_PWM3, NOMAP, RW }, 
                                LM85_SYSCTL_PWM_ZONE3, VALUE(1), 0 },
    { { SENSORS_LM85_ZONE1_LIMIT, "zone1_limit", SENSORS_LM85_TEMP1, NOMAP, RW }, 
                                  LM85_SYSCTL_ZONE1, VALUE(1), 1 },
    { { SENSORS_LM85_ZONE2_LIMIT, "zone2_limit", SENSORS_LM85_TEMP2, NOMAP, RW }, 
                                  LM85_SYSCTL_ZONE2, VALUE(1), 1 },
    { { SENSORS_LM85_ZONE3_LIMIT, "zone3_limit", SENSORS_LM85_TEMP3, NOMAP, RW }, 
                                  LM85_SYSCTL_ZONE3, VALUE(1), 1 },
    { { SENSORS_LM85_ZONE1_HYST, "zone1_hyst", SENSORS_LM85_TEMP1, NOMAP, RW }, 
                                 LM85_SYSCTL_ZONE1, VALUE(2), 1 },
    { { SENSORS_LM85_ZONE2_HYST, "zone2_hyst", SENSORS_LM85_TEMP2, NOMAP, RW }, 
                                 LM85_SYSCTL_ZONE2, VALUE(2), 1 },
    { { SENSORS_LM85_ZONE3_HYST, "zone3_hyst", SENSORS_LM85_TEMP3, NOMAP, RW }, 
                                 LM85_SYSCTL_ZONE3, VALUE(2), 1 },
    { { SENSORS_LM85_ZONE1_RANGE, "zone1_range", SENSORS_LM85_TEMP1, NOMAP, RW }, 
                                  LM85_SYSCTL_ZONE1, VALUE(3), 1 },
    { { SENSORS_LM85_ZONE2_RANGE, "zone2_range", SENSORS_LM85_TEMP2, NOMAP, RW }, 
                                  LM85_SYSCTL_ZONE2, VALUE(3), 1 },
    { { SENSORS_LM85_ZONE3_RANGE, "zone3_range", SENSORS_LM85_TEMP3, NOMAP, RW }, 
                                  LM85_SYSCTL_ZONE3, VALUE(3), 1 },
    { { SENSORS_LM85_ZONE1_CRITICAL, "zone1_critical", SENSORS_LM85_TEMP1, NOMAP, 
                                     RW }, 
                                     LM85_SYSCTL_ZONE1, VALUE(4), 1 },
    { { SENSORS_LM85_ZONE2_CRITICAL, "zone2_critical", SENSORS_LM85_TEMP2, NOMAP, 
                                     RW }, 
                                     LM85_SYSCTL_ZONE2, VALUE(4), 1 },
    { { SENSORS_LM85_ZONE3_CRITICAL, "zone3_critical", SENSORS_LM85_TEMP3, NOMAP, 
                                     RW }, 
                                     LM85_SYSCTL_ZONE3, VALUE(4), 1 },

    { { SENSORS_LM85_PWM1_SPINUP_CTL, "pwm1_spinup_ctl", SENSORS_LM85_PWM1, 
                                      NOMAP, RW }, 
                                      LM85_SYSCTL_SPINUP_CTL, VALUE(1), 0 },
    { { SENSORS_LM85_PWM2_SPINUP_CTL, "pwm2_spinup_ctl", SENSORS_LM85_PWM2, 
                                      NOMAP, RW }, 
                                      LM85_SYSCTL_SPINUP_CTL, VALUE(2), 0 },
    { { SENSORS_LM85_PWM3_SPINUP_CTL, "pwm3_spinup_ctl", SENSORS_LM85_PWM3, 
                                      NOMAP, RW }, 
                                      LM85_SYSCTL_SPINUP_CTL, VALUE(3), 0 },
    { { SENSORS_LM85_FAN1_TACH_MODE, "fan1_tach_mode", SENSORS_LM85_FAN1, NOMAP, 
                                     RW }, 
                                     LM85_SYSCTL_TACH_MODE, VALUE(1), 0 },
    { { SENSORS_LM85_FAN2_TACH_MODE, "fan2_tach_mode", SENSORS_LM85_FAN2, NOMAP, 
                                     RW }, 
                                     LM85_SYSCTL_TACH_MODE, VALUE(2), 0 },
    { { SENSORS_LM85_FAN3_TACH_MODE, "fan3_tach_mode", SENSORS_LM85_FAN3, NOMAP, 
                                     RW }, 
                                     LM85_SYSCTL_TACH_MODE, VALUE(3), 0 },
    { { SENSORS_LM85_ZONE1_SMOOTH, "zone1_smooth", SENSORS_LM85_TEMP1, NOMAP, RW }, 
                                   LM85_SYSCTL_SMOOTH1, VALUE(1), 1 },
    { { SENSORS_LM85_ZONE2_SMOOTH, "zone2_smooth", SENSORS_LM85_TEMP2, NOMAP, RW }, 
                                   LM85_SYSCTL_SMOOTH2, VALUE(1), 1 },
    { { SENSORS_LM85_ZONE3_SMOOTH, "zone3_smooth", SENSORS_LM85_TEMP3, NOMAP, RW }, 
                                   LM85_SYSCTL_SMOOTH3, VALUE(1), 1 },
    { { 0 }, 0 }
  };

/* NOTE: Many shared values with lm85_features !!! */
static sensors_chip_feature adm1027_features[] =
  { 
    { { SENSORS_LM85_IN0, "in0", NOMAP, NOMAP, R }, 
                          LM85_SYSCTL_IN0, VALUE(3), 3 },
    { { SENSORS_LM85_IN1, "in1", NOMAP, NOMAP, R }, 
                          LM85_SYSCTL_IN1, VALUE(3), 3 },
    { { SENSORS_LM85_IN2, "in2", NOMAP, NOMAP, R }, 
                          LM85_SYSCTL_IN2, VALUE(3), 3 },
    { { SENSORS_LM85_IN3, "in3", NOMAP, NOMAP, R }, 
                          LM85_SYSCTL_IN3, VALUE(3), 3 },
    { { SENSORS_LM85_IN4, "in4", NOMAP, NOMAP, R }, 
                          LM85_SYSCTL_IN4, VALUE(3), 3 },
    { { SENSORS_LM85_IN0_MIN, "in0_min", SENSORS_LM85_IN0, SENSORS_LM85_IN0, RW }, 
                              LM85_SYSCTL_IN0, VALUE(1), 3 },
    { { SENSORS_LM85_IN1_MIN, "in1_min", SENSORS_LM85_IN1, SENSORS_LM85_IN1, RW }, 
                              LM85_SYSCTL_IN1, VALUE(1), 3 },
    { { SENSORS_LM85_IN2_MIN, "in2_min", SENSORS_LM85_IN2, SENSORS_LM85_IN2, RW }, 
                              LM85_SYSCTL_IN2, VALUE(1), 3 },
    { { SENSORS_LM85_IN3_MIN, "in3_min", SENSORS_LM85_IN3, SENSORS_LM85_IN3, RW }, 
                              LM85_SYSCTL_IN3, VALUE(1), 3 },
    { { SENSORS_LM85_IN4_MIN, "in4_min", SENSORS_LM85_IN4, SENSORS_LM85_IN4, RW }, 
                              LM85_SYSCTL_IN4, VALUE(1), 3 },
    { { SENSORS_LM85_IN0_MAX, "in0_max", SENSORS_LM85_IN0, SENSORS_LM85_IN0, RW }, 
                              LM85_SYSCTL_IN0, VALUE(2), 3 },
    { { SENSORS_LM85_IN1_MAX, "in1_max", SENSORS_LM85_IN1, SENSORS_LM85_IN1, RW }, 
                              LM85_SYSCTL_IN1, VALUE(2), 3 },
    { { SENSORS_LM85_IN2_MAX, "in2_max", SENSORS_LM85_IN2, SENSORS_LM85_IN2, RW }, 
                              LM85_SYSCTL_IN2, VALUE(2), 3 },
    { { SENSORS_LM85_IN3_MAX, "in3_max", SENSORS_LM85_IN3, SENSORS_LM85_IN3, RW }, 
                              LM85_SYSCTL_IN3, VALUE(2), 3 },
    { { SENSORS_LM85_IN4_MAX, "in4_max", SENSORS_LM85_IN4, SENSORS_LM85_IN4, RW }, 
                              LM85_SYSCTL_IN4, VALUE(2), 3 },
    { { SENSORS_LM85_FAN1, "fan1", NOMAP, NOMAP, R }, 
                           LM85_SYSCTL_FAN1, VALUE(2), 0 },
    { { SENSORS_LM85_FAN2, "fan2", NOMAP, NOMAP, R }, 
                           LM85_SYSCTL_FAN2, VALUE(2), 0 },
    { { SENSORS_LM85_FAN3, "fan3", NOMAP, NOMAP, R }, 
                           LM85_SYSCTL_FAN3, VALUE(2), 0 },
    { { SENSORS_LM85_FAN4, "fan4", NOMAP, NOMAP, R }, 
                           LM85_SYSCTL_FAN4, VALUE(2), 0 },
    { { SENSORS_LM85_FAN1_MIN, "fan1_min", SENSORS_LM85_FAN1, SENSORS_LM85_FAN1, 
                               RW }, 
                               LM85_SYSCTL_FAN1, VALUE(1), 0 },
    { { SENSORS_LM85_FAN2_MIN, "fan2_min", SENSORS_LM85_FAN2, SENSORS_LM85_FAN2, 
                               RW }, 
                               LM85_SYSCTL_FAN2, VALUE(1), 0 },
    { { SENSORS_LM85_FAN3_MIN, "fan3_min", SENSORS_LM85_FAN3, SENSORS_LM85_FAN3, 
                               RW }, 
                               LM85_SYSCTL_FAN3, VALUE(1), 0 },
    { { SENSORS_LM85_FAN4_MIN, "fan4_min", SENSORS_LM85_FAN4, SENSORS_LM85_FAN4, 
                               RW }, 
                               LM85_SYSCTL_FAN4, VALUE(1), 0 },
    { { SENSORS_LM85_TEMP1, "temp1", NOMAP, NOMAP, R }, 
                            LM85_SYSCTL_TEMP1, VALUE(3), 2 },
    { { SENSORS_LM85_TEMP2, "temp2", NOMAP, NOMAP, R }, 
                            LM85_SYSCTL_TEMP2, VALUE(3), 2 },
    { { SENSORS_LM85_TEMP3, "temp3", NOMAP, NOMAP, R }, 
                            LM85_SYSCTL_TEMP3, VALUE(3), 2 },
    { { SENSORS_LM85_TEMP1_MIN, "temp1_min", SENSORS_LM85_TEMP1, 
                                SENSORS_LM85_TEMP1, RW }, 
                                LM85_SYSCTL_TEMP1, VALUE(1), 2 },
    { { SENSORS_LM85_TEMP2_MIN, "temp2_min", SENSORS_LM85_TEMP2, 
                                SENSORS_LM85_TEMP2, RW }, 
                                LM85_SYSCTL_TEMP2, VALUE(1), 2 },
    { { SENSORS_LM85_TEMP3_MIN, "temp3_min", SENSORS_LM85_TEMP3, 
                                SENSORS_LM85_TEMP3, RW }, 
                                LM85_SYSCTL_TEMP3, VALUE(1), 2 },
    { { SENSORS_LM85_TEMP1_MAX, "temp1_max", SENSORS_LM85_TEMP1, 
                                SENSORS_LM85_TEMP1, RW }, 
                                LM85_SYSCTL_TEMP1, VALUE(2), 2 },
    { { SENSORS_LM85_TEMP2_MAX, "temp2_max", SENSORS_LM85_TEMP2, 
                                SENSORS_LM85_TEMP2, RW }, 
                                LM85_SYSCTL_TEMP2, VALUE(2), 2 },
    { { SENSORS_LM85_TEMP3_MAX, "temp3_max", SENSORS_LM85_TEMP3, 
                                SENSORS_LM85_TEMP3, RW }, 
                                LM85_SYSCTL_TEMP3, VALUE(2), 2 },
    { { SENSORS_LM85_VID, "vid", NOMAP, NOMAP, R }, 
                          LM85_SYSCTL_VID, VALUE(1), 3 },
    { { SENSORS_LM85_VRM, "vrm", NOMAP, NOMAP, RW }, 
                          LM85_SYSCTL_VRM, VALUE(1), 1 },
    { { SENSORS_LM85_ALARMS, "alarms", NOMAP, NOMAP, R }, 
                             LM85_SYSCTL_ALARMS, VALUE(1), 0 },
    { { SENSORS_LM85_PWM1, "pwm1", NOMAP, NOMAP, RW }, 
                           LM85_SYSCTL_PWM1, VALUE(1), 0 },
    { { SENSORS_LM85_PWM2, "pwm2", NOMAP, NOMAP, RW }, 
                           LM85_SYSCTL_PWM2, VALUE(1), 0 },
    { { SENSORS_LM85_PWM3, "pwm3", NOMAP, NOMAP, RW }, 
                           LM85_SYSCTL_PWM3, VALUE(1), 0 },
    { { SENSORS_LM85_PWM1_SPINUP, "pwm1_spinup", SENSORS_LM85_PWM1, NOMAP, RW }, 
                                  LM85_SYSCTL_PWM_CFG1, VALUE(1), 1 },
    { { SENSORS_LM85_PWM2_SPINUP, "pwm2_spinup", SENSORS_LM85_PWM2, NOMAP, RW }, 
                                  LM85_SYSCTL_PWM_CFG2, VALUE(1), 1 },
    { { SENSORS_LM85_PWM3_SPINUP, "pwm3_spinup", SENSORS_LM85_PWM3, NOMAP, RW }, 
                                  LM85_SYSCTL_PWM_CFG3, VALUE(1), 1 },
    { { SENSORS_LM85_PWM1_MIN, "pwm1_min", SENSORS_LM85_PWM1, NOMAP, RW }, 
                               LM85_SYSCTL_PWM_CFG1, VALUE(2), 1 },
    { { SENSORS_LM85_PWM2_MIN, "pwm2_min", SENSORS_LM85_PWM2, NOMAP, RW }, 
                               LM85_SYSCTL_PWM_CFG2, VALUE(2), 1 },
    { { SENSORS_LM85_PWM3_MIN, "pwm3_min", SENSORS_LM85_PWM3, NOMAP, RW }, 
                               LM85_SYSCTL_PWM_CFG3, VALUE(2), 1 },
    { { SENSORS_LM85_PWM1_FREQ, "pwm1_freq", SENSORS_LM85_PWM1, NOMAP, RW }, 
                                LM85_SYSCTL_PWM_CFG1, VALUE(3), 1 },
    { { SENSORS_LM85_PWM2_FREQ, "pwm2_freq", SENSORS_LM85_PWM2, NOMAP, RW }, 
                                LM85_SYSCTL_PWM_CFG2, VALUE(3), 1 },
    { { SENSORS_LM85_PWM3_FREQ, "pwm3_freq", SENSORS_LM85_PWM3, NOMAP, RW }, 
                                LM85_SYSCTL_PWM_CFG3, VALUE(3), 1 },
    { { SENSORS_LM85_PWM1_MIN_CTL, "pwm1_min_ctl", SENSORS_LM85_PWM1, NOMAP, RW }, 
                                   LM85_SYSCTL_PWM_CFG1, VALUE(4), 1 },
    { { SENSORS_LM85_PWM2_MIN_CTL, "pwm2_min_ctl", SENSORS_LM85_PWM2, NOMAP, RW }, 
                                   LM85_SYSCTL_PWM_CFG2, VALUE(4), 1 },
    { { SENSORS_LM85_PWM3_MIN_CTL, "pwm3_min_ctl", SENSORS_LM85_PWM3, NOMAP, RW }, 
                                   LM85_SYSCTL_PWM_CFG3, VALUE(4), 1 },
    { { SENSORS_LM85_PWM1_INVERT, "pwm1_invert", SENSORS_LM85_PWM1, NOMAP, RW }, 
                                  LM85_SYSCTL_PWM_CFG1, VALUE(5), 0 },
    { { SENSORS_LM85_PWM2_INVERT, "pwm2_invert", SENSORS_LM85_PWM2, NOMAP, RW }, 
                                  LM85_SYSCTL_PWM_CFG2, VALUE(5), 0 },
    { { SENSORS_LM85_PWM3_INVERT, "pwm3_invert", SENSORS_LM85_PWM3, NOMAP, RW }, 
                                  LM85_SYSCTL_PWM_CFG3, VALUE(5), 0 },
    { { SENSORS_LM85_PWM1_ZONE, "pwm1_zone", SENSORS_LM85_PWM1, NOMAP, RW }, 
                                LM85_SYSCTL_PWM_ZONE1, VALUE(1), 0 },
    { { SENSORS_LM85_PWM2_ZONE, "pwm2_zone", SENSORS_LM85_PWM2, NOMAP, RW }, 
                                LM85_SYSCTL_PWM_ZONE2, VALUE(1), 0 },
    { { SENSORS_LM85_PWM3_ZONE, "pwm3_zone", SENSORS_LM85_PWM3, NOMAP, RW }, 
                                LM85_SYSCTL_PWM_ZONE3, VALUE(1), 0 },
    { { SENSORS_LM85_ZONE1_LIMIT, "zone1_limit", SENSORS_LM85_TEMP1, NOMAP, RW }, 
                                  LM85_SYSCTL_ZONE1, VALUE(1), 1 },
    { { SENSORS_LM85_ZONE2_LIMIT, "zone2_limit", SENSORS_LM85_TEMP2, NOMAP, RW }, 
                                  LM85_SYSCTL_ZONE2, VALUE(1), 1 },
    { { SENSORS_LM85_ZONE3_LIMIT, "zone3_limit", SENSORS_LM85_TEMP3, NOMAP, RW }, 
                                  LM85_SYSCTL_ZONE3, VALUE(1), 1 },
    { { SENSORS_LM85_ZONE1_HYST, "zone1_hyst", SENSORS_LM85_TEMP1, NOMAP, RW }, 
                                 LM85_SYSCTL_ZONE1, VALUE(2), 1 },
    { { SENSORS_LM85_ZONE2_HYST, "zone2_hyst", SENSORS_LM85_TEMP2, NOMAP, RW }, 
                                 LM85_SYSCTL_ZONE2, VALUE(2), 1 },
    { { SENSORS_LM85_ZONE3_HYST, "zone3_hyst", SENSORS_LM85_TEMP3, NOMAP, RW }, 
                                 LM85_SYSCTL_ZONE3, VALUE(2), 1 },
    { { SENSORS_LM85_ZONE1_RANGE, "zone1_range", SENSORS_LM85_TEMP1, NOMAP, RW }, 
                                  LM85_SYSCTL_ZONE1, VALUE(3), 1 },
    { { SENSORS_LM85_ZONE2_RANGE, "zone2_range", SENSORS_LM85_TEMP2, NOMAP, RW }, 
                                  LM85_SYSCTL_ZONE2, VALUE(3), 1 },
    { { SENSORS_LM85_ZONE3_RANGE, "zone3_range", SENSORS_LM85_TEMP3, NOMAP, RW }, 
                                  LM85_SYSCTL_ZONE3, VALUE(3), 1 },
    { { SENSORS_LM85_ZONE1_CRITICAL, "zone1_critical", SENSORS_LM85_TEMP1, NOMAP, 
                                     RW }, 
                                     LM85_SYSCTL_ZONE1, VALUE(4), 1 },
    { { SENSORS_LM85_ZONE2_CRITICAL, "zone2_critical", SENSORS_LM85_TEMP2, NOMAP, 
                                     RW }, 
                                     LM85_SYSCTL_ZONE2, VALUE(4), 1 },
    { { SENSORS_LM85_ZONE3_CRITICAL, "zone3_critical", SENSORS_LM85_TEMP3, NOMAP, 
                                     RW }, 
                                     LM85_SYSCTL_ZONE3, VALUE(4), 1 },

    { { SENSORS_ADM1027_FAN1_TACH_MODE, "fan1_tach_mode", SENSORS_LM85_FAN1, 
                                        NOMAP, RW }, 
                                        ADM1027_SYSCTL_TACH_MODE, VALUE(1), 0 },
    { { SENSORS_ADM1027_FAN2_TACH_MODE, "fan2_tach_mode", SENSORS_LM85_FAN2, 
                                        NOMAP, RW }, 
                                        ADM1027_SYSCTL_TACH_MODE, VALUE(2), 0 },
    { { SENSORS_ADM1027_FAN3_TACH_MODE, "fan3_tach_mode", SENSORS_LM85_FAN3, 
                                        NOMAP, RW }, 
                                        ADM1027_SYSCTL_TACH_MODE, VALUE(3), 0 },
    { { SENSORS_ADM1027_FAN4_TACH_MODE, "fan4_tach_mode", SENSORS_LM85_FAN4, 
                                        NOMAP, RW }, 
                                        ADM1027_SYSCTL_TACH_MODE, VALUE(4), 0 },
    { { SENSORS_ADM1027_PWM1_SMOOTH, "pwm1_smooth", SENSORS_LM85_PWM1, NOMAP, RW }, 
                                     LM85_SYSCTL_SMOOTH1, VALUE(1), 1 },
    { { SENSORS_ADM1027_PWM2_SMOOTH, "pwm2_smooth", SENSORS_LM85_PWM2, NOMAP, RW }, 
                                     LM85_SYSCTL_SMOOTH2, VALUE(1), 1 },
    { { SENSORS_ADM1027_PWM3_SMOOTH, "pwm3_smooth", SENSORS_LM85_PWM3, NOMAP, RW }, 
                                     LM85_SYSCTL_SMOOTH3, VALUE(1), 1 },
    { { SENSORS_ADM1027_TEMP1_OFFSET, "temp1_offset", SENSORS_LM85_TEMP1, 
                                      SENSORS_LM85_TEMP1, RW }, 
                                      ADM1027_SYSCTL_TEMP_OFFSET1, VALUE(1), 2 },
    { { SENSORS_ADM1027_TEMP2_OFFSET, "temp2_offset", SENSORS_LM85_TEMP2, 
                                      SENSORS_LM85_TEMP2, RW }, 
                                      ADM1027_SYSCTL_TEMP_OFFSET2, VALUE(1), 2 },
    { { SENSORS_ADM1027_TEMP3_OFFSET, "temp3_offset", SENSORS_LM85_TEMP3, 
                                      SENSORS_LM85_TEMP3, RW }, 
                                      ADM1027_SYSCTL_TEMP_OFFSET3, VALUE(1), 2 },
    { { SENSORS_ADM1027_FAN1_PPR, "fan1_ppr", SENSORS_LM85_FAN1, NOMAP, RW }, 
                                  ADM1027_SYSCTL_FAN_PPR, VALUE(1), 0 },
    { { SENSORS_ADM1027_FAN2_PPR, "fan2_ppr", SENSORS_LM85_FAN2, NOMAP, RW }, 
                                  ADM1027_SYSCTL_FAN_PPR, VALUE(2), 0 },
    { { SENSORS_ADM1027_FAN3_PPR, "fan3_ppr", SENSORS_LM85_FAN3, NOMAP, RW }, 
                                  ADM1027_SYSCTL_FAN_PPR, VALUE(3), 0 },
    { { SENSORS_ADM1027_FAN4_PPR, "fan4_ppr", SENSORS_LM85_FAN4, NOMAP, RW }, 
                                  ADM1027_SYSCTL_FAN_PPR, VALUE(4), 0 },
    { { SENSORS_ADM1027_ALARM_MASK, "alarm_mask", NOMAP, NOMAP, RW }, 
                                    ADM1027_SYSCTL_ALARM_MASK, VALUE(1), 0 },
    { { 0 }, 0 }
  };

static sensors_chip_feature lm87_features[] =
  { 
    { { SENSORS_LM87_IN0, "2.5V", NOMAP, NOMAP, R }, 
                          LM87_SYSCTL_IN0, VALUE(3), 2, "in0_input", 3 },
    { { SENSORS_LM87_IN1, "Vccp1", NOMAP, NOMAP, R }, 
                          LM87_SYSCTL_IN1, VALUE(3), 2, "in1_input", 3 },
    { { SENSORS_LM87_IN2, "3.3V", NOMAP, NOMAP, R }, 
                          LM87_SYSCTL_IN2, VALUE(3), 2, "in2_input", 3 },
    { { SENSORS_LM87_IN3, "5V", NOMAP, NOMAP, R }, 
                          LM87_SYSCTL_IN3, VALUE(3), 2, "in3_input", 3 },
    { { SENSORS_LM87_IN4, "12V", NOMAP, NOMAP, R }, 
                          LM87_SYSCTL_IN4, VALUE(3), 2, "in4_input", 3 },
    { { SENSORS_LM87_IN5, "Vccp2", NOMAP, NOMAP, R }, 
                          LM87_SYSCTL_IN5, VALUE(3), 2, "in5_input", 3 },
    { { SENSORS_LM87_AIN1, "AIN1", NOMAP, NOMAP, R }, 
                           LM87_SYSCTL_AIN1, VALUE(3), 0, "in6_input", 3 },
    { { SENSORS_LM87_AIN2, "AIN2", NOMAP, NOMAP, R }, 
                           LM87_SYSCTL_AIN2, VALUE(3), 0, "in7_input", 3 },
    { { SENSORS_LM87_IN0_MIN, "2.5V_min", SENSORS_LM87_IN0, SENSORS_LM87_IN0, RW }, 
                              LM87_SYSCTL_IN0, VALUE(1), 2, "in0_min", 3 },
    { { SENSORS_LM87_IN1_MIN, "Vccp1_min", SENSORS_LM87_IN1, SENSORS_LM87_IN1, 
                              RW }, 
                              LM87_SYSCTL_IN1, VALUE(1), 2, "in1_min", 3 },
    { { SENSORS_LM87_IN2_MIN, "3.3V_min", SENSORS_LM87_IN2, SENSORS_LM87_IN2, RW }, 
                              LM87_SYSCTL_IN2, VALUE(1), 2, "in2_min", 3 },
    { { SENSORS_LM87_IN3_MIN, "5V_min", SENSORS_LM87_IN3, SENSORS_LM87_IN3, RW }, 
                              LM87_SYSCTL_IN3, VALUE(1), 2, "in3_min", 3 },
    { { SENSORS_LM87_IN4_MIN, "12V_min", SENSORS_LM87_IN4, SENSORS_LM87_IN4, RW }, 
                              LM87_SYSCTL_IN4, VALUE(1), 2, "in4_min", 3 },
    { { SENSORS_LM87_IN5_MIN, "Vccp2_min", SENSORS_LM87_IN5, SENSORS_LM87_IN5, 
                              RW }, 
                              LM87_SYSCTL_IN5, VALUE(1), 2, "in5_min", 3 },
    { { SENSORS_LM87_AIN1_MIN, "AIN1_min", SENSORS_LM87_AIN1, SENSORS_LM87_AIN1, 
                               RW }, 
                               LM87_SYSCTL_AIN1, VALUE(1), 0, "in6_min", 3 },
    { { SENSORS_LM87_AIN2_MIN, "AIN2_min", SENSORS_LM87_AIN2, SENSORS_LM87_AIN2, 
                               RW }, 
                               LM87_SYSCTL_AIN2, VALUE(1), 0, "in7_min", 3 },
    { { SENSORS_LM87_IN0_MAX, "2.5V_max", SENSORS_LM87_IN0, SENSORS_LM87_IN0, RW }, 
                              LM87_SYSCTL_IN0, VALUE(2), 2, "in0_max", 3 },
    { { SENSORS_LM87_IN1_MAX, "Vccp1_max", SENSORS_LM87_IN1, SENSORS_LM87_IN1, 
                              RW }, 
                              LM87_SYSCTL_IN1, VALUE(2), 2, "in1_max", 3 },
    { { SENSORS_LM87_IN2_MAX, "3.3V_max", SENSORS_LM87_IN2, SENSORS_LM87_IN2, RW }, 
                              LM87_SYSCTL_IN2, VALUE(2), 2, "in2_max", 3 },
    { { SENSORS_LM87_IN3_MAX, "5V_max", SENSORS_LM87_IN3, SENSORS_LM87_IN3, RW }, 
                              LM87_SYSCTL_IN3, VALUE(2), 2, "in3_max", 3 },
    { { SENSORS_LM87_IN4_MAX, "12V_max", SENSORS_LM87_IN4, SENSORS_LM87_IN4, RW }, 
                              LM87_SYSCTL_IN4, VALUE(2), 2, "in4_max", 3 },
    { { SENSORS_LM87_IN5_MAX, "Vccp2_max", SENSORS_LM87_IN5, SENSORS_LM87_IN5, 
                              RW }, 
                              LM87_SYSCTL_IN5, VALUE(2), 2, "in5_max", 3 },
    { { SENSORS_LM87_AIN1_MAX, "AIN1_max", SENSORS_LM87_AIN1, SENSORS_LM87_AIN1, 
                               RW }, 
                               LM87_SYSCTL_AIN1, VALUE(2), 0, "in6_max", 3 },
    { { SENSORS_LM87_AIN2_MAX, "AIN2_max", SENSORS_LM87_AIN2, SENSORS_LM87_AIN2, 
                               RW }, 
                               LM87_SYSCTL_AIN2, VALUE(2), 0, "in7_max", 3 },
    { { SENSORS_LM87_FAN1, "fan1", NOMAP, NOMAP, R }, 
                           LM87_SYSCTL_FAN1, VALUE(2), 0 },
    { { SENSORS_LM87_FAN2, "fan2", NOMAP, NOMAP, R }, 
                           LM87_SYSCTL_FAN2, VALUE(2), 0 },
    { { SENSORS_LM87_FAN1_MIN, "fan1_min", SENSORS_LM87_FAN1, SENSORS_LM87_FAN1, 
                               RW }, 
                               LM87_SYSCTL_FAN1, VALUE(1), 0 },
    { { SENSORS_LM87_FAN2_MIN, "fan2_min", SENSORS_LM87_FAN2, SENSORS_LM87_FAN2, 
                               RW }, 
                               LM87_SYSCTL_FAN2, VALUE(1), 0 },
    { { SENSORS_LM87_TEMP3, "temp3", NOMAP, NOMAP, R }, 
                            LM87_SYSCTL_TEMP3, VALUE(3), 1 },
    { { SENSORS_LM87_TEMP2, "CPU_Temp", NOMAP, NOMAP, R }, 
                            LM87_SYSCTL_TEMP2, VALUE(3), 1, "temp2_input", 3 },
    { { SENSORS_LM87_TEMP1, "temp1", NOMAP, NOMAP, R }, 
                            LM87_SYSCTL_TEMP1, VALUE(3), 1 },
    { { SENSORS_LM87_TEMP1_HYST, "temp1_min", SENSORS_LM87_TEMP1, 
                                 SENSORS_LM87_TEMP1, RW }, 
                                 LM87_SYSCTL_TEMP1, VALUE(2), 1 },
    { { SENSORS_LM87_TEMP1_OVER, "temp1_max", SENSORS_LM87_TEMP1, 
                                 SENSORS_LM87_TEMP1, RW }, 
                                 LM87_SYSCTL_TEMP1, VALUE(1), 1 },
    { { SENSORS_LM87_TEMP2_HYST, "temp2_min", SENSORS_LM87_TEMP2, 
                                 SENSORS_LM87_TEMP2, RW }, 
                                 LM87_SYSCTL_TEMP2, VALUE(2), 1 },
    { { SENSORS_LM87_TEMP2_OVER, "temp2_max", SENSORS_LM87_TEMP2, 
                                 SENSORS_LM87_TEMP2, RW }, 
                                 LM87_SYSCTL_TEMP2, VALUE(1), 1 },
    { { SENSORS_LM87_TEMP3_HYST, "temp3_min", SENSORS_LM87_TEMP3, 
                                 SENSORS_LM87_TEMP3, RW }, 
                                 LM87_SYSCTL_TEMP3, VALUE(2), 1 },
    { { SENSORS_LM87_TEMP3_OVER, "temp3_max", SENSORS_LM87_TEMP3, 
                                 SENSORS_LM87_TEMP3, RW }, 
                                 LM87_SYSCTL_TEMP3, VALUE(1), 1 },
    { { SENSORS_LM87_VID, "vid", NOMAP, NOMAP, R }, 
                          LM87_SYSCTL_VID, VALUE(1), 3 },
    { { SENSORS_LM87_VRM, "vrm", NOMAP, NOMAP, RW }, 
                          LM87_SYSCTL_VRM, VALUE(1), 1 },
/* Ho hum, this will be wrong if fan1 is disabled, but fan2 isn't.. fix?? */
    { { SENSORS_LM87_FAN1_DIV, "fan1_div", SENSORS_LM87_FAN1, NOMAP, RW }, 
                               LM87_SYSCTL_FAN_DIV, VALUE(1), 0 },
    { { SENSORS_LM87_FAN2_DIV, "fan2_div", SENSORS_LM87_FAN2, NOMAP, RW }, 
                               LM87_SYSCTL_FAN_DIV, VALUE(2), 0 },
    { { SENSORS_LM87_ALARMS, "alarms", NOMAP, NOMAP, R }, 
                             LM87_SYSCTL_ALARMS, VALUE(1), 0 },
    { { SENSORS_LM87_ANALOG_OUT, "analog_out", NOMAP, NOMAP, RW }, 
                                 LM87_SYSCTL_ANALOG_OUT, VALUE(1), 0 , 
                                 "aout_output", 3 },
    { { 0 }, 0 }
  };

static sensors_chip_feature mtp008_features[] =
  { 
    { { SENSORS_MTP008_IN0, "in0", NOMAP, NOMAP, R }, 
                            MTP008_SYSCTL_IN0, VALUE(3), 2 },
    { { SENSORS_MTP008_IN1, "in1", NOMAP, NOMAP, R }, 
                            MTP008_SYSCTL_IN1, VALUE(3), 2 },
    { { SENSORS_MTP008_IN2, "in2", NOMAP, NOMAP, R }, 
                            MTP008_SYSCTL_IN2, VALUE(3), 2 },
    { { SENSORS_MTP008_IN3, "in3", NOMAP, NOMAP, R }, 
                            MTP008_SYSCTL_IN3, VALUE(3), 2 },
    { { SENSORS_MTP008_IN4, "in4", NOMAP, NOMAP, R }, 
                            MTP008_SYSCTL_IN4, VALUE(3), 2 },
    { { SENSORS_MTP008_IN5, "in5", NOMAP, NOMAP, R }, 
                            MTP008_SYSCTL_IN5, VALUE(3), 2 },
    { { SENSORS_MTP008_IN6, "in6", NOMAP, NOMAP, R }, 
                            MTP008_SYSCTL_IN6, VALUE(3), 2 },
    { { SENSORS_MTP008_IN0_MIN, "in0_min", SENSORS_MTP008_IN0, 
                                SENSORS_MTP008_IN0, RW }, 
                                MTP008_SYSCTL_IN0, VALUE(1), 2 },
    { { SENSORS_MTP008_IN1_MIN, "in1_min", SENSORS_MTP008_IN1, 
                                SENSORS_MTP008_IN1, RW }, 
                                MTP008_SYSCTL_IN1, VALUE(1), 2 },
    { { SENSORS_MTP008_IN2_MIN, "in2_min", SENSORS_MTP008_IN2, 
                                SENSORS_MTP008_IN2, RW }, 
                                MTP008_SYSCTL_IN2, VALUE(1), 2 },
    { { SENSORS_MTP008_IN3_MIN, "in3_min", SENSORS_MTP008_IN3, 
                                SENSORS_MTP008_IN3, RW }, 
                                MTP008_SYSCTL_IN3, VALUE(1), 2 },
    { { SENSORS_MTP008_IN4_MIN, "in4_min", SENSORS_MTP008_IN4, 
                                SENSORS_MTP008_IN4, RW }, 
                                MTP008_SYSCTL_IN4, VALUE(1), 2 },
    { { SENSORS_MTP008_IN5_MIN, "in5_min", SENSORS_MTP008_IN5, 
                                SENSORS_MTP008_IN5, RW }, 
                                MTP008_SYSCTL_IN5, VALUE(1), 2 },
    { { SENSORS_MTP008_IN6_MIN, "in6_min", SENSORS_MTP008_IN6, 
                                SENSORS_MTP008_IN6, RW }, 
                                MTP008_SYSCTL_IN6, VALUE(1), 2 },
    { { SENSORS_MTP008_IN0_MAX, "in0_max", SENSORS_MTP008_IN0, 
                                SENSORS_MTP008_IN0, RW }, 
                                MTP008_SYSCTL_IN0, VALUE(2), 2 },
    { { SENSORS_MTP008_IN1_MAX, "in1_max", SENSORS_MTP008_IN1, 
                                SENSORS_MTP008_IN1, RW }, 
                                MTP008_SYSCTL_IN1, VALUE(2), 2 },
    { { SENSORS_MTP008_IN2_MAX, "in2_max", SENSORS_MTP008_IN2, 
                                SENSORS_MTP008_IN2, RW }, 
                                MTP008_SYSCTL_IN2, VALUE(2), 2 },
    { { SENSORS_MTP008_IN3_MAX, "in3_max", SENSORS_MTP008_IN3, 
                                SENSORS_MTP008_IN3, RW }, 
                                MTP008_SYSCTL_IN3, VALUE(2), 2 },
    { { SENSORS_MTP008_IN4_MAX, "in4_max", SENSORS_MTP008_IN4, 
                                SENSORS_MTP008_IN4, RW }, 
                                MTP008_SYSCTL_IN4, VALUE(2), 2 },
    { { SENSORS_MTP008_IN5_MAX, "in5_max", SENSORS_MTP008_IN5, 
                                SENSORS_MTP008_IN5, RW }, 
                                MTP008_SYSCTL_IN5, VALUE(2), 2 },
    { { SENSORS_MTP008_IN6_MAX, "in6_max", SENSORS_MTP008_IN6, 
                                SENSORS_MTP008_IN6, RW }, 
                                MTP008_SYSCTL_IN6, VALUE(2), 2 },
    { { SENSORS_MTP008_FAN1, "fan1", NOMAP, NOMAP, R }, 
                             MTP008_SYSCTL_FAN1, VALUE(2), 0 },
    { { SENSORS_MTP008_FAN2, "fan2", NOMAP, NOMAP, R }, 
                             MTP008_SYSCTL_FAN2, VALUE(2), 0 },
    { { SENSORS_MTP008_FAN3, "fan3", NOMAP, NOMAP, R }, 
                             MTP008_SYSCTL_FAN3, VALUE(2), 0 },
    { { SENSORS_MTP008_FAN1_MIN, "fan1_min", SENSORS_MTP008_FAN1, 
                                 SENSORS_MTP008_FAN1, RW }, 
                                 MTP008_SYSCTL_FAN1, VALUE(1), 0 },
    { { SENSORS_MTP008_FAN2_MIN, "fan2_min", SENSORS_MTP008_FAN2, 
                                 SENSORS_MTP008_FAN2, RW }, 
                                 MTP008_SYSCTL_FAN2, VALUE(1), 0 },
    { { SENSORS_MTP008_FAN3_MIN, "fan3_min", SENSORS_MTP008_FAN3, 
                                 SENSORS_MTP008_FAN3, RW }, 
                                 MTP008_SYSCTL_FAN3, VALUE(1), 0 },
    { { SENSORS_MTP008_TEMP1, "temp1", NOMAP, NOMAP, R }, 
                              MTP008_SYSCTL_TEMP1, VALUE(3), 1 },
    { { SENSORS_MTP008_TEMP2, "temp2", NOMAP, NOMAP, R }, 
                              MTP008_SYSCTL_TEMP2, VALUE(3), 1 },
    { { SENSORS_MTP008_TEMP3, "temp3", NOMAP, NOMAP, R }, 
                              MTP008_SYSCTL_TEMP3, VALUE(3), 1 },
    { { SENSORS_MTP008_TEMP1_OVER, "temp1_over", SENSORS_MTP008_TEMP1, 
                                   SENSORS_MTP008_TEMP1, RW }, 
                                   MTP008_SYSCTL_TEMP1, VALUE(1), 1 },
    { { SENSORS_MTP008_TEMP2_OVER, "temp2_over", SENSORS_MTP008_TEMP2, 
                                   SENSORS_MTP008_TEMP2, RW }, 
                                   MTP008_SYSCTL_TEMP2, VALUE(1), 1 },
    { { SENSORS_MTP008_TEMP3_OVER, "temp3_over", SENSORS_MTP008_TEMP3, 
                                   SENSORS_MTP008_TEMP3, RW }, 
                                   MTP008_SYSCTL_TEMP3, VALUE(1), 1 },
    { { SENSORS_MTP008_TEMP1_HYST, "temp1_hyst", SENSORS_MTP008_TEMP1, 
                                   SENSORS_MTP008_TEMP1, RW }, 
                                   MTP008_SYSCTL_TEMP1, VALUE(2), 1 },
    { { SENSORS_MTP008_TEMP2_HYST, "temp2_hyst", SENSORS_MTP008_TEMP2, 
                                   SENSORS_MTP008_TEMP2, RW }, 
                                   MTP008_SYSCTL_TEMP2, VALUE(2), 1 },
    { { SENSORS_MTP008_TEMP3_HYST, "temp3_hyst", SENSORS_MTP008_TEMP3, 
                                   SENSORS_MTP008_TEMP3, RW }, 
                                   MTP008_SYSCTL_TEMP3, VALUE(2), 1 },
    { { SENSORS_MTP008_VID, "vid", NOMAP, NOMAP, R }, 
                            MTP008_SYSCTL_VID, VALUE(1), 2 },
    { { SENSORS_MTP008_FAN1_DIV, "fan1_div", NOMAP, NOMAP, RW }, 
                                 MTP008_SYSCTL_FAN_DIV, VALUE(1), 0 },
    { { SENSORS_MTP008_FAN2_DIV, "fan2_div", NOMAP, NOMAP, RW }, 
                                 MTP008_SYSCTL_FAN_DIV, VALUE(2), 0 },
    { { SENSORS_MTP008_FAN3_DIV, "fan3_div", NOMAP, NOMAP, RW }, 
                                 MTP008_SYSCTL_FAN_DIV, VALUE(3), 0 },
    { { SENSORS_MTP008_ALARMS, "alarms", NOMAP, NOMAP, R }, 
                               MTP008_SYSCTL_ALARMS, VALUE(1), 0 },
    { { SENSORS_MTP008_BEEP, "beeps", NOMAP, NOMAP, R }, 
                             MTP008_SYSCTL_BEEP, VALUE(1), 0 },
    { { 0 }, 0 }
  };

static sensors_chip_feature ds1621_features[] =
  {
    { { SENSORS_DS1621_TEMP, "temp", NOMAP, NOMAP, R }, 
                             DS1621_SYSCTL_TEMP, VALUE(3), 1 },
    { { SENSORS_DS1621_TEMP_HYST, "temp_hyst", SENSORS_DS1621_TEMP, 
                                  SENSORS_DS1621_TEMP, RW }, 
                                  DS1621_SYSCTL_TEMP, VALUE(2), 1, "temp1_min", 
                                  3 },
    { { SENSORS_DS1621_TEMP_OVER, "temp_over", SENSORS_DS1621_TEMP, 
                                  SENSORS_DS1621_TEMP, RW }, 
                                  DS1621_SYSCTL_TEMP, VALUE(1), 1 },
    { { SENSORS_DS1621_ALARMS, "alarms", NOMAP, NOMAP, R }, 
                               DS1621_SYSCTL_ALARMS, VALUE(1), 0 },
    { { 0 }, 0 }
  };

static sensors_chip_feature it87_features[] =
  {
    { { SENSORS_IT87_IN0, "in0", NOMAP, NOMAP, R }, 
                          IT87_SYSCTL_IN0, VALUE(3), 2 },
    { { SENSORS_IT87_IN1, "in1", NOMAP, NOMAP, R }, 
                          IT87_SYSCTL_IN1, VALUE(3), 2 },
    { { SENSORS_IT87_IN2, "in2", NOMAP, NOMAP, R }, 
                          IT87_SYSCTL_IN2, VALUE(3), 2 },
    { { SENSORS_IT87_IN3, "in3", NOMAP, NOMAP, R }, 
                          IT87_SYSCTL_IN3, VALUE(3), 2 },
    { { SENSORS_IT87_IN4, "in4", NOMAP, NOMAP, R }, 
                          IT87_SYSCTL_IN4, VALUE(3), 2 },
    { { SENSORS_IT87_IN5, "in5", NOMAP, NOMAP, R }, 
                          IT87_SYSCTL_IN5, VALUE(3), 2 },
    { { SENSORS_IT87_IN6, "in6", NOMAP, NOMAP, R }, 
                          IT87_SYSCTL_IN6, VALUE(3), 2 },
    { { SENSORS_IT87_IN7, "in7", NOMAP, NOMAP, R }, 
                          IT87_SYSCTL_IN7, VALUE(3), 2 },
    { { SENSORS_IT87_IN8, "in8", NOMAP, NOMAP, R }, 
                          IT87_SYSCTL_IN8, VALUE(3), 2 },
    { { SENSORS_IT87_IN0_MIN, "in0_min", SENSORS_IT87_IN0, SENSORS_IT87_IN0, RW }, 
                              IT87_SYSCTL_IN0, VALUE(1), 2 },
    { { SENSORS_IT87_IN1_MIN, "in1_min", SENSORS_IT87_IN1, SENSORS_IT87_IN1, RW }, 
                              IT87_SYSCTL_IN1, VALUE(1), 2 },
    { { SENSORS_IT87_IN2_MIN, "in2_min", SENSORS_IT87_IN2, SENSORS_IT87_IN2, RW }, 
                              IT87_SYSCTL_IN2, VALUE(1), 2 },
    { { SENSORS_IT87_IN3_MIN, "in3_min", SENSORS_IT87_IN3, SENSORS_IT87_IN3, RW }, 
                              IT87_SYSCTL_IN3, VALUE(1), 2 },
    { { SENSORS_IT87_IN4_MIN, "in4_min", SENSORS_IT87_IN4, SENSORS_IT87_IN4, RW }, 
                              IT87_SYSCTL_IN4, VALUE(1), 2 },
    { { SENSORS_IT87_IN5_MIN, "in5_min", SENSORS_IT87_IN5, SENSORS_IT87_IN5, RW }, 
                              IT87_SYSCTL_IN5, VALUE(1), 2 },
    { { SENSORS_IT87_IN6_MIN, "in6_min", SENSORS_IT87_IN6, SENSORS_IT87_IN6, RW }, 
                              IT87_SYSCTL_IN6, VALUE(1), 2 },
    { { SENSORS_IT87_IN7_MIN, "in7_min", SENSORS_IT87_IN7, SENSORS_IT87_IN7, RW }, 
                              IT87_SYSCTL_IN7, VALUE(1), 2 },
    { { SENSORS_IT87_IN0_MAX, "in0_max", SENSORS_IT87_IN0, SENSORS_IT87_IN0, RW }, 
                              IT87_SYSCTL_IN0, VALUE(2), 2 },
    { { SENSORS_IT87_IN1_MAX, "in1_max", SENSORS_IT87_IN1, SENSORS_IT87_IN1, RW }, 
                              IT87_SYSCTL_IN1, VALUE(2), 2 },
    { { SENSORS_IT87_IN2_MAX, "in2_max", SENSORS_IT87_IN2, SENSORS_IT87_IN2, RW }, 
                              IT87_SYSCTL_IN2, VALUE(2), 2 },
    { { SENSORS_IT87_IN3_MAX, "in3_max", SENSORS_IT87_IN3, SENSORS_IT87_IN3, RW }, 
                              IT87_SYSCTL_IN3, VALUE(2), 2 },
    { { SENSORS_IT87_IN4_MAX, "in4_max", SENSORS_IT87_IN4, SENSORS_IT87_IN4, RW }, 
                              IT87_SYSCTL_IN4, VALUE(2), 2 },
    { { SENSORS_IT87_IN5_MAX, "in5_max", SENSORS_IT87_IN5, SENSORS_IT87_IN5, RW }, 
                              IT87_SYSCTL_IN5, VALUE(2), 2 },
    { { SENSORS_IT87_IN6_MAX, "in6_max", SENSORS_IT87_IN6, SENSORS_IT87_IN6, RW }, 
                              IT87_SYSCTL_IN6, VALUE(2), 2 },
    { { SENSORS_IT87_IN7_MAX, "in7_max", SENSORS_IT87_IN7, SENSORS_IT87_IN7, RW }, 
                              IT87_SYSCTL_IN7, VALUE(2), 2 },
    { { SENSORS_IT87_FAN1, "fan1", NOMAP, NOMAP, R }, 
                           IT87_SYSCTL_FAN1, VALUE(2), 0 },
    { { SENSORS_IT87_FAN2, "fan2", NOMAP, NOMAP, R }, 
                           IT87_SYSCTL_FAN2, VALUE(2), 0 },
    { { SENSORS_IT87_FAN3, "fan3", NOMAP, NOMAP, R }, 
                           IT87_SYSCTL_FAN3, VALUE(2), 0 },
    { { SENSORS_IT87_FAN4, "fan4", NOMAP, NOMAP, R },
                           NOSYSCTL, VALUE(2), 0 },
    { { SENSORS_IT87_FAN5, "fan5", NOMAP, NOMAP, R },
                           NOSYSCTL, VALUE(2), 0 },
    { { SENSORS_IT87_FAN1_MIN, "fan1_min", SENSORS_IT87_FAN1, SENSORS_IT87_FAN1, 
                               RW }, 
                               IT87_SYSCTL_FAN1, VALUE(1), 0 },
    { { SENSORS_IT87_FAN2_MIN, "fan2_min", SENSORS_IT87_FAN2, SENSORS_IT87_FAN2, 
                               RW }, 
                               IT87_SYSCTL_FAN2, VALUE(1), 0 },
    { { SENSORS_IT87_FAN3_MIN, "fan3_min", SENSORS_IT87_FAN3, SENSORS_IT87_FAN3, 
                               RW }, 
                               IT87_SYSCTL_FAN3, VALUE(1), 0 },
    { { SENSORS_IT87_FAN4_MIN, "fan4_min", SENSORS_IT87_FAN4, SENSORS_IT87_FAN4,
                               RW },
                               NOSYSCTL, VALUE(1), 0 },
    { { SENSORS_IT87_FAN5_MIN, "fan5_min", SENSORS_IT87_FAN5, SENSORS_IT87_FAN5,
                               RW },
                               NOSYSCTL, VALUE(1), 0 },
    { { SENSORS_IT87_TEMP1, "temp1", NOMAP, NOMAP, R }, 
                            IT87_SYSCTL_TEMP1, VALUE(3), 1 },
    { { SENSORS_IT87_TEMP2, "temp2", NOMAP, NOMAP, R }, 
                            IT87_SYSCTL_TEMP2, VALUE(3), 1 },
    { { SENSORS_IT87_TEMP3, "temp3", NOMAP, NOMAP, R }, 
                            IT87_SYSCTL_TEMP3, VALUE(3), 1 },
    { { SENSORS_IT87_TEMP1_LOW, "temp1_low", SENSORS_IT87_TEMP1, 
                                SENSORS_IT87_TEMP1, RW }, 
                                IT87_SYSCTL_TEMP1, VALUE(2), 1 },
    { { SENSORS_IT87_TEMP2_LOW, "temp2_low", SENSORS_IT87_TEMP2, 
                                SENSORS_IT87_TEMP2, RW }, 
                                IT87_SYSCTL_TEMP2, VALUE(2), 1 },
    { { SENSORS_IT87_TEMP3_LOW, "temp3_low", SENSORS_IT87_TEMP3, 
                                SENSORS_IT87_TEMP3, RW }, 
                                IT87_SYSCTL_TEMP3, VALUE(2), 1 },
    { { SENSORS_IT87_TEMP1_HIGH, "temp1_over", SENSORS_IT87_TEMP1, 
                                 SENSORS_IT87_TEMP1, RW }, 
                                 IT87_SYSCTL_TEMP1, VALUE(1), 1 },
    { { SENSORS_IT87_TEMP2_HIGH, "temp2_over", SENSORS_IT87_TEMP2, 
                                 SENSORS_IT87_TEMP2, RW }, 
                                 IT87_SYSCTL_TEMP2, VALUE(1), 1 },
    { { SENSORS_IT87_TEMP3_HIGH, "temp3_over", SENSORS_IT87_TEMP3, 
                                 SENSORS_IT87_TEMP3, RW }, 
                                 IT87_SYSCTL_TEMP3, VALUE(1), 1 },
    { { SENSORS_IT87_VID, "vid", NOMAP, NOMAP, R }, 
                          IT87_SYSCTL_VID, VALUE(1), 2 },
    { { SENSORS_IT87_FAN1_DIV, "fan1_div", SENSORS_IT87_FAN1, NOMAP, RW }, 
                               IT87_SYSCTL_FAN_DIV, VALUE(1), 0 },
    { { SENSORS_IT87_FAN2_DIV, "fan2_div", SENSORS_IT87_FAN2, NOMAP, RW }, 
                               IT87_SYSCTL_FAN_DIV, VALUE(2), 0 },
    { { SENSORS_IT87_FAN3_DIV, "fan3_div", SENSORS_IT87_FAN3, NOMAP, RW }, 
                               IT87_SYSCTL_FAN_DIV, VALUE(3), 0 },
    { { SENSORS_IT87_ALARMS, "alarms", NOMAP, NOMAP, R }, 
                             IT87_SYSCTL_ALARMS, VALUE(1) },
    { { SENSORS_IT87_SENS1, "sensor1", SENSORS_IT87_TEMP1, NOMAP, RW }, 
                            IT87_SYSCTL_SENS1, VALUE(1), 0 },
    { { SENSORS_IT87_SENS2, "sensor2", SENSORS_IT87_TEMP2, NOMAP, RW }, 
                            IT87_SYSCTL_SENS2, VALUE(1), 0 },
    { { SENSORS_IT87_SENS3, "sensor3", SENSORS_IT87_TEMP3, NOMAP, RW }, 
                            IT87_SYSCTL_SENS3, VALUE(1), 0 },
    { { 0 }, 0 }
  };

static sensors_chip_feature fscpos_features[] =
  {
    { { SENSORS_FSCPOS_REV, "rev", NOMAP, NOMAP, R }, 
                            FSCPOS_SYSCTL_REV, VALUE(1), 0 },
    { { SENSORS_FSCPOS_EVENT, "event", NOMAP, NOMAP, R }, 
                              FSCPOS_SYSCTL_EVENT, VALUE(1), 0 },
    { { SENSORS_FSCPOS_CONTROL, "control", NOMAP, NOMAP, RW }, 
                                FSCPOS_SYSCTL_CONTROL, VALUE(1), 0 },
    { { SENSORS_FSCPOS_VOLTAGE1, "volt12", NOMAP, NOMAP, R }, 
                                 FSCPOS_SYSCTL_VOLT0, VALUE(1), 2, "in0_input", 
                                 3 },
    { { SENSORS_FSCPOS_VOLTAGE2, "volt5", NOMAP, NOMAP, R }, 
                                 FSCPOS_SYSCTL_VOLT1, VALUE(1), 2, "in1_input", 
                                 3 },
    { { SENSORS_FSCPOS_VOLTAGE3, "voltbatt", NOMAP, NOMAP, R }, 
                                 FSCPOS_SYSCTL_VOLT2, VALUE(1), 2, "in2_input", 
                                 3 },
    { { SENSORS_FSCPOS_TEMP1, "temp1", NOMAP, NOMAP, R }, 
                              FSCPOS_SYSCTL_TEMP0, VALUE(2), 0 },
    { { SENSORS_FSCPOS_TEMP2, "temp2", NOMAP, NOMAP, R }, 
                              FSCPOS_SYSCTL_TEMP1, VALUE(2), 0 },
    { { SENSORS_FSCPOS_TEMP3, "temp3", NOMAP, NOMAP, R }, 
                              FSCPOS_SYSCTL_TEMP2, VALUE(2), 0 },
    { { SENSORS_FSCPOS_TEMP1_STATE, "temp1_state", SENSORS_FSCPOS_TEMP1, 
                                    SENSORS_FSCPOS_TEMP1, RW }, 
                                    FSCPOS_SYSCTL_TEMP0, VALUE(1), 0 },
    { { SENSORS_FSCPOS_TEMP2_STATE, "temp2_state", SENSORS_FSCPOS_TEMP2, 
                                    SENSORS_FSCPOS_TEMP2, RW }, 
                                    FSCPOS_SYSCTL_TEMP1, VALUE(1), 0 },
    { { SENSORS_FSCPOS_TEMP3_STATE, "temp3_state", SENSORS_FSCPOS_TEMP3, 
                                    SENSORS_FSCPOS_TEMP3, RW }, 
                                    FSCPOS_SYSCTL_TEMP2, VALUE(1), 0 },
    { { SENSORS_FSCPOS_FAN1, "fan1", NOMAP, NOMAP, R }, 
                             FSCPOS_SYSCTL_FAN0, VALUE(4), 0 },
    { { SENSORS_FSCPOS_FAN2, "fan2", NOMAP, NOMAP, R }, 
                             FSCPOS_SYSCTL_FAN1, VALUE(4), 0 },
    { { SENSORS_FSCPOS_FAN3, "fan3", NOMAP, NOMAP, R }, 
                             FSCPOS_SYSCTL_FAN2, VALUE(4), 0 },
    { { SENSORS_FSCPOS_FAN1_MIN, "fan1_min", SENSORS_FSCPOS_FAN1, 
                                 SENSORS_FSCPOS_FAN1, RW }, 
                                 FSCPOS_SYSCTL_FAN0, VALUE(2), 0, "pwm1", 0 },
    { { SENSORS_FSCPOS_FAN2_MIN, "fan2_min", SENSORS_FSCPOS_FAN2, 
                                 SENSORS_FSCPOS_FAN2, RW }, 
                                 FSCPOS_SYSCTL_FAN1, VALUE(2), 0, "pwm2", 0 },
    { { SENSORS_FSCPOS_FAN1_STATE, "fan1_state", SENSORS_FSCPOS_FAN1, 
                                   SENSORS_FSCPOS_FAN1, RW }, 
                                   FSCPOS_SYSCTL_FAN0, VALUE(1), 0 },
    { { SENSORS_FSCPOS_FAN2_STATE, "fan2_state", SENSORS_FSCPOS_FAN2, 
                                   SENSORS_FSCPOS_FAN2, RW }, 
                                   FSCPOS_SYSCTL_FAN1, VALUE(1), 0 },
    { { SENSORS_FSCPOS_FAN3_STATE, "fan3_state", SENSORS_FSCPOS_FAN3, 
                                   SENSORS_FSCPOS_FAN3, RW }, 
                                   FSCPOS_SYSCTL_FAN2, VALUE(1), 0 },
    { { SENSORS_FSCPOS_FAN1_RIPPLE, "fan1_ripple", SENSORS_FSCPOS_FAN1, 
                                    SENSORS_FSCPOS_FAN1, RW }, 
                                    FSCPOS_SYSCTL_FAN0, VALUE(3), 0 },
    { { SENSORS_FSCPOS_FAN2_RIPPLE, "fan2_ripple", SENSORS_FSCPOS_FAN2, 
                                    SENSORS_FSCPOS_FAN2, RW }, 
                                    FSCPOS_SYSCTL_FAN1, VALUE(3), 0 },
    { { SENSORS_FSCPOS_FAN3_RIPPLE, "fan3_ripple", SENSORS_FSCPOS_FAN3, 
                                    SENSORS_FSCPOS_FAN3, RW }, 
                                    FSCPOS_SYSCTL_FAN2, VALUE(3), 0 },
    { { SENSORS_FSCPOS_WDOG_PRESET, "wdog_preset", NOMAP, NOMAP, RW }, 
                                    FSCPOS_SYSCTL_WDOG, VALUE(1), 0 },
    { { SENSORS_FSCPOS_WDOG_STATE, "wdog_state", SENSORS_FSCPOS_WDOG_PRESET, 
                                   SENSORS_FSCPOS_WDOG_PRESET, RW }, 
                                   FSCPOS_SYSCTL_WDOG, VALUE(2), 0 },
    { { SENSORS_FSCPOS_WDOG_CONTROL, "wdog_control", SENSORS_FSCPOS_WDOG_PRESET, 
                                     SENSORS_FSCPOS_WDOG_PRESET, RW }, 
                                     FSCPOS_SYSCTL_WDOG, VALUE(3), 0 },
    /* below this line are features only exported with the new fschmd drv */
    { { SENSORS_FSCHMD_TEMP_ALARM(1), "temp1_alarm", SENSORS_FSCPOS_TEMP1, 
                                    NOMAP, R }, NOSYSCTL, VALUE(1), 0 },
    { { SENSORS_FSCHMD_TEMP_ALARM(2), "temp2_alarm", SENSORS_FSCPOS_TEMP2, 
                                    NOMAP, R }, NOSYSCTL, VALUE(1), 0 },
    { { SENSORS_FSCHMD_TEMP_ALARM(3), "temp3_alarm", SENSORS_FSCPOS_TEMP3, 
                                    NOMAP, R }, NOSYSCTL, VALUE(1), 0 },
    { { SENSORS_FSCHMD_TEMP_FAULT(1), "temp1_fault", SENSORS_FSCPOS_TEMP1, 
                                    NOMAP, R }, NOSYSCTL, VALUE(1), 0 },
    { { SENSORS_FSCHMD_TEMP_FAULT(2), "temp2_fault", SENSORS_FSCPOS_TEMP2, 
                                    NOMAP, R }, NOSYSCTL, VALUE(1), 0 },
    { { SENSORS_FSCHMD_TEMP_FAULT(3), "temp3_fault", SENSORS_FSCPOS_TEMP3, 
                                    NOMAP, R }, NOSYSCTL, VALUE(1), 0 },
    { { SENSORS_FSCHMD_FAN_ALARM(1), "fan1_alarm", SENSORS_FSCPOS_FAN1, 
                                    NOMAP, R }, NOSYSCTL, VALUE(1), 0 },
    { { SENSORS_FSCHMD_FAN_ALARM(2), "fan2_alarm", SENSORS_FSCPOS_FAN2, 
                                    NOMAP, R }, NOSYSCTL, VALUE(1), 0 },
    { { SENSORS_FSCHMD_FAN_ALARM(3), "fan3_alarm", SENSORS_FSCPOS_FAN3, 
                                    NOMAP, R }, NOSYSCTL, VALUE(1), 0 },
    { { SENSORS_FSCHMD_FAN_FAULT(1), "fan1_fault", SENSORS_FSCPOS_FAN1, 
                                    NOMAP, R }, NOSYSCTL, VALUE(1), 0 },
    { { SENSORS_FSCHMD_FAN_FAULT(2), "fan2_fault", SENSORS_FSCPOS_FAN2, 
                                    NOMAP, R }, NOSYSCTL, VALUE(1), 0 },
    { { SENSORS_FSCHMD_FAN_FAULT(3), "fan3_fault", SENSORS_FSCPOS_FAN3, 
                                    NOMAP, R }, NOSYSCTL, VALUE(1), 0 },
    { { 0 }, 0 }
  };

static sensors_chip_feature fscscy_features[] =
  {
    { { SENSORS_FSCSCY_REV, "rev", NOMAP, NOMAP, R }, 
                            FSCSCY_SYSCTL_REV, VALUE(1), 0 },
    { { SENSORS_FSCSCY_EVENT, "event", NOMAP, NOMAP, R }, 
                              FSCSCY_SYSCTL_EVENT, VALUE(1), 0 },
    { { SENSORS_FSCSCY_CONTROL, "control", NOMAP, NOMAP, RW }, 
                                FSCSCY_SYSCTL_CONTROL, VALUE(1), 0 },
    { { SENSORS_FSCSCY_VOLTAGE1, "volt12", NOMAP, NOMAP, R }, 
                                 FSCSCY_SYSCTL_VOLT0, VALUE(1), 2 },
    { { SENSORS_FSCSCY_VOLTAGE2, "volt5", NOMAP, NOMAP, R }, 
                                 FSCSCY_SYSCTL_VOLT1, VALUE(1), 2 },
    { { SENSORS_FSCSCY_VOLTAGE3, "voltbatt", NOMAP, NOMAP, R }, 
                                 FSCSCY_SYSCTL_VOLT2, VALUE(1), 2 },
    { { SENSORS_FSCSCY_TEMP1, "temp1", NOMAP, NOMAP, R }, 
                              FSCSCY_SYSCTL_TEMP0, VALUE(2), 0 },
    { { SENSORS_FSCSCY_TEMP2, "temp2", NOMAP, NOMAP, R }, 
                              FSCSCY_SYSCTL_TEMP1, VALUE(2), 0 },
    { { SENSORS_FSCSCY_TEMP3, "temp3", NOMAP, NOMAP, R }, 
                              FSCSCY_SYSCTL_TEMP2, VALUE(2), 0 },
    { { SENSORS_FSCSCY_TEMP4, "temp4", NOMAP, NOMAP, R }, 
                              FSCSCY_SYSCTL_TEMP3, VALUE(2), 0 },
    { { SENSORS_FSCSCY_TEMP1_STATE, "temp1_state", SENSORS_FSCSCY_TEMP1, 
                                    SENSORS_FSCSCY_TEMP1, RW }, 
                                    FSCSCY_SYSCTL_TEMP0, VALUE(1), 0 },
    { { SENSORS_FSCSCY_TEMP2_STATE, "temp2_state", SENSORS_FSCSCY_TEMP2, 
                                    SENSORS_FSCSCY_TEMP2, RW }, 
                                    FSCSCY_SYSCTL_TEMP1, VALUE(1), 0 },
    { { SENSORS_FSCSCY_TEMP3_STATE, "temp3_state", SENSORS_FSCSCY_TEMP3, 
                                    SENSORS_FSCSCY_TEMP3, RW }, 
                                    FSCSCY_SYSCTL_TEMP2, VALUE(1), 0 },
    { { SENSORS_FSCSCY_TEMP4_STATE, "temp4_state", SENSORS_FSCSCY_TEMP4, 
                                    SENSORS_FSCSCY_TEMP4, RW }, 
                                    FSCSCY_SYSCTL_TEMP3, VALUE(1), 0 },
    { { SENSORS_FSCSCY_TEMP1_LIM, "temp1_lim", SENSORS_FSCSCY_TEMP1, 
                                  SENSORS_FSCSCY_TEMP1, R }, 
                                  FSCSCY_SYSCTL_TEMP0, VALUE(3), 0 },
    { { SENSORS_FSCSCY_TEMP2_LIM, "temp2_lim", SENSORS_FSCSCY_TEMP2, 
                                  SENSORS_FSCSCY_TEMP2, R }, 
                                  FSCSCY_SYSCTL_TEMP1, VALUE(3), 0 },
    { { SENSORS_FSCSCY_TEMP3_LIM, "temp3_lim", SENSORS_FSCSCY_TEMP3, 
                                  SENSORS_FSCSCY_TEMP3, R }, 
                                  FSCSCY_SYSCTL_TEMP2, VALUE(3), 0 },
    { { SENSORS_FSCSCY_TEMP4_LIM, "temp4_lim", SENSORS_FSCSCY_TEMP4, 
                                  SENSORS_FSCSCY_TEMP4, R }, 
                                  FSCSCY_SYSCTL_TEMP3, VALUE(3), 0 },
    { { SENSORS_FSCSCY_TEMP1_MIN, "temp1_min", SENSORS_FSCSCY_TEMP1, 
                                  SENSORS_FSCSCY_TEMP1, R }, 
                                  FSCSCY_SYSCTL_TEMP0, VALUE(4), 0 },
    { { SENSORS_FSCSCY_TEMP2_MIN, "temp2_min", SENSORS_FSCSCY_TEMP2, 
                                  SENSORS_FSCSCY_TEMP2, R }, 
                                  FSCSCY_SYSCTL_TEMP1, VALUE(4), 0 },
    { { SENSORS_FSCSCY_TEMP3_MIN, "temp3_min", SENSORS_FSCSCY_TEMP3, 
                                  SENSORS_FSCSCY_TEMP3, R }, 
                                  FSCSCY_SYSCTL_TEMP2, VALUE(4), 0 },
    { { SENSORS_FSCSCY_TEMP4_MIN, "temp4_min", SENSORS_FSCSCY_TEMP4, 
                                  SENSORS_FSCSCY_TEMP4, R }, 
                                  FSCSCY_SYSCTL_TEMP3, VALUE(4), 0 },
    { { SENSORS_FSCSCY_TEMP1_MAX, "temp1_max", SENSORS_FSCSCY_TEMP1, 
                                  SENSORS_FSCSCY_TEMP1, R }, 
                                  FSCSCY_SYSCTL_TEMP0, VALUE(5), 0 },
    { { SENSORS_FSCSCY_TEMP2_MAX, "temp2_max", SENSORS_FSCSCY_TEMP2, 
                                  SENSORS_FSCSCY_TEMP2, R }, 
                                  FSCSCY_SYSCTL_TEMP1, VALUE(5), 0 },
    { { SENSORS_FSCSCY_TEMP3_MAX, "temp3_max", SENSORS_FSCSCY_TEMP3, 
                                  SENSORS_FSCSCY_TEMP3, R }, 
                                  FSCSCY_SYSCTL_TEMP2, VALUE(5), 0 },
    { { SENSORS_FSCSCY_TEMP4_MAX, "temp4_max", SENSORS_FSCSCY_TEMP4, 
                                  SENSORS_FSCSCY_TEMP4, R }, 
                                  FSCSCY_SYSCTL_TEMP3, VALUE(5), 0 },
    { { SENSORS_FSCSCY_FAN1, "fan1", NOMAP, NOMAP, R }, 
                             FSCSCY_SYSCTL_FAN0, VALUE(4), 0 },
    { { SENSORS_FSCSCY_FAN2, "fan2", NOMAP, NOMAP, R }, 
                             FSCSCY_SYSCTL_FAN1, VALUE(4), 0 },
    { { SENSORS_FSCSCY_FAN3, "fan3", NOMAP, NOMAP, R }, 
                             FSCSCY_SYSCTL_FAN2, VALUE(4), 0 },
    { { SENSORS_FSCSCY_FAN4, "fan4", NOMAP, NOMAP, R }, 
                             FSCSCY_SYSCTL_FAN3, VALUE(4), 0 },
    { { SENSORS_FSCSCY_FAN5, "fan5", NOMAP, NOMAP, R }, 
                             FSCSCY_SYSCTL_FAN4, VALUE(4), 0 },
    { { SENSORS_FSCSCY_FAN6, "fan6", NOMAP, NOMAP, R }, 
                             FSCSCY_SYSCTL_FAN5, VALUE(4), 0 },
    { { SENSORS_FSCSCY_FAN1_MIN, "fan1_min", SENSORS_FSCSCY_FAN1, 
                                 SENSORS_FSCSCY_FAN1, RW }, 
                                 FSCSCY_SYSCTL_FAN0, VALUE(2), 0 },
    { { SENSORS_FSCSCY_FAN2_MIN, "fan2_min", SENSORS_FSCSCY_FAN2, 
                                 SENSORS_FSCSCY_FAN2, RW }, 
                                 FSCSCY_SYSCTL_FAN1, VALUE(2), 0 },
    { { SENSORS_FSCSCY_FAN3_MIN, "fan3_min", SENSORS_FSCSCY_FAN3, 
                                 SENSORS_FSCSCY_FAN3, RW }, 
                                 FSCSCY_SYSCTL_FAN2, VALUE(2), 0 },
    { { SENSORS_FSCSCY_FAN4_MIN, "fan4_min", SENSORS_FSCSCY_FAN4, 
                                 SENSORS_FSCSCY_FAN4, RW }, 
                                 FSCSCY_SYSCTL_FAN3, VALUE(2), 0 },
    { { SENSORS_FSCSCY_FAN5_MIN, "fan5_min", SENSORS_FSCSCY_FAN5, 
                                 SENSORS_FSCSCY_FAN5, RW }, 
                                 FSCSCY_SYSCTL_FAN4, VALUE(2), 0 },
    { { SENSORS_FSCSCY_FAN6_MIN, "fan6_min", SENSORS_FSCSCY_FAN6, 
                                 SENSORS_FSCSCY_FAN6, RW }, 
                                 FSCSCY_SYSCTL_FAN5, VALUE(2), 0 },
    { { SENSORS_FSCSCY_FAN1_STATE, "fan1_state", SENSORS_FSCSCY_FAN1, 
                                   SENSORS_FSCSCY_FAN1, RW }, 
                                   FSCSCY_SYSCTL_FAN0, VALUE(1), 0 },
    { { SENSORS_FSCSCY_FAN2_STATE, "fan2_state", SENSORS_FSCSCY_FAN2, 
                                   SENSORS_FSCSCY_FAN2, RW }, 
                                   FSCSCY_SYSCTL_FAN1, VALUE(1), 0 },
    { { SENSORS_FSCSCY_FAN3_STATE, "fan3_state", SENSORS_FSCSCY_FAN3, 
                                   SENSORS_FSCSCY_FAN3, RW }, 
                                   FSCSCY_SYSCTL_FAN2, VALUE(1), 0 },
    { { SENSORS_FSCSCY_FAN4_STATE, "fan4_state", SENSORS_FSCSCY_FAN4, 
                                   SENSORS_FSCSCY_FAN4, RW }, 
                                   FSCSCY_SYSCTL_FAN3, VALUE(1), 0 },
    { { SENSORS_FSCSCY_FAN5_STATE, "fan5_state", SENSORS_FSCSCY_FAN5, 
                                   SENSORS_FSCSCY_FAN5, RW }, 
                                   FSCSCY_SYSCTL_FAN4, VALUE(1), 0 },
    { { SENSORS_FSCSCY_FAN6_STATE, "fan6_state", SENSORS_FSCSCY_FAN6, 
                                   SENSORS_FSCSCY_FAN6, RW }, 
                                   FSCSCY_SYSCTL_FAN5, VALUE(1), 0 },
    { { SENSORS_FSCSCY_FAN1_RIPPLE, "fan1_ripple", SENSORS_FSCSCY_FAN1, 
                                    SENSORS_FSCSCY_FAN1, RW }, 
                                    FSCSCY_SYSCTL_FAN0, VALUE(3), 0 },
    { { SENSORS_FSCSCY_FAN2_RIPPLE, "fan2_ripple", SENSORS_FSCSCY_FAN2, 
                                    SENSORS_FSCSCY_FAN2, RW }, 
                                    FSCSCY_SYSCTL_FAN1, VALUE(3), 0 },
    { { SENSORS_FSCSCY_FAN3_RIPPLE, "fan3_ripple", SENSORS_FSCSCY_FAN3, 
                                    SENSORS_FSCSCY_FAN3, RW }, 
                                    FSCSCY_SYSCTL_FAN2, VALUE(3), 0 },
    { { SENSORS_FSCSCY_FAN4_RIPPLE, "fan4_ripple", SENSORS_FSCSCY_FAN4, 
                                    SENSORS_FSCSCY_FAN4, RW }, 
                                    FSCSCY_SYSCTL_FAN3, VALUE(3), 0 },
    { { SENSORS_FSCSCY_FAN5_RIPPLE, "fan5_ripple", SENSORS_FSCSCY_FAN5, 
                                    SENSORS_FSCSCY_FAN5, RW }, 
                                    FSCSCY_SYSCTL_FAN4, VALUE(3), 0 },
    { { SENSORS_FSCSCY_FAN6_RIPPLE, "fan6_ripple", SENSORS_FSCSCY_FAN6, 
                                    SENSORS_FSCSCY_FAN6, RW }, 
                                    FSCSCY_SYSCTL_FAN5, VALUE(3), 0 },
    { { SENSORS_FSCSCY_WDOG_PRESET, "wdog_preset", NOMAP, NOMAP, RW }, 
                                    FSCSCY_SYSCTL_WDOG, VALUE(1), 0 },
    { { SENSORS_FSCSCY_WDOG_STATE, "wdog_state", SENSORS_FSCSCY_WDOG_PRESET, 
                                   SENSORS_FSCSCY_WDOG_PRESET, RW }, 
                                   FSCSCY_SYSCTL_WDOG, VALUE(2), 0 },
    { { SENSORS_FSCSCY_WDOG_CONTROL, "wdog_control", SENSORS_FSCSCY_WDOG_PRESET, 
                                     SENSORS_FSCSCY_WDOG_PRESET, RW }, 
                                     FSCSCY_SYSCTL_WDOG, VALUE(3), 0 },
    /* below this line are features only exported with the new fschmd drv */
    { { SENSORS_FSCHMD_TEMP_ALARM(1), "temp1_alarm", SENSORS_FSCSCY_TEMP1, 
                                    NOMAP, R }, NOSYSCTL, VALUE(1), 0 },
    { { SENSORS_FSCHMD_TEMP_ALARM(2), "temp2_alarm", SENSORS_FSCSCY_TEMP2, 
                                    NOMAP, R }, NOSYSCTL, VALUE(1), 0 },
    { { SENSORS_FSCHMD_TEMP_ALARM(3), "temp3_alarm", SENSORS_FSCSCY_TEMP3, 
                                    NOMAP, R }, NOSYSCTL, VALUE(1), 0 },
    { { SENSORS_FSCHMD_TEMP_ALARM(4), "temp4_alarm", SENSORS_FSCSCY_TEMP4,
                                    NOMAP, R }, NOSYSCTL, VALUE(1), 0 },
    { { SENSORS_FSCHMD_TEMP_FAULT(1), "temp1_fault", SENSORS_FSCSCY_TEMP1, 
                                    NOMAP, R }, NOSYSCTL, VALUE(1), 0 },
    { { SENSORS_FSCHMD_TEMP_FAULT(2), "temp2_fault", SENSORS_FSCSCY_TEMP2, 
                                    NOMAP, R }, NOSYSCTL, VALUE(1), 0 },
    { { SENSORS_FSCHMD_TEMP_FAULT(3), "temp3_fault", SENSORS_FSCSCY_TEMP3, 
                                    NOMAP, R }, NOSYSCTL, VALUE(1), 0 },
    { { SENSORS_FSCHMD_TEMP_FAULT(4), "temp4_fault", SENSORS_FSCSCY_TEMP4,
                                    NOMAP, R }, NOSYSCTL, VALUE(1), 0 },
    { { SENSORS_FSCHMD_FAN_ALARM(1), "fan1_alarm", SENSORS_FSCSCY_FAN1, 
                                    NOMAP, R }, NOSYSCTL, VALUE(1), 0 },
    { { SENSORS_FSCHMD_FAN_ALARM(2), "fan2_alarm", SENSORS_FSCSCY_FAN2, 
                                    NOMAP, R }, NOSYSCTL, VALUE(1), 0 },
    { { SENSORS_FSCHMD_FAN_ALARM(3), "fan3_alarm", SENSORS_FSCSCY_FAN3, 
                                    NOMAP, R }, NOSYSCTL, VALUE(1), 0 },
    { { SENSORS_FSCHMD_FAN_ALARM(4), "fan4_alarm", SENSORS_FSCSCY_FAN4, 
                                    NOMAP, R }, NOSYSCTL, VALUE(1), 0 },
    { { SENSORS_FSCHMD_FAN_ALARM(5), "fan5_alarm", SENSORS_FSCSCY_FAN5, 
                                    NOMAP, R }, NOSYSCTL, VALUE(1), 0 },
    { { SENSORS_FSCHMD_FAN_ALARM(6), "fan6_alarm", SENSORS_FSCSCY_FAN6, 
                                    NOMAP, R }, NOSYSCTL, VALUE(1), 0 },
    { { SENSORS_FSCHMD_FAN_FAULT(1), "fan1_fault", SENSORS_FSCSCY_FAN1, 
                                    NOMAP, R }, NOSYSCTL, VALUE(1), 0 },
    { { SENSORS_FSCHMD_FAN_FAULT(2), "fan2_fault", SENSORS_FSCSCY_FAN2, 
                                    NOMAP, R }, NOSYSCTL, VALUE(1), 0 },
    { { SENSORS_FSCHMD_FAN_FAULT(3), "fan3_fault", SENSORS_FSCSCY_FAN3, 
                                    NOMAP, R }, NOSYSCTL, VALUE(1), 0 },
    { { SENSORS_FSCHMD_FAN_FAULT(4), "fan4_fault", SENSORS_FSCSCY_FAN4, 
                                    NOMAP, R }, NOSYSCTL, VALUE(1), 0 },
    { { SENSORS_FSCHMD_FAN_FAULT(5), "fan5_fault", SENSORS_FSCSCY_FAN5, 
                                    NOMAP, R }, NOSYSCTL, VALUE(1), 0 },
    { { SENSORS_FSCHMD_FAN_FAULT(6), "fan6_fault", SENSORS_FSCSCY_FAN6, 
                                    NOMAP, R }, NOSYSCTL, VALUE(1), 0 },
    { { 0 }, 0 }
  };

  static sensors_chip_feature fscher_features[] =
  {
    { { SENSORS_FSCHER_REV, "rev", NOMAP, NOMAP, R }, 
                            FSCHER_SYSCTL_REV, VALUE(1), 0 },
    { { SENSORS_FSCHER_EVENT, "alarms", NOMAP, NOMAP, R }, 
                              FSCHER_SYSCTL_EVENT, VALUE(1), 0 },
    { { SENSORS_FSCHER_CONTROL, "control", NOMAP, NOMAP, RW }, 
                                FSCHER_SYSCTL_CONTROL, VALUE(1), 0 },
    { { SENSORS_FSCHER_VOLTAGE1, "in0", NOMAP, NOMAP, R }, 
                                 FSCHER_SYSCTL_VOLT0, VALUE(1), 2 },
    { { SENSORS_FSCHER_VOLTAGE2, "in1", NOMAP, NOMAP, R }, 
                                 FSCHER_SYSCTL_VOLT1, VALUE(1), 2 },
    { { SENSORS_FSCHER_VOLTAGE3, "in2", NOMAP, NOMAP, R }, 
                                 FSCHER_SYSCTL_VOLT2, VALUE(1), 2 },
    { { SENSORS_FSCHER_TEMP1, "temp1", NOMAP, NOMAP, R }, 
                              FSCHER_SYSCTL_TEMP0, VALUE(2), 0 },
    { { SENSORS_FSCHER_TEMP2, "temp2", NOMAP, NOMAP, R }, 
                              FSCHER_SYSCTL_TEMP1, VALUE(2), 0 },
    { { SENSORS_FSCHER_TEMP3, "temp3", NOMAP, NOMAP, R }, 
                              FSCHER_SYSCTL_TEMP2, VALUE(2), 0 },
    { { SENSORS_FSCHER_TEMP1_STATE, "temp1_state", SENSORS_FSCHER_TEMP1, 
                                    SENSORS_FSCHER_TEMP1, RW }, 
                                    FSCHER_SYSCTL_TEMP0, VALUE(1), 0 },
    { { SENSORS_FSCHER_TEMP2_STATE, "temp2_state", SENSORS_FSCHER_TEMP2, 
                                    SENSORS_FSCHER_TEMP2, RW }, 
                                    FSCHER_SYSCTL_TEMP1, VALUE(1), 0 },
    { { SENSORS_FSCHER_TEMP3_STATE, "temp3_state", SENSORS_FSCHER_TEMP3, 
                                    SENSORS_FSCHER_TEMP3, RW }, 
                                    FSCHER_SYSCTL_TEMP2, VALUE(1), 0 },
    { { SENSORS_FSCHER_FAN1, "fan1", NOMAP, NOMAP, R }, 
                             FSCHER_SYSCTL_FAN0, VALUE(3), 0 },
    { { SENSORS_FSCHER_FAN2, "fan2", NOMAP, NOMAP, R }, 
                             FSCHER_SYSCTL_FAN1, VALUE(3), 0 },
    { { SENSORS_FSCHER_FAN3, "fan3", NOMAP, NOMAP, R }, 
                             FSCHER_SYSCTL_FAN2, VALUE(3), 0 },
    { { SENSORS_FSCHER_PWM1, "pwm1", SENSORS_FSCHER_FAN1, NOMAP, RW }, 
                             FSCHER_SYSCTL_PWM0, VALUE(1), 0 },
    { { SENSORS_FSCHER_PWM2, "pwm2", SENSORS_FSCHER_FAN2, NOMAP, RW }, 
                             FSCHER_SYSCTL_PWM1, VALUE(1), 0 },
    { { SENSORS_FSCHER_PWM3, "pwm3", SENSORS_FSCHER_FAN3, NOMAP, RW }, 
                             FSCHER_SYSCTL_PWM2, VALUE(1), 0 },
    { { SENSORS_FSCHER_FAN1_STATE, "fan1_state", SENSORS_FSCHER_FAN1, 
                                   SENSORS_FSCHER_FAN1, RW }, 
                                   FSCHER_SYSCTL_FAN0, VALUE(1), 0 },
    { { SENSORS_FSCHER_FAN2_STATE, "fan2_state", SENSORS_FSCHER_FAN2, 
                                   SENSORS_FSCHER_FAN2, RW }, 
                                   FSCHER_SYSCTL_FAN1, VALUE(1), 0 },
    { { SENSORS_FSCHER_FAN3_STATE, "fan3_state", SENSORS_FSCHER_FAN3, 
                                   SENSORS_FSCHER_FAN3, RW }, 
                                   FSCHER_SYSCTL_FAN2, VALUE(1), 0 },
    { { SENSORS_FSCHER_FAN1_RIPPLE, "fan1_ripple", SENSORS_FSCHER_FAN1, 
                                    SENSORS_FSCHER_FAN1, RW }, 
                                    FSCHER_SYSCTL_FAN0, VALUE(2), 0 },
    { { SENSORS_FSCHER_FAN2_RIPPLE, "fan2_ripple", SENSORS_FSCHER_FAN2, 
                                    SENSORS_FSCHER_FAN2, RW }, 
                                    FSCHER_SYSCTL_FAN1, VALUE(2), 0 },
    { { SENSORS_FSCHER_FAN3_RIPPLE, "fan3_ripple", SENSORS_FSCHER_FAN3, 
                                    SENSORS_FSCHER_FAN3, RW }, 
                                    FSCHER_SYSCTL_FAN2, VALUE(2), 0 },
    { { SENSORS_FSCHER_WDOG_PRESET, "wdog_preset", NOMAP, NOMAP, RW }, 
                                    FSCHER_SYSCTL_WDOG, VALUE(1), 0 },
    { { SENSORS_FSCHER_WDOG_STATE, "wdog_state", SENSORS_FSCHER_WDOG_PRESET, 
                                   SENSORS_FSCHER_WDOG_PRESET, RW }, 
                                   FSCHER_SYSCTL_WDOG, VALUE(2), 0 },
    { { SENSORS_FSCHER_WDOG_CONTROL, "wdog_control", SENSORS_FSCHER_WDOG_PRESET, 
                                     SENSORS_FSCHER_WDOG_PRESET, RW }, 
                                     FSCHER_SYSCTL_WDOG, VALUE(3), 0 },
    /* below this line are features only exported with the new fschmd drv */
    { { SENSORS_FSCHMD_TEMP_MAX(1), "temp1_max", SENSORS_FSCHER_TEMP1,
                                    SENSORS_FSCHER_TEMP1, RW },
                                    NOSYSCTL, VALUE(1), 0 },
    { { SENSORS_FSCHMD_TEMP_MAX(2), "temp2_max", SENSORS_FSCHER_TEMP2,
                                    SENSORS_FSCHER_TEMP2, RW },
                                    NOSYSCTL, VALUE(1), 0 },
    { { SENSORS_FSCHMD_TEMP_MAX(3), "temp3_max", SENSORS_FSCHER_TEMP3,
                                    SENSORS_FSCHER_TEMP3, RW },
                                    NOSYSCTL, VALUE(1), 0 },
    { { SENSORS_FSCHMD_TEMP_ALARM(1), "temp1_alarm", SENSORS_FSCHER_TEMP1,
                                    NOMAP, R }, NOSYSCTL, VALUE(1), 0 },
    { { SENSORS_FSCHMD_TEMP_ALARM(2), "temp2_alarm", SENSORS_FSCHER_TEMP2,
                                    NOMAP, R }, NOSYSCTL, VALUE(1), 0 },
    { { SENSORS_FSCHMD_TEMP_ALARM(3), "temp3_alarm", SENSORS_FSCHER_TEMP3, 
                                    NOMAP, R }, NOSYSCTL, VALUE(1), 0 },
    { { SENSORS_FSCHMD_TEMP_FAULT(1), "temp1_fault", SENSORS_FSCHER_TEMP1, 
                                    NOMAP, R }, NOSYSCTL, VALUE(1), 0 },
    { { SENSORS_FSCHMD_TEMP_FAULT(2), "temp2_fault", SENSORS_FSCHER_TEMP2, 
                                    NOMAP, R }, NOSYSCTL, VALUE(1), 0 },
    { { SENSORS_FSCHMD_TEMP_FAULT(3), "temp3_fault", SENSORS_FSCHER_TEMP3, 
                                    NOMAP, R }, NOSYSCTL, VALUE(1), 0 },
    { { SENSORS_FSCHMD_FAN_ALARM(1), "fan1_alarm", SENSORS_FSCHER_FAN1, 
                                    NOMAP, R }, NOSYSCTL, VALUE(1), 0 },
    { { SENSORS_FSCHMD_FAN_ALARM(2), "fan2_alarm", SENSORS_FSCHER_FAN2, 
                                    NOMAP, R }, NOSYSCTL, VALUE(1), 0 },
    { { SENSORS_FSCHMD_FAN_ALARM(3), "fan3_alarm", SENSORS_FSCHER_FAN3, 
                                    NOMAP, R }, NOSYSCTL, VALUE(1), 0 },
    { { SENSORS_FSCHMD_FAN_FAULT(1), "fan1_fault", SENSORS_FSCHER_FAN1, 
                                    NOMAP, R }, NOSYSCTL, VALUE(1), 0 },
    { { SENSORS_FSCHMD_FAN_FAULT(2), "fan2_fault", SENSORS_FSCHER_FAN2, 
                                    NOMAP, R }, NOSYSCTL, VALUE(1), 0 },
    { { SENSORS_FSCHMD_FAN_FAULT(3), "fan3_fault", SENSORS_FSCHER_FAN3, 
                                    NOMAP, R }, NOSYSCTL, VALUE(1), 0 },
    { { 0 }, 0 }
  };

  static sensors_chip_feature pcf8591_features[] =
  {
    { { SENSORS_PCF8591_AIN_CONF, "ain_conf", NOMAP, NOMAP, RW }, 
                                  PCF8591_SYSCTL_AIN_CONF, VALUE(1), 0 },
    { { SENSORS_PCF8591_CH0, "ch0", NOMAP, NOMAP, R }, 
                             PCF8591_SYSCTL_CH0, VALUE(1), 2, "in0_input", 3 },
    { { SENSORS_PCF8591_CH1, "ch1", NOMAP, NOMAP, R }, 
                             PCF8591_SYSCTL_CH1, VALUE(1), 2, "in1_input", 3 },
    { { SENSORS_PCF8591_CH2, "ch2", NOMAP, NOMAP, R }, 
                             PCF8591_SYSCTL_CH2, VALUE(1), 2, "in2_input", 3 },
    { { SENSORS_PCF8591_CH3, "ch3", NOMAP, NOMAP, R }, 
                             PCF8591_SYSCTL_CH3, VALUE(1), 2, "in3_input", 3 },
    { { SENSORS_PCF8591_AOUT_ENABLE, "aout_enable", NOMAP, NOMAP, RW }, 
                                     PCF8591_SYSCTL_AOUT_ENABLE, VALUE(1), 0, 
                                     "out0_enable", 0 },
    { { SENSORS_PCF8591_AOUT, "aout", NOMAP, NOMAP, RW }, 
                              PCF8591_SYSCTL_AOUT, VALUE(1), 2, "out0_output", 
                              3 },
    { { 0 }, 0 }
  };


static sensors_chip_feature vt1211_features[] =
  { 
    { { SENSORS_VT1211_IN0, "in0", NOMAP, NOMAP, R }, 
                            VT1211_SYSCTL_IN0, VALUE(3), 2 },
    { { SENSORS_VT1211_IN1, "in1", NOMAP, NOMAP, R }, 
                            VT1211_SYSCTL_IN1, VALUE(3), 2 },
    { { SENSORS_VT1211_IN2, "in2", NOMAP, NOMAP, R }, 
                            VT1211_SYSCTL_IN2, VALUE(3), 2 },
    { { SENSORS_VT1211_IN3, "in3", NOMAP, NOMAP, R }, 
                            VT1211_SYSCTL_IN3, VALUE(3), 2 },
    { { SENSORS_VT1211_IN4, "in4", NOMAP, NOMAP, R }, 
                            VT1211_SYSCTL_IN4, VALUE(3), 2 },
    { { SENSORS_VT1211_IN5, "in5", NOMAP, NOMAP, R }, 
                            VT1211_SYSCTL_IN5, VALUE(3), 2 },
    { { SENSORS_VT1211_IN0_MIN, "in0_min", SENSORS_VT1211_IN0, 
                                SENSORS_VT1211_IN0, RW }, 
                                VT1211_SYSCTL_IN0, VALUE(1), 2 },
    { { SENSORS_VT1211_IN1_MIN, "in1_min", SENSORS_VT1211_IN1, 
                                SENSORS_VT1211_IN1, RW }, 
                                VT1211_SYSCTL_IN1, VALUE(1), 2 },
    { { SENSORS_VT1211_IN2_MIN, "in2_min", SENSORS_VT1211_IN2, 
                                SENSORS_VT1211_IN2, RW }, 
                                VT1211_SYSCTL_IN2, VALUE(1), 2 },
    { { SENSORS_VT1211_IN3_MIN, "in3_min", SENSORS_VT1211_IN3, 
                                SENSORS_VT1211_IN3, RW }, 
                                VT1211_SYSCTL_IN3, VALUE(1), 2 },
    { { SENSORS_VT1211_IN4_MIN, "in4_min", SENSORS_VT1211_IN4, 
                                SENSORS_VT1211_IN4, RW }, 
                                VT1211_SYSCTL_IN4, VALUE(1), 2 },
    { { SENSORS_VT1211_IN5_MIN, "in5_min", SENSORS_VT1211_IN5, 
                                SENSORS_VT1211_IN5, RW }, 
                                VT1211_SYSCTL_IN5, VALUE(1), 2 },
    { { SENSORS_VT1211_IN0_MAX, "in0_max", SENSORS_VT1211_IN0, 
                                SENSORS_VT1211_IN0, RW }, 
                                VT1211_SYSCTL_IN0, VALUE(2), 2 },
    { { SENSORS_VT1211_IN1_MAX, "in1_max", SENSORS_VT1211_IN1, 
                                SENSORS_VT1211_IN1, RW }, 
                                VT1211_SYSCTL_IN1, VALUE(2), 2 },
    { { SENSORS_VT1211_IN2_MAX, "in2_max", SENSORS_VT1211_IN2, 
                                SENSORS_VT1211_IN2, RW }, 
                                VT1211_SYSCTL_IN2, VALUE(2), 2 },
    { { SENSORS_VT1211_IN3_MAX, "in3_max", SENSORS_VT1211_IN3, 
                                SENSORS_VT1211_IN3, RW }, 
                                VT1211_SYSCTL_IN3, VALUE(2), 2 },
    { { SENSORS_VT1211_IN4_MAX, "in4_max", SENSORS_VT1211_IN4, 
                                SENSORS_VT1211_IN4, RW }, 
                                VT1211_SYSCTL_IN4, VALUE(2), 2 },
    { { SENSORS_VT1211_IN5_MAX, "in5_max", SENSORS_VT1211_IN5, 
                                SENSORS_VT1211_IN5, RW }, 
                                VT1211_SYSCTL_IN5, VALUE(2), 2 },
    { { SENSORS_VT1211_FAN1, "fan1", NOMAP, NOMAP, R }, 
                             VT1211_SYSCTL_FAN1, VALUE(2), 0 },
    { { SENSORS_VT1211_FAN2, "fan2", NOMAP, NOMAP, R }, 
                             VT1211_SYSCTL_FAN2, VALUE(2), 0 },
    { { SENSORS_VT1211_FAN1_MIN, "fan1_min", SENSORS_VT1211_FAN1, 
                                 SENSORS_VT1211_FAN1, RW }, 
                                 VT1211_SYSCTL_FAN1, VALUE(1), 0 },
    { { SENSORS_VT1211_FAN2_MIN, "fan2_min", SENSORS_VT1211_FAN2, 
                                 SENSORS_VT1211_FAN2, RW }, 
                                 VT1211_SYSCTL_FAN2, VALUE(1), 0 },
    { { SENSORS_VT1211_TEMP1, "temp1", NOMAP, NOMAP, R }, 
                              VT1211_SYSCTL_TEMP1, VALUE(3), 1 },
    { { SENSORS_VT1211_TEMP1_HYST, "temp1_hyst", SENSORS_VT1211_TEMP1, 
                                   SENSORS_VT1211_TEMP1, RW }, 
                                   VT1211_SYSCTL_TEMP1, VALUE(2), 1 },
    { { SENSORS_VT1211_TEMP1_OVER, "temp1_over", SENSORS_VT1211_TEMP1, 
                                   SENSORS_VT1211_TEMP1, RW }, 
                                   VT1211_SYSCTL_TEMP1, VALUE(1), 1 },
    { { SENSORS_VT1211_TEMP2, "temp2", NOMAP, NOMAP, R }, 
                              VT1211_SYSCTL_TEMP2, VALUE(3), 1 },
    { { SENSORS_VT1211_TEMP2_HYST, "temp2_hyst", SENSORS_VT1211_TEMP2, 
                                   SENSORS_VT1211_TEMP2, RW }, 
                                   VT1211_SYSCTL_TEMP2, VALUE(2), 1 },
    { { SENSORS_VT1211_TEMP2_OVER, "temp2_over", SENSORS_VT1211_TEMP2, 
                                   SENSORS_VT1211_TEMP2, RW }, 
                                   VT1211_SYSCTL_TEMP2, VALUE(1), 1 },
    { { SENSORS_VT1211_TEMP3, "temp3", NOMAP, NOMAP, R }, 
                              VT1211_SYSCTL_TEMP3, VALUE(3), 3 },
    { { SENSORS_VT1211_TEMP3_HYST, "temp3_hyst", SENSORS_VT1211_TEMP3, 
                                   SENSORS_VT1211_TEMP3, RW }, 
                                   VT1211_SYSCTL_TEMP3, VALUE(2), 3 },
    { { SENSORS_VT1211_TEMP3_OVER, "temp3_over", SENSORS_VT1211_TEMP3, 
                                   SENSORS_VT1211_TEMP3, RW }, 
                                   VT1211_SYSCTL_TEMP3, VALUE(1), 3 },
    { { SENSORS_VT1211_TEMP4, "temp4", NOMAP, NOMAP, R }, 
                              VT1211_SYSCTL_TEMP4, VALUE(3), 3 },
    { { SENSORS_VT1211_TEMP4_HYST, "temp4_hyst", SENSORS_VT1211_TEMP4, 
                                   SENSORS_VT1211_TEMP4, RW }, 
                                   VT1211_SYSCTL_TEMP4, VALUE(2), 3 },
    { { SENSORS_VT1211_TEMP4_OVER, "temp4_over", SENSORS_VT1211_TEMP4, 
                                   SENSORS_VT1211_TEMP4, RW }, 
                                   VT1211_SYSCTL_TEMP4, VALUE(1), 3 },
    { { SENSORS_VT1211_TEMP5, "temp5", NOMAP, NOMAP, R }, 
                              VT1211_SYSCTL_TEMP5, VALUE(3), 3 },
    { { SENSORS_VT1211_TEMP5_HYST, "temp5_hyst", SENSORS_VT1211_TEMP5, 
                                   SENSORS_VT1211_TEMP5, RW }, 
                                   VT1211_SYSCTL_TEMP5, VALUE(2), 3 },
    { { SENSORS_VT1211_TEMP5_OVER, "temp5_over", SENSORS_VT1211_TEMP5, 
                                   SENSORS_VT1211_TEMP5, RW }, 
                                   VT1211_SYSCTL_TEMP5, VALUE(1), 3 },
    { { SENSORS_VT1211_TEMP6, "temp6", NOMAP, NOMAP, R }, 
                              VT1211_SYSCTL_TEMP6, VALUE(3), 3 },
    { { SENSORS_VT1211_TEMP6_HYST, "temp6_hyst", SENSORS_VT1211_TEMP6, 
                                   SENSORS_VT1211_TEMP6, RW }, 
                                   VT1211_SYSCTL_TEMP6, VALUE(2), 3 },
    { { SENSORS_VT1211_TEMP6_OVER, "temp6_over", SENSORS_VT1211_TEMP6, 
                                   SENSORS_VT1211_TEMP6, RW }, 
                                   VT1211_SYSCTL_TEMP6, VALUE(1), 3 },
    { { SENSORS_VT1211_TEMP7, "temp7", NOMAP, NOMAP, R }, 
                              VT1211_SYSCTL_TEMP7, VALUE(3), 3 },
    { { SENSORS_VT1211_TEMP7_HYST, "temp7_hyst", SENSORS_VT1211_TEMP7, 
                                   SENSORS_VT1211_TEMP7, RW }, 
                                   VT1211_SYSCTL_TEMP7, VALUE(2), 3 },
    { { SENSORS_VT1211_TEMP7_OVER, "temp7_over", SENSORS_VT1211_TEMP7, 
                                   SENSORS_VT1211_TEMP7, RW }, 
                                   VT1211_SYSCTL_TEMP7, VALUE(1), 3 },
    { { SENSORS_VT1211_FAN1_DIV, "fan1_div", SENSORS_VT1211_FAN1, NOMAP, RW }, 
                                 VT1211_SYSCTL_FAN_DIV, VALUE(1), 0 },
    { { SENSORS_VT1211_FAN2_DIV, "fan2_div", SENSORS_VT1211_FAN2, NOMAP, RW }, 
                                 VT1211_SYSCTL_FAN_DIV, VALUE(2), 0 },
    { { SENSORS_VT1211_ALARMS, "alarms", NOMAP, NOMAP, R }, 
                               VT1211_SYSCTL_ALARMS, VALUE(1), 0 },
    { { SENSORS_VT1211_VID, "vid", NOMAP, NOMAP, R }, 
                            VT1211_SYSCTL_VID, VALUE(1), 3 },
    { { SENSORS_VT1211_VRM, "vrm", NOMAP, NOMAP, RW }, 
                            VT1211_SYSCTL_VRM, VALUE(1), 1 },
    { { SENSORS_VT1211_UCH, "config", NOMAP, NOMAP, RW }, 
                            VT1211_SYSCTL_UCH, VALUE(1), 0, "uch_config" },
    { { 0 }, 0 }
  };

static sensors_chip_feature smsc47m1_features[] =
  { 
    { { SENSORS_SMSC47M1_FAN1, "fan1", NOMAP, NOMAP, R }, 
                               SMSC47M1_SYSCTL_FAN1, VALUE(2), 0 },
    { { SENSORS_SMSC47M1_FAN2, "fan2", NOMAP, NOMAP, R }, 
                               SMSC47M1_SYSCTL_FAN2, VALUE(2), 0 },
    { { SENSORS_SMSC47M1_FAN1_MIN, "fan1_min", SENSORS_SMSC47M1_FAN1, 
                                   SENSORS_SMSC47M1_FAN1, RW }, 
                                   SMSC47M1_SYSCTL_FAN1, VALUE(1), 0 },
    { { SENSORS_SMSC47M1_FAN2_MIN, "fan2_min", SENSORS_SMSC47M1_FAN2, 
                                   SENSORS_SMSC47M1_FAN2, RW }, 
                                   SMSC47M1_SYSCTL_FAN2, VALUE(1), 0 },
    { { SENSORS_SMSC47M1_FAN1_DIV, "fan1_div", SENSORS_SMSC47M1_FAN1, NOMAP, RW }, 
                                   SMSC47M1_SYSCTL_FAN_DIV, VALUE(1), 0 },
    { { SENSORS_SMSC47M1_FAN2_DIV, "fan2_div", SENSORS_SMSC47M1_FAN2, NOMAP, RW }, 
                                   SMSC47M1_SYSCTL_FAN_DIV, VALUE(2), 0 },
    { { SENSORS_SMSC47M1_ALARMS, "alarms", NOMAP, NOMAP, R }, 
                                 SMSC47M1_SYSCTL_ALARMS, VALUE(1), 0 },
    { { 0 }, 0 }
  };

static sensors_chip_feature smsc47m2_features[] =
  { 
    { { SENSORS_SMSC47M1_FAN1, "fan1", NOMAP, NOMAP, R },
                               SMSC47M1_SYSCTL_FAN1, VALUE(2), 0 },
    { { SENSORS_SMSC47M1_FAN2, "fan2", NOMAP, NOMAP, R },
                               SMSC47M1_SYSCTL_FAN2, VALUE(2), 0 },
    { { SENSORS_SMSC47M1_FAN3, "fan3", NOMAP, NOMAP, R },
                               SMSC47M1_SYSCTL_FAN3, VALUE(2), 0 },
    { { SENSORS_SMSC47M1_FAN1_MIN, "fan1_min", SENSORS_SMSC47M1_FAN1,
                                   SENSORS_SMSC47M1_FAN1, RW },
                                   SMSC47M1_SYSCTL_FAN1, VALUE(1), 0 },
    { { SENSORS_SMSC47M1_FAN2_MIN, "fan2_min", SENSORS_SMSC47M1_FAN2,
                                   SENSORS_SMSC47M1_FAN2, RW },
                                   SMSC47M1_SYSCTL_FAN2, VALUE(1), 0 },
    { { SENSORS_SMSC47M1_FAN3_MIN, "fan3_min", SENSORS_SMSC47M1_FAN3,
                                   SENSORS_SMSC47M1_FAN3, RW },
                                   SMSC47M1_SYSCTL_FAN3, VALUE(1), 0 },
    { { SENSORS_SMSC47M1_FAN1_DIV, "fan1_div", SENSORS_SMSC47M1_FAN1, NOMAP, RW },
                                   SMSC47M1_SYSCTL_FAN_DIV, VALUE(1), 0 },
    { { SENSORS_SMSC47M1_FAN2_DIV, "fan2_div", SENSORS_SMSC47M1_FAN2, NOMAP, RW },
                                   SMSC47M1_SYSCTL_FAN_DIV, VALUE(2), 0 },
    { { SENSORS_SMSC47M1_FAN3_DIV, "fan3_div", SENSORS_SMSC47M1_FAN3, NOMAP, RW },
                                   SMSC47M1_SYSCTL_FAN_DIV, VALUE(3), 0 },
    { { SENSORS_SMSC47M1_ALARMS, "alarms", NOMAP, NOMAP, R },
                                 SMSC47M1_SYSCTL_ALARMS, VALUE(1), 0 },
    { { 0 }, 0 }
  };

static sensors_chip_feature smsc47m192_features[] =
  { 
    { { SENSORS_SMSC47M192_IN(0), "in0", NOMAP, NOMAP, R }, 
                                  NOSYSCTL, VALUE(3), 3 },
    { { SENSORS_SMSC47M192_IN(1), "in1", NOMAP, NOMAP, R }, 
                                  NOSYSCTL, VALUE(3), 3 },
    { { SENSORS_SMSC47M192_IN(2), "in2", NOMAP, NOMAP, R }, 
                                  NOSYSCTL, VALUE(3), 3 },
    { { SENSORS_SMSC47M192_IN(3), "in3", NOMAP, NOMAP, R }, 
                                  NOSYSCTL, VALUE(3), 3 },
    { { SENSORS_SMSC47M192_IN(4), "in4", NOMAP, NOMAP, R }, 
                                  NOSYSCTL, VALUE(3), 3 },
    { { SENSORS_SMSC47M192_IN(5), "in5", NOMAP, NOMAP, R }, 
                                  NOSYSCTL, VALUE(3), 3 },
    { { SENSORS_SMSC47M192_IN(6), "in6", NOMAP, NOMAP, R }, 
                                  NOSYSCTL, VALUE(3), 3 },
    { { SENSORS_SMSC47M192_IN(7), "in7", NOMAP, NOMAP, R }, 
                                  NOSYSCTL, VALUE(3), 3 },
    { { SENSORS_SMSC47M192_IN_MIN(0), "in0_min", SENSORS_SMSC47M192_IN(0), 
                                      SENSORS_SMSC47M192_IN(0), RW }, 
                                      NOSYSCTL, VALUE(1), 3 },
    { { SENSORS_SMSC47M192_IN_MIN(1), "in1_min", SENSORS_SMSC47M192_IN(1), 
                                      SENSORS_SMSC47M192_IN(1), RW }, 
                                      NOSYSCTL, VALUE(1), 3 },
    { { SENSORS_SMSC47M192_IN_MIN(2), "in2_min", SENSORS_SMSC47M192_IN(2), 
                                      SENSORS_SMSC47M192_IN(2), RW }, 
                                      NOSYSCTL, VALUE(1), 3 },
    { { SENSORS_SMSC47M192_IN_MIN(3), "in3_min", SENSORS_SMSC47M192_IN(3), 
                                      SENSORS_SMSC47M192_IN(3), RW }, 
                                      NOSYSCTL, VALUE(1), 3 },
    { { SENSORS_SMSC47M192_IN_MIN(4), "in4_min", SENSORS_SMSC47M192_IN(4), 
                                      SENSORS_SMSC47M192_IN(4), RW }, 
                                      NOSYSCTL, VALUE(1), 3 },
    { { SENSORS_SMSC47M192_IN_MIN(5), "in5_min", SENSORS_SMSC47M192_IN(5), 
                                      SENSORS_SMSC47M192_IN(5), RW }, 
                                      NOSYSCTL, VALUE(1), 3 },
    { { SENSORS_SMSC47M192_IN_MIN(6), "in6_min", SENSORS_SMSC47M192_IN(6), 
                                      SENSORS_SMSC47M192_IN(6), RW }, 
                                      NOSYSCTL, VALUE(1), 3 },
    { { SENSORS_SMSC47M192_IN_MIN(7), "in7_min", SENSORS_SMSC47M192_IN(7), 
                                      SENSORS_SMSC47M192_IN(7), RW }, 
                                      NOSYSCTL, VALUE(1), 3 },
    { { SENSORS_SMSC47M192_IN_MAX(0), "in0_max", SENSORS_SMSC47M192_IN(0), 
                                      SENSORS_SMSC47M192_IN(0), RW }, 
                                      NOSYSCTL, VALUE(2), 3 },
    { { SENSORS_SMSC47M192_IN_MAX(1), "in1_max", SENSORS_SMSC47M192_IN(1), 
                                      SENSORS_SMSC47M192_IN(1), RW }, 
                                      NOSYSCTL, VALUE(2), 3 },
    { { SENSORS_SMSC47M192_IN_MAX(2), "in2_max", SENSORS_SMSC47M192_IN(2), 
                                      SENSORS_SMSC47M192_IN(2), RW }, 
                                      NOSYSCTL, VALUE(2), 3 },
    { { SENSORS_SMSC47M192_IN_MAX(3), "in3_max", SENSORS_SMSC47M192_IN(3), 
                                      SENSORS_SMSC47M192_IN(3), RW }, 
                                      NOSYSCTL, VALUE(2), 3 },
    { { SENSORS_SMSC47M192_IN_MAX(4), "in4_max", SENSORS_SMSC47M192_IN(4), 
                                      SENSORS_SMSC47M192_IN(4), RW }, 
                                      NOSYSCTL, VALUE(2), 3 },
    { { SENSORS_SMSC47M192_IN_MAX(5), "in5_max", SENSORS_SMSC47M192_IN(5), 
                                      SENSORS_SMSC47M192_IN(5), RW }, 
                                      NOSYSCTL, VALUE(2), 3 },
    { { SENSORS_SMSC47M192_IN_MAX(6), "in6_max", SENSORS_SMSC47M192_IN(6), 
                                      SENSORS_SMSC47M192_IN(6), RW }, 
                                      NOSYSCTL, VALUE(2), 3 },
    { { SENSORS_SMSC47M192_IN_MAX(7), "in7_max", SENSORS_SMSC47M192_IN(7), 
                                      SENSORS_SMSC47M192_IN(7), RW }, 
                                      NOSYSCTL, VALUE(2), 3 },
    { { SENSORS_SMSC47M192_IN_ALARM(0), "in0_alarm", SENSORS_SMSC47M192_IN(0), 
                                        NOMAP, R }, 
                                        NOSYSCTL, VALUE(1), 0 },
    { { SENSORS_SMSC47M192_IN_ALARM(1), "in1_alarm", SENSORS_SMSC47M192_IN(1), 
                                        NOMAP, R }, 
                                        NOSYSCTL, VALUE(1), 0 },
    { { SENSORS_SMSC47M192_IN_ALARM(2), "in2_alarm", SENSORS_SMSC47M192_IN(2), 
                                        NOMAP, R }, 
                                        NOSYSCTL, VALUE(1), 0 },
    { { SENSORS_SMSC47M192_IN_ALARM(3), "in3_alarm", SENSORS_SMSC47M192_IN(3), 
                                        NOMAP, R }, 
                                        NOSYSCTL, VALUE(1), 0 },
    { { SENSORS_SMSC47M192_IN_ALARM(4), "in4_alarm", SENSORS_SMSC47M192_IN(4), 
                                        NOMAP, R }, 
                                        NOSYSCTL, VALUE(1), 0 },
    { { SENSORS_SMSC47M192_IN_ALARM(5), "in5_alarm", SENSORS_SMSC47M192_IN(5), 
                                        NOMAP, R }, 
                                        NOSYSCTL, VALUE(1), 0 },
    { { SENSORS_SMSC47M192_IN_ALARM(6), "in6_alarm", SENSORS_SMSC47M192_IN(6), 
                                        NOMAP, R }, 
                                        NOSYSCTL, VALUE(1), 0 },
    { { SENSORS_SMSC47M192_IN_ALARM(7), "in7_alarm", SENSORS_SMSC47M192_IN(7), 
                                        NOMAP, R }, 
                                        NOSYSCTL, VALUE(1), 0 },
    { { SENSORS_SMSC47M192_TEMP(1), "temp1", NOMAP, NOMAP, R }, 
                                    NOSYSCTL, VALUE(3), 0 },
    { { SENSORS_SMSC47M192_TEMP(2), "temp2", NOMAP, NOMAP, R }, 
                                    NOSYSCTL, VALUE(3), 0 },
    { { SENSORS_SMSC47M192_TEMP(3), "temp3", NOMAP, NOMAP, R }, 
                                    NOSYSCTL, VALUE(3), 0 },
    { { SENSORS_SMSC47M192_TEMP_MIN(1), "temp1_min", SENSORS_SMSC47M192_TEMP(1), 
                                        SENSORS_SMSC47M192_TEMP(1), RW }, 
                                        NOSYSCTL, VALUE(2), 0 },
    { { SENSORS_SMSC47M192_TEMP_MIN(2), "temp2_min", SENSORS_SMSC47M192_TEMP(2), 
                                        SENSORS_SMSC47M192_TEMP(2), RW }, 
                                        NOSYSCTL, VALUE(2), 0 },
    { { SENSORS_SMSC47M192_TEMP_MIN(3), "temp3_min", SENSORS_SMSC47M192_TEMP(3), 
                                        SENSORS_SMSC47M192_TEMP(3), RW }, 
                                        NOSYSCTL, VALUE(2), 0 },
    { { SENSORS_SMSC47M192_TEMP_MAX(1), "temp1_max", SENSORS_SMSC47M192_TEMP(1), 
                                        SENSORS_SMSC47M192_TEMP(1), RW }, 
                                        NOSYSCTL, VALUE(1), 0 },
    { { SENSORS_SMSC47M192_TEMP_MAX(2), "temp2_max", SENSORS_SMSC47M192_TEMP(2), 
                                        SENSORS_SMSC47M192_TEMP(2), RW }, 
                                        NOSYSCTL, VALUE(1), 0 },
    { { SENSORS_SMSC47M192_TEMP_MAX(3), "temp3_max", SENSORS_SMSC47M192_TEMP(3), 
                                        SENSORS_SMSC47M192_TEMP(3), RW }, 
                                        NOSYSCTL, VALUE(1), 0 },
    { { SENSORS_SMSC47M192_TEMP_ALARM(1), "temp1_alarm", 
                                          SENSORS_SMSC47M192_TEMP(1), NOMAP, R }, 
                                          NOSYSCTL, VALUE(1), 0 },
    { { SENSORS_SMSC47M192_TEMP_ALARM(2), "temp2_alarm", 
                                          SENSORS_SMSC47M192_TEMP(2), NOMAP, R }, 
                                          NOSYSCTL, VALUE(1), 0 },
    { { SENSORS_SMSC47M192_TEMP_ALARM(3), "temp3_alarm", 
                                          SENSORS_SMSC47M192_TEMP(3), NOMAP, R }, 
                                          NOSYSCTL, VALUE(1), 0 },
    { { SENSORS_SMSC47M192_TEMP_FAULT(2), "temp2_fault", 
                                          SENSORS_SMSC47M192_TEMP(2), NOMAP, R }, 
                                          NOSYSCTL, VALUE(1), 0, "temp2_fault", 0, "temp2_input_fault" },
    { { SENSORS_SMSC47M192_TEMP_FAULT(3), "temp3_fault", 
                                          SENSORS_SMSC47M192_TEMP(3), NOMAP, R }, 
                                          NOSYSCTL, VALUE(1), 0, "temp3_fault", 0, "temp3_input_fault" },
    { { SENSORS_SMSC47M192_VID, "vid", NOMAP, NOMAP, R }, 
                                NOSYSCTL, VALUE(1), 3 },
    { { SENSORS_SMSC47M192_VRM, "vrm", NOMAP, NOMAP, RW }, 
                                NOSYSCTL, VALUE(1), 1 },
    { { 0 }, 0 }
  };


static sensors_chip_feature pc87360_features[] =
  { 
    { { SENSORS_PC87360_FAN1, "fan1", NOMAP, NOMAP, R }, 
                              PC87360_SYSCTL_FAN1, VALUE(2), 0 },
    { { SENSORS_PC87360_FAN2, "fan2", NOMAP, NOMAP, R }, 
                              PC87360_SYSCTL_FAN2, VALUE(2), 0 },
    { { SENSORS_PC87360_FAN3, "fan3", NOMAP, NOMAP, R }, 
                              PC87360_SYSCTL_FAN3, VALUE(2), 0 },
    { { SENSORS_PC87360_FAN1_MIN, "fan1_min", SENSORS_PC87360_FAN1, 
                                  SENSORS_PC87360_FAN1, RW }, 
                                  PC87360_SYSCTL_FAN1, VALUE(1), 0 },
    { { SENSORS_PC87360_FAN2_MIN, "fan2_min", SENSORS_PC87360_FAN2, 
                                  SENSORS_PC87360_FAN2, RW }, 
                                  PC87360_SYSCTL_FAN2, VALUE(1), 0 },
    { { SENSORS_PC87360_FAN3_MIN, "fan3_min", SENSORS_PC87360_FAN3, 
                                  SENSORS_PC87360_FAN3, RW }, 
                                  PC87360_SYSCTL_FAN3, VALUE(1), 0 },
    { { SENSORS_PC87360_FAN1_DIV, "fan1_div", SENSORS_PC87360_FAN1, NOMAP, R }, 
                                  PC87360_SYSCTL_FAN_DIV, VALUE(1), 0 },
    { { SENSORS_PC87360_FAN2_DIV, "fan2_div", SENSORS_PC87360_FAN2, NOMAP, R }, 
                                  PC87360_SYSCTL_FAN_DIV, VALUE(2), 0 },
    { { SENSORS_PC87360_FAN3_DIV, "fan3_div", SENSORS_PC87360_FAN3, NOMAP, R }, 
                                  PC87360_SYSCTL_FAN_DIV, VALUE(3), 0 },
    { { SENSORS_PC87360_FAN1_STATUS, "fan1_status", SENSORS_PC87360_FAN1, NOMAP, 
                                     R }, 
                                     PC87360_SYSCTL_FAN1_STATUS, VALUE(1), 0 },
    { { SENSORS_PC87360_FAN2_STATUS, "fan2_status", SENSORS_PC87360_FAN2, NOMAP, 
                                     R }, 
                                     PC87360_SYSCTL_FAN2_STATUS, VALUE(1), 0 },
    { { SENSORS_PC87360_FAN3_STATUS, "fan3_status", SENSORS_PC87360_FAN3, NOMAP, 
                                     R }, 
                                     PC87360_SYSCTL_FAN3_STATUS, VALUE(1), 0 },
    { { 0 }, 0 }
  };

static sensors_chip_feature pc87365_features[] =
  { 
    { { SENSORS_PC87360_ALARMS_IN, "alarms_in", NOMAP, NOMAP, R }, 
                                   PC87365_SYSCTL_ALARMS, VALUE(1), 0 },
    { { SENSORS_PC87360_ALARMS_TEMP, "alarms_temp", NOMAP, NOMAP, R }, 
                                     PC87365_SYSCTL_ALARMS, VALUE(2), 0 },
    { { SENSORS_PC87360_FAN1, "fan1", NOMAP, NOMAP, R }, 
                              PC87360_SYSCTL_FAN1, VALUE(2), 0 },
    { { SENSORS_PC87360_FAN2, "fan2", NOMAP, NOMAP, R }, 
                              PC87360_SYSCTL_FAN2, VALUE(2), 0 },
    { { SENSORS_PC87360_FAN3, "fan3", NOMAP, NOMAP, R }, 
                              PC87360_SYSCTL_FAN3, VALUE(2), 0 },
    { { SENSORS_PC87360_FAN1_MIN, "fan1_min", SENSORS_PC87360_FAN1, 
                                  SENSORS_PC87360_FAN1, RW }, 
                                  PC87360_SYSCTL_FAN1, VALUE(1), 0 },
    { { SENSORS_PC87360_FAN2_MIN, "fan2_min", SENSORS_PC87360_FAN2, 
                                  SENSORS_PC87360_FAN2, RW }, 
                                  PC87360_SYSCTL_FAN2, VALUE(1), 0 },
    { { SENSORS_PC87360_FAN3_MIN, "fan3_min", SENSORS_PC87360_FAN3, 
                                  SENSORS_PC87360_FAN3, RW }, 
                                  PC87360_SYSCTL_FAN3, VALUE(1), 0 },
    { { SENSORS_PC87360_FAN1_DIV, "fan1_div", SENSORS_PC87360_FAN1, NOMAP, R }, 
                                  PC87360_SYSCTL_FAN_DIV, VALUE(1), 0 },
    { { SENSORS_PC87360_FAN2_DIV, "fan2_div", SENSORS_PC87360_FAN2, NOMAP, R }, 
                                  PC87360_SYSCTL_FAN_DIV, VALUE(2), 0 },
    { { SENSORS_PC87360_FAN3_DIV, "fan3_div", SENSORS_PC87360_FAN3, NOMAP, R }, 
                                  PC87360_SYSCTL_FAN_DIV, VALUE(3), 0 },
    { { SENSORS_PC87360_FAN1_STATUS, "fan1_status", SENSORS_PC87360_FAN1, NOMAP, 
                                     R }, 
                                     PC87360_SYSCTL_FAN1_STATUS, VALUE(1), 0 },
    { { SENSORS_PC87360_FAN2_STATUS, "fan2_status", SENSORS_PC87360_FAN2, NOMAP, 
                                     R }, 
                                     PC87360_SYSCTL_FAN2_STATUS, VALUE(1), 0 },
    { { SENSORS_PC87360_FAN3_STATUS, "fan3_status", SENSORS_PC87360_FAN3, NOMAP, 
                                     R }, 
                                     PC87360_SYSCTL_FAN3_STATUS, VALUE(1), 0 },
    { { SENSORS_PC87360_IN0, "in0", NOMAP, NOMAP, R }, 
                             PC87365_SYSCTL_IN0, VALUE(3), 3 },
    { { SENSORS_PC87360_IN1, "in1", NOMAP, NOMAP, R }, 
                             PC87365_SYSCTL_IN1, VALUE(3), 3 },
    { { SENSORS_PC87360_IN2, "in2", NOMAP, NOMAP, R }, 
                             PC87365_SYSCTL_IN2, VALUE(3), 3 },
    { { SENSORS_PC87360_IN3, "in3", NOMAP, NOMAP, R }, 
                             PC87365_SYSCTL_IN3, VALUE(3), 3 },
    { { SENSORS_PC87360_IN4, "in4", NOMAP, NOMAP, R }, 
                             PC87365_SYSCTL_IN4, VALUE(3), 3 },
    { { SENSORS_PC87360_IN5, "in5", NOMAP, NOMAP, R }, 
                             PC87365_SYSCTL_IN5, VALUE(3), 3 },
    { { SENSORS_PC87360_IN6, "in6", NOMAP, NOMAP, R }, 
                             PC87365_SYSCTL_IN6, VALUE(3), 3 },
    { { SENSORS_PC87360_IN7, "in7", NOMAP, NOMAP, R }, 
                             PC87365_SYSCTL_IN7, VALUE(3), 3 },
    { { SENSORS_PC87360_IN8, "in8", NOMAP, NOMAP, R }, 
                             PC87365_SYSCTL_IN8, VALUE(3), 3 },
    { { SENSORS_PC87360_IN9, "in9", NOMAP, NOMAP, R }, 
                             PC87365_SYSCTL_IN9, VALUE(3), 3 },
    { { SENSORS_PC87360_IN10, "in10", NOMAP, NOMAP, R }, 
                              PC87365_SYSCTL_IN10, VALUE(3), 3 },
    { { SENSORS_PC87360_IN0_MIN, "in0_min", SENSORS_PC87360_IN0, 
                                 SENSORS_PC87360_IN0, RW }, 
                                 PC87365_SYSCTL_IN0, VALUE(1), 3 },
    { { SENSORS_PC87360_IN1_MIN, "in1_min", SENSORS_PC87360_IN1, 
                                 SENSORS_PC87360_IN1, RW }, 
                                 PC87365_SYSCTL_IN1, VALUE(1), 3 },
    { { SENSORS_PC87360_IN2_MIN, "in2_min", SENSORS_PC87360_IN2, 
                                 SENSORS_PC87360_IN2, RW }, 
                                 PC87365_SYSCTL_IN2, VALUE(1), 3 },
    { { SENSORS_PC87360_IN3_MIN, "in3_min", SENSORS_PC87360_IN3, 
                                 SENSORS_PC87360_IN3, RW }, 
                                 PC87365_SYSCTL_IN3, VALUE(1), 3 },
    { { SENSORS_PC87360_IN4_MIN, "in4_min", SENSORS_PC87360_IN4, 
                                 SENSORS_PC87360_IN4, RW }, 
                                 PC87365_SYSCTL_IN4, VALUE(1), 3 },
    { { SENSORS_PC87360_IN5_MIN, "in5_min", SENSORS_PC87360_IN5, 
                                 SENSORS_PC87360_IN5, RW }, 
                                 PC87365_SYSCTL_IN5, VALUE(1), 3 },
    { { SENSORS_PC87360_IN6_MIN, "in6_min", SENSORS_PC87360_IN6, 
                                 SENSORS_PC87360_IN6, RW }, 
                                 PC87365_SYSCTL_IN6, VALUE(1), 3 },
    { { SENSORS_PC87360_IN7_MIN, "in7_min", SENSORS_PC87360_IN7, 
                                 SENSORS_PC87360_IN7, RW }, 
                                 PC87365_SYSCTL_IN7, VALUE(1), 3 },
    { { SENSORS_PC87360_IN8_MIN, "in8_min", SENSORS_PC87360_IN8, 
                                 SENSORS_PC87360_IN8, RW }, 
                                 PC87365_SYSCTL_IN8, VALUE(1), 3 },
    { { SENSORS_PC87360_IN9_MIN, "in9_min", SENSORS_PC87360_IN9, 
                                 SENSORS_PC87360_IN9, RW }, 
                                 PC87365_SYSCTL_IN9, VALUE(1), 3 },
    { { SENSORS_PC87360_IN10_MIN, "in10_min", SENSORS_PC87360_IN10, 
                                  SENSORS_PC87360_IN10, RW }, 
                                  PC87365_SYSCTL_IN10, VALUE(1), 3 },
    { { SENSORS_PC87360_IN0_MAX, "in0_max", SENSORS_PC87360_IN0, 
                                 SENSORS_PC87360_IN0, RW }, 
                                 PC87365_SYSCTL_IN0, VALUE(2), 3 },
    { { SENSORS_PC87360_IN1_MAX, "in1_max", SENSORS_PC87360_IN1, 
                                 SENSORS_PC87360_IN1, RW }, 
                                 PC87365_SYSCTL_IN1, VALUE(2), 3 },
    { { SENSORS_PC87360_IN2_MAX, "in2_max", SENSORS_PC87360_IN2, 
                                 SENSORS_PC87360_IN2, RW }, 
                                 PC87365_SYSCTL_IN2, VALUE(2), 3 },
    { { SENSORS_PC87360_IN3_MAX, "in3_max", SENSORS_PC87360_IN3, 
                                 SENSORS_PC87360_IN3, RW }, 
                                 PC87365_SYSCTL_IN3, VALUE(2), 3 },
    { { SENSORS_PC87360_IN4_MAX, "in4_max", SENSORS_PC87360_IN4, 
                                 SENSORS_PC87360_IN4, RW }, 
                                 PC87365_SYSCTL_IN4, VALUE(2), 3 },
    { { SENSORS_PC87360_IN5_MAX, "in5_max", SENSORS_PC87360_IN5, 
                                 SENSORS_PC87360_IN5, RW }, 
                                 PC87365_SYSCTL_IN5, VALUE(2), 3 },
    { { SENSORS_PC87360_IN6_MAX, "in6_max", SENSORS_PC87360_IN6, 
                                 SENSORS_PC87360_IN6, RW }, 
                                 PC87365_SYSCTL_IN6, VALUE(2), 3 },
    { { SENSORS_PC87360_IN7_MAX, "in7_max", SENSORS_PC87360_IN7, 
                                 SENSORS_PC87360_IN7, RW }, 
                                 PC87365_SYSCTL_IN7, VALUE(2), 3 },
    { { SENSORS_PC87360_IN8_MAX, "in8_max", SENSORS_PC87360_IN8, 
                                 SENSORS_PC87360_IN8, RW }, 
                                 PC87365_SYSCTL_IN8, VALUE(2), 3 },
    { { SENSORS_PC87360_IN9_MAX, "in9_max", SENSORS_PC87360_IN9, 
                                 SENSORS_PC87360_IN9, RW }, 
                                 PC87365_SYSCTL_IN9, VALUE(2), 3 },
    { { SENSORS_PC87360_IN10_MAX, "in10_max", SENSORS_PC87360_IN10, 
                                  SENSORS_PC87360_IN10, RW }, 
                                  PC87365_SYSCTL_IN10, VALUE(2), 3 },
    { { SENSORS_PC87360_IN0_STATUS, "in0_status", SENSORS_PC87360_IN0, NOMAP, R }, 
                                    PC87365_SYSCTL_IN0_STATUS, VALUE(1), 0 },
    { { SENSORS_PC87360_IN1_STATUS, "in1_status", SENSORS_PC87360_IN1, NOMAP, R }, 
                                    PC87365_SYSCTL_IN1_STATUS, VALUE(1), 0 },
    { { SENSORS_PC87360_IN2_STATUS, "in2_status", SENSORS_PC87360_IN2, NOMAP, R }, 
                                    PC87365_SYSCTL_IN2_STATUS, VALUE(1), 0 },
    { { SENSORS_PC87360_IN3_STATUS, "in3_status", SENSORS_PC87360_IN3, NOMAP, R }, 
                                    PC87365_SYSCTL_IN3_STATUS, VALUE(1), 0 },
    { { SENSORS_PC87360_IN4_STATUS, "in4_status", SENSORS_PC87360_IN4, NOMAP, R }, 
                                    PC87365_SYSCTL_IN4_STATUS, VALUE(1), 0 },
    { { SENSORS_PC87360_IN5_STATUS, "in5_status", SENSORS_PC87360_IN5, NOMAP, R }, 
                                    PC87365_SYSCTL_IN5_STATUS, VALUE(1), 0 },
    { { SENSORS_PC87360_IN6_STATUS, "in6_status", SENSORS_PC87360_IN6, NOMAP, R }, 
                                    PC87365_SYSCTL_IN6_STATUS, VALUE(1), 0 },
    { { SENSORS_PC87360_IN7_STATUS, "in7_status", SENSORS_PC87360_IN7, NOMAP, R }, 
                                    PC87365_SYSCTL_IN7_STATUS, VALUE(1), 0 },
    { { SENSORS_PC87360_IN8_STATUS, "in8_status", SENSORS_PC87360_IN8, NOMAP, R }, 
                                    PC87365_SYSCTL_IN8_STATUS, VALUE(1), 0 },
    { { SENSORS_PC87360_IN9_STATUS, "in9_status", SENSORS_PC87360_IN9, NOMAP, R }, 
                                    PC87365_SYSCTL_IN0_STATUS, VALUE(1), 0 },
    { { SENSORS_PC87360_IN10_STATUS, "in10_status", SENSORS_PC87360_IN10, NOMAP, 
                                     R }, 
                                     PC87365_SYSCTL_IN10_STATUS, VALUE(1), 0 },
    { { SENSORS_PC87360_TEMP1, "temp1", NOMAP, NOMAP, R }, 
                               PC87365_SYSCTL_TEMP1, VALUE(4), 0 },
    { { SENSORS_PC87360_TEMP2, "temp2", NOMAP, NOMAP, R }, 
                               PC87365_SYSCTL_TEMP2, VALUE(4), 0 },
    { { SENSORS_PC87360_TEMP3, "temp3", NOMAP, NOMAP, R }, 
                               PC87365_SYSCTL_TEMP3, VALUE(4), 0 },
    { { SENSORS_PC87360_TEMP4, "temp4", NOMAP, NOMAP, R }, 
                               PC87365_SYSCTL_TEMP4, VALUE(4), 3 },
    { { SENSORS_PC87360_TEMP5, "temp5", NOMAP, NOMAP, R }, 
                               PC87365_SYSCTL_TEMP5, VALUE(4), 3 },
    { { SENSORS_PC87360_TEMP6, "temp6", NOMAP, NOMAP, R }, 
                               PC87365_SYSCTL_TEMP6, VALUE(4), 3 },
    { { SENSORS_PC87360_TEMP1_MIN, "temp1_min", SENSORS_PC87360_TEMP1, 
                                   SENSORS_PC87360_TEMP1, RW }, 
                                   PC87365_SYSCTL_TEMP1, VALUE(2), 0 },
    { { SENSORS_PC87360_TEMP2_MIN, "temp2_min", SENSORS_PC87360_TEMP2, 
                                   SENSORS_PC87360_TEMP2, RW }, 
                                   PC87365_SYSCTL_TEMP2, VALUE(2), 0 },
    { { SENSORS_PC87360_TEMP3_MIN, "temp3_min", SENSORS_PC87360_TEMP3, 
                                   SENSORS_PC87360_TEMP3, RW }, 
                                   PC87365_SYSCTL_TEMP3, VALUE(2), 0 },
    { { SENSORS_PC87360_TEMP4_MIN, "temp4_min", SENSORS_PC87360_TEMP4, 
                                   SENSORS_PC87360_TEMP4, RW }, 
                                   PC87365_SYSCTL_TEMP4, VALUE(2), 3 },
    { { SENSORS_PC87360_TEMP5_MIN, "temp5_min", SENSORS_PC87360_TEMP5, 
                                   SENSORS_PC87360_TEMP5, RW }, 
                                   PC87365_SYSCTL_TEMP5, VALUE(2), 3 },
    { { SENSORS_PC87360_TEMP6_MIN, "temp6_min", SENSORS_PC87360_TEMP6, 
                                   SENSORS_PC87360_TEMP6, RW }, 
                                   PC87365_SYSCTL_TEMP6, VALUE(2), 3 },
    { { SENSORS_PC87360_TEMP1_MAX, "temp1_max", SENSORS_PC87360_TEMP1, 
                                   SENSORS_PC87360_TEMP1, RW }, 
                                   PC87365_SYSCTL_TEMP1, VALUE(1), 0 },
    { { SENSORS_PC87360_TEMP2_MAX, "temp2_max", SENSORS_PC87360_TEMP2, 
                                   SENSORS_PC87360_TEMP2, RW }, 
                                   PC87365_SYSCTL_TEMP2, VALUE(1), 0 },
    { { SENSORS_PC87360_TEMP3_MAX, "temp3_max", SENSORS_PC87360_TEMP3, 
                                   SENSORS_PC87360_TEMP3, RW }, 
                                   PC87365_SYSCTL_TEMP3, VALUE(1), 0 },
    { { SENSORS_PC87360_TEMP4_MAX, "temp4_max", SENSORS_PC87360_TEMP4, 
                                   SENSORS_PC87360_TEMP4, RW }, 
                                   PC87365_SYSCTL_TEMP4, VALUE(1), 3 },
    { { SENSORS_PC87360_TEMP5_MAX, "temp5_max", SENSORS_PC87360_TEMP5, 
                                   SENSORS_PC87360_TEMP5, RW }, 
                                   PC87365_SYSCTL_TEMP5, VALUE(1), 3 },
    { { SENSORS_PC87360_TEMP6_MAX, "temp6_max", SENSORS_PC87360_TEMP6, 
                                   SENSORS_PC87360_TEMP6, RW }, 
                                   PC87365_SYSCTL_TEMP6, VALUE(1), 3 },
    { { SENSORS_PC87360_TEMP1_CRIT, "temp1_crit", SENSORS_PC87360_TEMP1, 
                                    SENSORS_PC87360_TEMP1, RW }, 
                                    PC87365_SYSCTL_TEMP1, VALUE(3), 0 },
    { { SENSORS_PC87360_TEMP2_CRIT, "temp2_crit", SENSORS_PC87360_TEMP2, 
                                    SENSORS_PC87360_TEMP2, RW }, 
                                    PC87365_SYSCTL_TEMP2, VALUE(3), 0 },
    { { SENSORS_PC87360_TEMP3_CRIT, "temp3_crit", SENSORS_PC87360_TEMP3, 
                                    SENSORS_PC87360_TEMP3, RW }, 
                                    PC87365_SYSCTL_TEMP3, VALUE(3), 0 },
    { { SENSORS_PC87360_TEMP4_CRIT, "temp4_crit", SENSORS_PC87360_TEMP4, 
                                    SENSORS_PC87360_TEMP4, RW }, 
                                    PC87365_SYSCTL_TEMP4, VALUE(3), 3 },
    { { SENSORS_PC87360_TEMP5_CRIT, "temp5_crit", SENSORS_PC87360_TEMP5, 
                                    SENSORS_PC87360_TEMP5, RW }, 
                                    PC87365_SYSCTL_TEMP5, VALUE(3), 3 },
    { { SENSORS_PC87360_TEMP6_CRIT, "temp6_crit", SENSORS_PC87360_TEMP6, 
                                    SENSORS_PC87360_TEMP6, RW }, 
                                    PC87365_SYSCTL_TEMP6, VALUE(3), 3 },
    { { SENSORS_PC87360_TEMP1_STATUS, "temp1_status", SENSORS_PC87360_TEMP1, 
                                      NOMAP, R }, 
                                      PC87365_SYSCTL_TEMP1_STATUS, VALUE(1), 0 },
    { { SENSORS_PC87360_TEMP2_STATUS, "temp2_status", SENSORS_PC87360_TEMP2, 
                                      NOMAP, R }, 
                                      PC87365_SYSCTL_TEMP2_STATUS, VALUE(1), 0 },
    { { SENSORS_PC87360_TEMP3_STATUS, "temp3_status", SENSORS_PC87360_TEMP3, 
                                      NOMAP, R }, 
                                      PC87365_SYSCTL_TEMP3_STATUS, VALUE(1), 0 },
    { { SENSORS_PC87360_TEMP4_STATUS, "temp4_status", SENSORS_PC87360_TEMP4, 
                                      NOMAP, R }, 
                                      PC87365_SYSCTL_TEMP4_STATUS, VALUE(1), 0 },
    { { SENSORS_PC87360_TEMP5_STATUS, "temp5_status", SENSORS_PC87360_TEMP5, 
                                      NOMAP, R }, 
                                      PC87365_SYSCTL_TEMP5_STATUS, VALUE(1), 0 },
    { { SENSORS_PC87360_TEMP6_STATUS, "temp6_status", SENSORS_PC87360_TEMP6, 
                                      NOMAP, R }, 
                                      PC87365_SYSCTL_TEMP6_STATUS, VALUE(1), 0 },
    { { SENSORS_PC87360_VID, "vid", NOMAP, NOMAP, R }, 
                             PC87365_SYSCTL_VID, VALUE(1), 3 },
    { { SENSORS_PC87360_VRM, "vrm", NOMAP, NOMAP, RW }, 
                             PC87365_SYSCTL_VRM, VALUE(1), 1 },
    { { 0 }, 0 }
  };

static sensors_chip_feature pc87427_features[] =
  {
    { { SENSORS_PC87427_FAN(1), "fan1", NOMAP, NOMAP, R }, 
                                NOSYSCTL, VALUE(2), 0 },
    { { SENSORS_PC87427_FAN(2), "fan2", NOMAP, NOMAP, R }, 
                                NOSYSCTL, VALUE(2), 0 },
    { { SENSORS_PC87427_FAN(3), "fan3", NOMAP, NOMAP, R }, 
                                NOSYSCTL, VALUE(2), 0 },
    { { SENSORS_PC87427_FAN(4), "fan4", NOMAP, NOMAP, R }, 
                                NOSYSCTL, VALUE(2), 0 },
    { { SENSORS_PC87427_FAN(5), "fan5", NOMAP, NOMAP, R }, 
                                NOSYSCTL, VALUE(2), 0 },
    { { SENSORS_PC87427_FAN(6), "fan6", NOMAP, NOMAP, R }, 
                                NOSYSCTL, VALUE(2), 0 },
    { { SENSORS_PC87427_FAN(7), "fan7", NOMAP, NOMAP, R }, 
                                NOSYSCTL, VALUE(2), 0 },
    { { SENSORS_PC87427_FAN(8), "fan8", NOMAP, NOMAP, R }, 
                                NOSYSCTL, VALUE(2), 0 },
    { { SENSORS_PC87427_FAN_MIN(1), "fan1_min", SENSORS_PC87427_FAN(1), 
                                    SENSORS_PC87427_FAN(1), RW }, 
                                    NOSYSCTL, VALUE(1), 0 },
    { { SENSORS_PC87427_FAN_MIN(2), "fan2_min", SENSORS_PC87427_FAN(2), 
                                    SENSORS_PC87427_FAN(2), RW }, 
                                    NOSYSCTL, VALUE(1), 0 },
    { { SENSORS_PC87427_FAN_MIN(3), "fan3_min", SENSORS_PC87427_FAN(3), 
                                    SENSORS_PC87427_FAN(3), RW }, 
                                    NOSYSCTL, VALUE(1), 0 },
    { { SENSORS_PC87427_FAN_MIN(4), "fan4_min", SENSORS_PC87427_FAN(4), 
                                    SENSORS_PC87427_FAN(4), RW }, 
                                    NOSYSCTL, VALUE(1), 0 },
    { { SENSORS_PC87427_FAN_MIN(5), "fan5_min", SENSORS_PC87427_FAN(5), 
                                    SENSORS_PC87427_FAN(5), RW }, 
                                    NOSYSCTL, VALUE(1), 0 },
    { { SENSORS_PC87427_FAN_MIN(6), "fan6_min", SENSORS_PC87427_FAN(6), 
                                    SENSORS_PC87427_FAN(6), RW }, 
                                    NOSYSCTL, VALUE(1), 0 },
    { { SENSORS_PC87427_FAN_MIN(7), "fan7_min", SENSORS_PC87427_FAN(7), 
                                    SENSORS_PC87427_FAN(7), RW }, 
                                    NOSYSCTL, VALUE(1), 0 },
    { { SENSORS_PC87427_FAN_MIN(8), "fan8_min", SENSORS_PC87427_FAN(8), 
                                    SENSORS_PC87427_FAN(8), RW }, 
                                    NOSYSCTL, VALUE(1), 0 },
    { { SENSORS_PC87427_FAN_ALARM(1), "fan1_alarm", SENSORS_PC87427_FAN(1), 
                                      NOMAP, R }, 
                                      NOSYSCTL, VALUE(1), 0 },
    { { SENSORS_PC87427_FAN_ALARM(2), "fan2_alarm", SENSORS_PC87427_FAN(2), 
                                      NOMAP, R }, 
                                      NOSYSCTL, VALUE(1), 0 },
    { { SENSORS_PC87427_FAN_ALARM(3), "fan3_alarm", SENSORS_PC87427_FAN(3), 
                                      NOMAP, R }, 
                                      NOSYSCTL, VALUE(1), 0 },
    { { SENSORS_PC87427_FAN_ALARM(4), "fan4_alarm", SENSORS_PC87427_FAN(4), 
                                      NOMAP, R }, 
                                      NOSYSCTL, VALUE(1), 0 },
    { { SENSORS_PC87427_FAN_ALARM(5), "fan5_alarm", SENSORS_PC87427_FAN(5), 
                                      NOMAP, R }, 
                                      NOSYSCTL, VALUE(1), 0 },
    { { SENSORS_PC87427_FAN_ALARM(6), "fan6_alarm", SENSORS_PC87427_FAN(6), 
                                      NOMAP, R }, 
                                      NOSYSCTL, VALUE(1), 0 },
    { { SENSORS_PC87427_FAN_ALARM(7), "fan7_alarm", SENSORS_PC87427_FAN(7), 
                                      NOMAP, R }, 
                                      NOSYSCTL, VALUE(1), 0 },
    { { SENSORS_PC87427_FAN_ALARM(8), "fan8_alarm", SENSORS_PC87427_FAN(8), 
                                      NOMAP, R }, 
                                      NOSYSCTL, VALUE(1), 0 },
    { { SENSORS_PC87427_FAN_FAULT(1), "fan1_fault", SENSORS_PC87427_FAN(1), 
                                      NOMAP, R }, 
                                      NOSYSCTL, VALUE(2), 0 },
    { { SENSORS_PC87427_FAN_FAULT(2), "fan2_fault", SENSORS_PC87427_FAN(2), 
                                      NOMAP, R }, 
                                      NOSYSCTL, VALUE(2), 0 },
    { { SENSORS_PC87427_FAN_FAULT(3), "fan3_fault", SENSORS_PC87427_FAN(3), 
                                      NOMAP, R }, 
                                      NOSYSCTL, VALUE(2), 0 },
    { { SENSORS_PC87427_FAN_FAULT(4), "fan4_fault", SENSORS_PC87427_FAN(4), 
                                      NOMAP, R }, 
                                      NOSYSCTL, VALUE(2), 0 },
    { { SENSORS_PC87427_FAN_FAULT(5), "fan5_fault", SENSORS_PC87427_FAN(5), 
                                      NOMAP, R }, 
                                      NOSYSCTL, VALUE(2), 0 },
    { { SENSORS_PC87427_FAN_FAULT(6), "fan6_fault", SENSORS_PC87427_FAN(6), 
                                      NOMAP, R }, 
                                      NOSYSCTL, VALUE(2), 0 },
    { { SENSORS_PC87427_FAN_FAULT(7), "fan7_fault", SENSORS_PC87427_FAN(7), 
                                      NOMAP, R }, 
                                      NOSYSCTL, VALUE(2), 0 },
    { { SENSORS_PC87427_FAN_FAULT(8), "fan8_fault", SENSORS_PC87427_FAN(8), 
                                      NOMAP, R }, 
                                      NOSYSCTL, VALUE(2), 0 },
    { { 0 }, 0 }
  };

static sensors_chip_feature lm92_features[] =
  {
    { { SENSORS_LM92_TEMP, "temp", NOMAP, NOMAP, R }, 
                           LM92_SYSCTL_TEMP, VALUE(1), 4 },
    { { SENSORS_LM92_TEMP_HIGH, "temp_high", SENSORS_LM92_TEMP, 
                                SENSORS_LM92_TEMP, RW }, 
                                LM92_SYSCTL_TEMP, VALUE(2), 4 },
    { { SENSORS_LM92_TEMP_LOW, "temp_low", SENSORS_LM92_TEMP, SENSORS_LM92_TEMP, 
                               RW }, 
                               LM92_SYSCTL_TEMP, VALUE(3), 4 },
    { { SENSORS_LM92_TEMP_CRIT, "temp_crit", SENSORS_LM92_TEMP, 
                                SENSORS_LM92_TEMP, RW }, 
                                LM92_SYSCTL_TEMP, VALUE(4), 4 },
    { { SENSORS_LM92_TEMP_HYST, "temp_hyst", SENSORS_LM92_TEMP, 
                                SENSORS_LM92_TEMP, RW }, 
                                LM92_SYSCTL_TEMP, VALUE(5), 4 , 
                                "temp1_crit_hyst", 3 },
    { { SENSORS_LM92_ALARMS, "alarms", NOMAP, NOMAP, R }, 
                             LM92_SYSCTL_ALARMS, VALUE(1), 0 },
    { { 0 }, 0 }
  };

static sensors_chip_feature vt8231_features[] =
  { 
    { { SENSORS_VT8231_IN0, "in0", NOMAP, NOMAP, R }, 
                            VT8231_SYSCTL_IN0, VALUE(3), 2 },
    { { SENSORS_VT8231_IN1, "in1", NOMAP, NOMAP, R }, 
                            VT8231_SYSCTL_IN1, VALUE(3), 2 },
    { { SENSORS_VT8231_IN2, "in2", NOMAP, NOMAP, R }, 
                            VT8231_SYSCTL_IN2, VALUE(3), 2 },
    { { SENSORS_VT8231_IN3, "in3", NOMAP, NOMAP, R }, 
                            VT8231_SYSCTL_IN3, VALUE(3), 2 },
    { { SENSORS_VT8231_IN4, "in4", NOMAP, NOMAP, R }, 
                            VT8231_SYSCTL_IN4, VALUE(3), 2 },
    { { SENSORS_VT8231_IN5, "in5", NOMAP, NOMAP, R }, 
                            VT8231_SYSCTL_IN5, VALUE(3), 2 },
    { { SENSORS_VT8231_IN0_MIN, "in0_min", SENSORS_VT8231_IN0, 
                                SENSORS_VT8231_IN0, RW }, 
                                VT8231_SYSCTL_IN0, VALUE(1), 2 },
    { { SENSORS_VT8231_IN1_MIN, "in1_min", SENSORS_VT8231_IN1, 
                                SENSORS_VT8231_IN1, RW }, 
                                VT8231_SYSCTL_IN1, VALUE(1), 2 },
    { { SENSORS_VT8231_IN2_MIN, "in2_min", SENSORS_VT8231_IN2, 
                                SENSORS_VT8231_IN2, RW }, 
                                VT8231_SYSCTL_IN2, VALUE(1), 2 },
    { { SENSORS_VT8231_IN3_MIN, "in3_min", SENSORS_VT8231_IN3, 
                                SENSORS_VT8231_IN3, RW }, 
                                VT8231_SYSCTL_IN3, VALUE(1), 2 },
    { { SENSORS_VT8231_IN4_MIN, "in4_min", SENSORS_VT8231_IN4, 
                                SENSORS_VT8231_IN4, RW }, 
                                VT8231_SYSCTL_IN4, VALUE(1), 2 },
    { { SENSORS_VT8231_IN5_MIN, "in5_min", SENSORS_VT8231_IN5, 
                                SENSORS_VT8231_IN5, RW }, 
                                VT8231_SYSCTL_IN5, VALUE(1), 2 },
    { { SENSORS_VT8231_IN0_MAX, "in0_max", SENSORS_VT8231_IN0, 
                                SENSORS_VT8231_IN0, RW }, 
                                VT8231_SYSCTL_IN0, VALUE(2), 2 },
    { { SENSORS_VT8231_IN1_MAX, "in1_max", SENSORS_VT8231_IN1, 
                                SENSORS_VT8231_IN1, RW }, 
                                VT8231_SYSCTL_IN1, VALUE(2), 2 },
    { { SENSORS_VT8231_IN2_MAX, "in2_max", SENSORS_VT8231_IN2, 
                                SENSORS_VT8231_IN2, RW }, 
                                VT8231_SYSCTL_IN2, VALUE(2), 2 },
    { { SENSORS_VT8231_IN3_MAX, "in3_max", SENSORS_VT8231_IN3, 
                                SENSORS_VT8231_IN3, RW }, 
                                VT8231_SYSCTL_IN3, VALUE(2), 2 },
    { { SENSORS_VT8231_IN4_MAX, "in4_max", SENSORS_VT8231_IN4, 
                                SENSORS_VT8231_IN4, RW }, 
                                VT8231_SYSCTL_IN4, VALUE(2), 2 },
    { { SENSORS_VT8231_IN5_MAX, "in5_max", SENSORS_VT8231_IN5, 
                                SENSORS_VT8231_IN5, RW }, 
                                VT8231_SYSCTL_IN5, VALUE(2), 2 },
    { { SENSORS_VT8231_FAN1, "fan1", NOMAP, NOMAP, R }, 
                             VT8231_SYSCTL_FAN1, VALUE(2), 0 },
    { { SENSORS_VT8231_FAN2, "fan2", NOMAP, NOMAP, R }, 
                             VT8231_SYSCTL_FAN2, VALUE(2), 0 },
    { { SENSORS_VT8231_FAN1_MIN, "fan1_min", SENSORS_VT8231_FAN1, 
                                 SENSORS_VT8231_FAN1, RW }, 
                                 VT8231_SYSCTL_FAN1, VALUE(1), 0 },
    { { SENSORS_VT8231_FAN2_MIN, "fan2_min", SENSORS_VT8231_FAN2, 
                                 SENSORS_VT8231_FAN2, RW }, 
                                 VT8231_SYSCTL_FAN2, VALUE(1), 0 },
    { { SENSORS_VT8231_TEMP, "temp1", NOMAP, NOMAP, R }, 
                             VT8231_SYSCTL_TEMP, VALUE(3), 1 },
    { { SENSORS_VT8231_TEMP_HYST, "temp1_hyst", SENSORS_VT8231_TEMP, 
                                  SENSORS_VT8231_TEMP, RW }, 
                                  VT8231_SYSCTL_TEMP, VALUE(2), 1 },
    { { SENSORS_VT8231_TEMP_OVER, "temp1_over", SENSORS_VT8231_TEMP, 
                                  SENSORS_VT8231_TEMP, RW }, 
                                  VT8231_SYSCTL_TEMP, VALUE(1), 1 },
    { { SENSORS_VT8231_TEMP2, "temp2", NOMAP, NOMAP, R }, 
                              VT8231_SYSCTL_TEMP2, VALUE(3), 2 },
    { { SENSORS_VT8231_TEMP2_HYST, "temp2_hyst", SENSORS_VT8231_TEMP2, 
                                   SENSORS_VT8231_TEMP2, RW }, 
                                   VT8231_SYSCTL_TEMP2, VALUE(2), 2 },
    { { SENSORS_VT8231_TEMP2_OVER, "temp2_over", SENSORS_VT8231_TEMP2, 
                                   SENSORS_VT8231_TEMP2, RW }, 
                                   VT8231_SYSCTL_TEMP2, VALUE(1), 2 },
    { { SENSORS_VT8231_TEMP3, "temp3", NOMAP, NOMAP, R }, 
                              VT8231_SYSCTL_TEMP3, VALUE(3), 2 },
    { { SENSORS_VT8231_TEMP3_HYST, "temp3_hyst", SENSORS_VT8231_TEMP3, 
                                   SENSORS_VT8231_TEMP3, RW }, 
                                   VT8231_SYSCTL_TEMP3, VALUE(2), 2 },
    { { SENSORS_VT8231_TEMP3_OVER, "temp3_over", SENSORS_VT8231_TEMP3, 
                                   SENSORS_VT8231_TEMP3, RW }, 
                                   VT8231_SYSCTL_TEMP3, VALUE(1), 2 },
    { { SENSORS_VT8231_TEMP4, "temp4", NOMAP, NOMAP, R }, 
                              VT8231_SYSCTL_TEMP4, VALUE(3), 2 },
    { { SENSORS_VT8231_TEMP4_HYST, "temp4_hyst", SENSORS_VT8231_TEMP4, 
                                   SENSORS_VT8231_TEMP4, RW }, 
                                   VT8231_SYSCTL_TEMP4, VALUE(2), 2 },
    { { SENSORS_VT8231_TEMP4_OVER, "temp4_over", SENSORS_VT8231_TEMP4, 
                                   SENSORS_VT8231_TEMP4, RW }, 
                                   VT8231_SYSCTL_TEMP4, VALUE(1), 2 },
    { { SENSORS_VT8231_TEMP5, "temp5", NOMAP, NOMAP, R }, 
                              VT8231_SYSCTL_TEMP5, VALUE(3), 2 },
    { { SENSORS_VT8231_TEMP5_HYST, "temp5_hyst", SENSORS_VT8231_TEMP5, 
                                   SENSORS_VT8231_TEMP5, RW }, 
                                   VT8231_SYSCTL_TEMP5, VALUE(2), 2 },
    { { SENSORS_VT8231_TEMP5_OVER, "temp5_over", SENSORS_VT8231_TEMP5, 
                                   SENSORS_VT8231_TEMP5, RW }, 
                                   VT8231_SYSCTL_TEMP5, VALUE(1), 2 },
    { { SENSORS_VT8231_TEMP6, "temp6", NOMAP, NOMAP, R }, 
                              VT8231_SYSCTL_TEMP6, VALUE(3), 2 },
    { { SENSORS_VT8231_TEMP6_HYST, "temp6_hyst", SENSORS_VT8231_TEMP6, 
                                   SENSORS_VT8231_TEMP6, RW }, 
                                   VT8231_SYSCTL_TEMP6, VALUE(2), 2 },
    { { SENSORS_VT8231_TEMP6_OVER, "temp6_over", SENSORS_VT8231_TEMP6, 
                                   SENSORS_VT8231_TEMP6, RW }, 
                                   VT8231_SYSCTL_TEMP6, VALUE(1), 2 },
    { { SENSORS_VT8231_FAN1_DIV, "fan1_div", SENSORS_VT8231_FAN1, NOMAP, RW }, 
                                 VT8231_SYSCTL_FAN_DIV, VALUE(1), 0 },
    { { SENSORS_VT8231_FAN2_DIV, "fan2_div", SENSORS_VT8231_FAN2, NOMAP, RW }, 
                                 VT8231_SYSCTL_FAN_DIV, VALUE(2), 0 },
    { { SENSORS_VT8231_ALARMS, "alarms", NOMAP, NOMAP, R }, 
                               VT8231_SYSCTL_ALARMS, VALUE(1), 0 },
    { { SENSORS_VT8231_VID, "vid", NOMAP, NOMAP, R }, 
                            VT8231_SYSCTL_VID, VALUE(1), 3 },
    { { SENSORS_VT8231_VRM, "vrm", NOMAP, NOMAP, RW }, 
                            VT8231_SYSCTL_VRM, VALUE(1), 1 },
    { { SENSORS_VT8231_UCH, "config", NOMAP, NOMAP, RW }, 
                            VT8231_SYSCTL_UCH, VALUE(1), 0 },
    { { 0 }, 0 }
  };

static sensors_chip_feature bmc_features[] =
  { 
    { { SENSORS_BMC_IN1, "in1", NOMAP, NOMAP, R }, 
                         BMC_SYSCTL_IN1, VALUE(3), 3 },
    { { SENSORS_BMC_IN1_MIN, "in1_min", SENSORS_BMC_IN1, SENSORS_BMC_IN1, RW }, 
                             BMC_SYSCTL_IN1, VALUE(1), 3 },
    { { SENSORS_BMC_IN1_MAX, "in1_max", SENSORS_BMC_IN1, SENSORS_BMC_IN1, RW }, 
                             BMC_SYSCTL_IN1, VALUE(2), 3 },
    { { SENSORS_BMC_IN1+1, "in2", NOMAP, NOMAP, R }, 
                           BMC_SYSCTL_IN1+1, VALUE(3), 3 },
    { { SENSORS_BMC_IN1_MIN+1, "in2_min", SENSORS_BMC_IN1+1, SENSORS_BMC_IN1+1, 
                               RW }, 
                               BMC_SYSCTL_IN1+1, VALUE(1), 3 },
    { { SENSORS_BMC_IN1_MAX+1, "in2_max", SENSORS_BMC_IN1+1, SENSORS_BMC_IN1+1, 
                               RW }, 
                               BMC_SYSCTL_IN1+1, VALUE(2), 3 },
    { { SENSORS_BMC_IN1+2, "in3", NOMAP, NOMAP, R }, 
                           BMC_SYSCTL_IN1+2, VALUE(3), 3 },
    { { SENSORS_BMC_IN1_MIN+2, "in3_min", SENSORS_BMC_IN1+2, SENSORS_BMC_IN1+2, 
                               RW }, 
                               BMC_SYSCTL_IN1+2, VALUE(1), 3 },
    { { SENSORS_BMC_IN1_MAX+2, "in3_max", SENSORS_BMC_IN1+2, SENSORS_BMC_IN1+2, 
                               RW }, 
                               BMC_SYSCTL_IN1+2, VALUE(2), 3 },
    { { SENSORS_BMC_IN1+3, "in4", NOMAP, NOMAP, R }, 
                           BMC_SYSCTL_IN1+3, VALUE(3), 3 },
    { { SENSORS_BMC_IN1_MIN+3, "in4_min", SENSORS_BMC_IN1+3, SENSORS_BMC_IN1+3, 
                               RW }, 
                               BMC_SYSCTL_IN1+3, VALUE(1), 3 },
    { { SENSORS_BMC_IN1_MAX+3, "in4_max", SENSORS_BMC_IN1+3, SENSORS_BMC_IN1+3, 
                               RW }, 
                               BMC_SYSCTL_IN1+3, VALUE(2), 3 },
    { { SENSORS_BMC_IN1+4, "in5", NOMAP, NOMAP, R }, 
                           BMC_SYSCTL_IN1+4, VALUE(3), 3 },
    { { SENSORS_BMC_IN1_MIN+4, "in5_min", SENSORS_BMC_IN1+4, SENSORS_BMC_IN1+4, 
                               RW }, 
                               BMC_SYSCTL_IN1+4, VALUE(1), 3 },
    { { SENSORS_BMC_IN1_MAX+4, "in5_max", SENSORS_BMC_IN1+4, SENSORS_BMC_IN1+4, 
                               RW }, 
                               BMC_SYSCTL_IN1+4, VALUE(2), 3 },
    { { SENSORS_BMC_IN1+5, "in6", NOMAP, NOMAP, R }, 
                           BMC_SYSCTL_IN1+5, VALUE(3), 3 },
    { { SENSORS_BMC_IN1_MIN+5, "in6_min", SENSORS_BMC_IN1+5, SENSORS_BMC_IN1+5, 
                               RW }, 
                               BMC_SYSCTL_IN1+5, VALUE(1), 3 },
    { { SENSORS_BMC_IN1_MAX+5, "in6_max", SENSORS_BMC_IN1+5, SENSORS_BMC_IN1+5, 
                               RW }, 
                               BMC_SYSCTL_IN1+5, VALUE(2), 3 },
    { { SENSORS_BMC_IN1+6, "in7", NOMAP, NOMAP, R }, 
                           BMC_SYSCTL_IN1+6, VALUE(3), 3 },
    { { SENSORS_BMC_IN1_MIN+6, "in7_min", SENSORS_BMC_IN1+6, SENSORS_BMC_IN1+6, 
                               RW }, 
                               BMC_SYSCTL_IN1+6, VALUE(1), 3 },
    { { SENSORS_BMC_IN1_MAX+6, "in7_max", SENSORS_BMC_IN1+6, SENSORS_BMC_IN1+6, 
                               RW }, 
                               BMC_SYSCTL_IN1+6, VALUE(2), 3 },

    { { SENSORS_BMC_IN1+7, "in8", NOMAP, NOMAP, R }, 
                           BMC_SYSCTL_IN1+7, VALUE(3), 3 },
    { { SENSORS_BMC_IN1_MIN+7, "in8_min", SENSORS_BMC_IN1+7, SENSORS_BMC_IN1+7, 
                               RW }, 
                               BMC_SYSCTL_IN1+7, VALUE(1), 3 },
    { { SENSORS_BMC_IN1_MAX+7, "in8_max", SENSORS_BMC_IN1+7, SENSORS_BMC_IN1+7, 
                               RW }, 
                               BMC_SYSCTL_IN1+7, VALUE(2), 3 },
    { { SENSORS_BMC_IN1+8, "in9", NOMAP, NOMAP, R }, 
                           BMC_SYSCTL_IN1+8, VALUE(3), 3 },
    { { SENSORS_BMC_IN1_MIN+8, "in9_min", SENSORS_BMC_IN1+8, SENSORS_BMC_IN1+8, 
                               RW }, 
                               BMC_SYSCTL_IN1+8, VALUE(1), 3 },
    { { SENSORS_BMC_IN1_MAX+8, "in9_max", SENSORS_BMC_IN1+8, SENSORS_BMC_IN1+8, 
                               RW }, 
                               BMC_SYSCTL_IN1+8, VALUE(2), 3 },
    { { SENSORS_BMC_IN1+9, "in10", NOMAP, NOMAP, R }, 
                           BMC_SYSCTL_IN1+9, VALUE(3), 3 },
    { { SENSORS_BMC_IN1_MIN+9, "in10_min", SENSORS_BMC_IN1+9, SENSORS_BMC_IN1+9, 
                               RW }, 
                               BMC_SYSCTL_IN1+9, VALUE(1), 3 },
    { { SENSORS_BMC_IN1_MAX+9, "in10_max", SENSORS_BMC_IN1+9, SENSORS_BMC_IN1+9, 
                               RW }, 
                               BMC_SYSCTL_IN1+9, VALUE(2), 3 },
    { { SENSORS_BMC_IN1+10, "in11", NOMAP, NOMAP, R }, 
                            BMC_SYSCTL_IN1+10, VALUE(3), 3 },
    { { SENSORS_BMC_IN1_MIN+10, "in11_min", SENSORS_BMC_IN1+10, 
                                SENSORS_BMC_IN1+10, RW }, 
                                BMC_SYSCTL_IN1+10, VALUE(1), 3 },
    { { SENSORS_BMC_IN1_MAX+10, "in11_max", SENSORS_BMC_IN1+10, 
                                SENSORS_BMC_IN1+10, RW }, 
                                BMC_SYSCTL_IN1+10, VALUE(2), 3 },
    { { SENSORS_BMC_IN1+11, "in12", NOMAP, NOMAP, R }, 
                            BMC_SYSCTL_IN1+11, VALUE(3), 3 },
    { { SENSORS_BMC_IN1_MIN+11, "in12_min", SENSORS_BMC_IN1+11, 
                                SENSORS_BMC_IN1+11, RW }, 
                                BMC_SYSCTL_IN1+11, VALUE(1), 3 },
    { { SENSORS_BMC_IN1_MAX+11, "in12_max", SENSORS_BMC_IN1+11, 
                                SENSORS_BMC_IN1+11, RW }, 
                                BMC_SYSCTL_IN1+11, VALUE(2), 3 },
    { { SENSORS_BMC_IN1+12, "in13", NOMAP, NOMAP, R }, 
                            BMC_SYSCTL_IN1+12, VALUE(3), 3 },
    { { SENSORS_BMC_IN1_MIN+12, "in13_min", SENSORS_BMC_IN1+12, 
                                SENSORS_BMC_IN1+12, RW }, 
                                BMC_SYSCTL_IN1+12, VALUE(1), 3 },
    { { SENSORS_BMC_IN1_MAX+12, "in13_max", SENSORS_BMC_IN1+12, 
                                SENSORS_BMC_IN1+12, RW }, 
                                BMC_SYSCTL_IN1+12, VALUE(2), 3 },
    { { SENSORS_BMC_IN1+13, "in14", NOMAP, NOMAP, R }, 
                            BMC_SYSCTL_IN1+13, VALUE(3), 3 },
    { { SENSORS_BMC_IN1_MIN+13, "in14_min", SENSORS_BMC_IN1+13, 
                                SENSORS_BMC_IN1+13, RW }, 
                                BMC_SYSCTL_IN1+13, VALUE(1), 3 },
    { { SENSORS_BMC_IN1_MAX+13, "in14_max", SENSORS_BMC_IN1+13, 
                                SENSORS_BMC_IN1+13, RW }, 
                                BMC_SYSCTL_IN1+13, VALUE(2), 3 },
    { { SENSORS_BMC_IN1+14, "in15", NOMAP, NOMAP, R }, 
                            BMC_SYSCTL_IN1+14, VALUE(3), 3 },
    { { SENSORS_BMC_IN1_MIN+14, "in15_min", SENSORS_BMC_IN1+14, 
                                SENSORS_BMC_IN1+14, RW }, 
                                BMC_SYSCTL_IN1+14, VALUE(1), 3 },
    { { SENSORS_BMC_IN1_MAX+14, "in15_max", SENSORS_BMC_IN1+14, 
                                SENSORS_BMC_IN1+14, RW }, 
                                BMC_SYSCTL_IN1+14, VALUE(2), 3 },
    { { SENSORS_BMC_IN1+15, "in16", NOMAP, NOMAP, R }, 
                            BMC_SYSCTL_IN1+15, VALUE(3), 3 },
    { { SENSORS_BMC_IN1_MIN+15, "in16_min", SENSORS_BMC_IN1+15, 
                                SENSORS_BMC_IN1+15, RW }, 
                                BMC_SYSCTL_IN1+15, VALUE(1), 3 },
    { { SENSORS_BMC_IN1_MAX+15, "in16_max", SENSORS_BMC_IN1+15, 
                                SENSORS_BMC_IN1+15, RW }, 
                                BMC_SYSCTL_IN1+15, VALUE(2), 3 },
    { { SENSORS_BMC_IN1+16, "in17", NOMAP, NOMAP, R }, 
                            BMC_SYSCTL_IN1+16, VALUE(3), 3 },
    { { SENSORS_BMC_IN1_MIN+16, "in17_min", SENSORS_BMC_IN1+16, 
                                SENSORS_BMC_IN1+16, RW }, 
                                BMC_SYSCTL_IN1+16, VALUE(1), 3 },
    { { SENSORS_BMC_IN1_MAX+16, "in17_max", SENSORS_BMC_IN1+16, 
                                SENSORS_BMC_IN1+16, RW }, 
                                BMC_SYSCTL_IN1+16, VALUE(2), 3 },
    { { SENSORS_BMC_IN1+17, "in18", NOMAP, NOMAP, R }, 
                            BMC_SYSCTL_IN1+17, VALUE(3), 3, },
    { { SENSORS_BMC_IN1_MIN+17, "in18_min", SENSORS_BMC_IN1+17, 
                                SENSORS_BMC_IN1+17, RW }, 
                                BMC_SYSCTL_IN1+17, VALUE(1), 3 },
    { { SENSORS_BMC_IN1_MAX+17, "in18_max", SENSORS_BMC_IN1+17, 
                                SENSORS_BMC_IN1+17, RW }, 
                                BMC_SYSCTL_IN1+17, VALUE(2), 3 },
    { { SENSORS_BMC_IN1+18, "in19", NOMAP, NOMAP, R }, 
                            BMC_SYSCTL_IN1+18, VALUE(3), 3 },
    { { SENSORS_BMC_IN1_MIN+18, "in19_min", SENSORS_BMC_IN1+18, 
                                SENSORS_BMC_IN1+18, RW }, 
                                BMC_SYSCTL_IN1+18, VALUE(1), 3 },
    { { SENSORS_BMC_IN1_MAX+18, "in19_max", SENSORS_BMC_IN1+18, 
                                SENSORS_BMC_IN1+18, RW }, 
                                BMC_SYSCTL_IN1+18, VALUE(2), 3 },
    { { SENSORS_BMC_IN1+19, "in20", NOMAP, NOMAP, R }, 
                            BMC_SYSCTL_IN1+19, VALUE(3), 3 },
    { { SENSORS_BMC_IN1_MIN+19, "in20_min", SENSORS_BMC_IN1+19, 
                                SENSORS_BMC_IN1+19, RW }, 
                                BMC_SYSCTL_IN1+19, VALUE(1), 3 },
    { { SENSORS_BMC_IN1_MAX+19, "in20_max", SENSORS_BMC_IN1+19, 
                                SENSORS_BMC_IN1+19, RW }, 
                                BMC_SYSCTL_IN1+19, VALUE(2), 3 },
    { { SENSORS_BMC_FAN1, "fan1", NOMAP, NOMAP, R }, 
                          BMC_SYSCTL_FAN1, VALUE(2), 0, },
    { { SENSORS_BMC_FAN1+1, "fan2", NOMAP, NOMAP, R }, 
                            BMC_SYSCTL_FAN1+1, VALUE(2), 0, },
    { { SENSORS_BMC_FAN1+2, "fan3", NOMAP, NOMAP, R }, 
                            BMC_SYSCTL_FAN1+2, VALUE(2), 0, },
    { { SENSORS_BMC_FAN1+3, "fan4", NOMAP, NOMAP, R }, 
                            BMC_SYSCTL_FAN1+3, VALUE(2), 0, },
    { { SENSORS_BMC_FAN1+4, "fan5", NOMAP, NOMAP, R }, 
                            BMC_SYSCTL_FAN1+4, VALUE(2), 0, },
    { { SENSORS_BMC_FAN1+5, "fan6", NOMAP, NOMAP, R }, 
                            BMC_SYSCTL_FAN1+5, VALUE(2), 0, },
    { { SENSORS_BMC_FAN1+6, "fan7", NOMAP, NOMAP, R }, 
                            BMC_SYSCTL_FAN1+6, VALUE(2), 0, },
    { { SENSORS_BMC_FAN1+7, "fan8", NOMAP, NOMAP, R }, 
                            BMC_SYSCTL_FAN1+7, VALUE(2), 0, },
    { { SENSORS_BMC_FAN1+8, "fan9", NOMAP, NOMAP, R }, 
                            BMC_SYSCTL_FAN1+8, VALUE(2), 0, },
    { { SENSORS_BMC_FAN1+9, "fan10", NOMAP, NOMAP, R }, 
                            BMC_SYSCTL_FAN1+9, VALUE(2), 0, },
    { { SENSORS_BMC_FAN1_MIN, "fan1_min", SENSORS_BMC_FAN1, SENSORS_BMC_FAN1, RW }, 
                              BMC_SYSCTL_FAN1, VALUE(1), 0 },
    { { SENSORS_BMC_FAN1_MIN+1, "fan2_min", SENSORS_BMC_FAN1+1, 
                                SENSORS_BMC_FAN1+1, RW }, 
                                BMC_SYSCTL_FAN1+1, VALUE(1), 0 },
    { { SENSORS_BMC_FAN1_MIN+2, "fan3_min", SENSORS_BMC_FAN1+2, 
                                SENSORS_BMC_FAN1+2, RW }, 
                                BMC_SYSCTL_FAN1+2, VALUE(1), 0 },
    { { SENSORS_BMC_FAN1_MIN+3, "fan4_min", SENSORS_BMC_FAN1+3, 
                                SENSORS_BMC_FAN1+3, RW }, 
                                BMC_SYSCTL_FAN1+3, VALUE(1), 0 },
    { { SENSORS_BMC_FAN1_MIN+4, "fan5_min", SENSORS_BMC_FAN1+4, 
                                SENSORS_BMC_FAN1+4, RW }, 
                                BMC_SYSCTL_FAN1+4, VALUE(1), 0 },
    { { SENSORS_BMC_FAN1_MIN+5, "fan6_min", SENSORS_BMC_FAN1+5, 
                                SENSORS_BMC_FAN1+5, RW }, 
                                BMC_SYSCTL_FAN1+5, VALUE(1), 0 },
    { { SENSORS_BMC_FAN1_MIN+6, "fan7_min", SENSORS_BMC_FAN1+6, 
                                SENSORS_BMC_FAN1+6, RW }, 
                                BMC_SYSCTL_FAN1+6, VALUE(1), 0 },
    { { SENSORS_BMC_FAN1_MIN+7, "fan8_min", SENSORS_BMC_FAN1+7, 
                                SENSORS_BMC_FAN1+7, RW }, 
                                BMC_SYSCTL_FAN1+7, VALUE(1), 0 },
    { { SENSORS_BMC_FAN1_MIN+8, "fan9_min", SENSORS_BMC_FAN1+8, 
                                SENSORS_BMC_FAN1+8, RW }, 
                                BMC_SYSCTL_FAN1+8, VALUE(1), 0 },
    { { SENSORS_BMC_FAN1_MIN+9, "fan10_min", SENSORS_BMC_FAN1+9, 
                                SENSORS_BMC_FAN1+9, RW }, 
                                BMC_SYSCTL_FAN1+9, VALUE(1), 0 },
    { { SENSORS_BMC_TEMP1, "temp1", NOMAP, NOMAP, R }, 
                           BMC_SYSCTL_TEMP1, VALUE(3), 2 },
    { { SENSORS_BMC_TEMP1_MIN, "temp1_min", SENSORS_BMC_TEMP1, SENSORS_BMC_TEMP1, 
                               RW }, 
                               BMC_SYSCTL_TEMP1, VALUE(2), 2 },
    { { SENSORS_BMC_TEMP1_MAX, "temp1_max", SENSORS_BMC_TEMP1, SENSORS_BMC_TEMP1, 
                               RW }, 
                               BMC_SYSCTL_TEMP1, VALUE(1), 2 },
    { { SENSORS_BMC_TEMP1+1, "temp2", NOMAP, NOMAP, R }, 
                             BMC_SYSCTL_TEMP1+1, VALUE(3), 2 },
    { { SENSORS_BMC_TEMP1_MIN+1, "temp2_min", SENSORS_BMC_TEMP1+1, 
                                 SENSORS_BMC_TEMP1+1, RW }, 
                                 BMC_SYSCTL_TEMP1+1, VALUE(2), 2 },
    { { SENSORS_BMC_TEMP1_MAX+1, "temp2_max", SENSORS_BMC_TEMP1+1, 
                                 SENSORS_BMC_TEMP1+1, RW }, 
                                 BMC_SYSCTL_TEMP1+1, VALUE(1), 2 },
    { { SENSORS_BMC_TEMP1+2, "temp3", NOMAP, NOMAP, R }, 
                             BMC_SYSCTL_TEMP1+2, VALUE(3), 2 },
    { { SENSORS_BMC_TEMP1_MIN+2, "temp3_min", SENSORS_BMC_TEMP1+2, 
                                 SENSORS_BMC_TEMP1+2, RW }, 
                                 BMC_SYSCTL_TEMP1+2, VALUE(2), 2 },
    { { SENSORS_BMC_TEMP1_MAX+2, "temp3_max", SENSORS_BMC_TEMP1+2, 
                                 SENSORS_BMC_TEMP1+2, RW }, 
                                 BMC_SYSCTL_TEMP1+2, VALUE(1), 2 },
    { { SENSORS_BMC_TEMP1+3, "temp4", NOMAP, NOMAP, R }, 
                             BMC_SYSCTL_TEMP1+3, VALUE(3), 2 },
    { { SENSORS_BMC_TEMP1_MIN+3, "temp4_min", SENSORS_BMC_TEMP1+3, 
                                 SENSORS_BMC_TEMP1+3, RW }, 
                                 BMC_SYSCTL_TEMP1+3, VALUE(2), 2 },
    { { SENSORS_BMC_TEMP1_MAX+3, "temp4_max", SENSORS_BMC_TEMP1+3, 
                                 SENSORS_BMC_TEMP1+3, RW }, 
                                 BMC_SYSCTL_TEMP1+3, VALUE(1), 2 },
    { { SENSORS_BMC_TEMP1+4, "temp5", NOMAP, NOMAP, R }, 
                             BMC_SYSCTL_TEMP1+4, VALUE(3), 2 },
    { { SENSORS_BMC_TEMP1_MIN+4, "temp5_min", SENSORS_BMC_TEMP1+4, 
                                 SENSORS_BMC_TEMP1+4, RW }, 
                                 BMC_SYSCTL_TEMP1+4, VALUE(2), 2 },
    { { SENSORS_BMC_TEMP1_MAX+4, "temp5_max", SENSORS_BMC_TEMP1+4, 
                                 SENSORS_BMC_TEMP1+4, RW }, 
                                 BMC_SYSCTL_TEMP1+4, VALUE(1), 2 },
    { { SENSORS_BMC_TEMP1+5, "temp6", NOMAP, NOMAP, R }, 
                             BMC_SYSCTL_TEMP1+5, VALUE(3), 2 },
    { { SENSORS_BMC_TEMP1_MIN+5, "temp6_min", SENSORS_BMC_TEMP1+5, 
                                 SENSORS_BMC_TEMP1+5, RW }, 
                                 BMC_SYSCTL_TEMP1+5, VALUE(2), 2 },
    { { SENSORS_BMC_TEMP1_MAX+5, "temp6_max", SENSORS_BMC_TEMP1+5, 
                                 SENSORS_BMC_TEMP1+5, RW }, 
                                 BMC_SYSCTL_TEMP1+5, VALUE(1), 2 },
    { { SENSORS_BMC_TEMP1+6, "temp7", NOMAP, NOMAP, R }, 
                             BMC_SYSCTL_TEMP1+6, VALUE(3), 2 },
    { { SENSORS_BMC_TEMP1_MIN+6, "temp7_min", SENSORS_BMC_TEMP1+6, 
                                 SENSORS_BMC_TEMP1+6, RW }, 
                                 BMC_SYSCTL_TEMP1+6, VALUE(2), 2 },
    { { SENSORS_BMC_TEMP1_MAX+6, "temp7_max", SENSORS_BMC_TEMP1+6, 
                                 SENSORS_BMC_TEMP1+6, RW }, 
                                 BMC_SYSCTL_TEMP1+6, VALUE(1), 2 },
    { { SENSORS_BMC_TEMP1+7, "temp8", NOMAP, NOMAP, R }, 
                             BMC_SYSCTL_TEMP1+7, VALUE(3), 2 },
    { { SENSORS_BMC_TEMP1_MIN+7, "temp8_min", SENSORS_BMC_TEMP1+7, 
                                 SENSORS_BMC_TEMP1+7, RW }, 
                                 BMC_SYSCTL_TEMP1+7, VALUE(2), 2 },
    { { SENSORS_BMC_TEMP1_MAX+7, "temp8_max", SENSORS_BMC_TEMP1+7, 
                                 SENSORS_BMC_TEMP1+7, RW }, 
                                 BMC_SYSCTL_TEMP1+7, VALUE(1), 2 },
    { { SENSORS_BMC_TEMP1+8, "temp9", NOMAP, NOMAP, R }, 
                             BMC_SYSCTL_TEMP1+8, VALUE(3), 2 },
    { { SENSORS_BMC_TEMP1_MIN+8, "temp9_min", SENSORS_BMC_TEMP1+8, 
                                 SENSORS_BMC_TEMP1+8, RW }, 
                                 BMC_SYSCTL_TEMP1+8, VALUE(2), 2 },
    { { SENSORS_BMC_TEMP1_MAX+8, "temp9_max", SENSORS_BMC_TEMP1+8, 
                                 SENSORS_BMC_TEMP1+8, RW }, 
                                 BMC_SYSCTL_TEMP1+8, VALUE(1), 2 },
    { { SENSORS_BMC_TEMP1+9, "temp10", NOMAP, NOMAP, R }, 
                             BMC_SYSCTL_TEMP1+9, VALUE(3), 2 },
    { { SENSORS_BMC_TEMP1_MIN+9, "temp10_min", SENSORS_BMC_TEMP1+9, 
                                 SENSORS_BMC_TEMP1+9, RW }, 
                                 BMC_SYSCTL_TEMP1+9, VALUE(2), 2 },
    { { SENSORS_BMC_TEMP1_MAX+9, "temp10_max", SENSORS_BMC_TEMP1+9, 
                                 SENSORS_BMC_TEMP1+9, RW }, 
                                 BMC_SYSCTL_TEMP1+9, VALUE(1), 2 },
    { { SENSORS_BMC_ALARMS, "alarms", NOMAP, NOMAP, R }, 
                            BMC_SYSCTL_ALARMS, VALUE(1), 0 },
    { { 0 }, 0 }
  };

static sensors_chip_feature xeontemp_features[] =
  {
    { { SENSORS_XEONTEMP_REMOTE_TEMP, "temp", NOMAP, NOMAP, R }, 
                                      XEONTEMP_SYSCTL_REMOTE_TEMP, VALUE(3), 0 },
    { { SENSORS_XEONTEMP_REMOTE_TEMP_HYST, "temp_low", 
                                           SENSORS_XEONTEMP_REMOTE_TEMP, 
                                           SENSORS_XEONTEMP_REMOTE_TEMP, RW }, 
                                           XEONTEMP_SYSCTL_REMOTE_TEMP, VALUE(2), 
                                           0 },
    { { SENSORS_XEONTEMP_REMOTE_TEMP_OVER, "temp_over", 
                                           SENSORS_XEONTEMP_REMOTE_TEMP, 
                                           SENSORS_XEONTEMP_REMOTE_TEMP, RW }, 
                                           XEONTEMP_SYSCTL_REMOTE_TEMP, VALUE(1), 
                                           0 },
    { { SENSORS_XEONTEMP_ALARMS, "alarms", NOMAP, NOMAP, R }, 
                                 XEONTEMP_SYSCTL_ALARMS, VALUE(1), 0 },
    { { 0 }, 0 }
  };

static sensors_chip_feature max1619_features[] =
  {
    { { SENSORS_MAX1619_LOCAL_TEMP, "temp1", NOMAP, NOMAP, R }, 
                                    MAX1619_SYSCTL_LOCAL_TEMP, VALUE(1), 0 },
    { { SENSORS_MAX1619_REMOTE_TEMP, "temp2", NOMAP, NOMAP, R }, 
                                     MAX1619_SYSCTL_REMOTE_TEMP, VALUE(3), 0 },
    { { SENSORS_MAX1619_REMOTE_LOW, "temp2_min", SENSORS_MAX1619_REMOTE_TEMP,
                                    SENSORS_MAX1619_REMOTE_TEMP, RW }, 
                                    MAX1619_SYSCTL_REMOTE_TEMP, VALUE(2), 0 },
    { { SENSORS_MAX1619_REMOTE_HIGH, "temp2_max", SENSORS_MAX1619_REMOTE_TEMP,
                                     SENSORS_MAX1619_REMOTE_TEMP, RW }, 
                                     MAX1619_SYSCTL_REMOTE_TEMP, VALUE(1), 0 },
    { { SENSORS_MAX1619_REMOTE_MAX, "temp2_crit", SENSORS_MAX1619_REMOTE_TEMP,
                                    SENSORS_MAX1619_REMOTE_TEMP, RW }, 
                                    MAX1619_SYSCTL_REMOTE_CRIT, VALUE(1), 0 },
    { { SENSORS_MAX1619_REMOTE_HYST, "temp2_hyst", SENSORS_MAX1619_REMOTE_TEMP,
                                     SENSORS_MAX1619_REMOTE_TEMP, RW }, 
                                     MAX1619_SYSCTL_REMOTE_CRIT, VALUE(2), 0, 
                                     "temp2_crit_hyst", 3 },
    { { SENSORS_MAX1619_ALARMS, "alarms", NOMAP, NOMAP, R }, 
                                MAX1619_SYSCTL_ALARMS, VALUE(1), 0 },
    { { 0 }, 0 }
  };

static sensors_chip_feature max6650_features[] =
  {
    { { SENSORS_MAX6650_FAN1_TACH, "fan1", NOMAP, NOMAP, R }, 
                                   MAX6650_SYSCTL_FAN1, VALUE(1), 0 },
    { { SENSORS_MAX6650_FAN2_TACH, "fan2", NOMAP, NOMAP, R }, 
                                   MAX6650_SYSCTL_FAN2, VALUE(1), 0 },
    { { SENSORS_MAX6650_FAN3_TACH, "fan3", NOMAP, NOMAP, R }, 
                                   MAX6650_SYSCTL_FAN3, VALUE(1), 0 },
    { { SENSORS_MAX6650_FAN4_TACH, "fan4", NOMAP, NOMAP, R }, 
                                   MAX6650_SYSCTL_FAN4, VALUE(1), 0 },
    { { SENSORS_MAX6650_SPEED, "speed", SENSORS_MAX6650_FAN1_TACH,
                               SENSORS_MAX6650_FAN1_TACH, RW }, 
                               MAX6650_SYSCTL_SPEED, VALUE(1), 0,
                               "fan1_target", 0 },
    { { 0 }, 0 }
  };

#define SENSORS_SMSC47B397_TEMP(nr) \
	{ { SENSORS_SMSC47B397_TEMP##nr, "temp" #nr, NOMAP, NOMAP, R }, \
	 	NOSYSCTL, VALUE(1), 0 }

#define SENSORS_SMSC47B397_FAN(nr) \
	{ { SENSORS_SMSC47B397_FAN##nr, "fan" #nr, NOMAP, NOMAP, R }, \
		NOSYSCTL, VALUE(1), 0 }

static sensors_chip_feature smsc47b397_features[] =
{
	SENSORS_SMSC47B397_TEMP(1),
	SENSORS_SMSC47B397_TEMP(2),
	SENSORS_SMSC47B397_TEMP(3),
	SENSORS_SMSC47B397_TEMP(4),
	SENSORS_SMSC47B397_FAN(1),
	SENSORS_SMSC47B397_FAN(2),
	SENSORS_SMSC47B397_FAN(3),
	SENSORS_SMSC47B397_FAN(4),
    { { 0 }, 0 }
};

static sensors_chip_feature f71805f_features[] =
{
/* 9 to 11 voltage inputs */
    { { SENSORS_F71805F_IN(0), "in0", NOMAP, NOMAP, R }, 
                            F71805F_SYSCTL_IN0, VALUE(3), 3 },
    { { SENSORS_F71805F_IN(1), "in1", NOMAP, NOMAP, R }, 
                            F71805F_SYSCTL_IN1, VALUE(3), 3 },
    { { SENSORS_F71805F_IN(2), "in2", NOMAP, NOMAP, R }, 
                            F71805F_SYSCTL_IN2, VALUE(3), 3 },
    { { SENSORS_F71805F_IN(3), "in3", NOMAP, NOMAP, R }, 
                            F71805F_SYSCTL_IN3, VALUE(3), 3 },
    { { SENSORS_F71805F_IN(4), "in4", NOMAP, NOMAP, R }, 
                            F71805F_SYSCTL_IN4, VALUE(3), 3 },
    { { SENSORS_F71805F_IN(5), "in5", NOMAP, NOMAP, R }, 
                            F71805F_SYSCTL_IN5, VALUE(3), 3 },
    { { SENSORS_F71805F_IN(6), "in6", NOMAP, NOMAP, R }, 
                            F71805F_SYSCTL_IN6, VALUE(3), 3 },
    { { SENSORS_F71805F_IN(7), "in7", NOMAP, NOMAP, R }, 
                            F71805F_SYSCTL_IN7, VALUE(3), 3 },
    { { SENSORS_F71805F_IN(8), "in8", NOMAP, NOMAP, R }, 
                            F71805F_SYSCTL_IN8, VALUE(3), 3 },
    { { SENSORS_F71805F_IN(9), "in9", NOMAP, NOMAP, R }, 
                            NOSYSCTL, VALUE(3), 3 },
    { { SENSORS_F71805F_IN(10), "in10", NOMAP, NOMAP, R }, 
                             NOSYSCTL, VALUE(3), 3 },
    { { SENSORS_F71805F_IN_MIN(0), "in0_min", SENSORS_F71805F_IN(0), 
                                SENSORS_F71805F_IN(0), RW }, 
                                F71805F_SYSCTL_IN0, VALUE(1), 3 },
    { { SENSORS_F71805F_IN_MIN(1), "in1_min", SENSORS_F71805F_IN(1), 
                                SENSORS_F71805F_IN(1), RW }, 
                                F71805F_SYSCTL_IN1, VALUE(1), 3 },
    { { SENSORS_F71805F_IN_MIN(2), "in2_min", SENSORS_F71805F_IN(2), 
                                SENSORS_F71805F_IN(2), RW }, 
                                F71805F_SYSCTL_IN2, VALUE(1), 3 },
    { { SENSORS_F71805F_IN_MIN(3), "in3_min", SENSORS_F71805F_IN(3), 
                                SENSORS_F71805F_IN(3), RW }, 
                                F71805F_SYSCTL_IN3, VALUE(1), 3 },
    { { SENSORS_F71805F_IN_MIN(4), "in4_min", SENSORS_F71805F_IN(4), 
                                SENSORS_F71805F_IN(4), RW }, 
                                F71805F_SYSCTL_IN4, VALUE(1), 3 },
    { { SENSORS_F71805F_IN_MIN(5), "in5_min", SENSORS_F71805F_IN(5), 
                                SENSORS_F71805F_IN(5), RW }, 
                                F71805F_SYSCTL_IN5, VALUE(1), 3 },
    { { SENSORS_F71805F_IN_MIN(6), "in6_min", SENSORS_F71805F_IN(6), 
                                SENSORS_F71805F_IN(6), RW }, 
                                F71805F_SYSCTL_IN6, VALUE(1), 3 },
    { { SENSORS_F71805F_IN_MIN(7), "in7_min", SENSORS_F71805F_IN(7), 
                                SENSORS_F71805F_IN(7), RW }, 
                                F71805F_SYSCTL_IN7, VALUE(1), 3 },
    { { SENSORS_F71805F_IN_MIN(8), "in8_min", SENSORS_F71805F_IN(8), 
                                SENSORS_F71805F_IN(8), RW }, 
                                F71805F_SYSCTL_IN8, VALUE(1), 3 },
    { { SENSORS_F71805F_IN_MIN(9), "in9_min", SENSORS_F71805F_IN(9), 
                                SENSORS_F71805F_IN(9), RW }, 
                                NOSYSCTL, VALUE(1), 3 },
    { { SENSORS_F71805F_IN_MIN(10), "in10_min", SENSORS_F71805F_IN(10), 
                                 SENSORS_F71805F_IN(10), RW }, 
                                 NOSYSCTL, VALUE(1), 3 },
    { { SENSORS_F71805F_IN_MAX(0), "in0_max", SENSORS_F71805F_IN(0), 
                                SENSORS_F71805F_IN(0), RW }, 
                                F71805F_SYSCTL_IN0, VALUE(2), 3 },
    { { SENSORS_F71805F_IN_MAX(1), "in1_max", SENSORS_F71805F_IN(1), 
                                SENSORS_F71805F_IN(1), RW }, 
                                F71805F_SYSCTL_IN1, VALUE(2), 3 },
    { { SENSORS_F71805F_IN_MAX(2), "in2_max", SENSORS_F71805F_IN(2), 
                                SENSORS_F71805F_IN(2), RW }, 
                                F71805F_SYSCTL_IN2, VALUE(2), 3 },
    { { SENSORS_F71805F_IN_MAX(3), "in3_max", SENSORS_F71805F_IN(3), 
                                SENSORS_F71805F_IN(3), RW }, 
                                F71805F_SYSCTL_IN3, VALUE(2), 3 },
    { { SENSORS_F71805F_IN_MAX(4), "in4_max", SENSORS_F71805F_IN(4), 
                                SENSORS_F71805F_IN(4), RW }, 
                                F71805F_SYSCTL_IN4, VALUE(2), 3 },
    { { SENSORS_F71805F_IN_MAX(5), "in5_max", SENSORS_F71805F_IN(5), 
                                SENSORS_F71805F_IN(5), RW }, 
                                F71805F_SYSCTL_IN5, VALUE(2), 3 },
    { { SENSORS_F71805F_IN_MAX(6), "in6_max", SENSORS_F71805F_IN(6), 
                                SENSORS_F71805F_IN(6), RW }, 
                                F71805F_SYSCTL_IN6, VALUE(2), 3 },
    { { SENSORS_F71805F_IN_MAX(7), "in7_max", SENSORS_F71805F_IN(7), 
                                SENSORS_F71805F_IN(7), RW }, 
                                F71805F_SYSCTL_IN7, VALUE(2), 3 },
    { { SENSORS_F71805F_IN_MAX(8), "in8_max", SENSORS_F71805F_IN(8), 
                                SENSORS_F71805F_IN(8), RW }, 
                                F71805F_SYSCTL_IN8, VALUE(2), 3 },
    { { SENSORS_F71805F_IN_MAX(9), "in9_max", SENSORS_F71805F_IN(9), 
                                SENSORS_F71805F_IN(9), RW }, 
                                NOSYSCTL, VALUE(2), 3 },
    { { SENSORS_F71805F_IN_MAX(10), "in10_max", SENSORS_F71805F_IN(10), 
                                 SENSORS_F71805F_IN(10), RW }, 
                                 NOSYSCTL, VALUE(2), 3 },
/* 3 fan tachometers */
    { { SENSORS_F71805F_FAN(1), "fan1", NOMAP, NOMAP, R }, 
                             F71805F_SYSCTL_FAN1, VALUE(2), 0 },
    { { SENSORS_F71805F_FAN(2), "fan2", NOMAP, NOMAP, R }, 
                             F71805F_SYSCTL_FAN2, VALUE(2), 0 },
    { { SENSORS_F71805F_FAN(3), "fan3", NOMAP, NOMAP, R }, 
                             F71805F_SYSCTL_FAN3, VALUE(2), 0 },
    { { SENSORS_F71805F_FAN_MIN(1), "fan1_min", SENSORS_F71805F_FAN(1), 
                                 SENSORS_F71805F_FAN(1), RW }, 
                                 F71805F_SYSCTL_FAN1, VALUE(1), 0 },
    { { SENSORS_F71805F_FAN_MIN(2), "fan2_min", SENSORS_F71805F_FAN(2), 
                                 SENSORS_F71805F_FAN(2), RW }, 
                                 F71805F_SYSCTL_FAN2, VALUE(1), 0 },
    { { SENSORS_F71805F_FAN_MIN(3), "fan3_min", SENSORS_F71805F_FAN(3), 
                                 SENSORS_F71805F_FAN(3), RW }, 
                                 F71805F_SYSCTL_FAN3, VALUE(1), 0 },
/* 3 temperature inputs */
    { { SENSORS_F71805F_TEMP(1), "temp1", NOMAP, NOMAP, R }, 
                              F71805F_SYSCTL_TEMP1, VALUE(3), 0 },
    { { SENSORS_F71805F_TEMP(2), "temp2", NOMAP, NOMAP, R }, 
                              F71805F_SYSCTL_TEMP2, VALUE(3), 0 },
    { { SENSORS_F71805F_TEMP(3), "temp3", NOMAP, NOMAP, R }, 
                              F71805F_SYSCTL_TEMP3, VALUE(3), 0 },
    { { SENSORS_F71805F_TEMP_MAX(1), "temp1_max", SENSORS_F71805F_TEMP(1), 
                                  SENSORS_F71805F_TEMP(1), RW }, 
                                  F71805F_SYSCTL_TEMP1, VALUE(1), 0 },
    { { SENSORS_F71805F_TEMP_MAX(2), "temp2_max", SENSORS_F71805F_TEMP(2), 
                                  SENSORS_F71805F_TEMP(2), RW }, 
                                  F71805F_SYSCTL_TEMP2, VALUE(1), 0 },
    { { SENSORS_F71805F_TEMP_MAX(3), "temp3_max", SENSORS_F71805F_TEMP(3), 
                                  SENSORS_F71805F_TEMP(3), RW }, 
                                  F71805F_SYSCTL_TEMP3, VALUE(1), 0 },
    { { SENSORS_F71805F_TEMP_HYST(1), "temp1_hyst", SENSORS_F71805F_TEMP(1), 
                                   SENSORS_F71805F_TEMP(1), RW }, 
                                   F71805F_SYSCTL_TEMP1, VALUE(2), 0 },
    { { SENSORS_F71805F_TEMP_HYST(2), "temp2_hyst", SENSORS_F71805F_TEMP(2), 
                                   SENSORS_F71805F_TEMP(2), RW }, 
                                   F71805F_SYSCTL_TEMP2, VALUE(2), 0 },
    { { SENSORS_F71805F_TEMP_HYST(3), "temp3_hyst", SENSORS_F71805F_TEMP(3), 
                                   SENSORS_F71805F_TEMP(3), RW }, 
                                   F71805F_SYSCTL_TEMP3, VALUE(2), 0 },
    { { SENSORS_F71805F_TEMP_TYPE(1), "sensor1", SENSORS_F71805F_TEMP(1), NOMAP, R }, 
                                   F71805F_SYSCTL_SENSOR1, VALUE(1), 0 },
    { { SENSORS_F71805F_TEMP_TYPE(2), "sensor2", SENSORS_F71805F_TEMP(2), NOMAP, R }, 
                                   F71805F_SYSCTL_SENSOR2, VALUE(1), 0 },
    { { SENSORS_F71805F_TEMP_TYPE(3), "sensor3", SENSORS_F71805F_TEMP(3), NOMAP, R }, 
                                   F71805F_SYSCTL_SENSOR3, VALUE(1), 0 },
/* 3 alarm bitvectors */
    { { SENSORS_F71805F_ALARMS_IN, "alarms_in", NOMAP, NOMAP, R }, 
                                F71805F_SYSCTL_ALARMS_IN, VALUE(1), 0 },
    { { SENSORS_F71805F_ALARMS_FAN, "alarms_fan", NOMAP, NOMAP, R }, 
                                 F71805F_SYSCTL_ALARMS_FAN, VALUE(1), 0 },
    { { SENSORS_F71805F_ALARMS_TEMP, "alarms_temp", NOMAP, NOMAP, R }, 
                                  F71805F_SYSCTL_ALARMS_TEMP, VALUE(1), 0 },
    { { 0 }, 0 }
};

#define SENSORS_ABITUGURU_IN_FEATURES(nr) \
	{ { SENSORS_ABITUGURU_IN(nr), "in" #nr, NOMAP, NOMAP, R }, \
		NOSYSCTL, VALUE(3), 3 }, \
	{ { SENSORS_ABITUGURU_IN_MIN(nr), "in" #nr "_min", \
		SENSORS_ABITUGURU_IN(nr), SENSORS_ABITUGURU_IN(nr), RW }, \
		NOSYSCTL, VALUE(1), 3 }, \
	{ { SENSORS_ABITUGURU_IN_MIN_ALARM(nr), "in" #nr "_min_alarm", \
		SENSORS_ABITUGURU_IN(nr), NOMAP, R }, \
		NOSYSCTL, VALUE(1), 0 }, \
	{ { SENSORS_ABITUGURU_IN_MAX(nr), "in" #nr "_max", \
		SENSORS_ABITUGURU_IN(nr), SENSORS_ABITUGURU_IN(nr), RW }, \
		NOSYSCTL, VALUE(2), 3 }, \
	{ { SENSORS_ABITUGURU_IN_MAX_ALARM(nr), "in" #nr "_max_alarm", \
		SENSORS_ABITUGURU_IN(nr), NOMAP, R }, \
		NOSYSCTL, VALUE(2), 0 }

#define SENSORS_ABITUGURU_TEMP_FEATURES(nr) \
	{ { SENSORS_ABITUGURU_TEMP(nr), "temp" #nr, NOMAP, NOMAP, R }, \
		NOSYSCTL, VALUE(3), 3 }, \
	{ { SENSORS_ABITUGURU_TEMP_ALARM(nr), "temp" #nr "_alarm", \
		SENSORS_ABITUGURU_TEMP(nr), NOMAP, R }, \
		NOSYSCTL, VALUE(1), 0 }, \
	{ { SENSORS_ABITUGURU_TEMP_MAX(nr), "temp" #nr "_max", \
		SENSORS_ABITUGURU_TEMP(nr), SENSORS_ABITUGURU_TEMP(nr), RW }, \
		NOSYSCTL, VALUE(1), 3 }, \
	{ { SENSORS_ABITUGURU_TEMP_CRIT(nr), "temp" #nr "_crit", \
		SENSORS_ABITUGURU_TEMP(nr), SENSORS_ABITUGURU_TEMP(nr), RW }, \
		NOSYSCTL, VALUE(2), 3 }

#define SENSORS_ABITUGURU_FAN_FEATURES(nr) \
	{ { SENSORS_ABITUGURU_FAN(nr), "fan" #nr, NOMAP, NOMAP, R }, \
		NOSYSCTL, VALUE(2), 0 }, \
	{ { SENSORS_ABITUGURU_FAN_ALARM(nr), "fan" #nr "_alarm", \
		SENSORS_ABITUGURU_FAN(nr), NOMAP, R }, \
		NOSYSCTL, VALUE(1), 0 }, \
	{ { SENSORS_ABITUGURU_FAN_MIN(nr), "fan" #nr "_min", \
		SENSORS_ABITUGURU_FAN(nr), SENSORS_ABITUGURU_FAN(nr), RW }, \
		NOSYSCTL, VALUE(1), 0 }

static sensors_chip_feature abituguru_features[] =
{
	SENSORS_ABITUGURU_IN_FEATURES(0),
	SENSORS_ABITUGURU_IN_FEATURES(1),
	SENSORS_ABITUGURU_IN_FEATURES(2),
	SENSORS_ABITUGURU_IN_FEATURES(3),
	SENSORS_ABITUGURU_IN_FEATURES(4),
	SENSORS_ABITUGURU_IN_FEATURES(5),
	SENSORS_ABITUGURU_IN_FEATURES(6),
	SENSORS_ABITUGURU_IN_FEATURES(7),
	SENSORS_ABITUGURU_IN_FEATURES(8),
	SENSORS_ABITUGURU_IN_FEATURES(9),
	SENSORS_ABITUGURU_IN_FEATURES(10),
	SENSORS_ABITUGURU_TEMP_FEATURES(1),
	SENSORS_ABITUGURU_TEMP_FEATURES(2),
	SENSORS_ABITUGURU_TEMP_FEATURES(3),
	SENSORS_ABITUGURU_TEMP_FEATURES(4),
	SENSORS_ABITUGURU_TEMP_FEATURES(5),
	SENSORS_ABITUGURU_TEMP_FEATURES(6),
	SENSORS_ABITUGURU_TEMP_FEATURES(7),
	SENSORS_ABITUGURU_FAN_FEATURES(1),
	SENSORS_ABITUGURU_FAN_FEATURES(2),
	SENSORS_ABITUGURU_FAN_FEATURES(3),
	SENSORS_ABITUGURU_FAN_FEATURES(4),
	SENSORS_ABITUGURU_FAN_FEATURES(5),
	SENSORS_ABITUGURU_FAN_FEATURES(6),
	{ { 0 }, 0 }
};

static sensors_chip_feature abituguru3_features[] =
{
	SENSORS_ABITUGURU_IN_FEATURES(0),
	SENSORS_ABITUGURU_IN_FEATURES(1),
	SENSORS_ABITUGURU_IN_FEATURES(2),
	SENSORS_ABITUGURU_IN_FEATURES(3),
	SENSORS_ABITUGURU_IN_FEATURES(4),
	SENSORS_ABITUGURU_IN_FEATURES(5),
	SENSORS_ABITUGURU_IN_FEATURES(6),
	SENSORS_ABITUGURU_IN_FEATURES(7),
	SENSORS_ABITUGURU_IN_FEATURES(8),
	SENSORS_ABITUGURU_IN_FEATURES(9),
	SENSORS_ABITUGURU_IN_FEATURES(10),
	SENSORS_ABITUGURU_IN_FEATURES(11),
	SENSORS_ABITUGURU_IN_FEATURES(12),
	SENSORS_ABITUGURU_IN_FEATURES(13),
	SENSORS_ABITUGURU_TEMP_FEATURES(1),
	SENSORS_ABITUGURU_TEMP_FEATURES(2),
	SENSORS_ABITUGURU_TEMP_FEATURES(3),
	SENSORS_ABITUGURU_TEMP_FEATURES(4),
	SENSORS_ABITUGURU_TEMP_FEATURES(5),
	SENSORS_ABITUGURU_TEMP_FEATURES(6),
	SENSORS_ABITUGURU_TEMP_FEATURES(7),
	SENSORS_ABITUGURU_FAN_FEATURES(1),
	SENSORS_ABITUGURU_FAN_FEATURES(2),
	SENSORS_ABITUGURU_FAN_FEATURES(3),
	SENSORS_ABITUGURU_FAN_FEATURES(4),
	SENSORS_ABITUGURU_FAN_FEATURES(5),
	SENSORS_ABITUGURU_FAN_FEATURES(6),
	SENSORS_ABITUGURU_FAN_FEATURES(7),
	SENSORS_ABITUGURU_FAN_FEATURES(8),
	{ { 0 }, 0 }
};


static sensors_chip_feature k8temp_features[] =
  {
    { { SENSORS_K8TEMP_TEMP1, "temp1", NOMAP, NOMAP, R }, 
                              NOSYSCTL, VALUE(1), 0 },
    { { SENSORS_K8TEMP_TEMP2, "temp2", NOMAP, NOMAP, R }, 
                              NOSYSCTL, VALUE(1), 0 },
    { { SENSORS_K8TEMP_TEMP3, "temp3", NOMAP, NOMAP, R }, 
                              NOSYSCTL, VALUE(1), 0 },
    { { SENSORS_K8TEMP_TEMP4, "temp4", NOMAP, NOMAP, R }, 
                              NOSYSCTL, VALUE(1), 0 },
    { { 0 }, 0 }
  };

static sensors_chip_feature coretemp_features[] =
  {
    { { SENSORS_CORETEMP_TEMP1, "temp1", NOMAP, NOMAP, R }, 
                                NOSYSCTL, VALUE(2), 0 },
    { { SENSORS_CORETEMP_TEMP1_CRIT, "temp1_crit", SENSORS_CORETEMP_TEMP1, 
                                     SENSORS_CORETEMP_TEMP1, R }, 
                                     NOSYSCTL, VALUE(1), 0 },
    { { SENSORS_CORETEMP_TEMP1_CRIT_ALARM, "temp1_crit_alarm", 
                                           SENSORS_CORETEMP_TEMP1, NOMAP, R }, 
                                           NOSYSCTL, VALUE(1), 0 },
    { { 0 }, 0 }
  };

#define SENSORS_DME1737_IN_FEATURES(nr) \
	{ { SENSORS_DME1737_IN(nr), "in" #nr, \
		NOMAP, NOMAP, R }, \
		NOSYSCTL, VALUE(3), 3 }, \
	{ { SENSORS_DME1737_IN_MIN(nr), "in" #nr "_min", \
		SENSORS_DME1737_IN(nr), SENSORS_DME1737_IN(nr), RW }, \
		NOSYSCTL, VALUE(1), 3 }, \
	{ { SENSORS_DME1737_IN_MAX(nr), "in" #nr "_max", \
		SENSORS_DME1737_IN(nr), SENSORS_DME1737_IN(nr), RW }, \
		NOSYSCTL, VALUE(2), 3 }, \
	{ { SENSORS_DME1737_IN_ALARM(nr), "in" #nr "_alarm", \
		SENSORS_DME1737_IN(nr), NOMAP, R }, \
		NOSYSCTL, VALUE(1), 0 }

#define SENSORS_DME1737_TEMP_FEATURES(nr) \
	{ { SENSORS_DME1737_TEMP(nr), "temp" #nr, \
		NOMAP, NOMAP, R }, \
		NOSYSCTL, VALUE(3), 3 }, \
	{ { SENSORS_DME1737_TEMP_MIN(nr), "temp" #nr "_min", \
		SENSORS_DME1737_TEMP(nr), SENSORS_DME1737_TEMP(nr), RW }, \
		NOSYSCTL, VALUE(2), 3 }, \
	{ { SENSORS_DME1737_TEMP_MAX(nr), "temp" #nr "_max", \
		SENSORS_DME1737_TEMP(nr), SENSORS_DME1737_TEMP(nr), RW }, \
		NOSYSCTL, VALUE(1), 3 }, \
	{ { SENSORS_DME1737_TEMP_ALARM(nr), "temp" #nr "_alarm", \
		SENSORS_DME1737_TEMP(nr), NOMAP, R }, \
		NOSYSCTL, VALUE(1), 0 }, \
	{ { SENSORS_DME1737_TEMP_FAULT(nr), "temp" #nr "_fault", \
		SENSORS_DME1737_TEMP(nr), NOMAP, R }, \
		NOSYSCTL, VALUE(2), 0 }

#define SENSORS_DME1737_FAN_FEATURES(nr) \
	{ { SENSORS_DME1737_FAN(nr), "fan" #nr, \
		NOMAP, NOMAP, R }, \
		NOSYSCTL, VALUE(2), 0 }, \
	{ { SENSORS_DME1737_FAN_MIN(nr), "fan" #nr "_min", \
		SENSORS_DME1737_FAN(nr), SENSORS_DME1737_FAN(nr), RW }, \
		NOSYSCTL, VALUE(1), 0 }, \
	{ { SENSORS_DME1737_FAN_ALARM(nr), "fan" #nr "_alarm", \
		SENSORS_DME1737_FAN(nr), NOMAP, R }, \
		NOSYSCTL, VALUE(1), 0 }

#define SENSORS_DME1737_PWM_FEATURES(nr) \
	{ { SENSORS_DME1737_PWM(nr), "pwm" #nr, \
		NOMAP, NOMAP, RW }, \
		NOSYSCTL, VALUE(1), 0 }, \
	{ { SENSORS_DME1737_PWM_ENABLE(nr), "pwm" #nr "_enable", \
		SENSORS_DME1737_PWM(nr), SENSORS_DME1737_PWM(nr), RW }, \
		NOSYSCTL, VALUE(2), 0 }, \
	{ { SENSORS_DME1737_PWM_FREQ(nr), "pwm" #nr "_freq", \
		SENSORS_DME1737_PWM(nr), SENSORS_DME1737_PWM(nr), RW }, \
		NOSYSCTL, VALUE(3), 0 }

static sensors_chip_feature dme1737_features[] =
{
	SENSORS_DME1737_IN_FEATURES(0),
	SENSORS_DME1737_IN_FEATURES(1),
	SENSORS_DME1737_IN_FEATURES(2),
	SENSORS_DME1737_IN_FEATURES(3),
	SENSORS_DME1737_IN_FEATURES(4),
	SENSORS_DME1737_IN_FEATURES(5),
	SENSORS_DME1737_IN_FEATURES(6),
	SENSORS_DME1737_TEMP_FEATURES(1),
	SENSORS_DME1737_TEMP_FEATURES(2),
	SENSORS_DME1737_TEMP_FEATURES(3),
	SENSORS_DME1737_FAN_FEATURES(1),
	SENSORS_DME1737_FAN_FEATURES(2),
	SENSORS_DME1737_FAN_FEATURES(3),
	SENSORS_DME1737_FAN_FEATURES(4),
	SENSORS_DME1737_FAN_FEATURES(5),
	SENSORS_DME1737_FAN_FEATURES(6),
	SENSORS_DME1737_PWM_FEATURES(1),
	SENSORS_DME1737_PWM_FEATURES(2),
	SENSORS_DME1737_PWM_FEATURES(3),
	SENSORS_DME1737_PWM_FEATURES(5),
	SENSORS_DME1737_PWM_FEATURES(6),
	{ { SENSORS_DME1737_VID, "cpu0_vid", NOMAP, NOMAP, R },
		NOSYSCTL, VALUE(1), 3 },
	{ { SENSORS_DME1737_VRM, "vrm", NOMAP, NOMAP, RW },
		NOSYSCTL, VALUE(1), 1 },
	{ { 0 }, 0 }
};

static sensors_chip_feature applesmc_features[] =
  {
    { { SENSORS_APPLESMC_TEMP(0), "temp1", NOMAP, NOMAP, R }, 
                                NOSYSCTL, VALUE(1), 3 },
    { { SENSORS_APPLESMC_TEMP(1), "temp2", NOMAP, NOMAP, R }, 
                                NOSYSCTL, VALUE(1), 3 },
    { { SENSORS_APPLESMC_TEMP(2), "temp3", NOMAP, NOMAP, R }, 
                                NOSYSCTL, VALUE(1), 3 },
    { { SENSORS_APPLESMC_TEMP(3), "temp4", NOMAP, NOMAP, R }, 
                                NOSYSCTL, VALUE(1), 3 },
    { { SENSORS_APPLESMC_TEMP(4), "temp5", NOMAP, NOMAP, R }, 
                                NOSYSCTL, VALUE(1), 3 },
    { { SENSORS_APPLESMC_TEMP(5), "temp6", NOMAP, NOMAP, R }, 
                                NOSYSCTL, VALUE(1), 3 },
    { { SENSORS_APPLESMC_TEMP(6), "temp7", NOMAP, NOMAP, R }, 
                                NOSYSCTL, VALUE(1), 3 },
    { { SENSORS_APPLESMC_TEMP(7), "temp8", NOMAP, NOMAP, R }, 
                                NOSYSCTL, VALUE(1), 3 },
    { { SENSORS_APPLESMC_TEMP(8), "temp9", NOMAP, NOMAP, R }, 
                                NOSYSCTL, VALUE(1), 3 },
    { { SENSORS_APPLESMC_TEMP(9), "temp10", NOMAP, NOMAP, R }, 
                                NOSYSCTL, VALUE(1), 3 },
    { { SENSORS_APPLESMC_TEMP(10), "temp11", NOMAP, NOMAP, R }, 
                                NOSYSCTL, VALUE(1), 3 },
    { { SENSORS_APPLESMC_TEMP(11), "temp12", NOMAP, NOMAP, R }, 
                                NOSYSCTL, VALUE(1), 3 },
    { { SENSORS_APPLESMC_FAN(0), "fan1", NOMAP, NOMAP, R }, 
                                NOSYSCTL, VALUE(2), 0 },
    { { SENSORS_APPLESMC_FAN(1), "fan2", NOMAP, NOMAP, R }, 
                                NOSYSCTL, VALUE(2), 0 },
    { { SENSORS_APPLESMC_FAN_MIN(0), "fan1_min", SENSORS_APPLESMC_FAN(0), 
		  SENSORS_APPLESMC_FAN(0), R }, NOSYSCTL, VALUE(1), 0 },
    { { SENSORS_APPLESMC_FAN_MIN(1), "fan2_min", SENSORS_APPLESMC_FAN(1),
		  SENSORS_APPLESMC_FAN(1), R }, NOSYSCTL, VALUE(1), 0 },
    { { SENSORS_APPLESMC_FAN_MAX(0), "fan1_max", SENSORS_APPLESMC_FAN(0), 
		  SENSORS_APPLESMC_FAN(0), R }, NOSYSCTL, VALUE(1), 0 },
    { { SENSORS_APPLESMC_FAN_MAX(1), "fan2_max", SENSORS_APPLESMC_FAN(1),
		  SENSORS_APPLESMC_FAN(1), R }, NOSYSCTL, VALUE(1), 0 },
    { { SENSORS_APPLESMC_FAN_SAFE(0), "fan1_safe", SENSORS_APPLESMC_FAN(0),
		  SENSORS_APPLESMC_FAN(0), R }, NOSYSCTL, VALUE(1), 0 },
    { { SENSORS_APPLESMC_FAN_SAFE(1), "fan2_safe", SENSORS_APPLESMC_FAN(1),
		  SENSORS_APPLESMC_FAN(1), R }, NOSYSCTL, VALUE(1), 0 },
    { { 0 }, 0 }
  };

#define SENSORS_F71882FG_IN_FEATURES(nr) \
	{ { SENSORS_F71882FG_IN(nr), "in" #nr, \
		NOMAP, NOMAP, R }, \
		NOSYSCTL, VALUE(3), 3 }

#define SENSORS_F71882FG_TEMP_FEATURES(nr) \
	{ { SENSORS_F71882FG_TEMP(nr), "temp" #nr, \
		NOMAP, NOMAP, R }, \
		NOSYSCTL, VALUE(3), 3 }, \
	{ { SENSORS_F71882FG_TEMP_MAX(nr), "temp" #nr "_max", \
		SENSORS_F71882FG_TEMP(nr), SENSORS_F71882FG_TEMP(nr), RW }, \
		NOSYSCTL, VALUE(1), 3 }, \
	{ { SENSORS_F71882FG_TEMP_MAX_HYST(nr), "temp" #nr "_max_hyst", \
		SENSORS_F71882FG_TEMP(nr), SENSORS_F71882FG_TEMP(nr), RW }, \
		NOSYSCTL, VALUE(2), 3 }, \
	{ { SENSORS_F71882FG_TEMP_CRIT(nr), "temp" #nr "_crit", \
		SENSORS_F71882FG_TEMP(nr), SENSORS_F71882FG_TEMP(nr), RW }, \
		NOSYSCTL, VALUE(4), 3 }, \
	{ { SENSORS_F71882FG_TEMP_CRIT_HYST(nr), "temp" #nr "_crit_hyst", \
		SENSORS_F71882FG_TEMP(nr), SENSORS_F71882FG_TEMP(nr), R }, \
		NOSYSCTL, VALUE(5), 3 }, \
	{ { SENSORS_F71882FG_TEMP_ALARM(nr), "temp" #nr "_alarm", \
		SENSORS_F71882FG_TEMP(nr), NOMAP, R }, \
		NOSYSCTL, VALUE(1), 0 }, \
	{ { SENSORS_F71882FG_TEMP_FAULT(nr), "temp" #nr "_fault", \
		SENSORS_F71882FG_TEMP(nr), NOMAP, R }, \
		NOSYSCTL, VALUE(2), 0 }, \
	{ { SENSORS_F71882FG_TEMP_TYPE(nr), "temp" #nr "_type", \
		SENSORS_F71882FG_TEMP(nr), NOMAP, R }, \
		NOSYSCTL, VALUE(3), 0 }

#define SENSORS_F71882FG_FAN_FEATURES(nr) \
	{ { SENSORS_F71882FG_FAN(nr), "fan" #nr, \
		NOMAP, NOMAP, R }, \
		NOSYSCTL, VALUE(2), 0 }, \
	{ { SENSORS_F71882FG_FAN_ALARM(nr), "fan" #nr "_alarm", \
		SENSORS_F71882FG_FAN(nr), NOMAP, R }, \
		NOSYSCTL, VALUE(1), 0 }

static sensors_chip_feature f71882fg_features[] =
{
	SENSORS_F71882FG_IN_FEATURES(0),
	SENSORS_F71882FG_IN_FEATURES(1),
	{ { SENSORS_F71882FG_IN_MAX(1), "in1_max",
		SENSORS_F71882FG_IN(1), SENSORS_F71882FG_IN(1), RW },
		NOSYSCTL, VALUE(1), 3 },
	{ { SENSORS_F71882FG_IN_ALARM(1), "in1_alarm",
		SENSORS_F71882FG_IN(1), NOMAP, R },
		NOSYSCTL, VALUE(1), 0 },
	SENSORS_F71882FG_IN_FEATURES(2),
	SENSORS_F71882FG_IN_FEATURES(3),
	SENSORS_F71882FG_IN_FEATURES(4),
	SENSORS_F71882FG_IN_FEATURES(5),
	SENSORS_F71882FG_IN_FEATURES(6),
	SENSORS_F71882FG_IN_FEATURES(7),
	SENSORS_F71882FG_IN_FEATURES(8),
	SENSORS_F71882FG_TEMP_FEATURES(1),
	SENSORS_F71882FG_TEMP_FEATURES(2),
	SENSORS_F71882FG_TEMP_FEATURES(3),
	SENSORS_F71882FG_FAN_FEATURES(1),
	SENSORS_F71882FG_FAN_FEATURES(2),
	SENSORS_F71882FG_FAN_FEATURES(3),
	SENSORS_F71882FG_FAN_FEATURES(4),
	{ { 0 }, 0 }
};

#define SENSORS_FSCHMD_IN_FEATURES(nr) \
	{ { SENSORS_FSCHMD_IN(nr), "in" #nr, \
		NOMAP, NOMAP, R }, \
		NOSYSCTL, VALUE(1), 3 }

#define SENSORS_FSCHMD_TEMP_FEATURES(nr) \
	{ { SENSORS_FSCHMD_TEMP(nr), "temp" #nr, \
		NOMAP, NOMAP, R }, \
		NOSYSCTL, VALUE(2), 3 }, \
	{ { SENSORS_FSCHMD_TEMP_MAX(nr), "temp" #nr "_max", \
		SENSORS_FSCHMD_TEMP(nr), SENSORS_FSCHMD_TEMP(nr), RW }, \
		NOSYSCTL, VALUE(1), 3 }, \
	{ { SENSORS_FSCHMD_TEMP_ALARM(nr), "temp" #nr "_alarm", \
		SENSORS_FSCHMD_TEMP(nr), NOMAP, R }, \
		NOSYSCTL, VALUE(1), 0 }, \
	{ { SENSORS_FSCHMD_TEMP_FAULT(nr), "temp" #nr "_fault", \
		SENSORS_FSCHMD_TEMP(nr), NOMAP, R }, \
		NOSYSCTL, VALUE(1), 0 }

#define SENSORS_FSCHMD_FAN_FEATURES(nr) \
	{ { SENSORS_FSCHMD_FAN(nr), "fan" #nr, \
		NOMAP, NOMAP, R }, \
		NOSYSCTL, VALUE(1), 0 }, \
	{ { SENSORS_FSCHMD_FAN_DIV(nr), "fan" #nr "_div", \
		SENSORS_FSCHMD_FAN(nr), NOMAP, RW }, \
		NOSYSCTL, VALUE(nr), 0 }, \
	{ { SENSORS_FSCHMD_FAN_ALARM(nr), "fan" #nr "_alarm", \
		SENSORS_FSCHMD_FAN(nr), NOMAP, R }, \
		NOSYSCTL, VALUE(1), 0 }, \
	{ { SENSORS_FSCHMD_FAN_FAULT(nr), "fan" #nr "_fault", \
		SENSORS_FSCHMD_FAN(nr), NOMAP, R }, \
		NOSYSCTL, VALUE(1), 0 }

static sensors_chip_feature fschmd_features[] =
{
	SENSORS_FSCHMD_IN_FEATURES(0),
	SENSORS_FSCHMD_IN_FEATURES(1),
	SENSORS_FSCHMD_IN_FEATURES(2),
	SENSORS_FSCHMD_TEMP_FEATURES(1),
	SENSORS_FSCHMD_TEMP_FEATURES(2),
	SENSORS_FSCHMD_TEMP_FEATURES(3),
	SENSORS_FSCHMD_TEMP_FEATURES(4),
	SENSORS_FSCHMD_TEMP_FEATURES(5),
	SENSORS_FSCHMD_FAN_FEATURES(1),
	SENSORS_FSCHMD_FAN_FEATURES(2),
	SENSORS_FSCHMD_FAN_FEATURES(3),
	SENSORS_FSCHMD_FAN_FEATURES(4),
	SENSORS_FSCHMD_FAN_FEATURES(5),
	{ { 0 }, 0 }
};

static sensors_chip_feature fschrc_features[] =
{
	SENSORS_FSCHMD_IN_FEATURES(0),
	SENSORS_FSCHMD_IN_FEATURES(1),
	SENSORS_FSCHMD_IN_FEATURES(2),
	SENSORS_FSCHMD_TEMP_FEATURES(1),
	SENSORS_FSCHMD_TEMP_FEATURES(2),
	SENSORS_FSCHMD_TEMP_FEATURES(3),
	SENSORS_FSCHMD_FAN_FEATURES(1),
	SENSORS_FSCHMD_FAN_FEATURES(2),
	SENSORS_FSCHMD_FAN_FEATURES(3),
	SENSORS_FSCHMD_FAN_FEATURES(4),
	{ { 0 }, 0 }
};

sensors_chip_features sensors_chip_features_list[] =
{
 { SENSORS_LM78_PREFIX, lm78_features },
 { SENSORS_LM78J_PREFIX, lm78j_features },
 { SENSORS_LM79_PREFIX, lm79_features },
 { SENSORS_LM75_PREFIX, lm75_features },
 { SENSORS_GL518_PREFIX, gl518_features },
 { SENSORS_GL520_PREFIX, gl520_features },
 { SENSORS_LM80_PREFIX, lm80_features },
 { SENSORS_W83781D_PREFIX, w83781d_features },
 { SENSORS_W83782D_PREFIX, w83782d_features },
 { SENSORS_W83783S_PREFIX, w83783s_features },
 { SENSORS_W83697HF_PREFIX, w83782d_features }, /* same as 782d */
		/* Cheat on 83627HF for now - no separate #defines */
 { SENSORS_W83627HF_PREFIX, w83782d_features },
		/* Cheat on 83627THF for now - no separate #defines */
 { SENSORS_W83627THF_PREFIX, w83782d_features },
		/* Cheat on 83637HF for now - no separate #defines */
 { SENSORS_W83637HF_PREFIX, w83782d_features },
 { SENSORS_W83687THF_PREFIX, w83782d_features }, /* Same as W83627THF */
 { SENSORS_W83791D_PREFIX, w83791d_features },
 { SENSORS_W83792D_PREFIX, w83792d_features },
 { SENSORS_W83793_PREFIX, w83793_features },
 { SENSORS_W83L785TS_PREFIX, w83l785ts_features },
 { SENSORS_W83627EHF_PREFIX, w83627ehf_features },
 { SENSORS_W83627DHG_PREFIX, w83627ehf_features }, /* Similar to W83627EHF but no in9 */
 { SENSORS_AS99127F_PREFIX, as99127f_features },
 { SENSORS_ASB100_PREFIX, asb100_features },
 { SENSORS_ADM9240_PREFIX, adm9240_features },
		/* Cheat on LM81 - same features */
 { SENSORS_LM81_PREFIX, adm9240_features },
		/* Cheat on DS1780 - same features */
 { SENSORS_DS1780_PREFIX, adm9240_features },
 { SENSORS_ADM1021_PREFIX, adm1021_features },
 { SENSORS_MAX1617_PREFIX, max1617_features },
 { SENSORS_MC1066_PREFIX, max1617_features },
 { SENSORS_MAX1617A_PREFIX, max1617a_features },
		/* Cheat on LM84 for now - no separate #defines */
 { SENSORS_LM84_PREFIX, adm1021_features },
		/* Cheat on GL523 for now - no separate #defines */
 { SENSORS_GL523_PREFIX, adm1021_features },
 { SENSORS_ADM1023_PREFIX, adm1023_features },
		/* Cheat on THMC10 for now - no separate #defines */
 { SENSORS_THMC10_PREFIX, adm1021_features },
 { SENSORS_SIS5595_PREFIX, sis5595_features },
 { SENSORS_MAXI_CG_PREFIX, maxi_cg_features },
 { SENSORS_MAXI_CO_PREFIX, maxi_co_features },
 { SENSORS_MAXI_AS_PREFIX, maxi_as_features },
		/* Cheat on NBA for now - no separate #defines */
 { SENSORS_MAXI_NBA_PREFIX, maxi_as_features },
 { SENSORS_THMC50_PREFIX, thmc50_features },
		/* Cheat on ADM1022 for now - no separate #defines */
 { SENSORS_ADM1022_PREFIX, thmc50_features },
 { SENSORS_ADM1025_PREFIX, adm1025_features },
 { SENSORS_NE1619_PREFIX, adm1025_features },
 { SENSORS_ADM1026_PREFIX, adm1026_features },
 { SENSORS_VIA686A_PREFIX, via686a_features },
 { SENSORS_DDCMON_PREFIX, ddcmon_features },
 { SENSORS_EEPROM_PREFIX, eeprom_features },
 { SENSORS_LM87_PREFIX, lm87_features },
 { SENSORS_MTP008_PREFIX, mtp008_features },
 { SENSORS_DS1621_PREFIX, ds1621_features },
 { SENSORS_ADM1024_PREFIX, adm1024_features },
 { SENSORS_IT87_PREFIX, it87_features },
 { SENSORS_IT8712_PREFIX, it87_features },
 { SENSORS_IT8716_PREFIX, it87_features },
 { SENSORS_IT8718_PREFIX, it87_features },
 { SENSORS_FSCPOS_PREFIX, fscpos_features },
 { SENSORS_FSCSCY_PREFIX, fscscy_features },
 { SENSORS_FSCHER_PREFIX, fscher_features },
 { SENSORS_PCF8591_PREFIX, pcf8591_features }, 
 { SENSORS_VT1211_PREFIX, vt1211_features }, 
 { SENSORS_SMSC47M192_PREFIX, smsc47m192_features }, 
 { SENSORS_SMSC47M1_PREFIX, smsc47m1_features }, 
 { SENSORS_SMSC47M2_PREFIX, smsc47m2_features }, 
 { SENSORS_PC87360_PREFIX, pc87360_features }, 
 { SENSORS_PC87363_PREFIX, pc87360_features }, 
 { SENSORS_PC87364_PREFIX, pc87360_features }, 
 { SENSORS_PC87365_PREFIX, pc87365_features }, 
 { SENSORS_PC87366_PREFIX, pc87365_features }, 
 { SENSORS_PC87427_PREFIX, pc87427_features },
 { SENSORS_LM92_PREFIX, lm92_features }, 
 { SENSORS_VT8231_PREFIX, vt8231_features }, 
 { SENSORS_BMC_PREFIX, bmc_features }, 
 { SENSORS_LM85_PREFIX, lm85_features },
 { SENSORS_LM85B_PREFIX, lm85_features },
 { SENSORS_LM85C_PREFIX, lm85_features },
 { SENSORS_EMC6D100_PREFIX, lm85_features },
 { SENSORS_EMC6D102_PREFIX, lm85_features },
 { SENSORS_ADM1027_PREFIX, adm1027_features },
 { SENSORS_ADT7463_PREFIX, adm1027_features },
 { SENSORS_LM83_PREFIX, lm83_features },
 { SENSORS_LM90_PREFIX, lm90_features },
 { SENSORS_ADM1032_PREFIX, lm90_features },
 { SENSORS_LM99_PREFIX, lm90_features },
 { SENSORS_LM86_PREFIX, lm90_features },
 { SENSORS_MAX6657_PREFIX, lm90_features },
 { SENSORS_ADT7461_PREFIX, lm90_features },
 { SENSORS_MAX6680_PREFIX, lm90_features },
 { SENSORS_LM63_PREFIX, lm63_features },
 { SENSORS_MAX1619_PREFIX, max1619_features },
 { SENSORS_XEONTEMP_PREFIX, xeontemp_features },
 { SENSORS_MAX6650_PREFIX, max6650_features },
 { SENSORS_ADM1029_PREFIX, adm1029_features },
 { SENSORS_ADM1030_PREFIX, adm1030_features },
 { SENSORS_ADM1031_PREFIX, adm1031_features },
 { SENSORS_LM93_PREFIX, lm93_features },
 { SENSORS_SMSC47B397_PREFIX, smsc47b397_features },
 { SENSORS_F71805F_PREFIX, f71805f_features },
 { SENSORS_F71872F_PREFIX, f71805f_features },
 { SENSORS_ABITUGURU_PREFIX, abituguru_features },
 { SENSORS_ABITUGURU3_PREFIX, abituguru3_features },
 { SENSORS_K8TEMP_PREFIX, k8temp_features },
 { SENSORS_CORETEMP_PREFIX, coretemp_features },
 { SENSORS_DME1737_PREFIX, dme1737_features },
 { SENSORS_SCH311X_PREFIX, dme1737_features },
 { SENSORS_APPLESMC_PREFIX, applesmc_features },
 { SENSORS_F71882FG_PREFIX, f71882fg_features },
 { SENSORS_FSCHMD_PREFIX, fschmd_features },
 { SENSORS_FSCHRC_PREFIX, fschrc_features },
 { 0 }
};
