/*
 * Copyright (c) 2003 Benedikt Meurer (benedikt.meurer@unix-ag.uni-siegen.de)
 *               2004 Jean-Franois Wauthy (pollux@xfce.org)
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <ctype.h>
#include <libxfcegui4/libxfcegui4.h>
#include <libxfce4util/libxfce4util.h>

#include <printing-system.h>
#include <filter.h>
#include <xfprintsettings.h>

#include "input_page.h"
#include "head_page.h"
#include "pprint_page.h"
#include "sheets_page.h"
#include "vpages_page.h"

/* Prototypes */
static void button_save_clicked_cb (GtkWidget * widget, gpointer data);
static gboolean print (PrintDlg * dlg, const gchar * ifile, const XfprintSettings * settings);
static XfprintFilter *a2ps_filter (const XfprintSettings * settings, const gchar * ifile);

/****************/
/* Print dialog */
/****************/
void
print_dialog_run (const gchar * ifile)
{
  GtkWidget *hbox;
  GtkWidget *label;

  PrintDlg *dlg;
  XfprintSettings *settings;
  GdkPixbuf *icon = NULL;

  dlg = g_new0 (PrintDlg, 1);

  dlg->icontheme = xfce_icon_theme_get_for_screen (NULL);
  dlg->dialog = gtk_dialog_new ();

  icon = xfce_icon_theme_load (dlg->icontheme, "xfprint", 48);
  if (icon) {
    gtk_window_set_icon (GTK_WINDOW (dlg->dialog), icon);
    g_object_unref (icon);
  }
  dlg->button_save = xfce_create_mixed_button (GTK_STOCK_SAVE, _("Save settings"));
  g_signal_connect (G_OBJECT (dlg->button_save), "clicked", G_CALLBACK (button_save_clicked_cb), dlg);
  gtk_dialog_add_action_widget (GTK_DIALOG (dlg->dialog), dlg->button_save, GTK_RESPONSE_APPLY);
  gtk_button_box_set_child_secondary (GTK_BUTTON_BOX (GTK_DIALOG (dlg->dialog)->action_area), dlg->button_save, TRUE);

  gtk_dialog_add_buttons (GTK_DIALOG (dlg->dialog),
                          GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL, GTK_STOCK_PRINT, GTK_RESPONSE_OK, NULL);

  gtk_window_set_default_size (GTK_WINDOW (dlg->dialog), 300, 10);
  gtk_window_set_title (GTK_WINDOW (dlg->dialog), _("XfPrint"));

  gtk_box_set_spacing (GTK_BOX (GTK_DIALOG (dlg->dialog)->vbox), 4);

  /* Notebook */
  dlg->notebook = gtk_notebook_new ();
  gtk_container_set_border_width (GTK_CONTAINER (dlg->notebook), 5);
  gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dlg->dialog)->vbox), dlg->notebook, TRUE, TRUE, 0);

  dlg->sheets = sheets_page_new ();
  label = gtk_label_new (_("Sheets"));
  gtk_misc_set_alignment (GTK_MISC (label), 0, 0.5);
  gtk_notebook_append_page (GTK_NOTEBOOK (dlg->notebook), dlg->sheets, label);

  dlg->vpages = vpages_page_new ();
  label = gtk_label_new (_("Virtual pages"));
  gtk_misc_set_alignment (GTK_MISC (label), 0, 0.5);
  gtk_notebook_append_page (GTK_NOTEBOOK (dlg->notebook), dlg->vpages, label);

  dlg->pprint = pprint_page_new ();
  label = gtk_label_new (_("Pretty-printing"));
  gtk_misc_set_alignment (GTK_MISC (label), 0, 0.5);
  gtk_notebook_append_page (GTK_NOTEBOOK (dlg->notebook), dlg->pprint, label);

  dlg->input = input_page_new ();
  label = gtk_label_new (_("Input"));
  gtk_misc_set_alignment (GTK_MISC (label), 0, 0.5);
  gtk_notebook_append_page (GTK_NOTEBOOK (dlg->notebook), dlg->input, label);

  dlg->head = head_page_new ();
  label = gtk_label_new (_("Headings"));
  gtk_misc_set_alignment (GTK_MISC (label), 0, 0.5);
  gtk_notebook_append_page (GTK_NOTEBOOK (dlg->notebook), dlg->head, label);

  /* number of copies */
  hbox = gtk_hbox_new (FALSE, 4);
  gtk_container_set_border_width (GTK_CONTAINER (hbox), 6);
  gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dlg->dialog)->vbox), hbox, FALSE, TRUE, 0);

  label = gtk_label_new (_("Copies:"));
  gtk_misc_set_alignment (GTK_MISC (label), 0, 0.5);
  gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 0);

  dlg->copies = gtk_spin_button_new_with_range (1.0, 1000.0, 1.0);
  gtk_box_pack_start (GTK_BOX (hbox), dlg->copies, FALSE, FALSE, 0);

  gtk_widget_show_all (dlg->dialog);

  printing_system_print_dialog_customize (dlg);

  /* load the settings */
  settings = xfprintsettings_load ();
  sheets_page_set_settings (SHEETS_PAGE (dlg->sheets), &settings->sheets);
  vpages_page_set_settings (VPAGES_PAGE (dlg->vpages), &settings->vpages);
  pprint_page_set_settings (PPRINT_PAGE (dlg->pprint), &settings->pprint);
  input_page_set_settings (INPUT_PAGE (dlg->input), &settings->input);
  head_page_set_settings (HEAD_PAGE (dlg->head), &settings->headings);

  gtk_spin_button_set_value (GTK_SPIN_BUTTON (dlg->copies), settings->copies);

  while (TRUE) {
    gint response = GTK_RESPONSE_NONE;
    gchar *decoded_ifile = NULL;

    response = gtk_dialog_run (GTK_DIALOG (dlg->dialog));

    if (response == GTK_RESPONSE_APPLY)
      continue;
    else if (response != GTK_RESPONSE_OK)
      break;

    if (g_str_has_prefix (ifile, "file://"))
      decoded_ifile = g_filename_from_uri (ifile, NULL, NULL);
    else if (g_path_is_absolute (ifile))
      decoded_ifile = g_strdup (ifile);
    else {
      gchar *current_dir = NULL;

      current_dir = g_get_current_dir ();
      decoded_ifile = g_build_filename (current_dir, ifile, NULL);

      g_free (current_dir);
    }
    DBG ("Printing file: %s", decoded_ifile);

    if (decoded_ifile && print (dlg, decoded_ifile, settings))
      xfce_info (_("Job sent to printer"));
    else {
      xfce_err (_("An error occured"));
      continue;
    }

    g_free (decoded_ifile);

    break;
  }

  xfprintsettings_free (settings);
  gtk_widget_destroy (dlg->dialog);
  g_object_unref (dlg->icontheme);
  g_free (dlg);
}

static void
button_save_clicked_cb (GtkWidget * widget, gpointer data)
{
  XfprintSettings *settings;
  PrintDlg *dlg;

  dlg = (PrintDlg *) data;

  settings = xfprintsettings_defaults ();

  sheets_page_get_settings (SHEETS_PAGE (dlg->sheets), &settings->sheets);
  vpages_page_get_settings (VPAGES_PAGE (dlg->vpages), &settings->vpages);
  pprint_page_get_settings (PPRINT_PAGE (dlg->pprint), &settings->pprint);
  input_page_get_settings (INPUT_PAGE (dlg->input), &settings->input);
  head_page_get_settings (HEAD_PAGE (dlg->head), &settings->headings);
  settings->copies = gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (dlg->copies));

  xfprintsettings_save (settings);
  xfprintsettings_free (settings);
}

static gboolean
print (PrintDlg * dlg, const gchar * ifile, const XfprintSettings * settings)
{
  XfprintFilterList *filters;
  XfprintFilter *filter;
  gboolean ret;
  int output;
  gchar *ofile;

  filters = xfprint_filterlist_new ();
  ret = FALSE;

  dlg->original_name = ifile;
  output = g_file_open_tmp ("xfprintXXXXXX", &ofile, NULL);

  if (output == -1)
    return ret;

  filter = a2ps_filter (settings, ifile);
  xfprint_filterlist_append (filters, filter);

  if (settings->sheets.reverse) {
    filter = xfprint_filter_new ("psselect");
    xfprint_filter_add (filter, "-r");
    xfprint_filterlist_append (filters, filter);
  }

  ret = !xfprint_filterlist_execute (filters, STDIN_FILENO, output, STDERR_FILENO);

  (void) close (output);

  if (!ret)
    xfce_err (_("Unable to print to temporary file"));
  else
    ret = printing_system_print_file (dlg, ofile);

  g_free (ofile);

  return (ret);
}

static XfprintFilter *
a2ps_filter (const XfprintSettings * settings, const gchar * ifile)
{
  XfprintFilter *filter;

  filter = xfprint_filter_new ("a2ps");

  /* sheet settings */
  xfprint_filter_add (filter, "--medium=%s", settings->sheets.papersize);
  if (settings->sheets.landscape)
    xfprint_filter_add (filter, "--landscape");
  else
    xfprint_filter_add (filter, "--portrait");
  xfprint_filter_add (filter, "--columns=%d", settings->sheets.cols);
  xfprint_filter_add (filter, "--rows=%d", settings->sheets.rows);
  if (settings->sheets.fillcols)
    xfprint_filter_add (filter, "--major=columns");
  else
    xfprint_filter_add (filter, "--major=rows");
  if (settings->sheets.borders)
    xfprint_filter_add (filter, "--borders=yes");
  else
    xfprint_filter_add (filter, "--borders=no");

  /* virtual pages settings */
  xfprint_filter_add (filter, "--line-numbers=%d", settings->vpages.linenumbers);
  if (settings->vpages.lpp > 0)
    xfprint_filter_add (filter, "--lines-per-page=%d", settings->vpages.lpp);
  if (settings->vpages.cpl > 0)
    xfprint_filter_add (filter, "--chars-per-line=%d", settings->vpages.cpl);
  xfprint_filter_add (filter, "--tabsize=%d", settings->vpages.tabsize);
  xfprint_filter_add (filter, "--non-printable-format=%s", settings->vpages.nonprtfmt);

  /* pretty print settings */
  if (strcmp (settings->pprint.language, "__auto__") != 0)
    xfprint_filter_add (filter, "--pretty-print=%s", settings->pprint.language);
  xfprint_filter_add (filter, "--highlight-level=%s", settings->pprint.highlight);
  xfprint_filter_add (filter, "--strip-level=%d", settings->pprint.strip);

  /* input settings */
  xfprint_filter_add (filter, "--encoding=%s", settings->input.encoding);
  if (!settings->input.all) {
    xfprint_filter_add (filter, "--pages=%d-%d", settings->input.from, settings->input.to);
  }
  if (settings->input.cut)
    xfprint_filter_add (filter, "--truncate-lines=yes");
  else
    xfprint_filter_add (filter, "--truncate-lines=no");
  if (settings->input.interpret)
    xfprint_filter_add (filter, "--interpret=yes");
  else
    xfprint_filter_add (filter, "--interpret=no");
  if (settings->input.binary)
    xfprint_filter_add (filter, "--print-anyway=yes");
  else
    xfprint_filter_add (filter, "--print-anyway=no");

  /* headings settings */
  if (!settings->headings.headers) {
    xfprint_filter_add (filter, "--no-header");
  }
  else {
    xfprint_filter_add (filter, "--header=%s", settings->headings.header);
    xfprint_filter_add (filter, "--underlay=%s", settings->headings.underlay);
    xfprint_filter_add (filter, "--center-title=%s", settings->headings.ctitle);
    xfprint_filter_add (filter, "--left-title=%s", settings->headings.ltitle);
    xfprint_filter_add (filter, "--right-title=%s", settings->headings.rtitle);
    xfprint_filter_add (filter, "--left-footer=%s", settings->headings.lfooter);
    xfprint_filter_add (filter, "--footer=%s", settings->headings.cfooter);
    xfprint_filter_add (filter, "--right-footer=%s", settings->headings.rfooter);
  }

  xfprint_filter_add (filter, "--output=-");
  xfprint_filter_add (filter, "%s", ifile);

  return (filter);
}
