/*
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                                             %
%                                                                             %
%                                                                             %
%                             CCCC  IIIII  N   N                              %
%                            C        I    NN  N                              %
%                            C        I    N N N                              %
%                            C        I    N  NN                              %
%                             CCCC  IIIII  N   N                              %
%                                                                             %
%                                                                             %
%                    Read/Write Kodak Cineon Image Format.                    %
%                Cineon Image Format is a subset of SMTPE DPX                 %
%                                                                             %
%                                                                             %
%                              Software Design                                %
%                                John Cristy                                  %
%                             Kelly Bergougnoux                               %
%                               October 2003                                  %
%                                                                             %
%                                                                             %
%  Copyright 1999-2005 ImageMagick Studio LLC, a non-profit organization      %
%  dedicated to making software imaging solutions freely available.           %
%                                                                             %
%  You may not use this file except in compliance with the License.  You may  %
%  obtain a copy of the License at                                            %
%                                                                             %
%    http://www.imagemagick.org/www/Copyright.html                            %
%                                                                             %
%  Unless required by applicable law or agreed to in writing, software        %
%  distributed under the License is distributed on an "AS IS" BASIS,          %
%  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.   %
%  See the License for the specific language governing permissions and        %
%  limitations under the License.                                             %
%                                                                             %
%                                                                             %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
%  Cineon image file format draft is available at
%  http://www.cineon.com/ff_draft.php.
%
%
*/

/*
  Include declarations.
*/
#include "magick/studio.h"
#include "magick/blob.h"
#include "magick/blob_private.h"
#include "magick/colorspace.h"
#include "magick/exception.h"
#include "magick/exception_private.h"
#include "magick/image.h"
#include "magick/image_private.h"
#include "magick/list.h"
#include "magick/magick.h"
#include "magick/memory_.h"
#include "magick/monitor.h"
#include "magick/static.h"
#include "magick/string_.h"

/*
  Forward declaractions.
*/
static MagickBooleanType
  WriteCINImage(const ImageInfo *,Image *);

/*
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                                             %
%                                                                             %
%                                                                             %
%   I s C I N E O N                                                           %
%                                                                             %
%                                                                             %
%                                                                             %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
%  IsCIN() returns MagickTrue if the image format type, identified by the magick
%  string, is CIN.
%
%  The format of the IsCIN method is:
%
%      MagickBooleanType IsCIN(const unsigned char *magick,const size_t length)
%
%  A description of each parameter follows:
%
%    o magick: This string is generally the first few bytes of an image file
%      or blob.
%
%    o length: Specifies the length of the magick string.
%
%
*/
static MagickBooleanType IsCIN(const unsigned char *magick,const size_t length)
{
  if (length < 4)
    return(MagickFalse);
  if (memcmp(magick,"\200\052\137\327",4) == 0)
    return(MagickTrue);
  return(MagickFalse);
}

/*
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                                             %
%                                                                             %
%                                                                             %
%   R e a d C I N E O N I m a g e                                             %
%                                                                             %
%                                                                             %
%                                                                             %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
%  ReadCINImage() reads an CIN X image file and returns it.  It allocates
%  the memory necessary for the new Image structure and returns a point to the
%  new image.
%
%  The format of the ReadCINImage method is:
%
%      Image *ReadCINImage(const ImageInfo *image_info,
%        ExceptionInfo *exception)
%
%  A description of each parameter follows:
%
%    o image_info: The image info.
%
%    o exception: return any errors or warnings in this structure.
%
%
*/
static Image *ReadCINImage(const ImageInfo *image_info,
  ExceptionInfo *exception)
{
#define MonoColorType  1
#define RGBColorType  3

  char
    magick[4];

  Image
    *image;

  long
    y;

  MagickBooleanType
    status;

  register long
    x;

  register PixelPacket
    *q;

  register long
    i;

  ssize_t
    count;

  unsigned char
    colortype;

  unsigned long
    headersize,
    pixel;

  /*
    Open image file.
  */
  assert(image_info != (const ImageInfo *) NULL);
  assert(image_info->signature == MagickSignature);
  if (image_info->debug != MagickFalse)
    (void) LogMagickEvent(TraceEvent,GetMagickModule(),image_info->filename);
  assert(exception != (ExceptionInfo *) NULL);
  assert(exception->signature == MagickSignature);
  image=AllocateImage(image_info);
  status=OpenBlob(image_info,image,ReadBinaryBlobMode,exception);
  if (status == MagickFalse)
    {
      image=DestroyImageList(image);
      return((Image *) NULL);
    }
  /*
    Read CIN image.
  */
  count=ReadBlob(image,4,(unsigned char *) magick);
  if ((count == 0) ||
      ((LocaleNCompare((char *) magick,"\200\052\137\327",4) != 0)))
    ThrowReaderException(CorruptImageError,"ImproperImageHeader");
  headersize=ReadBlobMSBLong(image);
  for (i=0; i < 185; i++)
    (void) ReadBlobByte(image);
  colortype=(unsigned char) ReadBlobByte(image);
  for (i=0; i < 4; i++)
    (void) ReadBlobByte(image);
  image->depth=ReadBlobByte(image) > 8 ? 16UL  : 8UL;
  (void) ReadBlobByte(image);
  image->columns= ReadBlobMSBLong(image);
  image->rows= ReadBlobMSBLong(image);
  (void) SeekBlob(image,(MagickOffsetType) headersize,SEEK_SET);
  if (image_info->ping)
    {
      CloseBlob(image);
      return(image);
    }
  /*
    Convert CIN raster image to pixel packets.
  */
  switch ((int) colortype)
  {
    case (MonoColorType):
    {
      q=SetImagePixels(image,0,0,image->columns,image->rows);
      for (x=0; x < (long) ((image->columns*image->rows)/3); x++)
      {
        pixel=ReadBlobMSBLong(image);
        q->red=ScaleXToQuantum((pixel >> 0) & 0x3ff,1023);
        q->green=q->red;
        q->blue=q->red;
        q++;
        q->red=ScaleXToQuantum((pixel >> 10) & 0x3ff,1023);
        q->green=q->red;
        q->blue=q->red;
        q++;
        q->red=ScaleXToQuantum((pixel >> 20) & 0x3ff,1023);
        q->green=q->red;
        q->blue=q->red;
        q++;
      }
      break;
    }
    case (RGBColorType):
    {
      for (y=0; y < (long) image->rows; y++)
      {
        q=SetImagePixels(image,0,y,image->columns,1);
        if (q == (PixelPacket *) NULL)
          break;
        for (x=0; x < (long) image->columns; x++)
        {
          pixel=ReadBlobMSBLong(image);
          q->red=ScaleXToQuantum((pixel >> 22) & 0x3ff,1023);
          q->green=ScaleXToQuantum((pixel >> 12) & 0x3ff,1023);
          q->blue=ScaleXToQuantum((pixel >> 2) & 0x3ff,1023);
          q++;
        }
        if (!SyncImagePixels(image))
          break;
        if (image->previous == (Image *) NULL)
          if ((image->progress_monitor != (MagickProgressMonitor) NULL) &&
              (QuantumTick(y,image->rows) != MagickFalse))
            {
              status=image->progress_monitor(LoadImageTag,y,image->rows,
                image->client_data);
              if (status == MagickFalse)
                break;
            }
      }
      break;
    }
    default:
      ThrowReaderException(CorruptImageError,"ColorTypeNotSupported");
  }
  if (EOFBlob(image))
    ThrowFileException(exception,CorruptImageError,"UnexpectedEndOfFile",
      image->filename);
  CloseBlob(image);
  return(GetFirstImageInList(image));
}

/*
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                                             %
%                                                                             %
%                                                                             %
%   R e g i s t e r C I N E O N I m a g e                                     %
%                                                                             %
%                                                                             %
%                                                                             %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
%  RegisterCINImage() adds attributes for the CIN image format to the list of
%  of supported formats.  The attributes include the image format tag, a method
%  to read and/or write the format, whether the format supports the saving of
%  more than one frame to the same file or blob, whether the format supports
%  native in-memory I/O, and a brief description of the format.
%
%  The format of the RegisterCINImage method is:
%
%      RegisterCINImage(void)
%
*/
ModuleExport void RegisterCINImage(void)
{
  MagickInfo
    *entry;

  entry=SetMagickInfo("CIN");
  entry->decoder=(DecoderHandler *) ReadCINImage;
  entry->encoder=(EncoderHandler *) WriteCINImage;
  entry->magick=(MagickHandler *) IsCIN;
  entry->description=AcquireString("Cineon Image File");
  entry->module=AcquireString("CIN");
  (void) RegisterMagickInfo(entry);
}

/*
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                                             %
%                                                                             %
%                                                                             %
%   U n r e g i s t e r C I N E O N I m a g e                                 %
%                                                                             %
%                                                                             %
%                                                                             %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
%  UnregisterCINImage() removes format registrations made by the CIN module
%  from the list of supported formats.
%
%  The format of the UnregisterCINImage method is:
%
%      UnregisterCINImage(void)
%
*/
ModuleExport void UnregisterCINImage(void)
{
  (void) UnregisterMagickInfo("CINEON");
}

/*
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                                             %
%                                                                             %
%                                                                             %
%   W r i t e C I N E O N I m a g e                                           %
%                                                                             %
%                                                                             %
%                                                                             %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
%  WriteCINImage() writes an image in CIN encoded image format.
%
%  The format of the WriteCINImage method is:
%
%      MagickBooleanType WriteCINImage(const ImageInfo *image_info,Image *image)
%
%  A description of each parameter follows.
%
%    o image_info: The image info.
%
%    o image:  The image.
%
%
*/
static MagickBooleanType WriteCINImage(const ImageInfo *image_info,Image *image)
{
  long
    y;

  MagickBooleanType
    status;

  register const PixelPacket
    *p;

  register long
    i,
    x;

  unsigned long
    pixel;

  unsigned long
    maxb,
    maxg,
    maxr,
    minb,
    ming,
    minr;

  /*
    Get the max values for each component... I guess it "needs" it...
  */
  maxr=maxg=maxb=0;
  minr=ming=minb=MaxRGB;
  for (y=0; y < (long) image->rows; y++)
  {
    p=AcquireImagePixels(image,0,y,image->columns,1,&image->exception);
    if (p == (const PixelPacket *) NULL)
      break;
    for (x=0; x < (long) image->columns; x++)
    {
      if ( (unsigned long) p->red > maxr ) maxr = p->red;
      if ( (unsigned long) p->green > maxg ) maxg = p->green;
      if ( (unsigned long) p->blue > maxb ) maxb = p->blue;
      if ( (unsigned long) p->red < minr ) minr = p->red;
      if ( (unsigned long) p->green < ming ) ming = p->green;
      if ( (unsigned long) p->blue < minb ) minb = p->blue;
      p++;
    }
  }
  /*
    Div by 64 to get 10bit values since that's all I'll do right now.
  */
  maxr/=64;
  maxg/=64;
  maxb/=64;
  minr/=64;
  ming/=64;
  minb/=64;
  /*
    Open output image file.
  */
  assert(image_info != (const ImageInfo *) NULL);
  assert(image_info->signature == MagickSignature);
  assert(image != (Image *) NULL);
  assert(image->signature == MagickSignature);
  if (image->debug != MagickFalse)
    (void) LogMagickEvent(TraceEvent,GetMagickModule(),image->filename);
  status=OpenBlob(image_info,image,WriteBinaryBlobMode,&image->exception);
  if (status == MagickFalse)
    return(status);
  /* Magick Number... */
  (void) SetImageColorspace(image,RGBColorspace);
  (void) WriteBlobMSBLong(image,0x802A5FD7UL);
  /* Offset to image */
  (void) WriteBlobMSBLong(image,0x0800);
  /* Generic section header length */
  (void) WriteBlobMSBLong(image,0x400);
  /* Industry Specific lenght */
  (void) WriteBlobMSBLong(image,0x400);
  /* variable length section */
  (void) WriteBlobMSBLong(image,0x0);
  /* Total image file size */
  (void) WriteBlobMSBLong(image,4*image->columns*image->rows+0x2000);
  /* Version number of header format */
  (void) WriteBlobString(image,"V4.5");
  (void) WriteBlobMSBLong(image,0x0);
  /* Filename */
  (void) WriteBlobString(image,image->filename);
  for ( i =0 ; i < (long) (100-strlen(image->filename)); i++ )
    (void) WriteBlobByte(image,0);
  /* Creation Date. */
  (void) WriteBlobString(image,"yyyy:mm:dd  ");
  /* Creation Time. */
  (void) WriteBlobString(image,"hh:mm:ssxxx ");
  for ( i =0 ; i < 36 ; i++)
    (void) WriteBlobByte(image,0);
  /* 0 left to right top to bottom */
  (void) WriteBlobByte(image,0);
  /* 3 channels */
  (void) WriteBlobByte(image,3);
  /* alignment */
  (void) WriteBlobByte(image,0x0);
  (void) WriteBlobByte(image,0x0);
  /* Do R */
  /* Channel 1 designator byte 0 */
  (void) WriteBlobByte(image,0);
  /* Channel 1 designator byte 1 */
  (void) WriteBlobByte(image,1);
  /* Bits per pixel... */
  (void) WriteBlobByte(image,10);
  /* alignment */
  (void) WriteBlobByte(image,0);
  /* pixels per line */
  (void) WriteBlobMSBLong(image,image->columns);
  /* lines per image */
  (void) WriteBlobMSBLong(image,image->rows);
  /* Minimum data value */
  (void) WriteBlobMSBLong(image,minr);
  /* Minimum quantity represented */
  (void) WriteBlobMSBLong(image,0x0);
  /* Maximum data value */
  (void) WriteBlobMSBLong(image,maxr);
  /* Maximum quantity represented */
  (void) WriteBlobMSBLong(image,0x40000000);
  /* Do G */
  /* see above. */
  (void) WriteBlobByte(image,0);
  (void) WriteBlobByte(image,2);
  (void) WriteBlobByte(image,10);
  (void) WriteBlobByte(image,0);
  (void) WriteBlobMSBLong(image,image->columns);
  (void) WriteBlobMSBLong(image,image->rows);
  (void) WriteBlobMSBLong(image,ming);
  (void) WriteBlobMSBLong(image,0x0);
  (void) WriteBlobMSBLong(image,maxg);
  (void) WriteBlobMSBLong(image,0x40000000);
  /* Go B */
  /* see above. */
  (void) WriteBlobByte(image,0);
  (void) WriteBlobByte(image,3);
  (void) WriteBlobByte(image,10);
  (void) WriteBlobByte(image,0);
  (void) WriteBlobMSBLong(image,image->columns);
  (void) WriteBlobMSBLong(image,image->rows);
  (void) WriteBlobMSBLong(image,minb);
  (void) WriteBlobMSBLong(image,0x0);
  (void) WriteBlobMSBLong(image,maxb);
  (void) WriteBlobMSBLong(image,0x40000000);
  /* pad channel 4-8 */
  for (i=0; i < 5*28; i++)
    (void) WriteBlobByte(image,0x00);
  /* k here's where it gets ugly and I should be shot */
  /* White point(colour temp.) x */
  (void) WriteBlobMSBLong(image,0x4EFF0000);
  /* White point(colour temp.) y */
  (void) WriteBlobMSBLong(image,0x4EFF0000);
  /* Red primary chromaticity x */
  (void) WriteBlobMSBLong(image,0x4EFF0000);
  /* Red primary chromaticity y */
  (void) WriteBlobMSBLong(image,0x4EFF0000);
  /* Green primary chromaticity x */
  (void) WriteBlobMSBLong(image,0x4EFF0000);
  /* Green primary chromaticity y */
  (void) WriteBlobMSBLong(image,0x4EFF0000);
  /* Blue primary chromaticity x */
  (void) WriteBlobMSBLong(image,0x4EFF0000);
  /* Blue primary chromaticity y */
  (void) WriteBlobMSBLong(image,0x4EFF0000);
  /* Label Text... */
  for (i=0; i < 200+28; i++)
    (void) WriteBlobByte(image,0x00);
  /* pixel interleave (rgbrgbr...) */
  (void) WriteBlobByte(image,0);
  /* Packing longword (32bit) boundaries */
  (void) WriteBlobByte(image,5);
  /* Data unsigned */
  (void) WriteBlobByte(image,0);
  /* Image sense: positive image */
  (void) WriteBlobByte(image,0);
  /* End of line padding */
  (void) WriteBlobMSBLong(image,0x0);
  /* End of channel padding */
  (void) WriteBlobMSBLong(image,0x0);
  /* Reseved for future Use.. */
  for (i=0; i < 20; i++)
    (void) WriteBlobByte(image,0x00);
  /* X offset */
  (void) WriteBlobMSBLong(image,0x0);
  /* Y offset */
  (void) WriteBlobMSBLong(image,0x0);
  /* Filename */
  (void) WriteBlobString(image,image->filename);
  for ( i =0 ; i < (long) (100-strlen(image->filename)); i++ )
    (void) WriteBlobByte(image,0);
  /* date. who cares ? */
  for ( i =0 ; i < 12 ; i++ )
    (void) WriteBlobByte(image,0);
  /* time who cares ? */
  for ( i =0 ; i < 12 ; i++ )
    (void) WriteBlobByte(image,0);
  (void) WriteBlobString(image,"ImageMagick");
  for (i = 0; i < 64-11 ; i++)
    (void) WriteBlobByte(image,0);
  for (i = 0; i < 32 ; i++)
    (void) WriteBlobByte(image,0);
  for (i = 0; i < 32 ; i++)
    (void) WriteBlobByte(image,0);
  /* X input device pitch */
  (void) WriteBlobMSBLong(image,0x4326AB85);
  /* Y input device pitch */
  (void) WriteBlobMSBLong(image,0x4326AB85);
  /* Image gamma */
  (void) WriteBlobMSBLong(image,0x3F800000);
  /* Reserved for future use */
  for (i = 0; i < 40 ; i++)
    (void) WriteBlobByte(image,0);
  for ( i = 0 ; i < 4 ; i++)
    (void) WriteBlobByte(image,0);
  (void) WriteBlobMSBLong(image,0x0);
  (void) WriteBlobMSBLong(image,0x0);
  for ( i = 0 ; i < 32 ; i++)
    (void) WriteBlobByte(image,0);
  (void) WriteBlobMSBLong(image,0x0);
  (void) WriteBlobMSBLong(image,0x0);
  for ( i = 0 ; i < 32 ; i++)
    (void) WriteBlobByte(image,0);
  for ( i = 0 ; i < 200 ; i++)
    (void) WriteBlobByte(image,0);
  for ( i = 0 ; i < 740 ; i++)
    (void) WriteBlobByte(image,0);
  /*
    Convert pixel packets to CIN raster image.
  */
  for (y=0; y < (long) image->rows; y++)
  {
    p=AcquireImagePixels(image,0,y,image->columns,1,&image->exception);
    if (p == (const PixelPacket *) NULL)
      break;
    for (x=0; x < (long) image->columns; x++)
    {
      pixel=(unsigned long)
        (((long) ((1023L*p->red+MaxRGB/2)/MaxRGB) & 0x3ff) << 22) |
        (((long) ((1023L*p->green+MaxRGB/2)/MaxRGB) &0x3ff) << 12) |
        (((long) ((1023L*p->blue+MaxRGB/2)/MaxRGB) &0x3ff) << 2);
      (void) WriteBlobMSBLong(image,pixel);
      p++;
    }
  }
  CloseBlob(image);
  return(status);
}
