/*
 * Copyright (C) 2004 the xine-project
 *
 * This program is free software; you can redistribute it and/or 
 * modify it under the terms of the GNU General Public License as 
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 *
 * $Id: time_widget.c,v 1.3 2004/11/10 01:29:13 dsalt Exp $
 *
 * a nice black time display area
 */

#include "globals.h"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <xine.h>

#include "time_widget.h"
#include "utils.h"
#include "globals.h"
#include "player.h"

#define LINESIZE 1025

struct time_widget_s {

  GtkWidget            *w;

  PangoLayout          *pl, *pl_big;
  PangoFontDescription *pfd, *pfd_big;

  GdkGC                *black_gc, *blue_gc, *white_gc;
};

static void paint_tw (time_widget_t *tw) {

  gint pos_stream, pos_time, length_time;
  gfloat pos;

  gdk_draw_rectangle (tw->w->window, tw->black_gc, TRUE,
		      0, 0, 
		      tw->w->allocation.width, tw->w->allocation.height+1);

  if ((xine_get_status (stream) == XINE_STATUS_PLAY) &&
      (xine_get_pos_length (stream, &pos_stream, &pos_time, &length_time))) {

    gint cur_time, length;

    gchar curtime_str[256];
    gchar length_str[256];

    pos = (gfloat) pos_stream;

    cur_time = pos_time / 1000;
    length   = length_time / 1000;

    int_to_timestring (cur_time, curtime_str, 256);
    int_to_timestring (length,   length_str, 256);

    pango_layout_set_text (tw->pl_big, curtime_str, -1);
    gdk_draw_layout (tw->w->window, tw->blue_gc, 7, 2, tw->pl_big);

    if (length_time>0)
      pango_layout_set_text (tw->pl, length_str, -1);
    else
      pango_layout_set_text (tw->pl, _("live"), -1);
    gdk_draw_layout (tw->w->window, tw->blue_gc, 7, 23, tw->pl);
  } 
}

static gboolean expose_event_cb (GtkWidget *widget, 
				 GdkEventExpose *event, 
				 gpointer data) {

  time_widget_t *tw = (time_widget_t *) data;

  gdk_window_clear_area (widget->window,
                         event->area.x, event->area.y,
                         event->area.width, event->area.height+1);

  gdk_gc_set_clip_rectangle (widget->style->fg_gc[widget->state],
                             &event->area);

  paint_tw (tw);

  gdk_gc_set_clip_rectangle (widget->style->fg_gc[widget->state],
                             NULL);
  return TRUE;
}

static void alloc_color(GtkWidget *window, GdkGC *gc, gint r, gint g, gint b) {

  GdkColor *color=malloc(sizeof(GdkColor));

  if (gc) {
    color->red = r; color->green = g; color->blue = b;
    if (!gdk_colormap_alloc_color (gtk_widget_get_colormap (window), color, FALSE, TRUE)) {
      printf ("failed to alloc color (%d/%d/%d)\n",color->red,color->green,color->blue);
      exit(1);
    }
    gdk_gc_set_foreground (gc, color);
  }
}

time_widget_t *create_time_widget (GtkWidget **w) {

  time_widget_t *tw;

  tw = (time_widget_t *) malloc (sizeof (time_widget_t));

  tw->w = gtk_drawing_area_new ();
  gtk_drawing_area_size (GTK_DRAWING_AREA(tw->w), 90, 40);

  *w = tw->w;

  return tw;
}

static gint update_time_widget_cb (gpointer data) {

  time_widget_t *tw = data;

  gdk_threads_enter ();
  paint_tw (tw);
  gdk_threads_leave ();

  return TRUE;
}

void postinit_time_widget (GtkWidget *window, time_widget_t *tw) {

  g_signal_connect (tw->w, "expose_event", 
		    G_CALLBACK (expose_event_cb), tw);

  tw->black_gc = gdk_gc_new (window->window);
  alloc_color (window, tw->black_gc, 0, 0, 0);

  tw->white_gc = gdk_gc_new (window->window);
  alloc_color (window, tw->white_gc, 65535, 65535, 65535);

  tw->blue_gc = gdk_gc_new (window->window);
  alloc_color (window, tw->blue_gc, 20000, 20000, 65535);

  /* 
   * pango stuff (antialiased font rendering
   */

  tw->pfd = pango_font_description_new ();
  pango_font_description_set_family (tw->pfd, "Sans");
  pango_font_description_set_weight (tw->pfd, PANGO_WEIGHT_BOLD);
  pango_font_description_set_size (tw->pfd, 8*PANGO_SCALE);
  tw->pfd_big = pango_font_description_new ();
  pango_font_description_set_family (tw->pfd_big, "Sans");
  pango_font_description_set_weight (tw->pfd_big, PANGO_WEIGHT_BOLD);
  pango_font_description_set_size (tw->pfd_big, 12*PANGO_SCALE);

  tw->pl = gtk_widget_create_pango_layout (tw->w, NULL);
  pango_layout_set_font_description (tw->pl, tw->pfd);

  tw->pl_big = gtk_widget_create_pango_layout (tw->w, NULL);
  pango_layout_set_font_description (tw->pl_big, tw->pfd_big);

  gtk_timeout_add (1000, update_time_widget_cb, tw); 
}
