/*
 * Copyright (C) 2003-2005 the xine project
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * $Id: snapshot.c,v 1.10 2005/01/05 20:35:22 dsalt Exp $
 *
 * snapshot implementation
 *
 * code based on:
 *    totem     Bastien Nocera <hadess@hadess.net>
 *    transcode Copyright (C) Thomas Oestreich - June 2001
 *    enix      enix.berlios.de
 */

#include "globals.h"

#include <stdlib.h>
#include <string.h>
#include <stdio.h>

#include <inttypes.h>

#include <gdk/gdk.h>
#include <gtk/gtk.h>

#include "snapshot.h"
#include "menu.h"
#include "utils.h"

#define clip_8bit(val) ((val) < 0 ? 0 : (val) > 255 ? 255 : (val))

static GtkWidget *bbox, *scalebtn, *blendbtn;

/*
 *   Create rgb data from yuy2
 */

static uint8_t *yuy2torgb (const uint8_t *data,
			   const int width, const int height)
{
  /* Packed YUV 422 is: Y1, U, Y2, V for each (horizontal) pixel pair.
   * Left pixel is Y1,U,V, right pixel is Y2,U,V.
   */

  int i;
  guchar *rgb, *ptr;

  ptr = rgb = g_malloc (width * height * 3);
  if (!rgb)
    return NULL;

  for (i = 0; i < width * height / 2; ++i)
  {
    int y = (data[0] - 16) * 11644;
    int u = data[1] - 128;
    int v = data[3] - 128;

    *ptr++ = clip_8bit ((y + 15960 * v) / 10000);		/* R */
    *ptr++ = clip_8bit ((y - 3918 * u - 8130 * v) / 10000);	/* G */
    *ptr++ = clip_8bit ((y + 20172 * u) / 10000);		/* B */

    y = (data[2] - 16) * 11644;
    data += 4;

    *ptr++ = clip_8bit ((y + 15960 * v) / 10000);		/* R */
    *ptr++ = clip_8bit ((y - 3918 * u - 8130 * v) / 10000);	/* G */
    *ptr++ = clip_8bit ((y + 20172 * u) / 10000);		/* B */
  }

  return rgb;
}

/*
 *   Create rgb data from yv12
 */

static uint8_t *yv12torgb (const uint8_t *data,
			   const int width, const int height)
{
  /* YV12 is planar: Y h*w, U (h/2)*(w/2), V (h/2)*(w/2).
   * => Each U and V byte is for 4 pixels (2x2).
   */

  const uint8_t *src_y = data;				/* [h][w] */
  const uint8_t *src_u = data + width * height;		/* [h/2][w/2] */
  const uint8_t *src_v = data + width * height * 5 / 4;	/* [h/2][w/2] */
  int i, j;
  guchar *rgb, *ptr;

  ptr = rgb = g_malloc (width * height * 3);
  if (!rgb)
    return NULL;

  for (i = 0; i < height; ++i)
  {
    for (j = 0; j < width; ++j)
    {
      int y = (src_y[j] - 16) * 11644;
      int u = src_u[j / 2] - 128;
      int v = src_v[j / 2] - 128;

      *ptr++ = clip_8bit ((y + 15960 * v) / 10000);		/* R */
      *ptr++ = clip_8bit ((y - 3918 * u - 8130 * v) / 10000);	/* G */
      *ptr++ = clip_8bit ((y + 20172 * u) / 10000);		/* B */
    }

    src_y += width;
    if (i & 1)
    {
      src_u += width / 2;
      src_v += width / 2;
    }
  }

  return rgb;
}

static GtkItemFactoryEntry blend_items[] = {
  { N_("/Nearest"),	NULL, NULL, 0, NULL },
  { N_("/Tiles"),	NULL, NULL, 0, NULL },
  { N_("/Bilinear"),	NULL, NULL, 0, NULL },
  { N_("/Hyper"),	NULL, NULL, 0, NULL },
};

void make_snapshot (char *fname, int scale, int blend)
{
  uint8_t   *yuv, *rgb;
  int        width, height, ratio, format = 0;
  GdkPixbuf *pixbuf, *scaled;
  double     desired_ratio, image_ratio, f;

  xine_get_current_frame (stream, &width, &height, &ratio, &format, NULL);

  /* catch "no image" and unrecognised formats */

  switch (format) {
  case XINE_IMGFMT_YUY2:
  case XINE_IMGFMT_YV12:
    break;

  case 0: /* no image */
    display_info (_("Snapshot"), _("Sorry, no image"));
    return;

  default:
    display_info (_("Snapshot"), _("Sorry, format '%.4s' is unsupported"),
		  (char *)&format);
    return;
  }

  yuv = malloc ((width+8) * (height+1) * 2);

  xine_get_current_frame (stream, &width, &height, &ratio, &format, yuv);

  if (scale >= 0)
    gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(scalebtn), scale != 0);
  if (blend >= 0 && blend < num_menu_items (blend_items))
    gtk_option_menu_set_history (GTK_OPTION_MENU(blendbtn), blend);

  /* get filename & options */

  if (!fname)
  {
    fname = modal_file_dialog_with_buttons (_("Enter filename for snapshot:"),
					    "*.png", bbox, NULL);
    if (!fname)
    {
      /* cancelled */
      free (yuv);
      return;
    }
    scale = blend = -1;
  }

  /* convert to rgb */

  switch (format) {
  case XINE_IMGFMT_YUY2:
    rgb = yuy2torgb (yuv, width, height);
    break;

  case XINE_IMGFMT_YV12:
    rgb = yv12torgb (yuv, width, height);
    break;

  default:
    printf ("snapshot: eek, how did I get here? (format '%.4s')",
	    (char *) &format);
    abort();
  }

  free (yuv); /* we've now finished with this */

  pixbuf = gdk_pixbuf_new_from_data (rgb,
				     GDK_COLORSPACE_RGB, FALSE,
				     8, width, height, 3 * width,
				     (GdkPixbufDestroyNotify) g_free, NULL);

  if (scale < 0)
    scale = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON(scalebtn));
  if (blend < 0)
    blend = gtk_option_menu_get_history (GTK_OPTION_MENU(blendbtn));

  if (!scale && !blend)
  {
    /* no scaling, no blending => simple save */
    gdk_pixbuf_save (pixbuf, fname, "png", NULL, NULL);
    gdk_pixbuf_unref (pixbuf);
    return;
  }

  /* scale the image */

  if (!scale)
  {
    ratio = XINE_VO_ASPECT_DONT_TOUCH;
    image_ratio = 1;
  }
  else
    image_ratio = (double) width / (double) height;

  switch (ratio) {
  case XINE_VO_ASPECT_ANAMORPHIC:  /* anamorphic     */
  case XINE_VO_ASPECT_PAN_SCAN:    /* we display pan&scan as widescreen */
    desired_ratio = 16.0 /9.0;
    break;
  case XINE_VO_ASPECT_DVB:         /* 2.11:1 */
    desired_ratio = 2.11/1.0;
    break;
  case XINE_VO_ASPECT_SQUARE:      /* square pels */
  case XINE_VO_ASPECT_DONT_TOUCH:  /* probably non-mpeg stream => don't touch aspect ratio */
    desired_ratio = image_ratio;
    break;
  default:
    printf (_("vo_scale: unknown/forbidden aspect ratio (%d) in stream => using 4:3\n"),
	    ratio);
  case XINE_VO_ASPECT_4_3:         /* 4:3             */
    desired_ratio = 4.0 / 3.0;
    break;
  }

  f = scale ? (desired_ratio / image_ratio) : 1;

  if (f >= 1.0)
    scaled = gdk_pixbuf_scale_simple (pixbuf,
				      (int) ((double) width * f), height,
				      (GdkInterpType) blend);
  else
    scaled = gdk_pixbuf_scale_simple (pixbuf,
				      width, (int) ((double) height / f),
				      (GdkInterpType) blend);
    
  gdk_pixbuf_unref (pixbuf);
    
  /* 
   * save pixbuf to file
   */

  gdk_pixbuf_save (scaled, fname, "png", NULL, NULL);

  gdk_pixbuf_unref (scaled);
}

void snapshot_init (void)
{
  GtkItemFactory *factory;
  GtkWidget *w;

  bbox = gtk_hbox_new (FALSE, 2);
  gtk_widget_ref (bbox);

  w = gtk_label_new_with_mnemonic (_("_Blend:"));
  gtk_box_pack_start (GTK_BOX(bbox), w, FALSE, FALSE, 2);

  blendbtn = gtk_option_menu_new ();
  gtk_option_menu_set_menu (GTK_OPTION_MENU(blendbtn),
			    create_menu_tree (&factory, NULL, NULL,
					      blend_items,
					      num_menu_items (blend_items)));
  gtk_box_pack_start (GTK_BOX(bbox), blendbtn, FALSE, FALSE, 2);
  gtk_option_menu_set_history (GTK_OPTION_MENU(blendbtn), 2); /* bilinear */

  gtk_label_set_mnemonic_widget (GTK_LABEL(w), blendbtn);

  scalebtn = gtk_check_button_new_with_mnemonic (_("_Scaled"));
  gtk_box_pack_start (GTK_BOX(bbox), scalebtn, FALSE, FALSE, 2);
  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(scalebtn), TRUE);
}
