/*
 * Copyright (C) 2001-2003 the xine project
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * $Id: preferences.c,v 1.19 2003/03/24 16:33:26 guenter Exp $
 *
 * Functions to stup and deal with a preferences dialog interfacing with
 * the xine config system.
 *
 * Richard Wareham <richwareham@users.sourceforge.net> -- March 2002
 */

#include <config.h>
#include <X11/Xlib.h>
#include <gtk/gtk.h>
#include <gdk/gdk.h>
#include <glib.h>
#include <stdio.h>

#include "globals.h"
#include "gtkxine.h"
#include "mediamarks.h"
#include "preferences.h"
#include <xine/xineutils.h>

static GtkWidget  *prefs_dialog;
static int         num_sections;
static char       *sections[20];
static int         is_visible;

#define do_two_columns { \
  gtk_table_attach(GTK_TABLE(table), label, 1,2, entries-1, entries, GTK_SHRINK|GTK_FILL, GTK_SHRINK|GTK_FILL, 5, 5); \
  gtk_table_attach(GTK_TABLE(table), widget, 0,1, entries-1, entries, GTK_FILL,GTK_SHRINK, 2,5); \
   widgets++; \
  gtk_label_set_justify (GTK_LABEL (label), GTK_JUSTIFY_LEFT); \
  gtk_misc_set_alignment (GTK_MISC (label), 0, 0.5);\
  sep = gtk_hseparator_new (); \
  gtk_table_attach(GTK_TABLE(table), sep, 0, 2, entries, entries+1, GTK_SHRINK|GTK_FILL, GTK_SHRINK|GTK_FILL, 5, 5);\
}

#define do_one_column { \
  gtk_table_attach(GTK_TABLE(table), widget, 0,2, entries-1, entries, GTK_EXPAND,0, 2,5); \
   widgets++; \
}

static void entry_cb (GtkEditable *editable, GdkEventFocus *even, 
		      gpointer user_data) {

  xine_cfg_entry_t      entry;
  gchar                *key = (gchar *) user_data;

  printf ("preferences: entry cb for key %s\n", key);

  if (!gtk_xine_config_lookup_entry (GTK_XINE(gtx), key, &entry))
    return;


  printf ("preferences: updating entry\n");

  entry.str_value = gtk_editable_get_chars (editable, 0, -1);

  gtk_xine_config_update_entry (GTK_XINE(gtx), &entry);

}

static void num_entry_cb (GtkEditable *editable, GdkEventFocus *even, 
			  gpointer user_data) {
  
  xine_cfg_entry_t      entry;
  gchar                *key = (gchar *) user_data;
  gchar                *str;
  int                   num;

  printf ("preferences: num entry cb for key %s\n", key);

  if (!gtk_xine_config_lookup_entry (GTK_XINE(gtx), key, &entry))
    return;


  printf ("preferences: updating entry\n");

  str = gtk_editable_get_chars (editable, 0, -1);
  if ( sscanf (str, "%d", &num) != 1)
    return;

  printf ("preferences: updating entry to %d\n", num);

  entry.num_value = num;

  gtk_xine_config_update_entry (GTK_XINE(gtx), &entry);

}

static void check_box_cb (GtkToggleButton *togglebutton,
			  gpointer user_data) {
  xine_cfg_entry_t      entry;
  gchar                *key = (gchar *) user_data;

  printf ("preferences: check box cb for key %s\n", key);

  if (!gtk_xine_config_lookup_entry (GTK_XINE(gtx), key, &entry))
    return;


  printf ("preferences: updating entry\n");

  entry.num_value = gtk_toggle_button_get_active (togglebutton);

  gtk_xine_config_update_entry (GTK_XINE(gtx), &entry);
}

static void range_cb (GtkWidget* widget, gpointer data) {

  xine_cfg_entry_t      entry;
  gchar                *key = (gchar *) data;

  printf ("preferences: check box cb for key %s\n", key);

  if (!gtk_xine_config_lookup_entry (GTK_XINE(gtx), key, &entry))
    return;

  entry.num_value = GTK_ADJUSTMENT(widget)->value;

  printf ("preferences: updating entry to %d\n", entry.num_value);

  gtk_xine_config_update_entry (GTK_XINE(gtx), &entry);
}

static void enum_cb (GtkWidget* widget, gpointer data) {

  xine_cfg_entry_t      entry;
  gchar                *key = (gchar *) data;
  GList                *sel;
  int                   pos;

  printf ("preferences: enum cb for key %s\n", key);

  if (!gtk_xine_config_lookup_entry (GTK_XINE(gtx), key, &entry))
    return;

  sel = GTK_LIST (widget)->selection;

  if (sel) {
    pos = g_list_index (GTK_LIST (widget)->children, sel->data);

    entry.num_value = pos;

    printf ("preferences: updating entry to %d\n", pos);

    gtk_xine_config_update_entry (GTK_XINE(gtx), &entry);
  }
}

GtkWidget* populate_section (int s) {
  /* Return a widget containing all the stuff for the
   * sections[s] page of the preferences dialog */

  xine_cfg_entry_t     entry;
  int                  len, entries, widgets;
  char                *section;
  const char          *labelkey;
  GtkWidget           *scroller;
  GtkWidget           *table=NULL;
  GtkWidget           *widget = NULL;
  GtkWidget           *label  = NULL;

  section = sections[s];
  len     = strlen (section);

  if (!gtk_xine_config_get_first_entry (GTK_XINE(gtx), &entry))
    return NULL;

  entries = 0; widgets = 0;

  table = gtk_table_new (1, 1, FALSE);

  scroller = gtk_scrolled_window_new (NULL, NULL);
  gtk_scrolled_window_add_with_viewport (GTK_SCROLLED_WINDOW (scroller),
					 table);
  gtk_widget_set_usize (scroller, 600, 350);
  gtk_scrolled_window_set_policy  ( GTK_SCROLLED_WINDOW (scroller),
				    GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);

  do {

    if (!strncmp (entry.key, section, len) && entry.description
	&& entry.exp_level<20) {
      char labeltext[256];
      GtkWidget *sep;

      labelkey = &entry.key[len+1];
      entries +=2;

      snprintf(labeltext, 255, "%s\n%s", labelkey, entry.description);

      label = gtk_label_new(labeltext);

      gtk_label_set_justify(GTK_LABEL(label), GTK_JUSTIFY_LEFT);
      /* gtk_misc_set_alignment (GTK_MISC(label), 0.0, 1.0); */
      gtk_label_set_line_wrap (GTK_LABEL(label), TRUE);

      switch (entry.type) {
      case XINE_CONFIG_TYPE_ENUM: {
	GList *items = NULL;
	char  *v;
	int    i;

	i = 0;
	while ( (v = entry.enum_values[i]) ) {
	  items = g_list_append (items, v);
	  i++;
	}

	widget = gtk_combo_new();
	gtk_combo_set_popdown_strings (GTK_COMBO (widget), items);
	gtk_combo_set_value_in_list   (GTK_COMBO (widget), TRUE, FALSE);
	gtk_entry_set_editable        (GTK_ENTRY (GTK_COMBO (widget)->entry), 
				       FALSE); 

	gtk_list_select_item (GTK_LIST (GTK_COMBO (widget)->list), 
			      entry.num_value);

	g_signal_connect (GTK_OBJECT(GTK_COMBO (widget)->list),
			    "selection-changed",
			    G_CALLBACK (enum_cb), entry.key);

	do_two_columns;
      }
	break;
	
      case XINE_CONFIG_TYPE_STRING:
	widget = gtk_entry_new();

	g_signal_connect (GTK_OBJECT (widget), "focus-out-event", 
			    G_CALLBACK (entry_cb), (gchar *) entry.key);


	gtk_entry_set_text(GTK_ENTRY(widget), entry.str_value);

	do_two_columns;
	break;

      case XINE_CONFIG_TYPE_RANGE: /* slider */
	 {
	  GtkAdjustment *adj;

	  adj = GTK_ADJUSTMENT(gtk_adjustment_new(entry.num_value, entry.range_min, entry.range_max,
						  1.0, 10.0, 0.0));
	  widget = gtk_hscale_new(adj);
	  gtk_scale_set_draw_value(GTK_SCALE(widget), TRUE);
	  gtk_scale_set_value_pos(GTK_SCALE(widget), GTK_POS_TOP);
	  gtk_scale_set_digits(GTK_SCALE(widget), 0);

	  g_signal_connect (GTK_OBJECT (adj), "value-changed",
			      G_CALLBACK (range_cb), 
			      (gchar *) entry.key );


	  do_two_columns;
	 }
	break;

      case XINE_CONFIG_TYPE_NUM:
	 {
	  char contents[20];

	  widget = gtk_entry_new();
	  snprintf(contents, 19, "%i", entry.num_value);

	  g_signal_connect (GTK_OBJECT (widget), "focus-out-event", 
			      G_CALLBACK (num_entry_cb), (gchar *) entry.key);

	  gtk_entry_set_text(GTK_ENTRY(widget), contents);

	  do_two_columns;
	 }
	break;

      case XINE_CONFIG_TYPE_BOOL:
	widget = gtk_check_button_new();
	if(entry.num_value == 1) {
	  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(widget), TRUE);
	} else {
	  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(widget), FALSE);
	}
	g_signal_connect (GTK_OBJECT (widget), "toggled", 
			    G_CALLBACK (check_box_cb), 
			    (gchar *) entry.key);

	/* gtk_misc_set_alignment (GTK_MISC(widget), 1.0, 1.0); */

	do_two_columns;
	break;

       default:
	printf("Error, unknown type for entry '%s'\n", entry.key);
	break;
      }
    }
  } while (gtk_xine_config_get_next_entry (GTK_XINE (gtx), &entry));

  if (widgets == 0) 
    return NULL;

  return scroller;
}

void populate_box (GtkWidget *notebook) {

  int                   i;
  xine_cfg_entry_t      entry;

  if (!gtk_xine_config_get_first_entry (GTK_XINE (gtx), &entry))
    return;

  num_sections = 0;

  do {
    char *point;

    point = strchr(entry.key, '.');

    if (point) {
      int found ;
      int i;
      int len;

      len = point - entry.key;
      found = 0;

      for (i=0; i<num_sections; i++) {
	if (!strncmp (sections[i], entry.key, len)) {
	  found = 1;
	  break;
	}
      }

      if (!found) {
	sections[num_sections] = xine_xmalloc (len+1);
	strncpy (sections[num_sections], entry.key, len);
	sections[num_sections][len] = 0;
	num_sections++;
      }
    }

  } while (gtk_xine_config_get_next_entry (GTK_XINE (gtx), &entry));

  for(i=0; i<num_sections; i++) {
    GtkWidget *label, *page;

    page = populate_section (i);

    if (page) {
      label = gtk_label_new(sections[i]);
    
      gtk_notebook_append_page(GTK_NOTEBOOK(notebook),
			       page, label);
    }
  }
}

static void delete_cb (GtkWidget* widget, gpointer data) {
  gtk_widget_hide (widget);
  is_visible = 0;
}

static gboolean close_cb (GtkWidget* widget, gpointer data) {
  gtk_widget_hide (prefs_dialog);
  is_visible = 0;
  return TRUE;
}

void preferences_init (void) {

  GtkWidget *tab_box, *b;

  prefs_dialog = gtk_dialog_new ();
  gtk_window_set_title (GTK_WINDOW (prefs_dialog), "Preferences...");
  gtk_window_set_default_size (GTK_WINDOW (prefs_dialog), 500, 150);
  g_signal_connect( GTK_OBJECT (prefs_dialog), "delete_event",
		      G_CALLBACK (delete_cb), NULL );
  b = gtk_dialog_add_button (GTK_DIALOG (prefs_dialog), 
			     GTK_STOCK_CLOSE, 1);
  g_signal_connect (GTK_OBJECT (b), "clicked", 
		    G_CALLBACK (close_cb), 
		    NULL);

  /* Make new tabbed box (notebook) */
  tab_box = gtk_notebook_new();

  /* Populate dialog box */
  populate_box (tab_box);

  /* Add it to the dialog */
  gtk_box_pack_end (GTK_BOX(GTK_DIALOG(prefs_dialog)->vbox), tab_box,
		    TRUE, TRUE, 0);
  gtk_widget_show_all(tab_box);

  is_visible = 0;
}

void preferences_show (void) {

  printf ("playlist: show, is_visible=%d\n",
	  is_visible);

  if (is_visible) {
    is_visible = FALSE;
    gtk_widget_hide (prefs_dialog);
  } else {
    is_visible = TRUE;
    gtk_widget_show_all (prefs_dialog);
    gtk_widget_map (prefs_dialog);
  }
}

