#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <stdio.h>
#include <ctype.h>
#include <time.h>

#include "support.h"
#include "main.h"
#include "conf.h"
#include "qsodata.h"
#include "macro.h"
#include "interface.h"
#include "trx.h"

#define	NUMMACROS	12

struct macro {
	gchar *name;
	gchar *text;
};

static struct macro macros[NUMMACROS];

#define confcpy(d,s)	{ strncpy((d),(s),sizeof(d)); d[sizeof(d)-1] = 0; }

static GtkWidget *macroconfigwin = NULL;   

/* ---------------------------------------------------------------------- */

static void set_macro_button_names(void)
{
	GtkButton *button;
	GtkLabel *label;
	gchar *str;
	gint i;

	for (i = 0; i < NUMMACROS; i++) {
		str = g_strdup_printf("macrobutton%d", i + 1);
		button = GTK_BUTTON(lookup_widget(appwindow, str));
		label = GTK_LABEL(GTK_BIN(button)->child);
		g_free(str);

		str = g_strdup_printf("%s (F%d)", macros[i].name, i + 1);
		gtk_label_set_text(label, str);
		g_free(str);
	}
}

/* ---------------------------------------------------------------------- */

static void fill_macroconfig(GtkWidget *win, gint n)
{
	GtkEntry *entry;
	GtkTextView *view;
	GtkTextBuffer *buffer;
	gchar *str;

	/* should not happen... */
	if (n < 1 || n > NUMMACROS) {
		g_print("fill_macroconfig: invalid macro number %d\n", n);
		return;
	}

	str = g_strdup_printf("macro %d", n);
	gtk_window_set_title(GTK_WINDOW(win), str);
	g_free(str);

	n--;

	entry = GTK_ENTRY(lookup_widget(win, "macroconfigentry"));
	gtk_entry_set_text(entry, macros[n].name);

	if (!macros[n].text)
		return;

	view = GTK_TEXT_VIEW(lookup_widget(win, "macroconfigtext"));
	buffer = gtk_text_view_get_buffer(view);
	gtk_text_buffer_set_text(buffer, macros[n].text, -1);
}

static void apply_macroconfig(GtkWidget *win)
{
	GtkEditable *editable;
	GtkTextView *view;
	GtkTextBuffer *buf;
	GtkTextIter start, end;
	gchar key[16];
	gint n;

	/*
	 * A very ugly hack... Get the macro number from the window
	 * title which is of the form "macro N". Does anyone have
	 * a better idea?
	 */
	n = atoi(gtk_window_get_title(GTK_WINDOW(win)) + 6);

	if (n < 1 || n > NUMMACROS) {
		g_print("apply_macroconfig: invalid macro number %d\n", n);
		return;
	}

	editable = GTK_EDITABLE(lookup_widget(win, "macroconfigentry"));

	g_free(macros[n - 1].name);
	macros[n - 1].name = gtk_editable_get_chars(editable, 0, -1);

	view = GTK_TEXT_VIEW(lookup_widget(win, "macroconfigtext"));
	buf = gtk_text_view_get_buffer(view);
	gtk_text_buffer_get_bounds(buf, &start, &end);

	g_free(macros[n - 1].text);
	macros[n - 1].text = gtk_text_buffer_get_text(buf, &start, &end, FALSE);

	sprintf(key, "macro/name%d", n);
	conf_set_string(key, macros[n - 1].name);

	sprintf(key, "macro/text%d", n);
	conf_set_string(key, macros[n - 1].text);

	set_macro_button_names();
}

/* ---------------------------------------------------------------------- */

void macroconfig_load(void)
{
	gchar key[16];
	gint i;

	for (i = 0; i < NUMMACROS; i++) {
		sprintf(key, "macro/name%d", i + 1);
		macros[i].name = conf_get_string(key, "");
		sprintf(key, "macro/text%d", i + 1);
		macros[i].text = conf_get_string(key, "");
	}

	set_macro_button_names();
}

/* ---------------------------------------------------------------------- */

static void macroconfig_response_callback(GtkDialog *dialog,
					  gint id,
					  gpointer data)
{
	GError *error = NULL;
	GtkWidget *w;
	GtkTextBuffer *b;

	switch (id) {
	case GTK_RESPONSE_HELP:
		gnome_help_display("gmfsk.xml", "gmfsk-macro", &error);
		if (error != NULL) {
			g_warning(error->message);
			g_error_free(error);
		}
		return;

	case GTK_RESPONSE_REJECT:
		w = lookup_widget(GTK_WIDGET(dialog), "macroconfigentry");
		gtk_editable_delete_text(GTK_EDITABLE(w), 0, -1);
		w = lookup_widget(GTK_WIDGET(dialog), "macroconfigtext");
		b = gtk_text_view_get_buffer(GTK_TEXT_VIEW(w));
		gtk_text_buffer_set_text(b, "", -1);
		break;

	case GTK_RESPONSE_OK:
		apply_macroconfig(GTK_WIDGET(dialog));
		/* note fall-trough */

	default:				/* CANCEL and NONE */
		gtk_widget_destroy(GTK_WIDGET(dialog));
		macroconfigwin = NULL;
		break;
	}
}

void macroconfig(gint n)
{
	if (macroconfigwin) {
		gtk_window_present(GTK_WINDOW(macroconfigwin));
		return;
	}

	macroconfigwin = create_macroconfigdialog();
	fill_macroconfig(macroconfigwin, n);

	g_signal_connect(G_OBJECT(macroconfigwin), "response",
			 G_CALLBACK(macroconfig_response_callback),
			 NULL);

	gtk_widget_show(macroconfigwin);
}

/* ---------------------------------------------------------------------- */

#define	CMDBUFSIZE	4096

static void run_command(gchar *cmd)
{
	gchar *outbuf;
	GError *error = NULL;
	gboolean ret;

	cmd++;				/* skip   '('	*/
	cmd[strlen(cmd) - 1] = 0;	/* delete ')'	*/

	ret = g_spawn_command_line_sync(cmd, &outbuf, NULL, NULL, &error);

	if (ret == FALSE) {
		errmsg("run_command: %s", error->message);
		g_error_free(error);
	}

	/* delete the last end-of-line if there is one */
	if (outbuf[strlen(outbuf) - 1] == '\n')
		outbuf[strlen(outbuf) - 1] = 0;

	send_string(outbuf);
	g_free(outbuf);
}

static void send_time(const gchar *fmt, gboolean utc)
{
	gchar buf[256];
	struct tm *tm;
	time_t t;

	time(&t);

	if (utc)
		tm = gmtime(&t);
	else
		tm = localtime(&t);

	strftime(buf, sizeof(buf), fmt, tm);
	buf[sizeof(buf) - 1] = 0;

	send_string(buf);
}

/* ---------------------------------------------------------------------- */

static gchar *getword(gchar **ptr)
{
	gchar *word, *p;

	if (**ptr == '(') {
		if ((p = strchr(*ptr, ')')) == NULL)
			return NULL;
		p++;
	} else {
		for (p = *ptr; *p && isalnum(*p); p++)
			;
	}

	word = g_memdup(*ptr, p - *ptr + 1);
	word[p - *ptr] = 0;

	*ptr = p;

	return word;
}

void send_macro(gint n)
{
	gchar *p, *str, *word;

	/* should not happen... */
	if (n < 1 || n > NUMMACROS) {
		g_print("send_macro: invalid macro number %d\n", n);
		return;
	}

	str = g_locale_from_utf8(macros[n - 1].text, -1, NULL, NULL, NULL);
	p = str;

	while (p && *p) {
		if (*p == '$') {
			p++;
			word = getword(&p);

			if (word == NULL)
				continue;

			if (word[0] == '(')
				run_command(word);

			if (!strcasecmp(word, "$"))
				send_char('$');

			if (!strcasecmp(word, "soft"))
				send_string(SoftString);

			/*
			 * Buttons
			 */
			if (!strcasecmp(word, "tx")) {
				trx_set_state_wait(TRX_STATE_TX);
				push_button("txbutton");
			}

			if (!strcasecmp(word, "rx"))
				push_button("rxbutton");

			if (!strcasecmp(word, "pause"))
				push_button("pausebutton");

			if (!strcasecmp(word, "abort"))
				push_button("abortbutton");

			/*
			 * My station info
			 */
			if (!strcasecmp(word, "mycall"))
				send_string(conf_get_mycall());

			if (!strcasecmp(word, "myname"))
				send_string(conf_get_myname());

			if (!strcasecmp(word, "myqth"))
				send_string(conf_get_myqth());

			if (!strcasecmp(word, "myloc"))
				send_string(conf_get_myloc());

			if (!strcasecmp(word, "myemail"))
				send_string(conf_get_myemail());

			/*
			 * Time and date
			 */
			if (!strcasecmp(word, "time"))
				send_time(conf_get_timefmt(), FALSE);

			if (!strcasecmp(word, "utctime"))
				send_time(conf_get_timefmt(), TRUE);

			if (!strcasecmp(word, "date"))
				send_time(conf_get_datefmt(), FALSE);

			if (!strcasecmp(word, "utcdate"))
				send_time(conf_get_datefmt(), TRUE);

			/*
			 * QSO data
			 */
			if (!strcasecmp(word, "call"))
				send_string(get_qsocall());

			if (!strcasecmp(word, "band"))
				send_string(get_qsoband());

			if (!strcasecmp(word, "rxrst"))
				send_string(get_qsorxrst());

			if (!strcasecmp(word, "txrst"))
				send_string(get_qsotxrst());

			if (!strcasecmp(word, "name"))
				send_string(get_qsoname());

			if (!strcasecmp(word, "qth"))
				send_string(get_qsoqth());

			if (!strcasecmp(word, "notes"))
				send_string(get_qsonotes());

			/*
			 * Mode
			 */
			if (!strcasecmp(word, "mode"))
				send_string(trx_get_mode_name());

			/* an unknown macro gets ignored */

			g_free(word);
			continue;
		}

		send_char(*p++);
	}

	g_free(str);
}

/* ---------------------------------------------------------------------- */
