/*--------------------------------------------------------
   child-main.c - The main routine for the children.

   Each RTSP connect causes a fork() and call to child_main()
   to handle the traffic for that connection.
--------------------------------------------------------*/


/*--------------------------------------------------------

REALNETWORKS LICENSE AGREEMENT AND WARRANTY 
             DISCLAIMER

_____________________________________________

Free Real-Time Streaming Protocol (RTSP) Firewall 
             Proxy License

IMPORTANT -- READ CAREFULLY: This RealNetworks 
License Agreement ("License Agreement") is a legal 
agreement between you (either an individual or an 
entity) and RealNetworks, Inc.  and its suppliers and 
licensors collectively ("RN") for the software product 
listed above, which includes computer software and 
associated media and printed material, whether provided 
in a physical form or received on-line form ("Software").  
By clicking on the "Accept" button or opening the 
package, you are consenting to be bound by this Agreement.  
If you do not agree to all of the terms of this agreement, 
click the "Do Not Accept" button and, if you received the 
Software by package, return the product to the place of 
purchase.

__________________________________________________________

1. GRANT OF LICENSE.

Subject to the provisions contained in this License Agreement, 
RN hereby grants you a non-exclusive, non-transferable, 
perpetual, worldwide license to use, modify or redistribute the 
Software subject to the following terms and conditions:

(a) The copyright notice (" 1998 RealNetworks, 
Inc.") and this copy of  this License Agreement shall 
appear on all copies and/or any derivative versions 
of the Software you create or distribute.

(b)	You acknowledge and agree that RN is and shall be 
the exclusive owner of all right, title and interest, 
including copyright, in the Software.

All rights not expressly granted to you are reserved to RN.

2.  SOFTWARE MAINTENANCE AND UPGRADES. 

RN is not obligated to provide maintenance or updates to you 
for the Software. However, any maintenance or updates 
provided by RN shall be covered by this Agreement.

3.  DISCLAIMER OF WARRANTY.

The Software is deemed accepted by you.  Because RN is 
providing you the Software for free, the Software is provided 
to you AS IS, WITHOUT WARRANTY OF ANY KIND. TO 
THE MAXIMUM EXTENT PERMITTED BY 
APPLICABLE LAW, REALNETWORKS FURTHER 
DISCLAIMS ALL
WARRANTIES, INCLUDING WITHOUT LIMITATION 
ANY IMPLIED WARRANTIES OF
MERCHANTABILITY, FITNESS FOR A PARTICULAR 
PURPOSE, AND NONINFRINGEMENT. THE ENTIRE 
RISK ARISING OUT OF THE USE OR PERFORMANCE 
OF THE SOFTWARE REMAINS WITH YOU. TO THE 
MAXIMUM EXTENT PERMITTED BY APPLICABLE 
LAW, IN NO
EVENT SHALL REALNETWORKS OR ITS SUPPLIERS 
BE LIABLE FOR ANY CONSEQUENTIAL, INCIDENTAL, 
DIRECT, INDIRECT, SPECIAL, PUNITIVE, OR OTHER 
DAMAGES WHATSOEVER (INCLUDING, WITHOUT 
LIMITATION, DAMAGES FOR LOSS OF BUSINESS 
PROFITS, BUSINESS INTERRUPTION, LOSS OF 
BUSINESS INFORMATION, OR OTHER PECUNIARY 
LOSS) ARISING OUT OF THIS AGREEMENT OR THE 
USE OF OR INABILITY TO USE THE SOFTWARE, EVEN 
IF REALNETWORKS HAS BEEN ADVISED OF THE 
POSSIBILITY OF SUCH DAMAGES. BECAUSE SOME 
STATES/JURISDICTIONS DO NOT ALLOW THE 
EXCLUSION OR LIMITATION OF LIABILITY FOR 
CONSEQUENTIAL OR INCIDENTAL DAMAGES, THE 
ABOVE LIMITATION MAY NOT APPLY TO YOU.

4. INDEMNIFICATION. 

You hereby agree to defend, indemnify, and hold RN, its 
directors, officers, employees and agents, harmless from any 
and all claims, damages, and expenses (including attorneys 
fees and costs) of any nature arising out of the use, 
modification, or redistribution the Software or any derivative 
versions thereof.

5. U.S. GOVERNMENT RESTRICTED RIGHTS AND 
EXPORT RESTRICTIONS. 

The Software is provided with RESTRICTED RIGHTS. Use, 
duplication, or disclosure by the Government is subject to 
restrictions as set forth in subparagraph (a) through (d) of the 
of the Commercial Computer Software-Restricted Rights at 
FAR 52.227-19, as applicable, or subparagraph (c)(1)(ii) of 
The Rights in Technical Data and Computer Software clause 
of DFARS 252.227-7013, and in similar clauses in the NASA 
FAR supplement, as applicable.  Manufacturer is 
RealNetworks, Inc., 1111 Third Avenue, Suite 500, Seattle, 
Washington 98101.  You acknowledge that none of the 
Software or underlying information or technology may be 
downloaded or otherwise exported or re-exported (i) into (or 
to a national or resident of) Cuba, Iraq, Libya, Yugoslavia, 
North Korea, Iran, Syria, Sudan or Angola or any other 
country to which the U.S. has embargoed goods; or (ii) to 
anyone on the U.S. Treasury Department's list of Specially 
Designated Nationals or the U.S. Commerce Department's 
Table of Denial Orders.  By using the Software, you are 
agreeing to the foregoing and you are representing and 
warranting that you are not located in, under the control of, or 
a national or resident or resident of any such country or on any 
such list.

6. GOVERNING LAW; ATTORNEYS FEES. 

This agreement shall be governed by the laws of the State of 
Washington and you further consent to jurisdiction by the state 
and federal courts sitting in the State of Washington. If either 
RN or you employs
attorneys to enforce any rights arising out of or relating to this 
Agreement, the prevailing party shall be entitled to recover 
reasonable attorneys' fees.

8.  ENTIRE AGREEMENT. 

This agreement constitutes the complete and exclusive 
agreement between RN and you with respect to the subject 
matter hereof, and supersedes all prior oral or written 
understandings, communications or agreements not 
specifically incorporated herein.  This agreement may not be 
modified except in a writing duly signed by an authorized 
representative of RN and you.    

Copyright  1997-1998 RealNetworks, Inc. and or its 
suppliers.  1111 Third Avenue, Suite 2900, Seattle, 
Washington 98101 U.S.A.  All rights are reserved.



RTSP Proxy License Agreement 8-98

--------------------------------------------------------*/

#include <stdio.h>
#include <strings.h>
#ifdef NEED_STRING_H
#include <string.h>
#endif /* NEED_STRING_H */
#include <stdlib.h>
#include <signal.h>
#include <unistd.h>
#include <fcntl.h>
#include <netdb.h>
#include <time.h>

#include <errno.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/time.h>
#include <sys/wait.h>

#include <netinet/in.h>
#include <arpa/inet.h>


#include "rtsp.h"
#include "util.h"
#include "simplib.h"
#include "setup.h"


/* a possible return value of inet_addr() */
#if !defined (INADDR_NONE)
#define INADDR_NONE (-1)
#endif /* !defined (INADDR_NONE) */

/* maximum number of file descriptors */
#define DESCRIPTORS  (FD_SETSIZE - 1)


/* structures */


/* UDPTunnel - an object which records a file descriptor which
   is connected to a server or a client and information about the remote
   end of the connection.  Data read from this file descriptor will
   be written to the sibling tunnel and vice versa.
*/
typedef struct _UDPTunnel UDPTunnel;
struct _UDPTunnel
{
  int      udpt_sibling;            /* index of sibling connection */
  int      udpt_sockfd;             /* socket fd of connection */
  unsigned short udpt_port;         /* local port tunnel connects to */
  struct sockaddr_in udpt_sendaddr; /* address to transmit data to */
};



/* CtrlConn - An RTSP Control connection is for talking to a server.
   One is allocated for each proxy<->server RTSP Control connection.
   The rtsp_server[] array is an array of pointers to these objects.
*/
typedef struct _CtrlConn CtrlConn;
struct _CtrlConn
{
   int     fd;    /* file descriptor for the RTSP Control connection */
   char   *obuff; /* output buffer being written to this fd */
   int     olen;  /* length of this buffer */
};


/* prototypes */
static int  add_tunnel              (int sockfd);
static void remove_tunnel           (int conidx);
static void connect_to_server       (char * hostname, unsigned short port, int rtspid);
static void socket_set_keepalive    (int sockfd);
static int  create_udp_tunnels      (int rtspid, unsigned short cport, unsigned short sport);
static char *get_peer_ipaddress     (int sockfd);
static CtrlConn *new_ctrlconn       (void);
       void close_udp_tunnel        (int idx, int sibflag);
       int create_rtp_udp_tunnel    (int port, int toserver, int count, int rtspid, int local);
       int rtp_udp_tunnel_port_ok   (int idx, unsigned short *pport);
       int rtp_rtcp_tunnel_port_ok  (int idx2, int idx1, unsigned short *pport);
       void set_tunnel_siblings     (int idx1, int idx2);
       int get_sibling_index        (int idx);
       void append_rtsp_conn_buff   (int id, char *buff, int len);
       void show_rtsp_conn_buffs    (void);
       void write_rtsp_conn_buffs   (void);




/* rtsp-module callback hooks */
static int  rtsp_connect_to_server (char * hostname, unsigned short port, int rtspid);
static int  rtsp_client_write (int id, char * buff, int n);
static int  rtsp_server_write (int id, char * buff, int n);
static int  rtsp_udp_client_tunnel_open (int rtspid,  unsigned short dport,  unsigned short * pport);
static int  rtsp_udp_server_tunnel_open (int id,  unsigned short dport,  unsigned short * pport);
static void rtsp_udp_tunnel_close (int id, int udp_id);




/* global */
static int    maxfd = -1;               /* top file descriptor */
static int    maxci = -1;               /* top index in the tunnels array */
static UDPTunnel *tunnels[DESCRIPTORS]; /* array of UDPTunnel structure pointers */
static fd_set all_rset;                 /* set of all open sockets */
static int    clientfd = -1;            /* client RTSP Control file descriptor*/
static CtrlConn *rtsp_server[DESCRIPTORS]; /* RTSP Control connections */


/* for forwarding outgoing RTSP traffic to another proxy (defined in main.c) */
extern char proxy_forward_host[];       /* forward to this host */
extern int  proxy_forward_port;         /* forward to this port */



/*----------------------------------------------------------------------
   child_main() -- Main routine for child processes (one per client.)

   Paramaters:
     cfd - The file descriptor for the child processes's RTSP
           Control connection to the client.

   Return: nothing (exits before returning)

   A child process is started for each child<->proxy RTSP connection.
----------------------------------------------------------------------*/
void
child_main (int cfd)
{
  int i=0, sockfd=0, nreads=0, sibidx=0;
  fd_set read_set;
  int rtspid=-1, rtspid2=-1;

  log_message (INFO_LOG, "proxy starting; client_ip=%s", get_peer_ipaddress (cfd));

  clientfd = cfd;
  socket_set_keepalive (clientfd);

  /* install signal handlers */
  set_signal_handler (SIGPIPE, SIG_IGN);

  /* initalize the RTSP proxy module */
  rtsp_class_init 
    ((RTSPServerConnectFunc)  rtsp_connect_to_server,
     (RTSPWriteFunc)          rtsp_client_write,
     (RTSPWriteServersFunc)   write_rtsp_conn_buffs,
     (RTSPAppendServerFunc)   append_rtsp_conn_buff,
     (RTSPUDPTunnelOpenFunc)  rtsp_udp_client_tunnel_open,
     (RTSPUDPTunnelOpenFunc)  rtsp_udp_server_tunnel_open,
     (RTSPUDPTunnelCloseFunc) rtsp_udp_tunnel_close
     );

  /* initalize the file descriptor sets */
  for (i = 0; i < DESCRIPTORS; i++)
    tunnels[i] = NULL;

  FD_ZERO(&all_rset);
  FD_SET(clientfd, &all_rset);
  maxfd = clientfd;

  for (i = 0; i < DESCRIPTORS; i++)
    rtsp_server[i] = NULL;


	/* This is our initial, client, RTSP structure.  Others will
           be created as the client establishes connections to servers.
         */
  rtspid = rtsp_new (CLIENT_IDSTR, NULL, CLIENT_OBJECT);



  /* MAIN LOOP FOR CHILD PROCESSES */
  while (1)
    {
      memcpy (&read_set, &all_rset, sizeof(fd_set));


      nreads = select (maxfd + 1, &read_set, NULL, NULL, NULL);
      if (nreads < 0 && errno != EINTR)
	exit_proxy (EXIT_SELECT_ERROR);

      if (nreads <= 0)
	continue;


      /* read data from client */
      if (FD_ISSET(clientfd, &read_set))
	{
	  int n=0, o=0;
	  char buff[MAX_TCP_BUFF];


	  nreads--;


	  if ( (n = read (clientfd, buff, MAX_TCP_BUFF)) > 0)
	    {
#if defined LOG_RTSP_TRAFFIC
              fprintf (stderr, "\n---------- C->P ---------- [pid=%d fd=%d rtspid=%d len=%d]\n", (int)getpid(), clientfd, rtspid, n);
              write (2, buff, n);
#endif /* LOG_RTSP_TRAFFIC */
	      o = rtsp_client_read (rtspid, buff, n);
	    }

	  if (n <= 0 || o < 0)
	    exit_proxy (EXIT_CLIENT_READ_ERROR);
	}

      /* read data from server(s) */
      for (rtspid2=0; rtspid2 <= last_rtsp_id(); ++rtspid2)
        {
          if (rtsp_server[rtspid2] != NULL && FD_ISSET(rtsp_server[rtspid2]->fd, &read_set))
	    {
	      int n=0, o=0;
	      char buff[MAX_TCP_BUFF];


	      nreads--;

	      if ( (n = read (rtsp_server[rtspid2]->fd, buff, MAX_TCP_BUFF/16)) > 0)
	        {
#if defined LOG_RTSP_TRAFFIC
                  fprintf (stderr, "\n---------- S->P ---------- [pid=%d fd=%d rtspid=%d len=%d]\n", (int)getpid(), rtsp_server[rtspid2]->fd, rtspid2, n);
                  write (2, buff, n);
#endif /* LOG_RTSP_TRAFFIC */
	          o = rtsp_server_read (rtspid2, buff, n);
	        }

	      if (n <= 0 || o < 0)
	        exit_proxy (EXIT_SERVER_READ_ERROR);
	    }
        }


      /* process tunnels */
      for (i = 0; i <= maxci; i++)
	{
	  if (!tunnels[i])
	    continue;

	  sockfd = tunnels[i]->udpt_sockfd;
	  sibidx = tunnels[i]->udpt_sibling;


	  /* process active file descriptors for UDP tunnels */
	  if (FD_ISSET(sockfd, &read_set))
	    {
	      int n=0, o=0;
	      char buff[MAX_UDP_BUFF];

      
	      nreads--;
      
	      if ( (n = recvfrom (sockfd, buff, MAX_UDP_BUFF, 0, NULL, NULL)) > 0)
                {
                    if ( sibidx == -1 || ! tunnels[sibidx] )
                      { /* can't forward it to a sibling we don't have... */
                        log_message (DEBUG_LOG, "UDP: Dropping packet since sibling is undefined [idx=%d sibidx=%d]", i, sibidx);
                        continue;
                      }
		    o = sendto (tunnels[sibidx]->udpt_sockfd,
			      buff, n, 0,
			      (struct sockaddr *) &tunnels[i]->udpt_sendaddr,
			      sizeof (struct sockaddr_in));
                    if (o < 0)
                      log_message (ERROR_LOG, "sendto() error: errno=%d (%s)",
                                          errno, strerror(errno));
#if defined LOG_EACH_UDP_PACKET
#ifdef DEBUG
                    log_message (DEBUG_LOG, "UDP: RECEIVED %5d BYTES ON PORT %5d, SENT %5d TO PORT %5d", n, tunnels[i]->udpt_port, o, tunnels[sibidx]->udpt_port);
#else
                    log_message (INFO_LOG, "UDP: RECEIVED %5d BYTES ON PORT %5d, SENT %5d TO PORT %5d", n, tunnels[i]->udpt_port, o, tunnels[sibidx]->udpt_port);
#endif /* DEBUG */
#endif /* LOG_EACH_UDP_PACKET */
	        }
	    }
	  
	} /* for */
    } /* while (1) */
}



/*----------------------------------------------------------------------
   add_tunnel() -- Add a UDPTunnel structure to the tunnels[] array.

   Paramaters:
     sockfd - A file descriptor corresponding to the tunnel.

   Return:
     The index of the tunnels[] array that was initialized for this tunnel.
----------------------------------------------------------------------*/      
static int
add_tunnel (int sockfd)
{
  int i=0;
  UDPTunnel *uptr=NULL;


  /* use the first available slot in the tunnels array */
  for (i = 0; i < DESCRIPTORS; i++)
    if (tunnels[i] == NULL)
      {
	uptr = g_malloc (sizeof (UDPTunnel));
	tunnels[i] = uptr;

	uptr->udpt_sibling = -1;
	uptr->udpt_sockfd  = sockfd;

	break;
      }
  
  /* no more slots, then return an error */
  if (i == DESCRIPTORS)
    return -1;
  
  /* recompute the maximum file descriptor number for select() */   
  maxfd = MAX(maxfd, uptr->udpt_sockfd);

  /* calculate the maximum used slot in the tunnels array,
   * this helps us keep things fast
   */
  maxci = MAX(maxci, i);

  FD_SET(sockfd, &all_rset);

  return i;
}



/*----------------------------------------------------------------------
   remove_tunnel() -- Remove a UDPTunnel structure from the tunnels[] array.
   Paramaters:
     idx - The index of the tunnel to remove.

   Return:  nothing
----------------------------------------------------------------------*/
static void
remove_tunnel (int idx)
{
  int i=0;
  UDPTunnel *uptr = tunnels[idx];

  if (!uptr)
    {
      log_message (DEBUG_LOG,
                   "**ERROR** Trying do remove index %d, not used!", idx);
      return;
    }

  tunnels[idx] = NULL;


  /* reset maximum index counters */
  if (idx == maxci)
    {
      for (i = idx - 1; i >= 0; i--)
	if (tunnels[i])
	  break;

      maxci = i;
    }

  if (uptr->udpt_sockfd == maxfd)
    {
      maxfd = clientfd;

      for (i = 0; i <= maxci; i++)
	if (tunnels[i])
	  maxfd = MAX (maxfd, tunnels[i]->udpt_sockfd);
    }

  FD_CLR(uptr->udpt_sockfd, &all_rset);
  g_free (uptr);
}



/*----------------------------------------------------------------------
   connect_to_server() -- Establish a RTSP Control connection to a server.

   Paramaters:
     hostname - The name of the server to connect to.
     port - The port on the server to connect to.
     rtspid - An unique identifier for this proxy<->server RTSP Control
              connection.

   Return: nothing
----------------------------------------------------------------------*/
static void                                                                   
connect_to_server (char * hostname, unsigned short port, int rtspid)
{                                                          
  struct sockaddr_in server_addr;
  int n=0;
  struct hostent *host=NULL;
  unsigned long ip=0;
  int *serverfd_p=NULL;
  extern unsigned int server_port;   /* defined and assigned in main.c */
  extern struct hostent *proxy_host; /* defined and assigned in main.c */

  rtsp_server[rtspid] = new_ctrlconn();
  serverfd_p = &(rtsp_server[rtspid]->fd);

  /* log_message (DEBUG_LOG,
                  "rtspid=%d serverfd_p=%x *serverfd_p=%d",
                  rtspid, serverfd_p, *serverfd_p); */

  /* If we're forwarding everything to another proxy, don't look up
   * the real hostname, look up the other proxy's hostname so everything
   * goes there.
   */
  if (proxy_forward_host[0])
  {
    strncpy (hostname, proxy_forward_host, HOSTNAME_MAX);
    proxy_forward_host[HOSTNAME_MAX-1] = '\0';
    port = proxy_forward_port;
  }

  log_message (DEBUG_LOG, "opening RTSP Control to host=%s port=%d",
               hostname, port);

  
  /* perform a blocking DNS lookup on hostname */
  if (((ip = inet_addr(hostname)) == INADDR_NONE)  &&
      (strcmp(hostname, "255.255.255.255") != 0))
    {
      if ((host = gethostbyname(hostname)) != NULL)
	{
	  memcpy(&ip, host->h_addr, sizeof(ip));
	}
      endhostent();
    }
  
  log_message (DEBUG_LOG, "DNS LOOKUP %s -> 0x%08lx", hostname, ip);


  bzero ((char *) &server_addr, sizeof (struct sockaddr_in));
  memcpy(&server_addr.sin_addr.s_addr, &ip, sizeof(ip));
  server_addr.sin_family = AF_INET;
  server_addr.sin_port   = htons(port);                 


  /* Compare destination with proxy info to prevent looping */

  /* is the destination port the same port we're listening to? */
  if (port == server_port)
    {
      /* compare the official hostname of destination server with ours */
      if (strncmp (host->h_name, proxy_host->h_name, HOSTNAME_MAX) == 0)
        {
          log_message (ERROR_LOG, "Error: Destination host and port is proxy host and port.");
          exit_proxy (EXIT_INFINITE_LOOP);
        }
    }


  /* allocate a socket  */
  if ( (*serverfd_p = socket (AF_INET, SOCK_STREAM, 0)) < 0)
    {
      exit_proxy (EXIT_SOCKET_CREATE_ERROR);
    }

  /* connect our socket to the server address */
  log_message (DEBUG_LOG, "CONNECTING TO %s", hostname);
  if ( (n = connect (*serverfd_p, (struct sockaddr *) &server_addr,
                     sizeof (struct sockaddr_in))) < 0)
  {
    exit_proxy (EXIT_SOCKET_CONNECT_ERROR);
  }

  maxfd = MAX(maxfd, *serverfd_p);
  FD_SET(*serverfd_p, &all_rset);

  socket_set_keepalive (*serverfd_p);

  log_message (INFO_LOG,
               "server=%s(%s) server_port=%d fd=%d",
               get_peer_ipaddress(*serverfd_p), hostname, port, *serverfd_p);
}





/*----------------------------------------------------------------------
   socket_set_keepalive() -- Set SO_KEEPALIVE on a socket file descriptior.

   Paramaters:
     sockfd - The socket file descriptor to update.

   Return:  nothing
----------------------------------------------------------------------*/
static void
socket_set_keepalive (int sockfd)
{
  int on = 1;

  setsockopt (sockfd, SOL_SOCKET, SO_KEEPALIVE, (char *) &on, sizeof (on));
}




/*----------------------------------------------------------------------
   get_peer_ipaddress() -- Given a socket, return the remote IP address.

   Paramaters:
     The socket file descriptor to examine.

   Return:
     A character string indicating the IP address of the remote host.

   Returns static buffer, so copy to use.
----------------------------------------------------------------------*/
static char *
get_peer_ipaddress (int sockfd)
{
  struct sockaddr_in peer_addr;
  ADDRESS_LENGTH len=sizeof (struct sockaddr_in);

  bzero ((char *) &peer_addr, sizeof (struct sockaddr_in));
  getpeername (sockfd, (struct sockaddr *)&peer_addr, &len);
  return inet_ntoa (peer_addr.sin_addr);
}



/*----------------------------------------------------------------------
   create_udp_tunnels() -- Create UDP tunnels (this version is for X-REAL-RDT).

   Paramaters:
     rtspid - An identifier for the proxy-to-server connection.
     cport - The port on the client to connect to.
     sport - The port on the server to connect to.

   Return:
     The index of the client side of the tunnel.
     exit on error
   
   The client tunnel by definition shoots data at the client.
   The server tunnel shoots any data it receives at the server.
----------------------------------------------------------------------*/
static int
create_udp_tunnels (int rtspid, unsigned short cport, unsigned short sport)
{
  int seridx=0, cliidx=0, csockfd = -1, ssockfd=0;
  ADDRESS_LENGTH len=0;
  struct sockaddr_in addr;

  if ( (csockfd = socket (AF_INET, SOCK_DGRAM, 0)) < 0)
    exit_proxy (EXIT_SOCKET_CREATE_ERROR);

  if ( (ssockfd = socket (AF_INET, SOCK_DGRAM, 0)) < 0)
    exit_proxy (EXIT_SOCKET_CREATE_ERROR);

  cliidx = add_tunnel (csockfd);
  seridx = add_tunnel (ssockfd);
  tunnels[cliidx]->udpt_sibling = seridx;
  tunnels[seridx]->udpt_sibling = cliidx;


  /* bind the client side of the tunnel */
  bzero ((char *) &addr, sizeof (struct sockaddr_in));
  addr.sin_family      = AF_INET;
  addr.sin_addr.s_addr = htonl(INADDR_ANY);
  addr.sin_port        = htons(0);

  if ( bind (csockfd, (struct sockaddr *) &addr,
             sizeof (struct sockaddr_in)) < 0)
    exit_proxy (EXIT_BIND_ERROR);

  /* set forwarding addresses and ports */
  len = sizeof (struct sockaddr_in);
  getsockname (csockfd, (struct sockaddr *) &addr, &len);
  tunnels[cliidx]->udpt_port = ntohs(addr.sin_port);

  log_message (DEBUG_LOG, "CLIENT TUNNEL BOUND TO LOCAL PORT %d",
               ntohs(addr.sin_port));

  len = sizeof (struct sockaddr_in);
  getpeername (clientfd, 
	       (struct sockaddr *) &tunnels[cliidx]->udpt_sendaddr, 
	       &len);
  tunnels[cliidx]->udpt_sendaddr.sin_port = htons(cport);


  /* bind the server side of the tunnel */
  bzero ((char *) &addr, sizeof (struct sockaddr_in));
  addr.sin_family      = AF_INET;
  addr.sin_addr.s_addr = htonl(INADDR_ANY);
  addr.sin_port        = htons(0);

  if ( bind (ssockfd, (struct sockaddr *) &addr,
                       sizeof (struct sockaddr_in)) < 0)
    exit_proxy (EXIT_BIND_ERROR);

  /* set forwarding addresses and ports */
  len = sizeof (struct sockaddr_in);
  getsockname (ssockfd, (struct sockaddr *) &addr, &len);
  tunnels[seridx]->udpt_port = ntohs(addr.sin_port);

  log_message (DEBUG_LOG, "SERVER TUNNEL BOUND TO LOCAL PORT %d",
               ntohs(addr.sin_port));

  len = sizeof (struct sockaddr_in);
  getpeername (rtsp_server[rtspid]->fd, 
	       (struct sockaddr *) &tunnels[seridx]->udpt_sendaddr, 
	       &len);
  tunnels[seridx]->udpt_sendaddr.sin_port = htons(sport);


  return cliidx;
}



/*----------------------------------------------------------------------
   close_udp_tunnel() -- Shut down a UDP tunnel.

   Paramaters:
     idx - The index of the tunnel to shut down.
     sibflag - A true/false flag indicating to shut down the tunnel's sibling

   Return: nothing
----------------------------------------------------------------------*/
void
close_udp_tunnel (int idx, int sibflag)
{
  int sibidx=0;

  if (!tunnels[idx])
    return;

  log_message (DEBUG_LOG, "UDP TUNNEL TEARDOWN(idx=%d sibflag=%d)",
               idx, sibflag);

  if ( sibflag )
  {
    sibidx = tunnels[idx]->udpt_sibling;
    if ((sibidx >= 0) && (tunnels[sibidx] != NULL))
        close (tunnels[sibidx]->udpt_sockfd);
    if (sibidx >= 0)
        remove_tunnel (sibidx);
  }

  close (tunnels[idx]->udpt_sockfd);
  remove_tunnel (idx);
}





/*======================================================================

    RTSP module callback hooks

======================================================================*/



/*----------------------------------------------------------------------
   rtsp_connect_to_server() -- A RTSP module callback for connecting to a server

   Paramaters:
     hostname - The name of the server to connect to.
     port     - The port on the server to connect to.
     rtspid   - An unique identifier for this proxy<->server RTSP Control
                connection.

   Return:
     always 1
----------------------------------------------------------------------*/
static int
rtsp_connect_to_server (char * hostname, unsigned short port, int rtspid)
{
  connect_to_server (hostname, port, rtspid);
  return 1;
}



/*----------------------------------------------------------------------
   rtsp_client_write() -- An RTSP module callback for writing to the client.

   Paramaters:
     id   - The RTSP id for this connection (for logging).
     buff - The buffer to write.
     n    - The size of the buffer.

   Return:
     number of bytes written (possibly zero)
     -1 on error
----------------------------------------------------------------------*/
static int
rtsp_client_write (int id, char * buff, int n)
{

  /* log_message (DEBUG_LOG, "made it to rtsp_client_write() buff=0x%x n=%d",
                  buff, n); */


  if (n <= 0 || buff == NULL)
    return 0;

#if defined LOG_RTSP_TRAFFIC
  fprintf (stderr, "\n---------- P->C ---------- "
           "[pid=%d fd=%d rtspid=%d len=%d]\n",
           (int)getpid(), clientfd, id, n);
  write (2, buff, n);
#endif /* LOG_RTSP_TRAFFIC */
  return (write (clientfd, buff, n));
}



/*----------------------------------------------------------------------
   rtsp_server_write() -- An RTSP module callback for writing to the server.

   Paramaters:
     id   - The RTSP id for this connection (for logging).
     buff - The buffer to write.
     n    - The size of the buffer.

   Return:
     number of bytes written (possibly zero)
     -1 on error
----------------------------------------------------------------------*/
static int 
rtsp_server_write (int id, char * buff, int n)
{
  if (n <= 0 || buff == NULL)
    return 0;

#if defined LOG_RTSP_TRAFFIC
  fprintf (stderr, "\n---------- P->S ---------- "
           "[pid=%d fd=%d rtspid=%d len=%d]\n",
           (int)getpid(), rtsp_server[id]->fd, id, n);
  write (2, buff, n);
#endif /* LOG_RTSP_TRAFFIC */
  return (write (rtsp_server[id]->fd, buff, n));
}



/*----------------------------------------------------------------------
   rtsp_udp_client_tunnel_open() -- An RTSP module callback for opening
                                    a UDP tunnel to the client.

   Paramaters:
     rtspid - The RTSP id for this connection.
     cport  - The port on the client to connect to.
     pport  - The port on the proxy to listen on.

   Return:
     The index of the tunnel opened.
----------------------------------------------------------------------*/
static int
rtsp_udp_client_tunnel_open (int rtspid, unsigned short cport,  unsigned short * pport)
{
  int udp_id=0;

  udp_id = create_udp_tunnels (rtspid, cport, 0);
  *pport = tunnels[udp_id]->udpt_port;

  return udp_id;
}



/*----------------------------------------------------------------------
   rtsp_udp_server_tunnel_open() -- An RTSP module callback for opening
                                    a UDP tunnel to the server.

   Paramaters:
     rtspid - The RTSP id for this connection.
     cport  - The port on the server to connect to.
     pport  - The port on the proxy to listen on.

   Return:
     The index of the tunnel opened.
----------------------------------------------------------------------*/
static int
rtsp_udp_server_tunnel_open (int id, unsigned short sport,  unsigned short * pport)
{
  int udp_id = tunnels[id]->udpt_sibling;

  tunnels[udp_id]->udpt_sendaddr.sin_port = htons(sport);
  *pport = tunnels[udp_id]->udpt_port;

  return udp_id;
}



/*----------------------------------------------------------------------
   rtsp_udp_tunnel_close() -- An RTSP module callback for closing a UDP tunnel.

   Paramaters:
     id - The RTSP id for this connection (unused).
     udp_id - The index of the tunnel to close.

   Return: nothing
----------------------------------------------------------------------*/
static void
rtsp_udp_tunnel_close (int id, int udp_id)
{
  close_udp_tunnel (udp_id, 1);
}



/*----------------------------------------------------------------------
   create_rtp_udp_tunnel() -- Create a UDP tunnel for RTP/AVP/UDP.

   Paramaters:
     port - The port on the remote host to connect to.
     toserver - A true/false flag for whether this is to a client or server.
     count  - A counter value, used only for logging.
     rtspid - The RTSP id for this connection.
     local - The desired local port to bind to.

   Return:
     The index of the tunnel opened.
 
   It is up to the caller to check if the port number is 2*N or 2*N+1.
----------------------------------------------------------------------*/
int
create_rtp_udp_tunnel (int port, int toserver, int count, int rtspid, int local)
{
  int idx=0, sockfd=-1, rtspfd=0;
  ADDRESS_LENGTH len=0;
  struct sockaddr_in addr;

  /* log_message (DEBUG_LOG,
                  "entered create_rtp_udp_tunnel(%d,%d,%d,%d)", 
                  port, toserver, count, rtspid); */

     /* which side of the RTSP connection are we talking to? */
  if (toserver == 1) 
    rtspfd = rtsp_server[rtspid]->fd;
  else
    rtspfd = clientfd;

     /* create a socket */
  if ( (sockfd = socket (AF_INET, SOCK_DGRAM, 0)) < 0)
    exit_proxy (EXIT_SOCKET_CREATE_ERROR);

    /* initialize a tunnel structure for this socket */
  idx = add_tunnel (sockfd);

    /* bind the remote side of the tunnel */
  bzero ((char *) &addr, sizeof (struct sockaddr_in));
  addr.sin_family      = AF_INET;
  addr.sin_addr.s_addr = htonl(INADDR_ANY);
  addr.sin_port        = htons(local);

  if ( bind (sockfd, (struct sockaddr *) &addr,
             sizeof (struct sockaddr_in)) < 0)
    return -1;

     /* set forwarding addresses and ports */
  len = sizeof (struct sockaddr_in);
  getsockname (sockfd, (struct sockaddr *) &addr, &len);
  tunnels[idx]->udpt_port = ntohs(addr.sin_port);

  log_message (DEBUG_LOG, "RTP/AVP/UDP TUNNEL %d BOUND TO LOCAL PORT %d",
		count, ntohs(addr.sin_port));

  len = sizeof (struct sockaddr_in);
  getpeername (rtspfd,
               (struct sockaddr *) &tunnels[idx]->udpt_sendaddr,
               &len);
  tunnels[idx]->udpt_sendaddr.sin_port = htons(port);

  return (idx);

}



/*----------------------------------------------------------------------
   rtp_udp_tunnel_port_ok() -- Verify the first UDP port for RTP/AVP/UDP.

   Paramaters:
     idx - The index of the tunnel structure to examine.
     pport - A short int to assign the port number to.

   Return:
     non-zero on success
     0 on error
   
   For RTP, the RTP Data UDP port number must be divisible by two.
----------------------------------------------------------------------*/
int
rtp_udp_tunnel_port_ok (int idx, unsigned short *pport)
{
    *pport = tunnels[idx]->udpt_port;
    return (tunnels[idx]->udpt_port % 2 == 0) ;
}



/*----------------------------------------------------------------------
   rtp_rtcp_tunnel_port_ok() -- Verify the second UDP port for RTP/AVP/UDP.

   Paramaters:
     idx1 - The index of the first tunnel structure to examine.
     idx2 - The index of the second tunnel structure to examine.
     pport - A short int to assign the second port number to.

   Return:
     non-zero on success
     0 on error

   For RTP, the RTCP Reports port number must not be divisible by 2
   and must be one more than the RTP Data port number, which should
   already be assigned before calling this routine.
----------------------------------------------------------------------*/
int
rtp_rtcp_tunnel_port_ok (int idx2, int idx1, unsigned short *pport)
{
    *pport = tunnels[idx2]->udpt_port;
    return (tunnels[idx2]->udpt_port % 2 != 0 &&
            tunnels[idx2]->udpt_port == tunnels[idx1]->udpt_port + 1) ;
}



/*----------------------------------------------------------------------
   set_tunnel_siblings() -- Set two tunnels to be each others siblings.

   Paramaters:
     idx1 - The index of the first tunnel
     idx2 - The index of the second tunnel

   Return: nothing

   Traffic received on one will be sent to the other.
----------------------------------------------------------------------*/
void
set_tunnel_siblings (int idx1, int idx2)
{
  tunnels[idx1]->udpt_sibling = idx2;
  tunnels[idx2]->udpt_sibling = idx1;
}



/*----------------------------------------------------------------------
   get_sibling_index() -- Return the index for the sibling of a tunnel.

   Paramaters:
     idx - The index of the tunnel structure to examine.

   Return:
     The index of the sibling tunnel
     -1 on error
----------------------------------------------------------------------*/
int
get_sibling_index (int idx)
{
  int sibidx=-1;

  if (tunnels[idx])
    sibidx = tunnels[idx]->udpt_sibling;

  return (sibidx);
}



/*----------------------------------------------------------------------
   new_ctrlconn() -- Allocate an RTSP Control connection structure.

   Paramaters:  nothing

   Return:
     A pointer to the allocated object.
----------------------------------------------------------------------*/
static CtrlConn *
new_ctrlconn (void)
{
  CtrlConn *con = g_malloc (sizeof (CtrlConn));
  if (con)
    {
      con->fd = -1;
      con->olen = 0;
      con->obuff = g_malloc (MAX_TCP_BUFF);
      if (con->obuff == NULL)
        {
          g_free (con);
          con = NULL;
        }
    }
    return (con);
}



/*----------------------------------------------------------------------
   append_rtsp_conn_buff() -- Append a buffer to a server's RTSP Control
                              connection buffer.

   Paramaters:
     id   - The RTSP id for this connection
     buff - The buffer to append.
     len  - The length of the buffer.

   Return: nothing
----------------------------------------------------------------------*/
void
append_rtsp_conn_buff (int id, char *buff, int len)
{

  /* log_message (DEBUG_LOG, "appending %d bytes for server id=%d (buff:0x%x)",
                  len, id, buff); */

  if (rtsp_server[id] != NULL)
    {
      memcpy (rtsp_server[id]->obuff + rtsp_server[id]->olen, buff, len);
      rtsp_server[id]->olen += len;
    }
}



/*----------------------------------------------------------------------
   show_rtsp_conn_buffs() -- Display the pending output buffers for all
                             allocated RTSP Control connection structures.

   Paramaters:  none

   Return:  nothing

   This is for debugging.
----------------------------------------------------------------------*/
void
show_rtsp_conn_buffs (void)
{

#ifdef DEBUG
  int id=0;

  log_message (DEBUG_LOG, "entering show_rtsp_conn_buffs()...");

  for (id=0; id < DESCRIPTORS; ++id)
    {
      if (rtsp_server[id])
        {
          fprintf (stderr, "==== CONNECTION id=%d fd=%d len=%d ====\n",
                   id, rtsp_server[id]->fd, rtsp_server[id]->olen);
          write (1, rtsp_server[id]->obuff, rtsp_server[id]->olen);
          fprintf (stderr, "\n----\n");
        }
    }
#endif /* DEBUG */

}



/*----------------------------------------------------------------------
   write_rtsp_conn_buffs() -- Callback to flush the output buffers for all
                              RTSP Control connections (writing the data
                              to the network).

   Paramaters:  none

   Return:  nothing
----------------------------------------------------------------------*/
void
write_rtsp_conn_buffs (void)
{
  int id=0;

  /* log_message (DEBUG_LOG, "entering write_rtsp_conn_buffs()..."); */

  for (id=0; id < DESCRIPTORS; ++id)
    {
      if (rtsp_server[id])
        {
          rtsp_server_write (id, rtsp_server[id]->obuff, rtsp_server[id]->olen);
          rtsp_server[id]->olen = 0;
        }
    }
}

