/*--------------------------------------------------------
    simplib.c - Set of functions for hashes, doubly linked-lists, and
    memory malloc'ing.

--------------------------------------------------------*/


/*--------------------------------------------------------

REALNETWORKS LICENSE AGREEMENT AND WARRANTY 
             DISCLAIMER

_____________________________________________

Free Real-Time Streaming Protocol (RTSP) Firewall 
             Proxy License

IMPORTANT -- READ CAREFULLY: This RealNetworks 
License Agreement ("License Agreement") is a legal 
agreement between you (either an individual or an 
entity) and RealNetworks, Inc.  and its suppliers and 
licensors collectively ("RN") for the software product 
listed above, which includes computer software and 
associated media and printed material, whether provided 
in a physical form or received on-line form ("Software").  
By clicking on the "Accept" button or opening the 
package, you are consenting to be bound by this Agreement.  
If you do not agree to all of the terms of this agreement, 
click the "Do Not Accept" button and, if you received the 
Software by package, return the product to the place of 
purchase.

__________________________________________________________

1. GRANT OF LICENSE.

Subject to the provisions contained in this License Agreement, 
RN hereby grants you a non-exclusive, non-transferable, 
perpetual, worldwide license to use, modify or redistribute the 
Software subject to the following terms and conditions:

(a) The copyright notice (" 1998 RealNetworks, 
Inc.") and this copy of  this License Agreement shall 
appear on all copies and/or any derivative versions 
of the Software you create or distribute.

(b)	You acknowledge and agree that RN is and shall be 
the exclusive owner of all right, title and interest, 
including copyright, in the Software.

All rights not expressly granted to you are reserved to RN.

2.  SOFTWARE MAINTENANCE AND UPGRADES. 

RN is not obligated to provide maintenance or updates to you 
for the Software. However, any maintenance or updates 
provided by RN shall be covered by this Agreement.

3.  DISCLAIMER OF WARRANTY.

The Software is deemed accepted by you.  Because RN is 
providing you the Software for free, the Software is provided 
to you AS IS, WITHOUT WARRANTY OF ANY KIND. TO 
THE MAXIMUM EXTENT PERMITTED BY 
APPLICABLE LAW, REALNETWORKS FURTHER 
DISCLAIMS ALL
WARRANTIES, INCLUDING WITHOUT LIMITATION 
ANY IMPLIED WARRANTIES OF
MERCHANTABILITY, FITNESS FOR A PARTICULAR 
PURPOSE, AND NONINFRINGEMENT. THE ENTIRE 
RISK ARISING OUT OF THE USE OR PERFORMANCE 
OF THE SOFTWARE REMAINS WITH YOU. TO THE 
MAXIMUM EXTENT PERMITTED BY APPLICABLE 
LAW, IN NO
EVENT SHALL REALNETWORKS OR ITS SUPPLIERS 
BE LIABLE FOR ANY CONSEQUENTIAL, INCIDENTAL, 
DIRECT, INDIRECT, SPECIAL, PUNITIVE, OR OTHER 
DAMAGES WHATSOEVER (INCLUDING, WITHOUT 
LIMITATION, DAMAGES FOR LOSS OF BUSINESS 
PROFITS, BUSINESS INTERRUPTION, LOSS OF 
BUSINESS INFORMATION, OR OTHER PECUNIARY 
LOSS) ARISING OUT OF THIS AGREEMENT OR THE 
USE OF OR INABILITY TO USE THE SOFTWARE, EVEN 
IF REALNETWORKS HAS BEEN ADVISED OF THE 
POSSIBILITY OF SUCH DAMAGES. BECAUSE SOME 
STATES/JURISDICTIONS DO NOT ALLOW THE 
EXCLUSION OR LIMITATION OF LIABILITY FOR 
CONSEQUENTIAL OR INCIDENTAL DAMAGES, THE 
ABOVE LIMITATION MAY NOT APPLY TO YOU.

4. INDEMNIFICATION. 

You hereby agree to defend, indemnify, and hold RN, its 
directors, officers, employees and agents, harmless from any 
and all claims, damages, and expenses (including attorneys 
fees and costs) of any nature arising out of the use, 
modification, or redistribution the Software or any derivative 
versions thereof.

5. U.S. GOVERNMENT RESTRICTED RIGHTS AND 
EXPORT RESTRICTIONS. 

The Software is provided with RESTRICTED RIGHTS. Use, 
duplication, or disclosure by the Government is subject to 
restrictions as set forth in subparagraph (a) through (d) of the 
of the Commercial Computer Software-Restricted Rights at 
FAR 52.227-19, as applicable, or subparagraph (c)(1)(ii) of 
The Rights in Technical Data and Computer Software clause 
of DFARS 252.227-7013, and in similar clauses in the NASA 
FAR supplement, as applicable.  Manufacturer is 
RealNetworks, Inc., 1111 Third Avenue, Suite 500, Seattle, 
Washington 98101.  You acknowledge that none of the 
Software or underlying information or technology may be 
downloaded or otherwise exported or re-exported (i) into (or 
to a national or resident of) Cuba, Iraq, Libya, Yugoslavia, 
North Korea, Iran, Syria, Sudan or Angola or any other 
country to which the U.S. has embargoed goods; or (ii) to 
anyone on the U.S. Treasury Department's list of Specially 
Designated Nationals or the U.S. Commerce Department's 
Table of Denial Orders.  By using the Software, you are 
agreeing to the foregoing and you are representing and 
warranting that you are not located in, under the control of, or 
a national or resident or resident of any such country or on any 
such list.

6. GOVERNING LAW; ATTORNEYS FEES. 

This agreement shall be governed by the laws of the State of 
Washington and you further consent to jurisdiction by the state 
and federal courts sitting in the State of Washington. If either 
RN or you employs
attorneys to enforce any rights arising out of or relating to this 
Agreement, the prevailing party shall be entitled to recover 
reasonable attorneys' fees.

8.  ENTIRE AGREEMENT. 

This agreement constitutes the complete and exclusive 
agreement between RN and you with respect to the subject 
matter hereof, and supersedes all prior oral or written 
understandings, communications or agreements not 
specifically incorporated herein.  This agreement may not be 
modified except in a writing duly signed by an authorized 
representative of RN and you.    

Copyright  1997-1998 RealNetworks, Inc. and or its 
suppliers.  1111 Third Avenue, Suite 2900, Seattle, 
Washington 98101 U.S.A.  All rights are reserved.



RTSP Proxy License Agreement 8-98

--------------------------------------------------------*/



#include <stdlib.h>
#include <stdio.h>

#include "simplib.h"
#include "util.h"





/*======================================================================

	*** HASHES ***

	These routies provide a basic linear hash table implementation.
	This implementation is geared towards data sets which don't have
	an excessive amount of hash key collisions.  Hint:  Try to call
	the init routine with an extra large size to avoid on-the-fly
	hash table resizing.

======================================================================*/


/*----------------------------------------------------------------------
   hash_table_new() -- Create a hash table.

   Paramaters:
     size      - The initial size of the hash table (will be resized as needed).
     thresh    - When the table is 1/thresh full it will be resized.
     fact      - When a resize occurs, the table size will be multiplied
                 by this.
     hash_func - The function to hash keys.
     key_compare_func - The function to compare keys.

   Return:
     A pointer to a new hash table.
     NULL on error.
----------------------------------------------------------------------*/
HashTable *
hash_table_new (unsigned int size, int thresh, int fact,
		HashFunc hash_func, CompareFunc key_compare_func)
{
	int i=0;
	HashTable *ht=NULL;

	if (size <= 0)
		size = HT_DEF_SIZE;

	if ( ! (ht = (HashTable *) mem_malloc (sizeof(HashTable))) )
		return NULL;

		/* This will grow dynamically as needed */
	ht->items = (HashItem *) mem_malloc (sizeof(HashItem) * size);
	ht->size = ( ht->items ? size : 0 );
	for (i=0; i < ht->size; ++i)
	{
		ht->items[i].key = EMPTY_ITEM;
		ht->items[i].value = 0;
	}
	ht->used          = 0;
	ht->rsz_threshold = ( (thresh > 1) ? thresh : HT_THRESHOLD );
	ht->rsz_factor    = ( (fact > 1) ? fact : HT_FACTOR );
	ht->hash_func     = ( hash_func ? hash_func : (HashFunc)int_hash_func );
	ht->key_compare_func = ( key_compare_func ? key_compare_func
			: (CompareFunc)int_equal_func );

	return ht;
}



/*----------------------------------------------------------------------
   hash_table_destroy() -- Free up resources associated with a hash table.

   Paramaters:
     ht - The hash table to destroy.

   Return:
     always NULL
----------------------------------------------------------------------*/
HashTable *
hash_table_destroy (HashTable *ht)
{
	if (ht)
	{
		if (ht->items)
		{
			free (ht->items);
		}

		free (ht);
	}

	return NULL;
}



/*----------------------------------------------------------------------
   hash_table_lookup() -- Lookup the value for a key in a hash table.

   Paramaters:
     ht  - The hash table to examine.
     key - The key to use for the hash table lookup.

   Return:
     A pointer to the associated value
     NULL if not found
----------------------------------------------------------------------*/
void *
hash_table_lookup (HashTable *ht, void *key)
{
	/* startkey is where we started, curkey is where we currently are */
	int startkey=0, curkey=0;
	void *ret = NULL;
	int done = 0;

	if (!ht || !key)
		return ret;

	if (ht->used <= 0)
		return ret;

	startkey = hash_key (ht, key);
	curkey   = startkey;
	while (!done)
	{   
		if (ht->items[curkey].key == EMPTY_ITEM)
			done = 1;

		else if (ht->items[curkey].key != DELETED_ITEM &&
			(ht->key_compare_func)
					(ht->items[curkey].key, key))
		{
			ret = ht->items[curkey].value;
			done = 1;
		}
		else
		{
			++curkey;
			if (curkey == ht->size)
				curkey = 0;
			if (curkey == startkey)  /* wrapped around */
				done = 1;
		}
	}

	return ret;
}



/*----------------------------------------------------------------------
   hash_table_resize() -- Increase the size of a hash table by some multiple N.

   Paramaters:
     ht     - The hash table to resize.
     factor - The hash table's size will be multiplied by this number.

   Return:
     A pointer the new hash table
     NULL on error
----------------------------------------------------------------------*/
HashTable *
hash_table_resize (HashTable *ht, int factor)
{
	HashTable *new_ht=NULL;
	int n=0;

	if (!ht)
		return NULL;

	if (factor <= 0)
		factor = HT_FACTOR;

		/* create a new, larger hash table (temporary) */
        new_ht = hash_table_new (ht->size * factor, ht->rsz_threshold,
			ht->rsz_factor, ht->hash_func, ht->key_compare_func);

		/* copy all the items from the old to the new */
	for (n=0; n < ht->size; ++n)
	{
		if ( ht->items[n].key != EMPTY_ITEM &&
		     ht->items[n].key != DELETED_ITEM )
		{
			hash_table_insert_item ( new_ht,
						 ht->items[n].key,
						 ht->items[n].value ); 
		}
	}

		/* destroy the old hash table items, replacing them
		   with the items from the new hash table, and destroy
		   the new (temporary) hash table we created.
		 */
	free (ht->items);
	ht->items = new_ht->items;
	ht->size = new_ht->size;
	new_ht->items = NULL;
	free (new_ht);
	return ht;
}




/*----------------------------------------------------------------------
   hash_table_insert_item() -- Insert a key/value pair into a hash table.

   Paramaters:
     ht    - The hash table to insert the item into.
     key   - The key identifying this key/value pair.
     value - The corresponding value.

   Return:
     A pointer the updated hash table
----------------------------------------------------------------------*/
HashTable *
hash_table_insert_item (HashTable *ht, void *key, void *value)
{
	int startkey=0, curkey=0;
	int done=0;

	if (!ht || !key)
		return ht;

		/* hash table is too full, so resize it */
	if (ht->used > ht->size / ht->rsz_threshold)
	{
		ht = hash_table_resize (ht, ht->rsz_factor);
	}

	startkey = hash_key (ht, key);
	curkey   = startkey;
	while ( !done )
	{
		/* if the item is empty or deleted, or currently
		   holds the same key...
		 */
		if (ht->items[curkey].key == EMPTY_ITEM    ||
		    ht->items[curkey].key == DELETED_ITEM  ||
		    ((CompareFunc)(ht->key_compare_func))
				(ht->items[curkey].key, key))
		{
			/* assign the item */
			ht->items[curkey].key   = key;
			ht->items[curkey].value = value;
			++ ht->used;
			done = 1;
		}
		else	/* next item */
		{
			++curkey;
			if (curkey == ht->size)
				curkey = 0;
			if (curkey == startkey) /* wrapped around */
				done = 1;
		}
	}

	return ht;
}



/*----------------------------------------------------------------------
   hash_table_delete_item() -- Delete a key/value pair from the hash table.

   Paramaters:
     ht  - The hash table to delete the item from.
     key - The key identifying this key/value pair.

   Return:
     A pointer the updated hash table
----------------------------------------------------------------------*/
HashTable *
hash_table_delete_item (HashTable *ht, void *key)
{
	int startkey=0, curkey=0;
	int done=0;

	if ( !ht || !key )
		return ht;

	startkey = hash_key (ht, key);
	curkey   = startkey;
	while (!done)
	{
		/* if the item is empty or deleted, or currently
		   holds the same key...
		 */
		if (ht->items[curkey].key == EMPTY_ITEM    ||
		    ht->items[curkey].key == DELETED_ITEM  ||
		    (ht->key_compare_func) (ht->items[curkey].key, key))
		{
			/* delete the item */
			ht->items[curkey].key = DELETED_ITEM;
			ht->items[curkey].value = EMPTY_ITEM;
			-- ht->used;
			done = 1;
		}
		else	/* next item */
		{
			++curkey;
			if (curkey == ht->size)
				curkey = 0;
			if (curkey == startkey) /* wrapped around */
				done = 1;
		}
	}

	return ht;
}



/*----------------------------------------------------------------------
   hash_table_foreach() -- Call a routine for every key/value pair.

   Paramaters:
     ht        - The hash table to iterate across.
     func      - The function to call for each key/value pair.
     user_data - Arbitrary user-supplied data to pass to this function.

   Return:
     nothing
----------------------------------------------------------------------*/
void
hash_table_foreach (HashTable *ht, HTblFunc func, void *user_data)
{
	unsigned int i=0;

	if ( !ht )
		return;

	for (i=0; i < ht->size; ++i)
	{
			/* if it contains an item, call func for it */
		if (ht->items[i].key != EMPTY_ITEM    &&
		    ht->items[i].key != DELETED_ITEM )
		{
			func (ht->items[i].key, ht->items[i].value, user_data);
		}
	}
}



/*----------------------------------------------------------------------
   int_hash_func() -- Hashing routine for integers.

   Paramaters:
     x - A pointer to an integer.

   Return:
     The integer the pointer refers to
     0 on error

   The return value will get mod'd with the hash table size.
----------------------------------------------------------------------*/
int
int_hash_func (void *x)
{
	if (x)
	    return *(int *)x;

	return 0;
}



/*----------------------------------------------------------------------
   int_equal_func() -- Returns whether the two referenced integers are equal.

   Paramaters:
     a - A pointer to an integer.
     b - A pointer to an integer.

   Return:
     1 if the two referenced integers are the same
     0 if not equal and on error
----------------------------------------------------------------------*/
int
int_equal_func (void *a, void *b)
{
	if (a == NULL || b == NULL)
		return 0;

	return (*(int *)a == *(int *)b);
}



/*----------------------------------------------------------------------
   str_hash_func() -- Hashing routine for strings.

   Paramaters:
     str - A pointer to a string of characters.

   Return:
     The sum of all the characters in the string.
     0 if str is NULL

   The return value will get mod'd with the hash table size.
----------------------------------------------------------------------*/
int
str_hash_func (void *str)
{
	int ret=0;
	int i=0;
	int len=0;


	if (str)
	{	/* just sum up the characters in the string */
		len = strlen ((char *)str);
		for (i = 0; i < len; ++i)
			ret += *(char *)str;
	}

 	return ret;
}



/*----------------------------------------------------------------------
   str_equal_func() -- Returns whether two strings are equal.

   Paramaters:
     a - A pointer to a string of characters.
     b - A pointer to a string of characters.

   Return:
     1 if the two strings are the same
     0 if not equal and on error
----------------------------------------------------------------------*/
int
str_equal_func (void *a, void *b)
{
	if (a == NULL || b == NULL)
		return 0;

	return (strcmp ((char *)a, (char *)b) == 0);
}





/*======================================================================

	*** DOUBLY LINKED LISTS ***

	A basic doubly linked list implementation.

======================================================================*/



/*----------------------------------------------------------------------
   list_init() -- Return a new linked list, initialized with the first item.

   Paramaters:
     data - A pointer to the data item to use for the first item in the list.

   Return:
     A pointer to the new list.
----------------------------------------------------------------------*/
List *
list_init (void *data)
{
	List *list = (List *) mem_malloc (sizeof(List));
	if (list)
	{
		list->data = data;
		list->next = NULL;
		list->prev = NULL;
	}
        
	return list;
}



/*----------------------------------------------------------------------
   list_delete() -- Delete an entire linked list.

   Paramaters:
     list - A pointer to an item in the linked list to delete.

   Return:
     1 on success
     0 on error

   Starting at any item in the list, the entire list will be deleted.
----------------------------------------------------------------------*/
int
list_delete (List *list)
{
	List *top=list;
	List *next=NULL;

	if (!list)
		return 0;

	while (top->prev)
		top = top->prev;

	while (top)
	{
		next = top->next;
		free (top);
		top = next;
	}

	return 1;
}



/*----------------------------------------------------------------------
   list_append() -- Append a new item to the end of a list.

   Paramaters:
     list - A pointer to an item in the linked list.
     data - A pointer to the data item to append to the end of the list.

   Return:
     A pointer to the updated list.
----------------------------------------------------------------------*/
List *
list_append (List *list, void *data)
{
	List *newitem = list_init (data);
	List *last = list;

	if (list)
	{
		while (last->next) /* find the last item in the list */
			last = last->next;
		newitem->prev = last;
		last->next = newitem;
	}

	return newitem;
}



/*----------------------------------------------------------------------
   list_remove() -- Find and remove an item from a list.

   Paramaters:
     list - A pointer to an item in the linked list.
     data - A pointer to the data item to remove from the list.

   Return:
     A pointer to the updated list.
----------------------------------------------------------------------*/
List *
list_remove (List *list, void *data)
{
	List *item=list;

	if (item) /* start at the head of the list */
	  while (item->prev)
            item = item->prev;

	while (item)
	{
		if (item->data == data) /* found it */
		{
			list = list_remove_item (item);
			item = NULL;
		}
		else
		{
			item = item->next;
		}
	}

	return list;
}



/*----------------------------------------------------------------------
   list_remove_item() -- Remove an item from a list.

   Paramaters:
     item - A pointer to the list item to remove from the list.

   Return:
     A pointer to the updated list.

   This is for when we already know which item to remove.
----------------------------------------------------------------------*/
List *
list_remove_item (List *item)
{
	List *list=NULL;

	if ( ! item )
		return list;

	if (item->prev)
		item->prev->next = item->next;
	if (item->next)
		item->next->prev = item->prev;

        list = (item->next) ? item->next : item->prev;

	free (item);
	return list;
}





/*======================================================================

	*** MEMORY ALLOCATION ***

	This provides an extra-paranoid malloc wrapper which exits
	if the malloc fails.

======================================================================*/



/*----------------------------------------------------------------------
   mem_malloc() -- A safety wrapper for malloc().

   Paramaters:
     size - The amount of memory to allocate.

   Return:
     A pointer to the allocated memory.

   If the memory allocation fails, we exit.
----------------------------------------------------------------------*/
void *mem_malloc (size_t size)
{
  void *mem = malloc (size);
  if (!mem)
    exit_proxy (EXIT_MALLOC_ERROR);
  return mem;
}


/*======================================================================*/
