/* $Id$
******************************************************************************

   LibGIC - Recognizers

   Copyright (C) 1999 Andreas Beck	[becka@ggi-project.org]

   Permission is hereby granted, free of charge, to any person obtaining a
   copy of this software and associated documentation files (the "Software"),
   to deal in the Software without restriction, including without limitation
   the rights to use, copy, modify, merge, publish, distribute, sublicense,
   and/or sell copies of the Software, and to permit persons to whom the
   Software is furnished to do so, subject to the following conditions:

   The above copyright notice and this permission notice shall be included in
   all copies or substantial portions of the Software.

   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
   THE AUTHOR(S) BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
   IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
   CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

******************************************************************************
*/

#include <string.h>

#include <ggi/gg.h>
#include <ggi/internal/gic.h>

int _gicRecognizerDriverRegister(gic_handle_t hand,
				 gic_recognizerdriver * driver,
				 void *handle)
{
	gic_recognizerlist *listitem;

	if (NULL == (listitem = malloc(sizeof(gic_recognizer))))
		return GGI_ENOMEM;
	listitem->next = hand->reclist;
	listitem->handle = handle;
	listitem->driver = driver;
	hand->reclist = listitem;
	return 0;
}

int gicRecognizerDriverRegister(gic_handle_t hand,
				gic_recognizerdriver * driver)
{
	return _gicRecognizerDriverRegister(hand, driver, NULL);
}

int gicRecognizerDriverUnregister(gic_handle_t hand,
				  gic_recognizerdriver * driver)
{
	gic_recognizerlist *listitem, **last;

	for (listitem = *(last = &(hand->reclist));
	     listitem != NULL; listitem = *(last = &listitem->next)) {
		if (listitem->driver == driver) {

			if (listitem->handle)
				ggFreeModule(listitem->handle);

			*last = listitem->next;
			free(listitem);
			return 0;
		}
	}
	return GGI_ENOMATCH;
}

gic_recognizerdriver *gicRecognizerDriverLookup(gic_handle_t hand,
						const char *name)
{
	gic_recognizerlist *listitem;

	for (listitem = hand->reclist;
	     listitem != NULL; listitem = listitem->next) {
		if (0 == strcmp(listitem->driver->drivername, name)) {
			return listitem->driver;
		}
	}
	return NULL;
}

int gicRecognizerWrite(gic_handle_t hand, gic_recognizer * recognizer,
		       FILE * where)
{
	char buffer[1024];
	fprintf(where, "gic:        Recognizer \"%s\"\n",
		recognizer->driver->drivername);
	recognizer->driver->write_pvtdata(hand, recognizer, buffer,
					  (int)sizeof(buffer));
	fprintf(where, "gic:          \"%s\"\n", buffer);
	fprintf(where, "gic:        Recognizer END\n");

	return 0;
}

int gicRecognizerTrain(gic_handle_t hand, gic_recognizer ** recognizerlist,
		       gii_event * event)
{
	gic_recognizer *listitem, *lastitem;
	gic_recognizerlist *listitem2;
	int rc = 0;
	int rc1;

	if (event == NULL) {	/* Start a new training sequence. */

		for (listitem = *recognizerlist; listitem != NULL;) {
			if (listitem->privdata)
				free(listitem->privdata);
			lastitem = listitem;
			listitem = listitem->next;
			free(lastitem);
		}
		*recognizerlist = NULL;
	}
	for (listitem2 = hand->reclist; listitem2 != NULL;
	     listitem2 = listitem2->next) {
		rc1 =
		    listitem2->driver->train(hand, recognizerlist, event);
		if (rc1 < 0)
			return rc1;
		else
			rc += rc1;
	}
	return rc;
}

int gicRecognizerTrainAdd(gic_handle_t hand,
			  gic_recognizer ** recognizerlist,
			  gic_recognizer * newone)
{
	gic_recognizer *listitem, **last;

	for (listitem = *(last = recognizerlist);
	     listitem != NULL; listitem = *(last = &listitem->next)) {
		if (listitem->confidence < newone->confidence) {
			*last = newone;
			newone->next = listitem;
			return 0;
		}
	}
	*last = newone;
	newone->next = NULL;
	return 0;
}

int gicRecognizerTrainMove(gic_handle_t hand,
			   gic_recognizer ** recognizerlist,
			   gic_recognizer * theone)
{
	gic_recognizer *listitem, **last;

	for (listitem = *(last = recognizerlist);
	     listitem != NULL; listitem = *(last = &listitem->next)) {
		if (listitem == theone) {
			*last = theone->next;
			return gicRecognizerTrainAdd(hand, recognizerlist,
						     theone);
		}
	}
	return GGI_ENOMATCH;
}

int gicRecognizerFindConflict(gic_handle_t hand,
			      gic_recognizer * recognizer,
			      gic_recognizer * compare_to)
{
	return recognizer->driver->check_conflict(hand, recognizer,
						  compare_to);
}

int gicRecognizerGetName(gic_handle_t hand, gic_recognizer * recognizer,
			 char *string, size_t maxlen)
{
	return recognizer->driver->get_name(hand, recognizer, string,
					    maxlen);
}

int gicRecognizerGetOpposite(gic_handle_t hand,
			     gic_recognizer * recognizer,
			     gic_recognizer ** opposite)
{
	return recognizer->driver->get_opposite(hand, recognizer,
						opposite);
}
