# `rq script`

*[back to index](./README.md)*

`rq script` allows `rq` to run a complete Rego file, providing access to all of the options and features as the `rq query` command. This can be valuable if you have a lengthy query, or want to save your query for repeated use.

```plain
$ cat helloworld.rego
package helloworld

# rq: query data.helloworld.output
# rq: output-format yaml

output := {"message": sprintf("Hello, %s world!", ["rq"])}
$ rq script ./helloworld.rego
message: Hello, rq world!
```

## Configuring `rq` in script mode

Because command line arguments cannot be passed when `rq` is running in script mode, parameters are instead provided in specially formatted comments. A **directive** is a line that starts with `# rq:`. Everything after `rq:` and up to the first space is the **parameter**, and everything from the second space on is the **value**.

For example, this line...

```
# rq: Hello rq world!
```

corresponds to the parameter `Hello` and the value `rq world!`.

Note that this means comments are not allowed to share lines with directives.
This line:

```
# rq: Hello rq world! # my comment
```

would be interpreted as the parameter `Hello` with the value `rq world! # my
comment`. This behavior was chosen to avoid difficulties with text escaping
when `#` symbols are desired in a value.

The same parameter can be specified multiple times. With some parameters such as `data-paths`, this can allow for multiple values to be specified. In all other cases, specifying the same parameter multiple times causes the last (lowest down in the file) value to be used. Many parameters correspond to flags supported by the `rq` command, see [`rq --help`](./help.md) for details on them.

| Parameter          | Multi Values? | Value Type | Corresponding Flag |
|--------------------|---------------|------------|--------------------|
| `query`            |               | string     | `<query>`          |
| `input`            |               | string     | `--input`          |
| `in-place`         |               | string     | `--in-place`       |
| `input-format`     |               | string     | `--input-format`   |
| `output`           |               | string     | `--output`         |
| `output-format`    |               | string     | `--output-format`  |
| `no-color`         |               | bool       | `--no-color`       |
| `force-color`      |               | bool       | `--force-color`    |
| `indent`           |               | string     | `--indent`         |
| `style`            |               | string     | `--style`          |
| `csv-comma`        |               | string     | `--csv-comma`      |
| `csv-comment`      |               | string     | `--csv-comment`    |
| `csv-skip-lines`   |               | int        | `--csv-skip-lines` |
| `csv-headers`      |               | bool       | `--csv-headers`    |
| `csv-no-infer`     |               | bool       | `--csv-no-infer`   |
| `profile`          |               | bool       | `--profile`        |
| `data-paths`       | yes           | string...  | `--data`           |
| `bundle-paths`     | yes           | string...  | `--bundle`         |
| `raw`              |               | bool       | `--raw`            |
| `silent-query`     | yes           | string...  |                    |
| `save-bundle`      |               | string     | `--save-bundle`    |
| `template`         |               | string     | `--template`       |
| `check`            |               | bool       | `--check`          |
| `v0-compatible`    |               | bool       | `--v0-compatible`  |

If the `query` parameter is omitted, then the default query is `data.<package name>` where `<package name>` is the package for the Rego file.

For directives which correspond to an `rq query` flag, the same default value is used for both.

The following example demonstrates specifying multiple values for the `data-paths` parameter:

```plain
$ cat directives.rego
package directives

# rq: data-paths csv.headers=true:foo.csv
# rq: data-paths yaml:bar.yaml
# rq: query data.directives

from_csv := data.foo

from_yaml := data.bar
$ cat foo.csv
col1,col2
1,spam
5,ham
$ cat bar.yaml
- bar
- baz
- quux
$ rq script directives.rego
{
	"from_csv": [
		{
			"col1": 1,
			"col2": "spam"
		},
		{
			"col1": 5,
			"col2": "ham"
		}
	],
	"from_yaml": [
		"bar",
		"baz",
		"quux"
	]
}
```



## Using `rq` as a shebang

`rq` can also be used as a shebang:

```plain
$ cat helloworld.rego
#!/usr/bin/env rq

package helloworld

# rq: query data.helloworld.output
# rq: output-format yaml

output := {"message": sprintf("Hello, %s world!", ["rq"])}
$ chmod +x helloworld.rego
$ ./helloworld.rego
message: Hello, rq world!
```

If `rq` is invoked without a subcommand, and with a first argument that is a file which exists, it will assume the `rq script` subcommand, rather than `rq query` as it otherwise would.

## Accepting CLI Arguments

Any arguments given to the script on the commandline are exposed to the Rego program via the [`rq.args()` builtin](./builtins.md#coderqargscode).

`rq` does not currently include a builtin for parsing CLI arguments, however a
passable version of argument parsing can be implemented using Rego fairly
concisely:

```plain
$ cat flagtest.rego 
#!/usr/bin/env rq

# rq: query data.script.flags

args := rq.args()

flags := {
        "foo": object.get({"0": args[i+1] | args[i] == ["--foo", "-f"][_]}, "0", ""),
        "bar": object.get({"0": args[i+1] | args[i] == ["--bar", "-b"][_]}, "0", ""),
        "baz": object.get({"0": args[i+1] | args[i] == ["--baz", "-B"][_]}, "0", "defaultval"),
}

$ ./flagtest.rego  -b hi --foo frob
{
	"bar": "hi",
	"baz": "defaultval",
	"foo": "frob"
}
$ ./flagtest.rego --foo frob
{
	"bar": "",
	"baz": "defaultval",
	"foo": "frob"
}
$ ./flagtest.rego --foo frob -B 123
{
	"bar": "",
	"baz": "123",
	"foo": "frob"
}
```

## `package` In Scripts

Unlike in a normal Rego file, the `package` statement is optional in an `rq` script. If it is omitted, then `rq` will insert `package script` at the top of the file before evaluating it.

```plain
$ cat nopackage.rego
#!/usr/bin/env rq

# rq: query data

foo := "bar"
$ ./nopackage.rego
{
	"script": {
		"foo": "bar"
	}
}
$ cat withpackage.rego
#!/usr/bin/env rq

# rq: query data

package sample

baz := "quux"

$ ./withpackage.rego
{
	"sample": {
		"baz": "quux"
	}
}
```

## Forcing Side-Effects With `silent-query`

**WARNING:** This feature is intended for advanced users. It is a severe abuse of Rego's semantics, and is easy to mess up. Using this feature correctly and safely requires understanding implementation details of the Rego compiler. Proceed with caution.

**If you do not know what you are doing, see the *Alternatives* section below for safer ways to accomplish this functionality.**

The `silent-query` directive allows you to specify additional Rego queries that `rq` should run when executing your script. The result of executing these queries will be discarded (though errors will still be reported). The purpose of this directive is to allow you to force certain rules to be executed even if you don't care about their actual values in your output. Typically, this would be done because you want to force some side effect to happen by executing an `rq.run()` or `http.send()` call.

An example of usage is provided in the [`silent-query1` smoketest](../smoketest/silent-query1/script.rego).

`rq` will run each silent query one at a time before the main query is executed, in the order they appear. Note however that Rego, by design, does not grantee any order of execution for rules, nor even that a particular rule will run (if the compiler believes it can be optimized out), hence the danger of using this feature.

For best results, ensure any rule heads you are querying are guaranteed to always be defined, and that the result (e.g. status) of any side-effect causing functions you call is included in full within the values assigned to those rule heads. This will ensure that executing your desired side effects does not get optimized away.

### Alternatives

A safer way to ensure side effects of your `rq` script are properly executed is to not directly execute the relevant commands with `rq.run()`, but to instead have `rq` generate commands to be run by a wrapper script and include them in its output. The wrapper script can then ensure that the expected number of commands are generated and that they are run properly. This will also allow you to exercise better control over the order of execution of those commands, if you so wish.

An example of this might be something like:

```plain
$ cat script.rego 
#!/usr/bin/env rq

# rq: query data.script.commands
# rq: raw true

numbers := [1, 2, 3, 4]

commands := {concat(" ", ["echo", sprintf("%d", [x])]) | x := numbers[_]}
$ cat wrapper.sh 
#!/usr/bin/env sh

set -e
set -u
cd "$(dirname "$0")"

./script.rego | while read -r cmd ; do
	sh -c "$cmd"
done
$ ./script.rego 
echo 1
echo 2
echo 3
echo 4
$ ./wrapper.sh 
1
2
3
4
```

## See Also

* [`rq script --help`](./help.md#coderq-script---helpcode)
