// Package builtins contains rq-specific builtins.
package builtins

import (
	"bytes"
	"context"
	"encoding/json"
	"errors"
	"fmt"
	"io"
	"io/fs"
	"os"
	"os/exec"
	"path/filepath"
	"strconv"
	"strings"
	"text/template"
	"time"

	"git.sr.ht/~charles/rq/internal"
	rqio "git.sr.ht/~charles/rq/io"
	"git.sr.ht/~charles/rq/util"
	"git.sr.ht/~charles/rq/version"
	"github.com/araddon/dateparse"
	"github.com/bcicen/go-units"
	"github.com/mozillazg/go-slugify"
	"github.com/open-policy-agent/opa/v1/ast"
	"github.com/open-policy-agent/opa/v1/rego"
	"github.com/open-policy-agent/opa/v1/topdown"
	"github.com/open-policy-agent/opa/v1/topdown/builtins"
	"github.com/open-policy-agent/opa/v1/types"
)

// Args is exposed to the script via the rq.args() builtin.
var Args = []string{}

// ScriptPath is exposed to the script via the rq.scriptpath() builtin.
var ScriptPath = ""

var templateFuncs template.FuncMap

var run = &ast.Builtin{
	Name:        "rq.run",
	Description: "Run a shell command.",
	Decl: types.NewFunction(
		types.Args(
			types.Named("command", types.A).Description("command to run"),
			types.Named("options", types.NewObject(nil, types.NewDynamicProperty(types.A, types.A))),
		),
		types.Named("result", types.NewObject(nil, types.NewDynamicProperty(types.A, types.A))),
	),
}

var tree = &ast.Builtin{
	Name:        "rq.tree",
	Description: "Obtain a file tree rooted at the given path.",
	Decl: types.NewFunction(
		types.Args(
			types.Named("path", types.S).Description("path where tree should be obtained"),
			types.Named("options", types.NewObject(nil, types.NewDynamicProperty(types.A, types.A))),
		),
		types.Named("result", types.NewObject(nil, types.NewDynamicProperty(types.A, types.A))),
	),
}

var env = &ast.Builtin{
	Name:        "rq.env",
	Description: "Returns system environment variables as an object.",
	Decl: types.NewFunction(
		types.Args(),
		types.Named("result", types.NewObject(nil, types.NewDynamicProperty(types.S, types.S))),
	),
}

var args = &ast.Builtin{
	Name:        "rq.args",
	Description: "Returns CLI arguments as an array of strings.",
	Decl: types.NewFunction(
		types.Args(),
		types.Named("result", types.NewArray(nil, types.S)),
	),
}

var berror = &ast.Builtin{
	Name:        "rq.error",
	Description: "Forces rq to exit with an error.",
	Decl: types.NewFunction(
		types.Args(types.Named("message", types.S).Description("error message to display")),
		types.Named("result", types.B),
	),
}

var decode = &ast.Builtin{
	Name:        "rq.decode",
	Description: "Decode a string with an rq input handler.",
	Decl: types.NewFunction(
		types.Args(
			types.Named("input", types.S).Description("string to decode"),
			types.Named("spec", types.A).Description("dataspec"),
		),
		types.Named("result", types.A),
	),
}

var encode = &ast.Builtin{
	Name:        "rq.encode",
	Description: "Encode data to a string with an rq output handler.",
	Decl: types.NewFunction(
		types.Args(
			types.Named("input", types.A).Description("data to encode"),
			types.Named("spec", types.A).Description("dataspec"),
		),
		types.Named("result", types.A),
	),
}

var read = &ast.Builtin{
	Name:        "rq.read",
	Description: "Read a file with an rq input handler.",
	Decl: types.NewFunction(
		types.Args(
			types.Named("spec", types.A).Description("dataspec"),
		),
		types.Named("result", types.A),
	),
}

var write = &ast.Builtin{
	Name:        "rq.write",
	Description: "Write a file with an rq output handler.",
	Decl: types.NewFunction(
		types.Args(
			types.Named("value", types.A).Description("data to write"),
			types.Named("spec", types.A).Description("dataspec"),
		),
		types.Named("result", types.B).Description("always `true`"),
	),
}

var parsedate = &ast.Builtin{
	Name:        "rq.parsedate",
	Description: "Parse a date into a number of nanoseconds.",
	Decl: types.NewFunction(
		types.Args(
			types.Named("date", types.S).Description("date to parse"),
		),
		types.Named("result", types.N).Description("representation of date in ns"),
	),
}

var bversion = &ast.Builtin{
	Name:        "rq.version",
	Description: "Retrieve the rq version info object.",
	Decl: types.NewFunction(
		types.Args(),
		types.Named("result", types.A).Description("rq version info object"),
	),
}

var convert = &ast.Builtin{
	Name:        "rq.convert",
	Description: "Perform unit conversion.",
	Decl: types.NewFunction(
		types.Args(
			types.Named("amount", types.N).Description("quantity in 'from' units"),
			types.Named("from", types.S).Description("units to convert from"),
			types.Named("to", types.S).Description("units to convert to"),
		),
		types.Named("result", types.N).Description("quantity in 'to' units"),
	),
}

var abs = &ast.Builtin{
	Name:        "rq.abs",
	Description: "Find the absolute representation of a file path.",
	Decl: types.NewFunction(
		types.Args(
			types.Named("path", types.S).Description("Filepath to find the absolute version of."),
		),
		types.Named("result", types.S).Description("Absolute representation of the path."),
	),
}

var base = &ast.Builtin{
	Name:        "rq.base",
	Description: "Find the last element of the given file path.",
	Decl: types.NewFunction(
		types.Args(
			types.Named("path", types.S).Description("Path to find the base of."),
		),
		types.Named("result", types.S).Description("Last element of the given file path"),
	),
}

var ext = &ast.Builtin{
	Name:        "rq.ext",
	Description: "Find the extension of the given path.",
	Decl: types.NewFunction(
		types.Args(
			types.Named("path", types.S).Description("Path to find the extension of."),
		),
		types.Named("result", types.S).Description("Extension for the given path."),
	),
}

var dir = &ast.Builtin{
	Name:        "rq.dir",
	Description: "Find all but the last element of the given path.",
	Decl: types.NewFunction(
		types.Args(
			types.Named("path", types.S).Description("Path to take the Dir of."),
		),
		types.Named("result", types.S).Description("All but the last element of the given path."),
	),
}

var splitpath = &ast.Builtin{
	Name:        "rq.splitpath",
	Description: "Splits the given path into it's constituent elements.",
	Decl: types.NewFunction(
		types.Args(
			types.Named("path", types.S).Description("Path to split."),
		),
		types.Named("result", types.NewArray(nil, types.S)).Description("List containing the path elements."),
	),
}

var joinpath = &ast.Builtin{
	Name:        "rq.joinpath",
	Description: "Join a list of path components to a single path.",
	Decl: types.NewFunction(
		types.Args(
			types.Named("elements", types.NewArray(nil, types.S)).Description("Path elements to join."),
		),
		types.Named("result", types.S).Description("Joined path elements"),
	),
}

var getwd = &ast.Builtin{
	Name:        "rq.getwd",
	Description: "Return a path to the current working directory.",
	Decl: types.NewFunction(
		types.Args(),
		types.Named("result", types.S).Description("Path to the current working directory."),
	),
}

var chdir = &ast.Builtin{
	Name:        "rq.chdir",
	Description: "Change the current working directory.",
	Decl: types.NewFunction(
		types.Args(
			types.Named("dir", types.S).Description("Directory to change directories to."),
		),
		types.Named("result", types.B).Description("always `true`"),
	),
}

var scriptpath = &ast.Builtin{
	Name:        "rq.scriptpath",
	Description: "Retrieve the path the current Rego script.",
	Decl: types.NewFunction(
		types.Args(),
		types.Named("path", types.S).Description("Path to the Rego script."),
	),
}

var btemplate = &ast.Builtin{
	Name:        "rq.template",
	Description: "Template a string with user-supplied data.",
	Decl: types.NewFunction(
		types.Args(
			types.Named("s", types.S).Description("String to template."),
			types.Named("data", types.A).Description("Data to supply to the template."),
		),
		types.Named("result", types.S).Description("The string after templating has been provided."),
	),
}

var fake = &ast.Builtin{
	Name:        "rq.fake",
	Description: "Generate various kinds of fake data.",
	Decl: types.NewFunction(
		types.Args(
			types.Named("kind", types.S).Description("Kind of fake data to generate."),
		),
		types.Named("result", types.A).Description("Generated fake data"),
	),
}

var sfake = &ast.Builtin{
	Name:        "rq.sfake",
	Description: "Generate various kinds of fake data with symnbolic references.",
	Decl: types.NewFunction(
		types.Args(
			types.Named("kind", types.S).Description("Kind of fake data to generate."),
			types.Named("symbol", types.S).Description("Symbol to reference the output with."),
		),
		types.Named("result", types.A).Description("Generated fake data"),
	),
}

var quote = &ast.Builtin{
	Name:        "rq.quote",
	Description: "Wrap a string in quotes.",
	Decl: types.NewFunction(
		types.Args(
			types.Named("s", types.S).Description("String to quote."),
		),
		types.Named("result", types.S).Description("Quoted version of the string."),
	),
}

var unquote = &ast.Builtin{
	Name:        "rq.unquote",
	Description: "Unquote an already quoted string.",
	Decl: types.NewFunction(
		types.Args(
			types.Named("s", types.S).Description("String to unquote."),
		),
		types.Named("result", types.S).Description("Unquoted version of the string."),
	),
}

var slug = &ast.Builtin{
	Name:        "rq.slug",
	Description: "Convert a string to a URL-friendly slug.",
	Decl: types.NewFunction(
		types.Args(
			types.Named("s", types.S).Description("String to slug-ify."),
		),
		types.Named("result", types.S).Description("Slug-ified version of the string."),
	),
}

var strtonum = &ast.Builtin{
	Name:        "rq.strtonum",
	Description: "Construct a Rego number directly from a string, allowing dynamic arbitrary-precision JSON numeric output.",
	Decl: types.NewFunction(
		types.Args(
			types.Named("s", types.S).Description("String to construct the number from."),
		),
		types.Named("result", types.N).Description("Rego number value."),
	),
}

type dummyResolver struct {
}

func (r *dummyResolver) Resolve(ref ast.Ref) (interface{}, error) {
	return nil, fmt.Errorf("this is unused")
}

// InterfaceToSpec converts an arbitrary interface to a DataSpec. If it encodes
// a map[string]interface{}, then this is parsed out into a DataSpec as if it
// was JSON. If it encodes any other type, is is parsed as a DataSpec string.
func InterfaceToSpec(iface interface{}) (*rqio.DataSpec, error) {
	obj, ok := iface.(map[string]interface{})
	if !ok {
		return rqio.ParseDataSpec(util.ValueToString(iface))
	}

	ds := &rqio.DataSpec{Options: make(map[string]string)}

	if format, ok := obj["format"]; ok {
		if format != nil {
			ds.Format = util.ValueToString(format)
		}
	}

	if filePath, ok := obj["file_path"]; ok {
		if filePath != nil {
			ds.FilePath = util.ValueToString(filePath)
		}
	}

	if options, ok := obj["options"]; ok {
		if m, ok2 := options.(map[string]interface{}); ok2 {
			for k, v := range m {
				ds.Options[k] = util.ValueToString(v)
			}
		} else {
			return nil, errors.New("'options' is not an object")
		}
	}

	return ds, nil
}

// implements rq.run
//
// The options object can have these keys:
//
// stdin (string): piped into the command as it's standard input
//
// stdin (object): formatted using an rq output handler and piped into the
// command's standard input
//
// stdin_spec (string): a DataSpec for an rq output handler describing how the
// standard input for the command should be obtained. If the stdin key is also
// specified, then the FilePath part of the DataSpec is ignored and the given
// object is used as input. If both stdin_spec and stdin are specified, and
// stdin is a string, then stdin_spec is ignored. If stdin and stdin_spec are
// both provided, and the FilePath portion of the spec is non-empty, then an
// error is thrown.
//
// stdin_spec (object): treated as a pre-parsed dataspec using the same
// strucuture as the JSON form of a DataSpec.
//
// stdout_spec (string/object): like stdin_spec, but specifies a DataSpec for
// an rq input handler. If omitted, then the output is treated as a string. The
// FilePath is always ignored and an error will be thrown if it is non-empty.
//
// stderr_spec (string/object): like stdout_spec, for standard error.
//
// env (object): set environment variables for the process to be run.
func builtinRun(bctx rego.BuiltinContext, operands []*ast.Term, iter func(*ast.Term) error) error {
	cmdArr, err := builtins.ArrayOperand(operands[0].Value, 0)
	if err != nil {
		if _, err := builtins.StringOperand(operands[0].Value, 0); err == nil {
			// Allow single string commands.
			cmdArr = ast.NewArray(operands[0])
		} else {
			return topdown.Halt{Err: errors.New("rq.run: command must be a string or an array of strings")}
		}
	}

	optTerm := operands[1]
	_, err = builtins.ObjectOperand(operands[1].Value, 0)
	if err != nil {
		return topdown.Halt{Err: fmt.Errorf("rq.run: options must be an object (%w)", err)}
	}
	optI, err := ast.ValueToInterface(optTerm.Value, &dummyResolver{})
	if err != nil {
		return topdown.Halt{Err: fmt.Errorf("rq.run: failed to convert options to a Go type due to error: %w", err)}
	}
	// This is guaranteed to succeed since we already tried ObjectOperand
	options := optI.(map[string]interface{})

	cmdList := []string{}
	cmdArr.Foreach(func(t *ast.Term) {
		iface, err := ast.ValueToInterface(t.Value, &dummyResolver{})
		if err != nil {
			// This should never happen since the resolver is
			// unused, and we know t.Value is good.
			panic(err)
		}

		if s, ok := iface.(string); ok {
			cmdList = append(cmdList, s)
		} else {
			cmdList = append(cmdList, fmt.Sprintf("%v", iface))
		}
	})

	optStdin, haveStdin := options["stdin"]
	optStdinSpec, haveStdinSpec := options["stdin_spec"]
	optEnv, haveEnv := options["env"]
	optTimeoutI, haveTimeout := options["timeout"]
	var stdinReader io.Reader = &bytes.Buffer{}
	var stdinSpec *rqio.DataSpec = &rqio.DataSpec{Format: "json", Options: make(map[string]string)}

	env := map[string]string{}
	if haveEnv {
		if optEnvMS, ok := optEnv.(map[string]string); ok {
			for k, v := range optEnvMS {
				env[k] = v
			}
		} else if optEnvMI, ok := optEnv.(map[string]interface{}); ok {
			for k, v := range optEnvMI {
				env[k] = util.ValueToString(v)
			}
		} else {
			return topdown.Halt{Err: fmt.Errorf("rq.run: 'env' option has unexpected type %T", optEnv)}
		}
	}

	if haveStdinSpec {
		var err error
		stdinSpec, err = InterfaceToSpec(optStdinSpec)
		if err != nil {
			return topdown.Halt{Err: fmt.Errorf("rq.run options error: 'stdin_spec' was invalid due to error: %w", err)}
		}
		if (len(stdinSpec.FilePath) > 0) && haveStdin {
			return topdown.Halt{Err: fmt.Errorf("rq.run options error: 'stdin_spec' has a non-empty file path '%s', but it will be ignored because the 'stdin' field was provided", stdinSpec.FilePath)}
		}

	}

	if haveStdin {
		if _, ok := optStdin.(map[string]interface{}); ok {
			// The stdin value is an object, so we need to run
			// it through the output handler specified by
			// the stdinSpec.

			handler, err := rqio.SelectOutputHandler(stdinSpec.Format)
			if err != nil {
				return topdown.Halt{Err: fmt.Errorf("rq.run options error: output handler specified by 'stdin_spec', '%s', was invalid due to error: %w", stdinSpec.Format, err)}
			}

			buf := &bytes.Buffer{}

			defaults := &rqio.DataSpec{
				Options: map[string]string{
					"output.colorize": "false",
				},
			}
			stdinSpec.ResolveDefaults(defaults)
			err = rqio.WriteOutputToWriter(optStdin, stdinSpec, buf)
			if err != nil {
				return topdown.Halt{Err: fmt.Errorf("rq.run error: failed to format standard input with output handler '%s' due to error: %w", handler.Name(), err)}
			}

			stdinReader = buf

		} else {
			// Typically, this will be a string.
			stdinReader = bytes.NewBufferString(util.ValueToString(optStdin))
		}

	} else if haveStdinSpec {
		// The user may have specified a file path in the dataspec.
		if stdinSpec.FilePath != "" {
			if f, err := os.Open(stdinSpec.FilePath); err == nil {
				stdinReader = f
				defer func() { _ = f.Close() }()
			} else {
				return topdown.Halt{Err: fmt.Errorf("rq.run error: failed to open path '%s' from stdin_spec due to error: %w", stdinSpec.FilePath, err)}
			}

		}

	}

	ctx := bctx.Context
	if ctx == nil {
		ctx = context.Background()
	}

	if haveTimeout {
		timeout, err := strconv.ParseFloat(fmt.Sprintf("%v", optTimeoutI), 64)
		if err != nil {
			return topdown.Halt{Err: fmt.Errorf("rq.run error: failed to parse timeout value '%v' due to error: %w", optTimeoutI, err)}
		}

		tctx, cancel := context.WithTimeout(ctx, time.Duration(timeout*float64(time.Second)))
		defer cancel()
		ctx = tctx
	}

	cmd := exec.CommandContext(ctx, cmdList[0], cmdList[1:]...)
	stdin, err := cmd.StdinPipe()
	if err != nil {
		return topdown.Halt{Err: fmt.Errorf("rq.run error: failed to obtain stdin for command %s due to error: %w", cmdList[0], err)}
	}
	if stdin == nil {
		return topdown.Halt{Err: fmt.Errorf("rq.run error: failed to obtain stdin for command %s because StdinPipe returned a nil pipe", cmdList[0])}
	}
	defer func() { _ = stdin.Close() }()

	stderr, err := cmd.StderrPipe()
	if err != nil {
		return topdown.Halt{Err: fmt.Errorf("rq.run error: failed to obtain stderr for command %s due to error: %w", cmdList[0], err)}
	}
	if stderr == nil {
		return topdown.Halt{Err: fmt.Errorf("rq.run error: failed to obtain stderr for command %s because StderrPipe returned a nil pipe", cmdList[0])}
	}
	defer func() { _ = stderr.Close() }()

	stdout, err := cmd.StdoutPipe()
	if err != nil {
		return topdown.Halt{Err: fmt.Errorf("rq.run error: failed to obtain stdout for command %s due to error: %w", cmdList[0], err)}
	}
	if stdout == nil {
		return topdown.Halt{Err: fmt.Errorf("rq.run error: failed to obtain stdout for command %s because StdoutPipe returned a nil pipe", cmdList[0])}
	}
	defer func() { _ = stdout.Close() }()

	for k, v := range env {
		cmd.Env = append(cmd.Env, k+"="+v)
	}

	if err := cmd.Start(); err != nil {
		// Note that topdown.Halt is required if we want Rego to
		// actually surface this error.
		//
		// We want to do so in this particular case because this is
		// the one that will fire if the command is not found.
		return topdown.Halt{Err: err}
	}

	_, err = io.Copy(stdin, stdinReader)
	if err != nil {
		return err
	}
	_ = stdin.Close()
	stdoutBytes, _ := io.ReadAll(stdout)
	stderrBytes, _ := io.ReadAll(stderr)
	exitCode := 0

	if err := cmd.Wait(); err != nil {
		var eerr *exec.ExitError
		if !errors.As(err, &eerr) {
			// We don't expect to ever encounter an error
			panic(err)
		}
		exitCode = eerr.ExitCode()
	}

	timedout := errors.Is(ctx.Err(), context.DeadlineExceeded)

	stdoutTerm := ast.StringTerm(string(stdoutBytes))
	stderrTerm := ast.StringTerm(string(stderrBytes))

	if optStdoutSpec, ok := options["stdout_spec"]; ok {
		stdoutSpec, err := InterfaceToSpec(optStdoutSpec)
		if err != nil {
			return topdown.Halt{Err: fmt.Errorf("rq.run: failed convert 'stdout_spec' to a valid Go type: %w", err)}
		}

		if len(stdoutSpec.FilePath) > 0 {
			return topdown.Halt{Err: fmt.Errorf("rq.run options error: 'stdout_spec' has a non-empty file path '%s'", stdoutSpec.FilePath)}
		}

		obj, err := rqio.LoadInputFromReader(stdoutSpec, bytes.NewBuffer(stdoutBytes))
		if err != nil {
			return topdown.Halt{Err: fmt.Errorf("rq.run: failed to parse output according to 'stdout_spec' due to error: %w", err)}
		}

		stdoutTerm = &ast.Term{Value: ast.MustInterfaceToValue(obj)}
	}

	if optStderrSpec, ok := options["stderr_spec"]; ok {
		stderrSpec, err := InterfaceToSpec(optStderrSpec)
		if err != nil {
			return topdown.Halt{Err: fmt.Errorf("rq.run: failed convert 'stderr_spec' to a valid Go type: %w", err)}
		}

		if len(stderrSpec.FilePath) > 0 {
			return topdown.Halt{Err: fmt.Errorf("rq.run options error: 'stderr_spec' has a non-empty file path '%s'", stderrSpec.FilePath)}
		}

		obj, err := rqio.LoadInputFromReader(stderrSpec, bytes.NewBuffer(stderrBytes))
		if err != nil {
			return topdown.Halt{Err: fmt.Errorf("rq.run: failed to parse output according to 'stderr_spec' due to error: %w", err)}
		}

		stderrTerm = &ast.Term{Value: ast.MustInterfaceToValue(obj)}
	}

	if haveTimeout {
		return iter(ast.ObjectTerm(
			[2]*ast.Term{ast.StringTerm("stdout"), stdoutTerm},
			[2]*ast.Term{ast.StringTerm("stderr"), stderrTerm},
			[2]*ast.Term{ast.StringTerm("exitcode"), ast.IntNumberTerm(exitCode)},
			[2]*ast.Term{ast.StringTerm("timedout"), ast.BooleanTerm(timedout)},
		))
	} else {
		return iter(ast.ObjectTerm(
			[2]*ast.Term{ast.StringTerm("stdout"), stdoutTerm},
			[2]*ast.Term{ast.StringTerm("stderr"), stderrTerm},
			[2]*ast.Term{ast.StringTerm("exitcode"), ast.IntNumberTerm(exitCode)},
		))
	}

}

// implements rq.tree()
//
// The options object can have these keys:
//
// maxdepth (int): maximum depth to include in output, use -1 for unlimited,
// which is the default.
//
// hidden (bool): if true, hidden files are included in the output.
func builtinTree(_ rego.BuiltinContext, operands []*ast.Term, iter func(*ast.Term) error) error {
	path, err := builtins.StringOperand(operands[0].Value, 0)
	if err != nil {
		return topdown.Halt{Err: err}
	}

	optTerm := operands[1]
	_, err = builtins.ObjectOperand(operands[1].Value, 0)
	if err != nil {
		return topdown.Halt{Err: fmt.Errorf("rq.tree: options must be an object (%w)", err)}
	}
	optI, err := ast.ValueToInterface(optTerm.Value, &dummyResolver{})
	if err != nil {
		return topdown.Halt{Err: fmt.Errorf("rq.tree: failed to convert options to a Go type due to error: %w", err)}
	}
	// This is guaranteed to succeed since we already tried ObjectOperand
	options := optI.(map[string]interface{})

	maxdepth := -1
	if maxdepthI, ok := options["maxdepth"]; ok {
		maxdepth, err = strconv.Atoi(util.ValueToString(maxdepthI))
		if err != nil {
			return topdown.Halt{Err: fmt.Errorf("rq.tree: failed to parse maxdepth '%v' as an integer due to error: %w", maxdepthI, err)}
		}
	}

	hidden := false
	if hiddenI, ok := options["hidden"]; ok {
		hidden = util.Truthy(util.ValueToString(hiddenI))
	}

	basePath := string(path)

	abs, err := filepath.Abs(basePath)
	if err != nil {
		return topdown.Halt{Err: fmt.Errorf("rq.tree: failed to take absolute of '%s' due to error: %w", basePath, err)}
	}

	baseDepth := len(strings.Split(string(abs), string(os.PathSeparator)))

	files := [][2]*ast.Term{}
	walk := func(path string, d fs.DirEntry, err error) error {
		if err != nil {
			return err
		}

		base := filepath.Base(path)
		ext := filepath.Ext(path)
		if ext == base {
			ext = ""
		} else if ext != "" {
			ext = ext[1:]
		}

		abs, err := filepath.Abs(path)
		if err != nil {
			return err
		}

		depth := len(strings.Split(string(abs), string(os.PathSeparator))) - baseDepth

		if maxdepth != -1 {
			if depth > maxdepth {
				return nil
			}
		}

		info, err := d.Info()
		if err != nil {
			return err
		}

		rel, err := filepath.Rel(basePath, path)
		if err != nil {
			return err
		}

		if (strings.HasPrefix(base, ".") || strings.HasPrefix(rel, ".")) && !hidden {
			return nil
		}

		metadata := ast.ObjectTerm(
			[2]*ast.Term{ast.StringTerm("base"), ast.StringTerm(base)},
			[2]*ast.Term{ast.StringTerm("ext"), ast.StringTerm(ext)},
			[2]*ast.Term{ast.StringTerm("rel"), ast.StringTerm(rel)},
			[2]*ast.Term{ast.StringTerm("abs"), ast.StringTerm(abs)},
			[2]*ast.Term{ast.StringTerm("dir"), ast.StringTerm(filepath.Dir(abs))},
			[2]*ast.Term{ast.StringTerm("size"), ast.NumberTerm(json.Number(strconv.FormatInt(info.Size(), 10)))},
			[2]*ast.Term{ast.StringTerm("mode"), ast.StringTerm(info.Mode().String())},
			[2]*ast.Term{ast.StringTerm("depth"), ast.IntNumberTerm(depth)},
			[2]*ast.Term{ast.StringTerm("is_dir"), &ast.Term{Value: ast.MustInterfaceToValue(info.IsDir())}},
		)

		file := [2]*ast.Term{
			ast.StringTerm(abs),
			metadata,
		}

		files = append(files, file)

		return nil
	}

	err = filepath.WalkDir(string(path), walk)
	if err != nil {
		return topdown.Halt{Err: fmt.Errorf("rq.tree: error while scanning files: %w", err)}
	}

	return iter(ast.ObjectTerm(files...))
}

func builtinEnv(_ rego.BuiltinContext, operands []*ast.Term, iter func(*ast.Term) error) error {
	env := [][2]*ast.Term{}
	for _, v := range os.Environ() {
		split := strings.SplitN(v, "=", 2)
		if len(split) != 2 {
			panic("unreachable, you should not see this")
		}

		envvar := [2]*ast.Term{
			ast.StringTerm(split[0]),
			ast.StringTerm(split[1]),
		}

		env = append(env, envvar)

	}

	return iter(ast.ObjectTerm(env...))
}

func builtinArgs(_ rego.BuiltinContext, operands []*ast.Term, iter func(*ast.Term) error) error {
	return iter(&ast.Term{Value: ast.MustInterfaceToValue(Args)})
}

// ErrorFromBuiltin represents errors raised by the rq.error() builtin, which
// are treated specially for output formatting purposes elsewhere.
type ErrorFromBuiltin struct {
	message string
}

func (e ErrorFromBuiltin) Error() string {
	return e.message
}

func builtinError(_ rego.BuiltinContext, operands []*ast.Term, iter func(*ast.Term) error) error {
	message, err := builtins.StringOperand(operands[0].Value, 0)
	if err != nil {
		return topdown.Halt{Err: err}
	}

	return topdown.Halt{Err: ErrorFromBuiltin{string(message)}}
}

func builtinDecode(_ rego.BuiltinContext, operands []*ast.Term, iter func(*ast.Term) error) error {
	input, err := builtins.StringOperand(operands[0].Value, 0)
	if err != nil {
		return topdown.Halt{Err: err}
	}

	specTerm := operands[1]
	specI, err := ast.ValueToInterface(specTerm.Value, &dummyResolver{})
	if err != nil {
		return topdown.Halt{Err: err}
	}
	spec, err := InterfaceToSpec(specI)
	if err != nil {
		return topdown.Halt{Err: err}
	}
	if len(spec.FilePath) > 0 {
		return topdown.Halt{Err: fmt.Errorf("rq.decode options error: DataSpec has a non-empty file path '%s'", spec.FilePath)}
	}

	reader := bytes.NewBuffer([]byte(input))

	result, err := rqio.LoadInputFromReader(spec, reader)
	if err != nil {
		return topdown.Halt{Err: err}
	}

	resultValue, err := ast.InterfaceToValue(result)
	if err != nil {
		return topdown.Halt{Err: err}
	}

	return iter(&ast.Term{Value: resultValue})
}

func builtinEncode(_ rego.BuiltinContext, operands []*ast.Term, iter func(*ast.Term) error) error {
	inputTerm := operands[0]
	inputI, err := ast.ValueToInterface(inputTerm.Value, &dummyResolver{})
	if err != nil {
		return topdown.Halt{Err: err}
	}

	specTerm := operands[1]
	specI, err := ast.ValueToInterface(specTerm.Value, &dummyResolver{})
	if err != nil {
		return topdown.Halt{Err: err}
	}
	spec, err := InterfaceToSpec(specI)
	if err != nil {
		return topdown.Halt{Err: err}
	}
	if len(spec.FilePath) > 0 {
		return topdown.Halt{Err: fmt.Errorf("rq.encodeoptions error: DataSpec has a non-empty file path '%s'", spec.FilePath)}
	}

	buf := &bytes.Buffer{}

	err = rqio.WriteOutputToWriter(inputI, spec, buf)
	if err != nil {
		return topdown.Halt{Err: err}
	}

	resultValue, err := ast.InterfaceToValue(buf.String())
	if err != nil {
		return topdown.Halt{Err: err}
	}

	return iter(&ast.Term{Value: resultValue})
}

func builtinRead(_ rego.BuiltinContext, operands []*ast.Term, iter func(*ast.Term) error) error {
	specTerm := operands[0]
	specI, err := ast.ValueToInterface(specTerm.Value, &dummyResolver{})
	if err != nil {
		return topdown.Halt{Err: err}
	}
	spec, err := InterfaceToSpec(specI)
	if err != nil {
		return topdown.Halt{Err: err}
	}

	result, err := rqio.LoadInput(spec)
	if err != nil {
		return topdown.Halt{Err: err}
	}

	resultValue, err := ast.InterfaceToValue(result)
	if err != nil {
		return topdown.Halt{Err: err}
	}

	return iter(&ast.Term{Value: resultValue})
}

func builtinWrite(_ rego.BuiltinContext, operands []*ast.Term, iter func(*ast.Term) error) error {
	inputTerm := operands[0]
	inputI, err := ast.ValueToInterface(inputTerm.Value, &dummyResolver{})
	if err != nil {
		return topdown.Halt{Err: err}
	}

	specTerm := operands[1]
	specI, err := ast.ValueToInterface(specTerm.Value, &dummyResolver{})
	if err != nil {
		return topdown.Halt{Err: err}
	}
	spec, err := InterfaceToSpec(specI)
	if err != nil {
		return topdown.Halt{Err: err}
	}

	err = rqio.WriteOutput(inputI, spec)
	if err != nil {
		return topdown.Halt{Err: err}
	}

	return iter(ast.BooleanTerm(true))
}

func builtinParsedate(_ rego.BuiltinContext, operands []*ast.Term, iter func(*ast.Term) error) error {
	dateValue, err := builtins.StringOperand(operands[0].Value, 0)
	if err != nil {
		return err
	}

	dateS := string(dateValue)

	t, err := dateparse.ParseAny(dateS)
	if err != nil {
		return err
	}

	return iter(ast.NewTerm(ast.Number(json.Number(strconv.FormatInt(t.UnixNano(), 10)))))
}

func builtinVersion(_ rego.BuiltinContext, operands []*ast.Term, iter func(*ast.Term) error) error {
	vi := version.Version()

	return iter(ast.NewTerm(ast.MustInterfaceToValue(vi)))
}

func builtinConvert(_ rego.BuiltinContext, operands []*ast.Term, iter func(*ast.Term) error) error {
	amountValue, err := builtins.NumberOperand(operands[0].Value, 0)
	if err != nil {
		return topdown.Halt{Err: err}
	}
	amount, ok := amountValue.Float64()
	if !ok {
		return topdown.Halt{Err: fmt.Errorf("convert: failed to convert '%v' to float", amount)}
	}

	from, err := builtins.StringOperand(operands[1].Value, 0)
	if err != nil {
		return topdown.Halt{Err: err}
	}

	to, err := builtins.StringOperand(operands[2].Value, 0)
	if err != nil {
		return topdown.Halt{Err: err}
	}

	fromUnit, err := units.Find(string(from))
	if err != nil {
		return topdown.Halt{Err: err}
	}

	toUnit, err := units.Find(string(to))
	if err != nil {
		return topdown.Halt{Err: err}
	}

	converted, err := units.ConvertFloat(amount, fromUnit, toUnit)
	if err != nil {
		return topdown.Halt{Err: err}
	}

	return iter(ast.NewTerm(ast.MustInterfaceToValue(converted.Float())))
}

func builtinAbs(_ rego.BuiltinContext, operands []*ast.Term, iter func(*ast.Term) error) error {
	path, err := builtins.StringOperand(operands[0].Value, 0)
	if err != nil {
		return topdown.Halt{Err: err}
	}

	result, err := filepath.Abs(string(path))
	if err != nil {
		return topdown.Halt{Err: err}
	}

	return iter(ast.NewTerm(ast.MustInterfaceToValue(result)))
}

func builtinBase(_ rego.BuiltinContext, operands []*ast.Term, iter func(*ast.Term) error) error {
	path, err := builtins.StringOperand(operands[0].Value, 0)
	if err != nil {
		return topdown.Halt{Err: err}
	}

	result := filepath.Base(string(path))

	return iter(ast.NewTerm(ast.MustInterfaceToValue(result)))
}

func builtinExt(_ rego.BuiltinContext, operands []*ast.Term, iter func(*ast.Term) error) error {
	path, err := builtins.StringOperand(operands[0].Value, 0)
	if err != nil {
		return topdown.Halt{Err: err}
	}

	result := filepath.Ext(string(path))

	return iter(ast.NewTerm(ast.MustInterfaceToValue(result)))
}

func builtinDir(_ rego.BuiltinContext, operands []*ast.Term, iter func(*ast.Term) error) error {
	path, err := builtins.StringOperand(operands[0].Value, 0)
	if err != nil {
		return topdown.Halt{Err: err}
	}

	result := filepath.Dir(string(path))

	return iter(ast.NewTerm(ast.MustInterfaceToValue(result)))
}

func builtinSplitPath(_ rego.BuiltinContext, operands []*ast.Term, iter func(*ast.Term) error) error {
	path, err := builtins.StringOperand(operands[0].Value, 0)
	if err != nil {
		return topdown.Halt{Err: err}
	}

	components := []string{}

	dir := filepath.Clean(string(path))
	if dir[len(dir)-1] == '/' {
		dir = dir[0 : len(dir)-1]
	}
	file := ""
	for {
		dir, file = filepath.Split(dir)
		if file != "" {
			components = append([]string{file}, components...)
		}
		if dir == "" {
			break
		}

		// This is necessary because if there is a trailing / symbol,
		// filepath.Split() won't actually do the split and we get
		// caught in an infinite loop.
		dir = filepath.Clean(dir)
		if dir[len(dir)-1] == '/' {
			dir = dir[0 : len(dir)-1]
		}
	}

	return iter(ast.NewTerm(ast.MustInterfaceToValue(components)))
}

func builtinJoinPath(_ rego.BuiltinContext, operands []*ast.Term, iter func(*ast.Term) error) error {
	elements, err := builtins.ArrayOperand(operands[0].Value, 0)
	if err != nil {
		return topdown.Halt{Err: err}
	}

	sElements := make([]string, elements.Len())
	for i := 0; i < elements.Len(); i++ {
		et := elements.Get(ast.IntNumberTerm(i))
		if et == nil {
			return topdown.Halt{Err: fmt.Errorf("accessing element %d of path %v returned nil", i, elements)}
		}
		ei, err := ast.ValueToInterface(et.Value, &dummyResolver{})
		if err != nil {
			return topdown.Halt{Err: err}
		}
		sElements[i] = util.ValueToString(ei)
	}

	result := filepath.Join(sElements...)

	return iter(ast.NewTerm(ast.MustInterfaceToValue(result)))
}

func builtinGetwd(_ rego.BuiltinContext, operands []*ast.Term, iter func(*ast.Term) error) error {
	dir, err := os.Getwd()
	if err != nil {
		return topdown.Halt{Err: err}
	}

	return iter(ast.NewTerm(ast.MustInterfaceToValue(dir)))
}

func builtinChdir(_ rego.BuiltinContext, operands []*ast.Term, iter func(*ast.Term) error) error {
	path, err := builtins.StringOperand(operands[0].Value, 0)
	if err != nil {
		return topdown.Halt{Err: err}
	}

	err = os.Chdir(string(path))
	if err != nil {
		return topdown.Halt{Err: err}
	}

	return iter(ast.NewTerm(ast.MustInterfaceToValue(true)))
}

func builtinScriptPath(_ rego.BuiltinContext, operands []*ast.Term, iter func(*ast.Term) error) error {
	return iter(ast.NewTerm(ast.MustInterfaceToValue(ScriptPath)))
}

func builtinTemplate(_ rego.BuiltinContext, operands []*ast.Term, iter func(*ast.Term) error) error {
	s, err := builtins.StringOperand(operands[0].Value, 0)
	if err != nil {
		return topdown.Halt{Err: err}
	}

	dataTerm := operands[1]
	_, err = builtins.ObjectOperand(operands[1].Value, 0)
	if err != nil {
		return topdown.Halt{Err: fmt.Errorf("rq.template: data must be an object: %w", err)}
	}
	dataI, err := ast.ValueToInterface(dataTerm.Value, &dummyResolver{})
	if err != nil {
		return topdown.Halt{Err: fmt.Errorf("rq.template: failed to convert data to a Go type due to error: %w", err)}
	}
	// This is guaranteed to succeed since we already tried ObjectOperand
	data := dataI.(map[string]interface{})

	tmpl, err := template.New("").Funcs(templateFuncs).Parse(string(s))
	if err != nil {
		return topdown.Halt{Err: fmt.Errorf("rq.template: error while instantiating template from string '%s': %w", s, err)}
	}

	buf := &bytes.Buffer{}

	err = tmpl.Execute(buf, data)
	if err != nil {
		return topdown.Halt{Err: fmt.Errorf("rq.template: error while executing template from string '%s': %w", s, err)}
	}

	return iter(ast.NewTerm(ast.MustInterfaceToValue(buf.String())))
}

func builtinFake(_ rego.BuiltinContext, operands []*ast.Term, iter func(*ast.Term) error) error {
	kind, err := builtins.StringOperand(operands[0].Value, 0)
	if err != nil {
		return topdown.Halt{Err: err}
	}

	out, err := internal.FakeOptions.Fake(string(kind))
	if err != nil {
		return topdown.Halt{Err: err}
	}

	return iter(ast.NewTerm(ast.MustInterfaceToValue(out)))
}

func builtinSfake(_ rego.BuiltinContext, operands []*ast.Term, iter func(*ast.Term) error) error {
	kind, err := builtins.StringOperand(operands[0].Value, 0)
	if err != nil {
		return topdown.Halt{Err: err}
	}

	sym, err := builtins.StringOperand(operands[1].Value, 0)
	if err != nil {
		return topdown.Halt{Err: err}
	}

	out, err := internal.Sfake(string(kind), string(sym))
	if err != nil {
		return topdown.Halt{Err: err}
	}

	return iter(ast.NewTerm(ast.MustInterfaceToValue(out)))
}

func builtinQuote(_ rego.BuiltinContext, operands []*ast.Term, iter func(*ast.Term) error) error {
	s, err := builtins.StringOperand(operands[0].Value, 0)
	if err != nil {
		return topdown.Halt{Err: err}
	}

	out := strconv.Quote(string(s))

	return iter(ast.NewTerm(ast.MustInterfaceToValue(out)))
}

func builtinUnquote(_ rego.BuiltinContext, operands []*ast.Term, iter func(*ast.Term) error) error {
	s, err := builtins.StringOperand(operands[0].Value, 0)
	if err != nil {
		return topdown.Halt{Err: err}
	}

	out, err := strconv.Unquote(string(s))
	if err != nil {
		// if there is an error, we just return the string we already
		// had
		out = string(s)
	}

	return iter(ast.NewTerm(ast.MustInterfaceToValue(out)))
}

func builtinSlug(_ rego.BuiltinContext, operands []*ast.Term, iter func(*ast.Term) error) error {
	s, err := builtins.StringOperand(operands[0].Value, 0)
	if err != nil {
		return topdown.Halt{Err: err}
	}

	out := slugify.Slugify(string(s))

	return iter(ast.NewTerm(ast.MustInterfaceToValue(out)))
}

func builtinStrtonum(_ rego.BuiltinContext, operands []*ast.Term, iter func(*ast.Term) error) error {
	s, err := builtins.StringOperand(operands[0].Value, 0)
	if err != nil {
		return topdown.Halt{Err: err}
	}

	if !util.IsValidJSONNumber(string(s)) {
		return topdown.Halt{Err: fmt.Errorf("'%s' is not a valid JSON number", string(s))}
	}

	return iter(ast.NumberTerm(json.Number(string(s))))
}

func init() {

	templateFuncs = util.GetTemplateFuncs()
	templateFuncs["sfake"] = internal.Sfake

	ast.RegisterBuiltin(run)
	ast.RegisterBuiltin(tree)
	ast.RegisterBuiltin(env)
	ast.RegisterBuiltin(args)
	ast.RegisterBuiltin(berror)
	ast.RegisterBuiltin(decode)
	ast.RegisterBuiltin(encode)
	ast.RegisterBuiltin(read)
	ast.RegisterBuiltin(write)
	ast.RegisterBuiltin(parsedate)
	ast.RegisterBuiltin(bversion)
	ast.RegisterBuiltin(convert)
	ast.RegisterBuiltin(abs)
	ast.RegisterBuiltin(base)
	ast.RegisterBuiltin(ext)
	ast.RegisterBuiltin(dir)
	ast.RegisterBuiltin(splitpath)
	ast.RegisterBuiltin(joinpath)
	ast.RegisterBuiltin(getwd)
	ast.RegisterBuiltin(chdir)
	ast.RegisterBuiltin(scriptpath)
	ast.RegisterBuiltin(btemplate)
	ast.RegisterBuiltin(fake)
	ast.RegisterBuiltin(sfake)
	ast.RegisterBuiltin(quote)
	ast.RegisterBuiltin(unquote)
	ast.RegisterBuiltin(slug)
	ast.RegisterBuiltin(strtonum)

	topdown.RegisterBuiltinFunc(run.Name, builtinRun)
	topdown.RegisterBuiltinFunc(tree.Name, builtinTree)
	topdown.RegisterBuiltinFunc(env.Name, builtinEnv)
	topdown.RegisterBuiltinFunc(args.Name, builtinArgs)
	topdown.RegisterBuiltinFunc(berror.Name, builtinError)
	topdown.RegisterBuiltinFunc(decode.Name, builtinDecode)
	topdown.RegisterBuiltinFunc(encode.Name, builtinEncode)
	topdown.RegisterBuiltinFunc(read.Name, builtinRead)
	topdown.RegisterBuiltinFunc(write.Name, builtinWrite)
	topdown.RegisterBuiltinFunc(parsedate.Name, builtinParsedate)
	topdown.RegisterBuiltinFunc(bversion.Name, builtinVersion)
	topdown.RegisterBuiltinFunc(convert.Name, builtinConvert)
	topdown.RegisterBuiltinFunc(abs.Name, builtinAbs)
	topdown.RegisterBuiltinFunc(base.Name, builtinBase)
	topdown.RegisterBuiltinFunc(ext.Name, builtinExt)
	topdown.RegisterBuiltinFunc(dir.Name, builtinDir)
	topdown.RegisterBuiltinFunc(splitpath.Name, builtinSplitPath)
	topdown.RegisterBuiltinFunc(joinpath.Name, builtinJoinPath)
	topdown.RegisterBuiltinFunc(getwd.Name, builtinGetwd)
	topdown.RegisterBuiltinFunc(chdir.Name, builtinChdir)
	topdown.RegisterBuiltinFunc(scriptpath.Name, builtinScriptPath)
	topdown.RegisterBuiltinFunc(btemplate.Name, builtinTemplate)
	topdown.RegisterBuiltinFunc(fake.Name, builtinFake)
	topdown.RegisterBuiltinFunc(sfake.Name, builtinSfake)
	topdown.RegisterBuiltinFunc(quote.Name, builtinQuote)
	topdown.RegisterBuiltinFunc(unquote.Name, builtinUnquote)
	topdown.RegisterBuiltinFunc(slug.Name, builtinSlug)
	topdown.RegisterBuiltinFunc(strtonum.Name, builtinStrtonum)

}
