use crate::actions::GlobalArgs;
use crate::types::context::BergContext;

use clap::Parser;
use forgejo_api::structs::CreateKeyOption;
use miette::{Context, IntoDiagnostic};

/// Upload a new ssh public key, attaching it to your profile
#[derive(Parser, Debug)]
pub struct AddSshArgs {
    /// The path to an ssh public key
    ssh_pub_path: std::path::PathBuf,
}

impl AddSshArgs {
    pub async fn run(self, global_args: GlobalArgs) -> miette::Result<()> {
        let _ = global_args;
        let ctx = BergContext::new(self, global_args).await?;

        let key = std::fs::read_to_string(ctx.args.ssh_pub_path.as_path()).map_err(|e| {
            miette::miette!(
                help = format!(
                    "Check access permissions for the file '{path}'",
                    path = ctx.args.ssh_pub_path.display()
                ),
                "{e}"
            )
            .context("Reading the SSH public key from file failed!")
        })?;

        let title = key
            .split_whitespace()
            .last()
            .context("The provided key file seems to be empty")?
            .to_string();

        let form = CreateKeyOption {
            key,
            read_only: None,
            title,
        };

        ctx.client
            .user_current_post_key(form)
            .await
            .into_diagnostic()
            .context("Uploading SSH public key failed!")?;

        println!("Successfully added SSH key to your user profile!");

        Ok(())
    }
}
