/* 
 * xwave - an interactive audio player, recorder, editor 
 * for the XWindow System
 * 
 * Copyright (C) 1996 Kai Kollmorgen
 * (kkollmor@informatik.uni-rostock.de)
 *
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 * 
 */

#include <stdio.h>
#include <stdlib.h>
#include <dirent.h>
#include <sys/types.h>
#include <X11/Intrinsic.h>	/* Intrinsics Definitions */
#include <X11/StringDefs.h>	/* Standard Name-String definitions */
#include <X11/Shell.h>     	/* Shell Definitions */

#include <X11/Xaw/MenuButton.h>	/* Athena Command Widget */
#include <X11/Xaw/SimpleMenu.h>
#include <X11/Xaw/SmeBSB.h>
#include <X11/Xaw/SmeLine.h>
#include <X11/Xaw/Command.h>
#include <X11/Xaw/Box.h>
#include <X11/Xaw/Label.h>
#include <X11/Xaw/List.h>
#include <X11/Xaw/Form.h>
#include <X11/Xaw/Viewport.h>

#include "types.h"
#include "xwave.h"
#include "xwave_widget.h"
#include "fileop.h"
#include "edit.h"
#include "misc.h"
#include "menu.h"
#include "effects.h"
#include "button.h"
#include "help.h"
#include "graphics.h"

#include "bitmaps/pullright.xbm"

extern Main_Data *MD;

static void destroy(Widget w, XtPointer data, XtPointer junk);
static void new_menu_bar(Widget w,TopMenuBar menu_bar, Bar_Widget *bw);
static void create_item(Widget parent, TopMenuItem item, Top_Widget *tw);
static void right_action(Widget w, XEvent *event, String *params, 
			 Cardinal *nparams);
static void canvas_menu_activated(Widget w,XtPointer client_data,
				  XtPointer call_data);
static void setactual_cb(Widget w,XtPointer client_data,XtPointer call_data);
static void list_cb(Widget w,XtPointer client_data,XtPointer call_data);
static void separator(Widget w,bool add);
static void create_window_lister(Widget w);
static void open_list_cb(Widget w,XtPointer client_data,XtPointer call_data);
static void ca_cb(Widget w,XtPointer client_data,XtPointer call_data);
static Widget find_mentry(char *name);

static Bar_Widget *BarWidget;
static Pixmap pullBitmap = None;
static Widget listpopup,wlist;
static int lcounter=0;
static Widget cm_widgets[CM_ALL];
static Next_Wave **lnw=NULL;

static RightMenuItem items_paste[]={
    MI_PULL("p_merge",TRUE,paste_merge_call),
      MI_PULL("p_insert",TRUE,paste_insert_call),
      MI_PULL("p_new",TRUE,paste_new_call)
};

static RightMenuItem items_volume[]={
    MI_PULL("v_abs",TRUE,abs_vol_call),
      MI_PULL("v_env",TRUE,env_vol_call)
};

static TopMenuItem items_file[]={	
    MI_SIMPLE("new",TRUE,new_call),
      MI_SIMPLE("load",TRUE,load_call),
      MI_SIMPLE("close",FALSE,close_call),
      MI_SIMPLE("save",FALSE,save_call),
      MI_SIMPLE("saveas",FALSE,saveas_call),
      MI_LINE(),
      MI_SIMPLE("quit",TRUE,quit_call)
};

static TopMenuItem items_edit[]={	
    MI_SIMPLE("undo",FALSE,undo_call),
      MI_LINE(),
      MI_SIMPLE("cut",FALSE,cut_call),
      MI_SIMPLE("copy",FALSE,copy_call),
      MI_LINE(),
      MI_RIGHT("paste",FALSE,NULL,XtNumber(items_paste),items_paste)
};

static TopMenuItem items_effects[]=
{	
    MI_SIMPLE("echo",FALSE,echo_call),
      MI_RIGHT("volume",FALSE,NULL,XtNumber(items_volume),items_volume),
      MI_SIMPLE("reverse",FALSE,reverse_call),
      MI_SIMPLE("swap",FALSE,swap_call),
      MI_SIMPLE("props",FALSE,props_call)
};

static TopMenuItem items_help[]={	
    MI_SIMPLE("about",TRUE,about_call),
      MI_LINE(),
      MI_SIMPLE("help",FALSE,help_call),
};

static TopMenuBar menu_bar[]={
      {"file",XtNumber(items_file),items_file},
      {"edit",XtNumber(items_edit),items_edit},
      {"effects",XtNumber(items_effects),items_effects},
      {"windows",0,NULL},
      {"help",XtNumber(items_help),items_help}
};

static void canvas_menu_activated(Widget w,XtPointer client_data,
				  XtPointer call_data)
{
    int entry=(int) client_data;
    switch (entry) {
     case CM_PLAY:
	play_it(w,(XtPointer) MD,(XtPointer) NULL);
	break;
     case CM_STOP:
	stop_it(w,(XtPointer) MD,(XtPointer) NULL);
	break;
     case CM_ZOOM:
	zoom_canvas(MD);
	break;
     case CM_CUT:
	cut_call(w,(XtPointer) MD,(XtPointer) NULL);
	break;
     case CM_COPY:
	copy_call(w,(XtPointer) MD,(XtPointer) NULL);
	break;
     case CM_PASTE_I:
	paste_insert_call(w,(XtPointer) MD,(XtPointer) NULL);
	break;
     case CM_PASTE_M: 
	paste_merge_call(w,(XtPointer) MD,(XtPointer) NULL);
	break;
     case CM_CLOSE:
	close_call(w,(XtPointer) MD,(XtPointer) NULL);
	break;
    }
}

void set_cmenu_state(int from, int to, bool state)
{
    int i;
    if (to>CM_ALL) to=CM_ALL;
    if (from>CM_ALL) from=CM_ALL;
    
    for (i=from;i<=to;i++) {
	XtVaSetValues(cm_widgets[i],XtNsensitive,state,NULL);
    }
}

void create_canvas_menu(Widget w,Main_Widget *mw)
{
    Widget mbox,mlabel;
    static XtTranslations trans1,trans2;
    char buf[15];
    int i;
    
    trans1= XtParseTranslationTable("<BtnUp>: XtMenuPopdown(cv_menu)");
    trans2= XtParseTranslationTable("<EnterWindow>: set()\n\
<LeaveWindow>: unset()\n\
<BtnUp>: set() XtMenuPopdown(cv_menu) notify() unset()");
    
    mw->cmenu=XtVaCreatePopupShell("cv_menu",overrideShellWidgetClass,
                                   w,XtNtranslations, trans1,NULL);
    
    mbox=MW("cv_menu_box",boxWidgetClass,mw->cmenu,NULL);
    
    mlabel=MW("cv_menu_label",labelWidgetClass,mbox,NULL);
    
    for (i=0;i<=CM_ALL;i++) {
	sprintf(buf,"cv_menu_%d",i);
	cm_widgets[i]=MW(buf,commandWidgetClass,mbox,XtNtranslations,
			 trans2,NULL);
	XtAddCallback(cm_widgets[i],XtNcallback,canvas_menu_activated,
		      (XtPointer) i);
    }
}


static void destroy(Widget w, XtPointer data, XtPointer junk)
{
    XtFree((XtPointer)data);
}

static void right_action(Widget w, XEvent *event, String *params,
			 Cardinal *nparams)
{
    Dimension width,height;
    int x,y;
    Widget RightShell=NULL;
    Bar_Widget *bw;
    Top_Widget *tw;
    Widget cw=XawSimpleMenuGetActiveEntry(w);
    
    if (cw == None || event->type != MotionNotify) return;
    
    x = event->xmotion.x;
    y = event->xmotion.y;
    
    XtVaGetValues(w, XtNwidth, &width, XtNheight, &height, NULL);
    if (x < 0 || x >= width || y < 0 || y >= height) return;
    
    /*
     * Only the second half of the menu is sensitve to pulls
     */
    if (x < ((width / 2)+(width / 3))) return;
    
    bw=BarWidget;
    while ((bw!=NULL)&&(RightShell==NULL)) {
	tw=bw->tops;
	while ((RightShell==NULL)&&(tw!=NULL)) {
	    if (tw->widget==cw) {
		RightShell=tw->shell;
		break;
	    }
	    if (tw->widget==cw) break;
	    tw=tw->next;
	}
	bw=bw->next;
    }
    if (RightShell==NULL) return;
    
    x = event->xmotion.x_root - 20;
    y = event->xmotion.y_root - 20;
    
    
    XtVaSetValues(RightShell, XtNx, x, XtNy, y, NULL);
    
    XtPopup(RightShell, XtGrabExclusive);
}

static void new_menu_bar(Widget w,TopMenuBar menu_bar, Bar_Widget *bw)
{
    int i;
    char name[20],menuname[20],*nm;
    static Widget last_widget=None;
    Top_Widget *tw=NULL;
    
    strcpy(name,menu_bar.name);
    strcat(name,"_menu");
    strcpy(menuname,name);
    
    nm = XtNewString(menuname);
    
    strcat(name,"_btn");
    
    bw->button = MW (name,menuButtonWidgetClass,w,
		     XtNfromHoriz,last_widget,
		     XtNmenuName, nm,
		     NULL);
    
    last_widget=bw->button;
    
    XtAddCallback(bw->button, XtNdestroyCallback, destroy, (XtPointer)nm);
    
    bw->widget = XtCreatePopupShell (menuname,
				     simpleMenuWidgetClass,
				     bw->button,
				     NULL,0);
    
    bw->tops=NULL;
    if (menu_bar.nitems>0) {
	bw->tops=calloc(sizeof(Top_Widget),1); 
	tw=bw->tops;
	create_item(bw->widget,menu_bar.items[0],tw);
    }
    
    for (i=1;i<menu_bar.nitems;i++) {
	tw->next=calloc(sizeof(Top_Widget),1);
	tw=tw->next;
	create_item(bw->widget,menu_bar.items[i],tw);
	tw->next=NULL;
    }
}

static void create_item(Widget parent, TopMenuItem item, Top_Widget *tw)
{	
    char name[20];
    static XtTranslations trans1,trans2;
    int i;
    String nm=NULL;
    
    if (strlen(item.name)==0) {
	tw->widget= MW ("seperator",smeLineObjectClass,
			parent,
			NULL);
    } 
    else { 
	strcpy(name,item.name);
	strcat(name,"_item");
	if (item.nitems>0) {
	    Right_Widget *rw=NULL,*nrw=NULL;
	    
	    tw->widget = MW (name,smeBSBObjectClass,
			     parent,
			     XtNsensitive,item.sensitive,
			     XtNrightMargin, pullright_width,
			     XtNrightBitmap, pullBitmap,
			     NULL);
	    trans1 = XtParseTranslationTable("<BtnMotion>: highlight()\
popup_right_menu()");	   
	    trans2 = XtParseTranslationTable
	      ("<LeaveWindow>: unhighlight() MenuPopdown()\n\
<BtnUp>: notify() unhighlight() MenuPopdown()\n\
<BtnMotion>: highlight()");
	    
	    strcpy(name,item.name);
	    strcat(name,"_right");
	    tw->shell = XtVaCreatePopupShell(nm,
					     simpleMenuWidgetClass, parent,
					     XtNtranslations, trans2,
					     NULL);
	    XtAddCallback(tw->shell, XtNdestroyCallback, destroy,
			  (XtPointer)nm);
	    XtOverrideTranslations(parent, trans1);
	    
	    for (i = 0; i < item.nitems; i++) {
		strcpy(name,item.items[i].name);
		strcat(name,"_item");
		rw=calloc(sizeof(Right_Widget),1);
		rw->widget = MW (name,smeBSBObjectClass, tw->shell,
					NULL);
		XtAddCallback (rw->widget,XtNcallback,
			       item.items[i].callback,(XtPointer) MD);
		if (i==0) {
		    tw->right=rw;
		    nrw=rw;
		} else  {
		    nrw->next=rw;
		    nrw=rw;
		}
		rw->next=NULL;
	    }		   
	} else {
	    tw->widget = MW (name,smeBSBObjectClass,
			     parent,
			     XtNsensitive,item.sensitive,
			     NULL);
	    XtAddCallback (tw->widget,XtNcallback, item.callback,(XtPointer)MD);
	    tw->shell=NULL;
	}	
    }
}


void create_topmenu(Widget w) 
{
    
    static XtActionsRec act={"popup_right_menu", (XtActionProc)right_action};
    int i;
    Bar_Widget *bw;
    
    pullBitmap = XCreateBitmapFromData(XtDisplay(w),
				       DefaultRootWindow(XtDisplay(w)),
				       (char *)pullright_bits,
				       pullright_width, pullright_height);
    
    
    XtAppAddActions(XtWidgetToApplicationContext(w), &act, 1);
    
    /* alloc mem for top of all widgets in the list */ 
    
    if (XtNumber(menu_bar)>0) {
	bw=calloc(sizeof(Bar_Widget),1);
	BarWidget=bw;
	new_menu_bar(w,menu_bar[0],bw);
    }
    for (i=1;i<XtNumber(menu_bar);i++) {
	bw->next=calloc(sizeof(Bar_Widget),1);
	bw=bw->next;
	new_menu_bar(w,menu_bar[i],bw);
	bw->next=NULL;
    }
    
    create_window_lister(w);
}			 

void menu_state(bool state)
{
    int i;
    Bar_Widget *bw;
    
    bw=BarWidget;
    for (i=0;i<XtNumber(menu_bar);i++) {
	XtVaSetValues(bw->button,XtNsensitive,state,NULL); 
	bw=bw->next;  	
    }
}

void add_menu_entry(Next_Wave *nw)
{
    int i;
    char name[]="window  ";
    char *wname;
    Bar_Widget *bw;
    Top_Widget *tw;
    
    wname=nw->wd->name;   
    i=strlen(wname);
    wname+=i;
    while (*wname!='/') wname--;
    wname++;
    
    bw=BarWidget;
    for (i=0;i<XtNumber(menu_bar)-2;i++) bw=bw->next; 
    
    tw=bw->tops;
    i=0;
    if (bw->tops==NULL) {
	bw->tops=calloc(sizeof(Top_Widget),1);
	XtVaSetValues(bw->button,XtNsensitive,True,NULL); 
	separator(bw->widget,False);
	tw=bw->tops;
    } else {
	i++;
	while (tw->next!=NULL) {
	    tw=tw->next;
	    i++;
	}
	if (i<=MAXWINDOWENTRIES) {
	    tw->next=calloc(sizeof(Top_Widget),1);
	    tw=tw->next;
	    tw->widget=NULL;
	}
    }
    tw->next=NULL;
    tw->shell=NULL;
    
    strcat(name,itoa(i));
    
    if (i>=MAXWINDOWENTRIES) {
	String *list=NULL;
	lcounter++;
	if (tw->widget!=NULL) {
	    String *newlist;
	    Next_Wave **newlnw;
	    
	    XtVaGetValues(wlist,XtNlist,&list,NULL);
	    newlist=(String*)calloc(sizeof(String *), lcounter);
	    newlnw=(Next_Wave **)calloc(sizeof(Next_Wave*), lcounter);
	    for (i=0;i<lcounter-1;i++) {
		newlnw[i]=lnw[i];
		newlist[i]=XtNewString(newlnw[i]->wd->name);
	    }
	    newlist[lcounter-1]=XtNewString(nw->wd->name);
	    XawListChange(wlist,newlist,lcounter,0,True);
	    XtFree((char *)list);
	    XtFree((char *)lnw);
	    lnw=newlnw;
	    lnw[lcounter-1]=nw;
	    nw->cw->mentry=NULL;
	    return;
	}
	list=(String*)calloc(sizeof(String *), 1);
	list[0]=XtNewString(nw->wd->name);
	separator(bw->widget,True);
	XawListChange(wlist,list,1,0,True);
	tw->widget = MW ("morewin_item",smeBSBObjectClass,bw->widget,NULL);
	XtAddCallback (tw->widget,XtNcallback, open_list_cb,(XtPointer) nw);
	lnw=(Next_Wave **)XtCalloc(sizeof(Next_Wave*), lcounter);
	lnw[lcounter-1]=nw;
	nw->cw->mentry=NULL;
	return;
    }
    
    tw->widget = MW (name,smeBSBObjectClass,
		     bw->widget,
		     XtNlabel,wname,
		     NULL);
    XtAddCallback (tw->widget,XtNcallback, setactual_cb,(XtPointer) nw);
    nw->cw->mentry=tw->widget;
}

void setactual_cb(Widget w,XtPointer client_data,XtPointer call_data)
{
    set_actual_canvas((Next_Wave *)client_data);
}

void open_list_cb(Widget w,XtPointer client_data,XtPointer call_data)
{
    popup_centered(listpopup);
}


void remove_menu_entry(Next_Wave *nw)
{
    int i;
    Bar_Widget *bw;
    Top_Widget *tw;
    Top_Widget *lw;
    
    bw=BarWidget;
    for (i=0;i<XtNumber(menu_bar)-2;i++) bw=bw->next; 
    
    tw=lw=bw->tops;
    
    i=0;
    while ((tw->next!=NULL)&&(tw->widget!=nw->cw->mentry)) {
	tw=tw->next;
	i++;
    }
    
    if (tw->widget==nw->cw->mentry) {
	if (lcounter==0) {
	    if (i==0) {
		if (tw->next==NULL) {
		    separator(bw->widget,True);
		    XtVaSetValues(bw->button,XtNsensitive,False,NULL); 
		    XtDestroyWidget(tw->widget);
		    XtFree((char*)tw);
		    bw->tops=NULL;
		    return;
		} else {
		    bw->tops=tw->next;
		    XtDestroyWidget(tw->widget);
		    XtFree((char*)tw);
		    return;
		}
	    } else {
		if (tw->next==NULL) {
		    XtDestroyWidget(tw->widget);
		    while (i-->1) lw=lw->next;
		    lw->next=NULL;
		    XtFree((char*)tw);
		    return;
		} else {
		    XtDestroyWidget(tw->widget);
		    while (i-->1) lw=lw->next;
		    lw->next=tw->next;
		    XtFree((char*)tw);
		    return;
		}
	    }
	} else {
	    String *newlist;
	    String *list=NULL;
	    char *wname;
	    Next_Wave *nnw,**newlnw;
	    
	    XtVaGetValues(wlist,XtNlist,&list,NULL);
	    lcounter--;
	    /* get last string in list */
	    wname=list[lcounter];
	    i=strlen(wname);
	    wname+=i;
	    while (*wname!='/') wname--;
	    wname++;
	    /* set menu label to gotten string */
	    XtVaSetValues(nw->cw->mentry,XtNlabel,wname,NULL);
	    /* get last Next_Wave from lnw (**Next_Wave) list */ 
	    nnw=lnw[lcounter];
	    nnw->cw->mentry=nw->cw->mentry;
	    XtRemoveCallback(nw->cw->mentry,XtNcallback,setactual_cb,(XtPointer) nw);
	    XtAddCallback (nnw->cw->mentry,XtNcallback, setactual_cb,(XtPointer) nnw);
	    if (lcounter==0) {
		XawListChange(wlist,NULL,0,0,True);
		XtFree((char *)list);
		tw=lw=bw->tops;
		while (tw->next!=NULL) {
		    lw=tw; 
		    tw=tw->next;
		}
		XtDestroyWidget(tw->widget);
		XtFree((char*) tw);
		lw->next=NULL;
		separator(bw->widget,FALSE);
		XtFree((char *)lnw);
		return;
	    }
	    newlist=(String*)calloc(sizeof(String *), lcounter);
	    newlnw=(Next_Wave **)calloc(sizeof(Next_Wave*), lcounter);
	    for (i=0;i<lcounter;i++) {
		newlnw[i]=lnw[i];
		newlist[i]=XtNewString(newlnw[i]->wd->name);
	    }
	    XawListChange(wlist,newlist,lcounter,0,True);
	    XtFree((char *)list);
	    XtFree((char *)lnw);
	    lnw=newlnw;
	}
	return;
    }
    
    if (lcounter>0) {
	String *newlist;
	String *list=NULL;
	Next_Wave **newlnw;
	int j;
	
	XtVaGetValues(wlist,XtNlist,&list,NULL);
	lcounter--;
	if (lcounter==0) {
	    XawListChange(wlist,NULL,0,0,True);
	    XtFree((char *)list);
	    tw=lw=bw->tops;
	    while (tw->next!=NULL) {
		lw=tw; 
		tw=tw->next;
	    }
	    XtDestroyWidget(tw->widget);
	    XtFree((char*) tw);
	    lw->next=NULL;
	    separator(bw->widget,FALSE);
	    XtFree((char *)lnw);
	    return;
	}
	newlist=(String*)calloc(sizeof(String *), lcounter);
	newlnw=(Next_Wave **)calloc(sizeof(Next_Wave*), lcounter);
	for (i=0,j=0;i<lcounter;i++,j++) {
	    if (lnw[j]==nw) j++;
	    newlnw[i]=lnw[j];
	    newlist[i]=XtNewString(newlnw[i]->wd->name);
	}
	XawListChange(wlist,newlist,lcounter,0,True);
	XtFree((char *)list);
	XtFree((char *)lnw);
	lnw=newlnw;
	return;
    }
}

void separator(Widget w,bool add)
{
    static Widget sep=NULL;
    if (add)  {
	sep = MW ("seperator",smeLineObjectClass,w,NULL);
    } else {
	if (sep!=NULL) XtDestroyWidget(sep);
    }
}

void create_window_lister(Widget w)
{
    Widget form,viewport,button;
    static XtTranslations trans;
    
    listpopup=XtVaCreatePopupShell("Select Window",
				   transientShellWidgetClass,w,NULL);
    
    form  = MW ("wl_main_form",formWidgetClass,listpopup,
		XtNresizable,TRUE, 
		XtNborderWidth,0,
		NULL); 
    
    viewport = MW ("wl_main_vp",viewportWidgetClass,form,
		   XtNallowVert,TRUE,XtNallowHoriz,TRUE,XtNforceBars,False,
		   XtNtop,XawRubber,XtNbottom,XawRubber,
		   XtNleft,XawRubber,XtNright,XawRubber,NULL);
    
    trans=XtParseTranslationTable("#override\n\
<Btn1Up>(2) : Set() Notify()\n");
    
    wlist = MW ("wl_main_list",listWidgetClass,viewport,
		XtNdefaultColumns,1,
		XtNtranslations, trans,
		NULL);
    
    XtAddCallback (wlist, XtNcallback, list_cb, (XtPointer) wlist);
    
    button = MW ("wl_ok_btn",commandWidgetClass,form,
		 XtNfromVert,viewport,
		 XtNtop,XawChainBottom,XtNbottom,XawChainBottom,
		 XtNleft,XawChainLeft,XtNright,XawChainLeft,NULL);
    
    XtAddCallback (button, XtNcallback, list_cb,(XtPointer) wlist);
    
    button = MW ("wl_ca_btn",commandWidgetClass,form,
		 XtNfromVert,viewport,XtNfromHoriz,button,
		 XtNtop,XawChainBottom,XtNbottom,XawChainBottom,
		 XtNleft,XawChainLeft,XtNright,XawChainLeft,NULL);
    
    XtAddCallback (button, XtNcallback, ca_cb,(XtPointer) wlist);
    
    XtRealizeWidget(listpopup);
    
}

void list_cb(Widget w,XtPointer client_data,XtPointer call_data)
{
    Widget shell;
    XawListReturnStruct *item=XawListShowCurrent((Widget) client_data);
    if (item->list_index==-1) return;
    XawListUnhighlight((Widget) client_data);
    shell=XtParent(w);
    while (!XtIsShell(shell)) shell=XtParent(shell);
    XtPopdown(shell);
    set_actual_canvas(lnw[item->list_index]);
}

void ca_cb(Widget w,XtPointer client_data,XtPointer call_data)
{
    Widget shell;
    XawListUnhighlight((Widget) client_data);
    shell=XtParent(w);
    while (!XtIsShell(shell)) shell=XtParent(shell);
    XtPopdown(shell);
}

static Widget find_mentry(char *name)
{
    int i,j,k;
    Bar_Widget *bw;
    Top_Widget *tw;
    Right_Widget *rw;
    TopMenuItem *tmit;
    RightMenuItem *rmit;
    
    bw=BarWidget;
    for (i=0;i<XtNumber(menu_bar);i++) {
	tmit=menu_bar[i].items;
	tw=bw->tops;
	for (j=0;tw!=NULL;j++) {
	    if (strcmp(tmit[j].name,name)==0) return(tw->widget);
	    if (tmit[j].nitems>0) {
		rw=tw->right;
		rmit=tmit[j].items;
		for (k=0;k<tmit[j].nitems;k++) {
		    if (strcmp(rmit[k].name,name)==0) return(rw->widget);
		    rw=rw->next;
		}
	    }
	    tw=tw->next;
	}
	bw=bw->next;  	
    }
    return(None);
}

void set_menu_state(int from, int to, bool state)
{
    int i;Widget w;
    
    for (i=from;i<=to;i++) {
	switch (i) {
	 case M_SAVE: 
	    if ((w=find_mentry("save"))!=None)
	      XtVaSetValues(w,XtNsensitive,state,NULL);
	    break;
	 case M_SAVEAS:
	    if ((w=find_mentry("saveas"))!=None)
	      XtVaSetValues(w,XtNsensitive,state,NULL);
	    break;
	 case M_CLOSE: 
	    if ((w=find_mentry("close"))!=None)
	      XtVaSetValues(w,XtNsensitive,state,NULL);
	    break;
	 case M_CUT: 
	    if ((w=find_mentry("cut"))!=None)
	      XtVaSetValues(w,XtNsensitive,state,NULL);
	    break;
	 case M_COPY: 
	    if ((w=find_mentry("copy"))!=None)
	      XtVaSetValues(w,XtNsensitive,state,NULL);
	    break;
	 case M_PASTE: 
	    if ((w=find_mentry("paste"))!=None)
	      XtVaSetValues(w,XtNsensitive,state,NULL);
	    break;
	 case M_ECHO: 
	    if ((w=find_mentry("echo"))!=None)
	      XtVaSetValues(w,XtNsensitive,state,NULL);
	    break;
	 case M_VOLUME:
	    if ((w=find_mentry("volume"))!=None)
	      XtVaSetValues(w,XtNsensitive,state,NULL);
	    break;
	 case M_REVERSE:
	    if ((w=find_mentry("reverse"))!=None)
	      XtVaSetValues(w,XtNsensitive,state,NULL);
	    break;
	 case M_SWAP: 
	    if ((w=find_mentry("swap"))!=None)
	      XtVaSetValues(w,XtNsensitive,state,NULL);
	    break;
	 case M_PROPS: 
	    if ((w=find_mentry("props"))!=None)
	      XtVaSetValues(w,XtNsensitive,state,NULL);
	    break;
	 default: 
	    return;
	}
    }
}

void actual_menu_wname(Main_Data *md,char *name)
{
    int i;
    char *wname;
    
    if (lcounter==0) {
	wname=md->wd->name;
	i=strlen(wname);
	wname+=i;
	while (*wname!='/') wname--;
	wname++;
	XtVaSetValues(md->cw->mentry,XtNlabel,wname,NULL);
    } else {
	String *list=NULL;
	
	XtVaGetValues(wlist,XtNlist,&list,NULL);
	i=0;
	while (i<=lcounter) {
	    wname=list[i];
	    if (strcmp(wname,name)==0) break;
	    i++;
	}
	XtFree(list[i]);
	list[i]=XtNewString(md->wd->name);
	XtVaSetValues(wlist,XtNlist,list,NULL);
    }
}
