/*
 * xtopnm.c,
 * cause of speed considerations we use only raw PNM format types
 *
 * Copyright (C) 1997,98 Rasca, Berlin
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "../config.h"  /* autoconf output */

#include <stdio.h>
#include <stdlib.h>
#include <limits.h>
#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>
#include <X11/XWDFile.h>
#ifdef HAVE_LIBZ
#include <zlib.h>
#endif
#include "job.h"
#include "xtopnm.h"
#include "colors.h"
#include "app_data.h"

typedef struct {
	unsigned char red;
	unsigned char green;
	unsigned char blue;
} rgb;


/* global */
static unsigned char *line = NULL;
static unsigned char *pnm_image = NULL;


/*
 */
void *
PPMcolorTable (XColor *colors, int ncolors)
{
	rgb *color_tab;
	int i;

	color_tab = (rgb *) malloc (ncolors * sizeof (rgb));
	if (!color_tab)
		return (NULL);
	for (i =0; i < ncolors; i++) {
		color_tab[i].red = colors[i].red;
		color_tab[i].green = colors[i].green;
		color_tab[i].blue = colors[i].blue;
	}
	return (color_tab);
}

#ifdef DEBUG
void dump_ximage_info (XImage *img) {
	printf (" width %d\n", img->width);
	printf (" height %d\n", img->height);
	printf (" xoffset %d\n", img->xoffset);
	printf (" format %d\n", img->format);
	printf (" data addr 0x%X\n", (int)img->data);
	printf (" first four bytes of data 0x%X 0x%X 0x%X 0x%X\n",
			(unsigned char)(img->data[0]), (unsigned char)(img->data[1]), (unsigned char)img->data[2], (unsigned char)img->data[3]);
	printf (" byte_order %s\n", img->byte_order ? "MSBFirst":"LSBFirst");
	printf (" bitmap_unit %d\n", img->bitmap_unit);
	printf (" bitmap_bit_order %s\n", img->bitmap_bit_order ? "MSBFirst" : "LSBFirst");
	printf (" bitmap_pad %d\n", img->bitmap_pad);
	printf (" depth %d\n", img->depth);
	printf (" bytes_per_line %d\n", img->bytes_per_line);
	printf (" bits_per_pixel %d\n", img->bits_per_pixel);
	printf (" read_mask 0x%lX\n", img->red_mask);
	printf (" green_mask 0x%lX\n", img->green_mask);
	printf (" blue_mask 0x%lX\n", img->blue_mask);
	
}
#endif

/*
 * for TrueColor and DirectColor
 * write a PPM out to the named file
 */
void
XImageToPPMC (FILE *fp, XImage *image, Job *job)
{
	static char head[256];
	static int line_size, image_size;
	static ColorInfo c_info;
	static unsigned long max_val;
	register unsigned char *line_ptr, *p8, *p24;
	unsigned short *p16;
	int row, col;

	if ( job->state & VC_START ) {
		/* it's the first call, prepare the header and some statics
		 */
#ifdef DEBUG
		dump_ximage_info (image);
#endif
		/* get all the masks and max vals .. */
		GetColorInfo (image, &c_info);
		if (job->flags & FLG_EXPAND_TO_24BIT)
			max_val = 255;
		else
			max_val = c_info.max_val;
		sprintf (head, "P6\n%d %d\n%ld\n",
					image->width, image->height, max_val);

		if (image->bits_per_pixel > 16) {
			line_size = image->width * sizeof(unsigned char) * 3; /* RGB */
			line = (unsigned char *) malloc (line_size);
		} else {
			image_size = image->width * sizeof(unsigned char) * 3
							* image->height; /* RGB */
			pnm_image = (unsigned char *) malloc (image_size);
		}
	}
	if ((*job->write) (head, strlen (head), 1, fp) < 1)
		perror (job->file);

	/* for ZPixmap bits_per_pixel could be 1,4,8,16,24,32
	 * but for Direct- and TrueColor it could only be 8,16,24,32 (?)
	 */
	switch (image->bits_per_pixel) {
		case 8:
			/* for 8bpp x server
			 */
			p8 = (unsigned char *) image->data;
			line_ptr = pnm_image;
			for (row = 0; row < image->height; row++) {
				for (col = 0; col < image->width; col++) {
					*line_ptr++ =
							((*p8 & image->red_mask) >> c_info.red_shift)
							* max_val / c_info.red_max_val;
					*line_ptr++ =
							((*p8 & image->green_mask) >> c_info.green_shift)
							* max_val / c_info.green_max_val;
					*line_ptr++ =
							((*p8 & image->blue_mask) >> c_info.blue_shift)
							* max_val / c_info.blue_max_val;
					p8++;
				}
				/* eat paded bytes .. */
				p8 += image->bytes_per_line -
						 	image->bits_per_pixel / 8 * image->width;
			}
			/* write out the image
			 */
			if ((*job->write) (pnm_image, image_size, 1, fp) < 1)
				perror ("XImageToPPM()");
			break;

		case 16:
			/* for 16bpp and 15bpp x server
			 */
			p16 = (unsigned short *) image->data;
			line_ptr = pnm_image;
			if (job->quality == 100) {
				/*
				 * use exact calculation for the PPM output
				 */
				for (row = 0; row < image->height; row++) {
					for (col = 0; col < image->width; col++, p16++) {
						*line_ptr++ = (
								(*p16 & image->red_mask) >> c_info.red_shift
								) * max_val / c_info.red_max_val;
						*line_ptr++ = (
								(*p16 & image->green_mask) >> c_info.green_shift
								) * max_val / c_info.green_max_val;
						*line_ptr++ = (
								(*p16 & image->blue_mask) >> c_info.blue_shift
								) * max_val / c_info.blue_max_val;
					}
					/* eat paded bytes .. we have to devide by 2 because
					 * the p16 pointer is unsigned short *
					 */
					p16 += (image->bytes_per_line -
							 	image->bits_per_pixel / 8 * image->width)/2;
				}
			} else {
				/* still 16bpp code ..
				 *
				 * use not exact calculation, but should be faster :-)
				 * also it looks terrible ..
				 */
				register unsigned long
					left_shift,
					r_shift0 = c_info.red_shift,
					r_shift1 = c_info.red_bit_depth,
					r_shift2 = c_info.red_bit_depth * 2,
					g_shift0 = c_info.green_shift,
					g_shift1 = c_info.green_bit_depth,
					g_shift2 = c_info.green_bit_depth * 2,
					b_shift0 = c_info.blue_shift,
					b_shift1 = c_info.blue_bit_depth,
					b_shift2 = c_info.blue_bit_depth * 2;

				if (job->flags & FLG_EXPAND_TO_24BIT)
					left_shift=8;
				else
					left_shift=c_info.bit_depth;

				for (row = 0; row < image->height; row++) {
					for (col = 0; col < image->width; col++, p16++) {
						*line_ptr = (*p16 & image->red_mask) >>r_shift0;
						if (*line_ptr > 0)
						*line_ptr = (((*line_ptr << left_shift)-1) >> r_shift1)
								+ (((*line_ptr++<<left_shift)-1) >> r_shift2);
						else
							line_ptr++;
						*line_ptr = (*p16 & image->green_mask) >> g_shift0;
						if (*line_ptr > 0)
						*line_ptr = (((*line_ptr << left_shift)-1) >> g_shift1)
								+ (((*line_ptr++<<left_shift)-1) >> g_shift2);
						else
							line_ptr++;
						*line_ptr = (*p16 & image->blue_mask) >> b_shift0;
						if (*line_ptr > 0)
						*line_ptr = (((*line_ptr << left_shift)-1) >> b_shift1)
								+ (((*line_ptr++<<left_shift)-1) >> b_shift2);
						else line_ptr++;
					}
					/* eat paded bytes .. we have to devide by 2 because
					 * the p16 pointer is 'unsigned short *'
					 */
					p16 += (image->bytes_per_line -
							 (image->bits_per_pixel >> 3) * image->width) >> 1;
				}
			}
			if ((*job->write) (pnm_image, image_size, 1, fp) < 1)
				perror ("XImageToPPM()");
			break;

		case 24:
			p24 = (unsigned char *) image->data;
			if (image->byte_order == LSBFirst) {
				for (row = 0; row < image->height; row++) {
					line_ptr = line;
					for (col = 0; col < image->width; col++) {
						/* we have to swap */
						*line_ptr++ = p24[2];
						*line_ptr++ = p24[1];
						*line_ptr++ = p24[0];
						p24 += 3;
					}
					/* write out the row
					 */
					if ((*job->write) (line, line_size, 1, fp) < 1)
						perror ("XImageToPPM()");
					/* eat paded bytes .. */
					p24 += image->bytes_per_line -
						 	image->bits_per_pixel / 8 * image->width;
				}
			} else {
				/* MSBFirst: we don't have to swap the bytes
				 * in image->data, never tested .. does it work?
				 */
				for (row = 0; row < image->height; row++) {
					line_ptr = image->data + (image->bytes_per_line * row);
					/* we could directly write out the row
					 */
					if ((*job->write) (line_ptr, line_size, 1, fp) < 1) {
#ifdef DEBUG
						printf ("D: lsize=%d\n", line_size);
#endif
						perror ("XImageToPPM(), write()");
						job->state = VC_STOP;
					}
				}
			}
			break;

		case 32: { /* thanks to Lucian Plesea for the following code,
					* he said it works fine on a SGI workstation ..
					* i've done some minor changes to speed up (rasca)
					*/
			register unsigned int
					rm = image->red_mask,
					gm = image->green_mask,
					bm = image->blue_mask,
					rs = c_info.red_shift,
					gs = c_info.green_shift,
					bs = c_info.blue_shift,
					*p32 = (unsigned int *) image->data;

				for (row = 0; row < image->height; row++) {
					line_ptr = line;
					for (col = 0; col < image->width; col++) {
						*line_ptr++ = ((*p32 & rm) >> rs);
						*line_ptr++ = ((*p32 & gm) >> gs);
						*line_ptr++ = ((*p32 & bm) >> bs);
						p32++; /* ignore alpha values */
					}
					/* eat paded bytes, for better speed we use shifting,
					 * (bytes_per_line - bits_per_pixel / 8 * width ) / 4
					 */
					p32 += (image->bytes_per_line
							- (image->bits_per_pixel >> 3 )
							* image->width ) >> 2;
					if ((*job->write) (line, line_size, 1, fp) < 1)
						perror ("XImageToPPM()");
				}
			}
			break;

		default:
			printf ("bits_per_pixel not supported: %d\n",image->bits_per_pixel);
			break;
	}
}

/*
 * for PseudoColor/8bpp
 * write a PPM out to the named file
 */
void
XImageToPPM8 (FILE *fp, XImage *image, Job *job)
{
	static char head[256];
	static unsigned int image_size;
	register unsigned char *line_ptr, *col_ptr;
	register int row, col;

	if ( job->state & VC_START ) {
#ifdef DEBUG
		dump_ximage_info (image);
#endif
		sprintf (head, "P6\n%d %d\n%d\n", image->width, image->height, 255);
		image_size = image->width * 3 * image->height; /* RGB */
		pnm_image = (unsigned char *) malloc (image_size);
	}
	if ((*job->write) (head, strlen(head), 1, fp) < 1)
		perror (job->file);
	switch (image->bits_per_pixel) {
		case 8:
			line_ptr = pnm_image;
			for (row = 0; row < image->height; row++) {
				col_ptr = image->data + (row * image->bytes_per_line);
				for (col = 0; col < image->width; col++) {
					*line_ptr++ = ((rgb*)job->color_table)[*col_ptr].red;
					*line_ptr++ = ((rgb*)job->color_table)[*col_ptr].green;
					*line_ptr++ = ((rgb*)job->color_table)[*col_ptr].blue;
					col_ptr++;
				}
			}
			if ((*job->write) (pnm_image, image_size, 1, fp) < 1)
				perror ("XImageToPPM()");
			break;
		default:
			printf ("Visual not supported!\n");
			break;
	}
}

/*
 * GrayScale and StaticGray
 */
void
XImageToPGM (FILE *fp, XImage *image, Job *job)
{
	static char head[256];
	register unsigned char *row_ptr;
	register int row;

	if ( job->state & VC_START ) {
#ifdef DEBUG
		dump_ximage_info (image);
#endif
		sprintf (head, "P5\n%d %d\n%d\n", image->width, image->height, 255);
	}
	if ((*job->write) (head, strlen(head), 1, fp) < 1)
		perror (job->file);
	switch (image->bits_per_pixel) {
		case 8:
			for (row = 0; row < image->height; row++) {
				row_ptr = image->data + (row * image->bytes_per_line);
				if ((*job->write) (row_ptr, image->width, 1, fp) < 1)
					perror ("XImageToPGM()");
			}
			break;
		default:
			printf ("Visual not supported!\n");
			break;
	}
}


/*
 * call if stop was pressed, clean up some stuff..
 */
void
PnmClean (Job *job)
{
	if (line) {
		free (line);
		line = NULL;
	}
	if (pnm_image) {
		free (pnm_image);
		pnm_image = NULL;
	}
}

