/*
 * job.c
 *
 * Copyright (C) 1997,98 Rasca, Berlin
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *
 * This file defines the job data structure and some functions to
 * set/get info about a job (kinda tries to be OO ;S )
 *
 */

#include "../config.h" 	/* autoconf output */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/stat.h>
#include <errno.h>
#include <limits.h> /* PATH_MAX */
#include <X11/Intrinsic.h>
#ifdef HAVE_LIBZ
#include <zlib.h>
#endif
#include "job.h"
#include "capture.h"
#include "app_data.h"
#include "util.h"
//#include "sound.h"
#include "xtopnm.h"
#include "xtoxwd.h"
#include "xtopng.h"
#include "xtomng.h"
#include "xtojpg.h"
#include "xtoqtf.h"
#include "xtoffmpeg.h"

static Job *job;

/*
 */
Job *
job_init (int flags) {
#include "codecs.h"
    
    job = (Job *) malloc (sizeof (Job));
    if (!job) {
        perror ("job_init()");
        exit (1);
    }
    job->flags = flags;
    job->file = NULL;
    job->step = 1;
    job->target = CAP_XWD;
    job->targetCodec = CODEC_NONE;
    job->state = VC_STOP;
    //	job->toplevel = toplevel;
    job->color_table = NULL;
    job->get_colors = (void*(*)(XColor *, int))NULL;
    job->capture = TCbCaptureX11;
    job->clean = (void(*)(Job *))NULL;
    job->colors = NULL;
    job->movie_no = 0;
#ifdef DEBUG2
    printf ("job_init (flags=%d)\n", flags);
#endif
    return (job);
}

/*
 */
Job *
job_ptr (void) {
    return (job);
}




/*
 * Select Function to use for processing the captured frame
 */
void
job_set_save_function (Visual *vis, int type, Job *jobp) {
#include "codecs.h"
    
    switch (type) {
        case CAP_PNM:
            jobp->get_colors = PPMcolorTable;
            jobp->clean = PnmClean;
            jobp->flags &= ~FLG_MULTI_IMAGE;

            switch (vis->class) {
                case TrueColor:
                case DirectColor:
                    jobp->save = XImageToPPMC;
                    break;
                case PseudoColor:
                case StaticColor:
                    jobp->save = XImageToPPM8;
                    break;
                case StaticGray:
                case GrayScale:
                    printf ("Visual not supported\n");
                    jobp->save = XImageToPGM;
                    break;
            }
            break;
#ifdef HAVE_LIBPNG
        case CAP_PNG:
            jobp->get_colors = PNGcolorTable;
            jobp->clean = PngClean;
            jobp->flags &= ~FLG_MULTI_IMAGE;
            
            switch (vis->class) {
                case TrueColor:
                case DirectColor:
                    jobp->save = XImageToPNGC;
                    break;
                case PseudoColor:
                case StaticColor:
                case StaticGray:
                case GrayScale:
                    jobp->save = XImageToPNG8;
                    break;
            }
            break;
        case CAP_MNG:
            jobp->get_colors = PNGcolorTable;
            jobp->flags |= FLG_MULTI_IMAGE;
            jobp->clean = MngClean;
            
            switch (vis->class) {
                case TrueColor:
                case DirectColor:
                    jobp->save = XImageToMNGC;
                    break;
                case PseudoColor:
                case StaticColor:
                case StaticGray:
                case GrayScale:
                    jobp->save = XImageToMNG8;
                    break;
            }
            break;
#endif // HAVE_LIBPNG
#ifdef HAVE_LIBJPEG
        case CAP_JPG:
            jobp->get_colors = JPGcolorTable;
            jobp->clean = JpgClean;
            jobp->flags &= ~FLG_MULTI_IMAGE;
            
            switch (vis->class) {
                case TrueColor:
                case DirectColor:
                    jobp->save = XImageToJPGC;
                    break;
                case PseudoColor:
                case StaticColor:
                    jobp->save = XImageToJPG8;
                    break;
                case StaticGray:
                    jobp->save = XImageToJPGG;
                    break;
                case GrayScale:
                    jobp->save = XImageToJPGG;
                    printf ("Visual not supported!\n");
                    break;
            }
            break;
#endif // HAVE_LIBJPEG
#ifdef WITHGT
        case CAP_QTR:
        case CAP_QTJ:
        case CAP_QTP:
            job->clean = QtClean;
            job->flags |= FLG_MULTI_IMAGE;
            
            switch (vis->class) {
                case TrueColor:
                case DirectColor:
                    jobp->save = XImageToQTFC;
                    break;
                case StaticGray:
                case GrayScale:
                    jobp->save = XImageToQTFG;
                    break;
                default:
                    printf ("Visual not supported!\n");
                    exit (2);
                    break;
            }
            break;
#endif // WITHGT
#ifdef HAVE_LIBAVCODEC
#ifdef HAVE_LIBAVFORMAT
        case CAP_AVI:
        case CAP_ASF:
        case CAP_FLV:
            jobp->clean = FFMPEGClean;
            if (jobp->targetCodec == CODEC_NONE) {
                jobp->targetCodec = CODEC_MPEG1;
            }
            jobp->flags |= FLG_MULTI_IMAGE;
            jobp->get_colors = FFMPEGcolorTable;
    
            /*
             * make sure we have even width and height for ffmpeg
             */
            {
            Boolean changed = FALSE;
        
            if (jobp->flags & FLG_RUN_VERBOSE) {
                fprintf (stderr, "job.job_set_file(): Original dimensions: %i * %i\n", jobp->area->width, jobp->area->height);
                }
            if ((jobp->area->width % 2) > 0 ) {
                jobp->area->width--;
                changed = TRUE;
            }
            if ((jobp->area->height % 2) > 0) {
                jobp->area->height--;
                changed = TRUE;
            }
            if ( jobp->area->width < 20 ) {
                jobp->area->width = 20;
                changed = TRUE;
            }
            if ( jobp->area->height < 20 ) {
                jobp->area->height = 20;
                changed = TRUE;
            } 
        
            if (changed) {
                XVC_ChangeFrame (jobp->area->x, jobp->area->y, jobp->area->width, jobp->area->height);
                if (jobp->flags & FLG_RUN_VERBOSE) {
                    fprintf (stderr, "job.job_set_file(): Modified dimensions: %i * %i\n", jobp->area->width, jobp->area->height);
                }
            }
            }
            
            jobp->save = XImageToFFMPEG;
            break;
#endif // HAVE_LIBAVFORMAT
#endif // HAVE_LIBAVCODEC
        default:	/* XWD */
            jobp->save = XImageToXWD;
            break;
    }
}


/*
 */
void
job_set_fps (float fps) {
    if (fps <= 0)
        fps = 0.00001; /* min value */
    job->time_per_frame = 1000 / fps;
    job->fps = fps;
}

/*
 * return frames per second
 */
float
job_fps (void) {
    return (job->fps);
}


#ifdef HAVE_FFMPEG_AUDIO
/*
 * set and check some parameters for the sound device
 */
 
int
job_set_sound_dev (char *snd, int rate, int size, int channels) {
    extern int errno;
    struct stat statbuf;
    int stat_ret;
    
    job->snd_rate = rate;
    job->snd_smplsize = size;
    job->snd_channels = channels;
    job->snd_device = snd;
    
    if (job->flags & FLG_AUDIO_WANTED) {
        if (strcmp (snd, "-")!=0) {
            stat_ret = stat (snd, &statbuf);
            
            if ( stat_ret != 0 ) {
                switch (errno) {
                    case EACCES:
                        fprintf (stderr, "Insufficient permission to access sound input from %s\n", snd);
                        fprintf (stderr, "Sound disabled!\n");
                        job->flags &= ~FLG_AUDIO_WANTED;
                        break;
                    default:
                        fprintf (stderr, "Error accessing sound input from %s\n", snd);
                        fprintf (stderr, "Sound disabled!\n");
                        job->flags &= ~FLG_AUDIO_WANTED;
                        break;
                }
            }
        }
    } // end if soundWanted
}
#endif // HAVE_FFMPEG_AUDIO



/*
 */
void
job_set_file (char *file) {
    
    if (!file)
        return;
    if (strstr (file, ".ppm") ||
    strstr (file, ".pnm") ||
    strstr (file, ".pgm") ) {
        job->target = CAP_PNM;
    } else if (strstr (file, ".png")) {
#ifdef HAVE_LIBPNG
        job->target = CAP_PNG;
    } else if (strstr (file, ".mng")) {
        job->target = CAP_MNG;
#else // HAVE_LIBPNG
        printf ("PNG and MNG not supported!\n");
        exit (1);
#endif
    } else if (strstr (file, ".jpg") ||
    strstr (file, ".jpeg")) {
#ifdef HAVE_LIBJPEG
        job->target = CAP_JPG;
#else
        printf ("JPEG not supported!\n");
        exit (1);
#endif
    } else if (strstr (file, ".mov")) {
#ifdef WITHGT
        job->target = CAP_QTR;
#ifndef HAVE_LIBJPEG
        job->quality = 100;
#endif
#ifndef HAVE_LIBPNG
        job->compress= 0;
#endif
                if (job->quality && (job->quality < 100)) {
                    /* jpeg codec */
                    job->target = CAP_QTJ;
                    job->compress = 0;
                } else if (job->compress) {
                    /* png codec */
                    job->target = CAP_QTP;
                }
#else
                printf ("Quicktime encoding not supported!\n");
                exit (1);
#endif /* WITHGT */
    } 
#ifdef HAVE_LIBAVCODEC
#ifdef HAVE_LIBAVFORMAT
    else if (strstr (file, ".mpeg") || strstr (file, ".mpg") || strstr (file, ".avi")) {
        job->target = CAP_AVI;
    } else if (strstr (file, ".asf")) {
        job->target = CAP_ASF;
    } else if (strstr (file, ".flv") || strstr (file, ".flv1")) {
        job->target = CAP_FLV;
    }
#else /* HAVE_LIBAVFORMAT */
    else if (strstr (file, ".mpeg") || strstr (file, ".mpg") || strstr (file, ".avi") || 
    strstr (file, ".asf") || strstr (file, ".flv") || strstr (file, ".flv1")) {
        printf ("FFMPEG encoding not supported!\n");
        exit (1);
    }
#endif /* HAVE_LIBAVFORMAT */
#else /* HAVE_LIBAVCODEC */
    else if (strstr (file, ".mpeg") || strstr (file, ".mpg") || strstr (file, ".avi") ||
    strstr (file, ".asf") || strstr (file, ".flv") || strstr (file, ".flv1")) {
        printf ("FFMPEG encoding not supported!\n");
        exit (1);
    }
#endif
    else {
        /* default is XWD
         */
        job->target = CAP_XWD;
        job->get_colors = XWDcolorTable;
        job->flags &= ~FLG_MULTI_IMAGE;
        job->clean = NULL;
    }
    
#ifdef HAVE_LIBZ
        if (job->compress && !(
        (job->target == CAP_PNG) || (job->target == CAP_MNG) ||
        (job->target == CAP_QTP))) {
            int len = strlen (file);
            
            job->open = (void*(*)(char*, char*))gzopen;
            job->close= (int(*)(void*))gzclose;
            job->write = zwrite;
            if (job->file && (job->file != file))
                free (job->file);
            if (!strstr (file, ".gz")) {
                job->file = (char *) malloc (len + 4 + 1);
                sprintf (job->file, "%s.gz", file);
            } else {
                job->file = strdup (file);
            }
            sprintf (job->open_flags, "wb%d", job->compress);
        } else
#endif
            //
{
    job->open = (void*(*)(char*, char*))fopen;
    job->close= (int(*)(void*))fclose;
    if (job->target == CAP_QTJ) {
        job->write = (int(*)(void*, unsigned int, unsigned int, void *))jwrite;
    } else if (job->target == CAP_QTP) {
        job->write = (int(*)(void*, unsigned int, unsigned int, void *))xv_pwrite;
    } else {
        job->write = (int(*)(void*, unsigned int, unsigned int, void *))fwrite;
    }
    job->file = strdup (file);
    if (strstr (file, ".gz")) {
        int len = strlen (file);
        job->file[len-3] = '\0';
    }
    sprintf (job->open_flags, "wb");
}
#ifdef HasBTTV
        if (job->flags & FLG_USE_V4L) {
            job->capture = TCbCaptureV4L;
        }
#endif
        // make sure a change gets activated for the next capture, too
        job_set_save_function (job->win_attr.visual, job->target, job);
}

/*
 */
Boolean is_filename_mutable(char* filename) {
	char file[PATH_MAX+1];

	sprintf(file, filename, 1);
	if (strcmp(file, filename) == 0 ) {
		return FALSE;
	} else {
		return TRUE;
	}
}


/*
 */
char *
job_file (void) {
    return (job->file);
}

/*
 */
void
job_set_compression (int comp) {
    if (comp < 0)
        comp = 0;
    else if (comp > 9)
        comp = 9;
    job->compress = comp;
}

/*
 */
int
job_compression (void) {
    return (job->compress);
}

/*
 * find the correct capture function
 */
void
job_set_capture (void) {
    int input = job->flags & FLG_SOURCE;
    
    switch (input) {
        case FLG_USE_SHM:
            job->capture = TCbCaptureSHM;
            break;
        case FLG_USE_DGA:
            job->capture = TCbCaptureDGA;
            break;
        case FLG_USE_V4L:
            job->capture = TCbCaptureV4L;
            break;
        default:
            job->capture = TCbCaptureX11;
            break;
    }
}

/*
 * return if shared memory is used or not
 */
int
job_shm (void) {
    return (job->flags & FLG_USE_SHM);
}

/*
 */
void
job_set_quality (int quality) {
    if (quality < 1)
        quality = 1;
    else if (quality > 100)
        quality = 100;
    job->quality = quality;
}

/*
 */
int job_quality (void) {
    return (job->quality);
}






