/*
 * gtapi.c
 * GnuTime module
 *
 * Copyright (C) 1998 Rasca, Berlin
 * EMail: thron@gmx.de
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <gt/gt.h>

/*
 */
void
gt_atom_add (gt_atom *parent, gt_atom *child)
{
	if (!parent || !child)
		return;
	parent->suba[parent->memb++] = child;
	parent->size += child->size;
}

/*
 * create a new movie handle, not done
 */
gt_movie *
gt_movie_new (int type, int maxsubs, const char *file)
{
	gt_movie *movie;

	movie = malloc (sizeof (gt_movie));
	if (!movie)
		return NULL;
	/* have at least one child: mvhd */
	movie->moov = gt_alloc_atom (GTA_movie, 1 + maxsubs);
	if (!movie->moov) {
		free (movie);
		return NULL;
	}
	movie->max_subs = 1+ maxsubs; 	/* max sub atoms in the moov container */
	(gt_atom *) movie->mvhd = movie->moov->suba[0] =
			gt_alloc_atom (GTA_movie_header, 0);
	movie->moov->memb = 1;
	movie->moov->size += movie->mvhd->size;
	movie->mvhd->ctime = gt_time();
	movie->mvhd->mtime = gt_time();
	movie->mvhd->time_scale = 1000;
	movie->mvhd->duration = 1000 * 12;
	movie->mvhd->pref_rate.high = 1;
	movie->mvhd->matrix[1] = 1;
	movie->mvhd->matrix[17] = 1;
	movie->mvhd->matrix[32] = 64;
	movie->mvhd->next_track_id = 1;

	movie->mdat = gt_alloc_atom (GTA_movie_data, 0);
	if (!movie->mdat) {
		free (movie->moov);
		free (movie->mvhd);
		free (movie);
		return NULL;
	}
	if (file)
		movie->file = strdup (file);
	else
		movie->file = NULL;
	return movie;
}

/*
 * free a movie structure
 */
void
gt_movie_free (gt_movie *movie)
{
	if (!movie || !movie->moov)
		return;
	gt_atom_free (movie->moov);
	gt_atom_free (movie->mdat);
	free (movie);
}

/*
 * add a track or user data to the moov atom
 */
int
gt_movie_add_atom (gt_movie *movie, gt_atom *atom)
{
	if (!movie || !movie->moov)
		return FALSE;
	if (movie->max_subs < (movie->moov->memb +1) )
		return FALSE;
	movie->moov->suba[movie->moov->memb++] = atom;
	movie->moov->size += atom->size;
	if (atom->type == GTA_track) {
		movie->mvhd->next_track_id = ((gt_tkhd_atom *)
					(atom->suba[0]))->track_id+1;
	}
#ifdef DEBUG
	fprintf (stderr, "gt_movie_add_atom() size=%d\n", movie->moov->size);
#endif
	return TRUE;
}

/*
 * write the moov structure to a file
 */
int
gt_movie_write_moov (gt_movie *movie, const char *file)
{
	FILE *fp;
	int rc;

	if (!movie)
		return 0;
	if (!file)
		file = movie->file;
	if (!file)
		return 0;
	fp = fopen (file, "wb");
	if (!fp)
		return 0;
	rc = gt_write_atom (movie->moov, fp);
	fclose (fp);
	return rc;
}

/*
 */
gt_minf_atom *
gt_media_info_new (int type, int subatoms)
{
	gt_minf_atom *minf;
	gt_hdlr_atom *hdlr;
	gt_atom *mtype;
	minf = gt_alloc_atom (GTA_media_info, 2 + subatoms);

	minf->suba[0] = (gt_atom *)hdlr = gt_alloc_atom (GTA_handler_ref, 0);
	minf->memb = 1;
	minf->size += hdlr->size;

	hdlr->comp_type = GT_DATA_REF;
	hdlr->comp_subtype = GT_ALIAS;
	hdlr->comp_man  = GT_COMP_MAN;
	/* ? */
	/* hdlr->comp_flags = 0x40000001; */
	/* hdlr->comp_flags_mask = 0x00010034; */

	if (type == GT_VIDEO) {
		mtype = gt_alloc_atom (GTA_video_media_header, 0);
		minf->suba[1] = mtype;
		minf->size += mtype->size;
		((gt_vmhd_atom *)mtype)->flags[2] = 1;
		minf->memb++;
	} else if (type == GT_SOUND) {
		mtype = gt_alloc_atom (GTA_sound_media_header, 0);
		minf->suba[1] = mtype;
		minf->size += mtype->size;
		minf->memb++;
	} else { /* GT_BASE */
		gt_gmin_atom *gmin;
		mtype = gt_alloc_atom (GTA_base_media_header, 1);
		gmin  = (gt_gmin_atom *)gt_alloc_atom (GTA_base_media_info, 0);
		gmin->grmode = 0x0040; /* ? */
		gt_atom_add (mtype, (gt_atom *)gmin);
		minf->suba[1] = mtype;
		minf->size += mtype->size;
		minf->memb++;
	}
	return (minf);
}

/*
 * a media atom has at least one subatom: media header
 * and we add the media handler as well..
 */
gt_mdia_atom *
gt_media_new (int type, int time_scale, int subatoms)
{
	gt_mdia_atom *mdia;
	gt_mdhd_atom *mdhd;
	gt_hdlr_atom *mdia_hdlr;

	mdia = gt_alloc_atom (GTA_media, 2 + subatoms);
	if (!mdia)
		return NULL;
	mdia->memb = 1;
	mdia->suba[0] = (gt_atom *)mdhd = gt_alloc_atom (GTA_media_header, 0);

	mdia->size += mdhd->size;
	mdhd->ctime = gt_time();
	mdhd->mtime = gt_time();
	mdhd->time_scale = time_scale;
	mdhd->duration = time_scale * 12;

	if (type == GT_VIDEO) {
		/* add a video media handler
		 */
		(gt_atom *)mdia_hdlr = mdia->suba[1] =
				gt_alloc_atom (GTA_handler_ref, strlen(GT_MID_HDLR_NAME));
		mdia->size += mdia_hdlr->size;
		mdia->memb++;

		mdia_hdlr->comp_type = GT_MEDIA_REF;
		mdia_hdlr->comp_subtype = GT_VIDEO;
		mdia_hdlr->comp_man = GT_COMP_MAN;
		mdia_hdlr->comp_flags = 0x40000000;		/* ? */
		mdia_hdlr->comp_flags_mask = 0x00010047; /* or 1002b ? */
		strcpy (mdia_hdlr->comp_name+1, GT_MID_HDLR_NAME);
		mdia_hdlr->comp_name[0]= strlen(GT_MID_HDLR_NAME);
	} else if (type == GT_PANORAMA) {
		(gt_atom *)mdia_hdlr = mdia->suba[1] =
				gt_alloc_atom (GTA_handler_ref, strlen(GT_MID_HDLR_NAME));
		mdia->size += mdia_hdlr->size;
		mdia->memb++;

		mdia_hdlr->comp_type = GT_MEDIA_REF;
		mdia_hdlr->comp_subtype = GT_N_PANO;
		mdia_hdlr->comp_man = GT_COMP_MAN;
		mdia_hdlr->comp_flags = 0x40000000;		/* ? */
		mdia_hdlr->comp_flags_mask = 0x0001002b; /* ? */
		strcpy (mdia_hdlr->comp_name+1, GT_MID_HDLR_NAME);
		mdia_hdlr->comp_name[0]= strlen(GT_MID_HDLR_NAME);
	}
	return (mdia);
}

/*
 * a track atom has at least a track header and a media atom
 */
gt_trak_atom *
gt_track_new (int id, int width, int height, int type, int subatoms)
{
	gt_trak_atom *trak;
	gt_tkhd_atom *tkhd;

	trak = gt_alloc_atom (GTA_track, 1 + subatoms);
	if (trak) {
		trak->memb = 1;
		(gt_atom *) tkhd = trak->suba[0] = gt_alloc_atom (GTA_track_header, 0);
		tkhd->flags[2] = (char) (0x01 | 0x02);
		tkhd->ctime = gt_time();
		tkhd->mtime = gt_time();
		tkhd->track_id = id;
		tkhd->duration = 1000 * 12;
		tkhd->matrix[1] = 1;
		tkhd->matrix[17]= 1;
		tkhd->matrix[32]= 64;
		tkhd->width.high = width;
		tkhd->height.high= height;
		trak->size += tkhd->size;

#ifdef DEBUG
		fprintf (stderr, "gt_track_new() size=%d\n", trak->size);
#endif
	}
	return (trak);
}

/*
 * create a new sample table atom and it's needed subatoms
 */
gt_atom *
gt_sample_new (int format, int width, int height, int num, int subatoms)
{
	gt_atom *atom;
	gt_stsd_atom *stsd;
	gt_stsd_pano_entry *pe;

	atom = gt_alloc_atom (GTA_sample_table, 1 + subatoms);
	if (atom && format) {
		if (format == GT_VID_FMT_PANO) {
			stsd = gt_alloc_atom (GTA_sample_desc, 2);
			if (stsd) {
				stsd->count = 1;
				pe = (gt_stsd_pano_entry *)stsd->tab;
				pe->size = 152;
				pe->format = format;
				pe->index = 1;
				pe->track_id = 1;
				pe->hpan_start.high = 0;
				pe->hpan_end.high = 360;
				pe->vpan_top.high = 20;
				pe->vpan_bottom.high = -20;
				pe->size_x = width;
				pe->size_y = height * num;
				pe->no_frames = num;
				pe->no_frames_x = 1;
				pe->no_frames_y = num;
				pe->color_depth = 24;
				stsd->size = 16 + pe->size;
				atom->suba[0] = (gt_atom *)stsd;
				atom->size += stsd->size;
				atom->memb++;
			}
		} else {
			stsd = gt_alloc_atom (GTA_sample_desc, 1);
			if (stsd) {
				stsd->count = 1;
				stsd->tab->size = 86;
				stsd->tab->format = format;
				stsd->tab->index = 1; /* points to an entry in 'dref' */
				stsd->tab->spat_qual = 1023;
				stsd->tab->width = width;
				stsd->tab->height= height;
				stsd->tab->hres.high = 72;
				stsd->tab->vres.high = 72;
				stsd->tab->frame_count = 1;
				stsd->tab->depth = 24;
				stsd->tab->ctab_id = -1;
				atom->suba[0] = (gt_atom *)stsd;
				atom->size += stsd->size;
				atom->memb++;
			}
		}
	}
	return atom;
}

/*
 */
gt_atom *
gt_data_info_new (int subatoms)
{
	gt_atom *atom;
	gt_dref_atom *dref;

	atom = gt_alloc_atom (GTA_data_info, 1 + subatoms);
	if (atom) {
		dref = (gt_dref_atom *)gt_alloc_atom (GTA_data_ref, 1);
		dref->count = 1;
		dref->tab->size = 12;
		dref->tab->type = GT_ALIAS;
		dref->tab->flags[2] = 0x01; /* self reference */
		atom->memb++;
		atom->suba[0] = (gt_atom *)dref;
		atom->size += dref->size;
	}
	return atom;
}

