/*
 * gt.c
 * GnuTime module
 *
 * Copyright (C) 1998,99 Rasca, Berlin
 * EMail: thron@gmx.de
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <stdio.h>
#include <stdlib.h>
#include <time.h>
#include "gt.h"

/*
 * alloc space for a named atom, the following integer allocs space:
 *   for container atoms some space for pointers to subatoms
 *   for leaf atoms it depends on the atom type, e.g. some space for tab entries
 */
void *
gt_alloc_atom (int type, int nmb)
{
	gt_atom *p;
	int size = 8, mem = sizeof(gt_atom);
	int is_cont = 0;

	switch (type) {
		/* container atoms */
		case GTA_movie:
		case GTA_track:
		case GTA_media:
		case GTA_media_info:
		case GTA_data_info:
		case GTA_sample_table:
		case GTA_edit:
		case GTA_base_media_header:
			mem += sizeof(gt_atom*) * nmb;
			is_cont = 1;
			break;
		/* leaf atoms */
		case GTA_movie_header:
			mem = sizeof (gt_mvhd_atom);
			size = 108;
			break;
		case GTA_movie_data:
			mem = sizeof (mdat_atom) + nmb;
			size = 8 + nmb;
			break;
		case GTA_track_header:
			mem = sizeof (gt_tkhd_atom);
			size = 92;
			break;
		case GTA_media_header:
			mem = sizeof (gt_mdhd_atom);
			size = 32;
			break;
		case GTA_handler_ref:
			mem = sizeof (gt_hdlr_atom) + nmb + 2;
			size = 32 + nmb + 1;
			break;
		case GTA_video_media_header:
		case GTA_sound_media_header:
			mem = sizeof (gt_vmhd_atom);
			size = 20;
			break;
		case GTA_base_media_info:
			mem = sizeof (gt_gmin_atom);
			size = 24;
			break;
		case GTA_data_ref:
			mem = sizeof (gt_dref_atom) + nmb * sizeof(gt_dref_entry);
			size = 16 + nmb * 12;
			break;
		case GTA_sample_desc:
			mem = sizeof (gt_stsd_atom) + nmb * sizeof (gt_stsd_entry);
			size = 16 + nmb * 86;
			break;
		case GTA_time_to_sample:
			mem = sizeof (gt_stts_atom) + nmb * sizeof(gt_stts_entry);
			size = 16 + nmb * 8;
			break;
		case GTA_sample_size:
			mem = sizeof (gt_stsz_atom) + nmb * sizeof (gt_stsz_entry);
			size = 20 + nmb * 4;
			break;
		case GTA_sample_to_chunk:
			mem = sizeof (gt_stsc_atom) + nmb * sizeof (gt_stsc_entry);
			size = 16 + nmb * 12;
			break;
		case GTA_chunk_offset:
			mem = sizeof (gt_stco_atom) + nmb * sizeof (gt_stco_entry);
			size = 16 + nmb * 4;
			break;
		case GTA_free:
			mem = sizeof (free_atom);
			size = 8 + nmb;
			break;
		case GTA_skip:
			mem = sizeof (skip_atom);
			size = 8 + nmb;
			break;
		case GTA_user_data:
			mem = sizeof (gt_udta_atom) + nmb * sizeof(gt_udta_entry);
			size = 8 + 12 * nmb;	/* at least */
			break;
		default:
			fprintf (stderr, "gt_alloc_atom() unknown atom type: %X\n", type);
			return (NULL);
			break;
	}
	p = calloc (mem, 1);
	if (!p)
		return (NULL);
	p->size = size;
	p->type = type;
	if (is_cont && nmb) {
		p->suba = (gt_atom **)(p+1);
	} else if (nmb) {
		int i;
		switch (type) {
			case GTA_movie_data:
				((mdat_atom *)p)->data = (char *)((mdat_atom*)p+1);
				break;
			case GTA_handler_ref:
				((gt_hdlr_atom *)p)->comp_name = (char*)((gt_hdlr_atom*)p+1);
				break;
			case GTA_data_ref:
			case GTA_sample_desc:
			case GTA_time_to_sample:
			case GTA_sample_to_chunk:
			case GTA_chunk_offset:
				((gt_dref_atom *)p)->tab = (void *)((gt_dref_atom*)p+1);
				break;
			case GTA_sample_size:
				((gt_stsz_atom *)p)->tab = (void *)((gt_stsz_atom*)p+1);
				break;
			case GTA_skip:
				break;
			case GTA_user_data:
				((gt_udta_atom *)p)->count = nmb;
				for (i = 0; i < nmb; i++) {
					((gt_udta_atom *)p)->tab[i] = (gt_udta_entry *)
						(((char *)p)+sizeof(gt_udta_atom) +
						i * sizeof(gt_udta_entry));
				}
				break;
			default:
				fprintf (stderr, "error in gt_alloc_atom(%x[%c%c%c%c], %d)\n",
					type, *((char *)&type)+3, *((char*)&type)+2,
						*((char *)&type+1), *((char*)&type), nmb);
				break;
		}
	}
	return (p);
}

/*
 * free this and all subatoms
 */
void
gt_atom_free (gt_atom *atom)
{
	int i;

	if (!atom)
        return;
	if (GT_IS_CONTAINER(atom)) {
		for (i = 0; i < atom->memb; i++) {
			gt_atom_free (atom->suba[i]);
		}
	} else {
		free (atom);
	}
}


/*
 * write movie header leaf atom
 */
int
write_mvhd_atom (gt_atom *a, FILE *fp)
{
	gt_mvhd_atom *m = (gt_mvhd_atom *)a;
	int rc = 0;

	if (m->size == 0)
		m->size = 108 /* sizeof (mvhd_atom) doesn't work cause of alignm? */;
	rc += gt_write4byte (m->size, fp);
	rc += gt_write4byte (m->type, fp);
	rc += fwrite (&m->version, 1, 1, fp);
	rc += fwrite (&m->flags, 1, 3, fp);
	rc += gt_write4byte (m->ctime, fp);
	rc += gt_write4byte (m->mtime, fp);
	rc += gt_write4byte (m->time_scale, fp);
	rc += gt_write4byte (m->duration, fp);
	rc += gt_write2byte (m->pref_rate.high, fp);
	rc += gt_write2byte (m->pref_rate.low, fp);
	rc += fwrite (&m->pref_volume.high, 1, 1, fp);
	rc += fwrite (&m->pref_volume.low, 1, 1, fp);
	rc += fwrite (&m->reserved, 1, 10, fp);
	rc += fwrite (&m->matrix, 1, 36, fp);
	rc += gt_write4byte (m->preview_time, fp);
	rc += gt_write4byte (m->preview_duration, fp);
	rc += gt_write4byte (m->poster_time, fp);
	rc += gt_write4byte (m->sel_time, fp);
	rc += gt_write4byte (m->sel_duration, fp);
	rc += gt_write4byte (m->current_time, fp);
	rc += gt_write4byte (m->next_track_id, fp);
	return (rc);
}

/*
 * write a free, skip or mdat atom
 */
int
write_mdat_atom (gt_atom *a, FILE *fp)
{
	mdat_atom *m = (mdat_atom *)a;
	int rc = 0;

	rc += gt_write4byte (m->size, fp);
	rc += gt_write4byte (m->type, fp);
	if ((m->type == GTA_free) || (m->type == GTA_skip)) {
		/* write fill bytes */
		int i;
		for (i = 0; i < m->size-8; i++) {
			rc += fwrite ("\0", 1, 1, fp);
		}
	} else {
		rc += fwrite (m->data, 1, m->size-8, fp);
	}
	return (rc);
}

/*
 * write user data atom
 */
int
write_udta_atom (gt_atom *a, FILE *fp)
{
	gt_udta_atom *u = (gt_udta_atom *)a;
	int rc=0, i, j;

	if (u->type != GTA_user_data)
		return (0);
	if (u->size == 0) {
		u->size = 8;
	}
	rc += gt_write4byte (u->size, fp);
	rc += gt_write4byte (u->type, fp);

	for (i = 0; i < u->count; i++) {
		rc += gt_write4byte (u->tab[i]->size, fp);
		rc += gt_write4byte (u->tab[i]->type, fp);
		if (u->tab[i]->size > 12) {
			rc += gt_write2byte (u->tab[i]->tlen, fp);
			rc += gt_write2byte (u->tab[i]->lang, fp);
			for (j = 0; j < u->tab[i]->tlen; j++) {
				rc += fwrite (u->tab[i]->text+j, 1, 1, fp);
			}
		} else if (u->tab[i]->size > 8) {
			rc += gt_write4byte ((long)u->tab[i]->text, fp);
		}
	}
	return (rc);
}

/*
 * write track header leaf atom
 */
int
write_tkhd_atom (gt_atom *a, FILE *fp)
{
	gt_tkhd_atom *tkhd = (gt_tkhd_atom *)a;
	int rc = 0;

	if (tkhd->size == 0)
		tkhd->size = 92;
	rc += gt_write4byte (tkhd->size, fp);
	rc += gt_write4byte (tkhd->type, fp);
	rc += fwrite (&tkhd->version, 1, 1, fp);
	rc += fwrite (&tkhd->flags, 1, 3, fp);
	rc += gt_write4byte (tkhd->ctime, fp);
	rc += gt_write4byte (tkhd->mtime, fp);
	rc += gt_write4byte (tkhd->track_id, fp);
	rc += fwrite (&tkhd->reserved1, 1, 4, fp);
	rc += gt_write4byte (tkhd->duration, fp);
	rc += fwrite (&tkhd->reserved2, 1, 8, fp);
	rc += gt_write2byte (tkhd->layer, fp);
	rc += gt_write2byte (tkhd->alternate_group, fp);
	rc += fwrite (&tkhd->volume.high, 1, 1, fp);
	rc += fwrite (&tkhd->volume.low,  1, 1, fp);
	rc += fwrite (&tkhd->reserved3, 1, 2, fp);
	rc += fwrite (&tkhd->matrix, 1, 36, fp);
	rc += gt_write2byte (tkhd->width.high, fp);
	rc += gt_write2byte (tkhd->width.low, fp);
	rc += gt_write2byte (tkhd->height.high, fp);
	rc += gt_write2byte (tkhd->height.low, fp);
	return (rc);
}

/*
 * write media header leaf atom
 */
int
write_mdhd_atom (gt_atom *a, FILE *fp)
{
	gt_mdhd_atom *m = (gt_mdhd_atom *)a;
	int rc = 0;

	if (m->size == 0)
		m->size = sizeof (gt_mdhd_atom);
	rc += gt_write4byte (m->size, fp);
	rc += gt_write4byte (m->type, fp);
	rc += fwrite (&m->version, 1, 1, fp);
	rc += fwrite (&m->flags, 1, 3, fp);
	rc += gt_write4byte (m->ctime, fp);
	rc += gt_write4byte (m->mtime, fp);
	rc += gt_write4byte (m->time_scale, fp);
	rc += gt_write4byte (m->duration, fp);
	rc += gt_write2byte (m->language, fp);
	rc += gt_write2byte (m->quality, fp);
	return (rc);
}

/*
 * write video media information header leaf atom
 */
int
write_vmhd_atom (gt_atom *a, FILE *fp)
{
	gt_vmhd_atom *m = (gt_vmhd_atom *)a;
	int rc = 0;

	if (m->size == 0)
		m->size = sizeof (gt_vmhd_atom);
	rc += gt_write4byte (m->size, fp);
	rc += gt_write4byte (m->type, fp);
	rc += fwrite (&m->version, 1, 1, fp);
	rc += fwrite (&m->flags, 1, 3, fp);
	rc += gt_write2byte (m->grmode, fp);
	rc += gt_write2byte (m->opcolor[0], fp);
	rc += gt_write2byte (m->opcolor[1], fp);
	rc += gt_write2byte (m->opcolor[2], fp);
	return (rc);
}

/*
 * write video media information header leaf atom
 */
int
write_gmin_atom (gt_atom *a, FILE *fp)
{
	gt_gmin_atom *m = (gt_gmin_atom *)a;
	int rc = 0;

	if (m->size == 0)
		m->size = sizeof (gt_vmhd_atom);
	rc += gt_write4byte (m->size, fp);
	rc += gt_write4byte (m->type, fp);
	rc += fwrite (&m->version, 1, 1, fp);
	rc += fwrite (&m->flags, 1, 3, fp);
	rc += gt_write2byte (m->grmode, fp);
	rc += gt_write2byte (m->opcolor[0], fp);
	rc += gt_write2byte (m->opcolor[1], fp);
	rc += gt_write2byte (m->opcolor[2], fp);
	rc += gt_write2byte (m->balance, fp);
	rc += gt_write2byte (m->reserved, fp);
	return (rc);
}

/*
 * write media handler reference leaf atom
 */
int
write_hdlr_atom (gt_atom *a, FILE *fp)
{
	gt_hdlr_atom *m = (gt_hdlr_atom *)a;
	int rc = 0, len, i;

	if (m->size == 0)
		m->size = sizeof (gt_vmhd_atom);
	rc += gt_write4byte (m->size, fp);
	rc += gt_write4byte (m->type, fp);
	rc += fwrite (&m->version, 1, 1, fp);
	rc += fwrite (&m->flags, 1, 3, fp);
	rc += gt_write4byte (m->comp_type, fp);
	rc += gt_write4byte (m->comp_subtype, fp);
	rc += gt_write4byte (m->comp_man, fp);
	rc += gt_write4byte (m->comp_flags, fp);
	rc += gt_write4byte (m->comp_flags_mask, fp);
	if (!m->comp_name) {
		rc += fwrite ("\0", 1, 1, fp);
	} else {
		len = m->comp_name[0];
		if (len > 31) len = 31;
		for (i = 0; i < len+1; i++) {
			rc += fwrite (m->comp_name+i, 1, 1, fp);
		}
	}
	return (rc);
}

/*
 * todo
 */
int
write_dref_atom (gt_atom *a, FILE *fp)
{
	gt_dref_atom *d = (gt_dref_atom *)a;
	int rc = 0, i;

	rc += gt_write4byte (d->size, fp);
	rc += gt_write4byte (d->type, fp);
	rc += fwrite (&d->version, 1, 1, fp);
	rc += fwrite (&d->flags, 1, 3, fp);
	rc += gt_write4byte (d->count, fp);
	for (i = 0; i < d->count; i++) {
		rc += gt_write4byte (d->tab[i].size, fp);
		rc += gt_write4byte (d->tab[i].type, fp);
		rc += fwrite (&d->tab[i].version, 1, 1, fp);
		rc += fwrite (&d->tab[i].flags, 1, 3, fp);
	}
	return (rc);
}

/*
 * write sample desc atom
 */
int
write_stsd_atom (gt_atom *a, FILE *fp)
{
	int i, rc = 0;
	gt_stsd_atom *s = (gt_stsd_atom *)a;
	gt_stsd_entry *te;
	gt_stsd_pano_entry *pte;

	rc += gt_write4byte (s->size, fp);
	rc += gt_write4byte (s->type, fp);
	rc += fwrite (&s->version, 1, 1, fp);
	rc += fwrite (&s->flags, 1, 3, fp);
	rc += gt_write4byte (s->count, fp);
	te = s->tab;
	for (i = 0; i < s->count; i++) {
		if (te->format == GT_VID_FMT_PANO) {
			pte = (gt_stsd_pano_entry *)te;
			rc += gt_write4byte(pte->size, fp);
			rc += gt_write4byte(pte->format, fp);
			rc += fwrite (&pte->reserved1, 1, 6, fp);
			rc += gt_write2byte(pte->index, fp);
			rc += gt_write4byte(pte->version, fp);
			rc += gt_write4byte(pte->track_id, fp);
			rc += gt_write4byte(pte->lowres_id, fp);
			rc += fwrite (&pte->reserved2, 1, 24, fp);
			rc += gt_write4byte(pte->hotspot_id, fp);
			rc += fwrite (&pte->reserved3, 1, 36, fp);
			rc += gt_write2byte(pte->hpan_start.high, fp);
			rc += gt_write2byte(pte->hpan_start.low, fp);
			rc += gt_write2byte(pte->hpan_end.high, fp);
			rc += gt_write2byte(pte->hpan_end.low, fp);
			rc += gt_write2byte(pte->vpan_top.high, fp);
			rc += gt_write2byte(pte->vpan_top.low, fp);
			rc += gt_write2byte(pte->vpan_bottom.high, fp);
			rc += gt_write2byte(pte->vpan_bottom.low, fp);
			rc += gt_write2byte(pte->min_zoom.high, fp);
			rc += gt_write2byte(pte->min_zoom.low, fp);
			rc += gt_write2byte(pte->max_zoom.high, fp);
			rc += gt_write2byte(pte->max_zoom.low, fp);
			rc += gt_write4byte(pte->size_x, fp);
			rc += gt_write4byte(pte->size_y, fp);
			rc += gt_write4byte(pte->no_frames, fp);
			rc += gt_write2byte(pte->reserved4, fp);
			rc += gt_write2byte(pte->no_frames_x, fp);
			rc += gt_write2byte(pte->no_frames_y, fp);
			rc += gt_write2byte(pte->color_depth, fp);
			rc += gt_write4byte(pte->hs_size_x, fp);
			rc += gt_write4byte(pte->hs_size_y, fp);
			rc += gt_write2byte(pte->reserved5, fp);
			rc += gt_write2byte(pte->hs_no_frames_x, fp);
			rc += gt_write2byte(pte->hs_no_frames_y, fp);
			rc += gt_write2byte(pte->hs_color_depth, fp);
		} else {
			rc += gt_write4byte(te->size, fp);
			rc += gt_write4byte(te->format, fp);
			rc += fwrite (&te->reserved, 1, 6, fp);
			rc += gt_write2byte(te->index, fp);
			rc += gt_write2byte(te->version, fp);
			rc += gt_write2byte(te->rev_level, fp);
			rc += gt_write4byte(te->vendor, fp);
			rc += gt_write4byte(te->temp_qual, fp);
			rc += gt_write4byte(te->spat_qual, fp);
			rc += gt_write2byte(te->width, fp);
			rc += gt_write2byte(te->height, fp);
			rc += gt_write2byte(te->hres.high, fp);
			rc += gt_write2byte(te->hres.low, fp);
			rc += gt_write2byte(te->vres.high, fp);
			rc += gt_write2byte(te->vres.low, fp);
			rc += gt_write4byte(te->data_size, fp);
			rc += gt_write2byte(te->frame_count, fp);
			rc += fwrite (&te->comp_name, 1, 32, fp);
			rc += gt_write2byte(te->depth, fp);
			rc += gt_write2byte(te->ctab_id, fp);
		}
		te = (gt_stsd_entry *)(((char *)te) + te->size);
	}
	return (rc);
}

/*
 * write time to sample atom
 */
int
write_stts_atom (gt_atom *a, FILE *fp)
{
	int i;
	gt_stts_atom *s = (gt_stts_atom *)a;

	gt_write4byte (s->size, fp);
	gt_write4byte (s->type, fp);
	fwrite (&s->version, 1, 1, fp);
	fwrite (&s->flags, 1, 3, fp);
	gt_write4byte (s->count, fp);
	for (i = 0; i < s->count; i++) {
		gt_write4byte(s->tab[i].num_samples, fp);
		gt_write4byte(s->tab[i].duration, fp);
	}
	return (0);
}

/*
 * write sample size atom
 */
int
write_stsz_atom (gt_atom *a, FILE *fp)
{
	gt_stsz_atom *s = (gt_stsz_atom *)a;
	int i, rc = 0;

	rc += gt_write4byte (s->size, fp);
	rc += gt_write4byte (s->type, fp);
	rc += fwrite (&s->version, 1, 1, fp);
	rc += fwrite (&s->flags, 1, 3, fp);
	rc += gt_write4byte (s->sample_size, fp);
	rc += gt_write4byte (s->count, fp);
	for (i = 0; i < s->count; i++) {
		rc += gt_write4byte(s->tab[i].size, fp);
	}
	return (rc);
}

/*
 * write sample to chunk leaf atom
 */
int
write_stsc_atom (gt_atom *a, FILE *fp)
{
	gt_stsc_atom *s = (gt_stsc_atom *)a;
	int i, rc = 0;

	rc += gt_write4byte (s->size, fp);
	rc += gt_write4byte (s->type, fp);
	rc += fwrite (&s->version, 1, 1, fp);
	rc += fwrite (&s->flags, 1, 3, fp);
	rc += gt_write4byte (s->count, fp);
	for (i = 0; i < s->count; i++) {
		rc += gt_write4byte(s->tab[i].first_chunk, fp);
		rc += gt_write4byte(s->tab[i].samples_per_chunk, fp);
		rc += gt_write4byte(s->tab[i].sample_id, fp);
	}
	return (rc);
}
/*
 * write sample chunk offset atom
 */

int
write_stco_atom (gt_atom *a, FILE *fp)
{
	int i;
	gt_stco_atom *s = (gt_stco_atom *)a;

	gt_write4byte (s->size, fp);
	gt_write4byte (s->type, fp);
	fwrite (&s->version, 1, 1, fp);
	fwrite (&s->flags, 1, 3, fp);
	gt_write4byte (s->count, fp);
	for (i = 0; i < s->count; i++) {
		gt_write4byte(s->tab[i].offset, fp);
	}
	return (0);
}

/*
 * write an atom and all it's subatoms
 */
int
gt_write_atom (gt_atom *a, FILE *fp)
{
	int i, rc = 0;
	char c = 0;

#ifdef DEBUG2
	fprintf (stderr,"gt_write_atom(%c%c%c%c) size=0x%x\n", (char)(a->type>>24),
				(char)(a->type>>16),(char)(a->type>>8), (char)a->type,
				a->size);
#endif
	if (GT_IS_CONTAINER (a)) {
		rc += gt_write4byte (a->size, fp);
		rc += gt_write4byte (a->type, fp);
		for (i = 0; i < a->memb; i++) {
			rc += gt_write_atom (a->suba[i], fp);
		}
	} else
		switch (a->type) {
		/* leaf atoms
		 */
		case GTA_movie_header:
			rc = write_mvhd_atom (a, fp); break;
		case GTA_user_data:
			rc = write_udta_atom (a, fp); break;
		case GTA_track_header:
			rc = write_tkhd_atom (a, fp); break;
		case GTA_media_header:
			rc = write_mdhd_atom (a, fp); break;
		case GTA_handler_ref:
			rc = write_hdlr_atom (a, fp); break;
		case GTA_data_ref:
			rc = write_dref_atom (a, fp); break;
		case GTA_video_media_header:
			rc = write_vmhd_atom (a, fp); break;
		case GTA_base_media_info:
			rc = write_gmin_atom (a, fp); break;
		case GTA_sample_desc:
			rc = write_stsd_atom (a, fp); break;
		case GTA_time_to_sample:
			rc = write_stts_atom (a, fp); break;
		case GTA_sample_size:
			rc = write_stsz_atom (a, fp); break;
		case GTA_sample_to_chunk:
			rc = write_stsc_atom (a, fp); break;
		case GTA_chunk_offset:
			rc = write_stco_atom (a, fp); break;
		case GTA_movie_data:
		case GTA_free:
		case GTA_skip:
			rc = write_mdat_atom (a, fp); break;
		default:
			fprintf(stderr, "gt_write_atom() Unknown atom type: %X size=%d\n",
						a->type,a->size);
			/* write fill bytes */
			gt_write4byte (a->size, fp);
			gt_write4byte (a->type, fp);
			for (i = 0; i < a->size - 8; i++) {
				fwrite ((char *)&c, 1, 1, fp);
			}
			rc = -1;
			break;
	}
	return (rc);
}

/*
 * misc tool functions */

/*
 * get qt/mac time value from current unix time
 */
unsigned int
gt_time (void)
{
	time_t t;

	time (&t);
	/* todo: this is just about the value it should be :) */
	return (t+(66*31536000)+1468800);
}

/*
 * todo
 */
gt_atom *
gt_get_track (FILE *fp)
{
	long start, pos;
	int rc;
	gt_atom *track, ac;

	start = pos = ftell (fp);
	track = gt_alloc_atom (GTA_track, MAX_SUBATOMS);
	track->memb = 0;

	rc = gt_read4byte (&track->size, fp);
	rc+= gt_read4byte (&track->type, fp);
	if ((rc < 8) || (track->type != GTA_track)) {
		gt_atom_free (track);
		return (NULL);
	}
	while (pos < start+track->size) {
		rc = gt_read4byte (&ac.size, fp);
		rc+= gt_read4byte (&ac.type, fp);
		if ((rc < 8) || (ac.type < 0x20202020)) {
			gt_atom_free (track);
			return (NULL);
		}
		track->suba[track->memb++] = gt_alloc_atom (ac.type, 0);
		track->suba[track->memb-1]->type = ac.type;
		track->suba[track->memb-1]->size = ac.size;
		fseek (fp, ac.size-8, SEEK_CUR);
		pos += 8 + ac.size;
	}
	return (track);
}

/*
 * todo ..
 */
gt_atom *
gt_get_movie (FILE *fp)
{
	int rc, start, pos;
	gt_atom ac, *ta, *moov = gt_alloc_atom (GTA_movie, MAX_SUBATOMS);

	start = ftell (fp);
	while (1) {
		rc = gt_read4byte (&moov->size, fp);
		rc+= gt_read4byte (&moov->type, fp);
		if ((rc < 8) || (moov->type < 0x20202020)) {
			gt_atom_free (moov);
			return (NULL);
		}
#ifdef DEBUG2
		printf (" get_movie() %c%c%c%c size=0x%X\n", ((char *)&moov->type)[3],
				((char*)&moov->type)[2], ((char*)&moov->type)[1],
				((char*)&moov->type)[0],moov->size);
#endif
		if (moov->type == GTA_movie) {
			break;
		} else {
			fseek (fp, moov->size - 8, SEEK_CUR);
		}
	}

	while (1) {
		pos = ftell (fp);
		if ((pos - start) >= moov->size) {
			break;
		}
		rc = gt_read4byte (&ac.size, fp);
		rc+= gt_read4byte (&ac.type, fp);
		if (rc < 8) {
			gt_atom_free (moov);
			return (NULL);
		}
#ifdef DEBUG2
		printf (" -get_movie() %c%c%c%c size=0x%X\n", ((char *)&ac.type)[3],
				((char*)&ac.type)[2], ((char*)&ac.type)[1],
				((char*)&ac.type)[0],ac.size);
#endif
		switch (ac.type) {
			/* leaf atoms */
			case GTA_movie_header:
			case GTA_track_header:
			case GTA_edit_list:
			case GTA_handler_ref:
			case GTA_media_header:
			case GTA_data_ref:
			case GTA_video_media_header:
			case GTA_sound_media_header:
			case GTA_chunk_offset:
			case GTA_sample_to_chunk:
			case GTA_sample_desc:
			case GTA_time_to_sample:
			case GTA_sample_size:
			case GTA_sync_sample:
			case GTA_user_data:
				fseek (fp, ac.size-8, SEEK_CUR);
				break;
			/* container atoms */
			case GTA_track:
			case GTA_edit:
			case GTA_media:
			case GTA_media_info:
			case GTA_data_info:
			case GTA_sample_table:
				switch (ac.type) {
					case GTA_track:
						fseek (fp, -8, SEEK_CUR);
						moov->suba[moov->memb++] = gt_get_track (fp);
						break;
					default:
						ta = gt_alloc_atom (ac.type, MAX_SUBATOMS);
						ta->size = ac.size;
						ta->type = ac.type;
						moov->suba[moov->memb++] = ta;
						fseek (fp, ac.size-8, SEEK_CUR);
						break;
				}
				break;
			default:
				/* todo */
				printf ("Unknow atom: %X\n", ac.type);
				fseek (fp, ac.size-8, SEEK_CUR);
				/* return (NULL); */
				break;
		}
	}
	return (moov);
}


/*
 */
void
gt_swap4byte (ui32 *n)
{
	unsigned char t, *p = (unsigned char *)n;

	t = p[0];
	p[0] = p[3];
	p[3] = t;
	t = p[1];
	p[1] = p[2];
	p[2] = t;
}

/*
 */
void
gt_swap2byte (ui16 *n)
{
	unsigned char t, *p = (unsigned char *)n;
	t = p[0];
	p[0] = p[1];
	p[1] = t;
}

/* IO */

/*
 */
int
gt_write2byte (ui16 n, FILE *fp)
{
	int rc;
#ifdef LSB
	gt_swap2byte (&n);
#endif
	rc = fwrite (&n, 1, 2, fp);
	return (rc);
}

/*
 */
int
gt_write4byte (ui32 n, FILE *fp)
{
	int rc;
#ifdef LSB
	gt_swap4byte (&n);
#endif
	rc = fwrite (&n, 1, 4, fp);
	return (rc);
}

/*
 */
int
gt_read2byte (ui16 *n, FILE *fp)
{
	int rc;

	rc = fread (n, 1, 2, fp);
#ifdef LSB
	gt_swap2byte (n);
#endif
	return (rc);
}

/*
 */
int
gt_read4byte (ui32 *n, FILE *fp)
{
	int rc;

	rc = fread (n, 1, 4, fp);
#ifdef LSB
	gt_swap4byte (n);
#endif
#ifdef DEBUG2
	printf ("%s: gt_read4byte() pos=%ld val=%X ret=%d\n",
			__FILE__, ftell(fp), *n, rc);
#endif
	return (rc);
}

