/*

    xpuyopuyo - pnetcomm.c    Copyright(c) 1999,2000 Justin David Smith
    justins(at)chaos2.org     http://chaos2.org/
    
    Network communication code
    

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/


#include <xpuyopuyo.h>


#if USE_NETWORK /* Allow network support? */


#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include <fcntl.h>
#include <errno.h>
#include <pnetint.h>
#include <pconfig.h>
#include <pwindow.h>
#include <ptime.h>


int pnet_ready_to_send(pconfig *c, psocket *socket) {

   fd_set fdesc;        /* Descriptor */
   struct timeval t;    /* Do not delay on select() call */
   int selectresult;    /* Result of select call */
   
   /* If no socket, return "ready" to prevent lockup */
   if(socket == NULL) return(1);

   /* Setup descriptor list and timer */   
   FD_ZERO(&fdesc);
   FD_SET(P_COMM_SOCKET(socket), &fdesc);
   TV_SET(t, 0, P_NET_SEND_TIMEOUT);

   /* Check for ready to send */
   selectresult = select(P_COMM_SOCKET(socket) + 1, NULL, &fdesc, NULL, &t);
   if(selectresult < 0) {
      pnet_set_error("ready_to_send", strerror(errno));
      pnet_shutdown_all(socket);
   }

   /* Return result */
   return(selectresult);

}


int pnet_ready_to_recv(pconfig *c, psocket *socket) {

   fd_set fdesc;        /* Descriptor */
   struct timeval t;    /* Do not delay on select() call */
   int selectresult;    /* Result of select call */

   /* If no socket, return "ready" to prevent lockup */
   if(socket == NULL) return(1);

   /* Setup descriptor list and timer */   
   FD_ZERO(&fdesc);
   FD_SET(P_COMM_SOCKET(socket), &fdesc);
   TV_SET(t, 0, P_NET_SEND_TIMEOUT);

   /* Check for ready to send */
   selectresult = select(P_COMM_SOCKET(socket) + 1, &fdesc, NULL, NULL, &t);
   if(selectresult < 0) {
      pnet_set_error("ready_to_recv", strerror(errno));
      pnet_shutdown_all(socket);
   }

   /* Return result */
   return(selectresult);

}


int pnet_ready_to_recv_within(pconfig *c, psocket *socket, struct timeval *tv) {

   fd_set fdesc;        /* Descriptor */
   struct timeval t;    /* Do not delay on select() call */
   int selectresult;    /* Result of select call */

   /* If no socket, return "ready" to prevent lockup */
   if(socket == NULL) return(1);

   /* Setup descriptor list and timer */   
   FD_ZERO(&fdesc);
   FD_SET(P_COMM_SOCKET(socket), &fdesc);
   TV_COPY(t, *tv);

   /* Check for ready to send */
   selectresult = select(P_COMM_SOCKET(socket) + 1, &fdesc, NULL, NULL, &t);
   if(selectresult < 0) {
      pnet_set_error("ready_to_recv", strerror(errno));
      pnet_shutdown_all(socket);
   }

   /* Return result */
   return(selectresult);

}


int pnet_recv_null(pconfig *c, psocket *socket, int size) {

   byte *buffer;              /* Character buffer to read into */

   if(size <= 0) return(0);   
   buffer = (unsigned char *)malloc(size);
   if(buffer == NULL) {
      pnet_set_error("recv_null", "failed to allocate temporary buffer");
      return(-1);
   }
   
   if(pnet_recv_buffer(c, socket, buffer, size) < 0) {
      free(buffer);
      return(-1);
   }
   
   free(buffer);
   return(0);

}


int pnet_send_header(pconfig *c, psocket *socket, word type, word size) {

   if(pnet_send_word(c, socket, type) < 0) return(-1);
   if(pnet_send_word(c, socket, size) < 0) return(-1);
   return(0);

}


int pnet_recv_header(pconfig *c, psocket *socket, word *type, word *size) {

   if(pnet_recv_word(c, socket, type) < 0) return(-1);
   if(pnet_recv_word(c, socket, size) < 0) return(-1);
   return(0);

}


int pnet_send_word(pconfig *c, psocket *socket, word value) {

   value = htons(value);
   return(pnet_send_buffer(c, socket, (byte *)&value, sizeof(word)));

}


int pnet_recv_word(pconfig *c, psocket *socket, word *value) {

   if(value == NULL) return(-1);
   if(pnet_recv_buffer(c, socket, (byte *)value, sizeof(word)) < 0) return(-1);
   *value = ntohs(*value);
   return(0);

}


int pnet_send_dword(pconfig *c, psocket *socket, dword value) {

   value = htonl(value);
   return(pnet_send_buffer(c, socket, (byte *)&value, sizeof(dword)));

}


int pnet_recv_dword(pconfig *c, psocket *socket, dword *value) {

   if(value == NULL) return(-1);
   if(pnet_recv_buffer(c, socket, (byte *)value, sizeof(dword)) < 0) return(-1);
   *value = ntohl(*value);
   return(0);

}


int pnet_send_buffer(pconfig *c, psocket *socket, const byte *buffer, int size) {

   int sel;
   int res;

   if(socket == NULL) return(0);
   if(buffer == NULL) return(-1);

   while((sel = pnet_ready_to_send(c, socket)) == 0);
   if(sel < 0) return(-1);

   #ifdef P_NET_DEBUG_COMM_VERBOSE
   printf("send_buffer:  Sending %d bytes of data ... ", size);
   #endif /* P_NET_DEBUG_COMM_VERBOSE */

   res = send(P_COMM_SOCKET(socket), buffer, size, 0);

   #ifdef P_NET_DEBUG_COMM_VERBOSE
   printf("%d\n", res);
   #endif /* P_NET_DEBUG_COMM_VERBOSE */

   if(res < 0) {
      pnet_set_error("send_buffer", strerror(errno));
      pnet_shutdown_all(socket);
      return(-1);
   } else if(res != size) {
      pnet_set_error("send_buffer", "Could not send all data.");
      pnet_shutdown_all(socket);
      return(-1);
   }
   
   socket->sent += size;
   return(0);

}


int pnet_recv_buffer(pconfig *c, psocket *socket, byte *buffer, int size) {

   struct timeval tv;
   int read;
   int sel;
   int res;

   if(socket == NULL) return(0);
   if(buffer == NULL) return(-1);

   while((sel = pnet_ready_to_recv(c, socket)) == 0);
   if(sel < 0) return(-1);

   #ifdef P_NET_DEBUG_COMM_VERBOSE
   printf("recv_buffer:  Waiting for %d bytes of data ... ", size);
   #endif /* P_NET_DEBUG_COMM_VERBOSE */

   read = 0;
   while(read < size) {
      res = recv(P_COMM_SOCKET(socket), buffer + read, size - read, 0);

      #ifdef P_NET_DEBUG_COMM_VERBOSE
      printf("%d\n", res);
      #endif /* P_NET_DEBUG_COMM_VERBOSE */

      if(res < 0) {
         pnet_set_error("recv_buffer", strerror(errno));
         pnet_shutdown_all(socket);
         return(-1);
      } else {
         read += res;
         sel = 1;
         if(read > 0 && read < size) {
            TV_SET(tv, 0, 100000);
            sel = pnet_ready_to_recv_within(c, socket, &tv);
            if(sel < 0) return(-1);
         }
         if(sel == 0 || read == 0) {
            pnet_set_error("recv_buffer", "timeout while waiting for data stream");
            pnet_shutdown_all(socket);
            return(-1);
         } /* If haven't read all data yet */
      }
   
   }

   socket->recv += size;
   return(0);

}


#endif /* Allow network? */
