/*
   
   xpuyopuyo - psound.c       Copyright(c) 2000 Justin David Smith
   justins(at)chaos2.org      http://chaos2.org/
    
   xpp sound driver
   

   This program is free software; you can redistribute it and/or modify 
   it under the terms of the GNU General Public License as published by 
   the Free Software Foundation; either version 2 of the License, or 
   (at your option) any later version.

   This program is distributed in the hope that it will be useful, 
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU 
   General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software Foundation, 
   Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/
#include <psound.h>
#include <psnprintf.h>



#if USE_SOUND



/* List of song names */
static const char *_p_sound_names[] = {
   "prelude.xm",
   "game.xm",
   "gameover.xm",
   NULL
};



static bool _p_sound_try_load(p_sound *s, p_sound_music id) {
/* p_sound_try_load
   Attempt to load the song given by the ID.  On success, true is returned
   and s->module will refer to the song.  On failure, false is returned.
   Note, Any previously loaded song must be unloaded before this function 
   is called.  */

   /* Temporary buffer to write filename into */
   char filename[P_SOUND_BUFFER];

   /* Construct the suspected filename for the song */
   p_snprintf(filename, P_SOUND_BUFFER, "%s/%s/%s", P_GLOBAL_DIR, P_SOUND_DIR, _p_sound_names[id]);
   
   /* Attempt to load the song */
   s->module = Player_Load(filename, 64, 0);
   if(s->module == NULL) {
      /* Failure; see what went wrong, then give up */
      fprintf(stderr, "Could not load module \"%s\": \"%s\"\n", _p_sound_names[id], MikMod_strerror(MikMod_errno));
      return(0);
   }
   
   /* Success: set module parametres and exit */
   s->module->wrap = 1;
   s->module->loop = 1;
   s->module->reppos = 0;
   return(!0);

}



void p_sound_init(void) {
/* p_sound_init
   This function must be called _exactly_ once, to initialise the sound 
   driver.  This function must be called before sound structures are
   created; for MikMod we'll just register everything.  */

   MikMod_RegisterAllDrivers();
   MikMod_RegisterAllLoaders();
   
}



static void _p_sound_mdmode(p_sound *s, int hqmixer) {
/* p_sound_mdmode
   Set driver (global) options.  */

   s->hqmixer = hqmixer;
   md_mode |= DMODE_SOFT_MUSIC | DMODE_INTERP;
   if(hqmixer) {
      md_mode |= DMODE_HQMIXER;
   } else {
      md_mode &= (~DMODE_HQMIXER);
   }
   md_reverb = P_SOUND_REVERB;

}



static void _p_sound_reset(p_sound *s) {
/* p_sound_reset
   Reset the mikmod driver - Warning: all sound samples and modules must
   be reloaded after this call.  Otherwise, nasty segfaults will occur.  */

   /*
   if(s->playing) Player_Stop();
   if(s->module != NULL) Player_Free(s->module);
   */

   #if LIBMIKMOD_VERSION >= 0x030107
   if(MikMod_Reset("")) {
   #else /* Version is old */
   if(MikMod_Reset()) {
   #endif /* Libmikmod ok? */
      fprintf(stderr, "Could not reinitialise sound: \"%s\"\n", MikMod_strerror(MikMod_errno));
   }
   s->playing = 0;

}


   
p_sound *p_sound_new(int enable, int hqmixer) {

   p_sound *s;

   if(!enable) return(NULL);

   s = (p_sound *)malloc(sizeof(p_sound));
   if(s == NULL) return(NULL);
   s->playing = 0;
   s->module = NULL;
   s->selection = -1;

   _p_sound_mdmode(s, hqmixer);

   /* Beware the kludgery */
   #if LIBMIKMOD_VERSION >= 0x030107
   if(MikMod_Init("")) {
   #else /* Version is old */
   if(MikMod_Init()) {
   #endif /* Libmikmod ok? */
      fprintf(stderr, "Could not initialise sound: \"%s\"\n", MikMod_strerror(MikMod_errno));
      p_sound_free(&s);
      return(NULL);
   }
   
   return(s);
   
}



void p_sound_config(p_sound **s, int enable, int hqmixer) {

   if(s == NULL) return;
   if(*s == NULL) {
      *s = p_sound_new(enable, hqmixer);
   } else {
      if(enable) {
         _p_sound_mdmode(*s, hqmixer);
         _p_sound_reset(*s);
      } else {
         p_sound_free(s);
      }
   }

}



void p_sound_free(p_sound **s) {

   if(s == NULL || *s == NULL) return;

   Player_Stop();
   if((*s)->module != NULL) {
      Player_Free((*s)->module);
   }
   MikMod_Exit();

   free(*s);
   *s = NULL;
   
}



void p_sound_start(p_sound *s, p_sound_music id) {

   /*int voice;*/

   if(s == NULL) return;
   _p_sound_reset(s);
   
   _p_sound_try_load(s, id);
   if(s->module == NULL) return;
      
   Player_Start(s->module);
   /*
   Player_SetPosition(0);
   for(voice = 0; voice <= 99; ++voice) {
      Player_Unmute(voice);
      Voice_SetVolume(voice, 256);
      Voice_SetPanning(voice, PAN_SURROUND);
   }*/
   
   s->playing = 1;
   s->selection = id;
   p_sound_update(s);

}



void p_sound_update(p_sound *s) {
   
   if(s == NULL || s->module == NULL) return;
   if(s->playing) {
      MikMod_Update();
      if(!Player_Active()) {
         Player_SetPosition(0);
      }
   }

}



void p_sound_stop(p_sound *s) {

   if(s == NULL || s->module == NULL) return;
   Player_Stop();
   s->playing = 0;
   s->selection = -1;

}



void p_sound_pause(p_sound *s) {

   if(s == NULL || s->module == NULL) return;
   if(!Player_Paused()) Player_TogglePause();

}



void p_sound_unpause(p_sound *s) {

   if(s == NULL || s->module == NULL) return;
   if(Player_Paused()) Player_TogglePause();

}



#endif /* USE_SOUND? */

