/*

    xpuyopuyo - pdialog-gtk.c       Copyright(c) 1999,2000 Justin David Smith
    justins(at)chaos2.org           http://chaos2.org/

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/


#include <config.h>
#include <pdialog-gtk.h>
#include <psnprintf.h>
#include <pfile.h>
#include <string.h>
#include <stdlib.h>
#include <stdio.h>
#include <errno.h>



typedef struct _pquery_gtk {
   GtkWidget *query_entry;       /* Entry source in query */
   char *query_buffer;           /* Destination to query string */
   int query_size;               /* Size of destination in query */
} pquery_gtk;



static inline void p_dialog_puyo_name(char *b) {

   char *p;
   int i;

   i = 4;
   p = b;
   while(i) {
      *p = 0;
      while(!*p) {
         *p = (rand() % 26) + 'a';
         if(*p == 'a' || *p == 'e' || *p == 'i' || *p == 'o' || *p == 'u') {
            if(!(i % 2)) *p = 0;
         } else {
            if(i % 2) *p = 0;
         }
      }
      p++;
      i--;
   }
   *p = 0;
   return;

}



static void p_dialog_smiley_clicked(GtkWidget *widget, GdkEventAny *event, gpointer data) {

   GtkWidget *msg = data;
   char buf[P_GTK_STRING_BUFFER];
   char b[P_GTK_STRING_BUFFER];
   char *s1;

   if(!msg) return;
   switch(rand() % 7) {
      case 0:  s1 = "I am Velius...  the Devil...\nYour wish is granted.\n  - Wiegraf (possessed by Zodiac Stone)\n        Final Fantasy Tactics, Square\n";
               break;
      case 1:  s1 = "Life... Dreams... Hope...\nWhere did they come from...\nAnd where are they headed?\nThese things... I will destroy!!\n  - Kefka, Final Fantasy VI\n";
               break;
      case 2:  s1 = "The end comes....  beyond chaos....\n  - Kefka (Fourth Tier), Final Fantasy VI\n";
               break;
      case 3:  s1 = "Be careful of forgetfulness.  Your lucky color is... blue?\n  - Cait Sith, Final Fantasy VII\n";
               break;
      case 4:  s1 = "Well, now that the world's coming to an end, I think you should start breeding Chocobos.  It'd be a healthy thing for you to do.\n  - Farmer in Final Fantasy VII\n";
               break;
      case 5:  s1 = "What you pursue will be yours...  but you will lose something very dear.\n  - Cait Sith, Final Fantasy VII\n";
               break;
      default: s1 = "QSSI WEHT ERAU DLRO\n  - Final Fantasy VI\n";
               break;
   }
   p_dialog_puyo_name(b);
   p_snprintf(buf, P_GTK_STRING_BUFFER, "%s\nThe random puyo name of the day is \"x%s%s\"\n", s1, b, b);
   gtk_label_set_text(GTK_LABEL(msg), buf);
   return;

}



GtkWidget *p_dialog_puyos(pwindow_gtk *w, GtkWidget *msg) {

   GtkWidget *grid;
   GtkWidget *puyo;
   GtkWidget *eventbox;
   int ht[P_DIALOG_PUYO_SIZE];
   int rnd;
   int yx;
   int yy;
   int i;
   int j;

   for(j = 0; j < P_DIALOG_PUYO_SIZE; j++) {
      ht[j] = P_DIALOG_PUYO_SIZE - j - 2;
   }
   ht[P_DIALOG_PUYO_SIZE - 1] += 2;

   grid = gtk_table_new(P_DIALOG_PUYO_SIZE, P_DIALOG_PUYO_SIZE - 1, TRUE);   
   if(!w->ready) return(grid);
   gtk_table_set_row_spacings(GTK_TABLE(grid), 0);
   gtk_table_set_col_spacings(GTK_TABLE(grid), 0);
   yx = -1;
   while(yx < 0) {
      yx = rand() % P_DIALOG_PUYO_SIZE;
      yy = rand() %(P_DIALOG_PUYO_SIZE - 1);
      if(ht[yx] > yy) yx = -1;
   }
   for(j = 0; j < P_DIALOG_PUYO_SIZE; j++) {
      for(i = ht[j]; i < P_DIALOG_PUYO_SIZE - 1; i++) {
         if(i == yx && j == yy) {
            rnd = 0;
            puyo = gtk_pixmap_new(P_IMAGEW(w, P_IMAGE_ROCK_IND), P_IMAGE_MASKW(w, P_IMAGE_ROCK_IND));
         } else {
            rnd = rand() % (P_COLORS_MAX - 1) + 1;
            puyo = gtk_pixmap_new(P_IMAGEW(w, P_IMAGE_UNJOINED(rnd)), P_IMAGE_MASKW(w, P_IMAGE_UNJOINED(rnd)));
         }
         if(rnd == 0) {
            eventbox = gtk_event_box_new();
            gtk_widget_set_events(eventbox, GDK_BUTTON_PRESS_MASK);
            gtk_container_add(GTK_CONTAINER(eventbox), puyo);
            gtk_signal_connect(GTK_OBJECT(eventbox), "button_press_event", GTK_SIGNAL_FUNC(p_dialog_smiley_clicked), msg);
            puyo = eventbox;
         }            
         gtk_table_attach(GTK_TABLE(grid), puyo, j, j + 1, i, i + 1, 0, 0, 0, 0);
      }
   }
   return(grid);

}



static void p_dialog_window_destroy(GtkWidget *widget, gpointer data) {

   p_dialog_window *dw = data;
   dw->waiting_for_close = FALSE;
   return;
   
}



static void p_dialog_window_close(GtkWidget *widget, gpointer data) {

   p_dialog_window *dw = data;
   gtk_widget_destroy(dw->dialog);
   return;

}



static void p_dialog_window_ok_clicked(GtkWidget *widget, gpointer data) {

   p_dialog_window *dw = data;
   if(dw->apply != NULL) dw->apply(dw);
   dw->dialog_accepted = TRUE;
   p_dialog_window_close(widget, dw);
   return;

}



static void p_dialog_window_cancel_clicked(GtkWidget *widget, gpointer data) {

   p_dialog_window *dw = data;
   dw->dialog_accepted = FALSE;
   p_dialog_window_close(widget, dw);
   return;

}



static void p_dialog_window_apply_clicked(GtkWidget *widget, gpointer data) {

   p_dialog_window *dw = data;
   if(dw->apply != NULL) dw->apply(dw);
   return;

}



static void p_dialog_window_close_clicked(GtkWidget *widget, gpointer data) {

   p_dialog_window *dw = data;
   dw->dialog_accepted = FALSE;
   p_dialog_window_close(widget, dw);
   return;

}



void p_dialog_window_new(p_dialog_window *dw, pwindow_gtk *w, const char *title, const char *msgtext, int flags) {

   GtkWidget *vbox;
   GtkWidget *hbox;
   GtkWidget *msg;
   GtkWidget *btn;
   
   dw->dialog = gtk_window_new(GTK_WINDOW_DIALOG);
   if(title != NULL) gtk_window_set_title(GTK_WINDOW(dw->dialog), title);
   gtk_signal_connect(GTK_OBJECT(dw->dialog), "destroy", GTK_SIGNAL_FUNC(p_dialog_window_destroy), dw);

   vbox = gtk_vbox_new(FALSE, 5);
   hbox = gtk_hbox_new(FALSE, 5);
   dw->container = gtk_vbox_new(FALSE, 5);

   gtk_container_border_width(GTK_CONTAINER(dw->dialog), 10);
   gtk_container_add(GTK_CONTAINER(dw->dialog), vbox);

   gtk_box_pack_start(GTK_BOX(hbox), dw->container, FALSE, FALSE, 0);

   if(flags & P_DIALOG_PUYOS) {  
      msg = gtk_label_new(msgtext);
      gtk_label_set_line_wrap(GTK_LABEL(msg), TRUE);
      gtk_box_pack_start(GTK_BOX(dw->container), msg, TRUE, TRUE, 0);
      gtk_box_pack_start(GTK_BOX(hbox), p_dialog_puyos(w, msg), FALSE, FALSE, 0);
   }

   gtk_box_pack_start(GTK_BOX(vbox), hbox, TRUE, TRUE, 0);
   gtk_box_pack_start(GTK_BOX(vbox), gtk_hseparator_new(), FALSE, FALSE, 0);
   hbox = gtk_hbox_new(FALSE, 5);
   gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 0);

   if(flags & P_DIALOG_OK) {
      btn = gtk_button_new_with_label(" Ok ");
      gtk_signal_connect(GTK_OBJECT(btn), "clicked", GTK_SIGNAL_FUNC(p_dialog_window_ok_clicked), dw);
      gtk_box_pack_start(GTK_BOX(hbox), btn, FALSE, FALSE, 5);
   }
   if(flags & P_DIALOG_CANCEL) {
      btn = gtk_button_new_with_label(" Cancel ");
      gtk_signal_connect(GTK_OBJECT(btn), "clicked", GTK_SIGNAL_FUNC(p_dialog_window_cancel_clicked), dw);
      gtk_box_pack_start(GTK_BOX(hbox), btn, FALSE, FALSE, 5);
   }
   if(flags & P_DIALOG_APPLY) {
      btn = gtk_button_new_with_label(" Apply ");
      gtk_signal_connect(GTK_OBJECT(btn), "clicked", GTK_SIGNAL_FUNC(p_dialog_window_apply_clicked), dw);
      gtk_box_pack_start(GTK_BOX(hbox), btn, FALSE, FALSE, 5);
   }
   if(flags & P_DIALOG_CLOSE) {
      btn = gtk_button_new_with_label(" Close ");
      gtk_signal_connect(GTK_OBJECT(btn), "clicked", GTK_SIGNAL_FUNC(p_dialog_window_close_clicked), dw);
      gtk_box_pack_start(GTK_BOX(hbox), btn, FALSE, FALSE, 5);
   }

   dw->w = w;
   dw->dialog_modal = !(flags & P_DIALOG_NONMODAL);
   dw->dialog_accepted = FALSE;
   dw->waiting_for_close = TRUE;

   p_dialog_window_set_apply(dw, NULL);
   p_dialog_window_set_data(dw, NULL);
   
   return;

}



void p_dialog_window_set_apply(p_dialog_window *dw, p_dialog_window_handler apply) {

   dw->apply = apply;

}



void p_dialog_window_set_data(p_dialog_window *dw, gpointer data) {

   dw->data = data;

}



gint p_dialog_window_run(p_dialog_window *dw) {

   gtk_window_set_position(GTK_WINDOW(dw->dialog), GTK_WIN_POS_MOUSE);
   gtk_widget_show_all(dw->dialog);
   if(dw->dialog_modal) gtk_window_set_modal(GTK_WINDOW(dw->dialog), TRUE);

   while(dw->waiting_for_close) {
      p_window_idle((pwindow *)dw->w);
   }
   return(dw->dialog_accepted);

}



void p_dialog(pwindow_gtk *w, const char *t, const char *s) {

   p_dialog_window dw;

   p_dialog_window_new(&dw, w, t, s, P_DIALOG_MESSAGE);
   p_dialog_window_run(&dw);

}



void p_dialog_error(pwindow_gtk *w, const char *s) {

   char *err;
   char *buf;
   
   err = strerror(errno);
   if(err == NULL) {
      p_dialog(w, "Error", s);
      return;
   }
   
   buf = (char *)malloc(strlen(s) + strlen(err) + 4);
   if(buf == NULL) {
      p_dialog(w, "Error", s);
      return;
   }
   
   p_snprintf(buf, strlen(s) + strlen(err) + 4, "%s: %s", s, err);
   p_dialog(w, "Error", buf);
   free(buf);
   return;

}



static void p_dialog_query_changed(GtkWidget *widget, gpointer data) {

   pquery_gtk *q = data;

   strncpy(q->query_buffer, gtk_entry_get_text(GTK_ENTRY(q->query_entry)), q->query_size);
   return;

}



int p_dialog_query(pwindow_gtk *w, const char *t, const char *s, char *d, int size) {

   p_dialog_window dw;
   pquery_gtk q;

   p_dialog_window_new(&dw, w, t, s, P_DIALOG_QUERY);

   q.query_entry = gtk_entry_new_with_max_length(size - 1);
   q.query_buffer = d;
   q.query_size = size;

   gtk_entry_set_text(GTK_ENTRY(q.query_entry), d);
   gtk_entry_select_region(GTK_ENTRY(q.query_entry), 0, strlen(d));
   gtk_signal_connect(GTK_OBJECT(q.query_entry), "changed", GTK_SIGNAL_FUNC(p_dialog_query_changed), &q);
   gtk_box_pack_start(GTK_BOX(dw.container), q.query_entry, TRUE, TRUE, 0);

   return(p_dialog_window_run(&dw));

}



int p_dialog_query_value(pwindow_gtk *w, const char *t, const char *s, int *value) {

   char buf[P_GTK_STRING_BUFFER];
   char bufres[P_GTK_STRING_BUFFER];
   char *p;
   
   p_snprintf(buf, P_GTK_STRING_BUFFER, "%d", *value);
   if(!p_dialog_query(w, t, s, buf, P_GTK_STRING_BUFFER)) return(0);

   *value = strtol(buf, &p, 0);
   if(*p) {
      p_snprintf(bufres, P_GTK_STRING_BUFFER, "The number specified, \"%s\", is invalid (failed at character %d).", buf, p - buf);
      bufres[P_GTK_STRING_BUFFER - 1] = '\0';
      p_dialog_error(w, bufres);
      return(0);
   }

   return(1);
   
}



int p_dialog_query_double(pwindow_gtk *w, const char *t, const char *s, double *value) {

   char buf[P_GTK_STRING_BUFFER];
   char bufres[P_GTK_STRING_BUFFER];
   char *p;
   
   p_snprintf(buf, P_GTK_STRING_BUFFER, "%g", *value);
   if(!p_dialog_query(w, t, s, buf, P_GTK_STRING_BUFFER)) return(0);
   
   *value = strtod(buf, &p);
   if(*p) {
      p_snprintf(bufres, P_GTK_STRING_BUFFER, "The number specified, \"%s\", is invalid (failed at character %d).", buf, p - buf);
      p_dialog_error(w, bufres);
      return(0);
   }

   return(1);
   
}



void p_dialog_text(pwindow_gtk *w, const char *basename) {

   p_dialog_window dw;
   char buf[P_IO_BUFFER];
   char out[P_IO_BUFFER];
   char *pin;
   char *pout;
   int  width;
   int height;
   FILE *f;
   GdkFont *font;
   GdkFont *lfont;
   GdkFont *font_normal;
   GdkFont *font_italic;
   GdkFont *font_bold;
   GtkWidget *scroll;
   GtkWidget *message;
   
   p_dialog_window_new(&dw, w, basename, NULL, P_DIALOG_CLOSE | P_DIALOG_NONMODAL);

   p_snprintf(buf, P_IO_BUFFER, "%s/%s", P_GLOBAL_DIR, basename);
   if((f = fopen(buf, "r")) == NULL) {
      p_snprintf(buf, P_IO_BUFFER, "Cannot open file %s.\n", basename);
      p_dialog_error(w, buf);
      return;
   }
   
   /* Load the needed fonts */
   font_normal = gdk_font_load("-adobe-courier-medium-r-*-*-*-120-*");
   font_italic = gdk_font_load("-adobe-courier-medium-*-*-*-*-120-*");
   font_bold   = gdk_font_load("-adobe-courier-bold-r-*-*-*-120-*");
   if(font_normal == NULL) {
      width = 8;
      height = 16;
   } else {
      width = gdk_char_width(font_normal, 'W');
      height = (font_normal->ascent + font_normal->descent);
   }
   width  *= 82;
   height *= 33;

   scroll = gtk_scrolled_window_new(NULL, NULL);
   gtk_box_pack_start(GTK_BOX(dw.container), scroll, TRUE, TRUE, 0);
   gtk_widget_set_usize(scroll, width, height);

   message = gtk_text_new(NULL, NULL);
   gtk_text_set_editable(GTK_TEXT(message), FALSE);
   gtk_container_add(GTK_CONTAINER(scroll), message);

   /* Load in the text data */
   gtk_text_freeze(GTK_TEXT(message));
   while(fgets(buf, P_IO_BUFFER, f)) {
      pin   = buf;
      pout  = out;
      font  = NULL;
      lfont = font_normal;
      while(*pin != '\0') {
         if(*(pin + 1) != 0x08) {
            font = font_normal;
         } else {
            font = (*pin == '_' ? font_italic : font_bold);
            pin += 2;
         }
         if(font != lfont) {
            if(pout - out > 0) {
               gtk_text_insert(GTK_TEXT(message), lfont, NULL, NULL, out, pout - out);
            }
            pout  = out;
            lfont = font;
         }
         *(pout++) = *pin;
         if(*pin == '\n') {
            *(pout++) = ' ';
            *(pout++) = ' ';
         }
         pin++;
      }
      if(pout - out > 0) {
         gtk_text_insert(GTK_TEXT(message), lfont, NULL, NULL, out, pout - out);
      }
   }
   if(font_normal != NULL) gdk_font_unref(font_normal);
   if(font_italic != NULL) gdk_font_unref(font_italic);
   if(font_bold   != NULL) gdk_font_unref(font_bold);
   gtk_text_thaw(GTK_TEXT(message));
   fclose(f);

   p_dialog_window_run(&dw);
   return;

}
