/*

    xpuyopuyo - pconfig.c     Copyright(c) 1999,2000 Justin David Smith
    justins(at)chaos2.org     http://chaos2.org/
    
    Main game control code.
    

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/


/* System includes */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <getopt.h>
#include <errno.h>
#include <sys/stat.h>
#include <unistd.h>


/* Local includes */
#include <config.h>
#include <xpuyopuyo.h>
#include <pfile.h>
#include <pgame.h>
#include <pfield.h>
#include <pmanip.h>
#include <ppiece.h>
#include <pconfig.h>
#include <pconfigm.h>
#include <ptournament.h>
#include <pinfo.h>
#include <pnet.h>
#include <psnprintf.h>
#include <psound.h>



/***     Code to control the boolean toggle features     ***/



/* Toggle option configuration */
#define BOOL_LONG_BIT               0x8000      /* Used to toggle a long option toggle */
#define P_CONFIG_NO(x)              (BOOL_LONG_BIT | (x))   /* Negation of a long option */
#define BOOL_DEFAULT                0           /* Indicate toggle status for short options */
#define BOOL_INVERT                 2           /* The next short options should be inverted */


/* Options available that are long-only. */
#define P_CONFIG_WIDTH              0x102       /* Set width -- long option only */
#define P_CONFIG_HEIGHT             0x103       /* Set height -- long option only */
#define P_CONFIG_MATCH              0x104       /* Set number required for match */

#define P_CONFIG_NET_CLIENT         0x201       /* Set network client */
#define P_CONFIG_NET_SERVER         0x202       /* Set network server */
#define P_CONFIG_NET_PORT           0x203       /* Set communication port */
#define P_CONFIG_NET_AI             0x204       /* Allow AI in network game? */

#define P_CONFIG_THEME_NAME         0x400       /* Theme name */

#define P_CONFIG_INSANITY           0x600       /* Insanity */

#define P_CONFIG_HQMIXER            0x800       /* Use HQ mixer in sound? */


/* The full list of supported long options */
static const struct option longopt[] =
 { {  "help", 0, 0, 'h' },    /* Assistance? */
   {  "insanity", 0, 0, P_CONFIG_INSANITY },

   /* Long options for number of players */
   {  "oneplayer", 0, 0, '1' },
   {  "twoplayer", 0, 0, '2' },
   {  "aiplayer", 0, 0, 'a' },
   {  "demo", 0, 0, 'd'},

   /* Difficulty options */
   {  "easy", 0, 0, 'e' },
   {  "medium", 0, 0, 'm' },
   {  "hard", 0, 0, 'H' },

   /* General options */
   {  "number", 1, 0, 'n' },
   {  "restart", 0, 0, 't' },
   {  "norestart", 0, 0, P_CONFIG_NO('t') },
   {  "tutorial",  0, 0, 'T' },
   {  "notutorial",  0, 0, P_CONFIG_NO('T') },

   /* Sound options */
   {  "sound",  0, 0, 'S' },
   {  "nosound", 0, 0, P_CONFIG_NO('S') },
   {  "hqmixer", 0, 0, P_CONFIG_HQMIXER },
   {  "nohqmixer", 0, 0, P_CONFIG_NO(P_CONFIG_HQMIXER) },

   /* Advanced options */
   {  "width", 1, 0, P_CONFIG_WIDTH },
   {  "height", 1, 0, P_CONFIG_HEIGHT },
   {  "match", 1, 0, P_CONFIG_MATCH },
   {  "nullify", 0, 0, 'U' },
   {  "nonullify", 0, 0, P_CONFIG_NO('U') },
   {  "indestructibles", 0, 0, 'I' },
   {  "noindestructibles", 0, 0, P_CONFIG_NO('I') },

   /* Player speed options */ 
   {  "speed", 1, 0, 's' },
   {  "accel", 1, 0, 'A' },

   /* AI options */
   {  "airule1", 1, 0, 'R' },
   {  "airule2", 1, 0, 'r' },
   {  "aiscore", 0, 0, 'c' },
   {  "aiscore", 0, 0, P_CONFIG_NO('c') },
   {  "aidrop", 0, 0, 'i' },
   {  "noaidrop", 0, 0, P_CONFIG_NO('i') },

   /* Network options */
   {  "server", 0, 0, P_CONFIG_NET_SERVER },
   {  "client", 1, 0, P_CONFIG_NET_CLIENT },
   {  "port", 1, 0, P_CONFIG_NET_PORT },
   {  "networkai", 0, 0, P_CONFIG_NET_AI },
   {  "nonetworkai", 0, 0, P_CONFIG_NO(P_CONFIG_NET_AI) },

   /* Theme options */
   {  "theme", 1, 0, P_CONFIG_THEME_NAME },

   /* Tournament options */
   {  "tournament", 0, 0, 'z' },
   {  "hypertournament", 0, 0, 'Z' },

   /* Termination of list */
   {  0, 0, 0, 0 } 
};



static inline void boolclear(int *flag) {
/* boolclear

   Clear the boolean flag for the next short option(s).     */

   *flag = BOOL_DEFAULT;         /* Reset the flag to default (noninverted) */

}


static inline int isinvert(int *flag) {
/* isinvert

   Report if the short boolean flag is inverted.  After this call, the
   boolean toggle will be reset automatically since it should only affect
   the _next_ short option.  */

   int tmp;                      /* A temporary variable to store result */
   
   tmp = (*flag == BOOL_INVERT); /* Check if flag was inverted or not */
   boolclear(flag);              /* Clear the flag */
   return(tmp);                  /* Return whether flag was inverted */

}


static inline void shortinvert(int *flag) {
/* shortinvert

   Invert the short boolean toggle.  If it was already inverted, it will be
   reset to its default state.  */

   if(isinvert(flag))   *flag = BOOL_DEFAULT;      /* Un-invert */
   else                 *flag = BOOL_INVERT;       /* Invert it */

}


static inline int longinvert(int *flag, int code) {
/* longinvert

   Invert the long option given, by removing the BOOL_LONG_BIT from it. 
   This is needed to parse those --noxxx options as if they were short
   options all along.  Previous flag setting should not have any affect;
   this will always return an inverted toggle.  */

   *flag = BOOL_INVERT;             /* Invert the toggle flag */
   return(code & (~BOOL_LONG_BIT)); /* Return the code w/o LONG_BIT */

}



/***     Random Values     ***/



static inline void randomize(void) {
/* randomize */

   struct timeval tv;               /* Get a random time value */

   gettimeofday(&tv, NULL);         /* Get the current time of day */
   srand(tv.tv_usec);               /* Randomize by the microseconds field */

}



/***     Configuration file/directory     ***/



static void p_config_load(pconfig *c, int *ai1, int *ai2) {
/* p_config_load

   Load the configuration file into c.  ai1 and ai2 are markers for the
   default AI rules to be loaded, if they are specified in the configuration
   file.  */

   char buf[P_IO_BUFFER];        /* Temporary buffer to read into */
   char *p;                      /* Temporary character pointer */
   FILE *f;                      /* File input stream */
   
   /* Attempt to open the file */   
   if((f = p_file_open_read(P_CONFIG_FILE)) == NULL) return;

   /* Read the file, a line at a time */
   while(p_file_read_line(buf, P_IO_BUFFER, f) != NULL) {
      if((p = p_file_break(buf, '=')) != NULL) {
         if(!strcmp(buf, "numplayers"))            c->numplayers = CLAMP(atoi(p), 1, P_NUM_PLAYERS);
         else if(!strcmp(buf, "numai"))            c->ai = CLAMP(atoi(p), 1, P_NUM_PLAYERS);
         else if(!strcmp(buf, "numcolors"))        c->numcolors = CLAMP(atoi(p), P_COLORS_MIN, P_COLORS_MAX);
         else if(!strcmp(buf, "dropspeed"))        c->dropspeed = MAX(atoi(p), P_MINDROP_TIME);
         else if(!strcmp(buf, "accel"))            c->accel = CLAMP(atof(p), P_ACCEL_OFF, P_ACCEL_MAX);
         else if(!strcmp(buf, "nullifyrocks"))     c->nullifyrocks = atoi(p);
         else if(!strcmp(buf, "indestructibles"))  c->indestructibles = atoi(p);
         else if(!strcmp(buf, "fieldwidth"))       c->fieldwidth = CLAMP(atoi(p), P_FIELD_WIDTH_MIN, P_FIELD_WIDTH_MAX);
         else if(!strcmp(buf, "fieldheight"))      c->fieldheight = CLAMP(atoi(p), P_FIELD_HEIGHT_MIN, P_FIELD_HEIGHT_MAX);
         else if(!strcmp(buf, "fieldmatch"))       c->fieldmatch = CLAMP(atoi(p), P_MATCH_MIN, P_MATCH_MAX);
         else if(!strcmp(buf, "aiscore"))          c->aiscore = atoi(p);
         else if(!strcmp(buf, "aidrop"))           c->aicandrop = atoi(p);
         else if(!strcmp(buf, "restart"))          c->restart = atoi(p);
         else if(!strcmp(buf, "tutorial"))         c->tutorial = atoi(p);
         else if(!strcmp(buf, "quiet"))            c->quiet = atoi(p);
         #if USE_SOUND
            else if(!strcmp(buf, "enablesound"))      c->enablesound = atoi(p);
            else if(!strcmp(buf, "usehqmixer"))       c->usehqmixer = atoi(p);
         #endif /* SOUND? */
         else if(!strcmp(buf, "theme"))            p_strncpy(c->theme, p, P_THEME_NAME_SIZE);
         else if(!strcmp(buf, "ai1")) {
            if(*p == 'r' || *p == 'R') *ai1 = -1;
            else *ai1 = CLAMP(atoi(p), -1, P_AI_COUNT - 1);
         } else if(!strcmp(buf, "ai2")) {
            if(*p == 'r' || *p == 'R') *ai2 = -1;
            else *ai2 = CLAMP(atoi(p), -1, P_AI_COUNT - 1);
         }
      }  /* if a valid assignment line */
   }    /* while reading the file ... */

   /* Close and return */
   fclose(f);
   return;

}


void p_config_save(const pconfig *c) {
/* p_config_save

   Save the current configuration.  */

   FILE *f;                      /* Output file stream */

   /* Attempt to write the file */
   if((f = p_file_open_write(P_CONFIG_FILE)) == NULL) return;

   /* Write current values */
   fprintf(f, "# xpuyopuyo configuration file - automatically generated\n#\n");
   fprintf(f, "# WARNING:  There is no error checking on input values in this file\n");
   fprintf(f, "# Do not edit these directly unless you know what you are doing.  \n#\n");
   fprintf(f, "numplayers=%d\n",             c->numplayers);
   fprintf(f, "numai=%d\n",                  c->ai);
   fprintf(f, "numcolors=%d\n",              c->numcolors);
   fprintf(f, "dropspeed=%d\n",              c->dropspeed);
   fprintf(f, "accel=%g\n",                  c->accel);
   fprintf(f, "nullifyrocks=%d\n",           c->nullifyrocks);
   fprintf(f, "indestructibles=%d\n",        c->indestructibles);
   fprintf(f, "fieldwidth=%d\n",             c->fieldwidth);
   fprintf(f, "fieldheight=%d\n",            c->fieldheight);
   fprintf(f, "fieldmatch=%d\n",             c->fieldmatch);
   fprintf(f, "aiscore=%d\n",                c->aiscore);
   fprintf(f, "aidrop=%d\n",                 c->aicandrop);
   fprintf(f, "restart=%d\n",                c->restart);
   fprintf(f, "tutorial=%d\n",               c->tutorial);
   fprintf(f, "quiet=%d\n",                  c->quiet);
   fprintf(f, "theme=%s\n",                  c->theme);
   #if USE_SOUND
      fprintf(f, "enablesound=%d\n",            c->enablesound);
      fprintf(f, "usehqmixer=%d\n",             c->usehqmixer);
   #endif /* Sound? */
   fprintf(f, "ai1=%d\n",                    P_PLAYER_0(c)->airule);
   fprintf(f, "ai2=%d\n",                    P_PLAYER_1(c)->airule);

   /* Close and exit */
   fclose(f);
   return;

}



/***     Setup and parse a configuration     ***/



static void p_config_init(pconfig *c, int *airule1, int *airule2) {

   /* No window structure by default */
   c->window = NULL;

   /* Set default CONFIG options */   
   #if USE_NETWORK
      c->socket = NULL;
      c->networkai = false;
   #endif /* Network play? */
   c->player[0] = NULL;
   c->player[1] = NULL;
   c->paused = false;
   c->waiting = false;
   p_config_easy(c);
   p_config_players(c, P_CONFIG_HUMAN_HUMAN);
   p_config_size(c, P_FIELD_WIDTH_DEF, P_FIELD_HEIGHT_DEF, P_MATCH_DEF);
   c->restart = false;
   c->tutorial = false;
   #if P_ALLOW_TOURNAMENT
      c->tournament = false;
      c->hypertournament = false;
   #endif
   c->quiet = false;
   c->nodisplay = false;
   c->needredraw = true;
   
   #if USE_SOUND
      c->sound = NULL;
      c->enablesound = true;
      c->usehqmixer = false;
   #endif /* Sound setup? */

   /* Set default AI rules */
   *airule1 = P_AI_TRAINER;
   *airule2 = P_AI_DEFAULT;
   
   /* Set default theme name */
   p_strncpy(c->theme, P_DEFAULT_THEME, P_THEME_NAME_SIZE);

   return;

}



pconfig *p_config_new(int *argc, char ***argv) {
/* p_config_new */

   /* Options we support.  Yay!  */
   #if USE_NETWORK
   char *servername;    /* Network client only */
   int portnumber;      /* Network port number */
   int usenetwork;      /* 1 if using network */
   char *p;             /* Temporary pointer */
   #endif /* Network support? */
   
   pconfig *c;    /* The new configuration structure */
   int airule1;   /* Need temp variable since player structures */
   int airule2;   /* are not parsed until after getopt runs */
   int inv;       /* Inversion flag for toggles */
   int ch;

   /* Check for username, hostname */
   if(getenv("USER") == NULL) fprintf(stderr, "warning:  $USER is not set, I don't know who you are.\n");
   if(getenv("HOME") == NULL) fprintf(stderr, "warning:  $HOME is not set, I don't know where to save my files.\n");

   /* RANDOMIZE! */
   randomize();
   
   /* Initialise Network flags */
   #if USE_NETWORK
      servername = NULL;
      portnumber = P_NET_DEFAULT_PORT;
      usenetwork = false;
   #endif /* Network? */
   
   /* Allocate memory for configuration */
   if(!(c = (pconfig *)malloc(sizeof(pconfig)))) {
      perror("Cannot allocate pconfig");
      return(NULL);
   }
   p_config_init(c, &airule1, &airule2);
   
   /* Make configuration directory; load data files */
   p_make_config_dir();
   c->airules = p_ai_load_rules();
   p_config_load(c, &airule1, &airule2);
   p_high_score_load(c->high);

   /* Parse options */   
   while(EOF != (ch = getopt_long(*argc, *argv, "/h12ademHn:tTUIs:SA:R:r:cizZY", longopt, NULL))) {
      /* Check if a long inverted option was given */
      if(ch & BOOL_LONG_BIT) {
         /* Yes; set character to normal (uninverted) option and set invert flag */
         ch = longinvert(&inv, ch);
      }
      
      /* Check which option */
      switch(ch) {
      
      /* Other */
      case P_CONFIG_INSANITY:
         break;

      /* Boolean inversion */
      case '/':
         shortinvert(&inv);
         break;

      /* Number of players */
      case '1':
         boolclear(&inv);
         p_config_players(c, P_CONFIG_SINGLE_PLAYER);
         break;
      case '2':
         boolclear(&inv);
         p_config_players(c, P_CONFIG_HUMAN_HUMAN);
         break;
      case 'a':
         boolclear(&inv);
         p_config_players(c, P_CONFIG_HUMAN_AI);
         break;
      case 'd':
         boolclear(&inv);
         p_config_players(c, P_CONFIG_AI_AI);
         break;

      /* Difficulty */
      case 'e':
         boolclear(&inv);
         p_config_easy(c);
         break;
      case 'm':
         boolclear(&inv);
         p_config_medium(c);
         break;
      case 'H':
         boolclear(&inv);
         p_config_hard(c);
         break;

      /* Other General options */
      case 'n':
         boolclear(&inv);
         c->numcolors = CLAMP(atoi(optarg), P_COLORS_MIN, P_COLORS_MAX);
         break;
      case 't':
         c->restart = !isinvert(&inv);
         break;
      case 'T':
         c->tutorial = !isinvert(&inv);
         break;

      /* Sound options */
      #if USE_SOUND
         case 'S':
            c->enablesound = !isinvert(&inv);
            break;
         case P_CONFIG_HQMIXER:
            c->usehqmixer = !isinvert(&inv);
            break;
      #endif /* Sound? */

      /* Advanced options */
      case P_CONFIG_WIDTH:
         boolclear(&inv);
         c->fieldwidth = CLAMP(atoi(optarg), P_FIELD_WIDTH_MIN, P_FIELD_WIDTH_MAX);
         break;
      case P_CONFIG_HEIGHT:
         boolclear(&inv);
         c->fieldheight = CLAMP(atoi(optarg), P_FIELD_HEIGHT_MIN, P_FIELD_HEIGHT_MAX);
         break;
      case P_CONFIG_MATCH:
         boolclear(&inv);
         c->fieldmatch = CLAMP(atoi(optarg), P_MATCH_MIN, P_MATCH_MAX);
         break;
      case 'U':
         c->nullifyrocks = !isinvert(&inv);
         break;
      case 'I':
         c->indestructibles = !isinvert(&inv);
         break;

      /* Player options */
      case 's':
         boolclear(&inv);
         c->dropspeed = MAX(atoi(optarg), P_MINDROP_TIME);
         break;
      case 'A':
         boolclear(&inv);
         c->accel = CLAMP(atof(optarg), P_ACCEL_OFF, P_ACCEL_MAX);
         break;

      /* AI options */
      case 'r':
         boolclear(&inv);
         if(*optarg == 'r' || *optarg == 'R') {
            airule2 = -1;
         } else {
            airule2 = CLAMP(atoi(optarg) - 1, 0, P_AI_COUNT - 1);
         }
         break;
      case 'R':
         boolclear(&inv);
         if(*optarg == 'r' || *optarg == 'R') {
            airule1 = -1;
         } else {
            airule1 = CLAMP(atoi(optarg) - 1, 0, P_AI_COUNT - 1);
         }
         break;
      case 'c':
         c->aiscore = !isinvert(&inv);
         break;
      case 'i':
         c->aicandrop = !isinvert(&inv);
         break;

      /* Theme support */
      case P_CONFIG_THEME_NAME:
         p_strncpy(c->theme, optarg, P_THEME_NAME_SIZE);
         break;

      /* Network flags */
      #if USE_NETWORK

         case P_CONFIG_NET_PORT: 
            /* Get the port number */
            portnumber = strtol(optarg, &p, 0);
            if(*p != '\0') {
               printf("** Numerical value for port is invalid\n");
               p_susage(*argv[0]);
               return(NULL);
            }
            break;
         case P_CONFIG_NET_SERVER:
            /* Setup as a server */
            usenetwork = true;
            free(servername);
            servername = NULL;
            break;
         case P_CONFIG_NET_CLIENT:
            /* Setup as a client */
            usenetwork = true;
            free(servername);
            /* Allocate the servername */
            servername = (char *)malloc(strlen(optarg) + 1);
            if(servername == NULL) {
               perror("Allocation error reading server name");
               return(NULL);
            }
            /* Copy the servername */
            p_strncpy(servername, optarg, strlen(optarg) + 1);
            break;
         case P_CONFIG_NET_AI:
            /* Allow AI in a network game */
            c->networkai = !isinvert(&inv);
            break;

      #else /* No network support allowed */

         case P_CONFIG_NET_PORT:
         case P_CONFIG_NET_SERVER:
         case P_CONFIG_NET_CLIENT:
         case P_CONFIG_NET_AI:
            printf("** Network support cannot be enabled\n");
            p_susage(*argv[0]);
            return(NULL);
            break;

      #endif /* Network support? */

      /* Tournament flags */
      #if P_ALLOW_TOURNAMENT

         case 'z':
            boolclear(&inv);
            c->tournament = P_AI_COUNT;
            p_config_players(c, P_CONFIG_AI_AI);
         case 'Z':
            boolclear(&inv);
            c->tournament = P_AI_COUNT;
            p_config_players(c, P_CONFIG_AI_AI);
            c->hypertournament = true;
            c->restart = true;
            c->dropspeed = P_MINDROP_TIME;
            break;
         case 'Y':
            boolclear(&inv);
            if(P_TOURNAMENT(c)) {
               c->nodisplay = true;
            } else {
               printf("** This option makes no sense unless in tournament\n");
               p_susage(*argv[0]);
               return(NULL);
            }
            break;

      #else /* No tournaments allowed */

         case 'z':
            printf("** Tournament mode cannot be enabled\n");
            p_susage(*argv[0]);
            return(NULL);
            break;

      #endif /* Allow tournament mode? */
         
      /* User help */
      case '?':
      case ':':
         p_susage(*argv[0]);
         p_config_free(&c);
         return(NULL);
         break;
      default:
         p_usage(*argv[0]);
         p_config_free(&c);
         return(NULL);
         break;

      } /* End switch statement */
   } /* End loop to grab options */

   /* Check for incompatible options */
   if(p_config_incompat(c)) {
      printf("** Incompatible options were given\n");
      p_usage(*argv[0]);
      p_config_free(&c);
      return(NULL);
   }

   /* Tournament mode? */
   #if P_ALLOW_TOURNAMENT
      if(P_TOURNAMENT(c)) {
         p_game_tournament_init(c);
         p_game_tournament_randomize_current(c);
         p_game_tournament_clear_next(c);
      }
   #endif

   /* Setup player 0 (player 1 to the user) */
   if((c->player[0] = p_player_new(0, c->numcolors, c->fieldwidth, c->fieldheight, c->numcolors)) == NULL) {
      p_config_free(&c);
      return(NULL);
   }

   /* Setup player 1 (player 2...) */
   if((c->player[1] = p_player_new(1, c->numcolors, c->fieldwidth, c->fieldheight, c->numcolors)) == NULL) {
      p_config_free(&c);
      return(NULL);
   }

   /* Setup AI status */
   P_PLAYER_0(c)->airule = airule1;
   P_PLAYER_1(c)->airule = airule2;
   P_PLAYER_0(c)->ai = p_ai_get_rule(c->airules, airule1);
   P_PLAYER_1(c)->ai = p_ai_get_rule(c->airules, airule2);

   /* Write out current configuration */
   p_config_save(c);
   
   /* Network game? */
   #if USE_NETWORK
      if(usenetwork) {
         /* Client or server? */
         if(servername == NULL) {
            /* Server */
            c->socket = pnet_new_server(c, portnumber);
            if(c->socket == NULL) {
               printf("** Cannot start server: %s\n", pnet_get_error());
               return(NULL);
            } 
         } else {
            /* Client */
            c->socket = pnet_new_client(c, servername, portnumber);
            if(c->socket == NULL) {
               printf("** Cannot start client: %s\n", pnet_get_error());
               return(NULL);
            }
         } /* New client or server? */
      } /* Allow network mode? */
      
      /* We might need to free the servername string */
      if(servername != NULL) free(servername);
   #endif /* Network enabled? */
   
   /* Setup sound service? */
   #if USE_SOUND
      p_sound_init();
      c->sound = p_sound_new(c->enablesound, c->usehqmixer);
   #endif /* Sound allowed? */
   
   /* Initialise game */
   p_game_end_(c);

   /* Return the new configuration structure */
   return(c);

}



void p_config_initialise(pconfig *c) {

   #if USE_SOUND
      p_sound_start(c->sound, P_MUSIC_PRELUDE);
   #endif /* Sound? */

}



void p_config_free(pconfig **c) {
/* p_config_free */

   if(!c || !*c) return;
   p_player_free(&(*c)->player[0]);
   p_player_free(&(*c)->player[1]);
   p_ai_release_rules(&(*c)->airules);
   #if USE_SOUND
      p_sound_free(&(*c)->sound);
   #endif /* Shutdown sound? */
   free(*c);
   *c = NULL;

}


