/*
 * Copyright 1995 The University of Newcastle upon Tyne
 * 
 * Permission to use, copy, modify and distribute this software and its
 * documentation for any purpose other than its commercial exploitation
 * is hereby granted without fee, provided that the above copyright
 * notice appear in all copies and that both that copyright notice and
 * this permission notice appear in supporting documentation, and that
 * the name of The University of Newcastle upon Tyne not be used in
 * advertising or publicity pertaining to distribution of the software
 * without specific, written prior permission. The University of
 * Newcastle upon Tyne makes no representations about the suitability of
 * this software for any purpose. It is provided "as is" without express
 * or implied warranty.
 * 
 * THE UNIVERSITY OF NEWCASTLE UPON TYNE DISCLAIMS ALL WARRANTIES WITH
 * REGARD TO THIS SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL THE UNIVERSITY OF
 * NEWCASTLE UPON TYNE BE LIABLE FOR ANY SPECIAL, INDIRECT OR
 * CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF
 * USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR
 * OTHER TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR
 * PERFORMANCE OF THIS SOFTWARE.
 * 
 * Author:  Gerry Tomlinson (gerry.tomlinson@newcastle.ac.uk)
 *          Computing Laboratory, University of Newcastle upon Tyne, UK
 */

/*
 * xp - print a file in an X window 
 *
 */

#include <stdio.h>
#include <errno.h>
#include <signal.h>
#include <sys/wait.h>
#include <sys/ioctl.h>
#if defined (SVR4) || defined (SYSV)
#include <sgtty.h>
#endif
#include <ctype.h>
#include <X11/Xos.h>
#include <X11/Xatom.h>
#include <X11/cursorfont.h>
#include <X11/IntrinsicP.h>
#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>
#include <X11/Shell.h>
#include <X11/Xaw/Command.h>
#include <X11/Xaw/AsciiText.h>
#include <X11/Xaw/Label.h>
#include <X11/Xaw/Form.h>
#include <X11/Xaw/Toggle.h>
#include <X11/Xaw/Paned.h>
#include <X11/Xaw/MenuButton.h>
#include <X11/Xaw/SimpleMenu.h>
#include <X11/Xaw/Sme.h>
#include <X11/Xaw/SmeBSB.h>
#include <X11/Xaw/SmeLine.h>
#include <X11/Xmu/Misc.h>

#include "version.h"

#include "mfmarker.bm"
#include "sfmarker.bm"
#include "tick.bm"

#include "mysystem.h"
#include "text.h"

	

/*
 * xp main program 
 *
 */


int Argc;
char **Argv;

Display *display;
XtAppContext app;

Widget cmndtarget;

static Widget toplevel,top, paned, hpaned, textmain, textsplit, vpaned ; 
static Widget quit, files, split, flip, cmnds, textinput;
static Widget helpshell = NULL;
static Widget errorshell = NULL;
static Widget help,helppaned, helplabel,helptext,helpdismiss;
static Widget filesmenu,optionsmenu,cmndsmenu;
static Widget swrap, searchmany, casesens, shortname, quiet, hscroll, autoexec;



static Pixmap mfmarker,sfmarker,tick;
static Cursor busycursor, textcursor, buttoncursor;


#include "appres.h"

AppResources app_resources;

/* there follows the user specified application resources, note this
 *  includes geometry so we can handle WM size hints correctly
 */

static XtResource resources[] = {
        {"geometry","Geometry",XtRString, sizeof(String),
	XtOffset(AppResources *, geometry), XtRImmediate, (XtPointer) NULL
	},
        {"display","Display",XtRString, sizeof(String),
	XtOffset(AppResources *, display), XtRImmediate, (XtPointer) NULL
	},
        {"searchWrap", "SearchWrap", XtRBoolean, sizeof (Boolean),
         XtOffset(AppResources *, searchWrap), XtRImmediate, (XtPointer) TRUE
	},
	{"searchMany", "SearchMany", XtRBoolean, sizeof (Boolean),
         XtOffset(AppResources *, searchMany), XtRImmediate, (XtPointer) FALSE
        },
	{"setIconName", "SetIconName", XtRBoolean, sizeof (Boolean),
         XtOffset(AppResources *, setIconName), XtRImmediate, (XtPointer) TRUE
        },
	{"setTitle", "SetTitle", XtRBoolean, sizeof (Boolean),
         XtOffset(AppResources *, setTitle), XtRImmediate, (XtPointer) TRUE
        },
	{"shortName", "ShortName", XtRBoolean, sizeof (Boolean),
         XtOffset(AppResources *, shortName), XtRImmediate, (XtPointer) TRUE
        },
 	{"caseSensitive", "CaseSensitive", XtRBoolean, sizeof (Boolean),
         XtOffset(AppResources *, caseSensitive), XtRImmediate, (XtPointer) TRUE
        },
	{"command", "Command", XtRString, sizeof(String), 
	 XtOffset(AppResources *, command), (XtPointer) NULL
	},
	{"cmndsList", "CmndsList", XtRString, sizeof(String), 
	 XtOffset(AppResources *, cmndsList), (XtPointer) NULL
	},
	{"quiet", "Quiet", XtRBoolean, sizeof (Boolean),
         XtOffset(AppResources *, quiet), XtRImmediate, (XtPointer) TRUE
        },
	{"terse", "Terse", XtRBoolean, sizeof (Boolean),
         XtOffset(AppResources *, terse), XtRImmediate, (XtPointer) FALSE
        },
	{"autoExec", "AutoExec", XtRBoolean, sizeof (Boolean),
         XtOffset(AppResources *, autoExec), XtRImmediate, (XtPointer) FALSE
        },
        {"samDoubleClick", "SamDoubleClick", XtRBoolean, sizeof (Boolean),
         XtOffset(AppResources *, samDoubleClick), XtRImmediate, (XtPointer) FALSE
        },
        {"useMyShell", "UseMyShell", XtRBoolean, sizeof (Boolean),
         XtOffset(AppResources *, useMyShell), XtRImmediate, (XtPointer) FALSE
        },
	{"version", "Version", XtRString, sizeof (String),
         XtOffset(AppResources *, version),  (XtPointer) NULL
        },
};

static XrmOptionDescRec optiondesc[] = {
{"-m", "searchMany", XrmoptionNoArg, (XtPointer)"on"},
{"+m", "searchMany", XrmoptionNoArg, (XtPointer)"off"},
{"-c",  "command", XrmoptionSepArg, (XtPointer) NULL},
{"-i", "caseSensitive", XrmoptionNoArg, (XtPointer)"off"},
{"+i", "caseSensitive", XrmoptionNoArg, (XtPointer)"on"},
{"-w", "searchWrap", XrmoptionNoArg, (XtPointer)"off"},
{"+w", "searchWrap", XrmoptionNoArg, (XtPointer)"on"}
};

static char translations[] = "\
			<Btn1Up>(2):    DoubleClick()";

/*
 * define default row & column values for case when default geometry is
 *  overridden but doesn't include both height & width
 */
Dimension rows = 24;
Dimension columns = 80;

typedef struct _Cmndsmenuentry {
   char* name; char * value;}
Cmndsmenuentry ;

Cmndsmenuentry* cmndsmenuentries;

/* callback  & action routine declarations */

static void Quit(), Help(), Dismiss(),  Next(), Previous(),  Split(), Flip(),
WrapMenuSelect(), FilesMenuSelect(), CmndsMenuSelect(), Cmnd(), KeyRedirect(),
Searchwrap(), SearchMany(), CaseSens(), ShortName(), Quiet(), AutoExec(), Hscroll(),
AdjustGrip(), ErrorDismiss(), DoubleClick(), SetCmndTarget();


XtActionsRec actionTable[] =
{
	"Previous", Previous,
	"Next", Next,
	"Cmnd", Cmnd,
        "KeyRedirect", KeyRedirect,
	"AdjustGrip", AdjustGrip,
	"DoubleClick", DoubleClick,
	"SetCmndTarget", SetCmndTarget,
};


static struct {Widget w; char * name; XawTextWrapMode mode;} wrapnames [] = {
	 NULL,"wrap word", XawtextWrapWord, 
	 NULL, "wrap line", XawtextWrapLine,
};


/*
 * here we define some globals which maintain state of file display text widgets
 * note we use file numbers starting from 1 to indicate which file
 * we're interested in since Argv[1] is the first filename,
 * but the widget list of menu items is indexed from 0.
 *
 * So we define the macro file_entry which takes a file number
 * and returns the corresponding menu widget item
 *
 */

#define file_name(file) (Argv[file] ? Argv[file] : "")
#define file_name_display(file) !app_resources.shortName ? file_name(file) : strippath(file_name(file)) 

static XtOrientation orientation; /* orientation of pane containing the text widgets */

static Widget * file_entry_list;     	/* list of file menu items which
			 	           contain the file names */

#define file_entry(fileno) file_entry_list[fileno -1]

/*
* have a textextra struct for each file display text widget -
 * include things which can't be got from the text widget itself 
 */

typedef struct {
    Widget widget;
    int file;		/* file number */
    char *data;		/* can't get this from widget in X11R4 */
    int size;
} textextra;


textextra textmainextra = {NULL,0,NULL,0};
textextra textsplitextra = {NULL,0,NULL,0};

/*
 * return a textextra given a window
 *
 */
textextra*
wtote(w) 
Widget w;
{
   return textmain == w  ? &textmainextra
		         : &textsplitextra;
}


/*
 * now some error routines
 *
 */
void
usage()
{
   fprintf(stderr,"usage: xp [-c command] [-i] [-m] [-w] file .....\n");
   exit(1);
} 

/*ARGSUSED*/
static void
ErrorDismiss(widget,w,client_data)
Widget w,widget;
caddr_t client_data;
{
    XtPopdown(errorshell);
}


/*
 * xperror
 *
 */
void
xperror(message)
char * message;
{
    static Widget errorform,errorlabel,errordismiss;
    Dimension width, height, b_width;
    Position x, y, inside_y, max_x, max_y;
    Window root_return, child_return;
    int root_x_return, root_y_return;
    int win_x_return, win_y_return;
    unsigned int mask_return;

    if (!app_resources.quiet)
	XBell(display,0);

    if (app_resources.terse)
	return;

    if (errorshell == NULL){
	errorshell = XtVaCreatePopupShell("errorshell",
                                transientShellWidgetClass,toplevel,NULL);

	errorform = XtVaCreateManagedWidget("errorform",
				formWidgetClass,errorshell,
				NULL);

        errorlabel = XtVaCreateManagedWidget("errorlabel",
				labelWidgetClass, errorform,
				NULL);
        errordismiss = XtVaCreateManagedWidget("errordismiss",
				commandWidgetClass, errorform,
				NULL);
        XtAddCallback (errordismiss,XtNcallback, ErrorDismiss, NULL);
	XtInstallAccelerators(errorform,errordismiss);
    }

    XtVaSetValues(errorlabel,XtNlabel,message,NULL);

    XQueryPointer(display,XtWindow(top),&root_return,&child_return,
			 &root_x_return, &root_y_return,
			 &win_x_return, &win_y_return, &mask_return);

    x = (Position) root_x_return;
    y = (Position) root_y_return;

    XtVaGetValues(errordismiss,
		  XtNwidth, &width,
		  XtNheight, &height,
		  XtNy, &inside_y,
		  XtNborderWidth, &b_width,
		  NULL);

    width += 2 * b_width;
    inside_y+= 2 * b_width;

    x -= ( (Position) width/2 );
    if (x < 0) x = 0;

   if (x > (max_x = (Position) (XtScreen(errorshell)->width - width))) x = max_x;
    
    y -=  (Position) (height/2 + inside_y );
    if (y < 0) y = 0;
    if (y > (max_y = (Position) (XtScreen(errorshell)->height - height))) y = max_y;

    XtVaSetValues(errorshell,
		  XtNx, x,
		  XtNy, y,
		  NULL);


    XtPopup(errorshell,XtGrabExclusive);

}

/*
 * strippath
 *
 * given a pathname p strips off everything up to & including last '/'
 */
char *
strippath(p)
char *p;
{
    char* f;

    f = strrchr(p,'/');
    return f ? f+1 : p;
}


/*
 * maketitle
 *
 * set title & iconname properties on shell
 *
 * set them to filename + error indication if success is false
 *
 */

char *
maketitle(filename, mess)
    char *filename;
    char *mess;
{
   char iconname[200];
   char title[200];

   if (Argc > 1) {
        iconname[0] =  title[0] = '\0';
        strcat(iconname,app_resources.iconName);
        strcat(title, app_resources.title);
	strcat(iconname,": ");
        strcat(title,": ");
        strcat(iconname,filename);
	strcat(title,filename);
	if (mess) {
            strcat(iconname,": ");
            strcat(iconname, mess);
	    strcat(title,": ");
	    strcat(title,mess);
	}


   if (app_resources.setIconName) 
       XtVaSetValues(top,XtNiconName,iconname, NULL);
   if (app_resources.setTitle)
       XtVaSetValues(top,XtNtitle,title, NULL);
  }

}

/*
 * newfile
 *
 * reads newfile if data null & returns no. of bytes in it (else use size)
 * puts data into textmain
 * and update textmainextra with data & size
 *
 * also sets title
 *
 */

int
newfile(file,data, size)
    int file;
    char *data;
    int size;

{
    extern int errno;
    extern char* uerror();
    extern char* readfile();
    char *mess = NULL;
    Boolean reopening = FALSE;
    XawTextPosition curpos;

    if (textmainextra.file == file) {
	reopening = TRUE;
        curpos = XawTextGetInsertionPoint(textmain);
    } 

    if (!XtIsManaged(textsplit) || textmainextra.data != textsplitextra.data)	
	XtFree(textmainextra.data);		

    textmainextra.data = data ? data : readfile(file_name(file),&size);

    XtVaSetValues(textmain, XtNstring,textmainextra.data?textmainextra.data:"",
				XtNlength, size?size:0 , NULL);

    if (reopening)
	setinsertionpoint(textmain,curpos);

    if (size <0) {
        switch (size) {
 	case -2: mess = "malloc error"; break;
	default: mess = uerror();
        }
	if (!app_resources.quiet)
	    XBell(display,0);
    }
    maketitle(file_name_display(file),mess);

    if (XtIsSensitive(files)) {
      if (textmainextra.file)
          XtVaSetValues(file_entry(textmainextra.file),XtNleftBitmap,None,NULL);
      XtVaSetValues(file_entry(file),XtNleftBitmap,mfmarker,NULL);
      if (XtIsManaged(textsplit) && file != textsplitextra.file)
	XtVaSetValues(file_entry(textsplitextra.file), XtNleftBitmap,sfmarker,NULL);
    }

    textmainextra.file = file;
    textmainextra.size = size;

    return size;

}

String
GeometryPos(geometrySpec)
     String geometrySpec;
{
    int parse, x, y, width, height;
    String offsets, plus, minus;

    offsets = NULL;
    if (geometrySpec) {
        parse = XParseGeometry(geometrySpec, &x, &y, &width, &height);
        if (parse & (XValue | YValue))
        {
            plus = strchr(geometrySpec, '+');
            minus = strchr(geometrySpec, '-');
	    if (plus && minus)
	        offsets = (plus < minus) ? plus : minus;
            else
	        offsets = plus ? plus : minus;
        }
    }
    return offsets;
}


/*
 * getsizeof
 * gives the size in pixels given  rows & columns of  widget
 * assuming fixed pith font
 * 
 */
void
getsizeof (columns, rows, widget, width, height)
     Dimension columns, rows, *width, *height;
     Widget widget;
{
    XFontStruct *font;
    Dimension  lm, rm, tm, bm;
    XRectangle caret;
    Widget sink;
    
    XtVaGetValues(widget,
		  XtNfont, &font,
		  XtNleftMargin, &lm,
		  XtNrightMargin, &rm,
		  XtNtopMargin, &tm,
		  XtNbottomMargin, &bm,
		  XtNtextSink, &sink,
                 NULL);

    XawTextSinkGetCursorBounds(sink, &caret);

    *width = columns * font->max_bounds.width + lm + rm;
    *height = rows * (font->max_bounds.ascent +
#if XT_REVISION < 5
		      font->max_bounds.descent) + tm + bm + caret.height;
#else
		      font->max_bounds.descent) + tm + bm;
#endif
}


void 
setwmhints(shell,text)
     Widget shell,text;
{
    Widget sink;
    XSizeHints hints;
    long supp;
    XFontStruct *font;
    Dimension lm, rm, tm, bm, height, pheight,ibw;
    Atom wm_delete_window;
    XRectangle caret;
    XawTextScrollMode scrollmode;

    XtVaGetValues(vpaned, 
		XtNheight, &height, 
 		XtNinternalBorderWidth,&ibw,
		NULL);

    XtVaGetValues(text,
		  XtNfont, &font,
		  XtNleftMargin, &lm,
		  XtNrightMargin, &rm,
		  XtNtopMargin, &tm,
		  XtNbottomMargin, &bm,
		  XtNtextSink, &sink,
		  XtNscrollHorizontal,&scrollmode,
		  NULL);

    XawTextSinkGetCursorBounds(sink, &caret);

    XtVaGetValues(shell,
		  XtNheight, &pheight,
		  NULL);

    if (!XGetWMNormalHints(display, XtWindow(shell), &hints, &supp))
    {
	hints.flags = 0;
    }
    hints.flags |= PResizeInc | PBaseSize;
    if (app_resources.geometry) 
    {
	hints.flags |= USPosition | USSize;
    }
    hints.width_inc = font->max_bounds.width;
    hints.height_inc = font->max_bounds.ascent + font->max_bounds.descent;
    hints.base_width = lm + rm;
#if XT_REVISION < 5
    hints.base_height = pheight - (height - tm - bm - (Dimension) caret.height);
    if (XtIsManaged(textsplit))
	hints.base_height = hints.base_height +tm +bm +ibw + (Dimension) caret.height;
#else
    hints.base_height = pheight - (height - tm - bm); 
    
    if (XtIsManaged(textsplit)) 
	hints.base_height = hints.base_height +tm +bm +ibw;
#endif

/*
    if (scrollmode == XawtextScrollAlways)
        hints.base_height = hints.base_height + hints.height_inc;

    if (XtIsManaged(textsplit)) {
	XtVaGetValues(textsplit,
		XtNscrollHorizontal,&scrollmode,
		  NULL);
        if (scrollmode == XawtextScrollAlways)
            hints.base_height = hints.base_height + hints.height_inc;
    }
*/

    XSetWMNormalHints(display, XtWindow(shell), &hints);

    wm_delete_window = XInternAtom (display, "WM_DELETE_WINDOW", False);
    (void) XSetWMProtocols(display, XtWindow(shell), &wm_delete_window, 1);

}




main (argc, argv)
    int argc;
    char **argv;
{
    extern char* readfile();
    extern char* uerror();
    Boolean use_stdin;
    int size = 0;
    char *filedata = NULL;
    int parse, x, y, width, height;
    
    void makewidgets();


    toplevel = XtVaAppInitialize (&app,"Xp",
                        optiondesc,XtNumber(optiondesc),
                         (Cardinal *) &argc, argv, 
			NULL, NULL);


    if (argc > 1 && (argv[1][0] == '-' && argv[1][1]))
        usage();

    filedata = readfile(argc > 1 ? argv[1] : "-", &size);

    if (argc == 2 && !filedata) {
	extern int errno;

	fprintf(stderr,"%s: %s: %s\n", argv[0], argv[1], uerror());
	exit (1);
    }

    if (argc <= 2 && size == 0) exit(0);


    display = XtDisplay(toplevel);

    ioctl(ConnectionNumber(display), FIOCLEX, NULL);  /* for system/popen etc */

    XtVaGetApplicationResources(toplevel, (XtPointer) &app_resources, resources,
                                XtNumber(resources), NULL);

    if (!app_resources.version || !strstr(VERSION,app_resources.version)) {
 	fprintf(stderr,"xp: cannot find application defaults file v%s\n",VERSION);
	exit (1);
    }


    XtVaGetValues(toplevel,
                        XtNiconic, &app_resources.iconic,
			XtNiconName, &app_resources.iconName,
                        XtNtitle, &app_resources.title,
                        NULL);

    parse = XParseGeometry(app_resources.geometry, &x, &y, &width, &height);

    if (app_resources.geometry) {
    if (parse & WidthValue)
	columns = width;
    if (parse & HeightValue) { 
	char *f = filedata;
	if (argc == 1) {	/* if no args then chop to size if small */
	        rows =  0;	
	    	while (*f && rows < height) 
			if (*f++ == '\n') rows++;
	        if ((*(f-1)) != '\n') rows++; /* for incomplete last line */           } else rows = height;
    }

    if (app_resources.geometry)
        app_resources.geometry = GeometryPos(app_resources.geometry);
    }

    top = XtVaCreatePopupShell("shell", topLevelShellWidgetClass, toplevel,
				XtNiconic, app_resources.iconic,
				XtNiconName, app_resources.iconName,
				XtNtitle, app_resources.title,
				XtNallowShellResize, TRUE,
				NULL);

    if (app_resources.geometry)
	XtVaSetValues(top,XtNgeometry,app_resources.geometry,NULL);


    mfmarker = XCreateBitmapFromData(display,
		RootWindow(display, DefaultScreen(display)),
		mfmarker_bits, mfmarker_width, mfmarker_height);

    sfmarker = XCreateBitmapFromData(display,
		RootWindow(display, DefaultScreen(display)),
		sfmarker_bits, sfmarker_width, sfmarker_height);

    tick = 	XCreateBitmapFromData(display,
		RootWindow(display, DefaultScreen(display)),
		tick_bits, tick_width, tick_height);

    Argc = argc;
    Argv = argv;

    use_stdin = (argc == 1);

    makewidgets(top,  !use_stdin);

    busycursor  = XCreateFontCursor(XtDisplay(top), XC_watch);
    XtVaGetValues(textmain, XtNcursor, &textcursor,NULL);
    XtVaGetValues(files, XtNcursor, &buttoncursor,NULL);
 
    (void) newfile(1, filedata, size);

    {
	Dimension textwidth,textheight;
	
	getsizeof(columns,rows,textmain,&textwidth,&textheight);
	
	XtVaSetValues(textmain,
	            XtNheight, textheight,
	            XtNwidth, textwidth,
	            NULL);

        XtVaSetValues(paned,
		    XtNwidth, textwidth,
		    NULL);
    }

    XtAppAddActions(app, actionTable, XtNumber(actionTable));

    if (app_resources.samDoubleClick) {
        XtOverrideTranslations(textmain, XtParseTranslationTable(translations));
	XtOverrideTranslations(textsplit, XtParseTranslationTable(translations));

    }

    XtRealizeWidget(top);
    setwmhints(top,textmain);

    XtPopup(top,XtGrabNone);

    cmndtarget = textmain;

    redirect();

    if (app_resources.command) 
	Cmnd(textinput, (XEvent *)NULL, (String *)NULL, 0);
 

	
    XtAppMainLoop (app);

}


void
makewidgets (parent,need_files)
    Widget parent;                      /* into whom widget should be placed */
    Boolean need_files;
{
    int i;
    Widget options, entry;
    XawTextWrapMode wrapmode;
    XawTextScrollMode scrollmode;
    
    void makecmndsmenu();

    paned = XtVaCreateManagedWidget ("paned", panedWidgetClass, parent, 
				NULL);

    vpaned =  XtVaCreateManagedWidget ("vpaned", panedWidgetClass, paned,
				NULL);

    XtVaGetValues(vpaned,XtNorientation,&orientation,NULL);


    textmain = XtVaCreateManagedWidget ("textmain", asciiTextWidgetClass, vpaned,
                                  XtNuseStringInPlace, TRUE, XtNstring, "",
				  XtNscrollVertical, XawtextScrollAlways,
                                NULL );

    textmainextra.widget = textmain;

    textsplit = XtVaCreateWidget ("textsplit", asciiTextWidgetClass, vpaned,
				  XtNuseStringInPlace,TRUE, XtNstring, "",
				  XtNscrollVertical, XawtextScrollAlways,
				NULL );

    textsplitextra.widget = textsplit;

    hpaned = XtVaCreateManagedWidget ("hpaned", panedWidgetClass,paned,
				XtNorientation, XtorientHorizontal,
				XtNskipAdjust,TRUE,
                                        NULL );

    quit = XtVaCreateManagedWidget("quit", commandWidgetClass, hpaned,
					NULL );

    files = XtVaCreateManagedWidget("files", menuButtonWidgetClass, hpaned,
                                        XtNmenuName,"filesmenu",
					XtNsensitive, need_files,
                                        NULL);


    if (need_files) {
        filesmenu = XtVaCreatePopupShell("filesmenu",  simpleMenuWidgetClass,files,
					    NULL);
    
	for (i = 1; i < Argc; i++) {
	    char * item = Argv[i];
    
	    entry = XtVaCreateManagedWidget(file_name_display(i), smeBSBObjectClass,filesmenu,
				    XtNleftBitmap, i == 1 ? mfmarker : None,
				    NULL);
    
	    XtAddCallback(entry, XtNcallback, FilesMenuSelect, (XtPointer)i);
	    if (i == 1 ) XtVaSetValues(filesmenu,XtNpopupOnEntry,entry,NULL);
	}
	XtVaGetValues(filesmenu, XtNchildren, &file_entry_list,NULL);
    }


    options = XtVaCreateManagedWidget("options", menuButtonWidgetClass, hpaned,
					XtNmenuName,"optionsmenu",
					NULL);

    optionsmenu = XtVaCreatePopupShell("optionsmenu",simpleMenuWidgetClass,
					 options,
					NULL);


     XtVaGetValues(textmain, XtNwrap, &wrapmode, 
			XtNscrollHorizontal,&scrollmode,
			NULL);

     for (i = 0; i < (int) XtNumber(wrapnames); i++)  {
	char * item = wrapnames[i].name;
	entry = wrapnames[i].w;

        entry = XtVaCreateManagedWidget(item, smeBSBObjectClass,optionsmenu,
					NULL);
        wrapnames[i].w = entry;

        if (wrapnames[i].mode == wrapmode) 
	    XtVaSetValues(entry, XtNleftBitmap, tick, NULL);

        XtAddCallback(entry, XtNcallback, WrapMenuSelect, NULL);
	if (i == 0) XtVaSetValues(optionsmenu,XtNpopupOnEntry,entry,NULL);
    }
    hscroll = XtVaCreateManagedWidget("horiz scroll",  smeBSBObjectClass,
                        optionsmenu,
			XtNleftBitmap,scrollmode == XawtextScrollAlways ? tick : None,
			NULL);

     XtVaCreateManagedWidget("line",smeLineObjectClass,optionsmenu,
                                        NULL);

    casesens = XtVaCreateManagedWidget("case sensitive", smeBSBObjectClass,
			optionsmenu,
			XtNleftBitmap, app_resources.caseSensitive ? tick : None,
			NULL);

    swrap  = XtVaCreateManagedWidget("search wrap", smeBSBObjectClass,
			optionsmenu,
			XtNleftBitmap, app_resources.searchWrap ? tick : None,
			NULL);
  
   searchmany = XtVaCreateManagedWidget("search many", smeBSBObjectClass,
                        optionsmenu,
                        XtNleftBitmap, app_resources.searchMany ? tick : None,
			XtNsensitive, need_files,
                        NULL);

   XtVaCreateManagedWidget("line",smeLineObjectClass,optionsmenu,
                                        NULL);

   autoexec =  XtVaCreateManagedWidget("auto exec", smeBSBObjectClass,
			optionsmenu,
			XtNleftBitmap, app_resources.autoExec ? tick : None,
                        NULL);

   shortname =  XtVaCreateManagedWidget("short name", smeBSBObjectClass,
                        optionsmenu,
                        XtNleftBitmap, app_resources.shortName ? tick : None,
                        NULL);

   quiet =  XtVaCreateManagedWidget("quiet error", smeBSBObjectClass,
			optionsmenu,
			XtNleftBitmap, app_resources.quiet ? tick : None,
                        NULL);

    XtAddCallback(hscroll, XtNcallback, Hscroll,NULL);
    XtAddCallback(swrap, XtNcallback,Searchwrap,NULL);
    XtAddCallback(searchmany, XtNcallback,SearchMany,NULL);
    XtAddCallback(casesens, XtNcallback,CaseSens,NULL);
    XtAddCallback(shortname, XtNcallback, ShortName, NULL);
    XtAddCallback(quiet, XtNcallback, Quiet, NULL);
    XtAddCallback(autoexec, XtNcallback, AutoExec, NULL);

    help = XtVaCreateManagedWidget ("help", commandWidgetClass, hpaned,
					NULL);

    split = XtVaCreateManagedWidget("split",  toggleWidgetClass, hpaned,
                                        NULL);

    flip = XtVaCreateManagedWidget("flip",  commandWidgetClass, hpaned,
					 XtNsensitive, FALSE,
                                        NULL);

    cmnds = XtVaCreateManagedWidget("cmnds", menuButtonWidgetClass, hpaned,
                                        XtNmenuName,"cmndsmenu", 
					XtNlabel,"",
                                        NULL);

    makecmndsmenu();


    textinput = XtVaCreateManagedWidget("textinput",asciiTextWidgetClass,hpaned,
					XtNeditType,XawtextEdit,
					XtNstring, app_resources.command,
					NULL);


    XtAddCallback (quit, XtNcallback, Quit, NULL);
    XtAddCallback (help, XtNcallback, Help, NULL);
    XtAddCallback (split, XtNcallback, Split, NULL);
    XtAddCallback (flip, XtNcallback, Flip, NULL);

    XtInstallAccelerators(textmain,textinput);
    XtInstallAccelerators(textsplit,textinput);


}

void
makecmndsmenu()
/*
 * make menu items from cmndsList resource
 *
 */
{	
	Widget entry;
	int i;
  	int n = 0 ;
	char *s, *menulist;

        cmndsmenu = XtVaCreatePopupShell("cmndsmenu", simpleMenuWidgetClass,
							cmnds, NULL);
	
	
        menulist = XtNewString(app_resources.cmndsList);/* copy it as we're */
							/*	changing it */
	s  = menulist;
        if (!s) s = "";
        if (*s)  n++;	
        while  (*s) {
	    if (*s == '\n') {
		 *s = 0;
		 n++;
	    }
	     s++;
	 };
	if (!*--s) n--;
		
	if (n == 0) {
	    XtVaSetValues(cmnds, XtNsensitive, FALSE, NULL);
	    return;
	}
	
	cmndsmenuentries = (Cmndsmenuentry *) XtMalloc(sizeof(Cmndsmenuentry) *n);	
	s  = menulist; 
	for (i = 0; i < n; i++) {
	     while(isspace(*s)) s++;	 
 	     cmndsmenuentries[i].name = s;	 	
	     while(*s && *s++ != ':' );		
	     *(s-1) = 0;
	     cmndsmenuentries[i].value = s;		
	     while (*s++);
	}
	
	for (i = 0; i < n; i++) {
	    entry = XtVaCreateManagedWidget(cmndsmenuentries[i].name,
				smeBSBObjectClass,
				cmndsmenu,
			 	NULL);
	    
	    XtAddCallback(entry,XtNcallback,CmndsMenuSelect,(XtPointer)i);
	    if (i == 0) XtVaSetValues(cmndsmenu,XtNpopupOnEntry,entry,NULL);
	}

}


/*ARGSUSED*/
static void
Quit(w, client_data, call_data)
Widget w;
XtPointer client_data, call_data;
{
        exit(0);
}

/*
 * Next - action routine
 *
 * read the next file
 *
 * must also adjust sensitivity of filemenu
 *
 * changes textmainextra.file 
 */
/*ARGSUSED*/
static void
Next(w, event,params,num_params)
Widget w;
XEvent *event;
String *params;
Cardinal num_params;
{

    if (textmainextra.file  == Argc -1) {
  	 xperror("At last file!");
	 return;
    }

    newfile(textmainextra.file +1, (char *)NULL, 0);

}


/*
 * Previous - action routine
 *
 * inverse of Next (above)
 */
/*ARGSUSED*/
static void
Previous(w, event,params,num_params)
Widget w;
XEvent *event;
String *params;
Cardinal num_params;
{
     if (textmainextra.file == 1) {
         xperror("At first file!");
         return;
    }   

    newfile(textmainextra.file -1,NULL,0);

}


/*
 *
 * split callback
 *
 * getting split behaviour right tricky when oriention is vertical 'cos
 * newly managed pane always goes at end whereas we want it
 * at begining  so we unmanage textmain first & remanage it to force it to
 * end - setting preferredPaneSize for textsplit before its
 * managed & turning off resizeToPreferred afterwards so its skipAdjust
 * will be override instead - note preferredPaneSize for textsplit  
 * must be exactly right to get pane in correct position just after caret
 *
 * note also shinking & growing of pane to allow for gap between panes
 * and top & bottom margins of the split text widget + caret height if we
 * are pre R5 'cos of scrolling bug
 *
 * also we adjust displayPosition of textmain to be next line after caret,
 * this makes the lines visible on the screen the same as before the split
 */

/*ARGSUSED*/
static void
Split(w, client_data, call_data)
Widget w;
XtPointer client_data, call_data;
{
    Boolean state;
    XawTextPosition curpos,displaypos;
    XRectangle curloc;
    Dimension height,tm,bm,ibw;
/*    char *textdata;  needed only if textsplitextra.data & textmainextra.data not global */
    int length;

    XtVaGetValues(w, XtNstate, &state,NULL);

/* first adjust vpaned height */

#if XT_REVISION < 5
    XawTextSinkGetCursorBounds(getsink(textsplit), &curloc);
#endif

    XtVaGetValues(vpaned,XtNheight,&height, XtNinternalBorderWidth, &ibw,NULL); 
    XtVaGetValues(textsplit, XtNtopMargin, &tm, XtNbottomMargin, &bm, NULL);	
#if XT_REVISION < 5
    XtVaSetValues(vpaned,XtNheight,state ? height +(tm+bm+curloc.height+ibw)
					 : height - (tm+bm+curloc.height+ibw),
#else
    XtVaSetValues(vpaned,XtNheight,state ? height +(tm+bm+ibw)  
					 : height -(tm+bm+ibw),
#endif
	NULL);

    if (state) {
	textsplitextra = textmainextra;
        textsplitextra.widget = textsplit;

	XawTextSinkGetCursorBounds(getsink(textmain),&curloc);
	curpos = XawTextGetInsertionPoint(textmain);
	XtVaGetValues(textmain, XtNlength, &length,
			XtNheight,&height,
			XtNdisplayPosition, &displaypos,
			NULL);

	if (orientation == XtorientVertical)  {
	    XtVaSetValues(textsplit,XtNpreferredPaneSize, 
#if XT_REVISION < 5
				curloc.y+ tm + bm  + curloc.height,
#else 
				curloc.y+ tm + bm,
#endif
				XtNresizeToPreferred, TRUE,
				XtNskipAdjust, TRUE,
        			XtNstring, textmainextra.data,
				XtNlength, length,
				NULL );
            displaypos =  XawTextSourceScan(XawTextGetSource(textmain),
				curpos, XawstEOL, XawsdRight, 1, True) ;
	    XtVaSetValues(textmain,XtNdisplayPosition,displaypos,NULL);
	    XtUnmanageChild(textmain);
        } else  XtVaSetValues(textsplit, XtNstring, textmainextra.data, NULL);

        XtManageChild(textsplit);
	XtManageChild(textmain);
	XtVaSetValues(textsplit,XtNresizeToPreferred, FALSE, NULL);
	XtVaSetValues(textmain, XtNskipAdjust, FALSE, 
		NULL); 
	XawTextSetInsertionPoint(textsplit,curpos);
        XtVaSetValues(flip,XtNsensitive, TRUE,NULL);
    }
    else {
	Flip(w, client_data, call_data);	/* keep top window */
    	XtUnmanageChild(textsplit);
	XtVaSetValues(textmain, XtNskipAdjust, TRUE, NULL);
	XtVaSetValues(flip,XtNsensitive, FALSE, NULL);
	if (XtIsSensitive(files)) {
            XtVaSetValues(file_entry(textsplitextra.file), XtNleftBitmap, None,NULL);
            XtVaSetValues(file_entry(textmainextra.file), XtNleftBitmap, mfmarker,NULL);
	}
    textsplitextra.file = -1; 	/* mark not split */
    maketitle(file_name_display(textmainextra.file), (char *)NULL);
    }
    setwmhints(top,textmain);

}

/*
 * Flip callback
 *
 * flip the text display widgets round so the top (textsplit)
 * contains what was in the bottom (textmain) & vice-versa
 * maintaining position of grip
 *
 */

/*ARGSUSED*/
static void
Flip(w, client_data, call_data)
Widget w;
XtPointer client_data, call_data;
{
    Widget tempw;
    textextra tempx;
    Dimension height,width;

    
    XtVaGetValues(textsplit,XtNheight,&height, XtNwidth, &width, NULL);
 
    XtUnmanageChild(textmain);
    if (orientation == XtorientVertical) 
        XtUnmanageChild(textsplit);

    tempw = textmain;
    textmain = textsplit;
    textsplit = tempw;

    tempx = textmainextra;
    textmainextra = textsplitextra;
    textsplitextra = tempx;

    if (orientation == XtorientVertical) {
    XtVaSetValues(textsplit,XtNpreferredPaneSize, height,
			XtNresizeToPreferred, TRUE,
			NULL );
    XtManageChild(textsplit);
    XtVaSetValues(textsplit,XtNskipAdjust, TRUE, NULL);
    XtVaSetValues(textmain,XtNresizeToPreferred, FALSE,
		XtNskipAdjust, FALSE, NULL);
    XtManageChild(textmain);
    }
    else {
	XtVaSetValues(textsplit,XtNpreferredPaneSize, width,
		 XtNresizeToPreferred, TRUE,
                        NULL );
	XtManageChild(textsplit);
	XtVaSetValues(textmain,XtNresizeToPreferred, FALSE,
                XtNskipAdjust, FALSE, NULL);
    }

    if (textsplitextra.file != textmainextra.file) {

	if (XtIsSensitive(files)) {
	  XtVaSetValues(file_entry(textmainextra.file), XtNleftBitmap, mfmarker,NULL);
	  XtVaSetValues(file_entry(textsplitextra.file), XtNleftBitmap, sfmarker,NULL);
	}
        maketitle(file_name_display(textmainextra.file), (char *)NULL);
    }

 /* now fix wrap & scroll menu to reflect new textmain */
  
   { 
   XawTextWrapMode wrapmode;
   XawTextScrollMode scrollmode;
   int i;

   XtVaGetValues(textmain, XtNwrap, &wrapmode, 
			 XtNscrollHorizontal,&scrollmode,
        	  	NULL);
   for (i = 0; i < (int) XtNumber(wrapnames); i++)  
            XtVaSetValues(wrapnames[i].w, XtNleftBitmap, 
                	wrapnames[i].mode == wrapmode ? tick :None,
	        	NULL);
   
   XtVaSetValues(hscroll,XtNleftBitmap,
			scrollmode == XawtextScrollAlways ? tick : None,
      			NULL);
   }
}

/*
 * WrapMenuSelect
 * called when wrap menu item selected
 *
 * we just run down the wrapnames[] looking for the matching
 * widget we so we can set the wrap  value of the text
 * widgets to appropriate value
 * also for all the menu items, set or unset the widget's insensitivity
 *
 */

/*ARGSUSED*/
static void
WrapMenuSelect(w, junk, garbage)
Widget w;
XtPointer junk, garbage;
{
    int i;
    XawTextWrapMode wrapmode;

    XtVaGetValues(textmain, XtNwrap, &wrapmode, NULL);

    for (i = 0; i < (int) XtNumber(wrapnames); i++)  {
        if (wrapnames[i].w == w) {
            if (wrapmode != wrapnames[i].mode) {
	    XtVaSetValues(textmain, XtNwrap,wrapnames[i].mode,NULL);
	    XtVaSetValues(textsplit, XtNwrap,wrapnames[i].mode,NULL);
	    XtCallActionProc(textmain,"redraw-display",NULL,NULL,0);
             XtVaSetValues(w, XtNleftBitmap,  tick, NULL); 
            }
	    else {
	    XtVaSetValues(textmain, XtNwrap,XawtextWrapNever,NULL);
	    XtVaSetValues(textsplit, XtNwrap,XawtextWrapNever,NULL);
	    XtVaSetValues(wrapnames[i].w, XtNleftBitmap, None, NULL); }
	   
	} else
	    XtVaSetValues(wrapnames[i].w, XtNleftBitmap, None, NULL);
    }
    XtVaSetValues(optionsmenu,XtNpopupOnEntry,w,NULL);
}
    

/*
 * Searchwrap
 * called when search wrap menu item selected
 *
 */

/*ARGSUSED*/
static void
Searchwrap(w, client_data, garbage)
Widget w;
XtPointer client_data, garbage;
{
    app_resources.searchWrap = !app_resources.searchWrap;
    XtVaSetValues(w,XtNleftBitmap,app_resources.searchWrap?tick:None,NULL);
    XtVaSetValues(optionsmenu,XtNpopupOnEntry,w,NULL);
    
}

/*
 * SearchMany
 * called when search all menu item selected
 *
 */

/*ARGSUSED*/
static void
SearchMany(w, dummy, garbage)
Widget w;
XtPointer dummy, garbage;
{
    app_resources.searchMany = !app_resources.searchMany;
    XtVaSetValues(w,XtNleftBitmap,app_resources.searchMany?tick:None,NULL);
    XtVaSetValues(optionsmenu,XtNpopupOnEntry,w,NULL);
   
}

/*
 * CaseSens
 * called when casesens menu item selected
 *
 */

/*ARGSUSED*/
static void
CaseSens(w, dummy, garbage)
Widget w;
XtPointer dummy, garbage;
{
  app_resources.caseSensitive = !app_resources.caseSensitive;
  XtVaSetValues(w,XtNleftBitmap,app_resources.caseSensitive?tick:None,NULL);
  XtVaSetValues(optionsmenu,XtNpopupOnEntry,w,NULL);
}

/*
 * ShortName
 * called when shortname menu item selected
 *
 * must toggle resource & reset title & relabel file menu entries
 */

/*ARGSUSED*/
static void
ShortName(w, dummy, garbage)
Widget w;
XtPointer dummy, garbage;
{
  int i;

  app_resources.shortName = !app_resources.shortName;

  maketitle(file_name_display(textmainextra.file), (char *)NULL);

  for (i = 1; i < Argc; i++) 
     XtVaSetValues(file_entry_list[i-1], XtNlabel, file_name_display(i), NULL);

  XtVaSetValues(w,XtNleftBitmap,app_resources.shortName?tick:None,NULL);
  XtVaSetValues(optionsmenu,XtNpopupOnEntry,w,NULL);
}

/*
 * Quiet
 * called when quiet menu item selected
 *
 */

/*ARGSUSED*/
static void
Quiet(w, dummy, garbage)
Widget w;
XtPointer dummy, garbage;
{
  app_resources.quiet = !app_resources.quiet;
  XtVaSetValues(w,XtNleftBitmap,app_resources.quiet?tick:None,NULL);
  XtVaSetValues(optionsmenu,XtNpopupOnEntry,w,NULL);
}

/*
 * exec
 * called when autoexec menu item selected
 *
 */

/*ARGSUSED*/
static void
AutoExec(w, dummy, garbage)
Widget w;
XtPointer dummy, garbage;
{
  app_resources.autoExec = !app_resources.autoExec;
  XtVaSetValues(w,XtNleftBitmap,app_resources.autoExec?tick:None,NULL);
  XtVaSetValues(optionsmenu,XtNpopupOnEntry,w,NULL);
}

/*
 * Hscroll
 * called when horiz scroll menu item selected
 *
 */

/*ARGSUSED*/
static void
Hscroll(w, dummy, garbage)
Widget w;
XtPointer dummy, garbage;
{
  XawTextScrollMode smode;

  XtVaGetValues(textmain,XtNscrollHorizontal,&smode,NULL);
  XtVaSetValues(textmain, XtNscrollHorizontal, 
      smode == XawtextScrollNever ? XawtextScrollAlways : XawtextScrollNever, 
      NULL);
  XtVaSetValues(textsplit, XtNscrollHorizontal,
      smode == XawtextScrollNever ? XawtextScrollAlways : XawtextScrollNever,
      NULL);
  XtVaSetValues(w,XtNleftBitmap,smode == XawtextScrollNever ? tick : None,
      NULL);

  XtVaSetValues(optionsmenu,XtNpopupOnEntry,w,NULL);

  setwmhints(top,textmain);

}



    

/*
 * FilesMenuSelect
 * called when files menu item selected
 *
 */

/*ARGSUSED*/
static void
FilesMenuSelect(w, client_data, garbage)
Widget w;
XtPointer client_data, garbage;
{

	newfile((int)client_data, NULL,0);
        XtVaSetValues(filesmenu,XtNpopupOnEntry,w,NULL);
}

/*
 * CmndsMenuSelect
 *
 * callback for cmdsmenu -  put the command into the
 * textinput widget  and execute if autoexec resource set,
 * unless its the "special" Cmnd, in which case
 * execute the command already in textinput
 */

/*ARGSUSED*/
static void
CmndsMenuSelect(w, client_data, call_data)
Widget w;
XtPointer client_data, call_data;
{ 
    int i = (int) client_data;
    char *command = cmndsmenuentries[i].value;
    
    if (strcmp(command,"Cmnd"))
         XtVaSetValues(textinput,XtNstring,command,NULL); 

     if ( app_resources.autoExec || !strcmp(command,"Cmnd") )
	 Cmnd(textinput, (XEvent *)NULL, (String *)NULL, 0);

    XtVaSetValues(cmndsmenu,XtNpopupOnEntry,w,NULL);
}

/*ARGSUSED*/
static void
KeyRedirect(w, event,params,num_params)
Widget w;
XEvent *event;
String *params;
Cardinal num_params;
{
    event->xany.window = XtWindow(textinput);
    XtDispatchEvent(event);
}

/*ARGSUSED*/
static void
SetCmndTarget(w, event,params,num_params)
Widget w;
XEvent *event;
String *params;
Cardinal num_params;
{
    cmndtarget = w;
    if (XtIsManaged(w))
    	maketitle(file_name_display(wtote(w)->file), (char *)NULL);
}



void
setbusy()
{
      XtVaSetValues(textinput, XtNcursor, busycursor, NULL);
      XtVaSetValues(textmain, XtNcursor, busycursor, NULL);
      XtVaSetValues(textsplit, XtNcursor, busycursor, NULL);
      XtVaSetValues(hpaned, XtNcursor, busycursor, NULL);

      XFlush(display);
}

void
setnotbusy()
{
      XtVaSetValues(textinput, XtNcursor,textcursor,NULL);
      XtVaSetValues(textmain, XtNcursor,textcursor,NULL);
      XtVaSetValues(textsplit, XtNcursor,textcursor,NULL);
      XtVaSetValues(hpaned, XtNcursor,buttoncursor,NULL);

}

/*
 * DoubleClick
 * action routine for overriding the text widget double click
 * to give sam style bracket matching, word selection etc.
 * (note this disables button 1 "drag" extension)
 */

static void
DoubleClick(w, event,params,num_params)
Widget w;
XEvent *event;
String *params;
Cardinal num_params;
{
    XawTextPosition curpos = XawTextGetInsertionPoint(w);
    textextra* textra = (w == textmain ? &textmainextra : &textsplitextra);
    char *start = textra->data + curpos;
    char *s;
    char c;
    char *starts, *ends;
    char match = 0;
    int depth = 1;

/* first do matching char */


    if (curpos) c = (*(start-1));	    /* try char before caret */
    else c = '\n';			    /* treat pre-start as \n */

    switch (c) {
    case '{':   match = '}'; break;
    case '(':   match = ')'; break;
    case '[':   match = ']'; break;
    case '<':   match = '>'; break;
    case '"':   match = '"'; break;
    case '\'':  match = '\''; break;
    case '`':   match = '`'; break;
    case '\n':  switch (*start) {	  /* give char after precedence */
		case '}':
		case ')': 
		case ']': 
    		case '>': 
    		case '"':  
    		case '\'':  
    		case '`':  break;
		default: match = '\n';break;
		}
		break;
    }

   if (match)  {
       for (s = start ;s < textra->data + textra->size && depth;s++) {
	    if (*s  == c ) depth++;
            if (*s == match) {
	        if (c == match) {s++; depth = 0; break;}
	        depth--;
	    }
        }
	if (match == '\n') depth = 0; /* allow for no cr at eof */
        if (depth == 0) 
            SetSelection(w,curpos,(XawTextPosition)(s-1 - textra->data));
        return;
    }

    c  = *start;			/* try char after caret */
    switch (c) {
    case '\n':  match = '\n';break;
    case '}':	match = '{'; break;
    case ')':   match = '('; break;
    case ']':   match = '['; break;
    case '>':   match = '<'; break;
    case '"':   match = '"'; break;
    case '\'':  match = '\''; break;
    case '`':   match = '`'; break;
    }

    if (match) {
	for (s = start -1;s > textra->data && depth;s--) {
	    if (*s == c ) depth++;
	    if (*s == match) {
		 if (c == match) {s--; depth = 0; break;}
		 depth--;
	    }
        }
        if (depth == 0)
	     SetSelection(w,(XawTextPosition)(s+2 - textra->data),curpos
);
	return;
   }

/* now do "word" selection */

    starts = ends = start;
    for (s = start; isalnum(*s)  || *s == '_'; s++);
    ends = s;

    for (s = start-1; isalnum(*s) || *s == '_'; s--);
    starts = s+1;

    SetSelection(w,(XawTextPosition)(starts - textra->data),
				(XawTextPosition)(ends - textra->data));
}

/*
 * define warning message handlers to trap invalid action calls
 * when we treat unrecognised command as an action
 */
static validAction;

/*ARGSUSED*/
static void
TrapWarningMsgss(name, type, class, defaultp, params, num_params)
     String name, type, class, defaultp, *params;
     Cardinal *num_params;
{
    validAction = False;
}

/*ARGSUSED*/
static void
TrapWarnings(message)
     String message;
{
    validAction = False;
}	


/*
 * Cmnd action routine
 *
 * if params is not null then execute it as the command 
 * (i.e a string passed in from translation table)
 * otherwise execute command from content of textinput widget
 * (i.e. the normal case)
 *
 */

/*ARGSUSED*/
static void
Cmnd(w, event,params,num_params)
Widget w;
XEvent *event;
String *params;
Cardinal num_params;
{

    char *linesearch();
    char *lastlinesearch();
    static void shellr();
    String userdata;
    textextra *te;
    static char cmndbuf[BUFSIZ];
    char *uptr;

   te = wtote(cmndtarget);

   signal(SIGPIPE, SIG_IGN); /* stop sigpipe causing us to die */

   if (params)
	userdata = *params;
   else XtVaGetValues(textinput, XtNstring, &userdata, NULL);

   uptr = userdata;
   while (isspace(*uptr)) uptr++;

   if (!*uptr)
	XtCallActionProc(te->widget,"next-line",NULL,NULL,0);
   else 
   if (isdigit(*uptr)) {
	char *s = NULL;
	int line = atoi(userdata); 
        if (line > 0)
            s = linesearch(te->data,atoi(userdata));
	if (s) {
	     setinsertionpoint(te->widget,(XawTextPosition)(s - te->data));
	     DoubleClick(te->widget,NULL,NULL,0);
	}
	else xperror("Line number out of range!");
   } else {

  cmndbuf[0] = '\0';
  strcat(cmndbuf,uptr+1);

  switch (*uptr) {
  case '|' :
	shellr(te,uptr+1,FALSE);
	break;
  case '>' :
   	shellr(te,uptr+1,TRUE);
        break;
  case ',':
	    if (*++uptr == '>')	
		shellr(te,uptr+1,FALSE);		    
	    else
		SetSelection(te->widget,(XawTextPosition)0,
				(XawTextPosition)te->size,FALSE);

	break;

   case '!' : 
	{
	/*
	 * set cursor busy & flush before we wait in system()
	 */
         if (app_resources.display) {
	    xperror("'!' not allowed");
	    break;
	 }
	
	 setbusy();

	 if (mysystem(cmndbuf,file_name(te->file)) < 0)
		xperror("problem in shell command execution");
	 setnotbusy();
	}
	break;
   case '$' :
	XawTextSetInsertionPoint(te->widget,
		(XawTextPosition)(lastlinesearch(te->data,te->size) - te->data));
	   break;
   case '=':
	   {
	   char *s = te->data;
	   int n = 0;

 	   if (s && te->size) {
	       while (*s)
		if ((*s++) == '\n') n++;
	   if ((*--s) != '\n') n++;
	   }
	   sprintf(cmndbuf,"%d",n);
	   XtVaSetValues(textinput, XtNstring, cmndbuf,NULL);
	   XtCallActionProc(textinput, "end-of-file",NULL,NULL,0);
	   }
	   break;
   case '.':
   case '+':
   case '-':
        if (*uptr == '.') uptr++;
	switch (*uptr) {
            case '=': 
	        sprintf(cmndbuf,"%d", Where(te->widget));
                XtVaSetValues(textinput, XtNstring, cmndbuf,NULL);
                XtCallActionProc(textinput, "end-of-file",NULL,NULL,0);
	        break;
	    case '.': 
               XawTextSetInsertionPoint(te->widget,XawTextGetInsertionPoint(te->widget)); 
	       XtCallActionProc(te->widget,"redraw-display",NULL,NULL,0);
	       break;
	   case '+':
	   case '-':
		{
  		char *s = NULL;
		static int relative = 0;	/* remember for default */
		int n = 0;
		if (*uptr+1) n = atoi(uptr+1);
		if (n) relative = n;
		s = *uptr == '+' ?
			  linesearch(te->data, Where(te->widget) + relative)
			: linesearch(te->data, Where(te->widget)  -relative);
		if (s) {
			XawTextSetInsertionPoint(te->widget,(XawTextPosition) (s - te->data));			
	     		DoubleClick(te->widget,NULL,NULL,0);	
		}
	        else xperror("Line number out of range!");
		}
		break;
	   default:
		XawTextSetInsertionPoint(te->widget,XawTextGetInsertionPoint(te->widget));
		XtCallActionProc(te->widget,"redraw-display",NULL,NULL,0);
	}
	break;
   case '/':
	{
        extern char* readfile();
        extern char* regexperrmess;
	int i = te->file; 
	int size;
	char * data = NULL;
        char *startmatch, *endmatch;

	XawTextPosition curpos = XawTextGetInsertionPoint(te->widget);
	XawTextPosition startpos = curpos;
	Boolean found = FALSE;
	Boolean searchmany = app_resources.searchMany && Argc > 2;
	XawTextPosition  from,to; 
     

/* if searchmany  repeat search for subsequent files but only wrap 
 *  on individual files if searchMany false */

	if (uptr[strlen(uptr)-1] == '/') uptr[strlen(uptr)-1] = 0; 

        setbusy();
        do {
         if (i == te->file) 
	    data = te->data;
	 else if (i == textsplitextra.file)
	    data = textsplitextra.data;
	 else
	    data =  readfile(file_name(i), &size);

/* use selection if nothing after the / */
	if (strlen(uptr+1) == 0) {
	    XawTextGetSelectionPos(te->widget,&from,&to);
	    strncat(uptr+1,data+from,Min( (int)(to - from), BUFSIZ - strlen(uptr)));
	}   
	  if (regsearch(data, data+startpos, uptr+1,TRUE,
       		searchmany ? FALSE:app_resources.searchWrap,
		!app_resources.caseSensitive,
		&startmatch, &endmatch)) {
		    found = TRUE;
		    break;
	  }
	  startpos = 0; /* look from start of file from now on */
	  if ( !(i ==  te->file || i == textsplitextra.file)) XtFree(data);
	} while (searchmany && ++i <= Argc -1);

/* if not found then continue from first file only if searchMany &&
 * searchWrap are true */

	if (!found && searchmany &&  app_resources.searchWrap) {
	    i = 1;
	    do {
         	if (i == te->file)
            	    data = te->data;
        	 else if (i == textsplitextra.file)
         	    data = textsplitextra.data;
         	 else
            	    data =  readfile(file_name(i), &size);

              if (regsearch(data, data, uptr+1,TRUE,
                        app_resources.searchMany? FALSE:app_resources.searchWrap,
			!app_resources.caseSensitive,
               		 &startmatch, &endmatch)) {
                    found = TRUE;
                    break;
              }
              if ( !(i ==  te->file || i == textsplitextra.file)) XtFree(data);
            } while (searchmany && ++i <= te->file);
	}

	XawTextSetInsertionPoint(textinput,(XawTextPosition)(uptr - userdata +1));

	if (!found) xperror(regexperrmess? regexperrmess:"Pattern not found!");
	else  {
	    if (i != te->file) {
	        XtUnmapWidget(te->widget);
		(void) newfile(i,data,size);
	   }
	   SetSelection(te->widget,(XawTextPosition)(startmatch - te->data),
				(XawTextPosition)(endmatch - te->data),TRUE);
	   XtMapWidget(te->widget);
           
        }
        }
	setnotbusy();
	break;
   case '?':
	{
	extern char* readfile();
	int i = te->file, size;
	char * data = NULL;
        char *startmatch, *endmatch;
	XawTextPosition curpos = XawTextGetInsertionPoint(te->widget);
	XawTextPosition startpos = curpos;
	Boolean found = FALSE;
	Boolean searchmany = app_resources.searchMany && Argc > 2;
        XawTextPosition  from,to; 

	setbusy();
	do {
         if (i == te->file) 
	    data = te->data;
	 else if (i == textsplitextra.file)
	    data = textsplitextra.data;
	 else
	    data =  readfile(file_name(i), &size);

  	/* use selection if nothing after the ? */
	if (strlen(uptr+1) == 0) {
	    XawTextGetSelectionPos(te->widget,&from,&to);
	    strncat(uptr+1,data+from,Min( (int)(to - from), BUFSIZ - strlen(uptr)));
	}

        if (regsearch(data, data+startpos,uptr+1,FALSE,
		searchmany? FALSE:app_resources.searchWrap,
		!app_resources.caseSensitive,
	         &startmatch, &endmatch)){
	            found = TRUE;
		    break;
	  }
	  startpos = 0 ; /* look from end of file from now on */
	  if ( !(i ==  te->file || i == textsplitextra.file)) XtFree(data);
	  } while (searchmany && --i >= 1);

/* if not found then continue from last file but only if searchmany &&
 * searchWrap are true */

	 if (!found && searchmany &&  app_resources.searchWrap) {
	    i = Argc - 1;
	    do {
	        if (i == te->file)
            	    data = te->data;
        	 else if (i == textsplitextra.file)
         	    data = textsplitextra.data;
         	 else
            	    data =  readfile(file_name(i), &size);

		if (regsearch(data, data, uptr+1,FALSE,
                        searchmany? FALSE:app_resources.searchWrap,
			!app_resources.caseSensitive,
               		 &startmatch, &endmatch)) {
                    found = TRUE;
                    break;
	        }
              if ( !(i ==  te->file || i == textsplitextra.file)) XtFree(data);
	    }  while (searchmany && --i >= te->file);
	}

	XawTextSetInsertionPoint(textinput,(XawTextPosition) (uptr - userdata +1));
        if (!found) xperror("Pattern not found!");
	else  {
	    if (i != te->file) {
	        XtUnmapWidget(te->widget);
		(void) newfile(i,data,size);
	   }
	   SetSelection(te->widget,(XawTextPosition)(startmatch - te->data),
				(XawTextPosition)(endmatch - te->data),FALSE);
	   XtMapWidget(te->widget);
           
        }
	}
        setnotbusy();
        break;
   case 'b':
	{
		int i;
		while (isspace(*++uptr));
	  	for (i = 1; i < Argc; i++) {
			if (!strcmp(Argv[i],uptr)) {
				newfile(i,NULL,0);
				i = -1;
				break;
			}
		}
		if (i != -1)  xperror("file not in menu");
	}
	break;
   case 'n':
	Next();
	break;
   case 'p':
	Previous();
	break;
   case 'q':
   case 'Q':
        exit(0);
   default:
	{
	    XtErrorMsgHandler oldWarningMsgHandler =
		XtAppSetWarningMsgHandler(app, TrapWarningMsgss);
	    XtErrorHandler oldWarningHandler =
		XtAppSetWarningHandler(app, TrapWarnings);

	    validAction = True;
	    XtCallActionProc(te->widget,uptr,NULL,NULL,0);
	    if (!validAction) xperror("No such command!");
	}

  }
  }

/* now perform none-blocking wait for any children from  '|' command */
#ifdef X_NOT_POSIX
  while (wait3(NULL,WNOHANG,NULL) > 0); 
#else
  while (waitpid (0, NULL, WNOHANG) > 0);			
#endif
}

/*ARGSUSED*/
static void 
Help(widget,w,client_data)
Widget w,widget;
caddr_t client_data;
{
	if (helpshell == NULL) {
	   helpshell = XtVaCreatePopupShell("helpshell",
                                topLevelShellWidgetClass,toplevel,NULL);

	   helppaned = XtVaCreateManagedWidget("helppaned",
                                panedWidgetClass,helpshell,
                                NULL);

	   helplabel = XtVaCreateManagedWidget("helplabel",
				labelWidgetClass, helppaned,
				NULL);
				
	   helptext = XtVaCreateManagedWidget("helptext",
                                asciiTextWidgetClass,helppaned,
				XtNuseStringInPlace, TRUE,
                                NULL);

	   helpdismiss  = XtVaCreateManagedWidget("helpdismiss",
                                commandWidgetClass,helppaned,
				XtNskipAdjust,TRUE,
				NULL);

            XtAddCallback (helpdismiss, XtNcallback, Dismiss, NULL);
        }

        XtPopup(helpshell,XtGrabNone);

}

/*ARGSUSED*/
static void Dismiss(widget,w,client_data)
Widget w,widget;
caddr_t client_data;
{
        XtPopdown(helpshell);
}

/*ARGSUSED*/
static void
AdjustGrip(w, event,params,num_params)
Widget w;
XEvent *event;
String *params;
Cardinal num_params;
{
    XFontStruct *font;
    Dimension height,tm,bm;
    XRectangle caret;
    Widget sink;

    XtVaGetValues(textsplit, 
	XtNheight, &height,
	XtNfont, &font,
	XtNtopMargin, &tm,
	XtNbottomMargin, &bm,
	XtNtextSink, &sink,
                 NULL);

   XawTextSinkGetCursorBounds(sink, &caret);

  XtVaSetValues(textsplit, 
		XtNheight, XawTextSinkMaxLines(sink,height) * 
#if XT_REVISION < 5
		(font->max_bounds.ascent +font->max_bounds.descent) + tm + bm + caret.height,
#else
		(font->max_bounds.ascent +font->max_bounds.descent) + tm + bm,
#endif
		NULL);

   
}

/*
 * here are some input command processing functions
 *
 */

/*
 * shellr handles > command redirection
 *
 * selection  => output selection not whole file
 */

static void
shellr(target,cmndptr,selection)
textextra *target;
char * cmndptr;
Boolean selection;
{
	   FILE *fout, *mypopen();
	   SIGNAL_T (*sigint)();
	   SIGNAL_T (*sigquit)();
	   Boolean background = FALSE;
	   char cmndbuf[BUFSIZ];
	   char * amptr = NULL;
	   char * bptr;

           if (app_resources.display) {
		xperror("'>' not allowed");
		return;
	   }

           setbusy();

	   cmndbuf[0] = 0;
	   strcat(cmndbuf,cmndptr);

	   bptr = amptr = strrchr(cmndbuf,'&');
	
	   if (amptr) {
		while(isspace(*++bptr));
	        if (bptr == &cmndbuf[strlen(cmndbuf)]) {
		    background = TRUE;
		    *amptr = ' ';
		    sigint = signal(SIGINT, SIG_IGN);
		    sigquit = signal(SIGQUIT, SIG_IGN);
		}
	   }
	   
 	   if ( (fout = mypopen(cmndbuf,file_name(target->file)) ) == NULL) 
	       fprintf(stderr,"xp: error executing %s\n",cmndbuf);
           else {
	        if (selection)
		    outputselection(target->widget,fout,target->data);
		else
		    write(fileno(fout),target->data,target->size);

                 background ? fclose(fout) : mypclose(fout); 
           }
	   if (background) {
	       signal(SIGINT, sigint);
	       signal(SIGQUIT, sigquit);
           }

	setnotbusy();
}


/*
 * process stderr for child processes
 *
 */
void
Do_stderr(client_data, source, id)
    XtPointer client_data;
    int source;
    XtInputId *id;
{
    int fd = (int)client_data;
    char buf [4096];
    int n;

    unsigned int mask_return;

    n = read(fd, buf, 4096);
    buf[n] = 0;
    printf("got %d\n",n);
    xperror(buf);
}


/*
 *
 * set stdin to /dev/null
 * set stdout to /dev/null
 * divert stderr back here via pipe
 * can cause deadlock if pipe fills
 *
 */
redirect() 
{
  static int stderr_pipe[2];

    pipe(stderr_pipe);

    dup2(open("/dev/null",O_RDONLY),0);
    dup2(open("/dev/null",O_WRONLY),1);
    dup2(stderr_pipe[1],2);
    close(stderr_pipe[1]);
    XtAppAddInput(app,stderr_pipe[0],(XtPointer)(XtInputReadMask|XtInputExceptMask), Do_stderr,(XtPointer)stderr_pipe[0]);

}
