/*
 * Copyright (c) 1994, 1995, 1996, 1997, 1998, 2000, Mark Buser.
 * Copyright (c) 2003, 2004, Danny Backx.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are
 * met:
 *
 * Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the following disclaimer.
 * Redistributions in binary form must reproduce the above copyright
 * notice, this list of conditions and the following disclaimer in the
 * documentation and/or other materials provided with the distribution.
 * Neither the names the authors (see above), nor the names of other
 * contributors may be used to endorse or promote products derived from
 * this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 * A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 * OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * $Header: /pack/anoncvs/xinvest/src/trans.c,v 2.30 2004/04/24 07:58:40 danny Exp $
 */
#include <ctype.h>
#include <math.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>

#include <Xm/Xm.h>

#include "rate.h"
#include "trans.h"
#include "transP.h"
#include "status.h"
#include "util.h"
#include "account.h"

/* globals */
static int lineno = 0;
static int transno = 0;
static int transalloc = 0;
static TRANS *trans;
static char *title = NULL;
static char *ticker = NULL;
static char *eticker = NULL;
static char *cticker = NULL;
static char *ecticker = NULL;

/* Are there any transactions, returns how many */
int AreTrans() {
  return (transno);
}

/* Set transaction record to none */
static void ClrTrans()
{
  lineno = 0;
  title = NULL;
  ticker = NULL;
  cticker = NULL;
  transno = 0;
  transalloc = 0;
  trans = NULL;
}

/* Return pointer to trans structure */
void *GetTrans()
{
  return (trans);
}

/* Return pointer to transaction title, if there is one */
char *GetTransTitle()
{
  return title;
}

/* Return pointer to ticker tape symbol, if there is one */
char *GetTransTicker()
{
  return ticker;
}

/* Return pointer to ticker tape symbol, if there is one */
char *GetTransETicker()
{
  return eticker;
}

/* Return pointer to exchange ticker tape symbol, if there is one */
char *GetTransCurrTicker()
{
  return cticker;
}

/* Return pointer to exchange ticker tape symbol, if there is one */
char *GetTransCurrETicker()
{
  return ecticker;
}

/* 
** Return date of transaction "num"
*/
char *GetTransDate( void *trans, int num )
{
  return (((TRANS *)trans)[num].t_date);
}

long GetTransLDate( void *trans, int num )
{
  return (((TRANS *)trans)[num].t_ldate);
}

/* 
** Return NAV of transaction "num" not modified by exchange rate.
*/
double GetTransNav( void *trans, int num )
{
  return ( ((TRANS *)trans)[num].t_nav);
}

/* 
** Return type of transaction "num"
*/
int GetTransType( void *trans, int num )
{
  return ( ((TRANS *)trans)[num].t_type );
}

/* 
** Return load of transaction "num"
*/
double GetTransLoad( void *trans, int num )
{
  return ( ((TRANS *)trans)[num].t_load );
}

/* 
** Return number of shares of transaction "num"
*/
double GetTransShares( void *trans, int num )
{
  return (((TRANS *)trans)[num].t_shares);
}

/* 
** Return exchange rate of transction "num"
*/
double GetTransCurrency(void *trans, int num)
{
	int	ca = 0;
	TRANS	*tp = &((TRANS *)trans)[num];

#if 0
	fprintf(stderr, "GetTransCurrency(%s)\n", GetAccountTitle(tp->t_account));
#endif
	if (tp->t_account && (ca = Account2CurrencyAccount(tp->t_account))) {
		return GetCurrencyForDate(ca, tp->t_ldate);
	}
	return tp->t_currency;
}

/*
** Return value of indicated flag value at transaction num 
*/
int GetTransFlag( void *trans, int num, int flag )
{
  if (flag == TRANS_REINVEST)
    return (((TRANS *)trans)[num].t_flag & flag);
  else
    return 0;
}

/*
** Return value of split major:minor
*/
int GetTransSplit( void *trans, int num, int which )
{
  if (which == TRANS_MAJOR)
    return (((TRANS *)trans)[num].t_major);
  if (which == TRANS_MINOR)
    return (((TRANS *)trans)[num].t_minor);
  else
    return 1;
}

/*
** Read in a transaction from trans, reset to 1st line on new file.
** Return 0 on ok, -1 on error.
*/
char *rd_trans(int account, char *transline, int new )
{
    TRANS *tp;                  /* ptr to slots in trans[] array */

    char *newline, *token;
    char date[11];              /* store date here */
    int  type;                  /* T_BUY, T_SELL, or T_DIST */
    double nav = 0.0;           /* buy price, sell price, reinvest price */
    double shares = 0.0;        /* number of shares buy or sell */
    double load = 0.0;          /* transaction cost */
    char loadpercent[2] = " ";  /* is load in percentage */
    int  major = 1, minor = 1;  /* split major:minor */

    static double curr = 1.0;   /* exchange rate */
    static int reinvest = True; /* reinvest gains or not */
    static char err_msg[80];

    int num_read, expect;

    if (new) {
      ClrTrans();
      reinvest = True;
      curr = 1.0;
    }

    if ( (transline == NULL) || strlen(transline) == 0 )
      return (0);

    lineno++;

    /* skip comment lines */
    if (*transline == '#' || *transline == '*' || isspace(*transline))
      return (0);

    strcpy (err_msg, "");  /* No errors */
    newline = XtNewString (transline);
    token = strtok (newline, " \t");

    /* In order of likely occurence */

    if (strchr(token, '/')) { /* DATE */
      /* Error checks */
      if (strlen(token) > 10 || strtoday(token) == -1) 
        sprintf( err_msg, "Line %d: invalid date '%s'.", lineno, token );

      if (transno>0 && (strtoday (token) < strtoday (trans[transno].t_date)) )
        sprintf( err_msg, "Line %d: '%s' is earlier\nthan previous date '%s'.", 
                 lineno, token, trans[transno].t_date );
      if (strlen(err_msg)) {
	XtFree (newline);
	return (err_msg);
      }

      strcpy(date, token);   /* Date seems ok */
      { /* Adjust 50+ to 1950+, 49- to 2049- */
        int m, d, y;
        sscanf (date, "%d/%d/%d", &m, &d, &y); 
        if (y < 50)       y += 2000;
        else if (y < 100) y += 1900;
        sprintf (date, "%2d/%2d/%4d", m, d, y);
      }
      token = strtok (NULL, "\n");
      token += strspn (token, " \t");

    }
     
    if (strncasecmp ("BUY", token, strlen("buy")) == 0 ||
        strncasecmp ("exchange", token, strlen("exchange")) == 0) {
       type = T_BUY;              /* price shares [load[%]] */
       num_read = sscanf (token, "%*s %lf %lf %lf %1s", 
                          &nav, &shares, &load, loadpercent);
       if (num_read == -1) num_read = 0;
       else if (num_read == 4 && loadpercent[0] == '%') {
          load = fabs(shares) * nav * load / 100.0;
	  expect = 4;
       } else if (num_read == 3) expect = 3;
       else  expect = 2;
       num_read +=2; expect +=2; /* Minimize user confusion include date/type*/


    } else if (strncasecmp ("SELL", token, strlen("sell"))  == 0) {
       type = T_SELL;            /* price shares [load[%]] */
       num_read = sscanf (token, "%*s %lf %lf %lf %1s", 
                          &nav, &shares, &load, loadpercent);
       if (shares > 0) shares = -shares;
       if (num_read == -1) num_read = 0;
       else if (num_read == 4 && loadpercent[0] == '%') {
          load = fabs(shares) * nav * load / 100.0;
	  expect = 4;
       } else if (num_read == 3) expect = 3;
       else  expect = 2;
       num_read +=2; expect +=2; /* Minimize user confusion include date/type*/


    } else if (strncasecmp ("PRICE", token, strlen("price"))  == 0) {
       type = T_PRICE;           /* price */
       num_read = sscanf (token, "%*s %lf", &nav);
       if (num_read == -1) num_read = 0;
       expect = 1;
       num_read +=2; expect +=2; /* Minimize user confusion include date/type*/


    } else if (strncasecmp ("STCG", token, strlen("stcg")) == 0 ||
               strncasecmp ("LTCG", token, strlen("ltcg")) == 0 ||
               strncasecmp ("DIV", token, strlen("div")) == 0) {
       type = T_DIST;         /* price shares [load[%]] */
       num_read = sscanf (token, "%*s %lf %lf %lf %1s", 
                          &nav, &shares, &load, loadpercent);
       if (num_read == -1) num_read = 0;
       else if (num_read == 4 && loadpercent[0] == '%') {
          load = fabs(shares) * nav * load / 100.0;
	  expect = 4;
       } else if (num_read == 3) expect = 3;
       else  expect = 2;
       num_read +=2; expect +=2; /* Minimize user confusion include date/type*/


    } else if (strncasecmp ("SPLIT", token, strlen("split"))  == 0) {
       type = T_SPLIT;           /* [price] major:minor */
       num_read = sscanf (token, "%*s %lf %d:%d", &nav, &major, &minor);
       if (num_read == -1) num_read = 0;
       else if (num_read != 3) { /* Optional price might not be present */
         num_read = sscanf (token, "%*s %d:%d", &major, &minor);
	 expect = 2;
         if (transno>0 && major>0) /* Better if everybody has a price*/
           nav = (trans[transno].t_nav * minor)/ major;  /* adjust for split */
       } else expect = 3;
       num_read +=2; expect +=2; /* Minimize user confusion include date/type*/


    } else if (strncasecmp ("CURRENCY", token, strlen("currency"))  == 0) {
       type = T_CURR;          /* price */
       num_read = sscanf (token, "%*s %lf", &curr);
       if (num_read == -1) num_read = 0;
       if (transno>0) nav = trans[transno].t_nav;   /*Everybody needs a price*/
       expect = 1;
       num_read +=1; expect +=1; /* Minimize user confusion include date/type*/


    } else if (strncasecmp ("ESOP", token, strlen("esop"))  == 0) {
       type = T_ESOP;     /* strike_price shares vested_shares */
       num_read = sscanf (token, "%*s %lf %*f %lf", &nav, &shares);
       if (num_read == -1) num_read = 0;
       expect = 2;
       num_read +=3; expect +=3; /* Minimize user confusion include date/type*/


    /* 
    ** All of the following types return without generating a transaction.
    */
    } else if (strncasecmp ("REINVEST", token, strlen("reinvest"))  == 0) {
       type = T_REINV;         /* Reinvest {All,None} */
       token = strtok (NULL, " \t");
       if (token == NULL)
         sprintf (err_msg, 
		  "Line %d: missing parameter in reinvest directive.", lineno);
       else if (strcasecmp(token, "all") == 0)
         reinvest = True;
       else if (strcasecmp(token, "none") == 0)
         reinvest = False;
       else
         sprintf (err_msg, 
		  "Line %d: invalid parameter '%s' in reinvest directive.",
	          lineno, token);
       XtFree (newline);
       return (strlen(err_msg)?err_msg:0);


    } else if (strncasecmp("PERCENT", token, strlen("percent")) == 0) {
       type = T_PERC;          /* Percent percent% name[.name] */
       return(0);


    } else if (strncasecmp ("TITLE", token, strlen("title"))  == 0) {
       type = T_TITLE;         /* Title name */
       if ((token = strtok (NULL, "\n"))) {
         if (title) XtFree(title);
	 title = XtNewString(token);
       }
       XtFree (newline);
       return (0);


    } else if (strncasecmp ("TICKER", token, strlen("ticker"))  == 0) {
       type = T_TICK;          /* Ticker symbol expression */
       if ((token = strtok (NULL, " \t"))) {
         if (ticker) XtFree(ticker);
	 ticker = XtNewString(token);
       }
       if ((token = strtok (NULL, " \t"))) {
	 int end = strspn(token, "xX/*.0123456789");
	 if (end==0 || end != strlen(token)) {
           sprintf(err_msg,"Line %d: Unexpected characters in expression '%s'.",
		   lineno, token);
	 } else {
           if (eticker) XtFree(eticker);
	   eticker = XtNewString(token);
	 }
       }
       XtFree (newline);
       return (strlen(err_msg)?err_msg:0);


    } else if (strncasecmp ("CTICKER", token, strlen("cticker"))  == 0) {
       type = T_CTICK;         /* Cticker symbol expression */
       if ((token = strtok (NULL, " \t"))) {
         if (cticker) XtFree(cticker);
	 cticker = XtNewString(token);
       }
       if ((token = strtok (NULL, " \t"))) {
	 int end = strspn(token, "xX/*.0123456789");
	 if (end==0 || end != strlen(token)) {
           sprintf(err_msg,"Line %d: Unexpected characters in expression '%s'.",
		   lineno, token);
	 } else {
           if (ecticker) XtFree(ecticker);
	   ecticker = XtNewString(token);
	 }
       }
#if 0
       fprintf(stderr, "CTicker(%s,%s)\n", cticker, ecticker);
#endif
       XtFree (newline);
       return (strlen(err_msg)?err_msg:0);


    } else {
       sprintf (err_msg, "Line %d: Unknown directive '%s'.", lineno, token);
       XtFree (newline);
       return (err_msg);
    }

    /* Least to most important ordering of error checks */

    if (load < 0.0)              /* Check for a reasonable load */
      sprintf( err_msg, "Line %d, specified load '%.2f' is negative.\n",
               lineno, load );
    else if (shares > 0.0 && load > (shares * nav))
      sprintf( err_msg, 
               "Line %d, specified load '%.2f' is not\n"
               "in range 0 to 100%% of transaction value.\n",
               lineno, load );

    if (type == T_SPLIT) {       /* Check for in range splits */
      if (major <= 0 || major > 255 || minor <= 0 || minor > 255)
        sprintf( err_msg, 
                 "Line %d, split '%d:%d' out\nof valid range of 1 to 255.", 
                 lineno, major, minor );
    }

    if (num_read != expect)      /* Check for expected number of params */
      sprintf (err_msg, 
	       "Line %d: expected %d field(s), processed %d.\n" 
	       "Missing or error in field %d.\n",
               lineno, expect, num_read, num_read+1);

    if (nav == 0.0 && type != T_CURR)
      sprintf (err_msg, "Line %d: Price of 0.0 is not allowed.\n", lineno);


    /* Only prints last detected (most severe) error, but that's ok */
    if (strlen(err_msg)) {
      XtFree (newline);
      return (err_msg);
    }

    /* Seems ok, fill in new transaction structure */
    if (++transno >= transalloc) {
      trans = (TRANS *)
	   XtRealloc ((char *)trans, (transalloc+TRANS_ALLOC)*sizeof(TRANS));
      if (trans == NULL) {
        sprintf (err_msg, "Out of memory: loaded up to line %d", lineno-1);
        return (err_msg);
      }
      transalloc += TRANS_ALLOC;
    }

    tp = &trans[transno];
    tp -> t_type = type;
    /* Save expanded year, not the possibly shorthand one */
    tp -> t_date = strdup( date );
    tp -> t_ldate = strtoday( date );
    tp -> t_shares = shares;
    tp -> t_nav = nav;
    tp -> t_load = load;
    tp -> t_major = major;
    tp -> t_minor = minor;
    if (reinvest == True)
      tp -> t_flag |= TRANS_REINVEST;
    else
      tp -> t_flag &= ~TRANS_REINVEST;
    tp -> t_currency = curr;
    tp->t_account = account;
#if 0
    fprintf(stderr, "rd_trans t_account %d (%s)\n", account, GetAccountTitle(account));
#endif 
    XtFree (newline);
    return ( NULL );
}

#include "account.h"

long GetLastLDate(int acc)
{
	int	ntrans;
	long	d;
	TRANS	*trans;

	getAccount(acc, ACCOUNT_TRANS, &trans);
	getAccount(acc, ACCOUNT_NUM_TRANS, &ntrans);
	d = GetTransLDate(trans, ntrans);

	return d;
}
