/* xdaliclock - a melting digital clock
 * Copyright (c) 1991-2008 Jamie Zawinski <jwz@jwz.org>
 *
 * Permission to use, copy, modify, distribute, and sell this software and its
 * documentation for any purpose is hereby granted without fee, provided that
 * the above copyright notice appear in all copies and that both that
 * copyright notice and this permission notice appear in supporting
 * documentation.  No representations are made about the suitability of this
 * software for any purpose.  It is provided "as is" without express or
 * implied warranty.
 */

#import "DaliClockView.h"
#import "xdaliclock.h"
#import <sys/time.h>

/* Currently there are two redisplay methods implemented here.
   The Aqua way is straightforward, but maybe slow.
   The Quartz way is complicated, and a little flaky, and maybe faster
   but I can't tell.  Define this to do it the Quartz way.
 */
#ifdef USE_IPHONE
# define BE_QUARTZY
#endif /* !USE_IPHONE */

#if MAC_OS_X_VERSION_MAX_ALLOWED <= MAC_OS_X_VERSION_10_4
  /* In 10.5 and later, NSColor wants CGfloat args, but that type
     does not exist in 10.4. */
  typedef CGFloat CGFloat;
#endif


@interface DaliClockView (ForwardDeclarations)

#ifndef USE_IPHONE

+ (NSUserDefaultsController *)userDefaultsController;
+ (void)setUserDefaultsController:(NSUserDefaultsController *)ctl;
#endif /* !USE_IPHONE */

- (void)setForeground:(NSColor *)fg background:(NSColor *)bg;


- (void)clockTick;
- (void)colorTick;
- (void)dateTick;
@end


@implementation DaliClockView

#ifndef USE_IPHONE

static NSUserDefaultsController *staticUserDefaultsController = 0;

+ (NSUserDefaultsController *)userDefaultsController
{
  return staticUserDefaultsController;
}


+ (void)initializeDefaults:(NSUserDefaultsController *)controller
{
  static BOOL initialized_p = NO;
  if (initialized_p)
    return;
  initialized_p = YES;

  staticUserDefaultsController = controller;
  [controller retain];

  /* Set the defaults for all preferences handled by DaliClockView.
     (AppController or DaliClockSaverView handles other preferences).

     Depending on class-initialization order, another class (e.g.
     AppController) might have already put defaults in the preferences
     object, so make sure we add ours instead of overwriting it all.
   */
  NSColor *deffg = [NSColor  blueColor];
  NSColor *defbg = [NSColor colorWithCalibratedHue:0.50
                                        saturation:1.00
                                        brightness:0.70   // cyan, but darker
                                             alpha:0.30]; // and translucent
  NSDate *defdate = [NSDate dateWithTimeIntervalSinceNow:
                     (NSTimeInterval) 60 * 60 * 12];  // 12 hours from now

  NSDictionary* defaults = [NSDictionary dictionaryWithObjectsAndKeys:
    @"0", @"hourStyle",
    @"0", @"timeStyle",
    @"0", @"dateStyle",
    [NSArchiver archivedDataWithRootObject:deffg], @"initialForegroundColor",
    [NSArchiver archivedDataWithRootObject:defbg], @"initialBackgroundColor",
    @"10.0", @"cycleSpeed",
    @"NO", @"usesCountdownTimer",
    defdate,@"countdownDate",
    nil];

  NSMutableDictionary *dict = [NSMutableDictionary dictionaryWithCapacity:100];
  [dict addEntriesFromDictionary: [controller initialValues]];
  [dict addEntriesFromDictionary:defaults];
  [controller setInitialValues:dict];
  [[controller defaults] registerDefaults:dict];
}


- (void)bindPreferences
{
  NSUserDefaultsController *controller = staticUserDefaultsController;

  [self    bind:@"hourStyle"
       toObject:controller
    withKeyPath:@"values.hourStyle" options:nil];
  [self    bind:@"timeStyle"
       toObject:controller
    withKeyPath:@"values.timeStyle"
        options:nil];
  [self    bind:@"dateStyle"
       toObject:controller
    withKeyPath:@"values.dateStyle"
        options:nil];
  [self    bind:@"cycleSpeed"
       toObject:controller
    withKeyPath:@"values.cycleSpeed"
        options:nil];
  [self    bind:@"usesCountdownTimer"
       toObject:controller
    withKeyPath:@"values.usesCountdownTimer"
        options:nil];
  [self    bind:@"countdownDate"
       toObject:controller
    withKeyPath:@"values.countdownDate"
        options:nil];

  NSDictionary *colorBindingOptions =
    [NSDictionary dictionaryWithObject:@"NSUnarchiveFromData"
                                forKey:NSValueTransformerNameBindingOption];
  [self    bind:@"initialForegroundColor"
       toObject:controller
    withKeyPath:@"values.initialForegroundColor"
        options:colorBindingOptions];
  [self    bind:@"initialBackgroundColor"
       toObject:controller
    withKeyPath:@"values.initialBackgroundColor"
        options:colorBindingOptions];
}

#endif /* !USE_IPHONE */


- (id)initWithFrame:(NSRect)rect
{
  self = [super initWithFrame:rect];
  if (! self) return nil;

#warning 64BIT: Inspect use of sizeof
  memset (&config, 0, sizeof(config));

# ifndef BE_QUARTZY
  img = [[[NSImage alloc] init] retain];
  if (! img) abort();
# endif /* !BE_QUARTZY */

  config.max_fps = 12;

  [self setOwnWindow:YES];  // by default, we may change window background

#ifndef USE_IPHONE
  // Tell the color selector widget to show the "opacity" slider.
  [[NSColorPanel sharedColorPanel] setShowsAlpha:YES];
#endif /* !USE_IPHONE */

  // initialize the fonts and bitmaps
  [self setFrameSize:[self frame].size];

  [self clockTick];
  [self colorTick];

#ifndef USE_IPHONE
  [self bindPreferences];
#endif /* !USE_IPHONE */

  return self;
}


-(oneway void)dealloc
{
# ifndef BE_QUARTZY
  [img release];
# endif /* !BE_QUARTZY */
  [super dealloc];
}

#ifndef USE_IPHONE

/* Called when the user starts interactively resizing the window.
 */
- (void)viewWillStartLiveResize
{
}

/* Called when the user is done interactively sizing the window.
 */
- (void)viewDidEndLiveResize
{
  // Resize the frame one last time, now that we're finished dragging.
  [self setFrameSize:[self frame].size];
}


/* Called when the View is resized.
 */
- (void)setFrameSize:(NSSize)newSize
{
  [super setFrameSize:newSize];

  // If the user is interactively resizing the window, don't regenerate
  // the pixmap until we're done.  (We will just scale whatever image is
  // already on the window instead, which reduces flicker when the
  // target bitmap size shifts over a boundary).
  if ([self inLiveResize])  return;

  NSInteger ow = config.width;
  NSInteger oh = config.height;

   config.width  = newSize.width * 2;   // use the next-larger bitmap
   config.height = newSize.height * 2;
//  config.width = 1280;    // always use the biggest font image
//  config.height = 1024;

  render_bitmap_size (&config, &config.width, &config.height);

  if (config.render_state && (ow == config.width && oh == config.height))
    return;  // nothing to do

  // When the window is resized, re-create the bitmaps for the largest
  // font that will now fit in the window.
  //
  if (config.bitmap) free (config.bitmap);
  config.bitmap = calloc (1, config.height * (config.width << 3));
  if (! config.bitmap) abort();

  if (pixmap) free (pixmap);
  pixmap = calloc (1, config.height * config.width * 4);
  if (! pixmap) abort();

  if (config.render_state)
    render_free (&config);
  render_init (&config);
}


/* Announce our willingness to accept keyboard input.
 */
- (BOOL)acceptsFirstResponder
{
  return YES;
}


/* Display date when mouse clicked in window.
 */
- (void)mouseDown:(NSEvent *)ev
{
  config.display_date_p = 1;
}

/* Back to time display when mouse released.
 */
- (void)mouseUp:(NSEvent *)ev
{
  config.display_date_p = 0;
}


/* Typing Ctrl-0 through Ctrl-9 and Ctrl-hyphen are a debugging hack.
 */
- (void)keyDown:(NSEvent *)ev
{
  NSString *ns = [ev charactersIgnoringModifiers];
  if (! [ns canBeConvertedToEncoding:NSASCIIStringEncoding])
    goto FAIL;
  const char *s = [ns cStringUsingEncoding:NSASCIIStringEncoding];
  if (! s) goto FAIL;
  if (strlen(s) != 1) goto FAIL;
  if (! ([ev modifierFlags] & NSControlKeyMask)) goto FAIL;
  if (*s == '-' || (*s >= '0' && *s <= '9'))
    config.test_hack = *s;
  else goto FAIL;
  return;
FAIL:
  [super keyDown:ev];
}
#endif /* !USE_IPHONE */


/* This is called from the timer (and other places) to change the colors.
 */
- (void)setForeground:(NSColor *)new_fg background:(NSColor *)new_bg
{
  if (fg != new_fg) {
    if (fg) [fg release];
# ifdef USE_IPHONE
    fg = [[new_fg copy] retain];
# else  /* !USE_IPHONE */
    fg = [[new_fg colorUsingColorSpaceName:NSCalibratedRGBColorSpace] retain];
# endif /* !USE_IPHONE */
  }
  if (bg != new_bg) {
    if (bg) [bg release];
# ifdef USE_IPHONE
    bg = [[new_bg copy] retain];
# else  /* !USE_IPHONE */
    bg = [[new_bg colorUsingColorSpaceName:NSCalibratedRGBColorSpace] retain];
# endif /* !USE_IPHONE */
  }
  [self setNeedsDisplay:TRUE];
}



/* The big kahuna refresh method.  Draw the current contents of the bitmap
   onto the window.  Sounds easy, doesn't it?
 */
- (void)drawRect:(NSRect)rect
{
  NSRect framerect, fromrect, torect;
  framerect.size = [self frame].size;
  framerect.origin.x = framerect.origin.y = 0;
  fromrect.origin.x = fromrect.origin.y = 0;
  fromrect.size.width = config.width;
  fromrect.size.height = config.height;

  if (config.width <= 0 || config.height <= 0) abort();

  CGFloat img_aspect = (CGFloat) config.width / (CGFloat) config.height;
  CGFloat win_aspect = framerect.size.width / (CGFloat) framerect.size.height;

  // Scale the image to fill the window without changing its aspect ratio.
  //
  if (win_aspect > img_aspect) {
    torect.size.height = framerect.size.height;
    torect.size.width  = framerect.size.height * img_aspect;
  } else {
    torect.size.width  = framerect.size.width;
    torect.size.height = framerect.size.width / img_aspect;
  }

  // The animation slows down a lot if we use truly gigantic numbers,
  // so limit the number size in screensaver-mode.
  //
  if (constrainSizes) {
    NSInteger maxh = (config.time_mode == SS ? 512 : /*256*/ 200);
    if (torect.size.height > maxh) {
      torect.size.height = maxh;
      torect.size.width  = maxh * img_aspect;
    }
  }

  // put a margin between the numbers and the edge of the window.
  torect.size.width  *= 0.95;  // 5% horizontally
  torect.size.height *= 0.80;  // 20% vertically

//  torect.size = fromrect.size; // #### debugging: don't scale

  // center it in the window
  //
  torect.origin.x = (framerect.size.width  - torect.size.width ) / 2;
  torect.origin.y = (framerect.size.height - torect.size.height) / 2;

# ifndef BE_QUARTZY

  /* This is the straightforward Aqua way of doing things.
     It's simple, but I think it might be too slow.
     (It relies on the 32bpp pixmap created in colorizePixmap
     instead of using the 1bpp bitmap directly)
   */

  NSBitmapImageRep *imgrep = [NSBitmapImageRep alloc];
#if 1
  [imgrep
    initWithBitmapDataPlanes:&pixmap
                  pixelsWide:config.width
                  pixelsHigh:config.height
               bitsPerSample:8
             samplesPerPixel:4
                    hasAlpha:YES
                    isPlanar:NO
              colorSpaceName:NSCalibratedRGBColorSpace
                 bytesPerRow:config.width * 4
                bitsPerPixel:32
    ];
#else
    [imgrep
    initWithBitmapDataPlanes:&config.bitmap
                  pixelsWide:config.width
                  pixelsHigh:config.height
               bitsPerSample:1
             samplesPerPixel:1
                    hasAlpha:NO
                    isPlanar:YES
              colorSpaceName:NSCalibratedWhiteColorSpace
                 bytesPerRow:config.width >> 3
                bitsPerPixel:1
    ];
/*
 [imgrep colorizeByMappingGray:0.0
                      toColor:bg
                 blackMapping:bg
                 whiteMapping:fg];
 */
#endif
  if (! imgrep) abort();

  [img setSize:[imgrep size]];
  [img setScalesWhenResized:NO];

  if (ownWindow) {
    // [[self window] setOpaque:NO];
    [[self window] setBackgroundColor:bg];
  } else {
    [bg set];
    [NSBezierPath fillRect:framerect];
  }

  [img addRepresentation:imgrep];
  [img drawInRect:torect
         fromRect:fromrect
        operation:NSCompositeSourceOver
         fraction:1.0];
  [img removeRepresentation:imgrep];
  [imgrep release];

# else /* BE_QUARTZY */

  /* This uses lower level Quartz/CoreGraphics calls to do the drawing.
     It might be more efficient, but it's way more complicated...
   */

  CGFloat fgr, fgg, fgb, fga;
  CGFloat bgr, bgg, bgb, bga;
#ifndef USE_IPHONE
  [fg getRed:&fgr green:&fgg blue:&fgb alpha:&fga];
  [bg getRed:&bgr green:&bgg blue:&bgb alpha:&bga];
#else  /* USE_IPHONE */
  const CGFloat *rgba;
  rgba = CGColorGetComponents ([fg CGColor]);
  fgr = rgba[0]; fgg = rgba[1]; fgb = rgba[2]; fga = rgba[3]; 
  rgba = CGColorGetComponents ([bg CGColor]);
  bgr = rgba[0]; bgg = rgba[1]; bgb = rgba[2]; bga = rgba[3]; 
#endif /* USE_IPHONE */

  CGDataProviderRef provider =
    CGDataProviderCreateWithData (NULL /* info */, config.bitmap,
                                  config.height * (config.width>>3),
                                  NULL);
  CGFloat decode[] = { 1.0, 0.0 };  /* invert pixels */
  CGImageRef fgmask=0, bgmask=0;
  fgmask = CGImageMaskCreate (config.width, config.height,
                              1 /* bitsPerComponent */,
                              1 /* bitsPerPixel */,
                              config.width>>3 /* bytesPerRow */,
                              provider, decode,
                              NO /* shouldInterpolate */
                              );
  if (fga < 1.0)
    bgmask = CGImageMaskCreate (config.width, config.height,
                                1 /* bitsPerComponent */,
                                1 /* bitsPerPixel */,
                                config.width>>3 /* bytesPerRow */,
                                provider, NULL /* decode */,
                                NO /* shouldInterpolate */
                                );
  CGDataProviderRelease (provider);

#ifdef USE_IPHONE
  CGContextRef cgc = [[[self window] graphicsContext] graphicsPort];
#else
  CGContextRef cgc = [[NSGraphicsContext currentContext] graphicsPort];
#endif /* !USE_IPHONE */

  CGRect bgrect, fgrect;
  bgrect.origin.x    = framerect.origin.x;
  bgrect.origin.y    = framerect.origin.y;
  bgrect.size.width  = framerect.size.width;
  bgrect.size.height = framerect.size.height;
  fgrect.origin.x    = torect.origin.x;
  fgrect.origin.y    = torect.origin.y;
  fgrect.size.width  = torect.size.width;
  fgrect.size.height = torect.size.height;

  if (! bgmask) {
    /* foreground is solid; background is solid or transparent.
       we can just blast the background down as a rectangle.
     */
    CGContextSetRGBFillColor (cgc, bgr,bgg,bgb,bga);
    if (bga < 1.0)
      CGContextClearRect (cgc, bgrect);
    CGContextFillRect (cgc, bgrect);

  } else {
    /* foreground is transparent; background is solid or transparent.
       we have to draw the background with a mask so that it doesn't
       interfere with the foreground.
     */
    CGContextSaveGState (cgc);
    CGContextSetRGBFillColor (cgc, bgr,bgg,bgb,bga);
    CGContextClearRect (cgc, bgrect);

    /* Ok, this is fucked up.  Since our "mask" bitmap is not as large as
       the window itself, and there's no way to specify a mask origin for
       the inverse mask (that is, say "draw everything *but* this mask)
       we have to fill in the rectangles outside the masked rectangle first.
       But, this leaves artifacts!  Even though the rectangles line up
       exactly, there's a transparent box getting left around the characters.
       Or, if the retangles overlap, then we get a too-dark box.  SHIT!
     */
    CGRect rects[4];
    rects[0] = bgrect;
    rects[0].size.height = fgrect.origin.y - bgrect.origin.y /*+ 1*/;
    rects[1] = rects[0];
    rects[1].origin.y = fgrect.origin.y + fgrect.size.height /*- 1*/;
    rects[2] = rects[1];
    rects[2].origin.y = fgrect.origin.y;
    rects[2].size.width = fgrect.origin.x - bgrect.origin.x /*+ 1*/;
    rects[2].size.height = fgrect.size.height;
    rects[3] = rects[2];
    rects[3].origin.x = bgrect.origin.x+fgrect.origin.x+fgrect.size.width/*-1*/;
    rects[3].size.width = bgrect.size.width-fgrect.size.width-fgrect.origin.x;
    rects[3].size.height = fgrect.size.height;

    CGContextFillRects (cgc, rects, 4);

    CGContextClipToMask (cgc, fgrect, bgmask);
    CGContextFillRect (cgc, fgrect);
    CGImageRelease (bgmask);
    bgmask = 0;
    CGContextRestoreGState (cgc);
  }

  CGContextSetRGBFillColor (cgc, fgr,fgg,fgb,fga);
  CGContextClipToMask (cgc, fgrect, fgmask);
  CGContextFillRect (cgc, fgrect);

  CGImageRelease (fgmask);
  fgmask = 0;

# endif /* BE_QUARTZY */
}


/* The code in digital.c gives us back a 1bpp bitmap (in config.bitmap)
   To actually render that in color with Aqua, we need to scale it up to
   an RGBA image (in self.pixmap) using the colors in "fg" and "bg".

   If we're doing the hairy Quartz stuff, this isn't needed.
*/
#ifndef BE_QUARTZY
- (void)colorizePixmap
{
  NSInteger x, y;
  unsigned char *scanin  = config.bitmap;
  unsigned char *scanout = pixmap;

  CGFloat rf, gf, bf, af;
  [fg getRed:&rf green:&gf blue:&bf alpha:&af];
  unsigned char fgr = rf * 255.0;
  unsigned char fgg = gf * 255.0;
  unsigned char fgb = bf * 255.0;
  unsigned char fga = af * 255.0;

  // in the bitmap, background is transparent (composited with window bg).
  unsigned char bgr = 0;
  unsigned char bgg = 0;
  unsigned char bgb = 0;
  unsigned char bga = 0;

  for (y = 0; y < config.height; y++)
  {
    for (x = 0; x < config.width; x++)
    {
      unsigned char bit = scanin[x>>3] & (1 << (7 - (x & 7)));
      unsigned char r, g, b, a;
      if (bit)
        r = fgr, g = fgg, b = fgb, a = fga;
      else
        r = bgr, g = bgg, b = bgb, a = bga;
      *scanout++ = r;
      *scanout++ = g;
      *scanout++ = b;
      *scanout++ = a;
    }
    scanin += (config.width + 7) >> 3;
  }
}
#endif /* !BE_QUARTZY */


/* When this timer goes off, we re-generate the bitmap/pixmap,
   and mark the display as invalid.
*/
- (void)clockTick
{
  if (clockTimer && [clockTimer isValid]) {
    [clockTimer invalidate];
    clockTimer = 0;
  }

  if (config.max_fps <= 0) abort();

# ifndef USE_IPHONE
  NSWindow *w = [self window];
  if (w && ![w isMiniaturized])   // noop if no window yet, or if iconified.
# endif /* !USE_IPHONE */
  {
    struct timeval now;
    struct timezone tzp;
    gettimeofday (&now, &tzp);
    render_once (&config, now.tv_sec, now.tv_usec);
# ifndef BE_QUARTZY
    [self colorizePixmap];
# endif /* !BE_QUARTZY */
    // [[self window] invalidateShadow]; // windows with shadows flicker...
    [self setNeedsDisplay:TRUE];
  }

  // re-schedule the timer according to current fps.
  //
  CGFloat delay = 0.9 / config.max_fps;
  clockTimer = [NSTimer scheduledTimerWithTimeInterval:delay
                                                target:self
                                              selector:@selector(clockTick)
                                              userInfo:nil
                                               repeats:NO];
}


/* When this timer goes off, we re-pick the foreground/background colors,
   and mark the display as invalid.
 */
- (void)colorTick
{
  if (colorTimer && [colorTimer isValid]) {
    [colorTimer invalidate];
    colorTimer = 0;
  }

  if (config.max_cps <= 0) return;   // cycling is turned off, do nothing

  if ([self window]) {  // do nothing if no window yet

    CGFloat h, s, v, a;
    NSColor *fg2, *bg2;
    CGFloat tick = 1.0 / 360.0;   // cycle H by one degree per tick

    [fg getHue:&h saturation:&s brightness:&v alpha:&a];
    h += tick;
    while (h > 1.0) h -= 1.0;
    fg2 = [NSColor colorWithCalibratedHue:h saturation:s brightness:v alpha:a];

    [bg getHue:&h saturation:&s brightness:&v alpha:&a];
    h += tick * 0.91;   // cycle bg slightly slower than fg, for randomosity.
    while (h > 1.0) h -= 1.0;
    bg2 = [NSColor colorWithCalibratedHue:h saturation:s brightness:v alpha:a];

    [self setForeground:fg2 background:bg2];
  }

  /* re-schedule the timer according to current fps.
   */
  CGFloat delay = 1.0 / config.max_cps;
  colorTimer = [NSTimer scheduledTimerWithTimeInterval:delay
                                                target:self
                                              selector:@selector(colorTick)
                                              userInfo:nil
                                               repeats:NO];
}


/* When this timer goes off, we switch to "show date" mode.
 */
- (void)dateTick
{
  if (dateTimer && [dateTimer isValid]) {
    [dateTimer invalidate];
    dateTimer = 0;
  }

  if (autoDateInterval <= 0) return;

  BOOL was_on = config.display_date_p;
  config.display_date_p = !was_on;

  /* re-schedule the timer according to current fps.
   */
  CGFloat delay = autoDateInterval;
  if (!was_on) delay = 1.0;
  dateTimer = [NSTimer scheduledTimerWithTimeInterval:delay
                                               target:self
                                             selector:@selector(dateTick)
                                             userInfo:nil
                                              repeats:NO];
}


- (void)updateCountdown
{
  config.countdown = (usesCountdownTimer
                      ? [countdownDate timeIntervalSince1970]
                      : 0);
}


/* The About button in the menu bar or preferences sheet.
 */
- (IBAction)aboutClick:(id)sender
{
#ifndef USE_IPHONE
  NSBundle *nsb = [NSBundle bundleForClass:[self class]];
  NSAssert1 (nsb, @"no bundle for class %@", [self class]);

  NSDictionary *info = [nsb infoDictionary];
//  NSString *name = [info objectForKey:@"CFBundleExecutable"];
  NSString *name = @"Dali Clock";
  NSString *vers = [info objectForKey:@"CFBundleVersion"];
  NSString *ver2 = [info objectForKey:@"CFBundleShortVersionString"];
  NSString *icon = [info objectForKey:@"CFBundleIconFile"];

  NSString *cred_file = [nsb pathForResource:@"Credits" ofType:@"html"];
  NSAttributedString *cred = [[[NSAttributedString alloc]
                                initWithPath:cred_file
                                documentAttributes:(NSDictionary **)NULL]
                               autorelease];

  NSString *icon_file = [nsb pathForResource:icon ofType:@"icns"];
  NSImage *iimg = [[NSImage alloc] initWithContentsOfFile:icon_file];

  NSDictionary* dict = [NSDictionary dictionaryWithObjectsAndKeys:
    name, @"ApplicationName",
    vers, @"Version",
    ver2, @"ApplicationVersion",
    cred, @"Credits",
    iimg, @"ApplicationIcon",
    @"",  @"Copyright",        // This is in "Credits"
    nil];

  [[NSApplication sharedApplication]
    orderFrontStandardAboutPanelWithOptions:dict];
#endif /* !USE_IPHONE */
}


// hint for XCode popup menu
#pragma mark accessors

/* [ken] if you (1) only wanted these for bindings purposes (true) and
   (2) they were dumb methods that just manipulated ivars of the same
   name (not true, they mostly write into config), then we would not
   have to write them.  I use the freeware Accessorizer to give me
   accessor templates.
 */
- (NSInteger)hourStyle { return !config.twelve_hour_p; }
- (void)setHourStyle:(NSInteger)aHourStyle
{
  config.twelve_hour_p = !aHourStyle;
}


- (NSInteger)timeStyle { return config.time_mode; }
- (void)setTimeStyle:(NSInteger)aTimeStyle
{
  if (config.time_mode != aTimeStyle) {
    config.time_mode = aTimeStyle;
    config.width++;  // kludge: force regeneration of bitmap
    [self setFrameSize:[self frame].size];
  }
}

- (NSInteger)dateStyle { return config.date_mode; }
- (void)setDateStyle:(NSInteger)aDateStyle
{
  config.date_mode = aDateStyle;
}

- (CGFloat)cycleSpeed { return (CGFloat)config.max_cps; }
- (void)setCycleSpeed:(CGFloat)aCycleSpeed
{
  if (config.max_cps != aCycleSpeed) {
    config.max_cps = (NSInteger)aCycleSpeed;
    if (config.max_cps < 0.0001) {
      [self setForeground:initialForegroundColor
               background:initialBackgroundColor];
    } else {
      [self colorTick];
    }
  }
}

- (NSInteger)usesCountdownTimer { return usesCountdownTimer; }
- (void)setUsesCountdownTimer:(NSInteger)flag
{
  usesCountdownTimer = flag;
  [self updateCountdown];
}

- (NSDate *)countdownDate { return [[countdownDate retain] autorelease]; }
- (void)setCountdownDate:(NSDate *)aCountdownDate
{
  if (countdownDate != aCountdownDate) {
    [countdownDate release];
    countdownDate = [aCountdownDate retain];
    [self updateCountdown];
  }
}

- (NSColor *)initialForegroundColor
{
  return [[initialForegroundColor retain] autorelease];
}

- (void)setInitialForegroundColor:(NSColor *)anInitialForegroundColor
{
  if (initialForegroundColor != anInitialForegroundColor) {
    [initialForegroundColor release];
    initialForegroundColor = [anInitialForegroundColor copy];
    [self setForeground:initialForegroundColor
             background:initialBackgroundColor];
  }
}

- (NSColor *)initialBackgroundColor
{
  return [[initialBackgroundColor retain] autorelease];
}

- (void)setInitialBackgroundColor:(NSColor *)anInitialBackgroundColor
{
  if (initialBackgroundColor != anInitialBackgroundColor) {
    [initialBackgroundColor release];
    initialBackgroundColor = [anInitialBackgroundColor copy];
    [self setForeground:initialForegroundColor
             background:initialBackgroundColor];
  }
}

- (void)setOwnWindow:(BOOL)own_p
{
  ownWindow = own_p;
}

- (void)setConstrainSizes:(BOOL)constrain_p;
{
  constrainSizes = constrain_p;
}

- (void)setAutoDate:(CGFloat)interval
{
  autoDateInterval = interval;

  config.display_date_p = 1;
  [self dateTick];
  config.display_date_p = 0;
}

@end
