/*	$NetBSD: wiconfig.c,v 1.25 2002/04/09 02:56:17 thorpej Exp $	*/
/*
 * Copyright (c) 1997, 1998, 1999
 *	Bill Paul <wpaul@ctr.columbia.edu>.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *	This product includes software developed by Bill Paul.
 * 4. Neither the name of the author nor the names of any co-contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY Bill Paul AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL Bill Paul OR THE VOICES IN HIS HEAD
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
 * THE POSSIBILITY OF SUCH DAMAGE.
 *
 *	From: Id: wicontrol.c,v 1.6 1999/05/22 16:12:49 wpaul Exp $
 */

/*
 * $Id: wicontrol.c,v 1.5 2002/08/11 14:18:51 iseki Exp $
 *
 * NOTE:
 *	current version of wistumbler will not reset the interface flags
 *	when it terminate. if it reset the flags, it will make a very
 *	race condition in some environment (ex. running dhclient).
 */

#include <sys/types.h>
#include <sys/cdefs.h>
#include <sys/param.h>
#include <sys/socket.h>
#include <sys/ioctl.h>
#include <sys/socket.h>

#include <net/if.h>
#ifdef __FreeBSD__
#include <net/if_var.h>
#include <net/ethernet.h>

#include <machine/if_wavelan_ieee.h>
#else
#include <netinet/in.h>
#include <netinet/if_ether.h>
#ifdef __NetBSD__
#include <net/if_ieee80211.h>
#include <dev/ic/wi_ieee.h>
#else
#include <dev/pcmcia/if_wavelan_ieee.h>
#endif
#endif

#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <stdlib.h>
#include <unistd.h>
#include <errno.h>
#include <err.h>

#include "gpscontrol.h"

#if !defined(lint)
__COPYRIGHT(
"@(#) Copyright (c) 1997, 1998, 1999\
	Bill Paul. All rights reserved.");
__RCSID("$NetBSD: wiconfig.c,v 1.25 2002/04/09 02:56:17 thorpej Exp $");
#endif

/* already define in wireg.h XXX */
#define	WI_APRATE_0		0x00	/* NONE */
#define WI_APRATE_1		0x0A	/* 1 Mbps */
#define WI_APRATE_2		0x14	/* 2 Mbps */
#define WI_APRATE_5		0x37	/* 5.5 Mbps */
#define WI_APRATE_11		0x6E	/* 11 Mbps */

int wi_apscan			__P((char *));
static int  get_if_flags	__P((int, const char *));
static int  set_if_flags	__P((int, const char *, int));
static int  wi_setval		__P((char *, struct wi_req *));

#define MAX_APS					1024
struct access_point {
	struct wi_apinfo	wi_apinfo;
	long			first;
	long			last;
	int			cur_receive;
	struct dir		dir;
};
struct access_point *access_points[MAX_APS];
static struct access_point *lookup_bssid	__P((struct wi_apinfo *));
void print_aps					__P((FILE *));
#if 0
int keep_flags;			/* interface's flags, UP/DOWN */
#endif

extern int gps;

static int get_if_flags(s, name)
	int		s;
	const char	*name;
{
	struct ifreq	ifreq;

	strncpy(ifreq.ifr_name, name, sizeof(ifreq.ifr_name));
	if (ioctl(s, SIOCGIFFLAGS, (caddr_t)&ifreq) == -1) {
		perror("SIOCGIFFLAGS");
		return -1;
	}

	return ifreq.ifr_flags;
}

static int set_if_flags(s, name, flags)
	int		s;
	const char	*name;
	int		flags;
{
	struct ifreq ifreq;

	ifreq.ifr_flags = flags;
	strncpy(ifreq.ifr_name, name, sizeof(ifreq.ifr_name));
	if (ioctl(s, SIOCSIFFLAGS, (caddr_t)&ifreq) == -1) {
		perror("SIOCSIFFLAGS");
		return -1;
	}

	return 0;
}

int wi_init(iface)
	char			*iface;
{
#if 0
	int		s;

	s = socket(AF_INET, SOCK_DGRAM, 0);
	if (s == -1) {
		perror("socket");
		return -1;
	}
	if ((keep_flags = get_if_flags(s, iface)) == -1) {
		close(s);
		return -1;
	if ((keep_flags & IFF_UP) == 0) {
		if (set_if_flags(s, iface, keep_flags | IFF_UP) == -1)
			close(s);
			return -1;
	}
	close(s);
#endif
	return 0;
}

int wi_finish(iface)
	char			*iface;
{
#if 0
	int		s;

	s = socket(AF_INET, SOCK_DGRAM, 0);
	if (s == -1) {
		perror("socket");
		return -1;
	}
	if (keep_flags != -1)
		set_if_flags(s, iface, keep_flags);
#endif
	return 0;
}

int wi_apscan(iface)
	char			*iface;
{
	struct wi_req		wreq;
	struct ifreq		ifr;
	int			s;
	int			naps;
	int			retries = 10;
	struct wi_apinfo	*w;
	int			i;
	int			fail;
	int			flags;

	s = socket(AF_INET, SOCK_DGRAM, 0);

	if (s == -1) {
		perror("socket");
		return -1;
	}

	flags = get_if_flags(s, iface);
	if (flags == -1)
		return -1;
	if ((flags & IFF_UP) == 0 &&
	    set_if_flags(s, iface, flags | IFF_UP) == -1)
		return -1;

	memset((char *)&wreq, 0, sizeof(wreq));

	wreq.wi_type = WI_RID_SCAN_APS;
	wreq.wi_len = 4;
	/* note chan. 1 is the least significant bit */
	wreq.wi_val[0] = 0x3fff;	/* 1 bit per channel, 1-14 */
	wreq.wi_val[1] = 0xf;		/* tx rate */

	/* write the request */
	if (wi_setval(iface, &wreq) < 0)
		return -1;

	/* now poll for a result */
	memset((char *)&wreq, 0, sizeof(wreq));

	wreq.wi_type = WI_RID_READ_APS;
	wreq.wi_len = WI_MAX_DATALEN;

	/* we have to do this ourself as opposed to
	 * using getval, because we cannot bail if
 	 * the ioctl fails
	 */
	memset((char *)&ifr, 0, sizeof(ifr));
        strcpy(ifr.ifr_name, iface);
        ifr.ifr_data = (caddr_t)&wreq;

	while ((fail = (ioctl(s, SIOCGWAVELAN, &ifr) == -1))) {
		if (errno != EINPROGRESS && --retries < 0)
			break;
		usleep(200000);		/* 200mS */
	}

	if (fail) {
		perror("SIOCGWAVELAN");
		if ((flags & IFF_UP) == 0)
		    (void) set_if_flags(s, iface, flags | IFF_UP);
		close(s);
		return -1;
	}

	naps = *(int *)wreq.wi_val;
	w =  (struct wi_apinfo *)(((char *)&wreq.wi_val) + sizeof(int));
	for ( i = 0; i < naps; i++, w++) {

		struct	access_point *ap;
		int	new_ap = 0;
		int	i;

		if ((ap = lookup_bssid(w)) == NULL) {
			/* this is a new entry */
			new_ap = 1;
			for (i = 0; i < MAX_APS; i++) {
				if (access_points[i] == NULL) {
					ap = malloc(sizeof(struct access_point));
					bzero((char *)ap, sizeof(ap));
					access_points[i] = ap;
					break;
				}
			}
                }
		if (!ap) {
			/* XXX should malloc instead of using array XXX */
			fprintf(stderr, "Table full\n");
			if ((flags & IFF_UP) == 0)
			    (void) set_if_flags(s, iface, flags | IFF_UP);
			close(s);
			return -1;
		}

		if (new_ap) {
			time(&ap->first);
			ap->wi_apinfo.namelen = w->namelen;
			strncpy(ap->wi_apinfo.name, w->name, w->namelen);

			for (i = 0; i < 6; i++)
				ap->wi_apinfo.bssid[i] = w->bssid[i];

			ap->wi_apinfo.channel = w->channel;
			ap->wi_apinfo.interval = w->interval;
			ap->wi_apinfo.capinfo = w->capinfo;
		}
		time(&ap->last);
		ap->wi_apinfo.quality = w->quality;
		ap->wi_apinfo.signal = w->signal;
		ap->wi_apinfo.noise = w->noise;
		ap->wi_apinfo.rate = w->rate;
		ap->cur_receive = 1;

		/* copy current position from GPS receiver */
		if (gps) {
			if (dir.Ndeg && dir.Edeg) {
				ap->dir.N = dir.N;
				ap->dir.Ndeg = dir.Ndeg;
				ap->dir.E = dir.E;
				ap->dir.Edeg = dir.Edeg;
			}
		}
	}

	if ((flags & IFF_UP) == 0)
	    (void) set_if_flags(s, iface, flags | IFF_UP);
	close(s);

	return 0;
}

static int wi_setval(iface, wreq)
	char			*iface;
	struct wi_req		*wreq;
{
	struct ifreq		ifr;
	int			s;

	bzero((char *)&ifr, sizeof(ifr));

	strcpy(ifr.ifr_name, iface);
	ifr.ifr_data = (caddr_t)wreq;

	s = socket(AF_INET, SOCK_DGRAM, 0);

	if (s == -1) {
		perror("socket");
		return -1;
	}

	if (ioctl(s, SIOCSWAVELAN, &ifr) == -1) {
		perror("SIOCSWAVELAN");
		return -1;
	}

	close(s);

	return 0;
}

static struct access_point *
lookup_bssid(ap)
struct wi_apinfo 	*ap;
{
	int i,j;

	for (i = 0; i < MAX_APS; i++) {
		if (access_points[i] == NULL)
			continue;
		for (j = 0; j < 6; j++) {
			if (access_points[i]->wi_apinfo.bssid[j] != ap->bssid[j])
				break;
		}
		if (j == 6)
			return(access_points[i]);
	}
	/* not found */
	return(NULL);
}

void
print_aps(file)
FILE *file;
{
	int			i, j;
	float			rate;
	struct tm		*tm;
	struct access_point	*w;

	fprintf(file, "%-23s\t%-17s\t%s\t%s\t%s\t%s\t%s\t%s\t%s\t%s\n",
		"SSID", "BSSID", "Channel", "Signal", "Interval", "Capinfo",
		"First", "Last", "Rate", "Position");
	for (i = 0; i < MAX_APS; i++) {
		if (access_points[i] == NULL)
			continue;
		w = access_points[i];

		for (j = 0; j < w->wi_apinfo.namelen; j++) {
			fprintf(file, "%c", w->wi_apinfo.name[j]);
		}
		while (j < 23)
			fputc(' ', file), j++;

		fprintf(file, "\t%02x:%02x:%02x:%02x:%02x:%02x",
			w->wi_apinfo.bssid[0]&0xff, w->wi_apinfo.bssid[1]&0xff,
			w->wi_apinfo.bssid[2]&0xff, w->wi_apinfo.bssid[3]&0xff,
			w->wi_apinfo.bssid[4]&0xff, w->wi_apinfo.bssid[5]&0xff);
		fprintf(file, "\t%d", w->wi_apinfo.channel);
		fprintf(file, "\t%d%%", w->wi_apinfo.signal);
		fprintf(file, "\t%dmS", w->wi_apinfo.interval); 

		if ((w->wi_apinfo.capinfo &
		    (IEEE80211_CAPINFO_ESS | IEEE80211_CAPINFO_PRIVACY)) ==
		    (IEEE80211_CAPINFO_ESS | IEEE80211_CAPINFO_PRIVACY))
			fprintf(file, "\t\tESS/WEP");
		else if (w->wi_apinfo.capinfo & IEEE80211_CAPINFO_ESS)
			fprintf(file, "\t\tESS");
		else if (w->wi_apinfo.capinfo & IEEE80211_CAPINFO_PRIVACY)
			fprintf(file, "\t\tWEP");
		else
			fprintf(file, "\t\tUnknown");

		tm = localtime(&w->first);
		fprintf(file, "\t%02.2d:%02.2d", tm->tm_hour, tm->tm_min);
		tm = localtime(&w->last);
		fprintf(file, "\t%02.2d:%02.2d", tm->tm_hour, tm->tm_min);

		switch (w->wi_apinfo.rate) {
		case WI_APRATE_1:
			rate = 1;
			break;
		case WI_APRATE_2:
			rate = 2;
			break;
		case WI_APRATE_5:
			rate = 5.5;
			break;
		case WI_APRATE_11:
			rate = 11;
			break;
		case WI_APRATE_0:
		default:
			rate = 0;
			break;
		}
		fprintf(file, "\t");
		if (rate != 0)
			fprintf(file, "%g", rate);
		if (gps)
			fprintf(file, "\t%c %f\t%c %f",
				w->dir.N, w->dir.Ndeg, w->dir.E, w->dir.Edeg);
		fprintf(file, "\n");
	}
}
