/*
 * Copyright (c) 2004
 *      iMil <imil@gcu.info>.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *      This product includes software developed by iMil.
 * 4. Neither the name of the author nor the names of any co-contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY iMil AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL iMil OR THE VOICES IN HIS HEAD
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
 * THE POSSIBILITY OF SUCH DAMAGE.
 *
 * $Id: wimon.c,v 1.15 2004/07/13 07:38:18 imil Exp $
 */

#include "common.h"
#include <signal.h>
#include <curses.h>
#include "showi.h"

#define BORDER 2
#define BOTTOM 2
#define TOP 2
#define DEFAULT_SCALE 100
#define DEFAULT_DELAY 40000

#define WATTRON(a, b) if (colors != 0) { \
	wattron(a, b); \
} 
#define WATTROFF(a, b) if (colors != 0) { \
	wattroff(a, b); \
} 

#define MAX(a,b) ((a)<(b)?(b):(a))
#define ABS(a) ((a)>0?(a):-(a))
#define INTERVAL (5)

static int *update_array(int, int *, int *);
static void update_blah(struct wi_infos);
static void draw_graph(int, int *, int);
static void print_int_items(WINDOW *, const char *, int);
static void print_str_items(WINDOW *, const char *, char *);
static void show_infos(struct wi_infos);
static void finish(int);
static void usage(char *);
static void init_windows(void);

int *s_array = NULL, *n_array = NULL, *q_array = NULL;
int gwin_h, gwin_w, colors;
int use_sound = 0, audiodev;
WINDOW *gwin, *main_win;
const char progname[] = "wimon";

static int *
update_array(int num, int *array, int *max)
{
	int i;
	*max = 0;

	if (array == NULL) {
		/* we malloc gwin_w elements */
		array = (int *)malloc(gwin_w * sizeof(int));
		for (i = 0; i < gwin_w; i++)
			array[i] = 0;
	} else
		for (i = gwin_w-1; i >=0; i--) {
			array[i] = array[i-1];
			if (*max < array[i])
				*max = array[i];
		}
		
	array[0] = num;
	return (array);
}

static void
update_blah(struct wi_infos wi)
{
	mvwprintw(main_win, gwin_h + TOP + 1, 2, "[ signal level:");
	WATTRON(main_win, COLOR_PAIR(1));
	wattron(main_win, A_BOLD);
	wprintw(main_win, " %d / %d ", wi.signal, wi.scale);
	wattroff(main_win, A_BOLD);
	WATTROFF(main_win, COLOR_PAIR(1));
	wprintw(main_win, "]  [ noise level:");
	WATTRON(main_win, COLOR_PAIR(2));
	wattron(main_win, A_BOLD);
	wprintw(main_win, " %d / %d ", wi.noise, wi.scale);
	wattroff(main_win, A_BOLD);
	WATTROFF(main_win, COLOR_PAIR(2));
	wprintw(main_win, "]  -  hit ^C to quit");
	wrefresh(main_win);
}

static void
draw_graph(int scale, int *selected_array, int color)
{
	int i, j, nbar, nscale = 0;
	int *array = selected_array;
        
	/* draw grid and scale */
	for (i = TOP + BOTTOM; i < gwin_h ; i += 2)
		nscale++;
	nscale = scale / nscale;
	for (i = 0, j = 0; i < ( gwin_h - BORDER ); i += 2, j++) {

		WATTRON(gwin, COLOR_PAIR(3));
		mvwhline(gwin, gwin_h -  i, BORDER + 2, '-', 4);
		WATTROFF(gwin, COLOR_PAIR(3));

		WATTRON(gwin, COLOR_PAIR(4));
		mvwprintw(gwin, gwin_h - i, BORDER, " %d ", j * nscale);
		WATTROFF(gwin, COLOR_PAIR(4));
	}
	
	/* draw graph */
	WATTRON(gwin, COLOR_PAIR(color));
	wattron(gwin, A_BOLD);
	for (i = 0; i <= gwin_w; i++) {
		nbar = (array[i] * gwin_h) / scale;
		mvwaddch(gwin, gwin_h - nbar, i + BORDER + 4, '_');
	}
	WATTROFF(gwin, COLOR_PAIR(color));
	wattroff(gwin, A_BOLD);
	wrefresh(gwin);
}

static void
print_int_items(WINDOW *cur_win, const char *title, int value)
{
	wprintw(cur_win, "%s:", title);
	wattron(cur_win, A_BOLD);
	wprintw(cur_win, " %d", value);
	wattroff(cur_win, A_BOLD);
}

static void
print_str_items(WINDOW *cur_win, const char *title, char *value)
{
	wprintw(cur_win, "%s:", title);
	wattron(cur_win, A_BOLD);
	wprintw(cur_win, " %s", value);
	wattroff(cur_win, A_BOLD);
}

static void
show_infos(struct wi_infos wi)
{
	int x, y, cur_w = COLS - (BORDER * 2) - 6; /* 6 == strlen("ssid: ") */

	wmove(main_win, 1, 2);

	wprintw(main_win, "%s v%s | ", progname, VERSION);
	print_int_items(main_win, "channel", wi.curchan);
	wprintw(main_win, " | ");
	print_int_items(main_win, "tx rate (Mbps)", wi.curtxrate);
	wprintw(main_win, " | ");
	/* if stlen(ssid) > screen width, reduce ssid to window width - diff */
	getyx(main_win, y, x);
	x = cur_w - (x + wi.curssidlen);
	if (x <= 0) {
		wi.curssid[wi.curssidlen + x] = '\0';
	}
	print_str_items(main_win, "ssid", wi.curssid);

	wmove(main_win, LINES, COLS);
	wrefresh(main_win);

}

static void
finish(int sig)
{
	delwin(main_win);
	delwin(gwin);
	endwin();

	free(s_array);
	free(n_array);
	free(q_array);

	if (use_sound != 0) {
		wi_signal = -1;
	}
	exit(0);
}

static void
init_windows()
{
	/* create a window 1 char smaller than term */
	main_win = newwin(LINES - 2, COLS - 2, 1, 1);
	WATTRON(main_win, COLOR_PAIR(4));
	box(main_win, 0 , 0);
	WATTROFF(main_win, COLOR_PAIR(4));

	gwin_h = LINES - BOTTOM - TOP - BORDER - 1;
	gwin_w = COLS - BORDER;
	/* draw separation lines */
	mvwhline(main_win, gwin_h + TOP, 1, 0, gwin_w - 2);
	mvwhline(main_win, TOP, 1, 0, gwin_w - 2);

	wrefresh(main_win);

	/* create the graph window */
	gwin = newwin(gwin_h, gwin_w, TOP, 0);

	wrefresh(gwin);
}

static void
usage(char *arg)
{
	fprintf(stderr, "%s -i <iface> [-s <scale>] [-d delay in microsec] [-w]\n\n", arg);
	fprintf(stderr, "-i <interface> : act on <interface>\n");
	fprintf(stderr, "-s <scale>     : scale graph to <scale>\n");
	fprintf(stderr, "-d <delay>     : refresh every <delay>\n");
	fprintf(stderr, "-w             : play sound when signal is caught\n");

	fprintf(stderr, "\n");
}

int
main(int argc, char *argv[])
{
	struct wi_infos wi;
	char *iface = NULL;
	int scale = DEFAULT_SCALE, delay = DEFAULT_DELAY;
	int new_scale, autoscale = 1;
	int s_max, n_max, q_max;
	extern int wi_signal;
	pthread_t p_thread;
	/* those are for getopt */
	extern char *optarg;
	int ch;
	

	while ((ch = getopt(argc, argv, "i:s:d:w")) != -1) {
		switch (ch) {
		case 'i':
			iface = optarg;
			break;
		case 's':
			scale = atoi(optarg);
			autoscale = 0;
			break;
		case 'd':
			delay = atoi(optarg);
			break;
		case 'w':
			use_sound = 1;
			break;
		case '?':
		default:
			usage(argv[0]);
			exit(1);
		}
	}

	if (iface == NULL) {
		usage(argv[0]);
		exit(1);
	}

	signal(SIGINT, finish);

	/* init ncurses */
	initscr();
	nonl();
	cbreak();
	noecho();

	if ((scale - LINES) < 0)
		scale = LINES;


	if (has_colors() == FALSE)
		colors = 0;
	else {
		colors = 1;
		start_color();

		init_pair(1, COLOR_GREEN, COLOR_BLACK);
		init_pair(2, COLOR_RED, COLOR_BLACK);
		init_pair(3, COLOR_BLUE, COLOR_BLACK);
		init_pair(4, COLOR_WHITE, COLOR_BLACK);
	}

	init_windows();

	wi.scale = scale;

	if (use_sound != 0)
		pthread_create(&p_thread, NULL, do_noise, NULL);

	while (1) {
		get_wi_infos(iface, &wi);
		
		s_array = update_array(wi.signal, s_array, &s_max);
		n_array = update_array(wi.noise, n_array, &n_max);
		q_array = update_array(wi.quality, q_array, &q_max);
		wclear(gwin);

		if (autoscale != 0) {
			new_scale = MAX(s_max, MAX(n_max, q_max)) + INTERVAL;
			if (ABS(new_scale - scale) >= INTERVAL) {
				scale = new_scale;
				if ((scale / LINES) < 1)
					scale = LINES;
				wi.scale = scale;
			}
		}

		/* draw signal */
		draw_graph(scale, s_array, 1);

		/* draw noise */
		draw_graph(scale, n_array, 2);

		/* draw quality */
		draw_graph(scale, q_array, 3);

		/* update signal / noise */
		update_blah(wi);

		/* update info window */
		show_infos(wi);

		if (use_sound != 0)
			wi_signal = wi.signal;
       
		usleep(delay);
	
	}

	finish(0);
}
