(define (pmac-functor names-mod error-mod)

  (module pmac-sig ()

    (use scheme-sig)
    (use names-mod names-sig)
    (use error-mod error-sig)

    ;; checking bindings for let/let*
    (define (check-let/let*-bindings bl complain)
      (define (check-one b)
	(if (not (and (= (length b) 2)
		      (name? (car b))))
	    (complain)))
      (for-each check-one bl))

    ;; the ``let'' macro
    (define (let-macro exp rename compare se)
      (define (complain)
	(se "bad let expression:" exp))

      (if (< (length exp) 3)
	  (complain))
      (let ((my-letrec (rename 'letrec))
	    (my-lambda (rename 'lambda))
	    (first (cadr exp))
	    (rest (cddr exp)))
	(if (name? first)		; named let?
	    (begin
	      (if (< (length exp) 4)
		  (complain))
	      (let ((n first)
		    (bl (car rest))
		    (body (cdr rest)))
		(check-let/let*-bindings bl complain)
		(let ((vl (map car bl))
		      (el (map cadr bl)))
		  `(,my-letrec
		    ((,n (,my-lambda ,vl ,@body)))
		    (,n ,@el)))))
	    (begin
	      (check-let/let*-bindings first complain)
	      (let* ((vl (map car first))
		     (el (map cadr first)))
		`((,my-lambda ,vl ,@rest) ,@el))))))

    ;; the  ``let*'' macro
    (define (let*-macro exp rename compare se)
      (define (complain)
	(se "bad let* expression:" exp))
      (if (< (length exp) 3)
	  (complain))
      (let ((vl (cadr exp))
	    (body (cddr exp))
	    (my-lambda (rename 'lambda)))

	(define (r l)
	  (if (pair? (cdr l))
	      `((,my-lambda (,(caar l)) ,(r (cdr l))) ,(cadar l))
	      `((,my-lambda (,(caar l)) ,@body) ,(cadar l))))

	(if (pair? vl)
	    (begin
	      (check-let/let*-bindings vl complain)
	      (r vl))
	    `((,my-lambda () ,@body)))))

    ;; the ``or'' macro:
    (define (or-macro exp rename compare se)
      (if (pair? (cdr exp))
	  (let ((my-let (rename 'let))
		(my-tmp (rename 'tmp))
		(my-if (rename 'if)))
	    (define (r l)
	      (if (pair? (cdr l))
		  `(,my-let ((,my-tmp ,(car l)))
			    (,my-if ,my-tmp ,my-tmp ,(r (cdr l))))
		  (car l)))
	    (r (cdr exp)))
	  '#f))

    ;; the ``and'' macro:
    (define (and-macro exp rename compare se)
      (if (pair? (cdr exp))
	  (let ((my-if (rename 'if)))
	    (define (r l)
	      (if (pair? (cdr l))
		  `(,my-if ,(car l) ,(r (cdr l)) #f)
		  (car l)))
	    (r (cdr exp)))
	  #t))

    ;; the ``cond'' macro:
    (define (cond-macro exp rename compare se)
      (define (complain)
	(se "bad cond expression:" exp))
      (let ((my-if (rename 'if))
	    (my-let (rename 'let))
	    (my-else (rename 'else))
	    (my-begin (rename 'begin))
	    (my-tmp (rename 'tmp))
	    (my-=> (rename '=>)))
	(define (recur clauses)
	  (if (not (pair? clauses))
	      (complain))
	  (let* ((current (car clauses))
		 (other-clauses (cdr clauses))
		 (more-clauses (pair? other-clauses)))
	    (if (not (pair? current))
		(complain))
	    (cond ((compare (car current) my-else)
		   (let ((seq (cdr current)))
		     (if (or more-clauses (not (pair? seq))) (complain))
		     (cons my-begin seq)))
		  ((not (pair? (cdr current)))
		   (if more-clauses
		       `(,my-let ((,my-tmp ,(car current)))
			   (,my-if ,my-tmp ,my-tmp
				   ,(recur other-clauses)))
		       `(,my-let ((,my-tmp ,(car current)))
			   (,my-if ,my-tmp ,my-tmp))))
		  ((and (= (length current) 3)
			(compare (cadr current) my-=>))
		   (if more-clauses
		       `(,my-let ((,my-tmp ,(car current)))
			   (,my-if ,my-tmp
				   (,(caddr current) ,my-tmp)
				   ,(recur other-clauses)))
		       `(,my-let ((,my-tmp ,(car current)))
			   (,my-if ,my-tmp
				   (,(caddr current) ,my-tmp)))))
		  (more-clauses
		   `(,my-if ,(car current)
			    (,my-begin ,@(cdr current))
			    ,(recur other-clauses)))
		  (else
		   `(,my-if ,(car current)
			    (,my-begin ,@(cdr current)))))))
	(recur (cdr exp))))

    ;; the ``quasiquote'' macro (*slightly* complicated :-):
    (define (quasiquote-macro exp rename compare se)
      (let ((my-quasiquote (rename 'quasiquote))
	    (my-unquote (rename 'unquote))
	    (my-unquote-splicing (rename 'unquote-splicing))
	    (op-list (list (cons 'cons (rename 'cons))
			   (cons 'append (rename 'append))
			   (cons 'vector (rename 'vector))
			   (cons 'list->vector (rename 'list->vector)))))

	;; translate quasiquote form into a more explicit notation first:
	;; The translation procedures communicate by returning annotated
	;; subexpressions.  The annotations look like:
	;;   (quote <x>)	- <x> remains constant
	;;   (eval <x>)		- <x> is a subexpression to be evaluated
	;;   (splice <x>)	- <x> is a subexpression to be evaluated and
	;;			  ``spliced'' into the enclosing construct
	;;   (cons <x> <y>)	- <x> <y> to be cons'ed together
	;;   (vector <x> ...)	- <x> ... go into a vector
	;;   (list->vector <x>)	- list <x> converted to vector
	;;   (append <x> <y>)	- <x> <y> append'ed together

	;; recognize relevant sub-structures:
	(define (qq? x)
	  (and (pair? x)
	       (compare (car x) my-quasiquote) (= (length x) 2)))
	(define (uq? x)
	  (and (pair? x)
	       (compare (car x) my-unquote) (= (length x) 2)))
	(define (uqs? x)
	  (and (pair? x)
	       (compare (car x) my-unquote-splicing) (= (length x) 2)))

	;; recurse into (non-expanding) substructure...
	(define (substruct2 x lev)
	  (let ((sub (construct-any (cadr x) lev)))
	    (case (car sub)
	      ((quote) (list 'quote x))
	      ((splice) (list 'cons (list 'quote (car x))
			      (cadr sub)))
	      (else (list 'cons
			  (list 'quote (car x))
			  (list 'cons
				sub
				''()))))))

	;; general case...
	(define (construct-any x lev)
	  (cond ((qq? x) (substruct2 x (+ lev 1)))
		((uq? x)
		 (if (zero? lev)
		     (list 'eval (cadr x))
		     (substruct2 x (- lev 1))))
		((uqs? x)
		 (if (zero? lev)
		     (list 'splice (cadr x))
		     (substruct2 x (- lev 1))))
		((pair? x)
		 (construct-pair x lev))
		((vector? x)
		 (construct-vector x lev))
		(else (list 'quote x))))

	;; quote a ``constant heading''...
	;; (the heading (a list) was collected in reverse order)
	(define (q-c-h x)
	  (list 'quote (reverse x)))

	;; treatment of vectors...
	(define (construct-vector v lev)
	  (let* ((state 'quote)
		 (subs (map (lambda (y)
			      (let ((r (construct-any y lev)))
				(case (car r)
				  ((quote) r)
				  ((splice) (set! state 'splice) r)
				  (else
				   (if (not (eq? state 'splice))
				       (set! state 'collect))
				   r))))
			    (vector->list v))))
	    (case state
	      ((quote) (list 'quote v))
	      ((collect) (cons 'vector subs))
	      (else
	       (list 'list->vector
		     (letrec
			 ((recur
			   (lambda (constant-heading rest)
			     (cond ((not (pair? rest))
				    (q-c-h constant-heading))
				   ((eq? (caar rest) 'quote)
				    (recur
				     (cons (cadar rest)
					   constant-heading)
				     (cdr rest)))
				   ((eq? (caar rest) 'splice)
				    (if (null? constant-heading)
					(list 'append
					      (list 'eval (cadar rest))
					      (recur '() (cdr rest)))
					(list
					 'append
					 (q-c-h constant-heading)
					 (list 'append
					       (list 'eval (cadar rest))
					       (recur '() (cdr rest))))))
				   ((null? constant-heading)
				    (list 'cons
					  (car rest)
					  (recur '() (cdr rest))))
				   (else
				    (list 'append
					  (q-c-h constant-heading)
					  (list 'cons
						(car rest)
						(recur '()
						       (cdr rest)))))))))
		       (recur '() subs)))))))

	;; treatment of pairs...
	(define (construct-pair x lev)
	  (letrec
	      ((recur
		(lambda (constant-heading rest)
		  (cond ((not (pair? rest))
			 (list 'quote
			       (letrec
				   ((loop (lambda (l r)
					    (if (pair? l)
						(loop (cdr l)
						      (cons (car l) r))
						r))))
				 (loop constant-heading rest))))
			((and (zero? lev) (uq? rest))
			 (if (pair? constant-heading)
			     (list 'append
				   (q-c-h constant-heading)
				   (list 'eval (cadr rest)))
			     (list 'eval (cadr rest))))
			(else
			 (let ((item (construct-any (car rest) lev)))
			   (case (car item)
			     ((quote)
			      (recur (cons (cadr item) constant-heading)
				     (cdr rest)))
			     ((splice)
			      (if (null? constant-heading)
				  (list 'append
					(list 'eval (cadr item))
					(recur '() (cdr rest)))
				  (list 'append
					(q-c-h constant-heading)
					(list 'append
					      (list 'eval (cadr item))
					      (recur '() (cdr rest))))))
			     (else
			      (if (null? constant-heading)
				  (list 'cons
					item
					(recur '() (cdr rest)))
				  (list
				   'append
				   (q-c-h constant-heading)
				   (list 'cons
					 item
					 (recur '()
						(cdr rest)))))))))))))
	    (recur '() x)))

	;; transcribe back into Scheme...
	(define (transcribe-back x)
	  (let ((cx (car x)))
	    (case  cx
	      ((splice)
	       (se "ill-placed unquote-splicing in:" exp))
	      ((eval) (cadr x))
	      ((quote) x)
	      (else
	       (cons (cond
		      ((assq cx op-list) => cdr)
		      (else
		       (display "CX: ") (write cx) (newline)
		       (display "OP-LIST: ") (write op-list) (newline)
		       (bug
			"something went wrong in quasiquote macro")))
		     (map transcribe-back (cdr x)))))))

	;; body of quasiquote-macro...
	(if (qq? exp)
	    (transcribe-back (construct-any (cadr exp) 0))
	    (se "bad quasiquote syntax in:" exp))))

    ;; the ``do'' macro:
    (define (do-macro exp rename compare se)
      (define (complain)
	(se "bad do expression:" exp))
      (if (< (length exp) 3)
	  (complain))
      (let* ((my-letrec (rename 'letrec))
	     (my-lambda (rename 'lambda))
	     (my-if (rename 'if))
	     (my-loop (rename 'loop))
	     (my-tmp (rename 'tmp))
	     (my-begin (rename 'begin))

	     (bl (cadr exp))
	     (fin (caddr exp))
	     (body (cdddr exp))

	     (bl (map (lambda (b)
			(let ((bb (case (length b)
				    ((2) (list (car b) (cadr b) (car b)))
				    ((3) b)
				    (else (complain)))))
			  (if (not (name? (car bb)))
			      (complain))
			  bb))
		      bl))
	     (vl (map car bl))
	     (il (map cadr bl))
	     (rl (map caddr bl))

	     (test (if (pair? fin)
		       (car fin)
		       (complain)))
	     (seq (cdr fin))
	     (inner-body `(,my-begin ,@body (,my-loop ,@rl)))
	     (lambda-body (if (pair? seq)
			      `(,my-if ,test
				       (,my-begin ,@seq)
				       ,inner-body)
			      `((,my-lambda (,my-tmp)
				   (,my-if ,my-tmp ,my-tmp ,inner-body))
				,test))))
	`(,my-letrec ((,my-loop (,my-lambda ,vl ,lambda-body)))
	    (,my-loop ,@il))))

    ;; the ``syntax-rules'' macro
    (define (syntax-rules-macro exp rename compare se)

      (if (< (length exp) 3)
	  (se "bad syntax-rules: " exp))

      (let ((ellipsis (rename '...))
	    (my-lambda (rename 'lambda))
	    (my-primitive-transformer (rename 'primitive-transformer))
	    (my-letrec (rename 'letrec))
	    (my-if (rename 'if))
	    (my-cond (rename 'cond))
	    (my-else (rename 'else))
	    (my-quasiquote (rename 'quasiquote))
	    (my-unquote (rename 'unquote))
	    (my-unquote-splicing (rename 'unquote-splicing))
	    (my-eqv? (rename 'eqv?))
	    (my-= (rename '=))
	    (my-null? (rename 'null?))
	    (my-pair? (rename 'pair?))
	    (my-car (rename 'car))
	    (my-cdr (rename 'cdr))
	    (my-apply (rename 'apply))
	    (my-map (rename 'map))
	    (my-cons (rename 'cons))
	    (my-append (rename 'append))
	    (my-length (rename 'length))
	    (my-exp (rename 'exp))
	    (my-fail (rename 'fail))
	    (my-succeed (rename 'succeed))
	    (my-cmp (rename 'cmp))
	    (my-se (rename 'se))
	    (my-recur (rename 'recur))
	    (my-onestep (rename 'onestep))
	    (my-l1 (rename 'l1))
	    (my-l2 (rename 'l2))

	    (literals (cadr exp))
	    (rules (cddr exp))
	    (used-literals '())
	    (var-mapping '()))

	(define (mem name l)
	  (cond ((not (pair? l)) #f)
		((compare (car l) name) #t)
		(else (mem name (cdr l)))))

	(define (lit? name)
	  (mem name literals))

	(define (rename! name)
	  (cond ((assoc name var-mapping) => cdr)
		(else
		 (let ((r (rename name)))
		   (set! var-mapping (cons (cons name r) var-mapping))
		   r))))

	(define brand-new-name
	  (let ((ctr 0))
	    (lambda (name)
	      (cond ((assoc name var-mapping) => cdr)
		    (else
		     (let* ((c ctr)
			    (ign (set! ctr (+ ctr 1)))
			    (s
			     (string->symbol
			      (string-append
			       "#@$%:" (number->string c)))))
		       (set! var-mapping (cons (cons name s) var-mapping))
		       s))))))

	(define (literal! name)
	  (let ((r (brand-new-name name)))
	    (if (not (member name used-literals))
		(set! used-literals (cons name used-literals)))
	    r))

	(define (datum? x)
	  (or (vector? x) (string? x) (char? x) (boolean? x) (number? x)
	      (null? x)))

	;; every (sub-)pattern gets translated into a little
	;; procedure which takes two explicit continuations:
	;;  1. a failure thunk
	;;  2. a procedure with n arguments, where n is the number of
	;;     pattern variables in the pattern.  Each of these arguments
	;;     receives the value (or the list (of lists (of lists...))) of
	;;     values for that variable.

	(define (build-simple-check item my-cmp k)
	  (k `(,my-lambda (,my-exp ,my-fail ,my-succeed)
		  (,my-if (,my-cmp ,my-exp ,item)
		      (,my-succeed)
		      (,my-fail)))
	     '()))

	(define (bad-ellipsis-pattern)
	  (se "illegal use of ellipsis in pattern: " exp))

	(define (build-recognizer pattern k)
	  (cond ((datum? pattern)
		 (build-simple-check `',pattern my-eqv? k))
		((name? pattern)
		 (cond
		  ((compare pattern ellipsis) (bad-ellipsis-pattern))
		  ((lit? pattern)
		   (build-simple-check (literal! pattern) my-cmp k))
		  (else
		   (let ((r (rename! pattern)))
		     (k `(,my-lambda (,my-exp ,my-fail ,my-succeed)
			    (,my-succeed ,my-exp))
			(list (list 0 pattern r)))))))
		((not (pair? pattern))
		 (se "bad pattern: " exp))
		((compare (car pattern) ellipsis) (bad-ellipsis-pattern))
		((and (pair? (cdr pattern))
		      (compare (cadr pattern) ellipsis))
		 (if (not (null? (cddr pattern))) (bad-ellipsis-pattern))
		 (build-recognizer
		  (car pattern)
		  (lambda (rexp vdesc)
		    (k `(,my-lambda (,my-exp ,my-fail ,my-succeed)
			   (,my-letrec
			    ((,my-onestep ,rexp)
			     (,my-recur
			      (,my-lambda (,my-exp ,my-succeed)
				(,my-cond
				 ((,my-null? ,my-exp)
				  (,my-succeed
				   ,@(map (lambda (x) ''()) vdesc)))
				 ((,my-pair? ,my-exp)
				  (,my-onestep
				   (,my-car ,my-exp)
				   ,my-fail
				   (,my-lambda ,my-l1
				      (,my-recur
				       (,my-cdr ,my-exp)
				       (,my-lambda ,my-l2
					  (,my-apply
					   ,my-succeed
					   (,my-map ,my-cons
						    ,my-l1 ,my-l2)))))))
				 (,my-else (,my-fail))))))
			    (,my-recur ,my-exp ,my-succeed)))
		       (map (lambda (d)
			      (cons (+ (car d) 1) (cdr d)))
			    vdesc)))))
		(else
		 (build-recognizer
		  (car pattern)
		  (lambda (rexp1 vdesc1)
		    (build-recognizer
		     (cdr pattern)
		     (lambda (rexp2 vdesc2)
		       (k `(,my-lambda (,my-exp ,my-fail ,my-succeed)
			      (,my-if
			       (,my-pair? ,my-exp)
			       (,rexp1
				(,my-car ,my-exp)
				,my-fail
				(,my-lambda ,my-l1
				   (,rexp2
				    (,my-cdr ,my-exp)
				    ,my-fail
				    (,my-lambda ,my-l2
				       (,my-apply
					,my-succeed
					(,my-append ,my-l1 ,my-l2))))))
			       (,my-fail)))
			  (desc-append vdesc1 vdesc2)))))))))

	(define (desc-append d1 d2)
	  (let ((n2 (map cadr d2)))
	    (for-each
	     (lambda (n)
	       (if (mem n n2)
		   (se "syntax-rules: pattern variable: " n
		       " not unique in: " exp)))
	     (map cadr d1)))
	  (append d1 d2))


	(define (build-constructor template vdesc)
	  ;; vdesc: ((<pattern variable> <lev> <renamed pattern var>) ...)

	  (define (merge vd1 vd2)
	    (cond ((not (pair? vd1)) vd2)
		  ((not (pair? vd2)) vd1)
		  ((eq? (car vd1) (car vd2))
		   (cons (car vd1) (merge (cdr vd1) (cdr vd2))))
		  ((memq (car vd1) (memq (car vd2) vdesc))
		   (cons (car vd2) (merge vd1 (cdr vd2))))
		  (else (cons (car vd1) (merge (cdr vd1) vd2)))))

	  (define (build tmpl level k)
	    (cond ((datum? tmpl) (k tmpl '()))
		  ((name? tmpl)
		   (if (compare tmpl ellipsis)
		       (se "bad ellipsis in template: " exp))
		   (cond ((assoc tmpl vdesc)
			  =>
			  (lambda (vd)
			    (if (not (= level (cadr vd)))
				(se "mismatching ellipses for: "
				    tmpl " in: " exp))
			    (k `(,my-unquote ,(caddr vd)) (list vd))))
			 (else
			  (k `(,my-unquote ,(literal! tmpl)) '()))))
		  ((not (pair? tmpl))
		   (se "bad template in: " exp))
		  ((compare (car tmpl) ellipsis)
		   (if (and (pair? (cdr tmpl))
			    (null? (cddr tmpl))
			    (compare (cadr tmpl) ellipsis))
		       (k (literal! (car tmpl)) '())
		       (se "bad use of ellipsis in template: " exp)))
		  ((and (pair? (cdr tmpl))
			(compare (cadr tmpl) ellipsis))
		   (build
		    (car tmpl) (+ level 1)
		    (lambda (b1 vdesc1)
		      (if (not (pair? vdesc1))
			  (se "unexpected ellipsis in template: " exp))
		      (let ((vl1 (map caddr vdesc1)))
			(build
			 (cddr tmpl) level
			 (lambda (b2 vdesc2)
			   (let* ((ok `(,my-map
					(,my-lambda ,vl1
					  (,my-quasiquote ,b1))
					,@vl1))
				  (insert
				   (if (pair? (cdr vdesc1))
				       `(,my-if
					 (,my-= ,@(map
						   (lambda (v)
						     `(,my-length ,v))
						   vl1))
					 ,ok
					 (,my-se "length mismatch: " ,my-exp))
				       ok)))
			     (k
			      `((,my-unquote-splicing ,insert) ,@b2)
			      (merge vdesc1 vdesc2)))))))))
		  (else
		   (build
		    (car tmpl) level
		    (lambda (b1 vdesc1)
		      (build
		       (cdr tmpl) level
		       (lambda (b2 vdesc2)
			 (k (cons b1 b2)
			    (merge vdesc1 vdesc2)))))))))

	  (build template 0
		 (lambda (b vl)
		   `(,my-lambda
		     ,(map caddr vdesc)
		     (,my-quasiquote ,b)))))

	(define (translate-rules rules)
	  (cond ((not (pair? rules))
		 `(,my-se "syntax error: " ,my-exp))
		((not (= (length (car rules)) 2))
		 (se "bad rule in: " exp))
		(else
		 (build-recognizer
		  (caar rules)
		  (lambda (r vdesc0)
		    `(,r ,my-exp
			 (,my-lambda () ,(translate-rules (cdr rules)))
			 ,(build-constructor
			   (cadar rules)
			   (map (lambda (vd0)
				  (list (cadr vd0) (car vd0) (caddr vd0)))
				vdesc0))))))))

	(let* ((transl (translate-rules rules))
	       (argl `(,my-exp ,my-cmp ,my-se
			       ,@(map
				  (lambda (l)
				    (cdr (assoc l var-mapping)))
				  used-literals))))
	  `(,my-primitive-transformer ,used-literals () ,argl ,transl))))))
