/*
 * Code.c -- Implementation of Scheme Bytecode
 *
 * (C) m.b (Matthias Blume); Apr 1992, HUB; Jan 1993 PU/CS
 *         Humboldt-University of Berlin
 *         Princeton University, Dept. of Computer Science
 *
 * $Id: Code.c,v 2.43 1994/11/12 21:51:44 blume Exp $
 */

# include "rcs.h"
RCSID ("$Id: Code.c,v 2.43 1994/11/12 21:51:44 blume Exp $")

# include <stdio.h>
# include <string.h>
# include <stdlib.h>
# include <assert.h>
# include <setjmp.h>
# include <signal.h>

# include "storext.h"
# include "Code.h"
# include "Cont.h"
# include "identifier.h"
# include "type.h"
# include "except.h"
# include "speccont.h"
# include "mode.h"
# include "keyword.h"
# include "Cons.h"
# include "Procedure.h"
# include "Promise.h"
# include "Vector.h"
# include "Boolean.h"
# include "Cell.h"
# include "Symbol.h"
# include "Primitive.h"
# include "String.h"
# include "Numeric.h"
# include "Module.h"

# include "realloc.h"

/*
 * This is a description of the VSCM Bytecode Instruction Set (VBIS):
 *
 *  ! All ``JMP'' instructions except ``I_JMP_BW'' jump *forward*
 *
 * The instructions:
 *
 * I_JMP_FW	<distance>			-- unconditional
 * I_JMP_BW	<distance>
 *
 * I_TJMP	<distance>			-- no POP
 * I_FJMP	<distance>
 * I_TJMP_P	<distance>			-- POP if branch is taken
 * I_FJMP_P	<distance>
 * I_TJMP_EP	<distance>			-- POP if branch is not taken
 * I_FJMP_EP	<distance>
 * I_P_TJMP	<distance>			-- POP always
 * I_P_FJMP	<distance>
 *
 * I_EQJMP	<const. index> <distance>	-- eq?jump
 * I_NEQJMP	<const. index> <distance>	-- neq?jump
 * I_EQJMP_P	<const. index> <distance>	-- eq?jump+pop
 * I_NEQJMP_P	<const. index> <distance>	-- neq?jump+pop
 * I_EQJMP_EP	<const. index> <distance>	-- eq?jump:pop
 * I_NEQJMP_EP	<const. index> <distance>	-- neq?jump:pop
 * I_P_EQJMP	<const. index> <distance>	-- pop-eq?jump
 * I_P_NEQJMP	<const. index> <distance>	-- pop-neq?jump
 * I_EVJMP	<const. index> <distance>	-- eqv?jump
 * I_NEVJMP	<const. index> <distance>	-- neqv?jump
 * I_EVJMP_P	<const. index> <distance>	-- eqv?jump+pop
 * I_NEVJMP_P	<const. index> <distance>	-- neqv?jump+pop
 * I_EVJMP_EP	<const. index> <distance>	-- eqv?jump:pop
 * I_NEVJMP_EP	<const. index> <distance>	-- neqv?jump:pop
 * I_P_EVJMP	<const. index> <distance>	-- pop-eqv?jump
 * I_P_NEVJMP	<const. index> <distance>	-- pop-neqv?jump
 * I_MVJMP	<const. index> <distance>	-- memv?jump
 * I_NMVJMP	<const. index> <distance>	-- nmemv?jump
 * I_MVJMP_P	<const. index> <distance>	-- memv?jump+pop
 * I_NMVJMP_P	<const. index> <distance>	-- nmemv?jump+pop
 * I_MVJMP_EP	<const. index> <distance>	-- memv?jump:pop
 * I_NMVJMP_EP	<const. index> <distance>	-- nmemv?jump:pop
 * I_P_MVJMP	<const. index> <distance>	-- pop-memv?jump
 * I_P_NMVJMP	<const. index> <distance>	-- pop-nmemv?jump
 *
 * I_GET_L	<stack index>			-- ``plain'' access
 * I_GET_V	<stack index> <vector index>
 * I_GET_G	<const. index>
 *
 * I_PUT_L	<stack index>
 * I_PUT_L_P	<stack index>			-- with POP
 * I_PUT_V	<stack index> <vector index>
 * I_PUT_V_P	<stack index> <vector index>	-- with POP
 * I_PUT_G	<const. index>
 * I_PUT_G_P	<const. index>			-- with POP
 *
 * I_TAKE	<const. index>
 * I_TK_T					-- #t
 * I_TK_F					-- #f
 * I_TK_N					-- ()
 * I_TK_P	<prim. no>
 *
 * I_POP
 * I_MPOP	<count>				-- <count> times I_POP
 *
 * I_LAMBDA	<const. index>		-- environment taken from top of stack
 * I_DELAY	<const. index>		-- likewise
 *
 * I_CALL	<argcnt>
 * I_CALLE	<argcnt>
 * I_EXIT
 *
 * I_CONS
 * I_APPEND
 * I_LTV					-- list->vector
 *
 * I_CAR
 * I_CDR
 * I_NOT
 *
 * I_CHECK
 *
 * new stuff:
 * I_GET_LV	<stack index>			-- get-loc-void
 * I_VOID       <stack index>			-- kill stack item
 * I_REF
 * I_DEREF
 * I_ASSIGN
 * I_MCL	<count> <argcnt>		-- make closure
 * I_VECTOR	<argcnt>			-- build a vector
 * I_GET_VV	<stack index> <vector index>	-- get-vec-void
 * I_MODULE	<argcnt>			-- module
 * I_VALIDATE
 * I_FETCH	[0|1|2] <const. index>	-- fetch-{constant,read-only,variable}
 * I_VREF	<index>			-- vector-ref with constant index
 * I_VSET	<index>			-- vector-set with constant index
 */

/*
 * Define OP-Codes for the Scheme Bytecode Statements:
 */

# define INSTR(oc,a,n) oc,
enum opcode {
# include "vbis.sort"
  N_INSTRUCTIONS		/* aka TERMINATOR_1 (see below :) */
};
# undef INSTR

/*
 * Define names for instruction lengths (the name for the length of
 * instruction FOO will be FOO_len):
 */

# define INSTR(oc,a,n) oc ## _len = a,
enum {
# include "vbis.sort"
  TERMINATOR_2			/* to get rid of the trailing comma */
};
# undef INSTR

/*
 * Table of opcodes and associated names... (alphabetically sorted)
 */

# define INSTR(oc,a,n) { oc, a, n },
static
struct stat_desc {
  unsigned short opcode;
  int length;
  const char *name;
} stat_desc [N_INSTRUCTIONS] = {
# include "vbis.sort"
};
# undef INSTR

# ifdef VM_INSTRUCTION_COUNTING
static unsigned long vm_counts [N_INSTRUCTIONS];
static unsigned long vm_pair_counts [N_INSTRUCTIONS][N_INSTRUCTIONS];
static int VICo;
# define COUNT(x) (vm_counts [x]++, vm_pair_counts [VICo][x]++, VICo = (x))
static void vm_statistics (void)
{
  int i, j;

  for (i = 0; i < N_INSTRUCTIONS; i++)
    fprintf (stderr, "\t*\t%10lu\t%s\n", vm_counts [i], stat_desc [i].name);
  putc ('\n', stderr);
  for (i = 0; i < N_INSTRUCTIONS; i++)
    for (j = 0; j < N_INSTRUCTIONS; j++)
      fprintf (stderr, "\t+\t%10lu\t%s, %s\n",
	       vm_pair_counts [i][j],
	       stat_desc [i].name,
	       stat_desc [j].name);
}
# else
# define COUNT(x) ((void)0)
# endif

static
struct stat_desc
  *find_stat (const char *name, unsigned len, void *opsym)
{
  int start = 0;
  int stop  = N_INSTRUCTIONS - 1;
  int m, cmp, llen;

  while (start <= stop) {
    m = (start + stop) / 2;
    cmp = strncmp (name, stat_desc [m].name, len);
    if (cmp == 0) {
      llen = strlen (stat_desc [m].name);
      if (llen == len)
	return stat_desc + m;
      else
	cmp = -1;
    }
    if (cmp < 0)
      stop = m - 1;
    else
      start = m + 1;
  }
  error ("vscm-asm: bad operation code: %w", opsym);
  /*NOTREACHED*/
}

/*
 * And now the normal stuff needed by type management:
 */

static MEM_cnt measure (void *vcode)
{
  ScmCode *code = vcode;
  return MEM_UNITS (sizeof (ScmCode) +
		    (code->length - 1) * sizeof (unsigned short));
}

static void iterator (void *vcode, MEM_visitor proc, void *cd)
{
  ScmCode *code = vcode;

  (* proc) ((void *)&code->argument_names, cd);
  (* proc) ((void *)&code->constants, cd);
  (* proc) ((void *)&code->proc_name, cd);
}

static void dumper (void *vcode, FILE *file)
{
  ScmCode *code = (ScmCode *) vcode;
  unsigned i;

  MEM_dump_ul (code->arg_cnt, file);
  if (code->take_rest)
    putc ('y', file);
  else
    putc ('n', file);
  MEM_dump_ul (code->stack_requirement, file);
  MEM_dump_ul (code->length, file);
  i = 0;
  while(i < code->length) {
    MEM_dump_ul (code->array [i], file);
    i++;
  }
}

static void *excavator (FILE *file)
{
  ScmCode *code;
  unsigned i;
  unsigned length, arg_cnt, stack_requirement;
  int c;

  arg_cnt = MEM_restore_ul (file);
  c = getc (file);
  if (c == EOF)
    fatal ("bad dump file format (Code)");
  stack_requirement = MEM_restore_ul (file);
  length = MEM_restore_ul (file);
  SCM_VNEW (code, Code, length, unsigned short);
  code->arg_cnt = arg_cnt;
  code->stack_requirement = stack_requirement;
  code->length = length;
  code->take_rest = (c == 'y');
  i = 0;
  while(i < length) {
    code->array [i] = MEM_restore_ul (file);
    i++;
  }
  return code;
}

extern void ScmDisplayCode (void *vcode, putc_proc pp, void *cd);
void ScmDisplayCode (void *vcode, putc_proc pp, void *cd)
{
  char buf [32];
  ScmCode *code = vcode;
  display_object (code->proc_name, pp, cd);
  sprintf (buf, " %u%s", (unsigned) code->arg_cnt, code->take_rest ? "+" : "");
  putc_string (buf, pp, cd);
}

static void display (void *vcode, putc_proc pp, void *cd)
{
  char buf [32];
  sprintf (buf, "#<Code %p ", vcode);
  putc_string (buf, pp, cd);
  ScmDisplayCode (vcode, pp, cd);
  (* pp) ('>', cd);
}

static void *reverse_old = NULL;
static void *names_save = NULL;
static void *append_save = NULL;
static void *constants_save = NULL;
static void *cont_save = NULL;
static ScmProcedure *proc_save = NULL;

# define MAX_INTERRUPTS 32		/* this should be enough */
static unsigned pending_interrupts = 0;
static struct {
  void *vect;
  unsigned short cont;
} int_table [MAX_INTERRUPTS];

static void int_table_iterator (void *tab, MEM_visitor proc, void *cd)
{
  int i;

  for (i = 0; i < MAX_INTERRUPTS; i++)
    (* proc) ((void *)&int_table[i].vect, cd);
}

static void deallocate_all (void);

static void module_init (void)
{
  atexit (deallocate_all);
  MEM_root_var (reverse_old);
  MEM_root_var (names_save);
  MEM_root_var (append_save);
  MEM_root_var (constants_save);
  MEM_root_var (cont_save);
  MEM_root_var (proc_save);
  MEM_root (int_table, int_table_iterator);
}

MEM_VECTOR (Code,
	    0, measure,
	    iterator, dumper, excavator, MEM_NULL_revisor,
	    module_init, MEM_NULL_task, MEM_NULL_task,
	    EXT (SCM_NO_NUMBER,
		 cannot_cvt_real, display, display, NULL_eq, NULL_eq));
/*
 * some useful functions
 */

static int memv (void *item, void *list)
{
  while (ScmTypeOf (list) == ScmType (Cons))
    if (eqv_object (((ScmCons *) list)->car, item))
      return 1;
    else
      list = ((ScmCons *) list)->cdr;
  return 0;
}

unsigned long ScmListLength (void *l)
{
  unsigned long i;

  i = 0;
  while (ScmTypeOf (l) == ScmType (Cons))
    l = ((ScmCons *) l)->cdr,
    i++;
  return i;
}

void *ScmReverseList (void *l)
{
  ScmCons *cons;
  void *r;
  unsigned long len, i;

  len = ScmListLength (l);
  if (len == 0)
    return &ScmNil;
  reverse_old = l;
  SCM_ALLOC (cons, len * sizeof (ScmCons));
  r = reverse_old;
  reverse_old = NULL;
  for (i = len; i-- > 0; ) {
    cons [i]._ = ScmType (Cons);
    cons [i].car = ((ScmCons *) r)->car;
    r = ((ScmCons *) r)->cdr;
    cons [i].cdr = cons + i + 1;
  }
  cons [len - 1].cdr = &ScmNil;
  return cons;
}

void *ScmReverseIP2 (void *l, void *r)
{
  ScmCons *tmp;

  while (ScmTypeOf (l) == ScmType (Cons)) {
    tmp = l;
    l = tmp->cdr;
    tmp->cdr = r;
    r = tmp;
  }
  return r;
}

static
void *append_two_lists (void *l1, void *l2)
{
  ScmCons *cons;
  unsigned long len, i;
  void *r;

  len = ScmListLength (l1);
  if (len == 0)
    return l2;
  reverse_old = l1;
  append_save = l2;
  SCM_ALLOC (cons, len * sizeof (ScmCons));
  l2 = append_save;
  r = reverse_old;
  append_save = reverse_old = NULL;
  for (i = 0; i < len; i++) {
    cons [i]._ = ScmType (Cons);
    cons [i].car = ((ScmCons *) r)->car;
    r = ((ScmCons *) r)->cdr;
    cons [i].cdr = cons + i + 1;
  }
  cons [len - 1].cdr = l2;
  return cons;
}

void ScmListToVector (void)
{
  ScmCons *l;
  unsigned long len, i;
  ScmVector *vect;

  len = ScmListLength (PEEK ());
  SCM_NEW_VECTOR (vect, len);
  l = PEEK ();
  for (i = 0; i < len; i++) {
    vect->array [i] = l->car;
    l = l->cdr;
  }
  SET_TOP (vect);
}

/*
 * Allocate a new continuation frame.  Push procedure environment
 * onto stack followed by the arguments.  The last argument will be
 * the topmost element.
 */
void ScmPrepareProcedureCall (void  *vproc, unsigned argcnt)
{
  register unsigned i, siz;
  ScmCons *cons;
  register void **frombase, **tobase;

  proc_save = vproc;
  ScmPushContinuation (((ScmCode *) proc_save->code)->stack_requirement);
  ScmCC->u.s.constants = (ScmCC->u.s.code = proc_save->code)->constants;

  /* we waste one stack entry here in case that the environment has */
  /* nothing in it -- this is probably not a big problem but        */
  /* simplifies the compiler...                                     */
  PUSH (proc_save->env);

  if (argcnt < (siz = ScmCC->u.s.code->arg_cnt))
    error ("too few arguments to procedure %w", proc_save);

  /*
   * The following conceptual code is expanded below:
   *
   * for (i = 0; i < siz; i++)
   *   PUSH (CPOP (ScmCC->father));
   */
  frombase = &CPOS (ScmCC->father, ScmCC->father->top);
  tobase = &POS (1);		/* ScmCC->top == 1 */
  for (i = 0; i < siz; i++)
    *tobase++ = *--frombase;
  ScmCC->top += siz;
  ScmCC->father->top -= siz;
    
  if (ScmCC->u.s.code->take_rest)
    if (siz == argcnt)
      PUSH (&ScmNil);
    else {
      SCM_ALLOC (cons, (argcnt - siz) * sizeof (ScmCons));
      for (i = siz; i < argcnt; i++) {
	cons [i - siz]._ = ScmType (Cons);
	cons [i - siz].car = CPOP (ScmCC->father);
	cons [i - siz].cdr = cons + i - siz + 1;
      }
      cons [argcnt - siz - 1].cdr = &ScmNil;
      PUSH (cons);
    }
  else
    if (argcnt > siz)
      error ("too many arguments to procedure %w", proc_save);

  proc_save = NULL;
}

static unsigned active_primitive = SCM_VM_TRAP_CONT;

static volatile sig_atomic_t some_interrupt_pending = 0;

void ScmRegisterInterrupt (unsigned cont, void *vvect)
{
  some_interrupt_pending = 1;
  if (pending_interrupts >= MAX_INTERRUPTS)
    reset ("too many pending interrupts");
  int_table [pending_interrupts].vect = vvect;
  int_table [pending_interrupts].cont = cont;
  pending_interrupts++;
}

static void handle_interrupts (void)
{
  ScmVector *vect;
  void *tmp;
  unsigned i, j;

  i = pending_interrupts;
  pending_interrupts = 0;
  while (i-- > 0) {
    active_primitive = int_table [i].cont;
    vect = int_table [i].vect;
    ScmPushPrimitiveContinuation (vect, vect->length - 1);
    active_primitive = SCM_VM_TRAP_CONT;
    vect = ScmCC->u.c.environ;
    for (j = vect->length - 1; j > 0; j--) {
      tmp = vect->array [j];
      Push (tmp);
      vect = ScmCC->u.c.environ;
    }
    int_table [i].vect = NULL;
    ScmPrepareProcedureCall (vect->array [0], vect->length - 1);
  }
}

static volatile sig_atomic_t asyn_interrupt = 0;
static volatile sig_atomic_t instant_interrupt_handling = 0;
static int memorized_instant_interrupt_handling;
static jmp_buf reactivation_point;

void ScmInstantInterruptHandling (int state)
{
  instant_interrupt_handling = state;
}

void MEM_announce_gc_start (void)
{
  memorized_instant_interrupt_handling = instant_interrupt_handling;
  instant_interrupt_handling = 0;
}

void MEM_announce_gc_end (void)
{
  instant_interrupt_handling = memorized_instant_interrupt_handling;
}

void ScmRegisterAsynInterrupt (void)
{
  some_interrupt_pending = 1;
  asyn_interrupt = 1;
  if (instant_interrupt_handling) {
    instant_interrupt_handling = 0;
    Push (&ScmEof);		/* this is not completely safe... */
    longjmp (reactivation_point, 1);
  }
}

static void handle_asyn_interrupt (void)
{
  void *intmode = ScmMode (SCM_INTERRUPT_MODE);

  asyn_interrupt = 0;
  if (intmode == NULL)
    reset ("Interrupt");
  else {
    active_primitive = SCM_VM_INTERRUPT_CONT;
    ScmPushPrimitiveContinuation (intmode, 0);
    active_primitive = SCM_VM_TRAP_CONT;
    ScmPrepareProcedureCall (ScmCC->u.c.environ, 0);
  }
}

# ifdef TICKING
static int timer_interrupt = 0;

static void handle_timer_interrupt (void)
{
  ScmVector *vect = NewScmVector (1);
  void *te_mode = ScmMode (SCM_TIMER_EXPIRATION_MODE);

  timer_interrupt = 0;

  if (te_mode == NULL)
    reset ("no handler for expired timer");

  vect->array [0] = te_mode;
  ScmRegisterInterrupt (SCM_VM_INTERRUPT_CONT, vect);
}

# endif

static void handle_all_interrupts (void)
{
  some_interrupt_pending = 0;
  if (asyn_interrupt)
    handle_asyn_interrupt ();
  if (pending_interrupts > 0)
    handle_interrupts ();
# ifdef TICKING
  if (timer_interrupt)
    handle_timer_interrupt ();
# endif
}

/* ticking */
# ifdef TICKING
static long ticks_left = 0;

# define TICK() ((void) (ticks_left && \
			 --ticks_left == 0 && \
			 (timer_interrupt = some_interrupt_pending = 1)))

# else

# define TICK() ((void)0)

# endif

long ScmTimer (long units)
{
# ifdef TICKING
  long old = ticks_left;

  if (units >= 0)
    ticks_left = units;
  return old;
# else
  return 0;
# endif
}

/*
 * The VSCM virtual machine:
 */

void ScmVM (void)
{
  register const unsigned short *cp;
  unsigned int cp_save;
  unsigned index, argcnt;
  void *tmp;
  ScmCell *cell;
  ScmVector *vector;
  ScmSymbol *symbol;
  ScmProcedure *proc;
  ScmPromise *prom;
  ScmCons *cons;
  ScmPrimitive *prim;

# ifdef VM_INSTRUCTION_COUNTING
  atexit (vm_statistics);
# endif

  setjmp (reactivation_point);

  if (some_interrupt_pending)
    handle_all_interrupts ();

  if (ScmTypeOf (ScmCC) == ScmType (CCont))
    goto c_cont_loop;

  cp = ScmCC->u.s.code->array + ScmCC->u.s.pc;

  /* loop forever */
  for (;;) {

# ifdef TRACE_VM_EXECUTION
    {
      int ii;
      for (ii = 1; ii < ScmCC->top; ii++)
	warning (" * %w", POS (ii));
      warning ("Executing: %s, stack->top = %u",
	       stat_desc [cp [0]].name, (unsigned) ScmCC->top);
    }
# endif


    COUNT (cp [0]);

    switch (cp [0]) {

      /*
       * Various CALL instructions...
       */

    case I_CALL:		/* call */
      argcnt = cp [1];
      cp += I_CALL_len;
      ScmCC->u.s.pc = cp - ScmCC->u.s.code->array;
call_entry_point:
      tmp = POP ();
      if (ScmTypeOf (tmp) == ScmType (Procedure)) {
	ScmPrepareProcedureCall (tmp, argcnt);
	cp = ScmCC->u.s.code->array;
      } else if (ScmTypeOf (tmp) == ScmType (Primitive)) {
	prim = tmp;
	if (prim->expected_argcnt >= 0 && prim->expected_argcnt != argcnt)
	  error ("wrong argcnt to primitive procedure %w", prim);
	active_primitive = prim->seq_num;
	argcnt = (* prim->code) (argcnt);
	active_primitive = SCM_VM_TRAP_CONT;
	if (argcnt > 0) {
	  argcnt--;
	  goto call_entry_point;
	}
	goto c_cont_loop;
      } else if (ScmTypeOf (tmp) == ScmType (Cont) ||
		 ScmTypeOf (tmp) == ScmType (CCont)) {
	ScmDirtyModeCache (-1);
	ScmSetContinuation (tmp, argcnt);
	cp = ScmCC->u.s.code->array + ScmCC->u.s.pc;
	/*
	 * allow for interrupts in strange call/cc loops as in:
	 * ((call/cc call/cc) (call/cc call/cc))
	 */
	TICK ();
	if (some_interrupt_pending) {
	  ScmCC->u.s.pc = cp - ScmCC->u.s.code->array;
	  handle_all_interrupts ();
	  cp = ScmCC->u.s.code->array + ScmCC->u.s.pc;
	}
	goto c_cont_loop;
      } else
	error ("VM: non-procedure called: %w", tmp);
      break;

    case I_CALLE:		/* call-exit */
      {
	ScmCode *code;
	unsigned i, s, t, sr;
	
	argcnt = cp [1];
	tmp = PEEK ();
	if (ScmTypeOf (tmp) != ScmType (Procedure)) {
	  ScmRevertToFatherContinuation (argcnt + 1);
	  goto call_entry_point;
	}
	proc = tmp;
	code = proc->code;
	s = code->arg_cnt;
	sr = code->stack_requirement;

	if (code->take_rest) {
	  if (argcnt < s)
	    ScmCC->u.s.pc = cp - ScmCC->u.s.code->array,
	    error ("too few arguments to procedure %w", proc);
	  if (argcnt == s)
	    append_save = &ScmNil;
	  else {
	    SCM_ALLOC (cons, (argcnt - s) * sizeof (ScmCons));
	    t = ScmCC->top - s - 2;
	    for (i = argcnt - s; i-- > 0; ) {
	      cons [i]._ = ScmType (Cons);
	      cons [i].cdr = cons + i + 1;
	      cons [i].car = POS (t - i);
	    }
	    cons [argcnt - s - 1].cdr = &ScmNil;
	    append_save = cons;
	  }
	} else if (argcnt != s)
	  ScmCC->u.s.pc = cp - ScmCC->u.s.code->array,
	  error ("wrong number of arguments to procedure %w", proc);

	if (ScmCC->length < sr) {
	  ScmCont *oldc, *tmpc;

	  tmpc = ScmNewSCont (ScmCC->father, sr);

	  oldc = ScmCC;
	  ScmCC = tmpc;
	  
	  proc = CPOP (oldc);

	  t = oldc->top;
	  i = s;
	  while (i > 0) {
	    POS (i) = CPOS (oldc, t - i);
	    i--;
	  }

	  ScmDisposeCont (oldc);
	} else {
	  int overlap;
	  unsigned rbase_idx, rtop_idx;

	  proc = POP ();
	  t = ScmCC->top;

	  overlap = 2 * (int) s + 1 - (int) t;
	  if (overlap > 0) {

	    if (s == t) {	/* degenerate case */
	      if (s == 0)	/* even more so */
		goto done_argument_shuffling;
	      else {
		PUSH (POS (0));
		rbase_idx = 1;
		rtop_idx = t - 1;
		i = 0;
	      }
	    } else {
	      i = s - overlap;
	      rbase_idx = i + 1;
	      rtop_idx = rbase_idx + overlap - 1;
	    }

	    while (rbase_idx < rtop_idx) {
	      tmp = POS (rbase_idx);
	      POS (rbase_idx) = POS (rtop_idx);
	      POS (rtop_idx) = tmp;
	      rtop_idx--;
	      rbase_idx++;
	    }
	  } else
	    i = s;

	  while (i > 0) {
	    POS (i) = POS (t - i);
	    i--;
	  }
	}

      done_argument_shuffling:
	POS (0) = proc->env;
	ScmCC->top = s + 1;
	ScmCC->u.s.code = code = proc->code;
	ScmCC->u.s.constants = code->constants;
	cp = code->array;
	if (code->take_rest) {
	  PUSH (append_save);
	  append_save = NULL;
	}
      }
      break;

      /*
       * Returning control to the caller...
       */

    case I_EXIT:
      /* incrementing cp is not necessary */
      (* ScmCC->father->result) ();
c_cont_loop:
      while (ScmTypeOf (ScmCC) == ScmType (CCont)) {
	prim = GetScmPrimitive (active_primitive = ScmCC->u.c.prim_no);
	argcnt = (* prim->cont) ();
	active_primitive = SCM_VM_TRAP_CONT;
	if (argcnt > 0) {
	  argcnt--;
	  goto call_entry_point;
	}
      }
      cp = ScmCC->u.s.code->array + ScmCC->u.s.pc;
      break;

      /*
       * Timer and interrupts...
       */

    case I_CHECK:		/* check */
      cp += I_CHECK_len;
      TICK ();
      if (some_interrupt_pending) {
	ScmCC->u.s.pc = cp - ScmCC->u.s.code->array;
	handle_all_interrupts ();
	cp = ScmCC->u.s.code->array + ScmCC->u.s.pc;
      }
      break;

      /*
       * Unconditional jumps...
       */

    case I_JMP_FW:		/* jump-forward */
      cp += I_JMP_FW_len + cp [1];
      break;

    case I_JMP_BW:		/* jump-backward */
      index = cp [1];
      cp -= index - I_JMP_BW_len; /* NOT: cp += I_JMP_BW_len - index !! */
      break;

      /*
       * Conditional jumps...
       */

    case I_TJMP:		/* true?jump */
      if (PEEK () == &ScmFalse)
	cp += I_TJMP_len;
      else
	cp += I_TJMP_len + cp [1];
      break;

    case I_FJMP:		/* false?jump */
      if (PEEK () != &ScmFalse)
	cp += I_FJMP_len;
      else
	cp += I_FJMP_len + cp [1];
      break;

    case I_TJMP_P:		/* true?jump+pop */
      if (PEEK () == &ScmFalse)
	cp += I_TJMP_P_len;
      else {
	(void) POP ();
	cp += I_TJMP_P_len + cp [1];
      }
      break;

    case I_FJMP_P:		/* false?jump+pop */
      if (PEEK () != &ScmFalse)
	cp += I_FJMP_P_len;
      else {
	(void) POP ();
	cp += I_FJMP_P_len + cp [1];
      }
      break;

    case I_TJMP_EP:		/* true?jump:pop */
      if (PEEK () == &ScmFalse) {
	(void) POP ();
	cp += I_TJMP_EP_len;
      } else
	cp += I_TJMP_EP_len + cp [1];
      break;

    case I_FJMP_EP:		/* false?jump:pop */
      if (PEEK () != &ScmFalse) {
	(void) POP ();
	cp += I_FJMP_EP_len;
      } else
	cp += I_FJMP_EP_len + cp [1];
      break;

    case I_P_TJMP:		/* pop-true?jump */
      if (POP () == &ScmFalse)
	cp += I_P_TJMP_len;
      else
	cp += I_P_TJMP_len + cp [1];
      break;

    case I_P_FJMP:		/* pop-false?jump */
      if (POP () != &ScmFalse)
	cp += I_P_FJMP_len;
      else
	cp += I_P_FJMP_len + cp [1];
      break;

      /*
       * Comparisons, no pop...
       */

    case I_EQJMP:		/* eq?jump */
      if (PEEK () == ScmCC->u.s.constants->array [cp [1]])
	cp += I_EQJMP_len + cp [2];
      else
	cp += I_EQJMP_len;
      break;

    case I_NEQJMP:		/* neq?jump */
      if (PEEK () == ScmCC->u.s.constants->array [cp [1]])
	cp += I_NEQJMP_len;
      else
	cp += I_NEQJMP_len + cp [2];
      break;

    case I_EVJMP:		/* eqv?jump */
      if (eqv_object (PEEK (), ScmCC->u.s.constants->array [cp [1]]))
	cp += I_EVJMP_len + cp [2];
      else
	cp += I_EVJMP_len;
      break;

    case I_NEVJMP:		/* neqv?jump */
      if (eqv_object (PEEK (), ScmCC->u.s.constants->array [cp [1]]))
	cp += I_NEVJMP_len;
      else
	cp += I_NEVJMP_len + cp [2];
      break;

    case I_MVJMP:		/* memv?jump */
      if (memv (PEEK (), ScmCC->u.s.constants->array [cp [1]]))
	cp += I_MVJMP_len + cp [2];
      else
	cp += I_MVJMP_len;
      break;

    case I_NMVJMP:		/* nmemv?jump */
      if (memv (PEEK (), ScmCC->u.s.constants->array [cp [1]]))
	cp += I_NMVJMP_len;
      else
	cp += I_NMVJMP_len + cp [2];
      break;

      /*
       * Comparisons, pop in jump branch...
       */

    case I_EQJMP_P:		/* eq?jump+pop */
      if (PEEK () == ScmCC->u.s.constants->array [cp [1]]) {
	(void) POP ();
	cp += I_EQJMP_P_len + cp [2];
      } else
	cp += I_EQJMP_P_len;
      break;

    case I_NEQJMP_P:		/* neq?jump+pop */
      if (PEEK () == ScmCC->u.s.constants->array [cp [1]])
	cp += I_NEQJMP_len;
      else {
	(void) POP ();
	cp += I_NEQJMP_len + cp [2];
      }
      break;

    case I_EVJMP_P:		/* eqv?jump+pop */
      if (eqv_object (PEEK (), ScmCC->u.s.constants->array [cp [1]])) {
	(void) POP ();
	cp += I_EVJMP_P_len + cp [2];
      } else
	cp += I_EVJMP_P_len;
      break;

    case I_NEVJMP_P:		/* neqv?jump+pop */
      if (eqv_object (PEEK (), ScmCC->u.s.constants->array [cp [1]]))
	cp += I_NEVJMP_P_len;
      else {
	(void) POP ();
	cp += I_NEVJMP_P_len + cp [2];
      }
      break;

    case I_MVJMP_P:		/* memv?jump+pop */
      if (memv (PEEK (), ScmCC->u.s.constants->array [cp [1]])) {
	(void) POP ();
	cp += I_MVJMP_P_len + cp [2];
      } else
	cp += I_MVJMP_P_len;
      break;

    case I_NMVJMP_P:		/* nmemv?jump+pop */
      if (memv (PEEK (), ScmCC->u.s.constants->array [cp [1]]))
	cp += I_NMVJMP_P_len;
      else {
	(void) POP ();
	cp += I_NMVJMP_P_len + cp [2];
      }
      break;

      /*
       * Comparisons, pop in fall-through branch...
       */

    case I_EQJMP_EP:		/* eq?jump:pop */
      if (PEEK () == ScmCC->u.s.constants->array [cp [1]])
	cp += I_EQJMP_EP_len + cp [2];
      else {
	(void) POP ();
	cp += I_EQJMP_EP_len;
      }
      break;

    case I_NEQJMP_EP:		/* neq?jump:pop */
      if (PEEK () == ScmCC->u.s.constants->array [cp [1]]) {
	(void) POP ();
	cp += I_NEQJMP_EP_len;
      } else
	cp += I_NEQJMP_EP_len + cp [2];
      break;

    case I_EVJMP_EP:		/* eqv?jump:pop */
      if (eqv_object (PEEK (), ScmCC->u.s.constants->array [cp [1]]))
	cp += I_EVJMP_EP_len + cp [2];
      else {
	(void) POP ();
	cp += I_EVJMP_EP_len;
      }
      break;

    case I_NEVJMP_EP:		/* neqv?jump:pop */
      if (eqv_object (PEEK (), ScmCC->u.s.constants->array [cp [1]])) {
	(void) POP ();
	cp += I_NEVJMP_EP_len;
      } else
	cp += I_NEVJMP_EP_len + cp [2];
      break;

    case I_MVJMP_EP:		/* memv?jump:pop */
      if (memv (PEEK (), ScmCC->u.s.constants->array [cp [1]]))
	cp += I_MVJMP_EP_len + cp [2];
      else {
	(void) POP ();
	cp += I_MVJMP_EP_len;
      }
      break;

    case I_NMVJMP_EP:		/* nmemv?jump:pop */
      if (memv (PEEK (), ScmCC->u.s.constants->array [cp [1]])) {
	(void) POP ();
	cp += I_NMVJMP_EP_len;
      } else
	cp += I_NMVJMP_EP_len + cp [2];
      break;

      /*
       * Comparisons, always pop...
       */

    case I_P_EQJMP:		/* pop-eq?jump */
      if (POP () == ScmCC->u.s.constants->array [cp [1]])
	cp += I_P_EQJMP_len + cp [2];
      else
	cp += I_P_EQJMP_len;
      break;

    case I_P_NEQJMP:		/* pop-neq?jump */
      if (POP () == ScmCC->u.s.constants->array [cp [1]])
	cp += I_P_NEQJMP_len;
      else
	cp += I_P_NEQJMP_len + cp [2];
      break;

    case I_P_EVJMP:		/* pop-eqv?jump */
      if (eqv_object (POP (), ScmCC->u.s.constants->array [cp [1]]))
	cp += I_P_EVJMP_len + cp [2];
      else
	cp += I_P_EVJMP_len;
      break;

    case I_P_NEVJMP:		/* pop-neqv?jump */
      if (eqv_object (POP (), ScmCC->u.s.constants->array [cp [1]]))
	cp += I_P_NEVJMP_len;
      else
	cp += I_P_NEVJMP_len + cp [2];
      break;

    case I_P_MVJMP:		/* pop-memv?jump */
      if (memv (POP (), ScmCC->u.s.constants->array [cp [1]]))
	cp += I_P_MVJMP_len + cp [2];
      else
	cp += I_P_MVJMP_len;
      break;

    case I_P_NMVJMP:		/* pop-nmemv?jump */
      if (memv (POP (), ScmCC->u.s.constants->array [cp [1]]))
	cp += I_P_NMVJMP_len;
      else
	cp += I_P_NMVJMP_len + cp [2];
      break;

      /*
       * Fetching from variables + temporaries ...
       */

    case I_GET_L:		/* get-loc */
      tmp = POS (cp [1]);
      PUSH (tmp);
      cp += I_GET_L_len;
      break;

    case I_GET_V:		/* get-vec */
      vector = POS (cp [1]);
      PUSH (vector->array [cp [2]]);
      cp += I_GET_V_len;
      break;

    case I_GET_G:		/* get-glob */
      symbol = ScmCC->u.s.constants->array [cp [1]];
      if ((tmp = symbol->value) == NULL)
	ScmCC->u.s.pc = cp - ScmCC->u.s.code->array,
	error ("Unbound variable: %w", symbol);
      PUSH (tmp);
      cp += I_GET_G_len;
      break;

      /*
       * Storing into variables and temporaries...
       */

    case I_PUT_L:		/* put-loc */
      POS (cp [1]) = PEEK ();
      cp += I_PUT_L_len;
      break;

    case I_PUT_L_P:		/* put-loc-pop */
      POS (cp [1]) = POP ();
      cp += I_PUT_L_P_len;
      break;

    case I_PUT_V:		/* put-vec */
      vector = POS (cp [1]);
      vector->array [cp [2]] = PEEK ();
      cp += I_PUT_V_len;
      break;

    case I_PUT_V_P:		/* put-vec-pop */
      vector = POS (cp [1]);
      vector->array [cp [2]] = POP ();
      cp += I_PUT_V_P_len;
      break;

    case I_PUT_G:		/* put-glob */
      symbol = ScmCC->u.s.constants->array [cp [1]];
      symbol->value = PEEK ();
      cp += I_PUT_G_len;
      break;

    case I_PUT_G_P:		/* put-glob-pop */
      symbol = ScmCC->u.s.constants->array [cp [1]];
      symbol->value = POP ();
      cp += I_PUT_G_P_len;
      break;

      /*
       * Loading constants...
       */

    case I_TAKE:		/* take */
      PUSH (ScmCC->u.s.constants->array [cp [1]]);
      cp += I_TAKE_len;
      break;

    case I_TK_T:		/* take-true */
      PUSH (&ScmTrue);
      cp += I_TK_T_len;
      break;

    case I_TK_F:		/* take-false */
      PUSH (&ScmFalse);
      cp += I_TK_F_len;
      break;

    case I_TK_N:		/* take-nil */
      PUSH (&ScmNil);
      cp += I_TK_N_len;
      break;

    case I_TK_P:		/* take-primitive */
      assert (cp [1] < ScmPrimitive_num);
      PUSH (GetScmPrimitive (cp [1]));
      cp += I_TK_P_len;
      break;

      /*
       * Stack manipulation...
       */

    case I_POP:			/* pop */
      (void) POP ();
      cp += I_POP_len;
      break;

    case I_MPOP:		/* multi-pop */
      ScmCC->top -= cp [1];	/* a hack, obviously */
      cp += I_MPOP_len;
      break;

      /*
       * Building closures...
       */

    case I_LAMBDA:		/* lambda */
      index = cp [1];
      cp_save = cp - ScmCC->u.s.code->array;
      SCM_NEW (proc, Procedure);
      proc->env = POP ();
      proc->code = ScmCC->u.s.constants->array [index];
      PUSH (proc);
      cp = ScmCC->u.s.code->array + cp_save + I_LAMBDA_len;
      break;

    case I_DELAY:		/* delay */
      index = cp [1];
      cp_save = cp - ScmCC->u.s.code->array;
      SCM_NEW (prom, Promise);
      prom->env = POP ();
      prom->code_or_value = ScmCC->u.s.constants->array [index];
      PUSH (prom);
      cp = ScmCC->u.s.code->array + cp_save + I_DELAY_len;
      break;

      /*
       * Miscellaneous...
       */

    case I_CONS:		/* cons */
      cp_save = cp - ScmCC->u.s.code->array;
      SCM_NEW (cons, Cons);
      cons->car = POP ();
      cons->cdr = PEEK ();
      SET_TOP (cons);
      cp = ScmCC->u.s.code->array + cp_save + I_CONS_len;
      break;

    case I_APPEND:		/* append */
      cp_save = cp - ScmCC->u.s.code->array;
      tmp = POP ();
      tmp = append_two_lists (tmp, PEEK ());
      SET_TOP (tmp);
      cp = ScmCC->u.s.code->array + cp_save + I_APPEND_len;
      break;

    case I_LTV:			/* list->vector */
      cp_save = cp - ScmCC->u.s.code->array;
      ScmListToVector ();
      cp = ScmCC->u.s.code->array + cp_save + I_LTV_len;
      break;

    case I_CAR:			/* car */
      tmp = PEEK ();
      if (ScmTypeOf (tmp) != ScmType (Cons))
	ScmCC->u.s.pc = cp - ScmCC->u.s.code->array,
	badarg ("car", tmp);
      SET_TOP (((ScmCons *) tmp)->car);
      cp += I_CAR_len;
      break;

    case I_CDR:			/* cdr */
      tmp = PEEK ();
      if (ScmTypeOf (tmp) != ScmType (Cons))
	ScmCC->u.s.pc = cp - ScmCC->u.s.code->array,
	badarg ("cdr", tmp);
      SET_TOP (((ScmCons *) tmp)->cdr);
      cp += I_CDR_len;
      break;

    case I_NOT:			/* not */
      tmp = PEEK ();
      SET_TOP (tmp == &ScmFalse ? &ScmTrue : &ScmFalse);
      cp += I_NOT_len;
      break;

    case I_VOID:
      POS (cp [1]) = NULL;
      cp += I_VOID_len;
      break;

    case I_REF:
      cp_save = cp - ScmCC->u.s.code->array;
      SCM_NEW (cell, Cell);
      cell->item = PEEK ();
      SET_TOP (cell);
      cp = ScmCC->u.s.code->array + cp_save + I_REF_len;
      break;

    case I_DEREF:
      cell = PEEK ();
      SET_TOP (cell->item);
      cp += I_DEREF_len;
      break;

    case I_ASSIGN:
      cell = POP ();
      cell->item = PEEK ();
      cp += I_ASSIGN_len;
      break;

    case I_VECTOR:
      index = cp [1];
      cp_save = cp - ScmCC->u.s.code->array;
      SCM_NEW_VECTOR (vector, index);
      {
	unsigned int i;
	for (i = 0; i < index; i++)
	  vector->array [i] = POP ();
      }
      PUSH (vector);
      cp = ScmCC->u.s.code->array + cp_save + I_VECTOR_len;
      break;

    case I_MCL:			/* make-closure */
      {
	unsigned int eslots = cp [1];
	unsigned int i;
	argcnt = cp [2];
	index = eslots + argcnt;
	cp_save = cp - ScmCC->u.s.code->array;
	SCM_NEW_VECTOR (vector, index);
	for (i = 0; i < eslots; i++)
	  vector->array [i] = NULL;
	for ( ; i < index; i++)
	  vector->array [i] = POP ();
	PUSH (vector);
	cp = ScmCC->u.s.code->array + cp_save + I_MCL_len;
      }
      break;

    case I_GET_LV:		/* get-loc-void */
      tmp = POS (cp [1]);
      POS (cp [1]) = NULL;
      PUSH (tmp);
      cp += I_GET_LV_len;
      break;

    case I_GET_VV:		/* get-vec-void */
      vector = POS (cp [1]);
      PUSH (vector->array [cp [2]]);
      POS (cp [1]) = NULL;
      cp += I_GET_VV_len;
      break;

    case I_VREF:		/* vec-ref */
      tmp = PEEK ();
      index = cp [1];
      if (ScmTypeOf (tmp) != ScmType (Vector) ||
	  (vector = tmp)->length <= index)
	ScmCC->u.s.pc = cp - ScmCC->u.s.code->array,
	badarg ("vector-ref", tmp);
      SET_TOP (vector->array [index]);
      cp += I_VREF_len;
      break;

    case I_VSET:		/* vec-set */
      tmp = POP ();
      index = cp [1];
      if (ScmTypeOf (tmp) != ScmType (Vector) ||
	  (vector = tmp)->length <= index)
	ScmCC->u.s.pc = cp - ScmCC->u.s.code->array,
	badarg ("vector-set!", tmp);
      vector->array [index] = PEEK ();
      cp += I_VSET_len;
      break;

      /*
       * module system stuff
       */

    case I_MODULE:
      index = cp [1];
      {
	ScmModule *m;
	unsigned int i;
	cp_save = cp - ScmCC->u.s.code->array;
	SCM_NEW_MODULE (m, index);
	cp = ScmCC->u.s.code->array + cp_save + I_MODULE_len;
	m->signature = POP ();
	m->exports = POP ();
	for (i = 0; i < index; i++)
	  m->array [i] = POP ();
	PUSH (m);
      }
      break;

    case I_VALIDATE:
      tmp = PEEK ();
      if (ScmTypeOf (tmp) != ScmType (Module))
	ScmCC->u.s.pc = cp - ScmCC->u.s.code->array,
	error ("not a module: %w", tmp);
      cp += I_VALIDATE_len;
      break;

    case I_FETCH:
      {
	ScmModule *m = PEEK ();
	ScmCons *c;
	unsigned mode = cp [1];
	void *sym = ScmCC->u.s.constants->array [cp [2]];
	unsigned i;

	cp += I_FETCH_len;

	vector = m->exports;
	for (i = vector->length; i-- > 1; )
	  if (((ScmCons *) vector->array [i])->car == sym)
	    break;

	c = vector->array [i];
	/*
	 * This test is only necessary, because validate really does nothing:
	 */
	if (c->car != sym)
	  ScmCC->u.s.pc = cp - ScmCC->u.s.code->array,
	  error ("module %w doesn't export %w", m, sym);

	SET_TOP (m->array [i]);

	switch (mode) {
	case 0:			/* constant */
	  if (c->cdr != &ScmNil)
	    ScmCC->u.s.pc = cp - ScmCC->u.s.code->array,
	    error ("module %m doesn't export %w as constant", m, sym);
	  break;
	case 1:			/* read-only */
	  if (c->cdr == &ScmNil) {
	    cp_save = cp - ScmCC->u.s.code->array;
	    SCM_NEW (cell, Cell);
	    cell->item = PEEK ();
	    SET_TOP (cell);
	    cp = ScmCC->u.s.code->array + cp_save;
	  } else if (c->cdr != &ScmTrue)
	    ScmCC->u.s.pc = cp - ScmCC->u.s.code->array,
	    error ("module %m doesn't export %w as read-only", m, sym);
	  break;
	default:		/* variable */
	  if (c->cdr != &ScmFalse)
	    ScmCC->u.s.pc = cp - ScmCC->u.s.code->array,
	    error ("module %m doesn't export %w as variable", m, sym);
	  break;
	}
      }
      break;

      /*
       * Dealing with the unthinkable...
       */

    default:
      ScmCC->u.s.pc = cp - ScmCC->u.s.code->array,
      error ("bad VM code: %i", (int) cp [0]);
      break;
    }
  }
}

void ScmPushPrimitiveContinuation (void *environ, unsigned stackreq)
{
  assert (active_primitive != SCM_VM_TRAP_CONT);
  ScmPushCContinuation (stackreq, active_primitive, environ);
}

/*
 * Scheme's assembly language
 */

static unsigned short *label_address = NULL;
static int label_count = 0;

static void deallocate_all (void)
{
  if (label_address != NULL)
    free (label_address);
}

static void provide_n_labels (int n)
{
  if (n > label_count) {
    label_address = REALLOC (label_address, n * sizeof (unsigned short));
    if (label_address == NULL) {
      label_count = 0;
      reset ("Out of memory");
    }
    label_count = n;
  }
}

static
struct stat_desc *get_stat (void *opsym)
{
  if (ScmTypeOf (opsym) == ScmType (Symbol))
    return
      find_stat (((ScmSymbol *) opsym)->array,
		 ((ScmSymbol *) opsym)->length,
		 opsym);
  else if (ScmTypeOf (opsym) == ScmType (String))
    return
      find_stat (((ScmString *) opsym)->array,
		 ((ScmString *) opsym)->length,
		 opsym);
  else
    error ("vscm-asm: bad operation symbol: %w", opsym);
}

static
unsigned get_label_addresses (int nlabels, void *asmlist)
{
  void *stat;
  unsigned pc;

  pc = 0;
  provide_n_labels (nlabels);
  while (ScmTypeOf (asmlist) == ScmType (Cons)) {
    stat = ((ScmCons *) asmlist)->car;
    asmlist = ((ScmCons *) asmlist)->cdr;
    if (ScmTypeOf (stat) == ScmType (Cons))
      pc += get_stat (((ScmCons *) stat)->car)->length;
    else if (ScmUPred (SCM_EXACT_PRED, stat)) {
      label_address [ScmNumberToUShort (stat, "get_label_addresses")] = pc;
    } else
      error ("vscm-asm: bad assembly code: %w -- %w", stat, asmlist);
  }
  return pc;
}

static void check_0 (void *x)
{
  if (ScmTypeOf (x) != ScmType (Cons))
    error ("check_0 failed: %w", x);
}

static void check_1 (void *x)
{
  if (ScmTypeOf (x) != ScmType (Cons))
    error ("check_1 failed: %w", x);
  check_0 (((ScmCons *) x)->cdr);
}

static void check_2 (void *x)
{
  if (ScmTypeOf (x) != ScmType (Cons))
    error ("check_2 failed: %w", x);
  check_1 (((ScmCons *) x)->cdr);
}


static unsigned get_0 (void *x)
{
  return ScmNumberToUShort (((ScmCons *) x)->car, "get_0");
}

static unsigned get_1 (void *x)
{
  return get_0 (((ScmCons *) x)->cdr);
}

static unsigned get_primno (void *x)
{
  x = ((ScmCons *) x)->cdr;
  x = ((ScmCons *) x)->car;
  if (ScmTypeOf (x) == ScmType (Symbol)) {
    unsigned n = ((ScmSymbol *) x)->primno_succ;
    if (n > 0)
      return n - 1;
    else
      error ("get_primno: not a primitive: %w", x);
  } else
    return ScmNumberToUShort (x, "get_primno");
}

static unsigned get_2 (void *x)
{
  return get_1 (((ScmCons *) x)->cdr);
}

static ScmCode *do_asm (void *, void *);

static
ScmCode *do_assembly (
  unsigned argcnt, int takerest, void *names, void *constants,
  unsigned stackreq, int nlabels, void *asmlist, void *nsym)
{
  unsigned length = get_label_addresses (nlabels, asmlist);
  ScmCode *code;
  unsigned nconstants = ScmListLength (constants);
  ScmVector *cvect;
  unsigned pc;
  void *stat;
  struct stat_desc *desc;
  unsigned i;
  unsigned tmp;

  constants_save = constants;
  reverse_old = asmlist;
  names_save = names;
  append_save = nsym;
  Push (constants_save);
  Push (reverse_old);
  reverse_old = NULL;
  SCM_VNEW (code, Code, length, unsigned short);
  code->length = length;
  code->arg_cnt = argcnt;
  code->take_rest = takerest;
  code->stack_requirement = stackreq;
  code->argument_names = names_save;
  code->proc_name = append_save;
  code->constants = NULL;
  append_save = names_save = NULL;
  Push (code);
  cvect = NewScmVector (nconstants);
  code = POP ();
  code->constants = cvect;

  pc = 0;
  for (asmlist = POP ();
       ScmTypeOf (asmlist) == ScmType (Cons);
       asmlist = ((ScmCons *) asmlist)->cdr) {
    stat = ((ScmCons *) asmlist)->car;
    if (ScmTypeOf (stat) == ScmType (Cons)) {
      desc = get_stat (((ScmCons *) stat)->car);
      code->array [pc] = desc->opcode;
      switch (desc->opcode) {
      case I_JMP_FW:
      case I_TJMP:
      case I_FJMP:
      case I_TJMP_P:
      case I_FJMP_P:
      case I_TJMP_EP:
      case I_FJMP_EP:
      case I_P_TJMP:
      case I_P_FJMP:
	check_1 (stat);
	tmp = label_address [get_1 (stat)] - pc - desc->length;
	code->array [pc + 1] = tmp;
	break;
      case I_JMP_BW:
	check_1 (stat);
	tmp = pc + desc->length - label_address [get_1 (stat)];
	code->array [pc + 1] = tmp;
	break;
      case I_EQJMP:
      case I_NEQJMP:
      case I_EQJMP_P:
      case I_NEQJMP_P:
      case I_EQJMP_EP:
      case I_NEQJMP_EP:
      case I_P_EQJMP:
      case I_P_NEQJMP:
      case I_EVJMP:
      case I_NEVJMP:
      case I_EVJMP_P:
      case I_NEVJMP_P:
      case I_EVJMP_EP:
      case I_NEVJMP_EP:
      case I_P_EVJMP:
      case I_P_NEVJMP:
      case I_MVJMP:
      case I_NMVJMP:
      case I_MVJMP_P:
      case I_NMVJMP_P:
      case I_MVJMP_EP:
      case I_NMVJMP_EP:
      case I_P_MVJMP:
      case I_P_NMVJMP:
	check_2 (stat);
	tmp = get_1 (stat);
	code->array [pc + 1] = tmp;
	tmp = label_address [get_2 (stat)] - pc - desc->length;
	code->array [pc + 2] = tmp;
	break;
      case I_GET_L:
      case I_GET_LV:
      case I_GET_G:
      case I_PUT_L:
      case I_PUT_L_P:
      case I_PUT_G:
      case I_PUT_G_P:
      case I_TAKE:
      case I_VECTOR:
      case I_CALL:
      case I_CALLE:
      case I_MPOP:
      case I_LAMBDA:
      case I_DELAY:
      case I_VOID:
      case I_MODULE:
      case I_VREF:
      case I_VSET:
	check_1 (stat);
	tmp = get_1 (stat);
	code->array [pc + 1] = tmp;
	break;
      case I_TK_P:
	check_1 (stat);
	tmp = get_primno (stat);
	code->array [pc + 1] = tmp;
	break;
      case I_GET_V:
      case I_GET_VV:
      case I_PUT_V:
      case I_PUT_V_P:
      case I_MCL:
      case I_FETCH:
	check_2 (stat);
	tmp = get_1 (stat);
	code->array [pc + 1] = tmp;
	tmp = get_2 (stat);
	code->array [pc + 2] = tmp;
	break;
	/*
	 * case I_TK_T:
	 * case I_TK_F:
	 * case I_TK_N:
	 * case I_POP:
	 * case I_EXIT:
	 * case I_CONS:
	 * case I_APPEND:
	 * case I_LTV:
	 * case I_CAR:
	 * case I_CDR:
	 * case I_NOT:
	 * case I_CHECK:
	 * case I_REF:
	 * case I_DEREF:
	 * case I_ASSIGN:
	 * case I_VALIDATE:
	 */
      default:
	/* do nothing */
	break;
      }
      pc += desc->length;
    }
  }
  constants_save = POP ();
  for (i = 0; i < nconstants; i++) {
    stat = ((ScmCons *) constants_save)->car;
    constants_save = ((ScmCons *) constants_save)->cdr;
    if (ScmTypeOf (stat) != ScmType (Cons))
      error ("vscm-asm: bad constant: %w", stat);
    if (((ScmCons *) stat)->car == ScmQuotePtr) {
      stat = ((ScmCons *) stat)->cdr;
      if (ScmTypeOf (stat) != ScmType (Cons))
	error ("vscm-asm: bad quotation: %w", stat);
      else
	code->constants->array [i] = ((ScmCons *) stat)->car;
    } else {
      append_save = code->proc_name;
      Push (code);
      Push (constants_save);
      nsym = append_save;
      append_save = NULL;
      stat = do_asm (stat, nsym);
      constants_save = POP ();
      code = POP ();
      code->constants->array [i] = stat;
    }
  }
  constants_save = NULL;
  return code;
}

static ScmCode *do_asm (void *stat, void *nsym)
{
  unsigned new_argcnt;
  int new_takerest;
  void *new_names;
  void *new_constants;
  unsigned int new_stackreq;
  int new_nlabels;

  check_0 (stat);
  if (ScmTypeOf (((ScmCons *) stat)->car) == ScmType (String)) {
    nsym = ((ScmCons *) stat)->car;
    stat = ((ScmCons *) stat)->cdr;
    check_0 (stat);
  }
  new_argcnt = get_0 (stat);
  stat = ((ScmCons *) stat)->cdr;
  check_0 (stat);
  /* Since several Schemes write () for #f, we have to check for #t here */
  new_takerest = (((ScmCons *) stat)->car == &ScmTrue ? 1 : 0);
  stat = ((ScmCons *) stat)->cdr;
  check_0 (stat);
  new_names = ((ScmCons *) stat)->car;
  stat = ((ScmCons *) stat)->cdr;
  check_0 (stat);
  new_constants = ((ScmCons *) stat)->car;
  stat = ((ScmCons *) stat)->cdr;
  check_0 (stat);
  new_stackreq = get_0 (stat);
  stat = ((ScmCons *) stat)->cdr;
  check_0 (stat);
  new_nlabels = get_0 (stat);
  stat = ((ScmCons *) stat)->cdr;
  return do_assembly (new_argcnt, new_takerest, new_names, new_constants,
		new_stackreq, new_nlabels, stat, nsym);
}

void *ScmAsmToCode (void *asmlist)
{
  return do_asm (asmlist, NULL);
}

void *ScmCodeToProcedure (void *code)
{
  ScmProcedure *proc;

  Push (code);
  SCM_NEW (proc, Procedure);
  proc->code = POP ();
  proc->env = NULL;
  return proc;
}

void *ScmAsmToProcedure (void *asmlist)
{
  return ScmCodeToProcedure (ScmAsmToCode (asmlist));
}

/*
 * given a pointer to a ScmCode object (code) and a program counter (pc)
 * retrieve the string opcode (opcode), the instruction length (length)
 * which is the same as 1+#operands if possible the operands (op1,op2,op3)
 * return the ``new'' pc (pc + *length),
 *  -1 if pc is out of range,
 *  -2 if operation cannot be decoded
 */
long ScmDisassemble
  (ScmCode *code, unsigned long pc, const char **opcode, int *length,
   unsigned short *op1, unsigned short *op2, unsigned short *op3)
{
  unsigned num_opcode;
  int len;

  if (pc > code->length)
    return -1;
  if ((num_opcode = code->array [pc]) >= N_INSTRUCTIONS)
    return -2;
  if ((pc + (len = stat_desc [num_opcode].length)) > code->length)
    return -1;
  *opcode = stat_desc [num_opcode].name;
  *length = len;
  if (len > 1) {
    *op1 = code->array [pc + 1];
    if (len > 2) {
      *op2 = code->array [pc + 2];
      if (len > 3)
	*op3 = code->array [pc + 3];
    }
  }
  return pc + len;
}
