;;########################################################################
;; tranobj1.lsp
;; contains code to implement prototype transformation objects
;; Copyright (c) 1991-97 by Forrest W. Young
;;########################################################################

;;------------------------------------------------------------------------
;;transformation object proto
;;------------------------------------------------------------------------

(defproto transf-object-proto () () mv-model-object-proto)

(defmeth transf-object-proto :isnew (&rest args)
  (send self :model-abbrev (select args 3))
  (send self :new-object)
  (setf (select args 3) nil)
  (apply #'call-next-method args))

(defmeth transf-object-proto :new-object ()
  (setf current-transf self)
  )

;;------------------------------------------------------------------------
;;transpose-data object proto
;;------------------------------------------------------------------------

(defproto trnsps-transf-object-proto '() () transf-object-proto)

(defmeth trnsps-transf-object-proto :isnew (&rest args)
  (apply #'call-next-method args))

(defmeth trnsps-transf-object-proto :options () t)

(defmeth trnsps-transf-object-proto :analysis ()
    (data (send self :name)
          :created   (- (send *desktop* :num-icons) 1)
          :title     (concatenate 'string "Transposed " (send self :title))
          :variables (send current-data :active-labels) 
          :data      (combine 
                      (transpose (send self :active-data-matrix '(all))))
          :labels    (send self :active-variables '(all))
          :types     (repeat "numeric" (send self :nobs))))

(defun transpose-data 
  (&key 
   (data       current-data)
   (dialog     nil)
   (name       "Trnsps")
   (title      nil)
   )
  (send trnsps-transf-object-proto :new 9 data title name dialog))

;;------------------------------------------------------------------------
;;normalize-data object proto
;;------------------------------------------------------------------------

(defproto norm-transf-object-proto '(stdv mean) ()
  transf-object-proto)

(defmeth norm-transf-object-proto :isnew (mean stdv &rest args)
  (send self :mean mean)
  (send self :stdv stdv)
  (apply #'call-next-method args))

(defmeth norm-transf-object-proto :stdv (&optional (value nil set))
    (if set (setf (slot-value  'stdv) value))
    (slot-value 'stdv))

(defmeth norm-transf-object-proto :mean (&optional (value nil set))
    (if set (setf (slot-value  'mean) value))
    (slot-value 'mean))

(defmeth norm-transf-object-proto :options ()
  (when (send self :dialog)
        (let ((mean-stdv (send (send self :dialog-box) :modal-dialog))
              )
          (when mean-stdv
                (if (= (select mean-stdv 0) 0)
                    (send self :mean (select mean-stdv 1))
                    (send self :mean nil))
                (if (= (select mean-stdv 2) 0)
                    (send self :stdv (select mean-stdv 3))
                    (send self :stdv nil)))
          mean-stdv)))

(defmeth norm-transf-object-proto :dialog-box ()
  (let* ((var-text (send text-item-proto :new "NORMALIZE ..."))
         (mean-text (send text-item-proto :new "MEANS:"))
         (mean-item (send choice-item-proto :new
                    (list "Change all to:" "Change none.")))
         (mean-value-item 
          (send edit-text-item-proto :new "0.0" :text-length 3))
         (stdv-text (send text-item-proto :new "STANDARD DEVIATIONS:"))
         (stdv-item (send choice-item-proto :new
                    (list "Change all to:" "Change none.")))
         (stdv-value-item
          (send edit-text-item-proto :new "1.0" :text-length 3))
         (cancel    (send modal-button-proto :new "Cancel"))
         (ok        (send modal-button-proto :new "OK" :action #'(lambda () 
                    (list (send mean-item :value) 
                          (number-from-string (send mean-value-item :text))
                          (send stdv-item :value)
                          (number-from-string (send stdv-value-item :text))
                          )))))
    (send modal-dialog-proto :new
               (list var-text 
                     mean-text 
                     (list mean-item mean-value-item) 
                     stdv-text 
                     (list stdv-item stdv-value-item) 
                     (list ok cancel))
          :default-button ok)
        ))

(defmeth norm-transf-object-proto :analysis  ()
  (let* ((mean (send self :mean))
         (stdv (send self :stdv))
         (data-matrix (send self :active-data-matrix '(numeric)))
         (size (reverse (array-dimensions data-matrix)))
         (means (mapcar #' (lambda (x) (mean x)) (column-list data-matrix)))
         (result-matrix nil)
         )
    (setf data-matrix (center data-matrix))
    (when stdv (setf data-matrix (normalize data-matrix stdv)))
    (if mean 
        (setf data-matrix (+ mean  (column-list data-matrix)))
        (setf data-matrix (+ means (column-list data-matrix))))
    (setf data-matrix 
               (transpose (matrix size (combine data-matrix))))
    (data (send self :name)
          :created (- (send *desktop* :num-icons) 1)
          :title (concatenate 'string "Normalized " (send self :title))
          :labels (send current-data :active-labels) 
          :data (combine data-matrix)
          :variables (send self :active-variables '(numeric))
          :types     (send self :active-types     '(numeric)))
    ))

(defun normalize-data 
  (&key 
      (data      current-data)
      (mean      0)
      (stdv      1)
      (dialog    nil)
      (name      "Norm")
      (title     nil)
   )
  (send norm-transf-object-proto :new mean stdv 9 data title name dialog))

;;------------------------------------------------------------------------
;;orthogonalize-data object proto
;;------------------------------------------------------------------------

(defproto orthog-transf-object-proto '() () transf-object-proto)

(defmeth orthog-transf-object-proto :isnew (&rest args)
  (apply #'call-next-method args))

(defmeth orthog-transf-object-proto :options ()
  t)

(defmeth orthog-transf-object-proto :analysis ()
    (data (send self :name)
          :created   (- (send *desktop* :num-icons) 1)
          :title     (concatenate 'string "Orthogonalized " (send self :title))
          :labels    (send current-data :active-labels) 
          :data      (combine (second (gs-orthog
                           (send self :active-data-matrix '(numeric)))))
          :variables (send self :active-variables '(numeric))
          :types     (send self :active-types     '(numeric)))
  )

(defun orthogonalize-data 
  (&key 
   (data       current-data)
   (dialog     nil)
   (name       "Orthog")
   (title      nil)
   )
  (send orthog-transf-object-proto :new 9 data title name dialog))

;;------------------------------------------------------------------------
;;correlation object proto
;;------------------------------------------------------------------------

(defproto corr-transf-object-proto '() () transf-object-proto)

(defmeth corr-transf-object-proto :isnew (&rest args)
  (apply #'call-next-method args))

(defmeth corr-transf-object-proto :options ()
  t)

(defmeth corr-transf-object-proto :analysis ()
  (data (send self :name)
        :created   (- (send *desktop* :num-icons) 1)
        :matrices '("Correlations")
        :shapes   '("symmetric")
        :title     (strcat "Correlations of " (send self :title))
        :labels    (send current-data :active-labels) 
        :data      (combine (correlation-matrix 
                             (send self :active-data-matrix '(numeric))))
        :variables (send self :active-variables '(numeric))
        :types     (send self :active-types     '(numeric)))
  )

(defun correlations 
  (&key 
   (data       current-data)
   (dialog     nil)
   (name       "Corr")
   (title      nil)
   )
  (send corr-transf-object-proto :new 9 data title name dialog))

;;------------------------------------------------------------------------
;;covariance object proto
;;------------------------------------------------------------------------

(defproto covar-transf-object-proto '() () transf-object-proto)

(defmeth covar-transf-object-proto :isnew (&rest args)
  (apply #'call-next-method args))

(defmeth covar-transf-object-proto :options ()
  t)

(defmeth covar-transf-object-proto :analysis ()
    (data (send self :name)
          :created   (- (send *desktop* :num-icons) 1)
          :matrices '("Covariances")
          :shapes   '("symmetric")
          :title     (strcat "Covariances of " (send self :title))
          :labels    (send current-data :active-labels) 
          :data      (combine (covariance-matrix 
                               (send self :active-data-matrix '(numeric))))
          :variables (send self :active-variables '(numeric))
          :types     (send self :active-types     '(numeric)))
  )

(defun covariances 
  (&key 
   (data       current-data)
   (dialog     nil)
   (name       "Cov")
   (title      nil)
   )
  (send covar-transf-object-proto :new 9 data title name dialog))

;;------------------------------------------------------------------------
;;distance object proto
;;------------------------------------------------------------------------

(defproto dist-transf-object-proto '() () transf-object-proto)

(defmeth dist-transf-object-proto :isnew (&rest args)
  (apply #'call-next-method args))

(defmeth dist-transf-object-proto :options ()
  t)

(defmeth dist-transf-object-proto :analysis ()
    (data (send self :name)
          :created   (- (send *desktop* :num-icons) 1)
          :matrices '("Distances")
          :shapes   '("symmetric")
          :title     (strcat "Distances from " (send self :title))
          :labels    (send self :active-variables '(numeric)) 
          :data      (combine (distance-matrix 
                               (send self :active-data-matrix '(numeric))))
          :variables (send current-data :active-labels)
          :types     (repeat "numeric" (send self :active-nobs))))

(defun distances 
  (&key 
   (data       current-data)
   (dialog     nil)
   (name       "Dist")
   (title      nil)
   )
  (send dist-transf-object-proto :new 9 data title name dialog))

;;------------------------------------------------------------------------
;;sort object proto
;;------------------------------------------------------------------------

(defproto sort-transf-object-proto '(sort-label sort-var descending) ()
  transf-object-proto)

(defmeth sort-transf-object-proto 
             :isnew (sort-label sort-var descending &rest args)
  (if sort-var 
        (send self :sort-label nil)
        (send self :sort-label sort-label))
  (send self :descending descending)
  (send self :sort-var sort-var)
  (apply #'call-next-method args)
  )

(defmeth sort-transf-object-proto :sort-label (&optional (logical nil set))
    (if set (setf (slot-value  'sort-label) logical))
    (slot-value 'sort-label))

(defmeth sort-transf-object-proto :sort-var (&optional (value nil set))
    (if set (setf (slot-value  'sort-var) value))
    (slot-value 'sort-var))

(defmeth sort-transf-object-proto :descending (&optional (logical nil set))
    (if set (setf (slot-value  'descending) logical))
    (slot-value 'descending))

(defmeth sort-transf-object-proto :options ()
  (when (send self :dialog)
        (let ((result (send (send self :dialog-box) :modal-dialog))
              ) 
          (when result 
                (send self :sort-label (= 0 (first result)))
                (send self :sort-var (second result))
                (send self :descending (third result))) 
          (when (and (not (send self :sort-var)) 
                     (not (send self :sort-label)))
                (error-message 
                 "You must select labels or a variable for sorting."));fwy4.29
          result)))

(defmeth sort-transf-object-proto :dialog-box ()
  (let* ((var-text 
          (send text-item-proto :new "SORT ...    by selected variable"))
         (perm-text 
          (send text-item-proto :new "PERMUTE ... all other variables"))
         (var-item (send choice-item-proto :new
                         (list "Sort by observation labels"
                               "Sort by values of variable:")
                         :value 1)) ;fwy4.28
         (var-list (send current-data :active-variables '(all)))
         (var-list-item (send list-item-proto :new var-list
                              ;:size (list 200 96)
                              
                              ))
         (descending 
          (send toggle-item-proto :new "Sort into descending order"))
         (cancel    (send modal-button-proto :new "Cancel"))
         (ok        (send modal-button-proto :new "OK" :action #'
                          (lambda () 
                            (list (send var-item :value)
                                  (send var-list-item :selection)
                                  (send descending :value)))))
         )
    (send modal-dialog-proto :new
               (list var-text
                     perm-text
                     var-item
                     var-list-item
                     descending
                     (list ok cancel))
          :default-button ok)
    ))

(defmeth sort-transf-object-proto :analysis ()
  (let* ((data-matrix (send self :active-data-matrix '(all)))
         (labels (send current-data :active-labels))
         (rows-of-data-matrix (row-list data-matrix))
         (order-var nil)
         (var nil)
         (permuted-data nil)
         (permuted-labels nil)
         (permuted-data-matrix nil)
         (result nil))
    (when (stringp (send self :sort-var))
          (send self :sort-var 
                (select ($position 
                         (list (send self :sort-var)) 
                         (send self :active-variables '(all))) 0)))
    (if (send self :sort-label)
        (setf var labels)
        (setf var (send self :variable 
                        (select (send self :active-variables '(all)) 
                                (send self :sort-var)))))
    (setf result 
          (sort-and-permute-dob data-matrix labels var 
                                (send self :descending)))
    (data (send self :name)
          :created   (- (send *desktop* :num-icons) 1)
          :title     (concatenate 'string "Sorted " (send self :title))
          :labels    (second result) 
          :data      (combine (first result))
         ; :variables (send self :active-variables '(numeric))
         ; :types     (send self :active-types     '(numeric))
          :variables (send self :variables);fwy4.28
          :types     (send self :types));fwy4.28
    ))

(defun sort-permute
  (&key
   (data current-data)
   (dialog nil)
   (name "Sort")
   (title nil)
   (label t)
   (variable nil)
   (descending nil)
   )
  (let ((ok-types '(all)));fwy4.28
    (send sort-transf-object-proto 
          :new label variable descending 9 data title name dialog  
               ok-types)));fwy4.28

;;------------------------------------------------------------------------
;;rank object proto
;;------------------------------------------------------------------------

(defproto rank-transf-object-proto '() () transf-object-proto)

(defmeth rank-transf-object-proto :isnew (&rest args)
  (apply #'call-next-method args))

(defmeth rank-transf-object-proto :options ()
  t)

(defmeth rank-transf-object-proto :analysis ()
  (let* (
         (data-matrix 
          (send self :active-data-matrix '(numeric ordinal)));fwy4.28
         (size (reverse (array-dimensions data-matrix)))
         (rank-data 
          (mapcar #'rank-with-ties (column-list data-matrix)));fwy4.28
         (rank-data-matrix (transpose (matrix size (combine rank-data))))
         )
    (data (send self :name)
          :created   (- (send *desktop* :num-icons) 1)
          :title     (concatenate 'string "Ranked " (send self :title))
          :labels    (send current-data :active-labels) 
          :data      (combine rank-data-matrix)
          :variables (send self :active-variables '(numeric ordinal));fwy4.28
          :types     (send self :active-types     '(numeric ordinal)));fwy4.28
    ))

(defun ranks 
  (&key 
   (data       current-data)
   (dialog     nil)
   (name       "Rank")
   (title      nil)
   )
  (let ((ok-types '(numeric ordinal)));fwy4.28
    (send rank-transf-object-proto :new 9 data title name dialog
          ok-types)));fwy4.28

(provide "tranobj1")