;;************************************************************************
;; graphic1.lsp 
;; contains code for graph-proto new variable buttons and for new and 
;; revised methods for name-list, histogram, scatterplot and scatmat
;; copyright (c) 1991-98 by Forrest W. Young
;;************************************************************************

(require "vista")


;-----------------------------
;code for New Variable buttons
;-----------------------------

(defmeth graph-proto :switch-use-color ()
  (send self :use-color (not (send self :use-color)))
  (send self :redraw)
  (send self :use-color))

(defmeth graph-proto :new-x ()
  (send self :new-var "X" ))

(defmeth graph-proto :new-y ()
  (send self :new-var "Y" ))

(defmeth graph-proto :new-z ()
  (send self :new-var "Z" ))

(defmeth graph-proto :new-var (axis)
  (let* ((result (send self :new-variable-dialog axis))
         )
    (when (> (length result) 0)
          (setf result (select result 0))
          (cond 
            ((not result) (error-message "You must select a variable"))
            (t
             (send self :show-new-var axis result))))))

(defmeth graph-proto :make-show-variables-list ()
  (let* ((variables (combine (send self :variable-labels)))
         (cur-vars (send self :current-variables)))
    (set-difference variables (select variables cur-vars))))

(defmeth graph-proto :new-variable-dialog (axis)
"Arg: AXIS
Presents a dialog box to choose a variable to be used on AXIS x y or z. Returns (var-name) for choice, (nil) for OK but no choice, nil for cancel."
  (let* ((row-pix 16)
         (variables (send self :variable-labels))
         (cur-vars (send self :current-variables))
         (last-cur-var (first (last cur-vars)))
         (show-variables (send self :make-show-variables-list))
         (title (send text-item-proto :new 
                     (format nil "Choose new variable for ~a" axis)))
         (cancel (send modal-button-proto :new "Cancel"))
         (varlist nil)
         (ok (send modal-button-proto :new "OK" :action 
              #'(lambda () 
                  (let* ((selection (send varlist :selection))
                         )
                    (when selection
                          (setf selection 
                                (list (select show-variables selection))))
                    (when (not selection) (setf selection (list nil)))
                    selection))))
         (nshow nil)
         (dialog nil)
         (result nil))
    (when show-variables
          (setf nshow (length show-variables))
          (setf varlist 
                (if (> nshow 6) 
                    (send list-item-proto :new show-variables
                          :size (list 190 (* 6 row-pix)))
                    (send list-item-proto :new show-variables))))
    (cond
      ((not nshow) (setf result nil))
      ((= nshow 1) (setf result show-variables))
      ((> nshow 1)
       (setf dialog 
             (send modal-dialog-proto :new
                   (list title varlist (list ok cancel)) :default-button ok))
       (setf result (send dialog :modal-dialog))))
   result)) 


;;************************************************************************
;; name-list-proto methods
;;************************************************************************

(send name-list-proto :menu-template 
      '(LINK MOUSE DASH ERASE-SELECTION FOCUS-ON-SELECTION 
                        SHOW-ALL COLOR ))

(defmeth name-list-proto :fix-name-list ()
"Fixes the name-list bug by adjusting the scroll bars to be at least screen-size in length."
  (let ((ms (max (screen-size)))
        (hs (send self :has-h-scroll))
        (vs (send self :has-v-scroll)))
    (if vs
        (when (< (send self :canvas-height) ms) (send self :has-v-scroll ms))
        (send self :has-v-scroll ms))
    (if hs 
        (when (< (send self :canvas-width) ms) (send self :has-h-scroll ms))
        (send self :has-h-scroll ms))))
  
;;************************************************************************
;; histogram-proto methods
;;************************************************************************

(send histogram-proto :menu-template '(link mouse resize-brush dash 
                                            erase-selection
                                            focus-on-selection show-all
					    color dash change-bins))

(send histogram-proto :size 250 250)

(defmeth histogram-proto :new (&rest args)
  (let ((object (apply #'call-next-method args)))
    (send object :add-slot 'show-normal)
    (send object :add-slot 'show-kernel)
    (send object :add-slot 'show-density)
    (send object :add-slot 'kernel-type)
    (send object :add-slot 'max-normal-pixel)
    (send object :add-slot 'slider)
    (send object :add-slot 'dens-dialog)
    (defmeth object :show-normal (&optional (logical nil set))
      (if set (setf (slot-value 'show-normal) logical))
      (slot-value 'show-normal))
    (defmeth object :show-kernel (&optional (logical nil set))
      (if set (setf (slot-value 'show-kernel) logical))
      (slot-value 'show-kernel))
    (defmeth object :show-density (&optional (logical nil set))
      (if set (setf (slot-value 'show-density) logical))
      (slot-value 'show-density))
    (defmeth object :kernel-type (&optional (logical nil set))
      (if set (setf (slot-value 'kernel-type) logical))
      (slot-value 'kernel-type))
    (defmeth object :switch-add-normal (&key (color 'red) (kcolor 'blue) (line-width 2))
      (send object :show-normal (not (send object :show-normal)))
      (cond
        ((send object :show-normal)
         (send object :add-normal :color color :line-width line-width))
        (t
         (send object :clear-lines)
         (when (send object :show-kernel) 
               (send object :add-kernel 
                    (send object :kernel-type) 
                     :color kcolor :line-width line-width)))))
    (defmeth object :switch-add-kernel (&key (color 'blue) (line-width 2))
      (send object :show-kernel (not (send object :show-kernel)))
      (cond 
        ((send object :show-kernel)
         (send object :add-kernel 
              (send object :kernel-type) :color 'blue :line-width line-width))
        (t
         (send object :clear-lines)
         (when (send object :show-normal) 
               (send object :add-normal :color 'red :line-width line-width)))))
    (defmeth object :max-normal-pixel (&optional (logical nil set))
      (if set (setf (slot-value 'max-normal-pixel) logical))
      (slot-value 'max-normal-pixel))
    object))

(defmeth histogram-proto :choose-density ()
  (when (> (send self :num-points) 2)
        (cond 
          ((not (send self :slot-value 'dens-dialog))
           (let* ((den-fun-state (list nil nil nil))
                  (monopoly-arguments nil)
                  (title  (send text-item-proto :new "Choose Curve"))
                  (Kernel-text (send text-item-proto :new "Kernel:"))
                  (normal (send toggle-item-proto :new "Normal Density"
                                :value (select den-fun-state 0)
                                :action #'(lambda () (send self :switch-add-normal))))
                  (kernel-type (send choice-item-proto :new
                                     (list "Bisquare" "Gaussian" "Triangular" "Uniform" )
                                     :value 0
                                     :action #'(lambda () 
                                                 (send self :put-kernel-type))))
                  (kernel (send toggle-item-proto :new "Kernel Density"
                                :value (select den-fun-state 1)
                                :action #'(lambda ()
                                            (send self :kernel-type (send kernel-type :value))
                                            (send self :switch-add-kernel))))
                  
                  (monopoly (send toggle-item-proto :new "MonoPoly Density"
                                  :value (select den-fun-state 2)))
                  (dialog (send dialog-proto :new
                                (list title normal 
                                      (list (list kernel (list kernel-text kernel-type))
                                            ;monopoly
                                            ))
                                :title "Distribution Curve"))
                  )
             (defmeth self :put-kernel-type ()
               (send self :kernel-type (send kernel-type :value))
               (when (send self :show-kernel)
                     (send self :switch-add-kernel)
                     (send self :switch-add-kernel)))
             (send self :add-subordinate dialog)
             (send self :slot-value 'dens-dialog dialog)))
          (t
           (send (send self :slot-value 'dens-dialog) :show-window)))))
          

(defmeth histogram-proto :add-normal (&key (color 'red) (line-width 2))
        (let* ((current-variable (first (send self :content-variables)))
               (var (send self :point-coordinate  current-variable
                          (iseq (send self :num-points))))
               (ndim (- (send self :num-variables) 1))
               (mu (mean var))
               (s (standard-deviation var))
               (range (send self :range current-variable))
               (x (rseq (first range) (second range) 50))
               (y (/ (normal-dens (/ (- x mu) s)) s))
               (max-pix-now (second (send self :real-to-canvas mu (max y))))
               (zero-pix (second (send self :real-to-canvas mu 0)))
             ; (max-pix-b4  (send self :max-normal-pixel))
               (y (* y (/ (- zero-pix 50) (- zero-pix max-pix-now))))
               )
          (when (or (not (send self :use-color)) (= *color-mode* 0) )
                (setf color 'black))
          (send self :add-lines (append (repeat (list x) ndim) (list y))
               :color color :width line-width)
          ))


(defmeth histogram-proto :add-kernel 
                         (type-number &key (color 'blue) (line-width 2))
  (let* ((type (case type-number
                 (0 'b)
                 (1 'g)
                 (2 't)
                 (3 'u)))
         (current-variable (first (send self :content-variables)))
         (npts (send self :num-points))
         (ndim (- (send self :num-variables) 1))
         (varx (send self :point-coordinate current-variable (iseq npts)))
       ;  (range (send self :range current-variable))
         (x (rseq (min varx) (max varx) 50))
         (maxyaxis (max (send self :range (1- (send self :num-variables)))))
         (y (second (kernel-dens varx :xvals 50 :type type)))
         (maxy (max y))
         (x-at-maxy (select x (first (last (order y)))))
         (max-pix-now (second (send self :real-to-canvas x-at-maxy maxy)))
         (zero-pix (second (send self :real-to-canvas x-at-maxy 0)))
         (y (* y (/ (- zero-pix 50) (- zero-pix max-pix-now))))
         )
    (when (or (not (send self :use-color)) (= *color-mode* 0) )
                (setf color 'black))
    (send self :add-lines (append (repeat (list x) ndim) (list y)) 
          :color color :width line-width)
    ))


(defmeth histogram-proto :add-monopoly-dens 
                         ( &key (color 'green) (line-width 2))
  (let* ((current-variable (first (send self :content-variables)))
         (npts (send self :num-points))
         (ndim (- (send self :num-variables) 1))
         (varx (send self :point-coordinate current-variable (iseq npts)))
         (range (send self :range current-variable))
         (x (rseq (min varx) (max varx) 50))
         (maxyaxis (max (send self :range (1- (send self :num-variables)))))
         (y (second (monopoly-dens varx :xvals 50)))
         (maxy (max y))
         (x-at-maxy (select x (first (last (order y)))))
         (max-pix-now (second (send self :real-to-canvas x-at-maxy maxy)))
         (zero-pix (second (send self :real-to-canvas x-at-maxy 0)))
         (y (* y (/ (- zero-pix 50) (- zero-pix max-pix-now))))
         )
    (when (or (not (send self :use-color)) (= *color-mode* 0) )
                (setf color 'black))
    (send self :add-lines (append (repeat (list x) ndim) (list y)) 
          :color color :width line-width)
    ))

(defmeth histogram-proto :show-new-var (axis variable)
  (let* ((var-num (position variable (send self :variable-labels))))
    (send self :clear-lines :draw nil)
    (send self :content-variables var-num (- (send self :num-variables) 1))
    (send self :adjust-to-data)
    (when (send self :show-normal) (send self :add-normal))
    (when (send self :show-kernel) (send self :add-kernel (send self :kernel-type)))
    (when (send self :slot-value 'slider)
          (send (send self :slot-value 'slider) :value
                 (- (send self :num-bins) 2)))
    ))

(defmeth histogram-proto :make-show-variables-list ()
  (let* ((variables (combine (send self :variable-labels)))
         (cur-vars (list (first (send self :current-variables)))))
    (set-difference variables (select variables cur-vars))))

(defmeth histogram-proto :new-bins ()
  (let* ((loc (send self :location))
         (size (send self :frame-size))
         (slider nil)
         )
    (cond 
      ((< (send self :num-points) 5)
       (vista-message "Not available when the number of data values being plotted in the histogram is less than 5."))
      ((not (send self :slot-value 'slider))
       (setf slider (sequence-slider-dialog 
                     (iseq 2 (floor (/ (send self :num-points) 2)))
                     :title "Number of Bins"
                     :action #'(lambda (x)
                                 (send self :clear-lines) 
                                 (send self :num-bins x)
                                 (when (send self :show-normal)
                                       (send self :show-normal nil)
                                       (send self :switch-add-normal))
                                 (when (send self :show-kernel)
                                       (send self :show-kernel nil)
                                       (send self :switch-add-kernel)))))
                                   
       (send slider :location 
             (floor (- (+ (/ (first size) 2) (first loc)) 
                       (/ (first (send slider :size)) 2))) (second loc))
       (send slider :value (- (send self :num-bins) 2))
       (send self :add-subordinate slider)
       (send self :slot-value 'slider slider)
       )
      (t (setf slider (send self :slot-value 'slider))
         (send slider :show-window)))))

;;************************************************************************
;; scatterplot-proto methods
;;************************************************************************

(send scatterplot-proto :menu-template 
      '(LINK SHOWING-LABELS MOUSE RESIZE-BRUSH DASH 
             ERASE-SELECTION FOCUS-ON-SELECTION SHOW-ALL 
             SYMBOL COLOR ))

(defmeth scatterplot-proto :show-new-var (axis variable)
  (let* ((cur-var (send self :current-variables))
         (var-num (position variable (send self :variable-labels))))
    (if (equal axis "X") 
       (send self :current-variables var-num (second cur-var) :draw nil)
       (send self :current-variables (first cur-var) var-num :draw nil))
    (send self :adjust-to-data)))

(send scatterplot-proto :menu-title "Scatter")
(send scatterplot-proto :title "Scatterplot")

(defmeth scatterplot-proto :add-lowess ()
  (let* ((cur-var (send self :current-variables))
         (npts (send self :num-points))
         (varx (send self :point-coordinate (first cur-var) (iseq npts)))
         (vary (send self :point-coordinate (second cur-var) (iseq npts)))
         )
    (send self :add-lines (lowess varx vary))))


(defmeth scatterplot-proto :add-kernel-smooth ()
  (let* ((cur-var (send self :current-variables))
         (npts (send self :num-points))
         (varx (send self :point-coordinate (first cur-var) (iseq npts)))
         (vary (send self :point-coordinate (second cur-var) (iseq npts)))
         )
    (send self :add-lines (kernel-smooth varx vary :type 'u))))

(defmeth scatterplot-proto :add-spline ()
  (let* ((cur-var (send self :current-variables))
         (npts (send self :num-points))
         (varx (send self :point-coordinate (first cur-var) (iseq npts)))
         (vary (send self :point-coordinate (second cur-var) (iseq npts)))
         (maty (matrix (list npts 1) vary))
         (sortx (select varx (order varx)))
         (permy (combine (sort-and-permute varx maty)))
         )
    (break)
    (send self :add-lines (spline sortx permy))))

(defmeth scatterplot-proto :add-grid ()
         (let* ((rangex (send self :range 0))
                (rangey (send self :range 1))
                (minx (first  rangex))
                (maxx (second rangex))
                (miny (first  rangey))
                (maxy (second rangey)))
           (apply #'send self :draw-line 
                  (combine (send self :real-to-canvas minx 0) 
                           (send self :real-to-canvas maxx 0)))
           (apply #'send self :draw-line 
                  (combine (send self :real-to-canvas 0 miny) 
                           (send self :real-to-canvas 0 maxy)))
           (apply #'send self :draw-line 
                  (combine (send self :real-to-canvas minx maxy) 
                           (send self :real-to-canvas maxx maxy)))
           (apply #'send self :draw-line 
                  (combine (send self :real-to-canvas maxx miny) 
                           (send self :real-to-canvas maxx maxy)))
           ))

(defmeth scatterplot-proto :center-at-centroid (&key (draw t))
"Args: DRAW
Centers plot at centroid of points, using existing scale type."
 (let* ((scale-type (send self :scale-type))
         (numpts (send self :num-points))
         (numvar (send self :num-variables))
         (ranges (send self :range (iseq 0 (- numvar 1))))
         )
    (when (or (equal scale-type 'variable) (equal scale-type 'fixed))
          (mapcar 
           #'(lambda (i)
               (send self :center i
                     (mean (send self :point-coordinate i (iseq numpts)))
                     :draw nil)) (iseq numvar))
          (when draw (send self :redraw-content)))
    nil))

(defmeth scatterplot-proto :adjust-scatterplot-to-data 
             (scale-type &key (draw t))
"Args: SCALE-TYPE DRAW
Adjust scatterplot to show data. To be used for SCALE-TYPE of centroid-fixed or centroid-variable"
  (let* ((numpts (send self :num-points))
         (numvar (send self :num-variables))
         (ranges (send self :range (iseq 0 (- numvar 1))))
         (maxrange (max (- (min ranges)) (max ranges)))
         (gnr (get-nice-range (- maxrange) maxrange 5))
         (nice-min (first gnr))
         (nice-max (second gnr)) 
         )
       (mapcar 
        #'(lambda (i)
            (send self :center i
                  (mean (send self :point-coordinate i (iseq numpts)))
                  :draw nil)) (iseq numvar))
       (cond
         ((equal scale-type 'centroid-variable)
          (mapcar 
           #'(lambda (i)
               (send self :scaled-range i (- (sqrt numvar)) (sqrt numvar)
                     :draw nil)) (iseq numvar))
          (setf gnr (get-nice-range (- (sqrt numvar)) (sqrt numvar) 5))
          (send self :range (iseq numvar) (first gnr) (second gnr) :draw nil)
          
          )
         (t
          (send self :range (iseq numvar) nice-min nice-max :draw nil)))
    (send self :x-axis t t (third gnr))
    (send self :y-axis t t (third gnr))
    (when draw
          (send self :resize)
          (send self :redraw))
    nil))

;;************************************************************************
;; scatmat-proto methods
;;************************************************************************

(send scatmat-proto :menu-template 
      '(LINK SHOWING-LABELS MOUSE RESIZE-BRUSH DASH 
             ERASE-SELECTION FOCUS-ON-SELECTION SHOW-ALL SYMBOL COLOR ))

(defmeth scatmat-proto :do-new-variable-focus (x y m1 m2)
"Used by new spreadplots. Method to focus on variable subplots. Assumes that spreadplot object exists. A click on a diagonal cell selects one variable. A click on a subplot selects two variables. Shift-clicks select multiple variables."
  (when (not (send self :has-slot 'spin-var))
        (send self :add-slot 'spin-var)
        (defmeth self :spin-var (&optional (var-list nil set))
          (if set (setf (slot-value 'spin-var) var-list))
          (slot-value 'spin-var)))
  (let* ((cur-var (send self :current-variables))
         (spin-var (send self :spin-var))
         (nvar (send self :num-variables))
         (var-labs (send self :variable-label cur-var))
         (obs-nums (iseq (send self :num-points)))
         (cur-data  
          (mapcar #'(lambda (x) 
                      (send self :point-coordinate x obs-nums)) cur-var))
         (sp (send self :spreadplot-object))
         )
    (when (not m1)
          (setf spin-var cur-var)
       ;   (when (= (length spin-var) 2) ;fixes 1D receiver plots
       ;         (setf spin-var (reverse spin-var))
       ;         (setf var-labs (reverse var-labs))
       ;         (setf cur-data (reverse cur-data)))
          (send sp :update-spreadplot 
                0 0 
                spin-var
                (list var-labs cur-data))
          )
    (when m1
          (if (not spin-var) (setf spin-var cur-var))
          (setf cur-var (remove-duplicates cur-var))
          (setf spin-var (combine (adjoin spin-var cur-var)))
          (setf spin-var (remove-duplicates spin-var))
          (setf var-labs (send self :variable-label spin-var))
          (setf cur-data  
                (mapcar #' (lambda (x) 
                   (send self :point-coordinate x obs-nums)) spin-var))
          (send sp :update-spreadplot 
                0 0 
                spin-var
                (list var-labs cur-data)))
    (send self :spin-var spin-var)
    ))

(defmeth scatmat-proto :do-variable-focus (x y m1 m2)
"Used by old spreadplots. Method to focus on variable subplots. Assumes that three other plots exist.  These plots MUST be named scatterplot, spin-plot and histogram.  A click on a subplot sends it to scatterplot and sends its horizontal variable to histogram.  A click on a diagonal cell sends that variable to histogram.  Shift-clicks send first three selected variables to spin-plot as well."
  (let ((cur-var (send self :current-variables))
        (nvar (send self :num-variables))
        (scale-type (send scatterplot :scale-type)))
    (when (not m1)
          (when (/= (select cur-var 0) (select cur-var 1))
                (send scatterplot :current-variables 
                      (select cur-var 0) (select cur-var 1) :draw nil)
                (if scale-type 
                    (send scatterplot :redraw)
                    (send scatterplot :adjust-to-data)))
          (send histogram :current-variables 
                (select cur-var 0) nvar :draw nil) 
          (send histogram   :adjust-to-data)
          (setf spin-var ()))
    (when m1
          (when (= (select cur-var 0) (select cur-var 1))
                (setf cur-var (list (select cur-var 0))))
          (when (< (length spin-var) 3)
                (setf spin-var (adjoin (select cur-var 0) spin-var))
                (when (< (length spin-var) 3)
                      (if (equal (length cur-var) 2)
                          (setf spin-var 
                                (adjoin (select cur-var 1) spin-var)))))
          (when (= (length spin-var) 3)
                (setf spin-var (reverse spin-var))
                (send spin-plot 
                      :current-variables (select spin-var 0)
                      (select spin-var 1) (select spin-var 2)
                      :draw nil)
                (send scatterplot :current-variables 
                      (select spin-var 0) (select spin-var 1) :draw nil)
                (send histogram :current-variables 
                      (select spin-var 0) nvar :draw nil) 
                (let ((cur-var (send spin-plot :current-variables)))
                  (send spin-plot :set-variables-with-labels cur-var
                        (select (send spin-plot :variable-labels) cur-var))
                  (send spin-plot :redraw))
                (if scale-type 
                    (send scatterplot :redraw)
                    (send scatterplot :adjust-to-data))
                (send histogram   :adjust-to-data)
                (setf spin-var ())))))