;;########################################################################
;; function.lsp
;; contains new functions and modifications of Tierney functions.
;; Copyright (c) 1991-98 by Forrest W. Young
;;########################################################################

(defun clean-open-file-dialog (arg)
"Redraws workmap after doing open-file-dialog"
  (let ((f (open-file-dialog t)))
    (when (send *workmap* :gui) (send *workmap* :redraw))
    (when f (load f) (format t "; finished loading ~s~%" f))
    f))
  
(defun do-nothing ())

(defun radians (deg) (/ (* deg pi) 360))

(defun print-matrix-to-window (a window-object &key labels (decimals 2))
"Args: (matrix window-object &key labels (decimals 2))
Prints MATRIX in WINDOW-OBJECT in a nice format with DECIMALS places after the decimal, and with optional row labels when the :LABELS keyword is followed by a list of labels. Prints to stdio if window-object is nil. Modified version of the print-matrix function written by Luke Tierney. Modified by FWY 08/24/91. Modified by FWY for XLSP+ 7/19/94. Modified again by FWY 8/30/96 for decimals."
  (unless (matrixp a) (error "not a matrix - ~a" a)) 
   (let ((size 0)
         (sizea 0)
         (j 0)) 
     (dotimes (i (length (row a 0)))
      (if (not (stringp (first (coerce (col a i) 'list))))
       (setf size (max size (+ 4 (flatsize (round (max (abs (col a i))))))))
       (setf sizea (max (mapcar #'length (coerce (col a i) 'list))))))
     (dolist (x (row-list a))
            (let ((n (length x)))
              (dotimes (i n)
                       (let ((y (aref x i)))
                         (cond
                           ((integerp y)
                            (if window-object
                                (send window-object :paste-string
                                      (format nil "~vd" size y))
                                (format t "~vd" size y))
                            )
                           ((floatp y)
                            (if window-object
                                (send window-object :paste-string
                                   (format nil "~v,vf" (+ decimals size)
                                          decimals y))
                                (format t "~v,vf" (+ decimals size)
                                        decimals y))
                            )
                           (t 
                            (if window-object
                                (send window-object :paste-string
                                      (format nil "~va" sizea y))
                                (format t "~va" sizea y))
                            )))
                       (if (< i (- n 1))
                          (if window-object
                              (send window-object :paste-string
                                    (format nil " "))
                              (format t " "))
                              )
                       (if (and (not (equal labels nil)) (= i (- n 1)))
                           (if window-object
                               (send window-object :paste-string
                                     (format nil " ~a" (select labels j)))
                               (format t " ~a" (select labels j)))
                               )
                       )
              (setf j (+ j 1))
                       (if window-object
                           (send window-object :paste-string 
                                 (format nil "~%"))
                           (format t "~%"))
                           ))
    nil))

;;
;;next function redefined in displayw.lsp
;;

(defun report-header (title)
"Function args :title
Macintosh: Opens a display window TITLE and returns display window object-id.
Dos/Unix:  Writes TITLE to listener and returns nil"
  (let ((w nil))
#+macintosh(setf w (send *vista* :report-window title))
#-macintosh(display-string 
            (format nil "~2% ~a ****************~2%" title))
    w))

(defmeth vista-system-object-proto :report-window (title)
"Function args :title
Creates a new report window on the macintosh."
  (let ((w (send display-window-proto :new 
                 :title title
                 :size *text-window-size*
                 :location (+ (list 0 20) *text-window-location*)))
        (L (send self :report-window-id-list))
        ) 
    (if L  (send self :report-window-id-list (combine L w))
           (send self :report-window-id-list (list w)))
    (defmeth w :remove () 
      (send *vista* :report-window-id-list
            (remove w (send *vista* :report-window-id-list)))
      (send w :close))
    w))

(defun display-string (string &optional (w nil))
"Function args: (string)
Writes string to display-window w if w specified and if system is Macintosh, otherwise writes string to standard output. Forrest Young 10/17/91"
  #+macintosh (if w (send w :paste-string string) (princ string))
  #-macintosh (princ string)
  nil)

(defun vector-origins (graphical-object 
                        &optional (var-indices 
                                   (iseq (send graphical-object
                                                       :num-variables))) )
"A function which can be used to find the origin wrt to any (sub)set of variables in any graph. Produces a list of centers for the specified variables. Primarily useful for adding vectors to a plot."
  (split-list (send graphical-object :center var-indices) 1) )

(defun row-list2 (matrix &key (list nil))
"Args: MATRIX &KEY (LIST NIL)
Row-list function which can optionally produce its output as a list of lists."
  (if list 
      (mapcar #'(lambda (x) (coerce x 'list)) (row-list matrix))
      (row-list matrix) ) )

(defun column-list2 (matrix &key (list nil))
"Args: MATRIX &KEY (LIST NIL)
Column-list function which can optionally produce its output as a list of lists."
  (if list 
      (mapcar #'(lambda (x) (coerce x 'list)) (column-list matrix))
      (column-list matrix) ) )

(defun bind-rows2 (items &rest args)
"Args: ITEMS &REST ARGS
Bind-rows function which can optionally operate on lists-of-items, as well as separate lists of items."
  (let (
        (items2 (if (some #'compound-data-p items)
                    (apply #'bind-rows items)
                    (bind-rows items)) )
       )
    (cond ( (null args) items2 )
          ( (some #'compound-data-p args)
            (bind-rows items2 (apply #'bind-rows2 args)) )
          ( t (bind-rows items2 args) ) ) ) )

(defun bind-columns2 (items &rest args)
"Args: ITEMS &REST ARGS
Bind-columns function which can optionally operate on lists-of-items, as well as separate lists of items."
  (let (
        (items2 (if (some #'compound-data-p items)
                    (apply #'bind-columns items)
                    (bind-columns items)) )
       )
    (cond ( (null args) items2 )
          ( (some #'compound-data-p args)
            (bind-columns items2 (apply #'bind-columns2 args)) )
          ( t (bind-columns items2 args) ) ) ) )

(defun row (a i &key (list nil))
"Args: (matrix row-number)
 Takes a matrix and returns the row specified by row-number. If the keyword
 argument :LIST is set to t, the row is returned as a list." 
  (select (row-list2 a :list list) i) )

(defun col (a i &key (list nil)) 
 "Args: (matrix column-number)
 Takes a matrix and returns the column specified by column-number. If the
 keyword argument :LIST is set to t, the row is returned as a list." 
  (select (column-list2 a :list list) i))

(defun lists-to-matrix (list-of-lists)
 "Args: (list-of-list)
 Takes a list of lists and returns a matrix with columns equal to the lists."
  (apply #'bind-columns list-of-lists)  )

(defun fuzz (numeric-arg &optional (num-decimal-places 2))
"Args: (numeric-arg &optional num-decimal-places)
 Fuzzes the precision of any type of numeric argument to at most num-decimal-places decimal places.  Default num-decimal-places = 2."
  (let ( (factor (^ 10 num-decimal-places)) )
    (/ (round (* factor numeric-arg)) factor) ) )


(defun pm (matrix &optional (decimals 2))
"Args: (matrix &optional (decimals 2))
Prints a matrix with precision specified by decimals (default is 2). Matrix may contain string values." 
  (print-matrix-to-window matrix nil :decimals decimals) 
  t)

(defun pms (matrix-list) 
"Args: MATRIX-LIST
Prints each matrix in a list of matrices using the pm function."
  (mapcar #'pm matrix-list) 
  t)

(defun gensym2 (&optional symbol-name val)
"Args: SYMBOL-NAME VAL
Redefines gensym to return a particular string with a particular value. If the two optional arguments are not specified, the function just performs a GENSYM."
  (if val 
      (let* (
            (x (gensym 0) )
            (y (dotimes (i (- val 1)) (gensym)) )

           )
             (gensym symbol-name) )
           (if symbol-name 
               (gensym symbol-name)
               (gensym)) ) )

(defun $= (x y) 
"Args: X Y
 A vectorized character comparison function.  Works like =, only for character arguments"
(map-elements #'equalp x y))

(defun search-string (string-a string-b)
"Args: STRING-A STRING-B
Search for first occurance of STRING-A (which must be a single character string) in STRING-B.  If found, return position index.  If not found, return nil. (Note that XLisp has no search function)."
  (let ((result nil)
        )
  (dotimes (i (length string-b))
           (when (char= (char string-a 0) (char string-b i))
                 (setf result i)
                 (return)))
    result))

(defun substitute-string (string-a string-b position)
"Args: STRING-A STRING-B POSITION
Substitute STRING-A into STRING-B at POSITION. (Note that XLisp has no substitute function)."
  (concatenate 'string 
               (subseq string-b 0 position)
               string-a
               (subseq string-b (+ position 1))))

(defun blanks-to-dashes (string)
"Arg: STRING
Convert all blanks in STRING to dashes, truncating terminating blank."
  (let ((position (search-string " " string))
        (new-string string))
    (cond 
      (position (setf new-string (substitute-string "-" string position))
                (when (equal "-" (subseq new-string (1- (length new-string))))
                      (setf new-string 
                            (subseq new-string 0 (1- (length new-string)))))
                (blanks-to-dashes new-string))
      (t string))))

(defun remove-period (string)
"Args: STRING
Removes period and following characters from a string. Returns string up to period, or entire string if none. [function.lsp]"
  (let ((position (search-string "." string)))
    (if position (subseq string 0 position) string)))

(defun $position (string-a string-b)
"Args: (string-a string-b)
String-a and string-b are lists of strings.  Returns a list whose length
equals the number of occurances of the elements of string-a in string-b.  
The list contains the position of each element of string-a in string-b 
(ignoring the case of the strings).  If an element of string-a is not in 
string-b, a nil element is returned."
  (let ((positions nil)
        )
    (dotimes (i (length string-a))
             (setf positions 
                   (combine positions 
                            (which ($= (select string-a i) string-b)))))
    (rest positions)
    ))

(defun not2 (x) (map-elements #'not x))

(defun unique-values (input-list)
  (remove-duplicates input-list) )

(defun number-from-string (string)
"Args: STRING
Converts a number represented as a string into a number."
  (eval (read (make-string-input-stream string) nil)))

(defun quit ()
 (cond 
   (*current-object* 
    (when (send (save-exit-dialog-box) :modal-dialog)
          (exit)))
   (t
    (prepare-to-quit)
    (exit))))

(defun save-exit-dialog-box () 
      (let* ((text1 (send text-item-proto :new
                    "Save ViSta data or models?"))
#-msdos(quit-exit-string "Quit")
#+msdos(quit-exit-string "Exit")
             (text2 (send text-item-proto :new (format nil
                   "(To Save, click Save, then~%use the save menu items.)")))
             (quit (send modal-button-proto :new quit-exit-string :action 
                     #'(lambda ()
                         (prepare-to-quit)
                         t)
                         ))
             (save (send modal-button-proto :new "Save" )))
        (send modal-dialog-proto :new
              (list text1 (list save quit) text2)
              :default-button save)))

(defun prepare-to-quit ()
  (close-all-plots)
  (when (send *vista* :help-window-object)
        (send (send *vista* :help-window-object) 
              :close)
        (send *vista* :help-window-object nil))
  (let* ((win-list 
          (send *vista* :report-window-id-list))
         (nw (length win-list))) 
    (when win-list
          (dotimes 
           (i nw)
           (send (select win-list i) :remove))))
  )

;;
;;next two functions redefined in displayw.lsp
;;

(defun file-to-stream (filename title &optional (out-stream *standard-output*))
  (gc)
  (if (equal title "Bug List")
      (format out-stream "~3%**************** ~a  ****************~2%" title)
      (format out-stream "~3%****************  Help for ~a  ****************~2%" title))
  (with-open-file (in-stream filename :direction :input)
     (let ((char nil))
       (loop ;loop until eof
          (if (setq char (read-char in-stream nil nil));returns nil on eof
              (write-char char out-stream)
              (return nil)))))
  (terpri))

;fwy 4.28 7/16/97 modified for show help
(defun file-to-window (filename title w)
  (with-open-file (g filename)
                  (when (not w) 
                        (setf w (send display-window-proto :new))
                        (send *vista* :help-window-object w)
                        (send *vista* :help-window-status t)
                        (send w :location 26 154)
                        (setf *help-window* w)
                        (defmeth w :remove ()
                          (send self :hide-window)
                          (send *vista* :help-window-status nil)
                          (when (send command-menu-show-help-item :mark)
                                (send *vista* :show-help nil)
                                (send command-menu-show-help-item :mark nil))
                          ))
                  (send w :flush-window)
                  (if (equal title "Bug List")
                      (send w :title title)
                      (send w :title (strcat "Help for " title)))
                  (send w :paste-stream g)
                  (send w :show-window)
                  (when (not (send *vista* :help-window-status)) 
                        (send w :show-window)
                        (send *vista* :help-window-status t))))

(defun make-matrix-list (matrix-list matrix)
"Args: MATRIX-LIST MATRIX
Adds MATRIX as a new element at the end of MATRIX-LIST."
  (let* ((n (length matrix-list))
         (new-matrix-list (make-list (+ 1 n) :initial-element nil)))
    (cond ((> n 0)
       (dotimes (i n)
                (setf (select new-matrix-list i) (select matrix-list i)))
       (setf (select new-matrix-list n) matrix))
      ((= n 0) (setf new-matrix-list (list matrix))))
    new-matrix-list))


(defun select2 (seq iseq)
"Args: Extends 3.46 select function to work on strings. Not needed for 3.50."
  (if (> xls-minor-release 49) (select seq iseq)
      (if iseq
          (if (sequencep iseq)
              (subseq seq (first iseq) (1+ (first (last iseq))))
              (select seq iseq))
          nil)))

(defun add-element-to-list (list element)
"Args: LIST ELEMENT
Adds ELEMENT as a new element to the end of LIST, creating a one element list when LIST is nil (there must be a much better way to do this)."
  (let* ((n (length list))
         (new-list (make-list (+ 1 n) :initial-element nil)))
    (cond ((> n 0)
           (dotimes (i n)
                    (setf (select new-list i) (select list i)))
           (setf (select new-list n) element))
      ((= n 0) (setf new-list (list element))))
    new-list))

(defmeth regression-model-proto :display (&optional (w nil))
"Message args: ()
Prints the least squares regression summary. Variables not used in the fit
are marked as aliased. Modified by Forrest Young from Luke Tierney's original version to include t-ratios and p-values and to display in window W when W not nil."
  (let* ((coefs (coerce (send self :coef-estimates) 'list))
         (se-s (send self :coef-standard-errors))
         (t-ratio nil)
         (p-value nil)
         (x (send self :x))
         (df (send self :df))
         (totaldf (- (send self :num-cases) 1))
         (modeldf (- (length (send self :coef-estimates)) 1))
         (errordf (- totaldf modeldf))
         (totalss (send self :total-sum-of-squares))
         (errorss (send self :sum-of-squares))
         (modelss (- totalss errorss))
         (r-squared (send self :r-squared))
         (adj-r-squared (+ 1 (* (- r-squared 1) 
                                (/ (- (send self :num-cases) 1) df))))
         (p-names (send self :predictor-names)))
    (if (send self :weights) 
        (display-string 
         (format nil "~%PARAMETER ESTIMATES (WEIGHTED LEAST SQUARES):") w)
        (display-string 
         (format nil "~%PARAMETER ESTIMATES (LEAST SQUARES)") w))
    (display-string (format nil " WITH TWO-TAILED T-TESTS.") w)
    (display-string (format nil "~%Term                     Estimate  Std. Error    t-Ratio  P-Value~%") w)
    (when (send self :intercept)
          (setf t-ratio (/ (car coefs) (car se-s)))
          (setf p-value (* 2 (- 1 (t-cdf (abs t-ratio) df))))
          (if (< p-value .0001) 
              (setf p-value "<.0001")
              (setf p-value (fuzz p-value 4)))
          (display-string (format nil "Constant               ~10,2f  ~10,2f ~10,2f   ~6f~%"
                  (car coefs) (car se-s) t-ratio p-value) w)
          (setf coefs (cdr coefs))
          (setf se-s (cdr se-s)))
    (dotimes (i (array-dimension x 1)) 
             (cond 
               ((member i (send self :basis))
                (setf t-ratio (/ (car coefs) (car se-s)))
                (setf p-value (* 2 (- 1 (t-cdf (abs t-ratio) df))))
                (if (< p-value .0001) 
                    (setf p-value "<.0001")
                    (setf p-value (fuzz p-value 4)))
                (display-string (format nil "~22a ~10,2f  ~10,2f ~10,2f   ~6f~%"
                        (select p-names i) (car coefs) (car se-s)
                        t-ratio p-value) w)
                (setf coefs (cdr coefs) se-s (cdr se-s)))
               (t (display-string (format nil "~22a    aliased~%" (select p-names i)) w))))
    (display-string 
     (format nil "~%SUMMARY OF FIT:~%") w)
    (display-string 
     (format nil "R Squared:             ~10,2f~%" r-squared) w)
    (display-string 
     (format nil "Adjusted R Squared:    ~10,2f~%" adj-r-squared) w)
    (display-string 
     (format nil "Sigma hat (RMS error): ~10,2f~%" (send self :sigma-hat)) w)
    (display-string 
     (format nil "Number of cases:       ~10d~%" (send self :num-cases)) w)
    (if (/= (send self :num-cases) (send self :num-included))
        (display-string 
         (format nil "Number of cases used:  ~10d~%" 
                 (send self :num-included)) w))
    (display-string (format nil "Degrees of freedom:    ~10d~%" df) w)
    (display-string (format nil "~%ANALYSIS OF VARIANCE: MODEL TEST") w)
    (display-string (format nil "~%Source             Sum-of-Squares   df  Mean-Square    F-Ratio   P-Value~%") w)
    (send self :anova-table-line w "Model" modelss modeldf errorss errordf)
    (send self :anova-table-line w "Error" errorss errordf)
    (send self :anova-table-line w "Total" totalss totaldf)
    (display-string (format nil "~%") w)))

(defmeth regression-model-proto :anova-table-line 
  (w source modelss &optional modeldf errorss errordf)
  (let* ((modelms nil)
         (pstring nil)
         (f nil)
         (p nil))
    (display-string (format nil "~22a ~10,2f " source  modelss) w)
    (when (not (equal source "Unique"))
          (display-string (format nil "~4d" modeldf) w))
    (when (not (or (equal source "Total") (equal source "Unique"))) 
          (setf modelms (/ modelss modeldf))
          (display-string (format nil "~13,2f" modelms) w))
    (when errorss 
          (setf f (/ modelms (/ errorss errordf)))
          (setf p (fuzz (- 1 (f-cdf f modeldf errordf)) 5))
          (setf f (fuzz f 5))
          (when (< p .0001) (setf pstring "   <.0001"))
          (if pstring
              (display-string (format nil "~11,2f ~9a" f pstring) w)
              (display-string (format nil "~11,2f ~9f" f p) w)))
    (display-string (format nil "~%") w)))

(defmeth workmap-proto :frame-poly (poly &optional (from-origin t))
  (if from-origin
      (call-next-method poly from-origin)
      (call-next-method (cumsum poly) t)))

(defun sort-and-permute (sort-by-variable data-matrix &optional descending)
"Args: SORT-BY-VARIABLE DATA-MATRIX &optional DESCENDING
Sorts elements of SORT-BY-VARIABLE into ascending (DESCENDING if t) order and permutes all values in each column of DATA-MATRIX into corresponding order. Returns list of permuted data."
  (let ((labels (iseq (length sort-by-variable))))
    (first (sort-and-permute-dob 
            data-matrix labels sort-by-variable descending))))

(defun sort-and-permute-dob (data-matrix labels sort-by-variable descending)
"Args: DATA-MATRIX LABELS SORT-BY-VARIABLE DESCENDING
Sorts elements of SORT-BY-VARIABLE into ascending (DESCENDING if t) order and permutes LABELS and all columns in DATA-MATRIX into corresponding order. Returns list of permuted data and labels."
  (let* ((rows-of-data-matrix (row-list data-matrix))
         (order-var (order sort-by-variable))
         (permuted-data nil)
         (permuted-labels nil)
         (permuted-data-matrix nil)) 
    (when descending (setf order-var (reverse order-var)))
    (setf permuted-data (select rows-of-data-matrix order-var))
    (setf permuted-labels (select labels order-var))
    (setf permuted-data-matrix (matrix (size data-matrix)
                                       (combine permuted-data)))
    (list permuted-data-matrix permuted-labels)))


(defun rank-with-ties (sequence)
"Args: sequence
Ranks elements in sequence, creating mean rank for tied elements. Ranks are 1-based so that lowest rank is 1 (not 0)"
  (let* ((ordered (select sequence (order (rank sequence))))
         (n (length ordered))
         (ntied 1)
         (ranksum 1)
         (meanrank nil)
         (tiedranks (repeat nil n))
         (k 0)
         )
    (dotimes (i n) 
             (cond
               ((= i (- n 1))
                (setf meanrank (/ ranksum ntied))
                (dotimes (j ntied)
                         (setf (select tiedranks k) meanrank)
                         (setf k (1+ k))))
               ((= (select ordered i) (select ordered (1+ i)))
                (setf ntied (1+ ntied))
                (setf ranksum (+ ranksum i 2)))
               (t
                (setf meanrank (/ ranksum ntied))
                (dotimes (j ntied)
                         (setf (select tiedranks k) meanrank)
                         (setf k (1+ k)))
                (setf ntied 1)
                (setf ranksum (+ 2 i)))))
    
    (select tiedranks (rank sequence)) )) ; fwy 4.28
;was(- (select tiedranks (rank sequence)) 1))) 

(defun yes-or-no-dialog (title text &optional (yes t))
"Args: TITLE STRING &optional YES
Presents Yes-No dialog with title TITLE and text TEXT. Default button is YES unless YES=NIL"
  (let* ((result nil)
         (title-item  (send text-item-proto :new title))
         (text-item   (send text-item-proto :new text))
         (yes-button  (send modal-button-proto :new "Yes" :action
                            (lambda () t)))
         (no-button   (send modal-button-proto :new "No"  :action
                            (lambda () nil)))
         (report-dialog nil))
    (if yes
        (setf report-dialog 
              (send modal-dialog-proto :new
                    (list title-item text-item (list yes-button no-button))
                    :default-button yes-button))
        (setf report-dialog 
              (send modal-dialog-proto :new
                    (list title-item text-item (list no-button yes-button))
                    :default-button no-button)))
    (send report-dialog :modal-dialog)))

(defun string-downcase-if-not-X11 (string)
"Function Args: string
Does string downcase if the X11 feature not present"
#-X11 (string-downcase string)
#+X11 string
  )

(defun compile-vista-file (file)
"Function Args: FILE
Compiles FILE in directory *code-dir-name*. FILE must have no extension."
  (compile-file (strcat *code-dir-name* file ".lsp")
                :output-file (strcat *code-dir-name* file ".fsl")
                :print t :verbose t))

(defun compile-vista-files (list)
    (mapcar #'compile-vista-file list))

(defun compile-vista-base ()
  (let ((vista-base-list '("vista" "vismenu1" "vismenu2" "iconobj1" "iconobj2" "workmap1" "workmap2" "workmap3" "systmob1" "systmob2" "systmob3" "dataobj1" "dataobj2" "dataobj3" "dataobj4" "datasim"  "graphic0" "graphic1" "spinplot" "boxplot1" "boxplot2" "qplotobj" "datavis"  "dashobj1" "dashobj2" "dashobj3" "dashobj4" "modelobj" "tranobj1" "tranobj2" "function" "statfunc" "sprdplot" "generic"  "displayw" "overlay"  "graphelp")))
    (mapcar #'compile-vista-base-file vista-base-list)))

(defun compile-vista-base-file (file)
"Function Args: FILE
Compiles FILE in directory *base-dir-name*. FILE must be a string filename with no extension. Places compiled result in directory *base-dir-name* and in *code-dir-name*"
  (compile-file (strcat *base-dir-name* file ".lsp")
                :output-file (strcat *base-dir-name* file ".fsl")
                :print t :verbose t)
  (with-open-file 
   (in-file (strcat *base-dir-name* file ".fsl"))
   (with-open-file 
    (out-file (strcat *code-dir-name* file ".fsl") :direction :output)
    (loop
     (setf line (read-line in-file nil nil))
     (when (not line) (return))
     (format out-file "~a~%" line))
    (format t "; writing file ~a~%~%"(strcat *code-dir-name* file ".fsl"))
    ))
  )

(defun initialize-vista-workspace ()
  (let ((separator
         #+macintosh ":"
         #+msdos "\\"
         #+X11 "/"
         ))
    (setf *vista-dir-name* (strcat (get-working-directory) "ViSta" separator))
    (setf *help-dir-name*  (strcat *vista-dir-name* "Help" separator))
    (setf *guide-dir-name* (strcat *vista-dir-name* "Guidance" separator)))
  (menus t)
  (show-workmap)
  (move-listener))

#-X11(defun move-listener ()
  (send *listener*  :hide-window)
  (send *listener*  :flush-window)
  (apply #'send *listener*  :size '(490 90))
  (apply #'send *listener*  :location *text-window-location*)
  (send *listener* :title "LispStat")
  (princ about)
  (terpri)
  (show-Xlisp-Stat)
  )
#+X11(defun move-listener () ())

(defun save-vista-workspace ()
  (setf *startup-functions* 
        (combine *startup-functions* 'initialize-vista-workspace))
  (save-workspace (strcat *startup-dir* "ViSta.wks")))

(defun run-time () 
"Returns run time in minutes"
  (/ (get-internal-real-time) internal-time-units-per-second 60))

(defun fatal-message (string)
  (error-message string nil t)
  (error string)
  )

(defun vista-message (text &key (title "ViSta Message") (beep t) 
                           size location frame-location)
"Args: text &key (title \"ViSta Message\") (beep t) size location
Displays TEXT in window of SIZE at LOCATION with TITLE. Beeps"
  (if beep (sysbeep))
  (if size (apply #'send *message-window* :size size))
  (if location
      (apply #'send *message-window* :location location)
      (apply #'send *message-window* :location frame-location)
      )
  (if frame-location 
      (apply #'send *message-window* :frame-location frame-location))
  (send *message-window* :show-message text title))

(defun about-message (text &key (title "About These Data") (beep t) 
                           size location frame-location)
"Args: text &key (title \"About These Data\") (beep t) size location
Displays TEXT in window of SIZE at LOCATION with TITLE. Beeps"
  (if beep (sysbeep))
  (if size (apply #'send *about-window* :size size))
  (if location
      (apply #'send *about-window* :location location)
      (apply #'send *about-window* :location frame-location)
      )
  (if frame-location 
      (apply #'send *about-window* :frame-location frame-location))
  (send *about-window* :show-message text title))

(provide "function")
  
