;;###########################################################################
;; dataobj3.lsp
;; Copyright (c) 1991-96 by Forrest W. Young
;; This file contains methods for reporting simple univariate statistics. 
;; Together, dataobj1.lsp, dataobj2.lsp, and dataobj3.lsp contain
;; all the code to implement multivariate data objects.
;;###########################################################################

(defmeth mv-data-object-proto :merge-variables (&optional name)
"Method Args: (&optional name)
Horizontally merges active variables of two data objects which have the same number of observations. The current and the previous data object are merged. The new data object is named NAME (a string) if specified, otherwise a dialog is presented for name. Returns object identification of the new data object."
  (if (not (eq current-object self)) (setcd self))
  (let ((object nil)
        (prev-data previous-data)
        (prev-data-icon (send previous-data :icon-number))
        (menu-name nil)
        )
    (cond 
      ((/= (send self :active-nobs) (send previous-data :active-nobs))
           (error-message "Data Objects cannot be merged because they do not have the same number of observations."))
      (t
       (if name
           (setf menu-name name)
           (setf menu-name 
                 (get-string-dialog "Please Name the Created Data Object:"
                                    :initial "Unnamed")))
       (cond 
         (menu-name
          (setf object
               (data menu-name
                     :created (send *desktop* :selected-icon)  
                     :data (combine (bind-columns
                           (send previous-data :active-data-matrix '(all)) 
                           (send self  :active-data-matrix '(all))))
                     :variables (concatenate 'list
                           (send previous-data :active-variables '(all))
                           (send self  :active-variables '(all)))
                     :types (concatenate 'list
                           (send previous-data :active-types '(all))
                           (send self  :active-types '(all)))
                     :labels (send self :active-labels)))
          (send object :title 
                (concatenate 'string "Merger of " (send prev-data :title)
                             " with " (send self :title)))
          (send *desktop* :connect-icons 
                (- prev-data-icon 1)
                (- (send *desktop* :num-icons) 1) :new t)
          (send object :dob-parents 
                (add-element-to-list (send object :dob-parents) prev-data))
          (send prev-data :dob-children
                (add-element-to-list (send prev-data :dob-children) object))
          ))))
       object))

(defmeth mv-data-object-proto :merge-observations (&optional name)
"Method Args: (&optional name)
Vertically merges active observations of two data objects which have the same number of variables. The current and the previous data object are merged. The new data object is named NAME (a string) if specified, otherwise a dialog is presented for name. Returns object identification of the new data object."
  (if (not (eq current-object self)) (setcd self))
  (let ((object nil)
        (prev-data previous-data)
        (prev-data-icon (send previous-data :icon-number))
        (menu-name nil)
        )
    (cond 
      ((/= (send self :active-nvar '(all)) 
           (send previous-data :active-nvar '(all)))
           (error-message "Data Objects cannot be merged because they do not have the same number of active variables."))
      (t
       (if name
           (setf menu-name name)
           (setf menu-name 
                 (get-string-dialog "Please Name the Created Data Object:"
                                    :initial "Unnamed")))
       (cond 
         (menu-name
          (setf object
               (data menu-name
                     :created (send *desktop* :selected-icon)  
                     :data (combine (bind-rows
                           (send previous-data :active-data-matrix '(all)) 
                           (send self  :active-data-matrix '(all))))
                     :variables (send self :active-variables '(all))
                     :types     (send self :active-types     '(all))
                     :labels (concatenate 'list
                             (send previous-data :active-labels)
                             (send self :active-labels))))
          (send object :title 
                (concatenate 'string "Merger of " (send previous-data :title)
                             " with " (send self :title)))
          (send *desktop* :connect-icons 
                (- prev-data-icon 1)
                (- (send *desktop* :num-icons) 1) :new t)
          (send object :dob-parents 
                (add-element-to-list (send object :dob-parents) prev-data))
          (send prev-data :dob-children
                (add-element-to-list (send prev-data :dob-children) object))
          ))))
       object))

(defmeth mv-data-object-proto :current-labels ()
"Method Args: none 
Returns a list of indices of the active observations. Active means their labels are (or were when the window was closed) visible in the observation window, or, if any labels are selected, visible AND selected."
  (let* ((nobs (send self :nobs))
         (states (send self :obs-states))
         (selected-labels 
          (which (mapcar #'equal (repeat 'SELECTED nobs) states))))
    (when (equal selected-labels nil) 
          (setf selected-labels (which (mapcar #'not (mapcar #'equal 
                  (repeat 'INVISIBLE nobs) states)))))
    selected-labels))

(defmeth mv-data-object-proto :current-variables (ok-types)
"Method Args: (oktypes) 
Takes a list of variable types and returns a list of indices
of the variables of those types which are also active.  Active means they
are (or were when the window was closed) visible in the variable window 
(if any variables are selected, visible AND selected). Oktypes can be Label, 
Category, Ordinal and Numeric, or All (which means select all types)."
  (let* ((nvar (send self :nvar))
         (states (send self :var-states))
         (selected-variables 
          (which (mapcar #'equal (repeat 'SELECTED nvar) states)))
         (types  (send self :types))
         (category-variables ())
         (label-variables ())
         (ordinal-variables ())
         (numeric-variables ())
         )
    (setf types (mapcar #'string-downcase types))
    (if (member 'Category ok-types) 
        (setf category-variables 
         (which (mapcar #'equal (repeat '"category" nvar) types))))
    (if (member 'label ok-types) 
        (setf label-variables 
         (which (mapcar #'equal (repeat '"label"    nvar) types))))
    (if (member 'ordinal ok-types) 
        (setf ordinal-variables 
         (which (mapcar #'equal (repeat '"ordinal"  nvar) types))))
    (if (member 'numeric ok-types) 
        (setf numeric-variables 
         (which (mapcar #'equal (repeat '"numeric"  nvar) types))))
    (if (member 'all ok-types)
        (setf numeric-variables (iseq nvar)))
    (if (equal selected-variables nil) 
        (setf selected-variables
             (which (mapcar #'not (mapcar #'equal 
                    (repeat 'INVISIBLE nvar) states)))))
    (setf selected-variables 
        (intersection selected-variables 
           (union category-variables 
                  (union label-variables 
                         (union ordinal-variables numeric-variables)))))
    (if selected-variables (sort-data selected-variables)
        nil)))

(defun summarize-data 
  (&key moments quartiles ranges correlations covariances dialog)
"Function Args:(&key moments quartiles ranges correlations covariances dialog)
Prints summary statistics for the active data.  If dialog is t a dialog box is presented to determine which statistics are to be printed, otherwise the other keywords determine which are printed."
  (send current-data :summary 
        :moments moments :quartiles quartiles :ranges ranges
        :correlations correlations :covariances covariances
        :dialog dialog)
  t)

(defmeth mv-data-object-proto :summary 
  (&key moments quartiles ranges correlations covariances dialog)
"Method Args: (&key moments quartiles ranges correlations covariances dialog)
Prints summary statistics for the active data.  If dialog is t a dialog box is presented to determine which statistics are to be printed, otherwise the other keywords determine which are printed."
  (if (not (eq current-object self)) (setcd self))
  (let ((summary-options nil))
    (if dialog
        (setf summary-options 
              (choose-subset-dialog 
               "Choose Summary Statistics:"
               '("Moments (Mean, StDv, etc.)" 
                 "Quartiles (5 Number Summary)" 
                 "Range, Interquartile Range"
                 "Correlation Matrix"
                 "Covariance Matrix")
               :initial (select (send self :summary-option-states) 0))
              )
        (setf summary-options (list (delete 'nil (list
             (when moments '0) (when quartiles '1)
             (when ranges  '2) (when correlations '3)
                               (when covariances '4)))))
        )
    (when (and (> (send self :active-nvar '(category)) 0)
               (> (send self :active-nvar '(numeric ordinal)) 1))
          (when (not (send self :vis-error-msg "summarize")) (top-level)))
    (when summary-options
          (when (not (select summary-options 0))
                (setf summary-options (send self :summary-option-states)))
          (when summary-options 
                (when (select summary-options 0)
                      (send self :summary-option-states summary-options)
                      (if (send self :active-types '(numeric ordinal))
                          (send self :describe-data 
                                (column-list
                                 (send self :active-data-matrix '(all)))
                                (if (send self :matrices)
                                    (send self :matrices)
                                    (send self :active-variables '(all)))
                                summary-options)
                          (error-message "There are no active numeric or ordinal variables.")
                          ))))
    (when (and (> (send self :active-nvar '(category)) 0)
               (= (send self :active-nvar '(numeric ordinal)) 1))
          (when (send self :mv-to-table) 
                (message-dialog 
                 "ViSta Message:~2%Your data are being converted to table~%~
                                   data for a more complete summary. Two~%~
                                   separate summaries will be provided.~%")
                (summarize-data 
                 :moments (not (not (member 0 (first summary-options)))) 
                 :quartiles (not (not (member 1 (first summary-options)))) 
                 :ranges (not (not (member 2 (first summary-options)))))
                 ))
    ))

(defmeth mv-data-object-proto :describe-data (variables varnames stats)
"Method Args: (data-matrix varnames stats)
Used by summary method to compute and print desired summary statistics for columns of DATA-MATRIX.  VARNAMES contains strings that are column names.
STATS is a list with up to five elements indicating stats to be printed:
0=moments, 1=quantiles, 2=ranges, 3=correlations, 4=covariances."
  (let* ((data-matrix nil)
         (nactvar (length variables))
         (w nil) 
         (test (repeat 0 5))
         (varnamenow nil)
         (novarvar nil)
         (varn nil)
         (varnow  nil) 
         (varmean nil) 
         (varstdv nil)
         (varvari nil) 
         (varskew nil) 
         (varkurt nil)
         (var5num nil) 
         (iqrange nil) 
         (range   nil))
    (setf w (report-header (strcat (send self :name) " Summary Statistics")))
    (setf stats (select stats 0))
    (dotimes (i (length stats)) (setf (select test (select stats i)) 1))
    (display-string (format nil   "Title: ~a" (send self :title)) w)
    (display-string (format nil "~%Data:  ~a ~2%" (send self :name)) w)
    (when (= 1 (select test 0))
          (if (send self :ways)
              (display-string (format nil "CELL NAMES         ") w)
              (display-string (format nil "VARIABLES (Numeric)") w))
          (display-string (format nil 
          "       MEAN      StDv   VARIANCE  SKEWNESS  KURTOSIS    N") w)
          (dotimes (i nactvar)
             (when 
              (or (send self :matrices)
                  (send self :ways)
                  (and (not (send self :matrices))
                       (equalp "numeric" 
                               (select (send self :active-types '(all)) i))))
              (setf varnamenow (select varnames i))
              (when (> (length varnamenow) 20)
                    (setf varnamenow (subseq varnamenow 0 20)))
              (setf varnow  (select variables i))
              (setf varn (length varnow))
              (setf varmean (mean varnow))
              (setf varstdv 0)
              (display-string 
               (format nil "~%~20a ~9,2f " 
                       varnamenow (+ .00001 varmean) ) w) 
              (when (> varn 1) (setf varstdv (standard-deviation varnow)))
              (when (= varn 1) (display-string (format nil "    N equals 1.  ") w))
              (cond 
                ((= 0 varstdv)
                 (setf novarvar t)
                 (display-string 
                  (format nil "Moment Statistics Undefined.") w))
                (t
                 (cond ((> varn 1)
                        (display-string (format nil "~9,2f ~9,2f "
                           (+ .00001 varstdv) (+ .00001 (^ varstdv 2))) w))
                   (t (display-string (format nil "    undef     undef ") w)))
                 (cond ((and (> varn 2) (> varstdv 0))
                        (setf varskew (skewness varnow))
                        (display-string (format nil "~9,2f " 
                                                (+ .00001 varskew)) w))
                   (t (display-string (format nil "    undef ") w)))
                 (cond ((and (> varn 3) (> varstdv 0))
                        (setf varkurt (kurtosis varnow))
                        (display-string (format nil "~9,2f " 
                                                (+ .00001 varkurt)) w))
                   (t (display-string (format nil "    undef ") w)))
                 (display-string (format nil "  ~5f" varn) w))))))

    (when (= 1 (select test 1))
          (if (send self :ways)
              (display-string (format nil "~2%CELL NAMES             ") w)
              (display-string (format nil "~2%VARIABLES (Ord. & Num.)") w))
          (display-string (format nil 
                 "  MINIMUM     1st Q     MEDIAN      3rd Q    MAXIMUM") w)
          (dotimes (i nactvar)
             (when 
              (or (send self :matrices)
                  (send self :ways)
                  (and (not (send self :matrices))
                       (or 
                        (equalp "numeric" 
                                (select (send self :active-types '(all)) i))
                        (equalp "ordinal" 
                                (select (send self :active-types '(all)) i))
                        )))
              (setf varnamenow (select varnames i))
              (when (> (length varnamenow) 20)
                    (setf varnamenow (subseq varnamenow 0 20)))
              (setf varnow (select variables i))
              (setf var5num (fivnum varnow))
              (display-string 
               (format nil "~%~20a ~10,2f ~10,2f ~10,2f ~10,2f ~10,2f"
                       varnamenow (+ .00001 (select var5num 0)) (+ .00001 (select var5num 1))
                       (+ .00001 (select var5num 2)) (+ .00001 (select var5num 3))
                       (+ .00001 (select var5num 4))) w))))
    (when (= 1 (select test 2))
          (if (send self :ways)
              (display-string (format nil "~2%CELL NAMES         ") w)
              (display-string (format nil "~2%VARIABLES (Numeric)") w))
          (display-string (format nil "     IQ-RANGE     RANGE  MID-RANGE") w)
          (dotimes (i nactvar)
             (when 
              (or (send self :matrices)
                  (send self :ways)
                  (and (not (send self :matrices))
                       (equalp "numeric" 
                               (select (send self :active-types '(all)) i))))
              (setf varnamenow (select varnames i))
              (when (> (length varnamenow) 20)
                    (setf varnamenow (subseq varnamenow 0 20)))
              (setf varnow (select variables i))
              (setf iq-range (interquartile-range varnow))
              (setf mid-range (mid-range varnow))
              (setf var5num (fivnum varnow))
              (setf range (range varnow))
              (display-string 
               (format nil "~%~20a ~10,2f ~10,2f ~10,2f" 
                       varnamenow (+ .00001 iq-range) (+ .00001 range)
                       (+ .00001 mid-range)) w))))
    (cond
      ((and novarvar (= 1 (select test 3)))
           (display-string 
        (format nil "~%Correlations and Covariances cannot be computed.~%")))
      (t (when (= 1 (select test 3))
            (when (> nactvar 1)
                (display-string (format nil "~2%CORRELATIONS (Numeric Variables)~%") w)
                (setf data-matrix (lists-to-matrix variables))
                (print-matrix-to-window 
                 (if (send self :matrices)
                     (fuzz (correlation-matrix data-matrix))
                     (fuzz (correlation-matrix data-matrix :types (send self :active-types '(all)))))
                 w :labels 
                 (if (send self :matrices) 
                     varnames
                     (select varnames ($position '("numeric") (send self :active-types '(all))))))))
    (when (= 1 (select test 4))
          (when (and (> nactvar 1) (not novarvar))
                (display-string (format nil "~2%COVARIANCES (Numeric Variables)~%") w)
                (when (not data-matrix) (setf data-matrix (lists-to-matrix variables)))
                (print-matrix-to-window 
                 (if (send self :matrices)
                     (fuzz (covariance-matrix data-matrix))
                     (fuzz (covariance-matrix
                            (select data-matrix 
                                    (iseq (select (size data-matrix) 0))
                                    ($position '("numeric") (send self :active-types '(all)))))))
                 w :labels
                 (if (send self :matrices) 
                     varnames
                     (select varnames ($position '("numeric") (send self :active-types '(all))))))))))
    (when (not w) (terpri)) ; fwy4.28
    w ))


;;---------------------------------------------------------------------------
;;define methods for reporting simple univariate statistics
;;---------------------------------------------------------------------------

(defmeth mv-data-object-proto :means ()
"Args: none
Reports the means of the active numeric variables."
  (mapcar #'mean
          (column-list (send self :active-data-matrix '(numeric)))))

(defmeth mv-data-object-proto :medians ()
"Args: none
Reports the medians of the active numeric and ordinal variables."
  (mapcar #'median
          (column-list (send self :active-data-matrix '(numeric ordinal)))))

(defmeth mv-data-object-proto :standard-deviations ()
"Args: none
Reports the standard deviations of the active numeric variables."
  (mapcar #'standard-deviation 
          (column-list (send self :active-data-matrix '(numeric)))))

(defmeth mv-data-object-proto :variances ()
"Args: none
Reports the variances of the active numeric variables."
  (^ (send self :standard-deviations) 2))

(defmeth mv-data-object-proto :minimums ()
"Args: none
Reports the minimums of the active numeric and ordinal variables."
  (mapcar #'min
          (column-list (send self :active-data-matrix '(numeric ordinal)))))

(defmeth mv-data-object-proto :maximums ()
"Args: none
Reports the maximums of the active numeric and ordinal variables."
  (mapcar #'max
          (column-list (send self :active-data-matrix '(numeric ordinal)))))

(defmeth mv-data-object-proto :mid-ranges ()
"Args: none
Reports the mid-ranges of the active numeric and ordinal variables."
  (mapcar #'mid-range 
          (column-list (send self :active-data-matrix '(numeric ordinal)))))

(defmeth mv-data-object-proto :ranges ()
"Args: none
Reports the ranges of the active numeric variables."
  (mapcar #'range
          (column-list (send self :active-data-matrix '(numeric)))))
               
(defmeth mv-data-object-proto :interquartile-ranges ()

"Args: none
Reports the interquartile ranges of the active numeric and ordinal variables."
  (mapcar #'interquartile-range
          (column-list (send self :active-data-matrix '(numeric ordinal)))))

(defmeth mv-data-object-proto :skewnesses ()
"Args: none
Reports the skewnesses of the active numeric variables."
  (mapcar #'skewness
          (column-list (send self :active-data-matrix '(numeric)))))

(defmeth mv-data-object-proto :kurtoses ()
"Args: none
Reports the kurtoses of the active numeric variables."
  (mapcar #'kurtosis
          (column-list (send self :active-data-matrix '(numeric)))))

(defmeth mv-data-object-proto :covariance-matrix ()
"Args: none
Reports the covariance-matrix of the active numeric variables."
  (covariance-matrix (send self :active-data-matrix '(numeric))))

;;###########################################################################
;;Define prototype dissimilarity and table data objects.
;;These prototypes inherit from the multivariate data-object prototype.
;;The methods for these prototypes are defined in dissobj.lsp and table.obj.
;;The defproto statements appear here so that data guidance will work 
;;even though the dissobj and tableobj files have not been loaded.
;;###########################################################################

(defproto diss-data-object-proto 
  '(enames mshapes nmat nele mat-window mat-window-object mat-states) 
  () mv-data-object-proto)

(defproto table-data-object-proto 
  '(classes nways nclasses ncells cellfreqs source-names level-names 
            indicator-matrices obs-labels) 
  () mv-data-object-proto)

(provide "dataobj3")
