/*
Copyright (C) 2003, Nik Reiman - nik@aboleo.net

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

/****
  This file is responsible for calling the external media player and actually playing music on the air.  Note that the UBS does not actually play any music itself; it simply parses through the configuration settings and exec's out a new process.
****/

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <sys/types.h>
#include <sys/wait.h>

#include "ubs-play.h"

/*+
  Plays the song from the queue.  This function basically takes the "media" directive from the configuration file, and forks out a new process to play the song, waiting until this process ends before returning.

  int play_song Returns OK on success, NO_FORK if fork fails, FAIL if exec fails, and NO_FORK if waitpid fails

  char *songname The name of the song to play.  This function has nothing to due with the queue; it simply expects to get passed a filename which it can open and play.
+*/
int play_song(char *songname)
{
 int pid;
 char buf[STRBUF];

 if((pid = fork()) < 0) {
  log_error_msg(LOG_ERROR, "Can't create fork");
  return NO_FORK;
 }
 // child process
 else if(pid == 0) {
  exec_proc(ubs_table_data(&PLAY, "media"), songname);
  // should never reach here
  log_error_msg(LOG_DEBUG, "Return from exec_proc");
  return FAIL;
 }
 // parent process
 else {
  // record the PID of the media playing process so that we may kill it
  strncpy(buf, PROCNAME, STRBUF);
  strncpy(PROCNAME, "ubs-media", STRBUF);
  record_pid(pid);
  strncpy(PROCNAME, buf, STRBUF);

  // wait for the forked process to end
  if(waitpid(pid, NULL, 0) < 0) {
   log_error_msg(LOG_ERROR, "Waitpid failed");
   return NO_FORK;
  }
 }

 return OK;
}
